<?php
/**
 * Manage the colors.
 *
 * @package mateo
 */

namespace Mateo;

use function SSNepenthe\ColorUtils\desaturate;
use function SSNepenthe\ColorUtils\lighten;
use function SSNepenthe\ColorUtils\rgba;

/**
 * Manage theme CSS rules.
 * Generate custom colors.
 */
class ThemeColor {
	/**
	 * Associative array of colors with a key is a color id and a HEX code as value.
	 *
	 * @see mateo_get_color_schemes() function for colors ids (color section).
	 *
	 * @var array
	 */
	protected $colors = array();

	/**
	 * Array of colors not saved in DB.
	 * Derived from $this->colors
	 *
	 * @var array
	 */
	protected $variations = array();

	/**
	 * Required by css template colors keys.
	 *
	 * @var string[]
	 */
	protected $keys = array(
		'primary',
		'primaryLight',
		'primaryDark',
		'textOnPrimary',
		'link',
		'textBody',
		'title',
		'titleBgr',
		'titleNoHeader',
		'contentBgr',
		'pageBgr',
		'primarySubmenuBgr',
		'switchCheckedBgr',
		'currentMenuItem',
		'menuItemHover',
		'bmdLabelOnPrimary',
		'bmdLabelFooter',
		'footerInputInvalid',
		'preBgr',
		'lightTextBody',
		'varyLightTextBody',
		'inputDisabled',
		'cite',
	);
	/**
	 * Default color to used in place of missing color.
	 *
	 * @var string
	 */
	protected $default_fill_color = '#000000';

	/**
	 * Constructor.
	 *
	 * @param array $colors     Associative array of colors keys and their value.
	 */
	public function __construct( array $colors ) {
		if ( $colors ) {
			$this->set_colors( $colors );
			$this->init_variations();
		}
	}

	/**
	 * Initialize color variations used in css.
	 * Rules are the same as in scss files.
	 */
	protected function init_variations() {
		if ( isset( $this->colors['primary'] ) ) {
			$this->variations['primarySubmenuBgr'] = (string) ( rgba( $this->colors['primary'], 0.9 ) );
			$this->variations['switchCheckedBgr']  = (string) ( desaturate( lighten( $this->colors['primary'], 28 ), 32 ) );
		}
		if ( isset( $this->colors['textOnPrimary'] ) ) {
			$this->variations['currentMenuItem']    = (string) ( rgba( $this->colors['textOnPrimary'], 0.2 ) );
			$this->variations['menuItemHover']      = $this->variations['currentMenuItem'];
			$this->variations['bmdLabelOnPrimary']  = (string) ( rgba( $this->colors['textOnPrimary'] ) );
			$this->variations['bmdLabelFooter']     = (string) ( rgba( $this->colors['textOnPrimary'], 0.8 ) );
			$this->variations['footerInputInvalid'] = (string) ( rgba( $this->colors['textOnPrimary'], 0.26 ) );
		}
		if ( $this->colors['textBody'] ) {
			$this->variations['preBgr']            = (string) ( rgba( $this->colors['textBody'], 0.01 ) );
			$this->variations['lightTextBody']     = (string) ( rgba( $this->colors['textBody'], 0.8 ) );
			$this->variations['varyLightTextBody'] = (string) ( rgba( $this->colors['textBody'], 0.1 ) );
			$this->variations['inputDisabled']     = (string) ( rgba( $this->colors['textBody'], 0.5 ) );
			$this->variations['cite']              = (string) ( rgba( $this->colors['textBody'], 0.5 ) );
		}
	}

	/**
	 * Load customized colors from DB.
	 */
	public function init_user_settings() {
		$mods = get_theme_mods();
		foreach ( $this->colors as $id => $color ) {
			if ( isset( $mods[ $id ] ) ) {
				$this->set_color( $id, $mods[ $id ] );
			}
		}
		$this->init_variations();
	}

	/**
	 * Load colors from DB for the current change set and generate some color variants from them.
	 * Used during theme customisation.
	 */
	public function init_change_set() {
		foreach ( $this->colors as $id => $color ) {
			$uc = get_theme_mod( $id, null );
			if ( $uc ) {
				$this->set_color( $id, $uc );
			}
		}
		$this->init_variations();
	}

	/**
	 * Get ids(keys) of declared colors.
	 *
	 * @return array
	 */
	public function get_ids() {
		return array_merge(
			array_keys( $this->colors ),
			array_keys( $this->variations ),
			$this->keys
		);
	}

	/**
	 * Checks if color is valid.
	 *
	 * @param string $hex Hexadecimal color value with starting '#".
	 * @return bool True if color is OK otherwise false.
	 */
	public function is_color_valid( $hex ) {
		$value = sanitize_hex_color( $hex );
		return ! empty( $value );
	}
	/**
	 * Set color in $this->colors array.
	 *
	 * @param string $id    Color id.
	 * @param string $color Color value (hex code).
	 */
	public function set_color( $id, $color ) {
		if ( ! $this->is_color_valid( $color ) ) {
			$color = $this->default_fill_color;
		}
		$this->colors[ $id ] = $color;
	}

	/**
	 * Set colors in $this->colors array.
	 *
	 * @param array $colors Associative array of colors.
	 */
	public function set_colors( array $colors ) {
		foreach ( $colors as $key => $value ) {
			$this->set_color( $key, $value );
		}
	}

	/**
	 * Get color by its id(key) from $this->colors array.
	 *
	 * @param string $id    Color key.
	 *
	 * @return string|null  Color hex value or null if color key does not exist
	 */
	public function get_color( $id ) {
		if ( ! array_key_exists( $id, $this->colors ) ) {
			return null;
		}

		return $this->colors[ $id ];
	}

	/**
	 * Wrap css rules into html style tag.
	 * If rules are not provided as a parameter they will be generated.
	 *
	 * @param string $id    HTML id for generated style tag.
	 * @param string $css   CSS rules.
	 *
	 * @return string
	 */
	public function get_style_tag( $id = '', $css = '' ) {
		if ( ! $css ) {
			$css = $this->get_css( true );
		}
		return '<style type="text/css" ' . ( $id ? 'id="' . esc_attr( $id ) . '"' : '' ) . '>' . $css . '</style>';
	}

	/**
	 * Customizable CSS rules.
	 * Declared rules are used also to generate a css template with placeholders for customization.
	 *
	 * @param array $data   Associative array with color id as a key and hex code or template placeholder as a value.
	 *
	 * @return string
	 */
	protected function get_css_rules( array $data ) {
		$css = <<<CSS
:root{
	--primary: {$data['primary']};
	--primary-light: {$data['primaryLight']};
	--primary-dark: {$data['primaryDark']}
}
body {
	color: {$data['textBody']};
	background-color: {$data['pageBgr']}
}
body.custom-background {
background-color: {$data['pageBgr']}
}
a, a:hover {
	color: {$data['link']}
}
blockquote,
.wp-block-quote,
.wp-block-quote.is-large,
.wp-block-quote.is-style-large,
.wp-block-quote.has-text-align-right,
.wp-block-quote.has-text-align-center,
.wp-block-quote.has-text-align-left,
.wp-block-quote:not(.is-large):not(.is-style-large)
{
	border-color: {$data['primaryLight']};
}
.wp-block-pullquote.is-style-solid-color .has-text-color cite {
	color: inherit;
}
pre {
	background-color: {$data['preBgr']};
	border-color:{$data['varyLightTextBody']};
	color: {$data['textBody']}
}
kbd,
tt {
	color: {$data['contentBgr']};
	background-color: {$data['textBody']};
}
code {
	color:{$data['textBody']};
	background-color: {$data['varyLightTextBody']}
}

input[type="submit"],
input[type="button"],
input[type="reset"],
button,
.btn-float-edit {
	color: {$data['textOnPrimary']};
	background-color: {$data['primary']};
	border-color:{$data['primaryLight']}
}
input:hover[type="submit"],
input:hover[type="button"],
input:hover[type="reset"],
button:hover,
.btn-float-edit:hover {
	color: {$data['textOnPrimary']};
	background-color: {$data['primaryDark']};
	border-color: {$data['primary']}
}
input:focus[type="submit"],
input:focus[type="button"],
input:focus[type="reset"],
button:focus,
.btn-float-edit:focus {
	color: {$data['textOnPrimary']};
	background-color: {$data['primaryDark']};
	border-color: {$data['primaryDark']}
}
input:active[type="submit"],
input:active[type="button"],
input:active[type="reset"],
button:active,
.btn-float-edit:active {
	color: {$data['textOnPrimary']};
	background-color: {$data['primaryDark']};
	border-color: {$data['primaryDark']}
}
.btn-post-header-edit {
	color: {$data['primary']}
}
[type="radio"]:not(:checked) + label:before,
[type="radio"]:not(:checked) + label:after,
[type="radio"]:not(:checked) + .label:before,
[type="radio"]:not(:checked) + .label:after {
  border-color: {$data['lightTextBody']}
}
[type="radio"]:checked:not(:disabled) + label:after,
[type="radio"]:checked:not(:disabled) + .label:after{
	border-color:{$data['primary']};
	background-color: {$data['primary']}
}
[type="radio"]:disabled + label:before,
[type="radio"]:disabled + .label:before{
	background-color: {$data['inputDisabled']};
}
[type="radio"]:disabled + label,
[type="radio"]:disabled + .label{
	color: {$data['inputDisabled']}
}

[type="checkbox"] + label:before,
[type="checkbox"] + label:after,
[type="checkbox"] + .label:before,
[type="checkbox"] + .label:after {
	border-color: {$data['lightTextBody']}
}
[type="checkbox"]:checked:not(:disabled) + label:before,
[type="checkbox"]:checked:not(:disabled) + .label:before
 {
	border-right-color: {$data['primary']};
	border-bottom-color: {$data['primary']}
}

.rtl [type="checkbox"]:checked:not(:disabled) + label:before,
.rtl [type="checkbox"]:checked:not(:disabled) + .label:before
{
	border-right-color: transparent;
	border-left-color: {$data['primary']};
	border-bottom-color: {$data['primary']}
}

[type="checkbox"]:indeterminate + label:before,
[type="checkbox"]:indeterminate + .label:before{
	border-right-color: {$data['primary']}
}

.is-focused [class^='bmd-label'],
.is-focused [class*=' bmd-label'],
.is-focused [class^='bmd-label'],
.is-focused [class*=' bmd-label']{
	color: {$data['primary']}
}
.comment-author .avatar {
	border-color:{$data['varyLightTextBody']}
}
.comment-action > a, #cancel-comment-reply-link {
	color: {$data['primary']}
}
.comment-action > a:hover, #cancel-comment-reply-link:hover {
	color: {$data['textOnPrimary']};
	background-color:{$data['primary']}
}
.comment-action > a:focus,
.comment-action > a.focus,
#cancel-comment-reply-link:focus {
	color: {$data['textOnPrimary']};
	background-color:{$data['primary']}
}
.comment-action > a:active,
 #cancel-comment-reply-link:active {
	 color: {$data['textOnPrimary']};
	 background-color: {$data['primaryDark']}
 }
.tag-cloud-link,
.chips-links a {
	color: {$data['lightTextBody']};
	background-color: {$data['varyLightTextBody']}
}
.tag-cloud-link:hover,
.chips-links a:hover {
	color: {$data['textOnPrimary']};
	background-color: {$data['primary']}
}
.tag-cloud-link:focus,
.tag-cloud-link.focus,
.chips-links a:focus,
.chips-links a.focus {
	color: {$data['textOnPrimary']};
	background-color: {$data['primary']}
}
.tag-cloud-link:active,
.chips-links a:active{
	color: {$data['textOnPrimary']};
	background-color: {$data['primaryDark']}
}
body > header {
	background-color: {$data['primary']}
}
body > header a, body > header .navbar-toggler-icon {
	color: {$data['textOnPrimary']}
}
.sub-menu {
	background-color: {$data['primarySubmenuBgr']}
}
.site-title, .site-title a {
	color: {$data['textOnPrimary']}
}
.site-title a:before {
	background-color: {$data['textOnPrimary']};
}
.main-menu > li.current-menu-item,
.main-menu > li.current-menu-ancestor{
	background-color: {$data['currentMenuItem']}
}

.navbar-toggler,
.main-menu a,
.s-main-menu a,
.top-search-btn {
	color: {$data['textOnPrimary']}
}

.main-menu a:before,
.s-main-menu a:before,
.top-search-btn:before,
.navbar-toggler:before {
	background-color: {$data['textOnPrimary']}
}
.main-menu a:hover,
.main-menu a:focus,
.s-main-menu a:hover,
.s-main-menu a:focus,
.top-search-btn:hover,
.top-search-btn:focus,
.navbar-toggler:hover,
.navbar-toggler:focus{
	color:{$data['textOnPrimary']};
	background-color: {$data['menuItemHover']}
}

.s-main-menu {
	background-color: {$data['primarySubmenuBgr']}
}

.search-submit ,
.search-cancel{
	color: {$data['primary']}
}
.search-submit:hover,
.search-submit:focus,
.search-cancel:hover,
.search-cancel:focus
{
	color: {$data['textOnPrimary']};
	background-color:{$data['primary']}
}
.search-submit:active,
.search-cancel:active
{
	color: {$data['textOnPrimary']};
	background-color: {$data['primaryDark']}
}
form.top-search-form {
	background-color: {$data['primary']}
}
.page-title {
	color: {$data['titleNoHeader']}
}
.page-title span {
	background-color: transparent;
}
.has-image .page-title {
	color: {$data['title']}
}
.has-image .page-title span {
	background-color: {$data['titleBgr']}
}

.site-content {
	background-color: {$data['contentBgr']}
}
.page-footer, .page-footer a{
	color: {$data['textOnPrimary']}
}
.footer-widget-area {
	background-color: {$data['primary']}
}

.footer-widget-area .widget-title {
	color: {$data['textOnPrimary']}
}
.footer-copyright-area {
	background-color:{$data['primaryDark']}
}
.pagination .nav-links li .page-numbers{
	color: {$data['textBody']}
}
.pagination .nav-links li .current{
	color: {$data['textOnPrimary']};
	background-color: {$data['primary']}
}

.comment-navigation .material-icons,
.post-navigation .material-icons{
	color: {$data['textOnPrimary']};
	background-color: {$data['primary']}
}
.btn-post-header-edit:hover,
.btn-post-header-edit:focus{
	color: {$data['textOnPrimary']};
	background-color:{$data['primaryDark']}
}
.btn-post-header-edit:active{
	color: {$data['textOnPrimary']};
	background-color: {$data['primary']}
}
.form-control,
input:not([type="button"]):not([type="submit"]):not([type="reset"]):not([type="checkbox"]):not([type="radio"]),
select,
textarea,
.is-focused .form-control,
.is-focused input:not([type="button"]):not([type="submit"]):not([type="reset"]):not([type="checkbox"]):not([type="radio"]),
.is-focused select,
.is-focused textarea {
	background-image: linear-gradient(to top,  {$data['primary']} 2px, rgba(233,30,99,0) 2px),linear-gradient(to top, rgba(0,0,0,0.26) 1px, rgba(0,0,0,0) 1px)
}
.form-control:invalid,
input:invalid:not([type="button"]):not([type="submit"]):not([type="reset"]):not([type="checkbox"]):not([type="radio"]),
select:invalid,
textarea:invalid {
	background-image: linear-gradient(to top,  {$data['primaryDark']} 2px, rgba(213,0,0,0) 2px),linear-gradient(to top, rgba(0,0,0,0.26) 1px,  rgba(0,0,0,0) 1px)
}
.footer-widget-area .search-submit {
	color: {$data['bmdLabelOnPrimary']}
}
.footer-widget-area input:not([type="button"]):not([type="submit"]):not([type="checkbox"]):not([type="radio"]):not([type="reset"]),
.footer-widget-area select {
	color: {$data['textOnPrimary']};
}

.footer-widget-area [class^="bmd-label"],
.footer-widget-area [class*=" bmd-label"] {
	color: {$data['bmdLabelFooter']}
}
.footer-widget-area .form-control:invalid,
.footer-widget-area input:invalid:not([type="button"]):not([type="submit"]):not([type="reset"]):not([type="checkbox"]):not([type="radio"]),
.footer-widget-area select:invalid,
.footer-widget-area textarea:invalid {
	background-image: linear-gradient(to top,  {$data['textOnPrimary']} 2px, rgba(213,0,0,0) 2px),linear-gradient(to top, {$data['footerInputInvalid']} 1px,  rgba(0,0,0,0) 1px)
}
.footer-widget-area .form-control,
.footer-widget-area input:not([type="button"]):not([type="submit"]):not([type="reset"]):not([type="checkbox"]):not([type="radio"]),
.footer-widget-area select,
.footer-widget-area textarea,
.footer-widget-area .is-focused .form-control,
.footer-widget-area .is-focused input:not([type="button"]):not([type="submit"]):not([type="reset"]):not([type="checkbox"]):not([type="radio"]),
.footer-widget-area .is-focused select,
.footer-widget-area .is-focused textarea {
	background-image: linear-gradient(to top,  {$data['textOnPrimary']} 2px, rgba(213,0,0,0) 2px),linear-gradient(to top, {$data['footerInputInvalid']} 1px,  rgba(0,0,0,0) 1px)
}
.alert-primary {
	background-color: {$data['primaryLight']};
	color: {$data['textOnPrimary']}
}
.content-area a:before {
	background-color: {$data['link']}
}
.sidebar a:before {
	background-color: {$data['link']}
}
.calendar_wrap table tbody a,
.wp-block-calendar table tbody a {
	background-color: {$data['primary']};
	color: {$data['textOnPrimary']}
}
.calendar_wrap table caption,
.wp-block-calendar table caption {
	color:  {$data['textBody']}
}
.page-footer .calendar_wrap th,
.page-footer .calendar_wrap td {
	color: {$data['textOnPrimary']}
}
.page-footer .calendar_wrap tbody a {
	background-color: {$data['textOnPrimary']};
	color: {$data['primary']}
}
.page-footer .calendar_wrap caption {
	color: {$data['textOnPrimary']}
}
.page-footer a:before {
	background-color: {$data['textOnPrimary']}
}

.wp-block-quote__citation, .wp-block-quote cite, .wp-block-quote footer {
  color: {$data['cite']};
}
.has-primary-color {
  color: {$data['primary']};
}
.has-primary-light-color {
  color: {$data['primaryLight']};
}
.has-primary-dark-color {
  color: {$data['primaryDark']};
}
.has-text-on-primary-color {
  color:  {$data['textOnPrimary']};
}
.has-text-body-color {
  color: {$data['textBody']};
}
.has-primary-background-color {
  background-color: {$data['primary']};
}
.has-primary-light-background-color {
  background-color: {$data['primaryLight']};
}
.has-primary-dark-background-color {
  background-color:  {$data['primaryDark']};
}
.has-text-on-primary-background-color {
  background-color: {$data['textOnPrimary']};
}
.has-text-body-background-color {
  background-color: {$data['textBody']};
}
.wp-block-button__link, .wp-block-file__button {
	background-color:  {$data['primary']};
	color:  {$data['textOnPrimary']};
}
.is-style-outline .wp-block-button__link {
  border-color: {$data['primary']};
  background-color: transparent;
}

.is-style-outline .wp-block-button__link:not(.has-text-color) {
  color: {$data['primary']};
}

CSS;

		return $css;
	}

	/**
	 * Customizable CSS rules.
	 * Declared rules are used also to generate a css template with placeholders for customization.
	 *  background-color: {$data['pageBgr']}
	 *
	 * @param array $data   Associative array with color id as a key and hex code or template placeholder as a value.
	 *
	 * @return string
	 */
	protected function get_css_rules_editor( array $data ) {
		$css = <<<CSS
:root {
	--mateo-content-bgr: {$data['contentBgr']};
	--mateo-page-bgr: {$data['pageBgr']};
	--mateo-page-title-color: {$data['title']};
	--mateo-page-title-color-no-header: {$data['titleNoHeader']};
	--mateo-page-title-bgr: {$data['titleBgr']};
	--mateo-page-title-bgr-no-header: transparent;
}
body  {
	color: {$data['textBody']};
}
a, a:hover, a:focus {
	color: {$data['link']}
}
a:before {
	 background-color: {$data['link']};
}
blockquote,
.wp-block-quote,
.wp-block-quote.is-large,
.wp-block-quote.is-style-large,
.wp-block-quote.has-text-align-right,
.wp-block-quote.has-text-align-center,
.wp-block-quote.has-text-align-left,
.wp-block-quote:not(.is-large):not(.is-style-large)
{
	border-color: {$data['primaryLight']};
}
.wp-block-pullquote.is-style-solid-color .has-text-color cite {
	color: inherit;
}
pre {
	background-color: {$data['preBgr']};
	border-color:{$data['varyLightTextBody']};
	color: {$data['textBody']}
}
kbd,
tt {
	color: {$data['contentBgr']};
	background-color: {$data['textBody']};
}
code {
	color:{$data['textBody']};
	background-color: {$data['varyLightTextBody']}
}

.wp-block-quote__citation,
.wp-block-quote cite,
.wp-block-quote footer {
  color: {$data['cite']};
}
.wp-block-button__link, .wp-block-file__button {
	background-color:  {$data['primary']};
	color:  {$data['textOnPrimary']};
}
.is-style-outline .wp-block-button__link {
  border-color: {$data['primary']};
  background-color: transparent;
}

.is-style-outline .wp-block-button__link:not(.has-text-color) {
  color: {$data['primary']};
}
CSS;

		return $css;
	}

	/**
	 * Generated css rules.
	 *
	 * @param bool $strip_eol   TRUE to remove line breaks, FALSE otherwise.
	 * @param bool $is_editor_css TRUE to return block editor css for the backend, FALSE for the frontend css.
	 * @return string
	 */
	public function get_css( $strip_eol = false, $is_editor_css = false ) {
		$colors = $this->fill_with_required(
			array_merge( $this->variations, $this->colors )
		);
		$css    = ! $is_editor_css ? $this->get_css_rules( $colors ) : $this->get_css_rules_editor( $colors );

		return $strip_eol ? static::strip_eol( $css ) : $css;
	}

	/**
	 * Check if some required in css template colors are missing.
	 * Fill passed in parameter colors array with missing colors if required.
	 *
	 * @param array $colors Colors.
	 * @return array
	 */
	public function fill_with_required( $colors ) {
		if ( count( $colors ) === count( $this->get_ids() ) ) {
			return $colors;
		}
		$diff = array_diff( array_keys( $colors ), $this->get_ids() );
		if ( $diff ) {
			$colors = array_merge( $colors, array_fill_keys( $diff, $this->default_fill_color ) );
		}

		return $colors;
	}
	/**
	 * Generate css template for customer.
	 *
	 * @param bool $strip_eol   TRUE to remove line breaks, FALSE otherwise.
	 *
	 * @return string
	 */
	public function get_css_template( $strip_eol = false ) {
		$placeholders = array();
		foreach ( $this->get_ids() as $id ) {
			$placeholders[ $id ] = sprintf( '{{data.%s}} ', $id ); // final space in '{{data.%s}} ' is important!
		}

		$tpl = $this->get_css_rules( $placeholders );

		return $strip_eol ? static::strip_eol( $tpl ) : $tpl;
	}

	/**
	 * Remove tabs and line breaks from the content.
	 *
	 * @param string $content   Some content to work with.
	 *
	 * @return string
	 */
	public static function strip_eol( $content ) {
		return str_replace( array( "\t", "\r", "\n" ), '', $content );
	}

}
