<?php
/**
 * Metabox for managing post/page display options.
 *
 * @package mateo
 */

namespace Mateo;

/**
 * Usage: get instance of the class ThemeOptionsMetabox::get_instance();
 *
 * @package Mateo
 */
class ThemeOptionsMetabox {

	/**
	 * Metabox id.
	 *
	 * @var string
	 */
	const ID = 'mateo_post_theme_opts';
	/**
	 * Nonce field name.
	 */
	const NONCE_NAME = 'mateo_post_theme_opts';
	/**
	 * Nonce field value.
	 */
	const NONCE_ACTION = '2df9336d-4d7f-4299-9757-d4f5d0aa9c70';
	/**
	 * Meta name for the option allowing to apply one column layout to the post.
	 * The option name is used in javascript for dynamic switch between 1 and 2 column layout.
	 *
	 * @var string
	 */
	const META_LAYOUT = '_mateo_column_layout';
	/**
	 * Meta name for the option allowing to disable a display of a post header (title and image).
	 *
	 * @var string
	 */
	const META_DISABLE_HEADER = '_mateo_disable_header';
	/**
	 * Meta name for the option allowing to customize a page background color thus overriding the site setting.
	 *
	 * @var string
	 */
	const META_PAGE_BGR = '_mateo_page_bgr';

	/**
	 * Instance of the class.
	 *
	 * @var ThemeOptionsMetabox
	 */
	private static $instance;
	/**
	 * Keys :
	 * - disable: an array of meta names do not display.
	 *
	 * @var array Metabox options array
	 */
	protected $options = array(
		'disable' => array(),
	);

	/**
	 * ThemeOptionsMetabox constructor.
	 * Class implements singleton pattern.
	 *
	 * @use ThemeOptionsMetabox::get_instance() method.
	 */
	protected function __construct() {
	}

	/**
	 * Sets options array.
	 *
	 * @param array $options Options.
	 */
	public function set_options( array $options ) {
		$this->options = $options;
	}

	/**
	 * Gets options array.
	 *
	 * @return array
	 */
	public function get_options() {
		return $this->options;
	}

	/**
	 * Checks if an option is disabled.
	 *
	 * @param string $name Option name.
	 *
	 * @return bool
	 */
	public function is_option_disabled( $name ) {
		if ( is_array( $this->options['disable'] ) && in_array( $name, $this->options['disable'], true ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Gets instance of the class.
	 *
	 * @return ThemeOptionsMetabox
	 */
	public static function get_instance() {
		if ( null === static::$instance ) {
			static::$instance = new ThemeOptionsMetabox();
		}

		return static::$instance;
	}

	/**
	 * Plug into WP.
	 *
	 * @param array $options Options array.
	 */
	public static function init( $options = array() ) {
		$instance = static::get_instance();
		$instance->set_options( array_merge( $instance->get_options(), $options ) );
		add_action( 'add_meta_boxes', array( $instance, 'add' ) );
		add_action( 'save_post', array( $instance, 'save' ) );
	}

	/**
	 * Adds metabox to WP.
	 */
	public function add() {
		add_meta_box(
			static::ID,
			__( 'Theme options', 'mateo' ),
			array( $this, 'display_metabox' ),
			null,
			'side'
		);
	}

	/**
	 * Saves in DB.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save( $post_id ) {
		// Post meta data is not saved with autosave, so we don't want to do anything.
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// Check if nonce is set.
		if ( ! isset( $_POST[ static::NONCE_NAME ] ) ) {
			return;
		}
		// Verify that the nonce is valid.
		if ( ! wp_verify_nonce( wp_unslash( $_POST[ static::NONCE_NAME ] ), static::NONCE_ACTION ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			return;
		}
		if ( ! $this->is_option_disabled( static::META_LAYOUT ) ) {
			$this->save_layout_option( $post_id );
		}
		if ( ! $this->is_option_disabled( static::META_DISABLE_HEADER ) ) {
			$this->save_disable_header_option( $post_id );
		}
		if ( ! $this->is_option_disabled( static::META_PAGE_BGR ) ) {
			$this->save_page_background_option( $post_id );
		}
	}

	/**
	 * Displays metabox input field.
	 *
	 * @param WP_Post $post Post.
	 */
	public function display_metabox( $post ) {
		$this->output_header_html();
		if ( ! $this->is_option_disabled( static::META_LAYOUT ) ) {
			$this->output_layout_html( $post );
		}
		if ( ! $this->is_option_disabled( static::META_DISABLE_HEADER ) ) {
			$this->output_disable_header_html( $post );
		}

		if ( ! $this->is_option_disabled( static::META_PAGE_BGR ) ) {
			$this->output_page_background_html( $post );
		}
		$this->output_footer_html();
		wp_nonce_field( static::NONCE_ACTION, static::NONCE_NAME );
	}
	/**
	 * Outputs starting html for the metabox.
	 */
	public function output_header_html() {
		echo '<p><em>';
		esc_html_e( 'To preview theme options changes in a new tab you need to save the post.', 'mateo' );
		echo '</em></p>';
	}
	/**
	 * Outputs concluding html for the metabox.
	 */
	public function output_footer_html() {
	}


	/**
	 * Outputs an input to disable a header display.
	 *
	 * @param WP_Post $post Post.
	 */
	public function output_disable_header_html( $post ) {
		$value = (int) get_post_meta( $post->ID, static::META_DISABLE_HEADER, true );
		$id    = esc_attr( static::META_DISABLE_HEADER );
		?>
		<div class="components-panel__row">
			<div class="components-base-control">
				<div class="components-base-control__field">
					<input id="<?php echo $id; /* phpcs:ignore */ ?>" name="<?php echo $id; /* phpcs:ignore */ ?>" type="checkbox" value="1" <?php checked( 1, $value ); ?>>
					<label for="<?php echo $id; /* phpcs:ignore */ ?>">
						<?php esc_html_e( 'Presentation Page', 'mateo' ); ?>
					</label>
					<p><em>
						<?php esc_html_e( 'This option allows you to build a fully custom page with editor. It disables the display of header area with title, breadcrumb, top and bottom spacing.', 'mateo' ); ?>
					</em>
					</p>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Saves option in database.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_disable_header_option( $post_id ) {
		$value = (int) filter_input( INPUT_POST, static::META_DISABLE_HEADER, FILTER_SANITIZE_NUMBER_INT );
		if ( 1 === $value ) {
			update_post_meta( $post_id, static::META_DISABLE_HEADER, $value );
			return;
		}
		delete_post_meta( $post_id, static::META_DISABLE_HEADER );
	}

	/**
	 * Outputs an input for choosing a post layout.
	 *
	 * @param WP_Post $post Post.
	 */
	public function output_layout_html( $post ) {
		$value = (int) get_post_meta( $post->ID, static::META_LAYOUT, true );
		$id    = esc_attr( static::META_LAYOUT );
		?>
		<div class="components-panel__row">
			<div class="components-base-control">
				<div class="components-base-control__field">
					<input id="<?php echo $id; /* phpcs:ignore */ ?>" name="<?php echo $id; /* phpcs:ignore */ ?>" type="checkbox" value="1" <?php checked( 1, $value ); ?>>
					<label for="<?php echo $id; /* phpcs:ignore */ ?>">
						<?php esc_html_e( 'Disable sidebar', 'mateo' ); ?>
					</label>
				</div>
			</div>
		</div>

		<?php
	}

	/**
	 * Saves post's layout option in database.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_layout_option( $post_id ) {
		$value = (int) filter_input( INPUT_POST, static::META_LAYOUT, FILTER_SANITIZE_NUMBER_INT );
		if ( 1 === $value ) {
			update_post_meta( $post_id, static::META_LAYOUT, $value );
			return;
		}
		delete_post_meta( $post_id, static::META_LAYOUT );
	}

	/**
	 * Outputs an input for choosing a color for the page background.
	 * WP color picker is initialized in editor.js.
	 *
	 * @param WP_Post $post Post.
	 */
	public function output_page_background_html( $post ) {
		$value = (string) get_post_meta( $post->ID, static::META_PAGE_BGR, true );
		$id    = esc_attr( static::META_PAGE_BGR );
		// Css for the 'default' message.
		$indicator_css = '';
		// Css for custom color.
		$preview_css = '';
		if ( $value ) {
			$indicator_css = 'display : none';
			$preview_css   = 'background-color: ' . $value;
		} else {
			$indicator_css = 'display : inline-block';
			$preview_css   = 'display : none';
		}
		?>
		<div class="components-panel__row">
			<span class="components-base-control__label">
				<?php echo esc_html_x( 'Page background', 'Color palette', 'mateo' ); ?>
			</span>
			<span
				id="mateo_page_bgr_indicator"
				style="<?php echo esc_attr( $indicator_css ); ?>">
				<?php esc_html_e( '(default)', 'mateo' ); ?>
			</span>
			<span
				id="mateo_page_bgr_preview"
				class="component-color-indicator"
				style="<?php echo esc_attr( $preview_css ); ?>"
				aria-label="<?php esc_attr_e( 'Color', 'mateo' ); ?>">
			</span>
		</div>
		<div class="components-panel__row">
			<div class="components-panel__row">
				<div class="components-base-control">
					<div class="components-base-control__field">
						<input
							class="components-text-control__input mateo_pagebgr_control"
							autocomplete="off"
							type="text"
							name="<?php echo $id; /* phpcs:ignore */ ?>" value="<?php echo esc_attr( $value ); ?>">
					</div>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Saves post's background color option in database.
	 *
	 * @param int $post_id Post ID.
	 */
	public function save_page_background_option( $post_id ) {
		$value = sanitize_hex_color( filter_input( INPUT_POST, static::META_PAGE_BGR, FILTER_SANITIZE_STRING ) );
		if ( ! $value ) {
			delete_post_meta( $post_id, static::META_PAGE_BGR );
		} else {
			update_post_meta( $post_id, static::META_PAGE_BGR, $value );
		}
	}
}
