<?php
/**
 * Theme extension of WP Core Nav Menu API.
 *
 * @package mateo
 */

// phpcs:ignore
require_once ABSPATH . 'wp-includes/class-walker-nav-menu.php';

/**
 * Theme class used to implement an HTML list of nav menu items.
 *
 * @see Walker_Nav_Menu
 */
class Mateo_Walker_Nav_Menu extends Walker_Nav_Menu {

	/**
	 * Mateo_Walker_Nav_Menu instance.
	 *
	 * @var Mateo_Walker_Nav_Menu
	 */
	protected static $instance;
	/**
	 * CSS class applied to a submenu.
	 *
	 * @var array
	 */
	protected $sub_menu_classes = array( 'sub-menu' );
	/**
	 * CSS class applied to a menu button/link .
	 *
	 * @var array
	 */
	protected $trigger_classes = array( 'menu-trigger' );
	/**
	 * CSS class for a menu state indicator (array down/ arrow up) .
	 *
	 * @var array
	 */
	protected $trigger_content_classes = array( 'menu-arrow-down' );

	/**
	 * Singleton pattern.
	 *
	 * @return Mateo_Walker_Nav_Menu
	 */
	public static function get_instance() {
		if ( ! static::$instance instanceof Mateo_Walker_Nav_Menu ) {
			static::$instance = new static();
		}
		return static::$instance;
	}
	/**
	 * Starts the list before the elements are added.
	 *
	 * @see Walker::start_lvl()
	 *
	 * @param string         $output Used to append additional content (passed by reference).
	 * @param int            $depth  Depth of menu item. Used for padding.
	 * @param stdClass|array $args   An object of wp_nav_menu() arguments.
	 */
	public function start_lvl( &$output, $depth = 0, $args = array() ) {
		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent  = str_repeat( $t, $depth );
		$classes = $this->sub_menu_classes;
		/**
		 * Filters the CSS class(es) applied to a submenu trigger (button or link).
		 *
		 * @param string[] $classes Array of the CSS classes to apply.
		 * @param stdClass $args An object of `wp_nav_menu()` arguments.
		 * @param int $depth Depth of menu item.
		 */
		$class_trigger = join( ' ', apply_filters( 'mateo_nav_menu_submenu_trigger_class', $this->trigger_classes, $args, $depth ) );
		/**
		 * Filters the CSS class(es) applied to a submenu trigger content (arrow) .
		 *
		 * @param string[] $classes Array of the CSS classes to apply.
		 * @param stdClass $args An object of `wp_nav_menu()` arguments.
		 * @param int $depth Depth of menu item.
		 */
		// Menu arrow classes.
		$class_arrow = join( ' ', apply_filters( 'mateo_nav_menu_submenu_trigger_content_class', $this->trigger_content_classes, $args, $depth ) );
		$output     .= sprintf(
			'<a href="#" class="%s" role="button" aria-haspopup="menu" aria-expanded="false" aria-label="%s"><span>',
			$class_trigger,
			esc_attr__( 'Show menu', 'mateo' )
		);
		$output     .= sprintf( '<i class="%s" aria-hidden="true">', $class_arrow );
		$output     .= '</i></span></a>';
		/**
		 * Filters the CSS class(es) applied to a submenu.
		 *
		 * @param string[] $classes Array of the CSS classes that are applied to the submenu element.
		 * @param stdClass $args An object of `wp_nav_menu()` arguments.
		 * @param int $depth Depth of menu item.
		 */
		$class_names = join( ' ', apply_filters( 'mateo_nav_menu_submenu_css_class', $classes, $args, $depth ) );
		$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';
		$output     .= "{$n}{$indent}<ul$class_names>{$n}";
	}
}
