<?php
/**
 * Class Fonts
 *
 * @package Mateo
 */

namespace Mateo;

/**
 * Class Fonts
 *
 * @package Mateo
 */
class Fonts {

	const INHERIT_OFF            = 'inherit_off';
	const FAMILY_APPEND_CATEGORY = 'FAMILY_APPEND_CATEGORY';
	const FAMILY_MONOSPACE       = 'monospace';
	/**
	 * Google fonts.
	 *
	 * @var array
	 */
	protected $fonts = array();
	/**
	 * Google fonts API URI.
	 *
	 * @var string
	 */
	protected $fonts_api = 'https://fonts.googleapis.com/css?';
	/**
	 * Current font key.
	 *
	 * @var string
	 */
	protected $current_key = '';

	/**
	 * Associative array of fonts keys and font family names.
	 * Fonts to load.
	 *
	 * @var array
	 */
	protected $font_family = array();
	/**
	 * Associative array of fonts keys and font variants.
	 *
	 * @var array
	 */
	protected $font_variants = array();
	/**
	 * Default variants used in CSS: light (300), normal (regular) and bold (700).
	 *
	 * @var string[]
	 */
	protected $default_variants = array( 'regular', '300', '700' );
	/**
	 * Associative array with a font key as key and font name as a value.
	 *
	 * @var array
	 */
	protected $fonts_options = array();
	/**
	 * Associative array of all fonts variants having font variant as a key and a value.
	 *
	 * @var array
	 */
	protected $variants_options = array();
	/**
	 * Associative array with css rules and their default values.
	 *
	 * @var string[]
	 */
	protected $css_rules = array(
		'fontSize'            => 16,
		'fontFamily'          => 'Roboto',
		'fontCategory'        => 'sans-serif',
		'fontStyle'           => 'normal',
		'fontWeight'          => 400,
		'lineHeight'          => 1.5,
		'fontSizeHeaders'     => 1,
		'fontFamilyHeaders'   => 'inherit',
		'fontStyleHeaders'    => 'normal',
		'fontWeightHeaders'   => 400,
		'lineHeightHeaders'   => 1.2,
		'fontSizeTitle'       => 4,
		'fontFamilyTitle'     => 'inherit',
		'fontStyleTitle'      => 'inherit',
		'fontWeightTitle'     => 200,
		'lineHeightTitle'     => 1.2,
		'fontSizeMonospace'   => 0.87,
		'fontFamilyMonospace' => 'inherit',
		'fontStyleMonospace'  => 'normal',
		'fontWeightMonospace' => 400,
		'lineHeightMonospace' => 1.5,
	);

	/**
	 * Instance of the class.
	 *
	 * @var null|static
	 */
	private static $instance = null;

	/**
	 * Fonts constructor.
	 */
	public function __construct() {
		$this->fonts_options    = array( '' => esc_html__( 'Inherit', 'mateo' ) );
		$this->variants_options = array( '' => esc_html__( 'Default', 'mateo' ) );
	}

	/**
	 * Returns existing instance of class or a new instance.
	 * Singleton pattern.
	 *
	 * @return Fonts
	 */
	public static function get_instance() {
		if ( null === static::$instance ) {
			static::$instance = new static();
		}
		return static::$instance;
	}

	/**
	 * Verifies if array of fonts are initialized.
	 *
	 * @return bool TRUE if fonts are initialized FALSE if not.
	 */
	public function has_fonts() {
		return ! empty( $this->fonts );
	}
	/**
	 * Sets collection of fonts.
	 *
	 * @param array $fonts Associative array with fonts.
	 */
	public function set_fonts( $fonts ) {
		$this->fonts = $fonts;
	}

	/**
	 * Initializes options used in customizer controls.
	 */
	public function init_options() {
		$options = array();
		foreach ( $this->fonts as $key => $family ) {
			$this->fonts_options[ $key ] = sprintf( '%s (%s)', $family['name'], $family['categoryGoogle'] );
			// Append variants if they not exist.
			$options += array_combine( $family['variants'], $family['variants'] );
		}
		asort( $options, SORT_NATURAL );
		$this->variants_options += $options;
	}

	/**
	 * Verifies if fonts and variants options are initialized.
	 *
	 * @return bool TRUE if options are initialized FALSE if they are empty.
	 */
	public function is_option_ready() {
		return count( $this->fonts_options ) > 1 && count( $this->variants_options ) > 1;
	}

	/**
	 * Returns fonts to display in "option" tag.
	 *
	 * @param string $option If equals to Fonts::INHERIT_OFF the "Inherit" entry will be removed from the resulting array.
	 * @return array Fonts options.
	 */
	public function get_fonts_options( $option = '' ) {
		if ( static::INHERIT_OFF === $option ) {
			return array_diff_key( $this->fonts_options, array( '' => '' ) );
		}
		if ( static::FAMILY_MONOSPACE === $option ) {
			// Get monospace fonts.
			$fonts_mono = array_filter(
				$this->fonts,
				function ( $font ) {
					return static::FAMILY_MONOSPACE === $font['category'];
				}
			);
			// Append default 'Inherit' option.
			$fonts_mono[''] = array( '' => esc_html__( 'Inherit', 'mateo' ) );
			// Get monospace options.
			return array_intersect_key( $this->fonts_options, $fonts_mono );
		}
		return $this->fonts_options;
	}

	/**
	 * Returns fonts variants to display in "option" tag.
	 *
	 * @return array Font variants.
	 */
	public function get_variants_options() {
		return $this->variants_options;
	}

	/**
	 * Checks if some font exists in an array of fonts.
	 *
	 * @param string $key Font key.
	 *
	 * @return bool TRUE if font exists in the array of fonts FALSE if not.
	 */
	public function exists( $key ) {
		return isset( $this->fonts[ $key ] );
	}

	/**
	 * Queue a font family to the list of fonts.
	 *
	 * @param string $key Font key.
	 *
	 * @return bool|\WP_Error TRUE if font is added and instance of WP_Error if font is not found.
	 */
	public function add_font( $key ) {
		if ( ! $this->exists( $key ) ) {
			return new \WP_Error( 'font_not_found', __( 'Font key is not found', 'mateo' ) );
		}
		if ( ! isset( $this->font_family[ $key ] ) ) {
			$this->current_key           = $key;
			$this->font_family[ $key ]   = $this->fonts[ $key ]['name'];
			$this->font_variants[ $key ] = array();
		}

		return true;
	}

	/**
	 * Add font variant(s) to the current font.
	 *
	 * @param array $variants Array of variants.
	 *
	 * @return int Number of font variants.
	 */
	public function add_font_variants( $variants = array() ) {

		if ( ! $this->current_key ) {
			return 0;
		}
		if ( empty( $variants ) ) {
			$variants = $this->get_default_variants();
		}

		$new_variants                              = array_merge( $this->font_variants[ $this->current_key ], $variants );
		$this->font_variants[ $this->current_key ] = array_unique( $new_variants );

		return count( $this->font_variants[ $this->current_key ] );
	}

	/**
	 * Tries to get default variants from the fonts variants.
	 *
	 * @return string[] Array of font variants.
	 */
	public function get_default_variants() {
		$variants = array_intersect( $this->default_variants, $this->fonts[ $this->current_key ]['variants'] );
		// If a font has some weired variants not matching default variants take the 1st available.
		if ( empty( $variants ) ) {
			$variants = $this->fonts[ $this->current_key ]['variants'][0];
		}
		return $variants;
	}

	/**
	 * Generates a Google Fonts API url for queued fonts.
	 * If no fonts are set up it returns a link without fonts.
	 *
	 * @return string Google font API URL.
	 */
	public function get_google_api_link() {
		if ( empty( $this->font_family ) ) {
			return $this->fonts_api;
		}
		$fonts = array();
		foreach ( $this->font_family as $key => $family ) {
			$font = $family;
			if ( ! empty( $this->font_variants[ $key ] ) ) {
				$variants = implode( ',', $this->font_variants[ $key ] );
				$font    .= ':' . $variants;
			}
			$fonts[] = $font;
		}
		// Update (may be) resulting array with a filter.
		$result_fonts = apply_filters( 'mateo_custom_google_fonts', $fonts );

		return $this->fonts_api . 'family=' . implode( '|', $result_fonts );
	}

	/**
	 * Gets font family name to use in CSS.
	 *
	 * @param string $key  Font key.
	 * @param string $option Pass Fonts::FAMILY_APPEND_CATEGORY constant to append a generic font family to the selected font family.
	 * @return string|null Name or NULL if font is not found.
	 */
	public function get_family_name( $key, $option = '' ) {
		if ( ! $this->exists( $key ) ) {
			return null;
		}
		$result = '"' . $this->fonts[ $key ]['name'] . '"';

		if ( static::FAMILY_APPEND_CATEGORY === $option ) {
			$result .= ', ' . $this->fonts[ $key ]['category'];
		}
		return $result;
	}

	/**
	 * Gets font category (generic font family: serif, sans-serif, etc).
	 *
	 * @param string $key Font key.
	 *
	 * @return string|null Generic font category or NULL if font is not found.
	 */
	public function get_category( $key ) {
		return $this->exists( $key ) ? $this->fonts[ $key ]['category'] : null;
	}

	/**
	 * Analyses if passed as a parameter variant is italic font style.
	 *
	 * @param string $variant Font variant.
	 *
	 * @return bool TRUE if fnt is italic FALSE otherwise.
	 */
	public function is_italic( $variant ) {
		return strpos( $variant, 'ital' ) !== false;
	}

	/**
	 * Gets font weight value from the font variant.
	 *
	 * @param string $variant Font variant.
	 *
	 * @return int Font weight size.
	 */
	public function get_font_weight( $variant ) {
		return (int) $variant;
	}
	/**
	 * Setter for the Fonts::css_rules array.
	 *
	 * @param string $rule Rule in camel case format (used in underscore template).
	 * @param mixed  $value CSS value.
	 */
	public function add_css( $rule, $value ) {
		switch ( $rule ) {
			case 'fontWeight':
				$this->css_rules['fontWeight']        = $value;
				$this->css_rules['fontWeightHeaders'] = $value;
				break;
			case 'fontStyle':
				$this->css_rules['fontStyle']        = $value;
				$this->css_rules['fontStyleHeaders'] = $value;
				break;
			case 'fontFamilyHeaders':
				$this->css_rules['fontFamilyHeaders'] = $value;
				$this->css_rules['fontFamilyTitle']   = $value;
				break;
			case 'lineHeightHeaders':
				$this->css_rules['lineHeightHeaders'] = $value;
				$this->css_rules['lineHeightTitle']   = $value;
				break;
			default:
				$this->css_rules[ $rule ] = $value;
				break;
		}
	}

	/**
	 * Getter for css_rules array.
	 *
	 * @return string[] Array.
	 */
	public function get_css_rules() {
		return $this->css_rules;
	}

	/**
	 * Gets CSS rule setting.
	 *
	 * @param string $key CSS rule in camel case.
	 * @param bool   $default Value to return if rule is not found. FALSE by default.
	 *
	 * @return bool|string Rule value if rule exists or default value.
	 */
	public function get_css_rule( $key, $default = false ) {
		return isset( $this->css_rules[ $key ] ) ? $this->css_rules[ $key ] : $default;
	}

	/**
	 * Underscore template with CSS rules.
	 *
	 * @return string Underscore template.
	 */
	public function get_css_template() {
		return <<<EOD
		:root {
			--mateo-headers-font-size-base: {{data.fontSizeHeaders}}rem;
			--mateo-page-title-size: {{data.fontSizeTitle}}rem;
			--mateo-page-title-font-family:  {{data.fontFamilyTitle}};
		    --mateo-page-title-font-weight:  {{data.fontWeightTitle}};
		    --mateo-page-title-font-style:  {{data.fontStyleTitle}};
		    --mateo-page-title-line-height: {{data.lineHeightTitle}};
		    --mateo-monospace-font-family:  {{data.fontFamilyMonospace}};
			--mateo-monospace-font-size: {{data.fontSizeMonospace}}rem;
			--mateo-monospace-font-weight: {{data.fontWeightMonospace}};
			--mateo-monospace-font-style: {{data.fontStyleMonospace}};
			--mateo-monospace-line-height: {{data.lineHeightMonospace}};
		}
		html {
			font-size: {{data.fontSize}}px;
		}
		body {
			font-family: {{data.fontFamily}}, {{data.fontCategory}};
			font-size: {{data.fontSize}}px ;
			font-weight: {{data.fontWeight}};
			font-style: {{data.fontStyle}};
			line-height: {{data.lineHeight}};
		}
		h1,h2, h3, h4, h5, h6 {
			font-family: {{data.fontFamilyHeaders}};
			font-weight: {{data.fontWeightHeaders}};
			font-style: {{data.fontStyleHeaders}};
			line-height: {{data.lineHeightHeaders}};
		}

EOD;

	}

	/**
	 * Replaces placeholders by values in the CSS template.
	 *
	 * @return string CSS rules.
	 */
	public function get_parsed_template() {
		if ( ! empty( $this->css_rules ) ) {
			$template = $this->get_css_template();
			foreach ( $this->css_rules as $key => $value ) {
				$template = str_replace( '{{data.' . $key . '}}', $value, $template );
			}
			return $template;
		}

		return '';
	}
}
