<?php
/**
 * Mateo Theme functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 * @package mateo
 */

// phpcs:ignore
require_once __DIR__ . '/vendor/autoload.php';

define( 'MATEO_THEME_ABSPATH', __DIR__ );
define( 'MATEO_VERSION', '2.5.2');
define( 'MATEO_MAX_CONTENT_WIDTH', 960 );
define( 'MATEO_CSS_STRIP_EOL', true );
// Handler name for the frontend css.
define( 'MATEO_STYLE_ID', 'mateo-style' );
// Handler name for the customizer css.
define( 'MATEO_CSS_CUSTOMIZE_HANDLER', 'mateo-css-customize' );
// Handler name for the customizer js.
define( 'MATEO_JS_CUSTOMIZE_HANDLER', 'mateo-js-customize' );
// Handlers for editor.
define( 'MATEO_CSS_EDITOR_HANDLER', 'mateo-css-editor' );
define( 'MATEO_JS_EDITOR_HANDLER', 'mateo-js-editor' );
// Theme settings.
define( 'MATEO_SETTING_LAYOUT_ID', 'site_layout' );
define( 'MATEO_LAYOUT_1', '1-column' );
define( 'MATEO_LAYOUT_2', '2-column' );
define( 'MATEO_CSS_FONT_HANDLER', 'mateo-fonts' );
define( 'MATEO_CSS_CUSTOM_FONT_HANDLER', 'mateo-fonts-custom' );
// Total width of left and right gutters in pixels.
define( 'MATEO_CSS_GUTTERS', 30 );

require __DIR__ . '/inc/common.php';

/**
 * Custom template tags for this theme.
 */
require __DIR__ . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require __DIR__ . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require __DIR__ . '/inc/customizer.php';


if ( ! function_exists( 'mateo_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function mateo_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on mateo, use a find and replace
		 * to change 'mateo' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'mateo', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		set_post_thumbnail_size( 825, 300 );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus(
			array(
				/* translators: primary menu */
				'menu-1' => esc_html__( 'Primary', 'mateo' ),
			)
		);

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support(
			'html5',
			array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
			)
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support(
			'custom-logo',
			array(
				'height'      => 64,
				'width'       => 64,
				'flex-width'  => true,
				'flex-height' => true,
			)
		);
		$custom_header_args = apply_filters(
			'mateo_custom_header_args',
			array(
				'default-image'      => get_template_directory_uri() . '/assets/img/header.jpg',
				'header_text'        => false,
				'default-text-color' => 'fff',
				'width'              => 1920,
				'height'             => 1080,
				'flex-height'        => true,
				'flex-width'         => true,
			)
		);
		add_theme_support( 'custom-header', $custom_header_args );

		add_theme_support(
			'custom-background',
			array(
				'default-image'          => '',
				'default-preset'         => 'fill',
				'default-position-x'     => 'left',
				'default-position-y'     => 'left',
				'default-size'           => 'cover',
				'default-repeat'         => 'no-repeat',
				'default-attachment'     => 'scroll',
				'default-color'          => '',
				'wp-head-callback'       => '_custom_background_cb',
				'admin-head-callback'    => '',
				'admin-preview-callback' => '',
			)
		);

		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );

		add_theme_support( 'align-wide' );

		// Block editor styles.
		add_theme_support( 'editor-styles' );
		add_editor_style( 'assets/css/editor-styles.css' );

		// Adds theme colors to the editor color palette.
		add_theme_support( 'editor-color-palette', mateo_get_editor_colors() );
		// This one is required for embedded content sizing.
		$GLOBALS['content_width'] = mateo_get_display_width(); // phpcs:ignore WPThemeReview.CoreFunctionality.PrefixAllGlobals.NonPrefixedVariableFound
	}
endif;
add_action( 'after_setup_theme', 'mateo_setup' );

/**
 * Add some required settings before other theme scripts are loaded.
 *
 * @return string
 */
function mateo_js_config() {
	$text = array(
		'show'       => esc_attr__( 'Show menu', 'mateo' ),
		'hide'       => esc_attr__( 'Hide menu', 'mateo' ),
		'showParent' => esc_attr__( 'Show parent menu', 'mateo' ),
		'previous'   => esc_attr__( 'Previous', 'mateo' ),
		'next'       => esc_attr__( 'Next', 'mateo' ),
	);

	return 'window.Mateo = {}; Mateo.labels = ' . wp_json_encode( $text ) . ';';
}


/**
 * Enqueue frontend assets.
 */
function mateo_scripts() {
	$assets_root = get_stylesheet_directory_uri() . '/assets';
	// CSS.
	$custom_fonts = mateo_custom_fonts();
	if ( ! empty( $custom_fonts ) ) {
		$font_link = $custom_fonts['link'] . '|Material+Icons';
	} else {
		$font_link = 'https://fonts.googleapis.com/css?family=Roboto:300,400,500,700|Material+Icons';
	}
	wp_enqueue_style( MATEO_CSS_FONT_HANDLER, $font_link, array() ); // phpcs:ignore
	// Initialize a css handler. Inline styles will be appended to the handler by mateo_color_scheme_css() function.
	wp_enqueue_style( MATEO_STYLE_ID, get_stylesheet_uri(), array(), MATEO_VERSION );

	wp_style_add_data( MATEO_STYLE_ID, 'rtl', 'replace' );
	// JS.
	wp_enqueue_script( 'jquery-core', false, array(), MATEO_VERSION, true );
	wp_add_inline_script( 'jquery-core', mateo_js_config() );
	wp_enqueue_script(
		'mateo-custom',
		$assets_root . '/js/' . mateo_get_js_filename( 'theme' ),
		array( 'jquery-core' ),
		MATEO_VERSION,
		true
	);

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}

add_action( 'wp_enqueue_scripts', 'mateo_scripts' );


/**
 * Add inline front-end customized CSS settings.
 */
function mateo_frontend_inline_custom_css() {
	wp_add_inline_style( MATEO_STYLE_ID, mateo_custom_css() );
}

add_action( 'wp_enqueue_scripts', 'mateo_frontend_inline_custom_css' );

/**
 * Register block editor assets.
 */
function mateo_block_editor_assets() {
	wp_enqueue_style( MATEO_CSS_EDITOR_HANDLER, 'https://fonts.googleapis.com/css?family=Roboto:300,400,500,700', array('wp-color-picker') ); // phpcs:ignore
	wp_enqueue_script(
		MATEO_JS_EDITOR_HANDLER,
		get_stylesheet_directory_uri() . '/assets/js/' . mateo_get_js_filename( 'editor' ),
		array( 'jquery-core', 'wp-color-picker' ),
		MATEO_VERSION,
		true
	);
	wp_localize_script(
		MATEO_JS_EDITOR_HANDLER,
		'Mateo',
		array(
			'Layout'            => mateo_get_theme_layout(),
			'ItemLayout'        => mateo_get_post_layout(),
			'HasHeaderImg'      => has_header_image(),
			'FeaturedImgHeader' => mateo_is_active_featured_in_header(),
			'HasFeaturedImg'    => has_post_thumbnail(),
		)
	);
	$custom_fonts = mateo_custom_fonts();
	if ( $custom_fonts ) {
		wp_enqueue_style( MATEO_CSS_CUSTOM_FONT_HANDLER, $custom_fonts['link'], array(), MATEO_VERSION );
	}
}
add_action( 'enqueue_block_editor_assets', 'mateo_block_editor_assets' );

/**
 * Filter function to append user custom CSS to the very end of editor styles to override default WP CSS settings.
 *
 * @param array $editor_settings    Array with editor settings.
 * @return array Editor settings
 */
function mateo_block_editor_styles_filter( $editor_settings ) {
	$editor_settings['styles'][] = array( 'css' => mateo_editor_custom_css() );

	return $editor_settings;
}
add_filter( 'block_editor_settings', 'mateo_block_editor_styles_filter', 10 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function mateo_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'mateo' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here.', 'mateo' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);

	register_sidebar(
		array(
			/* translators: Footer widgets area */
			'name'          => esc_html__( 'Primary Footer', 'mateo' ),
			'id'            => 'footer-1',
			'description'   => esc_html__( 'Add widgets here.', 'mateo' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);

	register_sidebar(
		array(
			/* translators: Centered footer widgets */
			'name'          => esc_html__( 'Secondary Footer', 'mateo' ),
			'id'            => 'footer-2',
			'description'   => esc_html__( 'Add widgets here.', 'mateo' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		)
	);
}

add_action( 'widgets_init', 'mateo_widgets_init' );
