<?php
/**
* MagPal Theme Customizer.
*
* @package MagPal WordPress Theme
* @copyright Copyright (C) 2025 ThemesDNA
* @license http://www.gnu.org/licenses/gpl-2.0.html GNU/GPLv2 or later
* @author ThemesDNA <themesdna@gmail.com>
*/

if ( ! class_exists( 'WP_Customize_Control' ) ) {return null;}

/**
* MagPal_Customize_Static_Text_Control class
*/

class MagPal_Customize_Static_Text_Control extends WP_Customize_Control {
    public $type = 'magpal-static-text';

    public function __construct( $manager, $id, $args = [] ) {
        parent::__construct( $manager, $id, $args );
    }

    protected function render_content() {
        if ( empty( $this->label ) && empty( $this->description ) ) {
            return; // Prevent rendering if both label and description are empty
        }

        if ( ! empty( $this->label ) ) {
            echo '<span class="customize-control-title">' . esc_html( $this->label ) . '</span>';
        }

        if ( ! empty( $this->description ) ) {
            echo '<div class="description customize-control-description">' . wp_kses_post( $this->description ) . '</div>';
        }
    }
}

/**
* MagPal_Customize_Button_Control class
*/

class MagPal_Customize_Button_Control extends WP_Customize_Control {
    public $type = 'magpal-button';
    protected $button_tag = 'button';
    protected $button_class = 'button button-primary';
    protected $button_href = 'javascript:void(0)';
    protected $button_target = '';
    protected $button_onclick = '';
    protected $button_tag_id = '';

    public function render_content() {
        if ( empty( $this->label ) ) {
            return; // Prevent rendering an empty button
        }
        ?>
        <span class="center">
            <?php
            echo '<' . esc_html( $this->button_tag );

            if ( ! empty( $this->button_class ) ) {
                echo ' class="' . esc_attr( $this->button_class ) . '"';
            }

            if ( $this->button_tag === 'button' ) {
                echo ' type="button"';
            } else {
                echo ' href="' . esc_url( $this->button_href ) . '"';
                if ( ! empty( $this->button_target ) ) {
                    echo ' target="' . esc_attr( $this->button_target ) . '"';
                }
            }

            if ( ! empty( $this->button_onclick ) ) {
                echo ' onclick="' . esc_js( $this->button_onclick ) . '"';
            }

            if ( ! empty( $this->button_tag_id ) ) {
                echo ' id="' . esc_attr( $this->button_tag_id ) . '"';
            }

            echo '>';
            echo esc_html( $this->label );
            echo '</' . esc_html( $this->button_tag ) . '>';
            ?>
        </span>
        <?php
    }
}

/**
* Magpal_Toggle_Switch_Control class
*/

if ( class_exists( 'WP_Customize_Control' ) ) {
    /**
     * Toggle Switch Custom Control Class
     */
    class Magpal_Toggle_Switch_Control extends WP_Customize_Control {
        /**
         * The type of control being rendered
         */
        public $type = 'magpal-toggle-switch';

        /**
         * On/Off labels
         *
         * @var string
         */
        public $on_text;
        public $off_text;

        /**
         * Constructor
         *
         * @param WP_Customize_Manager $manager Customize manager object.
         * @param string               $id      Control ID.
         * @param array                $args    Arguments to override class property defaults.
         */
        public function __construct( $manager, $id, $args = array() ) {
            parent::__construct( $manager, $id, $args );

            // Set the on/off texts, translating them here.
            $this->on_text  = isset( $args['on_text'] ) ? esc_html( $args['on_text'] ) : esc_html__( 'ON', 'magpal' );
            $this->off_text = isset( $args['off_text'] ) ? esc_html( $args['off_text'] ) : esc_html__( 'OFF', 'magpal' );
        }

        /**
         * Enqueue scripts and styles.
         */
        public function enqueue() {
            wp_enqueue_style( 'magpal-toggle-switch-css', get_template_directory_uri() . '/css/magpal-toggle-switch.css', array(), '1.0' );
        }

        /**
         * Render the control's content.
         */
        public function render_content() {
            ?>
            <div class="customize-control magpal-customize-control-toggle-switch">
                <?php if ( ! empty( $this->label ) ) : ?>
                    <span class="customize-control-title"><?php echo esc_html( $this->label ); ?></span>
                <?php endif; ?>

                <?php if ( ! empty( $this->description ) ) : ?>
                    <span class="description customize-control-description"><?php echo wp_kses_post( $this->description ); ?></span>
                <?php endif; ?>

                <div class="magpal-toggle-switch-wrapper">
                    <input type="checkbox" id="<?php echo esc_attr( $this->id ); ?>" class="magpal-toggle-switch-input" <?php $this->link(); ?> <?php checked( $this->value() ); ?> aria-label="<?php echo esc_attr( $this->label ); ?>" />
                    <label class="magpal-toggle-switch-slider" for="<?php echo esc_attr( $this->id ); ?>">
                        <span class="magpal-toggle-switch-on"><?php echo $this->on_text; ?></span>
                        <span class="magpal-toggle-switch-off"><?php echo $this->off_text; ?></span>
                    </label>
                </div>
            </div>
            <?php
        }

        /**
         * Empty content template to prevent default rendering.
         */
        public function content_template() {}
    }
}

/**
* Sanitization callback functions
*
* These functions are used to sanitize and validate user inputs in the WordPress Customizer.
* They ensure that only safe and valid values are stored in the database.
*/

function magpal_sanitize_checkbox( $input ) {
    return ( ( isset( $input ) && ( true == $input ) ) ? true : false );
}

function magpal_sanitize_html( $input ) {
    return wp_kses_post( force_balance_tags( $input ) );
}

function magpal_sanitize_date_type( $input, $setting ) {
    $valid = array('published','updated');
    return in_array( $input, $valid, true ) ? $input : $setting->default;
}

function magpal_sanitize_yes_no( $input, $setting ) {
    $valid = array('yes','no');
    return in_array( $input, $valid, true ) ? $input : $setting->default;
}

function magpal_sanitize_thumbnail_alignment( $input, $setting ) {
    $valid = array('magpal-thumbnail-aligncenter','magpal-thumbnail-alignwide','magpal-thumbnail-alignfull');
    return in_array( $input, $valid, true ) ? $input : $setting->default;
}

function magpal_sanitize_posts_navigation_type( $input, $setting ) {
    $valid = array('normalnavi','numberednavi');
    return in_array( $input, $valid, true ) ? $input : $setting->default;
}

function magpal_sanitize_email( $input, $setting ) {
    return ( ! empty( $input ) && is_email( $input ) ) ? sanitize_email( $input ) : $setting->default;
}

function magpal_sanitize_logo_location( $input, $setting ) {
    $valid = array('beside-title','above-title');
    return in_array( $input, $valid, true ) ? $input : $setting->default;
}

function magpal_sanitize_positive_integer( $input, $setting ) {
    $input = absint( $input ); // Force the value into non-negative integer.
    return ( $input > 0 ) ? $input : $setting->default;
}

function magpal_sanitize_positive_float( $input, $setting ) {
    $input = (float) $input;
    return ( $input > 0 ) ? $input : $setting->default;
}

function magpal_sanitize_snippet_type( $input, $setting ) {
    $valid = array('postexcerpt','postcontent');
    return in_array( $input, $valid, true ) ? $input : $setting->default;
}

function magpal_register_theme_customizer( $wp_customize ) {

    if ( method_exists( 'WP_Customize_Manager', 'add_panel' ) ) {
        $wp_customize->add_panel( 'magpal_main_options_panel', array(
            'title'    => esc_html__( 'Theme Options', 'magpal' ),
            'priority' => 10,
        ) );
    }

    // Assign sections to the 'magpal_main_options_panel' and set priorities
    $wp_customize->get_section( 'title_tagline' )->panel = 'magpal_main_options_panel';
    $wp_customize->get_section( 'title_tagline' )->priority = 20;
    $wp_customize->get_section( 'header_image' )->panel = 'magpal_main_options_panel';
    $wp_customize->get_section( 'header_image' )->priority = 26;
    $wp_customize->get_section( 'background_image' )->panel = 'magpal_main_options_panel';
    $wp_customize->get_section( 'background_image' )->priority = 27;
    $wp_customize->get_section( 'colors' )->panel = 'magpal_main_options_panel';
    $wp_customize->get_section( 'colors' )->priority = 40;

    // Enable live preview (postMessage transport) for site title, description, header text color, and background color
    $wp_customize->get_setting( 'blogname' )->transport = 'postMessage';
    $wp_customize->get_setting( 'blogdescription' )->transport = 'postMessage';
    $wp_customize->get_setting( 'header_textcolor' )->transport = 'postMessage';
    $wp_customize->get_setting( 'background_color' )->transport = 'postMessage';

    // Add a description for the background color control, instructing users to remove the background image first
    $wp_customize->get_control( 'background_color' )->description = esc_html__('To change Background Color, need to remove background image first:- go to Appearance : Customize : Theme Options : Background Image', 'magpal');

    // Check if selective refresh is available in the Customizer
    if ( isset( $wp_customize->selective_refresh ) ) {

        // Add selective refresh support for the site title
        $wp_customize->selective_refresh->add_partial( 'blogname', array(
            'selector'        => '.magpal-site-title a',
            'render_callback' => 'magpal_customize_partial_blogname',
        ) );

        // Add selective refresh support for the site description (tagline)
        $wp_customize->selective_refresh->add_partial( 'blogdescription', array(
            'selector'        => '.magpal-site-description',
            'render_callback' => 'magpal_customize_partial_blogdescription',
        ) );
    }

    /* Getting started options */
    $wp_customize->add_section( 'magpal_section_getting_started', array( 'title' => esc_html__( 'Getting Started', 'magpal' ), 'description' => esc_html__( 'Thanks for your interest in MagPal! If you have any questions or run into any trouble, please visit us the following links. We will get you fixed up!', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 5, ) );

    $wp_customize->add_setting( 'magpal_options[documentation]', array( 'default' => '', 'sanitize_callback' => '__return_false', ) );

    $wp_customize->add_control( new MagPal_Customize_Button_Control( $wp_customize, 'magpal_documentation_control', array( 'label' => esc_html__( 'Documentation', 'magpal' ), 'section' => 'magpal_section_getting_started', 'settings' => 'magpal_options[documentation]', 'type' => 'magpal-button', 'button_tag' => 'a', 'button_class' => 'button button-primary', 'button_href' => esc_url( 'https://themesdna.com/magpal-wordpress-theme/' ), 'button_target' => '_blank', ) ) );

    $wp_customize->add_setting( 'magpal_options[contact]', array( 'default' => '', 'sanitize_callback' => '__return_false', ) );

    $wp_customize->add_control( new MagPal_Customize_Button_Control( $wp_customize, 'magpal_contact_control', array( 'label' => esc_html__( 'Contact Us', 'magpal' ), 'section' => 'magpal_section_getting_started', 'settings' => 'magpal_options[contact]', 'type' => 'magpal-button', 'button_tag' => 'a', 'button_class' => 'button button-primary', 'button_href' => esc_url( 'https://themesdna.com/contact/' ), 'button_target' => '_blank', ) ) );


    /* Color options */
   $wp_customize->add_setting( 'magpal_options[header_text_hover_color]', array( 'default' => '#dddddd', 'type' => 'option', 'transport' => 'postMessage', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_hex_color' ) );

    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'magpal_header_text_hover_color_control', array( 'label' => esc_html__( 'Header Hover Text Color', 'magpal' ), 'section' => 'colors', 'settings' => 'magpal_options[header_text_hover_color]' ) ) );

    $wp_customize->add_setting( 'magpal_options[grid_posts_bg_color]', array( 'default' => '#efebe9', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_hex_color' ) );

    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'magpal_grid_posts_bg_color_control', array( 'label' => esc_html__( 'Grid Posts Area Background Color', 'magpal' ), 'description' => esc_html__( 'Note: As the background color of grid posts areas, use this background color option instead of the default "Background Color" option.', 'magpal' ), 'section' => 'colors', 'settings' => 'magpal_options[grid_posts_bg_color]' ) ) );

    $wp_customize->add_setting( 'magpal_options[disable_grid_posts_bg_color]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_grid_posts_bg_color_control', array( 'label' => esc_html__( 'Disable Grid Posts Area Background Color', 'magpal' ), 'description' => esc_html__( 'Checking this option disables "Grid Posts Area Background Color", making grid post areas use the default "Background Color" instead.', 'magpal' ), 'section' => 'colors', 'settings' => 'magpal_options[disable_grid_posts_bg_color]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Menu options */
    $wp_customize->add_section( 'magpal_section_menu_options', array( 'title' => esc_html__( 'Menu Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 100 ) );

    $wp_customize->add_setting( 'magpal_options[primary_menu_text]', array( 'default' => esc_html__( 'Menu', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field', ) );

    $wp_customize->add_control( 'magpal_primary_menu_text_control', array( 'label' => esc_html__( 'Primary Menu Mobile Text', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[primary_menu_text]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'magpal_options[disable_primary_menu]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_primary_menu_control', array( 'label' => esc_html__( 'Disable Primary Menu', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[disable_primary_menu]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_header_search_button]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_header_search_button_control', array( 'label' => esc_html__( 'Hide Header Search Button', 'magpal' ), 'description' => esc_html__( 'This option has no effect if you checked the option: "Disable Primary Menu"', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[hide_header_search_button]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[disable_responsive_primary_menu]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_responsive_primary_menu_control', array( 'label' => esc_html__( 'Disable Responsiveness(Mobile View) from Primary Menu', 'magpal' ), 'description' => 'Prevent the primary menu from switching to a mobile-friendly layout, forcing it to display in its desktop format on all screen sizes.', 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[disable_responsive_primary_menu]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[secondary_menu_text]', array( 'default' => esc_html__( 'Menu', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field', ) );

    $wp_customize->add_control( 'magpal_secondary_menu_text_control', array( 'label' => esc_html__( 'Secondary Menu Mobile Text', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[secondary_menu_text]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'magpal_options[disable_secondary_menu]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_secondary_menu_control', array( 'label' => esc_html__( 'Disable Secondary Menu', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[disable_secondary_menu]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[disable_responsive_secondary_menu]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_responsive_secondary_menu_control', array( 'label' => esc_html__( 'Disable Responsiveness(Mobile View) from Secondary Menu', 'magpal' ), 'description' => esc_html__( 'Prevent the secondary menu from switching to a mobile-friendly layout, forcing it to display in its desktop format on all screen sizes.', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[disable_responsive_secondary_menu]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[no_center_secondary_menu]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_no_center_secondary_menu_control', array( 'label' => esc_html__( 'Do not Center Secondary Menu', 'magpal' ), 'section' => 'magpal_section_menu_options', 'settings' => 'magpal_options[no_center_secondary_menu]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Header options */
    $wp_customize->add_section( 'magpal_section_header', array( 'title' => esc_html__( 'Header Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 120 ) );

    $wp_customize->add_setting( 'magpal_options[hide_tagline]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_tagline_control', array( 'label' => esc_html__( 'Hide Tagline', 'magpal' ), 'section' => 'magpal_section_header', 'settings' => 'magpal_options[hide_tagline]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[enable_sticky_header]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_enable_sticky_header_control', array( 'label' => esc_html__( 'Enable Sticky Feature on Header', 'magpal' ), 'description' => esc_html__( 'Turn off the sticky header feature, making it scroll normally with the page instead of staying fixed at the top.', 'magpal' ), 'section' => 'magpal_section_header', 'settings' => 'magpal_options[enable_sticky_header]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[enable_sticky_mobile_header]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_enable_sticky_mobile_header_control', array( 'label' => esc_html__( 'Enable Sticky Feature on Header on Small Screen', 'magpal' ), 'description' => esc_html__( 'Keep the sticky header feature on smaller screens for easy access while scrolling. This option has no effect until you checked the option: "Enable Sticky Feature on Header"', 'magpal' ), 'section' => 'magpal_section_header', 'settings' => 'magpal_options[enable_sticky_mobile_header]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_header_content]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_header_content_control', array( 'label' => esc_html__( 'Hide Header Content', 'magpal' ), 'section' => 'magpal_section_header', 'settings' => 'magpal_options[hide_header_content]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[logo_location]', array( 'default' => 'above-title', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_logo_location' ) );

    $wp_customize->add_control( 'magpal_logo_location_control', array( 'label' => esc_html__( 'Logo Location', 'magpal' ), 'description' => esc_html__('Select how you want to display the site logo with site title and tagline.', 'magpal'), 'section' => 'title_tagline', 'settings' => 'magpal_options[logo_location]', 'type' => 'select', 'choices' => array( 'beside-title' => esc_html__( 'Before Site Title and Tagline', 'magpal' ), 'above-title' => esc_html__( 'Above Site Title and Tagline', 'magpal' ) ), 'priority'   => 8 ) );

    $wp_customize->add_setting( 'magpal_options[limit_logo_width]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_limit_logo_width_control', array( 'label' => esc_html__( 'Limit the Logo Width', 'magpal' ), 'section' => 'title_tagline', 'settings' => 'magpal_options[limit_logo_width]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), 'priority'   => 8 ) ) );

    $wp_customize->add_setting( 'magpal_options[logo_max_width]', array( 'default' => '300px', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field' ) );

    $wp_customize->add_control( 'magpal_logo_max_width_control', array( 'label' => esc_html__( 'Max Width of the Logo', 'magpal' ), 'description' => __('Enter the max-width value for the logo. Default value is "300px". Note: This option has no effect until you check "Limit the Logo Width" option.', 'magpal'), 'section' => 'title_tagline', 'settings' => 'magpal_options[logo_max_width]', 'type' => 'text', 'priority'   => 8 ) );

    $wp_customize->add_setting( 'magpal_options[hide_header_image]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_header_image_control', array( 'label' => esc_html__( 'Hide Header Image from Everywhere', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[hide_header_image]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_header_image_details]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_header_image_details_control', array( 'label' => esc_html__( 'Hide both Title and Description from Header Image', 'magpal' ), 'description' => esc_html__( 'Remove the title and description overlay from the header image for a cleaner look.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[hide_header_image_details]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_header_image_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_header_image_title_control', array( 'label' => esc_html__( 'Hide Title from Header Image', 'magpal' ), 'description' => esc_html__( 'Hide header image title from all screen sizes. This option has no effect if you have checked the option: "Hide both Title and Description from Header Image"', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[hide_header_image_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_header_image_description]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_header_image_description_control', array( 'label' => esc_html__( 'Hide Description from Header Image', 'magpal' ), 'description' => esc_html__( 'Hide header image description from all screen sizes. This option has no effect if you have checked the option: "Hide both Title and Description from Header Image"', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[hide_header_image_description]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_title_mobile]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_header_image_title_mobile_control', array( 'label' => esc_html__( 'Show Header Image Title on Smaller Screens', 'magpal' ), 'description' => esc_html__( 'This option has no effect if you have checked any of these options: "Hide both Title and Description from Header Image", "Hide Title from Header Image"', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_title_mobile]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_description_mobile]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_header_image_description_mobile_control', array( 'label' => esc_html__( 'Show Header Image Description on Smaller Screens', 'magpal' ), 'description' => esc_html__( 'This option has no effect if you have checked any of these options: "Hide both Title and Description from Header Image", "Hide Description from Header Image"', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_description_mobile]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_custom_text]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_header_image_custom_text_control', array( 'label' => esc_html__( 'Add Custom Title/Custom Description to Header Image', 'magpal' ), 'description' => esc_html__( 'Display a custom title and description on the header image instead of the default site title and tagline.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_custom_text]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_custom_title]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_header_image_custom_title_control', array( 'label' => esc_html__( 'Header Image Custom Title', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_custom_title]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'magpal_options[header_image_custom_description]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_header_image_custom_description_control', array( 'label' => esc_html__( 'Header Image Custom Description', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_custom_description]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'magpal_options[header_image_destination]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_header_image_destination_control', array( 'label' => esc_html__( 'Header Image Destination URL', 'magpal' ), 'description' => esc_html__( 'Enter the URL a visitor should go when he/she click on the header image. If you did not enter a URL below, header image will be linked to the homepage of your website.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_destination]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[remove_header_image_destination]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_remove_header_image_destination_control', array( 'label' => esc_html__( 'Remove Destination URL from Header Image', 'magpal' ), 'description' => esc_html__( 'Disable the clickable link on the header image, making it non-interactive.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[remove_header_image_destination]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_destination_target]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_header_image_destination_target_control', array( 'label' => esc_html__( 'Open Header Image Destination URL in a New Window/Tab', 'magpal' ), 'description' => esc_html__( 'Enable this option to open the header image destination URL in a new browser tab or window when clicked.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_destination_target]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_destination_rel]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_header_image_destination_rel_control', array( 'label' => esc_html__( 'Add rel="nofollow" attribute to Header Image Destination URL', 'magpal' ), 'description' => esc_html__( 'Adds the rel="nofollow" attribute to the header image link, preventing search engines from passing link equity.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_destination_rel]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_cover]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_header_image_cover_control', array( 'label' => esc_html__( 'Add a Minimum Height to Header Image on Smaller Screens', 'magpal' ), 'description' => esc_html__( 'Set a minimum height for the header image on smaller screens to maintain proper visibility.', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_cover]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[header_image_height_mobile]', array( 'default' => '200px', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field' ) );

    $wp_customize->add_control( 'magpal_header_image_height_mobile_control', array( 'label' => esc_html__( 'Header Image Minimum Height on Smaller Screens', 'magpal' ), 'description' => __('Enter the minimum height value for header image height. Default is "200px". Note: This option has no effect until you check "Add a Minimum Height to Header Image on Smaller Screens" option.', 'magpal'), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_height_mobile]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[header_padding]', array( 'default' => '10px', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field' ) );

    $wp_customize->add_control( 'magpal_header_padding_control', array( 'label' => esc_html__( 'Header Padding', 'magpal' ), 'description' => __('Enter the padding value for header content. Default is "10px".', 'magpal'), 'section' => 'magpal_section_header', 'settings' => 'magpal_options[header_padding]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[header_image_alt_text]', array( 'default' => get_bloginfo( 'name' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field', ) );

    $wp_customize->add_control( 'magpal_header_image_alt_text_control', array( 'label' => esc_html__( 'Header Image Alt Text', 'magpal' ), 'section' => 'header_image', 'settings' => 'magpal_options[header_image_alt_text]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'magpal_options[search_box_placeholder_text]', array( 'default' => esc_html__( 'Enter your search keyword...', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field', ) );

    $wp_customize->add_control( 'magpal_search_box_placeholder_text_control', array( 'label' => esc_html__( 'Search Box Placeholder Text', 'magpal' ), 'description' => esc_html__( 'Set a custom placeholder text for the search box to guide users on what to search for.', 'magpal' ), 'section' => 'magpal_section_header', 'settings' => 'magpal_options[search_box_placeholder_text]', 'type' => 'text', ) );


    /* Posts Grid options */
    $wp_customize->add_section( 'magpal_section_posts_grid', array( 'title' => esc_html__( 'Posts Grid Options', 'magpal' ), 'description' => esc_html__('To display your latest posts as a grid on your homepage, please set the "Your homepage displays" option to "Your latest posts." You can find this setting in your WordPress Dashboard by navigating to "Settings" -> "Reading" -> "Your homepage displays."', 'magpal'), 'panel' => 'magpal_main_options_panel', 'priority' => 160 ) );

    $wp_customize->add_setting( 'magpal_options[hide_posts_heading]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_posts_heading_control', array( 'label' => esc_html__( 'Hide HomePage Posts Heading', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_posts_heading]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[posts_heading]', array( 'default' => esc_html__( 'Recent Posts', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field', ) );

    $wp_customize->add_control( 'magpal_posts_heading_control', array( 'label' => esc_html__( 'HomePage Posts Heading', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[posts_heading]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'posts_per_page', array( 'default' => get_option('posts_per_page'), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_positive_integer' ) );

    $wp_customize->add_control( 'magpal_posts_per_page_control', array( 'label' => esc_html__( 'Number of Posts per Page', 'magpal' ), 'description' => esc_html__('Set the maximum number of posts displayed on each blog page.', 'magpal'), 'section' => 'magpal_section_posts_grid', 'settings' => 'posts_per_page', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_thumbnail]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_thumbnail_control', array( 'label' => esc_html__( 'Hide Featured Images from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_thumbnail]', 'on_text'  => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_default_grid_thumbnail]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_default_grid_thumbnail_control', array( 'label' => esc_html__( 'Hide Default Images from Posts Grid', 'magpal' ), 'description' => esc_html__( 'The default image is shown when there is no featured image.', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_default_grid_thumbnail]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_post_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_post_title_control', array( 'label' => esc_html__( 'Hide Post Titles from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_post_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[remove_grid_post_title_link]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_remove_grid_post_title_link_control', array( 'label' => esc_html__( 'Remove Links of Post Titles from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[remove_grid_post_title_link]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[limit_grid_post_titles]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_limit_grid_post_titles_control', array( 'label' => esc_html__( 'Limit the Length of Post Titles in Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[limit_grid_post_titles]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[grid_post_title_length]', array( 'default' => 6, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_positive_integer' ) );

    $wp_customize->add_control( 'magpal_grid_post_title_length_control', array( 'label' => esc_html__( 'Maximum Length of Post Titles in Posts Grid', 'magpal' ), 'description' => esc_html__('Enter the maximum number of words need to display in grid post titles. Default is 6 words. This option only work if you have check the option "Limit the Length of Post Titles in Grid Posts Summaries".', 'magpal'), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[grid_post_title_length]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_post_snippet]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_post_snippet_control', array( 'label' => esc_html__( 'Hide Post Snippets from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_post_snippet]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[grid_post_snippet_type]', array( 'default' => 'postexcerpt', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_snippet_type' ) );

    $wp_customize->add_control( 'magpal_grid_post_snippet_type_control', array( 'label' => esc_html__( 'Post Snippets Type of Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[grid_post_snippet_type]', 'type' => 'select', 'choices' => array( 'postexcerpt' => esc_html__('Post Excerpt', 'magpal'), 'postcontent' => esc_html__('Post Content', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[grid_post_snippet_length]', array( 'default' => 17, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_positive_float' ) );

    $wp_customize->add_control( 'magpal_grid_post_snippet_length_control', array( 'label' => esc_html__( 'Post Snippets Length of Posts Grid', 'magpal' ), 'description' => esc_html__('Enter the number of words need to display in the post summary. Default is 17 words.', 'magpal'), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[grid_post_snippet_length]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_post_date]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_post_date_control', array( 'label' => esc_html__( 'Hide Post Dates from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_post_date]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[grid_post_date_type]', array( 'default' => 'published', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_date_type' ) );

    $wp_customize->add_control( 'magpal_grid_post_date_type_control', array( 'label' => esc_html__( 'Post Date Type for Posts Grid', 'magpal' ), 'description' => esc_html__('You can choose to display published or updated date of your post.', 'magpal'), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[grid_post_date_type]', 'type' => 'select', 'choices' => array( 'published' => esc_html__('Published', 'magpal'), 'updated' => esc_html__('Updated', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_post_author]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_post_author_control', array( 'label' => esc_html__( 'Hide Post Authors from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_post_author]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_comments_link]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_comments_link_control', array( 'label' => esc_html__( 'Hide Comment Links from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_comments_link]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_grid_post_categories]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_grid_post_categories_control', array( 'label' => esc_html__( 'Hide Post Categories from Posts Grid', 'magpal' ), 'section' => 'magpal_section_posts_grid', 'settings' => 'magpal_options[hide_grid_post_categories]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Post options */
    $wp_customize->add_section( 'magpal_section_post', array( 'title' => esc_html__( 'Post Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 180 ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_thumbnail]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_thumbnail_control', array( 'label' => esc_html__( 'Hide Featured Image from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_thumbnail]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[thumbnail_alignment_single]', array( 'default' => 'magpal-thumbnail-alignwide', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_thumbnail_alignment' ) );

    $wp_customize->add_control( 'magpal_thumbnail_alignment_single_control', array( 'label' => esc_html__( 'Featured Image Alignment', 'magpal' ), 'description' => esc_html__('Select the alignment of featured image in full post.', 'magpal'), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[thumbnail_alignment_single]', 'type' => 'select', 'choices' => array( 'magpal-thumbnail-aligncenter' => esc_html__('Center', 'magpal'), 'magpal-thumbnail-alignwide' => esc_html__('Wide', 'magpal'), 'magpal-thumbnail-alignfull' => esc_html__('Full', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[single_thumbnail_link]', array( 'default' => 'yes', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_yes_no' ) );

    $wp_customize->add_control( 'magpal_single_thumbnail_link_control', array( 'label' => esc_html__( 'Featured Image Link', 'magpal' ), 'description' => esc_html__('Do you want the featured image in a single post to be linked to its post?', 'magpal'), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[single_thumbnail_link]', 'type' => 'select', 'choices' => array( 'yes' => esc_html__('Yes', 'magpal'), 'no' => esc_html__('No', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[featured_media_under_post_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_featured_media_under_post_title_control', array( 'label' => esc_html__( 'Move Featured Image to Bottom of Full Post Title', 'magpal' ), 'description' => esc_html__( 'Display the featured image beneath the post title instead of above it.', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[featured_media_under_post_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_post_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_post_title_control', array( 'label' => esc_html__( 'Hide Post Header from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_post_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[remove_single_post_title_link]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_remove_single_post_title_link_control', array( 'label' => esc_html__( 'Remove Link from Full Post Title', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[remove_single_post_title_link]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_post_date]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_post_date_control', array( 'label' => esc_html__( 'Hide Post Date from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_post_date]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[single_post_date_type]', array( 'default' => 'published', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_date_type' ) );

    $wp_customize->add_control( 'magpal_single_post_date_type_control', array( 'label' => esc_html__( 'Post Date Type for Full Post', 'magpal' ), 'description' => esc_html__('You can choose to display published or updated date of your post.', 'magpal'), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[single_post_date_type]', 'type' => 'select', 'choices' => array( 'published' => esc_html__('Published', 'magpal'), 'updated' => esc_html__('Updated', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_post_author]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_post_author_control', array( 'label' => esc_html__( 'Hide Post Author from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_post_author]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_comments_link]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_comments_link_control', array( 'label' => esc_html__( 'Hide Comment Link from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_comments_link]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_comment_form]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_comment_form_control', array( 'label' => esc_html__( 'Hide Comments/Comment Form from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_comment_form]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_post_categories]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_post_categories_control', array( 'label' => esc_html__( 'Hide Post Categories from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_post_categories]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_post_tags]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_post_tags_control', array( 'label' => esc_html__( 'Hide Post Tags from Full Post', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_post_tags]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_breadcrumbs]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_breadcrumbs_control', array( 'label' => __( 'Hide Breadcrumbs from Full Post', 'magpal' ), 'description' => __( 'Note: You need to install and activate "Yoast SEO" plugin to display breadcrumbs.', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_breadcrumbs]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_author_bio_box]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_author_bio_box_control', array( 'label' => esc_html__( 'Hide Author Bio Box', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_author_bio_box]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_single_post_edit]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_single_post_edit_control', array( 'label' => esc_html__( 'Hide Post Edit Link', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[hide_single_post_edit]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[no_underline_content_links]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_no_underline_content_links_control', array( 'label' => esc_html__( 'Do not Underline the Links within the Content', 'magpal' ), 'section' => 'magpal_section_post', 'settings' => 'magpal_options[no_underline_content_links]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Navigation options */
    $wp_customize->add_section( 'magpal_section_navigation', array( 'title' => esc_html__( 'Post/Posts Navigation Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 185 ) );

    $wp_customize->add_setting( 'magpal_options[hide_post_navigation]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_post_navigation_control', array( 'label' => esc_html__( 'Hide Post Navigation from Full Posts', 'magpal' ), 'section' => 'magpal_section_navigation', 'settings' => 'magpal_options[hide_post_navigation]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_posts_navigation]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_posts_navigation_control', array( 'label' => esc_html__( 'Hide Posts Navigation from Home/Archive/Search Pages', 'magpal' ), 'section' => 'magpal_section_navigation', 'settings' => 'magpal_options[hide_posts_navigation]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[posts_navigation_type]', array( 'default' => 'numberednavi', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_posts_navigation_type' ) );

    $wp_customize->add_control( 'magpal_posts_navigation_type_control', array( 'label' => esc_html__( 'Posts Navigation Type', 'magpal' ), 'description' => esc_html__('Select posts navigation type you need. If you activate WP-PageNavi plugin, this navigation will be replaced by WP-PageNavi navigation.', 'magpal'), 'section' => 'magpal_section_navigation', 'settings' => 'magpal_options[posts_navigation_type]', 'type' => 'select', 'choices' => array( 'normalnavi' => esc_html__('Normal Navigation', 'magpal'), 'numberednavi' => esc_html__('Numbered Navigation', 'magpal') ) ) );


    /* Page options */
    $wp_customize->add_section( 'magpal_section_page', array( 'title' => esc_html__( 'Page Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 190 ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_thumbnail]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_thumbnail_control', array( 'label' => esc_html__( 'Hide Featured Image from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_thumbnail]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[thumbnail_alignment_page]', array( 'default' => 'magpal-thumbnail-alignwide', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_thumbnail_alignment' ) );

    $wp_customize->add_control( 'magpal_thumbnail_alignment_page_control', array( 'label' => esc_html__( 'Featured Image Alignment', 'magpal' ), 'description' => esc_html__('Select the alignment of featured image in single page.', 'magpal'), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[thumbnail_alignment_page]', 'type' => 'select', 'choices' => array( 'magpal-thumbnail-aligncenter' => esc_html__('Center', 'magpal'), 'magpal-thumbnail-alignwide' => esc_html__('Wide', 'magpal'), 'magpal-thumbnail-alignfull' => esc_html__('Full', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[page_thumbnail_link]', array( 'default' => 'yes', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_yes_no' ) );

    $wp_customize->add_control( 'magpal_page_thumbnail_link_control', array( 'label' => esc_html__( 'Featured Image Link', 'magpal' ), 'description' => esc_html__('Do you want the featured image in a page to be linked to its page?', 'magpal'), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[page_thumbnail_link]', 'type' => 'select', 'choices' => array( 'yes' => esc_html__('Yes', 'magpal'), 'no' => esc_html__('No', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[featured_media_under_page_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_featured_media_under_page_title_control', array( 'label' => esc_html__( 'Move Featured Image to Bottom of Page Title', 'magpal' ), 'description' => esc_html__( 'Display the featured image beneath the page title instead of above it.', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[featured_media_under_page_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_title_control', array( 'label' => esc_html__( 'Hide Page Header from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[remove_page_title_link]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_remove_page_title_link_control', array( 'label' => esc_html__( 'Remove Link from Single Page Title', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[remove_page_title_link]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_date]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_date_control', array( 'label' => esc_html__( 'Hide Posted Date from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_date]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[page_date_type]', array( 'default' => 'published', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_date_type' ) );

    $wp_customize->add_control( 'magpal_page_date_type_control', array( 'label' => esc_html__( 'Page Date Type', 'magpal' ), 'description' => esc_html__('You can choose to display published or updated date of your page.', 'magpal'), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[page_date_type]', 'type' => 'select', 'choices' => array( 'published' => esc_html__('Published', 'magpal'), 'updated' => esc_html__('Updated', 'magpal') ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_author]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_author_control', array( 'label' => esc_html__( 'Hide Page Author from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_author]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_comments_link]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_comments_link_control', array( 'label' => esc_html__( 'Hide Comment Link from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_comments_link]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_comment_form]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_comment_form_control', array( 'label' => esc_html__( 'Hide Comments/Comment Form from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_comment_form]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_page_edit]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_page_edit_control', array( 'label' => esc_html__( 'Hide Edit Link from Single Page', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_page_edit]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_static_page_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_static_page_title_control', array( 'label' => esc_html__( 'Hide Page Header from Static Homepage', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_static_page_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_static_page_meta]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_static_page_meta_control', array( 'label' => esc_html__( 'Hide Page Meta Data (Page Date, Page Author, Comment Link) from Static Homepage', 'magpal' ), 'section' => 'magpal_section_page', 'settings' => 'magpal_options[hide_static_page_meta]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Social profiles options */
    $wp_customize->add_section( 'magpal_section_social', array( 'title' => esc_html__( 'Social Links Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 240, ));

    $wp_customize->add_setting( 'magpal_options[hide_social_buttons]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_social_buttons_control', array( 'label' => esc_html__( 'Hide Footer Social + Login/Logout Buttons', 'magpal' ), 'description' => esc_html__( 'If you checked this option, all buttons will disappear from footer. There is no any effect from these option: "Show Login/Logout Button in Footer".', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[hide_social_buttons]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[show_footer_login_button]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_show_footer_login_button_control', array( 'label' => esc_html__( 'Show Login/Logout Button in Footer', 'magpal' ), 'description' => esc_html__( 'This option has no effect if you checked the option: "Hide Footer Social + Login/Logout Buttons"', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[show_footer_login_button]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[twitterlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_twitterlink_control', array( 'label' => esc_html__( 'X URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[twitterlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[facebooklink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_facebooklink_control', array( 'label' => esc_html__( 'Facebook URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[facebooklink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[threadslink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_threadslink_control', array( 'label' => esc_html__( 'Threads URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[threadslink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[pinterestlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_pinterestlink_control', array( 'label' => esc_html__( 'Pinterest URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[pinterestlink]', 'type' => 'text' ) );
    
    $wp_customize->add_setting( 'magpal_options[linkedinlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_linkedinlink_control', array( 'label' => esc_html__( 'Linkedin Link', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[linkedinlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[instagramlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_instagramlink_control', array( 'label' => esc_html__( 'Instagram Link', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[instagramlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[vklink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_vklink_control', array( 'label' => esc_html__( 'VK Link', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[vklink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[flickrlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_flickrlink_control', array( 'label' => esc_html__( 'Flickr Link', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[flickrlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[youtubelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_youtubelink_control', array( 'label' => esc_html__( 'Youtube URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[youtubelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[vimeolink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_vimeolink_control', array( 'label' => esc_html__( 'Vimeo URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[vimeolink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[soundcloudlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_soundcloudlink_control', array( 'label' => esc_html__( 'Soundcloud URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[soundcloudlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[messengerlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_messengerlink_control', array( 'label' => esc_html__( 'Messenger URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[messengerlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[whatsapplink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_whatsapplink_control', array( 'label' => esc_html__( 'WhatsApp URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[whatsapplink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[tiktoklink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_tiktoklink_control', array( 'label' => esc_html__( 'TikTok URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[tiktoklink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[lastfmlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_lastfmlink_control', array( 'label' => esc_html__( 'Lastfm URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[lastfmlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[mediumlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_mediumlink_control', array( 'label' => esc_html__( 'Medium URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[mediumlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[githublink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_githublink_control', array( 'label' => esc_html__( 'Github URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[githublink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[bitbucketlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_bitbucketlink_control', array( 'label' => esc_html__( 'Bitbucket URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[bitbucketlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[tumblrlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_tumblrlink_control', array( 'label' => esc_html__( 'Tumblr URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[tumblrlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[digglink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_digglink_control', array( 'label' => esc_html__( 'Digg URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[digglink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[deliciouslink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_deliciouslink_control', array( 'label' => esc_html__( 'Delicious URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[deliciouslink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[stumblelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_stumblelink_control', array( 'label' => esc_html__( 'Stumbleupon URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[stumblelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[mixlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_mixlink_control', array( 'label' => esc_html__( 'Mix URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[mixlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[redditlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_redditlink_control', array( 'label' => esc_html__( 'Reddit URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[redditlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[dribbblelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_dribbblelink_control', array( 'label' => esc_html__( 'Dribbble URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[dribbblelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[flipboardlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_flipboardlink_control', array( 'label' => esc_html__( 'Flipboard URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[flipboardlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[bloggerlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_bloggerlink_control', array( 'label' => esc_html__( 'Blogger URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[bloggerlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[etsylink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_etsylink_control', array( 'label' => esc_html__( 'Etsy URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[etsylink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[behancelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_behancelink_control', array( 'label' => esc_html__( 'Behance URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[behancelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[amazonlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_amazonlink_control', array( 'label' => esc_html__( 'Amazon URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[amazonlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[meetuplink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_meetuplink_control', array( 'label' => esc_html__( 'Meetup URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[meetuplink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[mixcloudlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_mixcloudlink_control', array( 'label' => esc_html__( 'Mixcloud URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[mixcloudlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[slacklink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_slacklink_control', array( 'label' => esc_html__( 'Slack URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[slacklink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[snapchatlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_snapchatlink_control', array( 'label' => esc_html__( 'Snapchat URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[snapchatlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[spotifylink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_spotifylink_control', array( 'label' => esc_html__( 'Spotify URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[spotifylink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[yelplink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_yelplink_control', array( 'label' => esc_html__( 'Yelp URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[yelplink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[wordpresslink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_wordpresslink_control', array( 'label' => esc_html__( 'WordPress URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[wordpresslink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[twitchlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_twitchlink_control', array( 'label' => esc_html__( 'Twitch URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[twitchlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[telegramlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_telegramlink_control', array( 'label' => esc_html__( 'Telegram URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[telegramlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[bandcamplink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_bandcamplink_control', array( 'label' => esc_html__( 'Bandcamp URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[bandcamplink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[quoralink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_quoralink_control', array( 'label' => esc_html__( 'Quora URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[quoralink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[foursquarelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_foursquarelink_control', array( 'label' => esc_html__( 'Foursquare URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[foursquarelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[deviantartlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_deviantartlink_control', array( 'label' => esc_html__( 'DeviantArt URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[deviantartlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[imdblink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_imdblink_control', array( 'label' => esc_html__( 'IMDB URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[imdblink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[codepenlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_codepenlink_control', array( 'label' => esc_html__( 'Codepen URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[codepenlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[jsfiddlelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_jsfiddlelink_control', array( 'label' => esc_html__( 'JSFiddle URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[jsfiddlelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[stackoverflowlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_stackoverflowlink_control', array( 'label' => esc_html__( 'Stack Overflow URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[stackoverflowlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[stackexchangelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_stackexchangelink_control', array( 'label' => esc_html__( 'Stack Exchange URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[stackexchangelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[bsalink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_bsalink_control', array( 'label' => esc_html__( 'BuySellAds URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[bsalink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[web500pxlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_web500pxlink_control', array( 'label' => esc_html__( '500px URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[web500pxlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[ellolink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_ellolink_control', array( 'label' => esc_html__( 'Ello URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[ellolink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[discordlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_discordlink_control', array( 'label' => esc_html__( 'Discord URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[discordlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[goodreadslink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_goodreadslink_control', array( 'label' => esc_html__( 'Goodreads URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[goodreadslink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[odnoklassnikilink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_odnoklassnikilink_control', array( 'label' => esc_html__( 'Odnoklassniki URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[odnoklassnikilink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[houzzlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_houzzlink_control', array( 'label' => esc_html__( 'Houzz URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[houzzlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[pocketlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_pocketlink_control', array( 'label' => esc_html__( 'Pocket URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[pocketlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[xinglink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_xinglink_control', array( 'label' => esc_html__( 'XING URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[xinglink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[mastodonlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_mastodonlink_control', array( 'label' => esc_html__( 'Mastodon URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[mastodonlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[blueskylink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_blueskylink_control', array( 'label' => esc_html__( 'Bluesky URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[blueskylink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[googleplaylink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_googleplaylink_control', array( 'label' => esc_html__( 'Google Play URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[googleplaylink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[slidesharelink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_slidesharelink_control', array( 'label' => esc_html__( 'SlideShare URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[slidesharelink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[dropboxlink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_dropboxlink_control', array( 'label' => esc_html__( 'Dropbox URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[dropboxlink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[paypallink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_paypallink_control', array( 'label' => esc_html__( 'PayPal URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[paypallink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[viadeolink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_viadeolink_control', array( 'label' => esc_html__( 'Viadeo URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[viadeolink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[wikipedialink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_wikipedialink_control', array( 'label' => esc_html__( 'Wikipedia URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[wikipedialink]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[skypeusername]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'sanitize_text_field' ) );

    $wp_customize->add_control( 'magpal_skypeusername_control', array( 'label' => esc_html__( 'Skype Username', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[skypeusername]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[emailaddress]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_email' ) );

    $wp_customize->add_control( 'magpal_emailaddress_control', array( 'label' => esc_html__( 'Email Address', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[emailaddress]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[rsslink]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'esc_url_raw' ) );

    $wp_customize->add_control( 'magpal_rsslink_control', array( 'label' => esc_html__( 'RSS Feed URL', 'magpal' ), 'section' => 'magpal_section_social', 'settings' => 'magpal_options[rsslink]', 'type' => 'text' ) );


    /* Sidebar options */
    $wp_customize->add_section( 'magpal_section_sidebar', array( 'title' => esc_html__( 'Sidebar Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 270 ) );

    $wp_customize->add_setting( 'magpal_options[disable_sidebar]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_sidebar_control', array( 'label' => esc_html__( 'Disable Off-Canvas Sidebar', 'magpal' ), 'section' => 'magpal_section_sidebar', 'settings' => 'magpal_options[disable_sidebar]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[sidebar_width]', array( 'default' => 350, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_positive_integer' ) );

    $wp_customize->add_control( 'magpal_sidebar_width_control', array( 'label' => esc_html__( 'Width of Off-Canvas Sidebar', 'magpal' ), 'description' => esc_html__('Enter the numerical pixel width value of the off-canvas sidebar. Default is 350.', 'magpal'), 'section' => 'magpal_section_sidebar', 'settings' => 'magpal_options[sidebar_width]', 'type' => 'text' ) );


    /* Footer options */
    $wp_customize->add_section( 'magpal_section_footer', array( 'title' => esc_html__( 'Footer Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 280 ) );

    $wp_customize->add_setting( 'magpal_options[footer_text]', array( 'default' => '', 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_footer_text_control', array( 'label' => esc_html__( 'Footer copyright notice', 'magpal' ), 'section' => 'magpal_section_footer', 'settings' => 'magpal_options[footer_text]', 'type' => 'text', ) );

    $wp_customize->add_setting( 'magpal_options[hide_footer_widgets]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_footer_widgets_control', array( 'label' => esc_html__( 'Hide Footer Widgets', 'magpal' ), 'description' => esc_html__( 'When you enable this option, it will hide your footer widgets across the entire site.', 'magpal' ), 'section' => 'magpal_section_footer', 'settings' => 'magpal_options[hide_footer_widgets]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[disable_backtotop]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_backtotop_control', array( 'label' => esc_html__( 'Disable Back to Top Button', 'magpal' ), 'section' => 'magpal_section_footer', 'settings' => 'magpal_options[disable_backtotop]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Archive pages options */
    $wp_customize->add_section( 'magpal_section_archive_pages', array( 'title' => esc_html__( 'Archive Pages Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 320 ) );

    $wp_customize->add_setting( 'magpal_options[hide_cats_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_cats_title_control', array( 'label' => esc_html__( 'Hide Category Title from Category Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_cats_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_cats_description]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_cats_description_control', array( 'label' => esc_html__( 'Hide Category Description from Category Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_cats_description]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_category_word]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_category_word_control', array( 'label' => esc_html__( 'Remove Word "Category:" from Category Archive Pages Titles', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_category_word]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_tags_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_tags_title_control', array( 'label' => esc_html__( 'Hide Tag Title from Tag Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_tags_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_tags_description]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_tags_description_control', array( 'label' => esc_html__( 'Hide Tag Description from Tag Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_tags_description]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_tag_word]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_tag_word_control', array( 'label' => esc_html__( 'Remove Word "Tag:" from Tag Archive Pages Titles', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_tag_word]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_author_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_author_title_control', array( 'label' => esc_html__( 'Hide Author Title from Author Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_author_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_author_description]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_author_description_control', array( 'label' => esc_html__( 'Hide Author Description from Author Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_author_description]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_author_word]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_author_word_control', array( 'label' => esc_html__( 'Remove Word "Author:" from Author Archive Pages Titles', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_author_word]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    $wp_customize->add_setting( 'magpal_options[hide_date_title]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_date_title_control', array( 'label' => esc_html__( 'Hide Title from Year/Month/Day Archive Pages', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_date_title]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[hide_date_word]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_date_word_control', array( 'label' => esc_html__( 'Remove Words "Year:"/"Month:"/"Day:" from Year/Month/Day Archive Pages Titles', 'magpal' ), 'section' => 'magpal_section_archive_pages', 'settings' => 'magpal_options[hide_date_word]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Search and 404 options */
    $wp_customize->add_section( 'magpal_section_search_404', array( 'title' => esc_html__( 'Search and 404 Pages Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 340 ) );

    $wp_customize->add_setting( 'magpal_options[hide_search_results_heading]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_search_results_heading_control', array( 'label' => esc_html__( 'Hide Search Results Heading', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[hide_search_results_heading]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[search_results_heading]', array( 'default' => esc_html__( 'Search Results for:', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_search_results_heading_control', array( 'label' => esc_html__( 'Search Results Heading', 'magpal' ), 'description' => esc_html__( 'Enter a sentence to display before the search query.', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[search_results_heading]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[no_search_heading]', array( 'default' => esc_html__( 'Nothing Found', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_no_search_heading_control', array( 'label' => esc_html__( 'No Search Results Heading', 'magpal' ), 'description' => esc_html__( 'Enter a heading to display when no search results are found.', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[no_search_heading]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[no_search_results]', array( 'default' => esc_html__( 'Sorry, but nothing matched your search terms. Please try again with some different keywords.', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_no_search_results_control', array( 'label' => esc_html__( 'No Search Results Message', 'magpal' ), 'description' => esc_html__( 'Enter a message to display when no search results are found.', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[no_search_results]', 'type' => 'textarea' ) );

    $wp_customize->add_setting( 'magpal_options[error_404_heading]', array( 'default' => esc_html__( 'Oops! That page can not be found.', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_error_404_heading_control', array( 'label' => esc_html__( '404 Error Page Heading', 'magpal' ), 'description' => esc_html__( 'Enter the heading for the 404 error page.', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[error_404_heading]', 'type' => 'text' ) );

    $wp_customize->add_setting( 'magpal_options[error_404_message]', array( 'default' => esc_html__( 'It looks like nothing was found at this location. Maybe try one of the links below or a search?', 'magpal' ), 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_html', ) );

    $wp_customize->add_control( 'magpal_error_404_message_control', array( 'label' => esc_html__( 'Error 404 Message', 'magpal' ), 'description' => esc_html__( 'Enter a message to display on the 404 error page.', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[error_404_message]', 'type' => 'textarea', ) );

    $wp_customize->add_setting( 'magpal_options[hide_404_search]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_hide_404_search_control', array( 'label' => esc_html__( 'Hide Search Box from 404 Page', 'magpal' ), 'section' => 'magpal_section_search_404', 'settings' => 'magpal_options[hide_404_search]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Other options */
    $wp_customize->add_section( 'magpal_section_other_options', array( 'title' => esc_html__( 'Other Options', 'magpal' ), 'panel' => 'magpal_main_options_panel', 'priority' => 600 ) );

    $wp_customize->add_setting( 'magpal_options[enable_widgets_block_editor]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_enable_widgets_block_editor_control', array( 'label' => esc_html__( 'Enable Gutenberg Widget Block Editor', 'magpal' ), 'section' => 'magpal_section_other_options', 'settings' => 'magpal_options[enable_widgets_block_editor]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[disable_loading_animation]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_loading_animation_control', array( 'label' => esc_html__( 'Disable Site Loading Animation', 'magpal' ), 'description' => esc_html__( 'You can turn off the animation displayed while the site loads, allowing pages to appear instantly.', 'magpal' ), 'section' => 'magpal_section_other_options', 'settings' => 'magpal_options[disable_loading_animation]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );

    $wp_customize->add_setting( 'magpal_options[disable_fitvids]', array( 'default' => false, 'type' => 'option', 'capability' => 'edit_theme_options', 'sanitize_callback' => 'magpal_sanitize_checkbox', ) );

    $wp_customize->add_control( new Magpal_Toggle_Switch_Control( $wp_customize, 'magpal_disable_fitvids_control', array( 'label' => esc_html__( 'Disable FitVids.JS', 'magpal' ), 'description' => esc_html__( 'You can disable fitvids.js script if you are not using videos on your website or if you do not want fluid width videos in your post content.', 'magpal' ), 'section' => 'magpal_section_other_options', 'settings' => 'magpal_options[disable_fitvids]', 'on_text' => esc_html__( 'Yes', 'magpal' ), 'off_text' => esc_html__( 'No', 'magpal' ), ) ) );


    /* Upgrade to pro options */
    $wp_customize->add_section( 'magpal_section_upgrade', array( 'title' => esc_html__( 'Upgrade to Pro Version', 'magpal' ), 'priority' => 400 ) );
    
    $wp_customize->add_setting( 'magpal_options[upgrade_text]', array( 'default' => '', 'sanitize_callback' => '__return_false', ) );
    
    $wp_customize->add_control( new MagPal_Customize_Static_Text_Control( $wp_customize, 'magpal_upgrade_text_control', array(
        'label'       => esc_html__( 'MagPal Pro', 'magpal' ),
        'section'     => 'magpal_section_upgrade',
        'settings' => 'magpal_options[upgrade_text]',
        'description' => esc_html__( 'Do you enjoy MagPal? Upgrade to MagPal Pro now and get:', 'magpal' ).
            '<ul class="magpal-customizer-pro-features">' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Color Options', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Font Options', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( '10 Thumbnail Size Options for Post Grids', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Grid Item Size Options for Post Grids', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Width Options for Different Layout Areas', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( '6 Header Layouts', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Grid Featured Posts Widget (Recent/Categories/Tags/Post IDs based) - with capability to display posts according to Likes/Views/Comments/Dates/... and there are many options', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'List Featured Posts Widget (Recent/Categories/Tags/Post IDs based) - with capability to display posts according to Likes/Views/Comments/Dates/... and there are many options', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Tabbed Featured Posts Widget (Recent/Categories/Tags/Post IDs based) - with capability to display posts according to Likes/Views/Comments/Dates/... and there are many options', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'News Ticker (Recent/Categories/Tags/Post IDs based) - It can display posts according to Likes/Views/Comments/Dates/... and there are many options.', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Advanced Header Image Support – Assign unique images, links, titles, and descriptions for each post or page', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Footer with Layout Options (1/2/3/4/5/6 columns)', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Custom Page Templates', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Settings Panel to Control Options in Each Post', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Settings Panel to Control Options in Each Page', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Posts Likes System', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Posts Views Counter', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Infinite Scroll and Load More Features for Post Grids', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Display Ads/Custom Contents between Posts in the Grid', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'About and Social Widget – Supports 60+ social media buttons', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Post Share Buttons - 25+ Social Networks are Supported', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Related Posts (Categories/Tags/Author/Post IDs based) with Many Options', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Author Bio Box with Social Buttons - 60+ Social Buttons', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Ad Placement Options – Add ads below post titles and within post content', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Disable Google Fonts Option – Optimize for faster loading', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Post Navigation with Thumbnails', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Additional Widget Areas', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Built-in Contact Form', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'WooCommerce Compatibility', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Elementor Pro Support', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Yoast SEO Breadcrumbs Support', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Full RTL Language Support', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Search Engine Optimized (SEO)', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'Extensive Theme Options in the Customizer', 'magpal' ) . '</li>' .
                '<li><i class="fas fa-check" aria-hidden="true"></i> ' . esc_html__( 'And Many More Features…', 'magpal' ) . '</li>' .
            '</ul>'.
            '<strong><a href="'.MAGPAL_PROURL.'" class="button button-primary" target="_blank"><i class="fas fa-shopping-cart" aria-hidden="true"></i> ' . esc_html__( 'Upgrade To MagPal PRO', 'magpal' ) . '</a></strong>'
    ) ) );

}
add_action( 'customize_register', 'magpal_register_theme_customizer' );

/**
 * Render the site title for the selective refresh partial.
 *
 * @return void
 */
function magpal_customize_partial_blogname() {
    bloginfo( 'name' );
}

/**
 * Render the site tagline for the selective refresh partial.
 *
 * @return void
 */
function magpal_customize_partial_blogdescription() {
    bloginfo( 'description' );
}

/**
 * Enqueue customizer JavaScript for live preview.
 */
function magpal_customizer_js_scripts() {
    wp_enqueue_script(
        'magpal-theme-customizer-js',
        get_template_directory_uri() . '/assets/js/customizer.js',
        array( 'jquery', 'customize-preview' ),
        wp_get_theme()->get( 'Version' ),
        true
    );
}
add_action( 'customize_preview_init', 'magpal_customizer_js_scripts' );