<?php
/**
* Theme Functions
*
* @package MagPal WordPress Theme
* @copyright Copyright (C) 2025 ThemesDNA
* @license http://www.gnu.org/licenses/gpl-2.0.html GNU/GPLv2 or later
* @author ThemesDNA <themesdna@gmail.com>
*/

/**
 * Retrieve the value of a given theme option from the database.
 *
 * @since 1.0.0
 *
 * @param string $option Theme option name to retrieve.
 * @param mixed $default Default value to return if the option is not set.
 * @return mixed The value of the theme option, or the default value if not set.
 */
function magpal_get_option($option, $default = '') {
    // Validate input: Ensure the option is a non-empty string
    if (!is_string($option) || trim($option) === '') {
        return $default; // Return the default value for invalid input
    }

    // Retrieve the magpal_options array from the database
    $magpal_options = get_option('magpal_options');

    // Ensure magpal_options is a valid array
    if (!is_array($magpal_options)) {
        return $default; // Return the default value if magpal_options is not an array
    }

    // Return the option value if it exists, otherwise return the default value
    if (array_key_exists($option, $magpal_options)) {
        return $magpal_options[$option];
    }

    return $default;
}

/**
 * Check if a specific theme option is set in the magpal_options array.
 *
 * @param string $option The key of the option to check.
 * @return bool True if the option is set, false otherwise.
 */
function magpal_is_option_set($option) {
    // Validate input: Ensure the option is a non-empty string
    if (!is_string($option) || trim($option) === '') {
        return false;
    }

    // Retrieve the magpal_options array from the database
    $magpal_options = get_option('magpal_options');

    // Return true if the option exists in the array, otherwise false
    return is_array($magpal_options) && array_key_exists($option, $magpal_options);
}

/**
 * Remove a specific option from the magpal_options array.
 *
 * @param string $option The key of the option to remove.
 * @return bool True if the option was removed and updated successfully, false otherwise.
 */
function magpal_remove_option($option) {
    // Validate input: Ensure the option is a non-empty string
    if (!is_string($option) || trim($option) === '') {
        return false;
    }

    // Retrieve the magpal_options array
    $magpal_options = get_option('magpal_options');

    // Ensure magpal_options is a valid array and contains the specified option
    if (is_array($magpal_options) && array_key_exists($option, $magpal_options)) {
        // Remove the specific option
        unset($magpal_options[$option]);

        // Update the options array in the database and return the result
        return (bool) update_option('magpal_options', $magpal_options);
    }

    // Return false if the key does not exist or magpal_options is invalid
    return false;
}

if ( ! function_exists( 'magpal_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function magpal_setup() {
    
    global $wp_version;

    /*
     * Make theme available for translation.
     * Translations can be filed in the /languages/ directory.
     * If you're building a theme based on MagPal, use a find and replace
     * to change 'magpal' to the name of your theme in all the template files.
     */
    load_theme_textdomain( 'magpal', get_template_directory() . '/languages' );

    // Add default posts and comments RSS feed links to head.
    add_theme_support( 'automatic-feed-links' );

    /*
     * Let WordPress manage the document title.
     * By adding theme support, we declare that this theme does not use a
     * hard-coded <title> tag in the document head, and expect WordPress to
     * provide it for us.
     */
    add_theme_support( 'title-tag' );

    /*
     * Enable support for Post Thumbnails on posts and pages.
     *
     * @link http://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
     */
    add_theme_support( 'post-thumbnails' );

    if ( function_exists( 'add_image_size' ) ) {
        add_image_size( 'magpal-1222w-autoh-image', 1222, 9999, false );
        add_image_size( 'magpal-300w-autoh-image', 300, 9999, false );
    }

    // This theme uses wp_nav_menu() in one location.
    register_nav_menus( array(
    'primary' => esc_html__('Primary Menu', 'magpal'),
    'secondary' => esc_html__('Secondary Menu', 'magpal')
    ) );

    /*
     * Switch default core markup for search form, comment form, and comments
     * to output valid HTML5.
     */
    $markup = array( 'comment-list', 'comment-form', 'search-form', 'gallery', 'caption', 'navigation-widgets' );
    add_theme_support( 'html5', $markup );

    /**
     * Enable custom logo support
     *
     * Add support for a custom logo in the theme, allowing users to upload 
     * and set a site logo from the WordPress Customizer. The logo supports flexible width 
     * and height adjustments.
     */
    add_theme_support( 'custom-logo', array(
        'height'      => 37,
        'width'       => 280,
        'flex-height' => true,
        'flex-width'  => true,
        'header-text' => array( 'site-title', 'site-description' ),
    ) );

    // Support for Custom Header
    add_theme_support( 'custom-header', apply_filters( 'magpal_custom_header_args', array(
    'default-image'          => '',
    'default-text-color'     => 'ffffff',
    'width'                  => 1920,
    'height'                 => 400,
    'flex-width'            => true,
    'flex-height'            => true,
    'wp-head-callback'       => 'magpal_header_style',
    'uploads'                => true,
    ) ) );

    // Set up the WordPress core custom background feature.
    $background_args = array(
            'default-color'          => 'ffffff',
            'default-image'          => '',
            'default-repeat'         => 'repeat',
            'default-position-x'     => 'left',
            'default-position-y'     => 'top',
            'default-size'     => 'auto',
            'default-attachment'     => 'fixed',
            'wp-head-callback'       => '_custom_background_cb',
            'admin-head-callback'    => 'admin_head_callback_func',
            'admin-preview-callback' => 'admin_preview_callback_func',
    );
    add_theme_support( 'custom-background', apply_filters( 'magpal_custom_background_args', $background_args) );
    
    // Support for Custom Editor Style
    add_editor_style( 'assets/css/editor-style.css' );

    // Add support for responsive embedded content.
    add_theme_support( 'responsive-embeds' );

    // Enable support for default block styles provided by WordPress.
    // This allows the theme to use the default styling for WordPress core blocks.
    add_theme_support( 'wp-block-styles' );

    // Enable support for wide and full alignment options for blocks.
    // This allows blocks like images and cover blocks to be aligned wide or full width in the theme.
    add_theme_support( 'align-wide' );

    // Enable support for WooCommerce in the theme, allowing it to integrate WooCommerce features and functionality
    add_theme_support( 'woocommerce' );

    // Enable support for the WooCommerce product gallery zoom feature
    add_theme_support( 'wc-product-gallery-zoom' );

    // Enable support for the WooCommerce product gallery lightbox feature
    add_theme_support( 'wc-product-gallery-lightbox' );

    // Enable support for the WooCommerce product gallery slider feature
    add_theme_support( 'wc-product-gallery-slider' );

    // Enable support for Yoast SEO breadcrumbs in the theme, allowing the theme to display breadcrumb navigation generated by Yoast SEO
    add_theme_support( 'yoast-seo-breadcrumbs' );

    // Enable Elementor support for this theme
    add_theme_support( 'elementor' );

    if ( !(magpal_get_option('enable_widgets_block_editor')) ) {
        remove_theme_support( 'widgets-block-editor' );
    }

}
endif;
add_action( 'after_setup_theme', 'magpal_setup' );

/**
 * Register Elementor theme locations.
 *
 * Registers multiple theme locations in Elementor to allow 
 * custom header, footer, single post, and archive templates.
 *
 * @param \Elementor\ThemeManager $elementor_theme_manager Elementor theme manager instance.
 */
function magpal_register_elementor_locations($elementor_theme_manager) {
    $locations = ['header', 'footer', 'single', 'archive'];

    foreach ($locations as $location) {
        $elementor_theme_manager->register_location($location);
    }
}

add_action('elementor/theme/register_locations', 'magpal_register_elementor_locations');

/**
* Layout Functions
*/

function magpal_is_footer_widgets_active() {
    if ( magpal_get_option( 'hide_footer_widgets' ) ) {
        return apply_filters( 'magpal_is_footer_widgets_active', false );
    }

    return apply_filters( 'magpal_is_footer_widgets_active', true );
}

function magpal_is_header_content_active() {
    if ( magpal_get_option( 'hide_header_content' ) ) {
        return apply_filters( 'magpal_is_header_content_active', false );
    }

    return apply_filters( 'magpal_is_header_content_active', true );
}

function magpal_is_primary_menu_active() {
    if ( magpal_get_option( 'disable_primary_menu' ) ) {
        return apply_filters( 'magpal_is_primary_menu_active', false );
    }

    return apply_filters( 'magpal_is_primary_menu_active', true );
}

function magpal_is_secondary_menu_active() {
    if ( magpal_get_option( 'disable_secondary_menu' ) ) {
        return apply_filters( 'magpal_is_secondary_menu_active', false );
    }

    return apply_filters( 'magpal_is_secondary_menu_active', true );
}

function magpal_is_primary_menu_responsive() {
    if ( magpal_get_option( 'disable_responsive_primary_menu' ) ) {
        return apply_filters( 'magpal_is_primary_menu_responsive', false );
    }

    return apply_filters( 'magpal_is_primary_menu_responsive', true );
}

function magpal_is_secondary_menu_responsive() {
    if ( magpal_get_option( 'disable_responsive_secondary_menu' ) ) {
        return apply_filters( 'magpal_is_secondary_menu_responsive', false );
    }

    return apply_filters( 'magpal_is_secondary_menu_responsive', true );
}

function magpal_is_sticky_header_active() {
    if ( magpal_get_option( 'enable_sticky_header' ) ) {
        return apply_filters( 'magpal_is_sticky_header_active', true );
    }

    return apply_filters( 'magpal_is_sticky_header_active', false );
}

function magpal_is_sticky_mobile_header_active() {
    if ( magpal_get_option( 'enable_sticky_mobile_header' ) ) {
        return apply_filters( 'magpal_is_sticky_mobile_header_active', true );
    }

    return apply_filters( 'magpal_is_sticky_mobile_header_active', false );
}

function magpal_is_footer_social_buttons_active() {
    if ( magpal_get_option( 'hide_social_buttons' ) ) {
        return apply_filters( 'magpal_is_footer_social_buttons_active', false );
    }

    return apply_filters( 'magpal_is_footer_social_buttons_active', true );
}

function magpal_is_fitvids_active() {
    if ( magpal_get_option( 'disable_fitvids' ) ) {
        return apply_filters( 'magpal_is_fitvids_active', false );
    }

    return apply_filters( 'magpal_is_fitvids_active', true );
}

function magpal_is_sidebar_active() {
    if ( magpal_get_option( 'disable_sidebar' ) ) {
        return apply_filters( 'magpal_is_sidebar_active', false );
    }

    return apply_filters( 'magpal_is_sidebar_active', true );
}

function magpal_is_backtotop_active() {
    if ( magpal_get_option( 'disable_backtotop' ) ) {
        return apply_filters( 'magpal_is_backtotop_active', false );
    }

    return apply_filters( 'magpal_is_backtotop_active', true );
}

function magpal_sidebar_width() {
    $sidebar_width = magpal_get_option( 'sidebar_width' ) ?: 350;

    return apply_filters( 'magpal_sidebar_width', $sidebar_width );
}

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function magpal_content_width() {
    $content_width = 1200;

    if ( is_singular() ) {
        $content_width = 1200;
    } else {
        $content_width = 1200;
    }

    $GLOBALS['content_width'] = apply_filters( 'magpal_content_width', $content_width ); /* phpcs:ignore WPThemeReview.CoreFunctionality.PrefixAllGlobals.NonPrefixedVariableFound */
}
add_action( 'template_redirect', 'magpal_content_width', 0 );

function magpal_thumbnail_alignment_single() {
    $alignment = magpal_get_option( 'thumbnail_alignment_single' ) ?: 'magpal-thumbnail-alignwide';

    return apply_filters( 'magpal_thumbnail_alignment_single', $alignment );
}

function magpal_thumbnail_alignment_page() {
    $alignment = magpal_get_option( 'thumbnail_alignment_page' ) ?: 'magpal-thumbnail-alignwide';

    return apply_filters( 'magpal_thumbnail_alignment_page', $alignment );
}

/**
* Register widget area.
*
* @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
*/

function magpal_widgets_init() {

register_sidebar(array(
    'id' => 'magpal-off-canvas-sidebar',
    'name' => esc_html__( 'Off-Canvas Sidebar', 'magpal' ),
    'description' => esc_html__( 'Widgets in this area will appear in the off-canvas sidebar.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-off-canvas-sidebar-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

register_sidebar(array(
    'id' => 'magpal-home-fullwidth-widgets',
    'name' => esc_html__( 'Top Full Width Widgets (Default HomePage)', 'magpal' ),
    'description' => esc_html__( 'This full-width widget area is located after the header of your website. Widgets of this widget area are displayed on the default homepage of your website (when you are showing your latest posts on homepage).', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-main-widget widget magpal-widget-box %2$s"><div class="magpal-outer-wrapper magpal-fullwidth-widgets-outer-wrapper"><div class="magpal-widget-box-inside">',
    'after_widget' => '</div></div></div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

register_sidebar(array(
    'id' => 'magpal-fullwidth-widgets',
    'name' => esc_html__( 'Top Full Width Widgets (Everywhere)', 'magpal' ),
    'description' => esc_html__( 'This full-width widget area is located after the header of your website. Widgets of this widget area are displayed on every page of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-main-widget widget magpal-widget-box %2$s"><div class="magpal-outer-wrapper magpal-fullwidth-widgets-outer-wrapper"><div class="magpal-widget-box-inside">',
    'after_widget' => '</div></div></div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

register_sidebar(array(
    'id' => 'magpal-home-fullwidth-bottom-widgets',
    'name' => esc_html__( 'Bottom Full Width Widgets (Default HomePage)', 'magpal' ),
    'description' => esc_html__( 'This full-width widget area is located before the footer of your website. Widgets of this widget area are displayed on the default homepage of your website (when you are showing your latest posts on homepage).', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-main-widget widget magpal-widget-box %2$s"><div class="magpal-outer-wrapper magpal-fullwidth-widgets-outer-wrapper"><div class="magpal-widget-box-inside">',
    'after_widget' => '</div></div></div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

register_sidebar(array(
    'id' => 'magpal-fullwidth-bottom-widgets',
    'name' => esc_html__( 'Bottom Full Width Widgets (Everywhere)', 'magpal' ),
    'description' => esc_html__( 'This full-width widget area is located before the footer of your website. Widgets of this widget area are displayed on every page of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-main-widget widget magpal-widget-box %2$s"><div class="magpal-outer-wrapper magpal-fullwidth-widgets-outer-wrapper"><div class="magpal-widget-box-inside">',
    'after_widget' => '</div></div></div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

register_sidebar(array(
    'id' => 'magpal-single-post-bottom-widgets',
    'name' => esc_html__( 'Single Post Bottom Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is located at the bottom of single post of any post type (except attachments and pages).', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-main-widget widget magpal-widget-box %2$s"><div class="magpal-widget-box-inside">',
    'after_widget' => '</div></div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

register_sidebar(array(
    'id' => 'magpal-top-footer',
    'name' => esc_html__( 'Footer Top Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is located on the top of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-footer-1',
    'name' => esc_html__( 'Footer 1 Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is the column 1 of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-footer-2',
    'name' => esc_html__( 'Footer 2 Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is the column 2 of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-footer-3',
    'name' => esc_html__( 'Footer 3 Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is the column 3 of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-footer-4',
    'name' => esc_html__( 'Footer 4 Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is the column 4 of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-footer-5',
    'name' => esc_html__( 'Footer 5 Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is the column 5 of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-bottom-footer',
    'name' => esc_html__( 'Footer Bottom Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is located on the bottom of the footer of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-footer-widget widget %2$s">',
    'after_widget' => '</div>',
    'before_title' => '<h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2>'));

register_sidebar(array(
    'id' => 'magpal-404-widgets',
    'name' => esc_html__( '404 Page Widgets', 'magpal' ),
    'description' => esc_html__( 'This widget area is located on the 404(not found) page of your website.', 'magpal' ),
    'before_widget' => '<div id="%1$s" class="magpal-main-widget widget magpal-widget-box %2$s"><div class="magpal-widget-box-inside">',
    'after_widget' => '</div></div>',
    'before_title' => '<div class="magpal-widget-header"><h2 class="magpal-widget-title"><span class="magpal-widget-title-inside">',
    'after_title' => '</span></h2></div>'));

}
add_action( 'widgets_init', 'magpal_widgets_init' );

function magpal_top_wide_widgets() {
    // Check if any of the targeted sidebars are active
    if (
        is_active_sidebar( 'magpal-home-fullwidth-widgets' ) || 
        is_active_sidebar( 'magpal-fullwidth-widgets' )
    ) {
        echo '<div class="magpal-top-wrapper-outer magpal-clearfix">
                <div class="magpal-featured-posts-area magpal-top-wrapper magpal-clearfix">';

        // Display widgets for specific conditions
        if ( is_front_page() && is_home() && !is_paged() ) {
            dynamic_sidebar( 'magpal-home-fullwidth-widgets' );
        }

        // Always display the general fullwidth widgets
        dynamic_sidebar( 'magpal-fullwidth-widgets' );

        echo '  </div>
              </div>';
    }
}

function magpal_bottom_wide_widgets() {
    // Check if any of the targeted sidebars are active
    if (
        is_active_sidebar( 'magpal-home-fullwidth-bottom-widgets' ) || 
        is_active_sidebar( 'magpal-fullwidth-bottom-widgets' )
    ) {
        echo '<div class="magpal-bottom-wrapper-outer magpal-clearfix">
                <div class="magpal-featured-posts-area magpal-bottom-wrapper magpal-clearfix">';

        // Display widgets for specific conditions
        if ( is_front_page() && is_home() && !is_paged() ) {
            dynamic_sidebar( 'magpal-home-fullwidth-bottom-widgets' );
        }

        // Always display the general fullwidth bottom widgets
        dynamic_sidebar( 'magpal-fullwidth-bottom-widgets' );

        echo '  </div>
              </div>';
    }
}

function magpal_404_widgets() {
    // Check if the 404 sidebar is active
    if ( is_active_sidebar( 'magpal-404-widgets' ) ) {
        echo '<div class="magpal-featured-posts-area magpal-featured-posts-area-top magpal-clearfix">';
        dynamic_sidebar( 'magpal-404-widgets' );
        echo '</div>';
    }
}

function magpal_post_bottom_widgets() {
    if ( is_singular() ) {
        // Display the bottom widgets if the sidebar is active
        if ( is_active_sidebar( 'magpal-single-post-bottom-widgets' ) ) {
            echo '<div class="magpal-featured-posts-area magpal-clearfix">';
            dynamic_sidebar( 'magpal-single-post-bottom-widgets' );
            echo '</div>';
        }
    }
}

/**
* Social buttons
*/

function magpal_footer_social_buttons() { ?>

<div class='magpal-footer-social-icons'>
    <?php if ( magpal_get_option('twitterlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('twitterlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-twitter" aria-label="<?php esc_attr_e('X Button','magpal'); ?>"><i class="fab fa-x-twitter" aria-hidden="true" title="<?php esc_attr_e('X','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('facebooklink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('facebooklink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-facebook" aria-label="<?php esc_attr_e('Facebook Button','magpal'); ?>"><i class="fab fa-facebook-f" aria-hidden="true" title="<?php esc_attr_e('Facebook','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('threadslink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('threadslink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-threads" aria-label="<?php esc_attr_e('Threads Button','magpal'); ?>"><i class="fab fa-threads" aria-hidden="true" title="<?php esc_attr_e('Threads','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('pinterestlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('pinterestlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-pinterest" aria-label="<?php esc_attr_e('Pinterest Button','magpal'); ?>"><i class="fab fa-pinterest" aria-hidden="true" title="<?php esc_attr_e('Pinterest','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('linkedinlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('linkedinlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-linkedin" aria-label="<?php esc_attr_e('Linkedin Button','magpal'); ?>"><i class="fab fa-linkedin-in" aria-hidden="true" title="<?php esc_attr_e('Linkedin','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('instagramlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('instagramlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-instagram" aria-label="<?php esc_attr_e('Instagram Button','magpal'); ?>"><i class="fab fa-instagram" aria-hidden="true" title="<?php esc_attr_e('Instagram','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('flickrlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('flickrlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-flickr" aria-label="<?php esc_attr_e('Flickr Button','magpal'); ?>"><i class="fab fa-flickr" aria-hidden="true" title="<?php esc_attr_e('Flickr','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('youtubelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('youtubelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-youtube" aria-label="<?php esc_attr_e('Youtube Button','magpal'); ?>"><i class="fab fa-youtube" aria-hidden="true" title="<?php esc_attr_e('Youtube','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('vimeolink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('vimeolink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-vimeo" aria-label="<?php esc_attr_e('Vimeo Button','magpal'); ?>"><i class="fab fa-vimeo-v" aria-hidden="true" title="<?php esc_attr_e('Vimeo','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('soundcloudlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('soundcloudlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-soundcloud" aria-label="<?php esc_attr_e('SoundCloud Button','magpal'); ?>"><i class="fab fa-soundcloud" aria-hidden="true" title="<?php esc_attr_e('SoundCloud','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('messengerlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('messengerlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-messenger" aria-label="<?php esc_attr_e('Messenger Button','magpal'); ?>"><i class="fab fa-facebook-messenger" aria-hidden="true" title="<?php esc_attr_e('Messenger','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('whatsapplink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('whatsapplink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-whatsapp" aria-label="<?php esc_attr_e('WhatsApp Button','magpal'); ?>"><i class="fab fa-whatsapp" aria-hidden="true" title="<?php esc_attr_e('WhatsApp','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('tiktoklink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('tiktoklink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-tiktok" aria-label="<?php esc_attr_e('TikTok Button','magpal'); ?>"><i class="fab fa-tiktok" aria-hidden="true" title="<?php esc_attr_e('TikTok','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('lastfmlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('lastfmlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-lastfm" aria-label="<?php esc_attr_e('Lastfm Button','magpal'); ?>"><i class="fab fa-lastfm" aria-hidden="true" title="<?php esc_attr_e('Lastfm','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('mediumlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('mediumlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-medium" aria-label="<?php esc_attr_e('Medium Button','magpal'); ?>"><i class="fab fa-medium-m" aria-hidden="true" title="<?php esc_attr_e('Medium','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('githublink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('githublink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-github" aria-label="<?php esc_attr_e('Github Button','magpal'); ?>"><i class="fab fa-github" aria-hidden="true" title="<?php esc_attr_e('Github','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('bitbucketlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('bitbucketlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-bitbucket" aria-label="<?php esc_attr_e('Bitbucket Button','magpal'); ?>"><i class="fab fa-bitbucket" aria-hidden="true" title="<?php esc_attr_e('Bitbucket','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('tumblrlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('tumblrlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-tumblr" aria-label="<?php esc_attr_e('Tumblr Button','magpal'); ?>"><i class="fab fa-tumblr" aria-hidden="true" title="<?php esc_attr_e('Tumblr','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('digglink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('digglink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-digg" aria-label="<?php esc_attr_e('Digg Button','magpal'); ?>"><i class="fab fa-digg" aria-hidden="true" title="<?php esc_attr_e('Digg','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('deliciouslink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('deliciouslink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-delicious" aria-label="<?php esc_attr_e('Delicious Button','magpal'); ?>"><i class="fab fa-delicious" aria-hidden="true" title="<?php esc_attr_e('Delicious','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('stumblelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('stumblelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-stumbleupon" aria-label="<?php esc_attr_e('Stumbleupon Button','magpal'); ?>"><i class="fab fa-stumbleupon" aria-hidden="true" title="<?php esc_attr_e('Stumbleupon','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('mixlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('mixlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-mix" aria-label="<?php esc_attr_e('Mix Button','magpal'); ?>"><i class="fab fa-mix" aria-hidden="true" title="<?php esc_attr_e('Mix','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('redditlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('redditlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-reddit" aria-label="<?php esc_attr_e('Reddit Button','magpal'); ?>"><i class="fab fa-reddit" aria-hidden="true" title="<?php esc_attr_e('Reddit','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('dribbblelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('dribbblelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-dribbble" aria-label="<?php esc_attr_e('Dribbble Button','magpal'); ?>"><i class="fab fa-dribbble" aria-hidden="true" title="<?php esc_attr_e('Dribbble','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('flipboardlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('flipboardlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-flipboard" aria-label="<?php esc_attr_e('Flipboard Button','magpal'); ?>"><i class="fab fa-flipboard" aria-hidden="true" title="<?php esc_attr_e('Flipboard','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('bloggerlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('bloggerlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-blogger" aria-label="<?php esc_attr_e('Blogger Button','magpal'); ?>"><i class="fab fa-blogger" aria-hidden="true" title="<?php esc_attr_e('Blogger','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('etsylink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('etsylink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-etsy" aria-label="<?php esc_attr_e('Etsy Button','magpal'); ?>"><i class="fab fa-etsy" aria-hidden="true" title="<?php esc_attr_e('Etsy','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('behancelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('behancelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-behance" aria-label="<?php esc_attr_e('Behance Button','magpal'); ?>"><i class="fab fa-behance" aria-hidden="true" title="<?php esc_attr_e('Behance','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('amazonlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('amazonlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-amazon" aria-label="<?php esc_attr_e('Amazon Button','magpal'); ?>"><i class="fab fa-amazon" aria-hidden="true" title="<?php esc_attr_e('Amazon','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('meetuplink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('meetuplink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-meetup" aria-label="<?php esc_attr_e('Meetup Button','magpal'); ?>"><i class="fab fa-meetup" aria-hidden="true" title="<?php esc_attr_e('Meetup','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('mixcloudlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('mixcloudlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-mixcloud" aria-label="<?php esc_attr_e('Mixcloud Button','magpal'); ?>"><i class="fab fa-mixcloud" aria-hidden="true" title="<?php esc_attr_e('Mixcloud','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('slacklink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('slacklink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-slack" aria-label="<?php esc_attr_e('Slack Button','magpal'); ?>"><i class="fab fa-slack" aria-hidden="true" title="<?php esc_attr_e('Slack','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('snapchatlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('snapchatlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-snapchat" aria-label="<?php esc_attr_e('Snapchat Button','magpal'); ?>"><i class="fab fa-snapchat" aria-hidden="true" title="<?php esc_attr_e('Snapchat','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('spotifylink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('spotifylink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-spotify" aria-label="<?php esc_attr_e('Spotify Button','magpal'); ?>"><i class="fab fa-spotify" aria-hidden="true" title="<?php esc_attr_e('Spotify','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('yelplink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('yelplink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-yelp" aria-label="<?php esc_attr_e('Yelp Button','magpal'); ?>"><i class="fab fa-yelp" aria-hidden="true" title="<?php esc_attr_e('Yelp','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('wordpresslink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('wordpresslink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-wordpress" aria-label="<?php esc_attr_e('WordPress Button','magpal'); ?>"><i class="fab fa-wordpress" aria-hidden="true" title="<?php esc_attr_e('WordPress','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('twitchlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('twitchlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-twitch" aria-label="<?php esc_attr_e('Twitch Button','magpal'); ?>"><i class="fab fa-twitch" aria-hidden="true" title="<?php esc_attr_e('Twitch','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('telegramlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('telegramlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-telegram" aria-label="<?php esc_attr_e('Telegram Button','magpal'); ?>"><i class="fab fa-telegram" aria-hidden="true" title="<?php esc_attr_e('Telegram','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('bandcamplink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('bandcamplink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-bandcamp" aria-label="<?php esc_attr_e('Bandcamp Button','magpal'); ?>"><i class="fab fa-bandcamp" aria-hidden="true" title="<?php esc_attr_e('Bandcamp','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('quoralink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('quoralink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-quora" aria-label="<?php esc_attr_e('Quora Button','magpal'); ?>"><i class="fab fa-quora" aria-hidden="true" title="<?php esc_attr_e('Quora','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('foursquarelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('foursquarelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-foursquare" aria-label="<?php esc_attr_e('Foursquare Button','magpal'); ?>"><i class="fab fa-foursquare" aria-hidden="true" title="<?php esc_attr_e('Foursquare','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('deviantartlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('deviantartlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-deviantart" aria-label="<?php esc_attr_e('DeviantArt Button','magpal'); ?>"><i class="fab fa-deviantart" aria-hidden="true" title="<?php esc_attr_e('DeviantArt','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('imdblink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('imdblink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-imdb" aria-label="<?php esc_attr_e('IMDB Button','magpal'); ?>"><i class="fab fa-imdb" aria-hidden="true" title="<?php esc_attr_e('IMDB','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('vklink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('vklink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-vk" aria-label="<?php esc_attr_e('VK Button','magpal'); ?>"><i class="fab fa-vk" aria-hidden="true" title="<?php esc_attr_e('VK','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('codepenlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('codepenlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-codepen" aria-label="<?php esc_attr_e('Codepen Button','magpal'); ?>"><i class="fab fa-codepen" aria-hidden="true" title="<?php esc_attr_e('Codepen','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('jsfiddlelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('jsfiddlelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-jsfiddle" aria-label="<?php esc_attr_e('JSFiddle Button','magpal'); ?>"><i class="fab fa-jsfiddle" aria-hidden="true" title="<?php esc_attr_e('JSFiddle','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('stackoverflowlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('stackoverflowlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-stackoverflow" aria-label="<?php esc_attr_e('Stack Overflow Button','magpal'); ?>"><i class="fab fa-stack-overflow" aria-hidden="true" title="<?php esc_attr_e('Stack Overflow','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('stackexchangelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('stackexchangelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-stackexchange" aria-label="<?php esc_attr_e('Stack Exchange Button','magpal'); ?>"><i class="fab fa-stack-exchange" aria-hidden="true" title="<?php esc_attr_e('Stack Exchange','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('bsalink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('bsalink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-buysellads" aria-label="<?php esc_attr_e('BuySellAds Button','magpal'); ?>"><i class="fab fa-buysellads" aria-hidden="true" title="<?php esc_attr_e('BuySellAds','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('web500pxlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('web500pxlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-web500px" aria-label="<?php esc_attr_e('500px Button','magpal'); ?>"><i class="fab fa-500px" aria-hidden="true" title="<?php esc_attr_e('500px','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('ellolink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('ellolink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-ello" aria-label="<?php esc_attr_e('Ello Button','magpal'); ?>"><i class="fab fa-ello" aria-hidden="true" title="<?php esc_attr_e('Ello','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('discordlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('discordlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-discord" aria-label="<?php esc_attr_e('Discord Button','magpal'); ?>"><i class="fab fa-discord" aria-hidden="true" title="<?php esc_attr_e('Discord','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('goodreadslink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('goodreadslink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-goodreads" aria-label="<?php esc_attr_e('Goodreads Button','magpal'); ?>"><i class="fab fa-goodreads" aria-hidden="true" title="<?php esc_attr_e('Goodreads','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('odnoklassnikilink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('odnoklassnikilink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-odnoklassniki" aria-label="<?php esc_attr_e('Odnoklassniki Button','magpal'); ?>"><i class="fab fa-odnoklassniki" aria-hidden="true" title="<?php esc_attr_e('Odnoklassniki','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('houzzlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('houzzlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-houzz" aria-label="<?php esc_attr_e('Houzz Button','magpal'); ?>"><i class="fab fa-houzz" aria-hidden="true" title="<?php esc_attr_e('Houzz','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('pocketlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('pocketlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-pocket" aria-label="<?php esc_attr_e('Pocket Button','magpal'); ?>"><i class="fab fa-get-pocket" aria-hidden="true" title="<?php esc_attr_e('Pocket','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('xinglink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('xinglink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-xing" aria-label="<?php esc_attr_e('XING Button','magpal'); ?>"><i class="fab fa-xing" aria-hidden="true" title="<?php esc_attr_e('XING','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('mastodonlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('mastodonlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-mastodon" aria-label="<?php esc_attr_e('Mastodon Button','magpal'); ?>"><i class="fab fa-mastodon" aria-hidden="true" title="<?php esc_attr_e('Mastodon','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('blueskylink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('blueskylink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-bluesky" aria-label="<?php esc_attr_e('Bluesky Button','magpal'); ?>"><i class="fab fa-bluesky" aria-hidden="true" title="<?php esc_attr_e('Bluesky','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('googleplaylink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('googleplaylink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-googleplay" aria-label="<?php esc_attr_e('Google Play Button','magpal'); ?>"><i class="fab fa-google-play" aria-hidden="true" title="<?php esc_attr_e('Google Play','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('slidesharelink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('slidesharelink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-slideshare" aria-label="<?php esc_attr_e('SlideShare Button','magpal'); ?>"><i class="fab fa-slideshare" aria-hidden="true" title="<?php esc_attr_e('SlideShare','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('dropboxlink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('dropboxlink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-dropbox" aria-label="<?php esc_attr_e('Dropbox Button','magpal'); ?>"><i class="fab fa-dropbox" aria-hidden="true" title="<?php esc_attr_e('Dropbox','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('paypallink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('paypallink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-paypal" aria-label="<?php esc_attr_e('PayPal Button','magpal'); ?>"><i class="fab fa-paypal" aria-hidden="true" title="<?php esc_attr_e('PayPal','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('viadeolink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('viadeolink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-viadeo" aria-label="<?php esc_attr_e('Viadeo Button','magpal'); ?>"><i class="fab fa-viadeo" aria-hidden="true" title="<?php esc_attr_e('Viadeo','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('wikipedialink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('wikipedialink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-wikipedia" aria-label="<?php esc_attr_e('Wikipedia Button','magpal'); ?>"><i class="fab fa-wikipedia-w" aria-hidden="true" title="<?php esc_attr_e('Wikipedia','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('skypeusername') ) : ?>
            <a href="skype:<?php echo esc_html( magpal_get_option('skypeusername') ); ?>?chat" class="magpal-footer-social-icon-skype" aria-label="<?php esc_attr_e('Skype Button','magpal'); ?>"><i class="fab fa-skype" aria-hidden="true" title="<?php esc_attr_e('Skype','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('emailaddress') ) : ?>
            <a href="mailto:<?php echo esc_html( magpal_get_option('emailaddress') ); ?>" class="magpal-footer-social-icon-email" aria-label="<?php esc_attr_e('Email Us Button','magpal'); ?>"><i class="far fa-envelope" aria-hidden="true" title="<?php esc_attr_e('Email Us','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('rsslink') ) : ?>
            <a href="<?php echo esc_url( magpal_get_option('rsslink') ); ?>" target="_blank" rel="nofollow" class="magpal-footer-social-icon-rss" aria-label="<?php esc_attr_e('RSS Button','magpal'); ?>"><i class="fas fa-rss" aria-hidden="true" title="<?php esc_attr_e('RSS','magpal'); ?>"></i></a><?php endif; ?>
    <?php if ( magpal_get_option('show_footer_login_button') ) { ?><?php if (is_user_logged_in()) : ?><a href="<?php echo esc_url( wp_logout_url( get_permalink() ) ); ?>" aria-label="<?php esc_attr_e( 'Logout Button', 'magpal' ); ?>" class="magpal-footer-social-icon-login"><i class="fas fa-sign-out-alt" aria-hidden="true" title="<?php esc_attr_e('Logout','magpal'); ?>"></i></a><?php else : ?><a href="<?php echo esc_url( wp_login_url( get_permalink() ) ); ?>" aria-label="<?php esc_attr_e( 'Login / Register Button', 'magpal' ); ?>" class="magpal-footer-social-icon-login"><i class="fas fa-sign-in-alt" aria-hidden="true" title="<?php esc_attr_e('Login / Register','magpal'); ?>"></i></a><?php endif;?><?php } ?>
</div>

<?php }

/**
* Author bio box
*/

function magpal_add_author_bio_box() {
    $content='';
    if (is_single()) {
        $content .= '
            <div class="magpal-author-bio">
            <div class="magpal-author-bio-inside">
            <div class="magpal-author-bio-top">
            <span class="magpal-author-bio-gravatar">
                '. get_avatar( get_the_author_meta('email') , 80 ) .'
            </span>
            <div class="magpal-author-bio-text">
                <div class="magpal-author-bio-name">'.esc_html__( 'Author: ', 'magpal' ).'<span>'. get_the_author_link() .'</span></div><div class="magpal-author-bio-text-description">'. wp_kses_post( get_the_author_meta('description',get_query_var('author') ) ) .'</div>
            </div>
            </div>
            </div>
            </div>
        ';
    }
    return apply_filters( 'magpal_add_author_bio_box', $content );
}

/**
* Post meta functions
*/

if ( ! function_exists( 'magpal_post_tags' ) ) :
/**
 * Prints HTML with meta information for the tags.
 */
function magpal_post_tags() {
    if ( 'post' === get_post_type() ) {
        /* Translators: used between list items, there is a space after the comma */
        $tags_list = get_the_tag_list( '', esc_html_x( ', ', 'list item separator', 'magpal' ) );
        if ( $tags_list ) {
            ?>
            <span class="magpal-tags-links">
                <i class="fas fa-tags" aria-hidden="true"></i>
                <?php
                /* Translators: %s: list of tags */
                echo wp_kses_post( sprintf(
                    __( 'Tagged %s', 'magpal' ),
                    $tags_list
                ) );
                ?>
            </span>
            <?php
        }
    }
}
endif;

if ( ! function_exists( 'magpal_grid_cats' ) ) :
/**
 * Displays categories for grid posts.
 */
function magpal_grid_cats() {
    if ( 'post' === get_post_type() ) {
        /* translators: used between list items, there is a comma and space */
        $categories_list = get_the_category_list( esc_html__( '&nbsp;', 'magpal' ) );
        if ( $categories_list ) {
            ?>
            <div class="magpal-grid-post-categories magpal-grid-post-block">
                <span class="screen-reader-text">
                    <?php esc_html_e( 'Posted in ', 'magpal' ); ?>
                </span>
                <?php echo wp_kses_post( $categories_list ); ?>
            </div>
            <?php
        }
    }
}
endif;

if ( ! function_exists( 'magpal_grid_postmeta' ) ) :
function magpal_grid_postmeta() {
    global $post;

    // Retrieve global options
    $hide_author_option   = magpal_get_option( 'hide_grid_post_author' );
    $hide_comments_option = magpal_get_option( 'hide_grid_comments_link' );

    // Determine which elements to show
    $show_author   = ! $hide_author_option;
    $show_comments = ! $hide_comments_option;

    // Check if any meta should be displayed
    if ( $show_author || $show_comments ) {
        ?>
        <div class="magpal-grid-post-footer magpal-grid-post-block">
            <div class="magpal-grid-post-footer-inside">
                <?php
                // Display the author
                if ( $show_author ) {
                    ?>
                    <span class="magpal-grid-post-author magpal-grid-post-meta">
                        <a href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
                            <?php echo esc_html( get_the_author() ); ?>
                        </a>
                    </span>
                    <?php
                }

                // Display the comments link
                if ( $show_comments ) {
                    if ( ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
                        ?>
                        <span class="magpal-grid-post-comment magpal-grid-post-meta">
                            <?php
                            comments_popup_link(
                                esc_html__( '0 Comment', 'magpal' ),
                                esc_html__( '1 Comment', 'magpal' ),
                                esc_html__( '% Comments', 'magpal' )
                            );
                            ?>
                        </span>
                        <?php
                    }
                }
                ?>
            </div>
        </div>
        <?php
    }
}
endif;

if ( ! function_exists( 'magpal_single_cats' ) ) :
/**
 * Displays categories for single posts.
 */
function magpal_single_cats() {
    if ( 'post' === get_post_type() ) {
        /* translators: used between list items, there is a space */
        $categories_list = get_the_category_list( esc_html__( ', ', 'magpal' ) );
        if ( $categories_list ) {
            ?>
            <span class="magpal-entry-meta-single-cats magpal-entry-meta-single-block">
                <i class="far fa-folder-open" aria-hidden="true"></i>&nbsp;
                <span class="screen-reader-text">
                    <?php esc_html_e( 'Posted in ', 'magpal' ); ?>
                </span>
                <?php echo wp_kses_post( $categories_list ); ?>
            </span>
            <?php
        }
    }
}
endif;

if ( ! function_exists( 'magpal_single_postmeta' ) ) :
/**
 * Displays post meta information on single posts.
 */
function magpal_single_postmeta() {
    global $post;

    // Retrieve global options
    $hide_author_option      = magpal_get_option( 'hide_single_post_author' );
    $hide_date_option        = magpal_get_option( 'hide_single_post_date' );
    $hide_comments_option    = magpal_get_option( 'hide_single_comments_link' );
    $hide_categories_option  = magpal_get_option( 'hide_single_post_categories' );
    $hide_edit_option        = magpal_get_option( 'hide_single_post_edit' );

    // Determine which elements to show
    $show_author      = ! $hide_author_option;
    $show_date        = ! $hide_date_option;
    $show_comments    = ! $hide_comments_option && ! post_password_required() && ( comments_open() || get_comments_number() );
    $show_categories  = ! $hide_categories_option && has_category();
    $show_edit        = ! $hide_edit_option;

    // Check if any meta should be displayed
    if ( $show_author || $show_date || $show_comments || $show_categories || $show_edit ) {
        ?>
        <div class="magpal-entry-meta-single">
            <?php
            // Display the author
            if ( $show_author ) {
                ?>
                <span class="magpal-entry-meta-single-author magpal-entry-meta-single-block">
                    <i class="far fa-user-circle" aria-hidden="true"></i>&nbsp;
                    <span class="author vcard" itemscope itemtype="http://schema.org/Person" itemprop="author">
                        <a class="url fn n" href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
                            <?php echo esc_html( get_the_author() ); ?>
                        </a>
                    </span>
                </span>
                <?php
            }

            // Display the date
            if ( $show_date ) {
                $single_post_date_type = magpal_get_option( 'single_post_date_type' );
                $date_format           = get_option( 'date_format' );

                if ( 'updated' === $single_post_date_type ) {
                    $date = get_the_modified_date( $date_format );
                } else {
                    $date = get_the_date( $date_format );
                }
                ?>
                <span class="magpal-entry-meta-single-date magpal-entry-meta-single-block">
                    <i class="far fa-clock" aria-hidden="true"></i>&nbsp;
                    <?php echo esc_html( $date ); ?>
                </span>
                <?php
            }

            // Display the comments link
            if ( $show_comments ) {
                ?>
                <span class="magpal-entry-meta-single-comments magpal-entry-meta-single-block">
                    <i class="far fa-comments" aria-hidden="true"></i>&nbsp;
                    <?php
                    comments_popup_link(
                        sprintf(
                            wp_kses(
                                /* translators: %s: post title */
                                __( 'Leave a Comment<span class="screen-reader-text"> on %s</span>', 'magpal' ),
                                array( 'span' => array( 'class' => array() ) )
                            ),
                            wp_kses_post( get_the_title() )
                        )
                    );
                    ?>
                </span>
                <?php
            }

            // Display categories
            if ( $show_categories ) {
                magpal_single_cats();
            }

            // Display edit link
            if ( $show_edit ) {
                edit_post_link(
                    sprintf(
                        wp_kses(
                            /* translators: %s: Name of current post. Only visible to screen readers */
                            __( 'Edit<span class="screen-reader-text"> %s</span>', 'magpal' ),
                            array( 'span' => array( 'class' => array() ) )
                        ),
                        wp_kses_post( get_the_title() )
                    ),
                    '<span class="edit-link magpal-entry-meta-single-block">&nbsp;&nbsp;<i class="far fa-edit" aria-hidden="true"></i> ',
                    '</span>'
                );
            }
            ?>
        </div>
        <?php
    }
}
endif;

if ( ! function_exists( 'magpal_page_postmeta' ) ) :
/**
 * Displays post meta information on pages.
 */
function magpal_page_postmeta() {
    global $post;

    // Retrieve global options
    $hide_author_option     = magpal_get_option( 'hide_page_author' );
    $hide_date_option       = magpal_get_option( 'hide_page_date' );
    $hide_comments_option   = magpal_get_option( 'hide_page_comments_link' );
    $hide_static_page_meta  = magpal_get_option( 'hide_static_page_meta' );
    $page_date_type         = magpal_get_option( 'page_date_type' );

    // Determine which elements to show
    $show_author   = ! $hide_author_option;
    $show_date     = ! $hide_date_option;
    $show_comments = ! $hide_comments_option && ! post_password_required() && ( comments_open() || get_comments_number() );
    $show_meta     = $show_author || $show_date || $show_comments;

    // Check if we should hide meta on the front page
    $is_static_front_page = is_front_page() && $hide_static_page_meta;

    // Display meta if needed and not hiding on front page
    if ( $show_meta && ! $is_static_front_page ) {
        ?>
        <div class="magpal-entry-meta-single">
            <?php
            // Display the author
            if ( $show_author ) {
                ?>
                <span class="magpal-entry-meta-single-author magpal-entry-meta-single-block">
                    <i class="far fa-user-circle" aria-hidden="true"></i>&nbsp;
                    <span class="author vcard" itemscope itemtype="http://schema.org/Person" itemprop="author">
                        <a class="url fn n" href="<?php echo esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ); ?>">
                            <?php echo esc_html( get_the_author() ); ?>
                        </a>
                    </span>
                </span>
                <?php
            }

            // Display the date
            if ( $show_date ) {
                $date_format = get_option( 'date_format' );

                if ( 'updated' === $page_date_type ) {
                    $date = get_the_modified_date( $date_format );
                } else {
                    $date = get_the_date( $date_format );
                }
                ?>
                <span class="magpal-entry-meta-single-date magpal-entry-meta-single-block">
                    <i class="far fa-clock" aria-hidden="true"></i>&nbsp;
                    <?php echo esc_html( $date ); ?>
                </span>
                <?php
            }

            // Display the comments link
            if ( $show_comments ) {
                ?>
                <span class="magpal-entry-meta-single-comments magpal-entry-meta-single-block">
                    <i class="far fa-comments" aria-hidden="true"></i>&nbsp;
                    <?php
                    comments_popup_link(
                        sprintf(
                            wp_kses(
                                /* translators: %s: post title */
                                __( 'Leave a Comment<span class="screen-reader-text"> on %s</span>', 'magpal' ),
                                array( 'span' => array( 'class' => array() ) )
                            ),
                            wp_kses_post( get_the_title() )
                        )
                    );
                    ?>
                </span>
                <?php
            }
            ?>
        </div>
        <?php
    }
}
endif;

if ( ! function_exists( 'magpal_grid_datebox' ) ) :
function magpal_grid_datebox() {
    global $post;

    // Retrieve options and meta values once
    $grid_post_date_type      = magpal_get_option( 'grid_post_date_type' );
    $hide_grid_post_date      = magpal_get_option( 'hide_grid_post_date' );

    // Check if the date box should be displayed
    if ( ! $hide_grid_post_date ) {
        // Decide whether to use the modified date or the original date
        if ( $grid_post_date_type === 'updated' ) {
            $day   = get_the_modified_date( 'd' );
            $month = get_the_modified_date( 'M' );
            $year  = get_the_modified_date( 'Y' );
        } else {
            $day   = get_the_date( 'd' );
            $month = get_the_date( 'M' );
            $year  = get_the_date( 'Y' );
        }
        ?>
        <div class="magpal-grid-datebox">
            <div class="magpal-grid-datebox-day"><?php echo esc_html( $day ); ?></div>
            <div class="magpal-grid-datebox-month"><?php echo esc_html( $month ); ?></div>
            <div class="magpal-grid-datebox-year"><?php echo esc_html( $year ); ?></div>
        </div>
        <?php
    }
}
endif;

/**
* Posts navigation functions
*/

/**
 * Display pagination using WP-PageNavi plugin.
 *
 * @return void
 */
if ( ! function_exists( 'magpal_wp_pagenavi' ) ) :
function magpal_wp_pagenavi() {
    if ( function_exists( 'wp_pagenavi' ) ) { // Check if WP-PageNavi is active
        ?>
        <nav class="navigation posts-navigation magpal-clearfix" role="navigation" aria-label="<?php esc_attr_e( 'Posts navigation', 'magpal' ); ?>">
            <?php wp_pagenavi(); ?>
        </nav><!-- .navigation -->
        <?php
    }
}
endif;

/**
 * Display posts navigation based on theme settings.
 */
if ( ! function_exists( 'magpal_posts_navigation' ) ) :
function magpal_posts_navigation() {
    // Check if post navigation is globally disabled.
    if ( magpal_get_option( 'hide_posts_navigation' ) ) {
        return;
    }

    // Use WP-PageNavi if available.
    if ( function_exists( 'wp_pagenavi' ) ) {
        magpal_wp_pagenavi();
        return;
    }

    // Determine the navigation type from theme options.
    $nav_type = magpal_get_option( 'posts_navigation_type' );

    // Render appropriate navigation based on the type.
    switch ( $nav_type ) {
        case 'normalnavi':
            the_posts_navigation( array(
                'prev_text' => esc_html__( 'Older posts', 'magpal' ),
                'next_text' => esc_html__( 'Newer posts', 'magpal' ),
            ) );
            break;

        default:
            the_posts_pagination( array(
                'mid_size'  => 2,
                'prev_text' => esc_html__( '&larr; Newer posts', 'magpal' ),
                'next_text' => esc_html__( 'Older posts &rarr;', 'magpal' ),
            ) );
            break;
    }
}
endif;

/**
 * Display post navigation based on theme settings.
 */
if ( ! function_exists( 'magpal_post_navigation' ) ) :
function magpal_post_navigation() {
    global $post;

    // Check if post navigation is globally disabled.
    if ( magpal_get_option( 'hide_post_navigation' ) ) {
        return;
    }

    the_post_navigation( array(
        'prev_text' => esc_html__( '%title &rarr;', 'magpal' ),
        'next_text' => esc_html__( '&larr; %title', 'magpal' ),
    ) );
}
endif;

/**
* WooCommerce support
*/

// Checking if WooCommerce is active
if ( defined('MAGPAL_WOOCOMMERCE_ACTIVE') && MAGPAL_WOOCOMMERCE_ACTIVE ) {

    // Remove WooCommerce default content wrappers
    remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10 );
    remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10 );

    // Add custom WooCommerce content wrappers
    add_action( 'woocommerce_before_main_content', 'magpal_woo_wrapper_start', 10 );
    add_action( 'woocommerce_after_main_content', 'magpal_woo_wrapper_end', 10 );

    /**
     * Start custom WooCommerce wrapper.
     */
    function magpal_woo_wrapper_start() {
        magpal_before_shop_content(); // Hook for additional actions or content
        echo '
        <div class="magpal-shop-wrapper" id="magpal-shop-wrapper" itemscope="itemscope" itemtype="http://schema.org/ItemList" role="main">
            <div class="magpal-outer-wrapper magpal-shop-outer-wrapper">
                <div class="magpal-shop-wrapper-inside magpal-clearfix">
                    <div class="magpal-shop-content">';
    }

    /**
     * End custom WooCommerce wrapper.
     */
    function magpal_woo_wrapper_end() {
        echo '
                    </div> <!-- /.magpal-shop-content -->
                </div> <!-- /.magpal-shop-wrapper-inside -->
            </div> <!-- /.magpal-shop-outer-wrapper -->
        </div> <!-- /#magpal-shop-wrapper -->';
        magpal_after_shop_content(); // Hook for additional actions or content
    }
}

/**
* Menu Functions
*/

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a "Home" link as the first item
 */
function magpal_page_menu_args( $args ) {
    $args['show_home'] = true;
    return $args;
}
add_filter( 'wp_page_menu_args', 'magpal_page_menu_args' );

/**
 * Retrieve the primary menu text.
 *
 * @return string The primary menu text.
 */
function magpal_primary_menu_text() {
    $menu_text = magpal_is_option_set( 'primary_menu_text' ) ? magpal_get_option( 'primary_menu_text' ) : esc_html__( 'Menu', 'magpal' );

    return apply_filters( 'magpal_primary_menu_text', $menu_text );
}

/**
 * Retrieve the secondary menu text.
 *
 * @return string The secondary menu text.
 */
function magpal_secondary_menu_text() {
    $menu_text = magpal_is_option_set( 'secondary_menu_text' ) ? magpal_get_option( 'secondary_menu_text' ) : esc_html__( 'Menu', 'magpal' );

    return apply_filters( 'magpal_secondary_menu_text', $menu_text );
}

/**
 * Render a fallback menu using wp_page_menu().
 *
 * This function generates a menu from the site's pages if no custom menu is assigned.
 */
function magpal_primary_fallback_menu() {
    wp_page_menu( array(
        'sort_column'  => 'menu_order, post_title', // Order by menu order first, then post title.
        'menu_id'      => 'magpal-menu-primary-navigation', // Add a unique ID for the menu.
        'menu_class'   => 'magpal-primary-nav-menu magpal-menu-primary', // Add CSS classes for styling.
        'container'    => 'ul', // Use an <ul> as the wrapper.
        'echo'         => true, // Output the menu directly.
        'link_before'  => '', // No additional text before links.
        'link_after'   => '', // No additional text after links.
        'before'       => '', // No additional content before menu items.
        'after'        => '', // No additional content after menu items.
        'item_spacing' => 'discard', // Remove extra spacing for clean HTML output.
        'walker'       => '', // No custom walker used.
    ) );
}

/**
 * Render a fallback menu using wp_page_menu().
 *
 * This function generates a menu from the site's pages if no custom menu is assigned.
 */
function magpal_secondary_fallback_menu() {
    wp_page_menu( array(
        'sort_column'  => 'menu_order, post_title', // Sort by menu order first, then post title.
        'menu_id'      => 'magpal-menu-secondary-navigation', // Add an ID to the <ul> element.
        'menu_class'   => 'magpal-secondary-nav-menu magpal-menu-secondary', // Add CSS classes to the <ul> element.
        'container'    => 'ul', // Do not wrap the menu in a <div>.
        'echo'         => true, // Output the menu.
        'link_before'  => '', // No text or HTML before links.
        'link_after'   => '', // No text or HTML after links.
        'before'       => '', // No text or HTML before each item.
        'after'        => '', // No text or HTML after each item.
        'item_spacing' => 'discard', // Remove extra spacing in the output.
        'walker'       => '', // Use the default walker.
    ) );
}

/**
 * Render the secondary menu area.
 */
function magpal_secondary_menu_area() {
    if ( !magpal_is_secondary_menu_active() ) {
        return; // Exit if the secondary menu is not active.
    }
    ?>
    <div class="magpal-secondary-menu-container magpal-clearfix">
        <div class="magpal-secondary-menu-container-inside magpal-clearfix">
            <nav class="magpal-nav-secondary" id="magpal-secondary-navigation" itemscope="itemscope" itemtype="http://schema.org/SiteNavigationElement" role="navigation" aria-label="<?php esc_attr_e( 'Secondary Menu', 'magpal' ); ?>">
                <div class="magpal-outer-wrapper magpal-secondary-menu-outer-wrapper">
                    <?php if ( magpal_is_secondary_menu_responsive() ) : ?>
                        <button class="magpal-secondary-responsive-menu-icon" aria-controls="magpal-menu-secondary-navigation" aria-expanded="false">
                            <i class="fas fa-bars" aria-hidden="true" title="<?php esc_attr_e( 'Menu', 'magpal' ); ?>"></i>
                            <?php if ( magpal_secondary_menu_text() ) : ?>&nbsp;<?php echo esc_html( magpal_secondary_menu_text() ); ?><?php endif; ?>
                        </button>
                    <?php endif; ?>

                    <?php
                    wp_nav_menu( array(
                        'theme_location' => 'secondary',
                        'menu_id'        => 'magpal-menu-secondary-navigation',
                        'menu_class'     => 'magpal-secondary-nav-menu magpal-menu-secondary magpal-clearfix',
                        'fallback_cb'    => 'magpal_secondary_fallback_menu',
                        'container'      => '',
                    ) );
                    ?>
                </div>
            </nav>
        </div>
    </div>
    <?php
}

/**
* Header Functions
*/

/**
 * Add a pingback URL auto-discovery header for single posts, pages, or attachments.
 */
function magpal_pingback_header() {
    if ( is_singular() && pings_open() ) {
        printf(
            '<link rel="pingback" href="%s">',
            esc_url( get_bloginfo( 'pingback_url' ) )
        );
    }
}
add_action( 'wp_head', 'magpal_pingback_header' );

/**
 * Get the custom logo URL.
 *
 * @return string|null Custom logo URL or null if no custom logo is set.
 */
function magpal_custom_logo() {
    // Check if a custom logo is set.
    if ( ! has_custom_logo() ) {
        return null;
    }

    // Get the custom logo ID.
    $custom_logo_id = get_theme_mod( 'custom_logo' );

    // Get the custom logo URL.
    $logo = wp_get_attachment_image_src( $custom_logo_id, 'full' );
    $logo_src = isset( $logo[0] ) ? $logo[0] : null;

    // Apply a filter and return the custom logo URL.
    return apply_filters( 'magpal_custom_logo', $logo_src );
}

/**
 * Render a skip link for accessibility.
 */
function magpal_skip_link() {
    // Determine the appropriate wrapper ID based on the current view.
    if ( is_singular() ) {
        $target_id = 'magpal-singular-wrapper';
    } elseif ( is_404() ) {
        $target_id = 'magpal-404-wrapper';
    } else {
        $target_id = 'magpal-posts-wrapper';
    }

    // Output the skip link.
    printf(
        '<a class="skip-link screen-reader-text" href="#%s">%s</a>',
        esc_attr( $target_id ),
        esc_html__( 'Skip to content', 'magpal' )
    );
}

/**
 * Display the site title and tagline.
 */
function magpal_site_title() {
    if ( is_front_page() && is_home() ) { ?>
            <h1 class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_home() ) { ?>
            <h1 class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_singular() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_category() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_tag() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_author() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_archive() && !is_category() && !is_tag() && !is_author() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_search() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } elseif ( is_404() ) { ?>
            <p class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php } else { ?>
            <h1 class="magpal-site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
            <?php if ( !(magpal_get_option('hide_tagline')) ) { ?><p class="magpal-site-description"><?php bloginfo( 'description' ); ?></p><?php } ?>
    <?php }
}

/**
 * Get the header image destination URL.
 *
 * @return string The URL for the header image destination.
 */
function magpal_header_image_destination() {
    $url = magpal_get_option( 'header_image_destination' ) ?: home_url( '/' );

    return apply_filters( 'magpal_header_image_destination', $url );
}

/**
 * Get the alt text for the header image.
 *
 * @return string The alt text for the header image.
 */
function magpal_header_image_alt_text() {
    global $post;

    // Default alt text to the site name.
    $alt_text = get_bloginfo( 'name' );

    // Fallback to global header image alt text if set.
    $custom_alt_text = magpal_get_option( 'header_image_alt_text' );
    if ( ! empty( $custom_alt_text ) ) {
        $alt_text = $custom_alt_text;
    }

    return apply_filters( 'magpal_header_image_alt_text', $alt_text );
}

/**
 * Render the header image markup.
 */
function magpal_header_image_markup() {
    $attributes = '';
    $attributes .= magpal_get_option( 'header_image_destination_target' ) ? ' target="_blank"' : '';
    $attributes .= magpal_get_option( 'header_image_destination_rel' ) ? ' rel="nofollow"' : '';
    $remove_link = magpal_get_option( 'remove_header_image_destination' ) ? true : false;

    // Global fallback: Use the default header image if no custom image is found.
    $header_image = get_header_image();
    if ( $header_image ) {
        if ( $remove_link ) {
            the_header_image_tag( array( 'class' => 'magpal-header-img', 'alt' => esc_attr( magpal_header_image_alt_text() ) ) );
        } else {
            echo sprintf(
                '<a href="%s" class="magpal-header-img-link"%s>',
                esc_url( magpal_header_image_destination() ),
                $attributes
            );
            the_header_image_tag( array( 'class' => 'magpal-header-img', 'alt' => esc_attr( magpal_header_image_alt_text() ) ) );
            echo '</a>';
        }
    }
}

/**
 * Render header image details (title and description).
 */
function magpal_header_image_details() {
    $header_image_custom_title = magpal_get_option('header_image_custom_title') ?: '';
    $header_image_custom_description = magpal_get_option('header_image_custom_description') ?: '';

    if ( !(magpal_get_option('hide_header_image_details')) ) {
    if ( magpal_get_option('header_image_custom_text') ) {
        if ( $header_image_custom_title || $header_image_custom_description ) { ?>
            <div class="magpal-header-image-info">
            <div class="magpal-header-image-info-inside">
                <?php if ( !(magpal_get_option('hide_header_image_title')) ) { ?><?php if ( $header_image_custom_title ) { ?><p class="magpal-header-image-site-title magpal-header-image-block"><?php echo wp_kses_post( force_balance_tags( do_shortcode($header_image_custom_title) ) ); ?></p><?php } ?><?php } ?>
                <?php if ( !(magpal_get_option('hide_header_image_description')) ) { ?><?php if ( $header_image_custom_description ) { ?><p class="magpal-header-image-site-description magpal-header-image-block"><?php echo wp_kses_post( force_balance_tags( do_shortcode($header_image_custom_description) ) ); ?></p><?php } ?><?php } ?>
            </div>
            </div>
        <?php }
    } else { ?>
        <div class="magpal-header-image-info">
        <div class="magpal-header-image-info-inside">
            <?php if ( !(magpal_get_option('hide_header_image_title')) ) { ?><p class="magpal-header-image-site-title magpal-header-image-block"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p><?php } ?>
            <?php if ( !(magpal_get_option('hide_header_image_description')) ) { ?><p class="magpal-header-image-site-description magpal-header-image-block"><?php bloginfo( 'description' ); ?></p><?php } ?>
        </div>
        </div>
    <?php }
    }
}

/**
 * Render the header image wrapper, including the image markup and details.
 */
function magpal_header_image_wrapper() { ?>
    <div class="magpal-header-image magpal-clearfix">
    <?php magpal_header_image_markup(); ?>
    <?php magpal_header_image_details(); ?>
    </div>
    <?php
}

/**
 * Render the header image for the site.
 */
function magpal_header_image() {
    if ( magpal_get_option( 'hide_header_image' ) ) {
        return; // Exit if the header image is sitewide hidden.
    }

    // Render global header image if no custom header image is found.
    if ( get_header_image() ) {
        magpal_header_image_wrapper();
    }
}

/**
 * Get the maximum width for the logo.
 *
 * @return string The maximum width of the logo.
 */
function magpal_logo_max_width() {
    $logo_max_width = magpal_get_option( 'logo_max_width' ) ?: '300px';

    return apply_filters( 'magpal_logo_max_width', $logo_max_width );
}

/**
* Css Classes Functions
*/

/**
 * Add category IDs as classes to the post class
 *
 * @param array $classes Existing post classes.
 * @return array Modified post classes.
 */
function magpal_category_id_class( $classes ) {
    if ( is_singular() ) { // Ensure this runs on singular views where global $post is available
        global $post;

        // Get the categories for the post
        $categories = get_the_category( $post->ID );

        // Add category ID classes if categories exist
        if ( ! empty( $categories ) ) {
            foreach ( $categories as $category ) {
                $classes[] = 'wpcat-' . $category->cat_ID . '-id';
            }
        }
    }

    return apply_filters( 'magpal_category_id_class', $classes );
}
add_filter( 'post_class', 'magpal_category_id_class' );

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Existing body classes.
 * @return array Modified body classes.
 */
function magpal_body_classes( $classes ) {
    // Add class if the site has multiple authors.
    if ( is_multi_author() ) {
        $classes[] = 'magpal-group-blog';
    }

    // Add loading animation classes if not disabled.
    if ( ! magpal_get_option( 'disable_loading_animation' ) ) {
        $classes[] = 'magpal-animated';
        $classes[] = 'magpal-fadein';
    }

    // Always add theme active class.
    $classes[] = 'magpal-theme-is-active';

    // Add class if header image is set.
    if ( get_header_image() ) {
        $classes[] = 'magpal-header-image-active';
    }

    if ( magpal_get_option('header_image_cover') ) {
        $classes[] = 'magpal-header-image-cover';
    }

    // Add class if custom logo is set.
    if ( has_custom_logo() ) {
        $classes[] = 'magpal-custom-logo-active';
    }

    // Add class based on whether it's a singular page.
    $classes[] = is_singular() ? 'magpal-singular-page' : 'magpal-non-singular-page';

    if ( is_singular() ) {
        global $post;

        // Check for featured media under title.
        $featured_media_under_title = false;

        if ( is_single() ) {
            $featured_media_under_title = magpal_get_option( 'featured_media_under_post_title' );
        }
        if ( is_page() ) {
            $featured_media_under_title = magpal_get_option( 'featured_media_under_page_title' );
        }

        if ( $featured_media_under_title ) {
            $classes[] = 'magpal-singular-media-under-title';
        }
    }

    $classes[] = 'magpal-header-style-logo-menu';

    // Add class if the tagline is hidden.
    if ( magpal_get_option( 'hide_tagline' ) ) {
        $classes[] = 'magpal-tagline-inactive';
    }

    $logo_location = magpal_get_option( 'logo_location' );
    if ( 'beside-title' === $logo_location ) {
        $classes[] = 'magpal-logo-beside-title';
    } elseif ( 'above-title' === $logo_location ) {
        $classes[] = 'magpal-logo-above-title';
    } else {
        $classes[] = 'magpal-logo-above-title';
    }

    // Primary and secondary menu classes.
    $classes[] = magpal_is_primary_menu_active() ? 'magpal-primary-menu-active' : '';
    $classes[] = magpal_is_primary_menu_responsive() ? 'magpal-primary-mobile-menu-active' : 'magpal-primary-mobile-menu-inactive';
    $classes[] = magpal_is_secondary_menu_active() ? 'magpal-secondary-menu-active' : '';
    $classes[] = magpal_is_secondary_menu_responsive() ? 'magpal-secondary-mobile-menu-active' : 'magpal-secondary-mobile-menu-inactive';
    $classes[] = magpal_get_option('no_center_secondary_menu') ? '' : 'magpal-secondary-menu-centered';

    $classes[] = 'magpal-secondary-menu-topup';
    $classes[] = 'magpal-table-css-active';
    $classes[] = magpal_get_option('no_underline_content_links') ? 'magpal-nouc-links' : 'magpal-uc-links';

    /**
     * Filters the list of body classes for the MagPal theme.
     *
     * @param array $classes An array of body classes.
     */
    return apply_filters( 'magpal_body_classes', $classes );
}
add_filter( 'body_class', 'magpal_body_classes' );

/**
* More Custom Functions
*/

/**
 * Filter the excerpt length.
 *
 * @param int $length Default excerpt length.
 * @return int Modified excerpt length.
 */
function magpal_excerpt_length( $length ) {
    // Return default length in the admin area unless it's an AJAX request.
    if ( is_admin() ) {
        return $length;
    }

    // Retrieve the custom excerpt length or fallback to 17.
    $grid_post_snippet_length = magpal_get_option( 'grid_post_snippet_length' ) ?: 17;

    /**
     * Filter the excerpt length.
     *
     * @param int $grid_post_snippet_length The custom excerpt length.
     */
    return apply_filters( 'magpal_excerpt_length', $grid_post_snippet_length );
}
add_filter( 'excerpt_length', 'magpal_excerpt_length' );

/**
 * Modify the "read more" text for excerpts.
 *
 * @param string $more The default "read more" text.
 * @return string The modified "read more" text.
 */
function magpal_excerpt_more( $more ) {
    // Return default "read more" text in the admin area unless it's an AJAX request.
    if ( is_admin() ) {
        return $more;
    }

    // Return the custom "read more" text.
    return apply_filters( 'magpal_excerpt_more', '...' );
}
add_filter( 'excerpt_more', 'magpal_excerpt_more' );

if ( ! function_exists( 'wp_body_open' ) ) :
    /**
     * Adds the `wp_body_open` function for backward compatibility.
     *
     * This function was introduced in WordPress 5.2. If it doesn't exist,
     * we define it to maintain compatibility with themes and plugins.
     *
     * @since WordPress 5.2.0
     */
    function wp_body_open() { // phpcs:ignore WPThemeReview.CoreFunctionality.PrefixAllGlobals.NonPrefixedFunctionFound
        /**
         * Fires the `wp_body_open` action.
         *
         * This action is triggered immediately after the opening <body> tag.
         * Developers can use this action to add custom scripts, analytics, or content.
         */
        do_action( 'wp_body_open' ); // phpcs:ignore WPThemeReview.CoreFunctionality.PrefixAllGlobals.NonPrefixedHooknameFound
    }
endif;

/**
 * Get the grid post title length.
 *
 * @return int The number of words for the grid post title.
 */
function magpal_grid_post_title_length() {
    $default_length = 6;
    $grid_post_title_length = magpal_get_option( 'grid_post_title_length' ) ?: $default_length;

    /**
     * Filter the grid post title length.
     *
     * @param int $grid_post_title_length The number of words for the grid post title.
     */
    return apply_filters( 'magpal_grid_post_title_length', $grid_post_title_length );
}

/**
 * Get the grid post snippet length.
 *
 * @return int The number of words for the grid post snippet.
 */
function magpal_grid_post_snippet_length() {
    $default_length = 17;
    $grid_post_snippet_length = magpal_get_option( 'grid_post_snippet_length' ) ?: $default_length;

    /**
     * Filter the grid post snippet length.
     *
     * @param int $grid_post_snippet_length The number of words for the grid post snippet.
     */
    return apply_filters( 'magpal_grid_post_snippet_length', $grid_post_snippet_length );
}

/**
 * Get the grid post snippet type.
 *
 * @return string The type of grid post snippet.
 */
function magpal_grid_post_snippet_type() {
    $default_type = 'postexcerpt';
    $grid_post_snippet_type = magpal_get_option( 'grid_post_snippet_type' ) ?: $default_type;

    /**
     * Filter the grid post snippet type.
     *
     * @param string $grid_post_snippet_type The type of grid post snippet.
     */
    return apply_filters( 'magpal_grid_post_snippet_type', $grid_post_snippet_type );
}

/**
 * Modify the archive titles for categories, tags, authors, and date-based archives.
 *
 * @param string $title The current archive title.
 * @return string The modified archive title.
 */
function magpal_modify_archive_titles( $title ) {
    // Modify category title.
    if ( is_category() && magpal_get_option( 'hide_category_word' ) ) {
        $title = single_cat_title( '', false );
    }

    // Modify tag title.
    if ( is_tag() && magpal_get_option( 'hide_tag_word' ) ) {
        $title = single_tag_title( '', false );
    }

    // Modify author title.
    if ( is_author() && magpal_get_option( 'hide_author_word' ) ) {
        $title = '<span class="vcard">' . get_the_author() . '</span>';
    }

    // Modify date-based titles.
    if ( magpal_get_option( 'hide_date_word' ) ) {
        if ( is_year() ) {
            $title = get_the_date( _x( 'Y', 'yearly archives date format', 'magpal' ) );
        } elseif ( is_month() ) {
            $title = get_the_date( _x( 'F Y', 'monthly archives date format', 'magpal' ) );
        } elseif ( is_day() ) {
            $title = get_the_date( _x( 'F j, Y', 'daily archives date format', 'magpal' ) );
        }
    }

    return apply_filters( 'magpal_modify_archive_titles', $title );
}
add_filter( 'get_the_archive_title', 'magpal_modify_archive_titles', 100 );

/**
 * Add a mobile menu close button to the primary menu.
 *
 * @param string   $items Menu items HTML.
 * @param stdClass $args  An object containing wp_nav_menu() arguments.
 * @return string Updated menu items HTML.
 */
function magpal_add_menu_search_button( $items, $args ) {
    // Ensure the function only modifies the primary menu.
    if ( $args->theme_location !== 'primary' ) {
        return $items;
    }

    // Add a close button for responsive menus.
    if ( magpal_is_primary_menu_responsive() ) {
        $mobile_menu_close_button  = '<li class="magpal-primary-menu-close-item">';
        $mobile_menu_close_button .= '<a href="#" aria-label="' . esc_attr__( 'Menu Close Button', 'magpal' ) . '" class="magpal-mobile-menu-close">';
        $mobile_menu_close_button .= '<i class="fas fa-xmark" aria-hidden="true" title="' . esc_attr__( 'Close', 'magpal' ) . '"></i>';
        $mobile_menu_close_button .= '</a>';
        $mobile_menu_close_button .= '</li>';

        $items .= $mobile_menu_close_button;
    }

    return $items;
}
add_filter( 'wp_nav_menu_items', 'magpal_add_menu_search_button', 10, 2 );

/**
 * Get the search box placeholder text.
 *
 * @return string The placeholder text for the search box.
 */
function magpal_search_box_placeholder_text() {
    // Default placeholder text.
    $placeholder_text = esc_html__( 'Enter your search keyword...', 'magpal' );

    // Check if a custom placeholder text is set in theme options.
    if ( magpal_get_option( 'search_box_placeholder_text' ) ) {
        $placeholder_text = magpal_get_option( 'search_box_placeholder_text' );
    }

    // Allow filtering of the placeholder text.
    return apply_filters( 'magpal_search_box_placeholder_text', $placeholder_text );
}

if ( ! function_exists( 'magpal_remove_theme_support' ) ) :
    /**
     * Remove theme support for responsive embeds if fitvids is active.
     *
     * This function checks if the FitVids plugin or similar functionality is active
     * and removes support for WordPress's responsive embeds to avoid conflicts.
     */
    function magpal_remove_theme_support() {
        if ( magpal_is_fitvids_active() ) {
            // Remove responsive embedded content support.
            remove_theme_support( 'responsive-embeds' );
        }
    }
endif;

// Hook the function after theme setup with a high priority to ensure it runs late.
add_action( 'after_setup_theme', 'magpal_remove_theme_support', 1000 );

/**
* Custom Hooks
*/

function magpal_before_header() {
    do_action('magpal_before_header');
}

function magpal_after_header() {
    do_action('magpal_after_header');
}

function magpal_before_main_content() {
    do_action('magpal_before_main_content');
}
add_action('magpal_before_main_content', 'magpal_top_wide_widgets', 20 );

function magpal_after_main_content() {
    do_action('magpal_after_main_content');
}
add_action('magpal_after_main_content', 'magpal_bottom_wide_widgets', 20 );

function magpal_before_shop_content() {
    do_action('magpal_before_shop_content');
}

function magpal_after_shop_content() {
    do_action('magpal_after_shop_content');
}

function magpal_before_single_post() {
    do_action('magpal_before_single_post');
}

function magpal_before_single_post_title() {
    do_action('magpal_before_single_post_title');
}

function magpal_after_single_post_title() {
    do_action('magpal_after_single_post_title');
}

function magpal_top_single_post_content() {
    do_action('magpal_top_single_post_content');
}

function magpal_bottom_single_post_content() {
    do_action('magpal_bottom_single_post_content');
}

function magpal_after_single_post_content() {
    do_action('magpal_after_single_post_content');
}

function magpal_after_single_post() {
    do_action('magpal_after_single_post');
}

function magpal_before_single_page() {
    do_action('magpal_before_single_page');
}

function magpal_before_single_page_title() {
    do_action('magpal_before_single_page_title');
}

function magpal_after_single_page_title() {
    do_action('magpal_after_single_page_title');
}

function magpal_after_single_page_content() {
    do_action('magpal_after_single_page_content');
}

function magpal_after_single_page() {
    do_action('magpal_after_single_page');
}

function magpal_before_comments() {
    do_action('magpal_before_comments');
}

function magpal_after_comments() {
    do_action('magpal_after_comments');
}

function magpal_before_footer() {
    do_action('magpal_before_footer');
}

function magpal_after_footer() {
    do_action('magpal_after_footer');
}

function magpal_before_nongrid_post_title() {
    do_action('magpal_before_nongrid_post_title');
}

function magpal_after_nongrid_post_title() {
    do_action('magpal_after_nongrid_post_title');
}

if ( !(magpal_get_option('enable_widgets_block_editor')) ) {
    // Disables the block editor from managing widgets in the Gutenberg plugin.
    add_filter( 'gutenberg_use_widgets_block_editor', '__return_false', 100 );

    // Disables the block editor from managing widgets.
    add_filter( 'use_widgets_block_editor', '__return_false' );
}

/**
* Media functions
*/

/**
 * Render the media content grid for posts.
 */
function magpal_media_content_grid() {
    global $post;

    // Check if the grid thumbnail is hidden globally or for the specific post.
    if ( magpal_get_option( 'hide_grid_thumbnail' ) ) {
        return; // Exit if the grid thumbnail is hidden.
    }

    // Render the grid thumbnail if the post has a featured image.
    if ( has_post_thumbnail( $post->ID ) ) {
        ?>
        <div class="magpal-grid-post-thumbnail">
            <a href="<?php echo esc_url( get_permalink() ); ?>" class="magpal-grid-post-thumbnail-link" title="<?php /* translators: %s: post title. */ echo esc_attr( sprintf( __( 'Permanent Link to %s', 'magpal' ), the_title_attribute( [ 'echo' => false ] ) ) ); ?>">
                <?php the_post_thumbnail( 'magpal-300w-autoh-image', [ 'class' => 'magpal-grid-post-thumbnail-img', 'title' => the_title_attribute( [ 'echo' => false ] ) ] ); ?>
            </a>
            <?php magpal_grid_datebox(); ?>
        </div>
        <?php
    } else {
        // Render the default grid thumbnail if no featured image exists and the default is not hidden.
        if ( !magpal_get_option( 'hide_default_grid_thumbnail' ) ) {
            ?>
            <div class="magpal-grid-post-thumbnail magpal-grid-post-thumbnail-default">
                <a href="<?php echo esc_url( get_permalink() ); ?>" class="magpal-grid-post-thumbnail-link" title="<?php /* translators: %s: post title. */ echo esc_attr( sprintf( __( 'Permanent Link to %s', 'magpal' ), the_title_attribute( [ 'echo' => false ] ) ) ); ?>">
                    <img src="<?php echo esc_url( get_template_directory_uri() . '/assets/images/no-image-300-225.jpg' ); ?>" class="magpal-grid-post-thumbnail-img" />
                </a>
                <?php magpal_grid_datebox(); ?>
            </div>
            <?php
        }
    }
}

/**
 * Render the featured image (thumbnail) for single posts.
 */
function magpal_media_content_single() {
    global $post;

    // Exit if the post does not have a featured image.
    if ( !has_post_thumbnail() ) {
        return;
    }

    // Exit if the single thumbnail display is not enabled globally.
    if ( magpal_get_option( 'hide_single_thumbnail' ) ) {
        return;
    }

    // Determine if the thumbnail should be linked.
    $thumbnail_link = magpal_get_option( 'single_thumbnail_link' ) == 'no' ? false : true;

    // Render the thumbnail.
    $thumbnail_classes = 'magpal-post-thumbnail-single ' . esc_attr( magpal_thumbnail_alignment_single() );
    ?>
    <div class="<?php echo esc_attr( $thumbnail_classes ); ?>">
        <?php if ( $thumbnail_link ) : ?>
            <a href="<?php echo esc_url( get_permalink() ); ?>" title="<?php /* translators: %s: post title. */ echo esc_attr( sprintf( __( 'Permanent Link to %s', 'magpal' ), the_title_attribute( [ 'echo' => false ] ) ) ); ?>" class="magpal-post-thumbnail-single-link">
                <?php the_post_thumbnail( 'magpal-1222w-autoh-image', [ 'class' => 'magpal-post-thumbnail-single-img', 'title' => the_title_attribute( [ 'echo' => false ] ) ] ); ?>
            </a>
        <?php else : ?>
            <?php the_post_thumbnail( 'magpal-1222w-autoh-image', [ 'class' => 'magpal-post-thumbnail-single-img', 'title' => the_title_attribute( [ 'echo' => false ] ) ] ); ?>
        <?php endif; ?>
    </div>
    <?php
}

/**
 * Dynamically set the location of media content in a single post.
 */
function magpal_media_content_single_location() {
    if ( is_single() ) {
        if ( magpal_get_option( 'featured_media_under_post_title' ) ) {
            // Use the Customizer setting to place the media under the post title.
            add_action( 'magpal_after_single_post_title', 'magpal_media_content_single', 10 );
        } else {
            // Default placement: Before the single post title.
            add_action( 'magpal_before_single_post_title', 'magpal_media_content_single', 10 );
        }
    }
}
add_action( 'template_redirect', 'magpal_media_content_single_location', 100 );

/**
 * Render the featured image (thumbnail) for single pages.
 */
function magpal_media_content_page() {
    global $post;

    // Exit if the post does not have a featured image.
    if ( !has_post_thumbnail() ) {
        return;
    }

    // Exit if the single thumbnail display is not enabled globally.
    if ( magpal_get_option( 'hide_page_thumbnail' ) ) {
        return;
    }

    // Determine if the thumbnail should be linked.
    $thumbnail_link = magpal_get_option( 'page_thumbnail_link' ) == 'no' ? false : true;

    // Render the thumbnail.
    $thumbnail_classes = 'magpal-post-thumbnail-single ' . esc_attr( magpal_thumbnail_alignment_page() );
    ?>
    <div class="<?php echo esc_attr( $thumbnail_classes ); ?>">
        <?php if ( $thumbnail_link ) : ?>
            <a href="<?php echo esc_url( get_permalink() ); ?>" title="<?php /* translators: %s: post title. */ echo esc_attr( sprintf( __( 'Permanent Link to %s', 'magpal' ), the_title_attribute( [ 'echo' => false ] ) ) ); ?>" class="magpal-post-thumbnail-single-link">
                <?php the_post_thumbnail( 'magpal-1222w-autoh-image', [ 'class' => 'magpal-post-thumbnail-single-img', 'title' => the_title_attribute( [ 'echo' => false ] ) ] ); ?>
            </a>
        <?php else : ?>
            <?php the_post_thumbnail( 'magpal-1222w-autoh-image', [ 'class' => 'magpal-post-thumbnail-single-img', 'title' => the_title_attribute( [ 'echo' => false ] ) ] ); ?>
        <?php endif; ?>
    </div>
    <?php
}

/**
 * Dynamically set the location of media content in a single page.
 */
function magpal_media_content_page_location() {
    if ( is_page() ) {
        // Check if the featured media should appear under the page title for this specific page.
        if ( magpal_get_option( 'featured_media_under_page_title' ) ) {
            // Use the Customizer setting to place the media under the page title.
            add_action( 'magpal_after_single_page_title', 'magpal_media_content_page', 10 );
        } else {
            // Default placement: Before the single page title.
            add_action( 'magpal_before_single_page_title', 'magpal_media_content_page', 10 );
        }
    }
}
add_action( 'template_redirect', 'magpal_media_content_page_location', 110 );

/**
* Enqueue scripts and styles
*/

function magpal_scripts() {
    // Enqueue styles
    wp_enqueue_style( 'magpal-maincss', get_stylesheet_uri(), array(), null );
    wp_enqueue_style( 'fontawesome', get_template_directory_uri() . '/assets/css/all.min.css', array(), null );
    wp_enqueue_style( 'magpal-webfont', '//fonts.googleapis.com/css2?family=Oswald:wght@400;500;700&amp;family=Roboto+Slab:wght@400;500;700&amp;family=Roboto+Condensed:ital,wght@0,400;0,700;1,400;1,700&amp;display=swap', array(), null );

    // Determine active features with explicit boolean values
    $magpal_primary_mobile_menu_active   = magpal_is_primary_menu_responsive() ? true : false;
    $magpal_secondary_mobile_menu_active = magpal_is_secondary_menu_responsive() ? true : false;
    $magpal_primary_menu_active          = magpal_is_primary_menu_active() ? true : false;
    $magpal_secondary_menu_active        = magpal_is_secondary_menu_active() ? true : false;
    $magpal_sticky_header_active         = magpal_is_sticky_header_active() ? true : false;
    $magpal_sticky_header_mobile_active  = magpal_is_sticky_mobile_header_active() ? true : false;
    $magpal_fitvids_active               = magpal_is_fitvids_active() ? true : false;
    $magpal_backtotop_active             = magpal_is_backtotop_active() ? true : false;
    $magpal_sidebar_active               = magpal_is_sidebar_active() ? true : false;

    // Enqueue scripts conditionally
    if ( $magpal_fitvids_active ) {
        wp_enqueue_script( 'fitvids', get_template_directory_uri() . '/assets/js/jquery.fitvids.min.js', array( 'jquery' ), null, true );
    }

    // Set other options
    $magpal_search_overlay_active = magpal_get_option( 'hide_header_search_button' ) ? false : true;

    $magpal_columnwidth = '.magpal-fixed-grid-col-sizer';
    $magpal_gutter      = '.magpal-fixed-grid-col-gutter';

    // Enqueue common scripts
    wp_enqueue_script( 'magpal-navigation', get_template_directory_uri() . '/assets/js/navigation.js', array(), null, true );
    wp_enqueue_script( 'magpal-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix.js', array(), null, true );

    // Enqueue custom script with appropriate dependencies
    wp_enqueue_script( 'magpal-customjs', get_template_directory_uri() .'/assets/js/custom.js', array( 'jquery', 'imagesloaded', 'masonry' ), null, true );

    // Localize script data
    wp_localize_script( 'magpal-customjs', 'magpal_ajax_object', array(
        'ajaxurl'                      => esc_url_raw( admin_url( 'admin-ajax.php' ) ),
        'primary_menu_active'          => $magpal_primary_menu_active,
        'secondary_menu_active'        => $magpal_secondary_menu_active,
        'primary_mobile_menu_active'   => $magpal_primary_mobile_menu_active,
        'secondary_mobile_menu_active' => $magpal_secondary_mobile_menu_active,
        'sticky_header_active'         => $magpal_sticky_header_active,
        'sticky_header_mobile_active'  => $magpal_sticky_header_mobile_active,
        'sidebar_active'               => $magpal_sidebar_active,
        'fitvids_active'               => $magpal_fitvids_active,
        'backtotop_active'             => $magpal_backtotop_active,
        'columnwidth'                  => $magpal_columnwidth,
        'gutter'                       => $magpal_gutter,
        'search_overlay_active'        => $magpal_search_overlay_active,
    ) );

    // Enqueue comment reply script if necessary
    if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
        wp_enqueue_script( 'comment-reply' );
    }
}
add_action( 'wp_enqueue_scripts', 'magpal_scripts' );

/**
 * Enqueue the WordPress Color Picker script and styles for the widget form.
 */
function magpal_enqueue_color_picker($hook_suffix) {
    // Enqueue color picker styles
    wp_enqueue_style('wp-color-picker');
    
    // Enqueue the color picker script along with jQuery
    wp_enqueue_script('wp-color-picker');
    wp_enqueue_script('magpal-color-picker', get_template_directory_uri() . '/assets/js/magpal-color-picker.js', array('wp-color-picker', 'jquery'), false, true);
}
//add_action('admin_enqueue_scripts', 'magpal_enqueue_color_picker');

/**
 * Enqueue styles for the block-based editor.
 */
function magpal_block_editor_styles() {
    wp_enqueue_style( 'magpal-block-editor-style', get_template_directory_uri() . '/assets/css/editor-blocks.css', array(), null );
}
add_action( 'enqueue_block_editor_assets', 'magpal_block_editor_styles' );

/**
 * Enqueue customizer styles.
 */
function magpal_enqueue_customizer_styles() {
    wp_enqueue_style( 'magpal-customizer-styles', get_template_directory_uri() . '/assets/css/customizer-style.css', array(), null );
    wp_enqueue_style('fontawesome', get_template_directory_uri() . '/assets/css/all.min.css', array(), null );
}
add_action( 'customize_controls_enqueue_scripts', 'magpal_enqueue_customizer_styles' );

/**
* Block Styles
*/

/**
 * Register Custom Block Styles
 */
if ( function_exists( 'register_block_style' ) ) :
    function magpal_register_block_styles() {

        // Ensure the style handle exists to prevent enqueue issues
        if ( wp_style_is( 'magpal-maincss', 'registered' ) ) {
            /**
             * Register block style for Core Button
             */
            register_block_style(
                'core/button',
                array(
                    'name'         => 'magpal-button',
                    'label'        => esc_html__( 'MagPal Button', 'magpal' ),
                    'style_handle' => 'magpal-maincss',
                )
            );
        }
    }
    add_action( 'init', 'magpal_register_block_styles' );
endif;

/**
* Customizer Options
*/

function magpal_inline_css() {
    $custom_css = '';

    if ( magpal_get_option('header_text_hover_color') ) {
        $custom_css .= '.magpal-site-title a:hover,.magpal-site-title a:focus,.magpal-site-title a:active{color:' . esc_attr( magpal_get_option('header_text_hover_color') ) . ';}';
    }

    if ( magpal_get_option('disable_grid_posts_bg_color') ) {
        $custom_css .= '.magpal-posts-wrapper{background:inherit !important;}';
    } else {
        if ( magpal_get_option('grid_posts_bg_color') ) {
            $custom_css .= '.magpal-posts-wrapper{background:' . esc_attr( magpal_get_option('grid_posts_bg_color') ) . ';}';
        }
    }

    if ( !magpal_is_backtotop_active() ) {
        $custom_css .= '.magpal-scroll-top{display:none !important;}';
    }

    if ( magpal_get_option('header_image_title_mobile') ) {
        $custom_css .= '.magpal-header-image .magpal-header-image-info{display:block !important;}';
        $custom_css .= '.magpal-header-image .magpal-header-image-info .magpal-header-image-site-title{display:block !important;}';
    }

    if ( magpal_get_option('header_image_description_mobile') ) {
        $custom_css .= '.magpal-header-image .magpal-header-image-info{display:block !important;}';
        $custom_css .= '.magpal-header-image .magpal-header-image-info .magpal-header-image-site-description{display:block !important;}';
    }

    if ( magpal_get_option('header_image_cover') ) {
        if ( magpal_get_option('header_image_height_mobile') ) {
            $custom_css .= '@media screen and (max-width: 599px){.magpal-header-image-cover .magpal-header-img{min-height:'.esc_html( magpal_get_option('header_image_height_mobile') ).';}}';
        }
    }

    if ( magpal_get_option('header_padding') ) {
        $custom_css .= '.magpal-header-inside-content{padding:'.esc_html( magpal_get_option('header_padding') ).';}';
    }

    if ( magpal_get_option('limit_logo_width') ) {
        if ( magpal_logo_max_width() ) {
            $custom_css .= '.magpal-logo-img{max-width:'.esc_html( magpal_logo_max_width() ).';}';
        }
    }

    if ( magpal_sidebar_width() ) {
        $custom_css .= '@media only screen and (min-width:380px){.magpal-off-canvas-sidebar{width:'.esc_html( magpal_sidebar_width() ).'px;}}';
    }

    if ( $custom_css !== '' ) {
        wp_add_inline_style( 'magpal-maincss', $custom_css );
    }
}
add_action( 'wp_enqueue_scripts', 'magpal_inline_css' );

// Header styles
if ( ! function_exists( 'magpal_header_style' ) ) :
function magpal_header_style() {
    $header_text_color = get_header_textcolor();
    //if ( get_theme_support( 'custom-header', 'default-text-color' ) === $header_text_color ) { return; }
    ?>
    <style type="text/css">
    <?php if ( ! display_header_text() ) : ?>
        .magpal-site-title, .magpal-site-description {position:absolute !important;word-wrap:normal !important;width:1px;height:1px;padding:0;margin:-1px;overflow:hidden;clip:rect(1px,1px,1px,1px);-webkit-clip-path:inset(50%);clip-path:inset(50%);white-space:nowrap;border:0;}
    <?php else : ?>
        .magpal-site-title, .magpal-site-title a, .magpal-site-title a:hover, .magpal-site-title a:focus, .magpal-site-title a:active, .magpal-site-description {color: #<?php echo esc_attr( $header_text_color ); ?>;}
    <?php endif; ?>
    </style>
    <?php
}
endif;