<?php
/**
 * Magic Magazine functions and definitions
 *
 * Sets up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme (see https://codex.wordpress.org/Theme_Development
 * and https://codex.wordpress.org/Child_Themes), you can override certain
 * functions (those wrapped in a function_exists() call) by defining them first
 * in your child theme's functions.php file. The child theme's functions.php
 * file is included before the parent theme's file, so the child theme
 * functions would be used.
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters, @link https://codex.wordpress.org/Plugin_API
 *
 * @package WordPress
 * @subpackage Magic_Magazine
 * @since Magic Magazine 1.0
 */


/*
 * Set up the content width value based on the theme's design.
 *
 * @see twentythirteen_content_width() for template-specific adjustments.
 */
if ( ! isset( $content_width ) )
	$content_width = 900;

/**
 * Magic Magazine setup.
 *
 * Sets up theme defaults and registers the various WordPress features that
 * Twenty Thirteen supports.
 *
 * @uses load_theme_textdomain() For translation/localization support.
 * @uses add_editor_style() To add Visual Editor stylesheets.
 * @uses add_theme_support() To add support for automatic feed links, post
 * formats, and post thumbnails.
 * @uses register_nav_menu() To add support for a navigation menu.
 * @uses set_post_thumbnail_size() To set a custom post thumbnail size.
 *
 * @since Magic Magazine 1.0
 */

function magic_magazine_setup(){
	/*
	 * Makes Magic Magazine available for translation.
	 *
	 * Translations can be added to the /languages/ directory.
	 * If you're building a theme based on Magic Magazine, use a find and
	 * replace to change 'magicmagazine' to the name of your theme in all
	 * template files.
	 */
	load_theme_textdomain( 'magicmagazin', get_template_directory() . '/languages' );

	/*
	 * This theme styles the visual editor to resemble the theme style,
	 * specifically font, colors, icons, and column width.
	 */
	add_editor_style();

	// Adds RSS feed links to <head> for posts and comments.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * This theme uses a custom image size for featured images, displayed on
	 * "standard" posts and pages.
	 */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 200, 170, true ); // Sets the Post Main Thumbnails
	add_image_size( 'magic_magazine_post-thumbnails', 260, 150, true ); // Sets Post Posts Thumbnails
	add_image_size( 'magic_magazine_single-post-thumbnails', 690, 365, true ); // Sets Single post Thumbnails

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( "title-tag" );

}
add_action('after_setup_theme', 'magic_magazine_setup');


/**
 * Enqueue scripts and styles for the front end.
 *
 * @since Magic Magazine 1.0
 */
function magic_magazine_scripts_styles() {

	wp_enqueue_script('jquery');

	// Loads JavaScript file for modernizr.
	wp_enqueue_script( 'magicmagagine-modernizr-script', get_template_directory_uri() . '/js/modernizr-2.8.3.js', array( 'jquery' ), '2.8.3' );

	// Loads JavaScript file for bootstrap.
	wp_enqueue_script( 'magicmagagine-bootstrap-script', get_template_directory_uri() . '/js/bootstrap.js', array( 'jquery' ), '3.3.5', true );

	// Loads JavaScript file for Slicknav.
	wp_enqueue_script( 'magicmagagine-slicknav-script', get_template_directory_uri() . '/js/jquery.slicknav.js', array( 'jquery' ), '1.0', true );

	// Loads JavaScript file for plugin.
	wp_enqueue_script( 'magicmagagine-plugin-script', get_template_directory_uri() . '/js/plugins.js', array( 'jquery' ), '1.0', true );

	// Loads JavaScript file for Active.
	wp_enqueue_script( 'magicmagagine-active-script', get_template_directory_uri() . '/js/active.js', array( 'jquery' ), '1.0', true );


	// Add font stylesheet.
	wp_enqueue_style( 'magicmagagine-font', '//fonts.googleapis.com/css?family=Open+Sans:400,700', array(), '1.0' );

	// Add bootstarp stylesheet.
	wp_enqueue_style( 'magicmagagine-bootstrap', get_template_directory_uri() . '/css/bootstrap.css', array(), '3.3.3' );

	// Add slicknav stylesheet.
	wp_enqueue_style( 'magicmagagine-font-slicknav', get_template_directory_uri() . '/css/slicknav.css', array(), '1.0' );

	// Add normalize stylesheet.
	wp_enqueue_style( 'magicmagagine-normalize', get_template_directory_uri() . '/css/normalize.css', array(), '1.0' );

	// Add comments stylesheet.
	wp_enqueue_style( 'magicmagagine-comments', get_template_directory_uri() . '/css/comments.css', array(), '1.0' );

	// Loads our main stylesheet.
	wp_enqueue_style( 'magicmagagine-style', get_stylesheet_uri(), array(), '2015-09-20' );

	// Add Responsive stylesheet.
	wp_enqueue_style( 'magicmagagine-media', get_template_directory_uri() . '/css/media.css', array(), '1.0' );
}
add_action( 'wp_enqueue_scripts', 'magic_magazine_scripts_styles' );


/**
 * Function for main navigation 
 *
 * @since Magic Magazine 1.0
 */
function magic_magazine_menu_regeste(){
// This theme uses wp_nav_menu() in two locations.
	if (function_exists('register_nav_menu')){ 
		register_nav_menu( 'magic-main-menu', __( 'Main Menu', 'magic-magazine') ); 
	} 
}
add_action('init','magic_magazine_menu_regeste');


/**
 * Functions for Excerpt length 
 *
 * @since Magic Magazine 1.0
 */
function magic_magazine_excerptLength( $length ){ 
	return 30; 
}
add_filter( 'excerpt_length', 'magic_magazine_excerptLength' );

/**
 * Register five widget areas.
 *
 * @since Magic Magazine 1.0
 */
function magic_magazine_widget_area() {
	//Sidebar widget	
	register_sidebar( array(
        'name' => __( 'Sidebar Widget', 'magic-magazine' ),
        'id' => 'sidebar_widget',
        'before_widget' => '<div class="single_sidebar_widget">',
        'after_widget' => '</div>',
        'before_title' => '<h2>',
        'after_title' => '</h2>',
    ) );

	//Footer widget one
    register_sidebar( array(
        'name' => __( 'Footer Widget One', 'magic-magazine' ),
        'id' => 'footer_widget_one',
        'before_widget' => '<div class="single_ft_wed">',
        'after_widget' => '</div>',
        'before_title' => '<h2>',
        'after_title' => '</h2>',
    ) );

    //Footer widget two
	register_sidebar( array(
        'name' => __( 'Footer Widget Two', 'magic-magazine' ),
        'id' => 'footer_widget_two',
        'before_widget' => '<div class="single_ft_wed">',
        'after_widget' => '</div>',
        'before_title' => '<h2>',
        'after_title' => '</h2>',
    ) );
	
	// Footer widget three
	register_sidebar( array(
        'name' => __( 'Footer Widget  Three', 'magic-magazine' ),
        'id' => 'footer_widget_three',
        'before_widget' => '<div class="single_ft_wed">',
        'after_widget' => '</div>',
        'before_title' => '<h2>',
        'after_title' => '</h2>',
    ) );
	
	//Footer widget four
	register_sidebar( array(
        'name' => __( 'Footer Widget Four', 'magic-magazine' ),
        'id' => 'footer_widget_four',
        'before_widget' => '<div class="single_ft_wed">',
        'after_widget' => '</div>',
        'before_title' => '<h2>',
        'after_title' => '</h2>',
    ) );
	
}

add_action('widgets_init', 'magic_magazine_widget_area');

/**
 * Functions for magic magazine post tags
 *
 * @since Magic Magazine 1.0
 */

add_filter('widget_tag_cloud_args','magic_magazine_tag_cloud_sizes');
function magic_magazine_tag_cloud_sizes($args) {
$args['smallest'] = 10;
$args['largest'] = 12;
$args['number'] = 20;
return $args; }

/**
 * Function for magic magazine comment reply
 *
 * @since Magic Magazine 1.0
 */
function magic_magazine_comments_reply() {
	if( get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'comment_form_before', 'magic_magazine_comments_reply' );


