<?php

/**
 * Theme functions and definitions.
 *
 * @link https://codex.wordpress.org/Functions_File_Explained
 *
 * @package Magazine_Plus
 */

if (! function_exists('magazine_plus_setup')) :

	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function magazine_plus_setup()
	{

		// Make theme available for translation.
		load_theme_textdomain('magazine-plus', get_template_directory() . '/languages');

		// Add default posts and comments RSS feed links to head.
		add_theme_support('automatic-feed-links');

		// Let WordPress manage the document title.
		add_theme_support('title-tag');

		// Enable support for Post Thumbnails on posts and pages.
		add_theme_support('post-thumbnails');
		add_image_size('magazine-plus-thumb', 370, 250);

		// Register menu locations.
		register_nav_menus(array(
			'primary'  => esc_html__('Primary Menu', 'magazine-plus'),
			'top'      => esc_html__('Top Menu', 'magazine-plus'),
			'footer'   => esc_html__('Footer Menu', 'magazine-plus'),
			'social'   => esc_html__('Social Menu', 'magazine-plus'),
			'notfound' => esc_html__('404 Menu', 'magazine-plus'),
		));

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support('html5', array(
			'comment-form',
			'search-form',
			'comment-list',
			'gallery',
			'caption',
		));

		// Enable support for Post Formats.
		add_theme_support('post-formats', array(
			'aside',
			'image',
			'video',
			'quote',
			'link',
		));

		// Set up the WordPress core custom background feature.
		add_theme_support('custom-background', apply_filters('magazine_plus_custom_background_args', array(
			'default-color' => 'FFFFFF',
			'default-image' => '',
		)));

		// Enable support for selective refresh of widgets in Customizer.
		add_theme_support('customize-selective-refresh-widgets');

		// Enable support for custom logo.
		add_theme_support('custom-logo');

		// Load default block styles.
		add_theme_support('wp-block-styles');

		// Add support for responsive embeds.
		add_theme_support('responsive-embeds');

		// Enable support for footer widgets.
		add_theme_support('footer-widgets', 4);

		// Load Supports.
		require get_template_directory() . '/inc/support.php';

		global $magazine_plus_default_options;
		$magazine_plus_default_options = magazine_plus_get_default_theme_options();
	}
endif;

add_action('after_setup_theme', 'magazine_plus_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function magazine_plus_content_width()
{
	$GLOBALS['content_width'] = apply_filters('magazine_plus_content_width', 640);
}
add_action('after_setup_theme', 'magazine_plus_content_width', 0);

/**
 * Register widget area.
 */
function magazine_plus_widgets_init()
{
	register_sidebar(array(
		'name'          => esc_html__('Primary Sidebar', 'magazine-plus'),
		'id'            => 'sidebar-1',
		'description'   => esc_html__('Add widgets here to appear in your Primary Sidebar.', 'magazine-plus'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	));
	register_sidebar(array(
		'name'          => esc_html__('Secondary Sidebar', 'magazine-plus'),
		'id'            => 'sidebar-2',
		'description'   => esc_html__('Add widgets here to appear in your Secondary Sidebar.', 'magazine-plus'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	));
	register_sidebar(array(
		'name'          => esc_html__('Front Page Widget Area', 'magazine-plus'),
		'id'            => 'sidebar-front-page-widget-area',
		'description'   => esc_html__('Add widgets here to appear in Front Page Widget Area.', 'magazine-plus'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	));
	register_sidebar(array(
		'name'          => esc_html__('Header Right Widget Area', 'magazine-plus'),
		'id'            => 'header-right',
		'description'   => esc_html__('Add widgets here to appear in Header Right Widget Area.', 'magazine-plus'),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	));
}
add_action('widgets_init', 'magazine_plus_widgets_init');

/**
 * Enqueue scripts and styles.
 */
function magazine_plus_scripts()
{

	$min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

	wp_enqueue_style('font-awesome', get_template_directory_uri() . '/third-party/font-awesome/css/all' . $min . '.css', '', '6.7.2');

	wp_enqueue_style('magazine-plus-google-fonts', magazine_plus_fonts_url(), array(), null);

	wp_enqueue_style('magazine-plus-style', get_stylesheet_uri(), null, date('Ymd-Gis', filemtime(get_template_directory() . '/style.css')));

	wp_enqueue_script('magazine-plus-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix' . $min . '.js', array(), '20130115', true);

	wp_enqueue_script('magazine-plus-navigation', get_template_directory_uri() . '/js/navigation' . $min . '.js', array(), '20200527', true);

	wp_localize_script('magazine-plus-navigation', 'MagazinePlusScreenReaderText', array(
		'expand'   => '<span class="screen-reader-text">' . esc_html__('expand child menu', 'magazine-plus') . '</span>',
		'collapse' => '<span class="screen-reader-text">' . esc_html__('collapse child menu', 'magazine-plus') . '</span>',
	));

	wp_enqueue_script('jquery-easytabs', get_template_directory_uri() . '/third-party/easytabs/js/jquery.easytabs' . $min . '.js', array('jquery'), '3.2.0', true);

	wp_enqueue_script('jquery-cycle2', get_template_directory_uri() . '/third-party/cycle2/js/jquery.cycle2' . $min . '.js', array('jquery'), '2.1.6', true);

	wp_enqueue_script('jquery-easy-ticker', get_template_directory_uri() . '/third-party/ticker/jquery.easy-ticker' . $min . '.js', array('jquery'), '2.0', true);

	wp_enqueue_script('magazine-plus-custom', get_template_directory_uri() . '/js/custom' . $min . '.js', array('jquery'), '1.0.4', true);

	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}
add_action('wp_enqueue_scripts', 'magazine_plus_scripts');

/**
 * Enqueue admin scripts and styles.
 */
function magazine_plus_admin_scripts($hook)
{

	$min = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

	if ('widgets.php' === $hook) {
		wp_enqueue_style('wp-color-picker');
		wp_enqueue_script('wp-color-picker');
		wp_enqueue_media();
		wp_enqueue_style('magazine-plus-custom-widgets-style', get_template_directory_uri() . '/css/widgets' . $min . '.css', array(), '1.0.0');
		wp_enqueue_script('magazine-plus-custom-widgets', get_template_directory_uri() . '/js/widgets' . $min . '.js', array('jquery'), '1.0.0', true);
	}
}
add_action('admin_enqueue_scripts', 'magazine_plus_admin_scripts');

if (! function_exists('magazine_plus_blocks_support')) :
	/**
	 * Create add default blocks support
	 */
	function magazine_plus_blocks_support()
	{
		// Add support for Block Styles.
		add_theme_support('wp-block-styles');

		// Add support for full and wide align images.
		add_theme_support('align-wide');

		// Add support for editor styles.
		add_theme_support('editor-styles');

		// Add support for responsive embeds.
		add_theme_support('responsive-embeds');

		// Add custom editor font sizes.
		add_theme_support(
			'editor-font-sizes',
			array(
				array(
					'name'      => esc_html__('Small', 'magazine-plus'),
					'shortName' => esc_html__('S', 'magazine-plus'),
					'size'      => 14,
					'slug'      => 'small',
				),
				array(
					'name'      => esc_html__('Normal', 'magazine-plus'),
					'shortName' => esc_html__('M', 'magazine-plus'),
					'size'      => 18,
					'slug'      => 'normal',
				),
				array(
					'name'      => esc_html__('Large', 'magazine-plus'),
					'shortName' => esc_html__('L', 'magazine-plus'),
					'size'      => 42,
					'slug'      => 'large',
				),
				array(
					'name'      => esc_html__('Huge', 'magazine-plus'),
					'shortName' => esc_html__('XL', 'magazine-plus'),
					'size'      => 54,
					'slug'      => 'huge',
				),
			)
		);

		// Add support for custom color scheme.
		add_theme_support('editor-color-palette', array(
			array(
				'name'  => esc_html__('White', 'magazine-plus'),
				'slug'  => 'white',
				'color' => '#ffffff',
			),
			array(
				'name'  => esc_html__('Black', 'magazine-plus'),
				'slug'  => 'black',
				'color' => '#111111',
			),
			array(
				'name'  => esc_html__('Gray', 'magazine-plus'),
				'slug'  => 'gray',
				'color' => '#f4f4f4',
			),
			array(
				'name'  => esc_html__('Blue', 'magazine-plus'),
				'slug'  => 'blue',
				'color' => '#1b8be0',
			),
			array(
				'name'  => esc_html__('Orange', 'magazine-plus'),
				'slug'  => 'orange',
				'color' => '#fa5742',
			),
		));
	}
	add_action('after_setup_theme', 'magazine_plus_blocks_support', 20);
endif; //magazine_plus_blocks_support

if (! function_exists('magazine_plus_add_blocks_style')) :
	/**
	 * Add Blocks Style
	 */
	function magazine_plus_add_blocks_style()
	{
		// Theme block stylesheet.
		wp_enqueue_style('magazine-plus-block-style', get_theme_file_uri('/css/blocks.css'), array('magazine-plus-style'), date('Ymd-Gis', filemtime(get_template_directory() . '/css/blocks.css')));
	}
	add_action('wp_enqueue_scripts', 'magazine_plus_add_blocks_style');
endif; //magazine_plus_add_blocks_style

if (! function_exists('magazine_plus_block_editor_styles')) :
	/**
	 * Enqueue editor styles for Blocks
	 */
	function magazine_plus_block_editor_styles()
	{
		// Block styles.
		wp_enqueue_style('magazine-plus-block-editor-style', get_theme_file_uri('/css/editor-blocks.css'), null, date('Ymd-Gis', filemtime(get_template_directory() . '/css/editor-blocks.css')));

		// Add custom fonts.
		wp_enqueue_style('magazine-plus-fonts', magazine_plus_fonts_url(), array(), null);
	}
	add_action('enqueue_block_editor_assets', 'magazine_plus_block_editor_styles');
endif; //magazine_plus_block_editor_styles

/**
 * Load init.
 */
require_once get_template_directory() . '/inc/init.php';
