<?php
/**
 * WooCommerce Compatibility File
 *
 * @link https://woocommerce.com/
 *
 * @package Lyrielle
 */

require get_template_directory() . '/inc/woocommerce_sidebar.php'; // phpcs:ignore WPThemeReview.CoreFunctionality.FileInclude.FileIncludeFound

/**
 * WooCommerce setup function.
 *
 * @link https://docs.woocommerce.com/document/third-party-custom-theme-compatibility/
 * @link https://github.com/woocommerce/woocommerce/wiki/Enabling-product-gallery-features-(zoom,-swipe,-lightbox)-in-3.0.0
 *
 * @return void
 */
function lyrielle_woocommerce_setup() {
	add_theme_support( 'woocommerce' );
	add_theme_support( 'wc-product-gallery-zoom' );
	add_theme_support( 'wc-product-gallery-lightbox' );
	add_theme_support( 'wc-product-gallery-slider' );
}
add_action( 'after_setup_theme', 'lyrielle_woocommerce_setup' );

/**
 * WooCommerce specific scripts & stylesheets.
 *
 * @return void
 */
function lyrielle_woocommerce_scripts() {
	wp_enqueue_style( 'lyrielle_woocommerce_style', get_template_directory_uri() . '/css/woocommerce.min.css' );

	$font_path   = WC()->plugin_url() . '/assets/fonts/';
	$inline_font = '@font-face {
			font-family: "star";
			src: url("' . $font_path . 'star.eot");
			src: url("' . $font_path . 'star.eot?#iefix") format("embedded-opentype"),
				url("' . $font_path . 'star.woff") format("woff"),
				url("' . $font_path . 'star.ttf") format("truetype"),
				url("' . $font_path . 'star.svg#star") format("svg");
			font-weight: normal;
			font-style: normal;
		}

		@font-face {
			font-family: "WooCommerce";
			src: url("' . $font_path . 'WooCommerce.eot");
			src: url("' . $font_path . 'WooCommerce.eot?#iefix") format("embedded-opentype"),
				url("' . $font_path . 'WooCommerce.woff") format("woff"),
				url("' . $font_path . 'WooCommerce.ttf") format("truetype"),
				url("' . $font_path . 'WooCommerce.svg#WooCommerce") format("svg");
			font-weight: normal;
			font-style: normal;
		}';

	wp_add_inline_style( 'lyrielle_woocommerce_style', $inline_font );
}
add_action( 'wp_enqueue_scripts', 'lyrielle_woocommerce_scripts' );

/**
 * Disable the default WooCommerce stylesheet.
 *
 * Removing the default WooCommerce stylesheet and enqueing your own will
 * protect you during WooCommerce core updates.
 *
 * @link https://docs.woocommerce.com/document/disable-the-default-stylesheet/
 */
add_filter( 'woocommerce_enqueue_styles', '__return_empty_array' );

/**
 * Add 'woocommerce-active' class to the body tag.
 *
 * @param  array $classes CSS classes applied to the body tag.
 * @return array $classes modified to include 'woocommerce-active' class.
 */
function lyrielle_woocommerce_active_body_class( $classes ) {
	$classes[] = 'woocommerce-active';

	return $classes;
}
add_filter( 'body_class', 'lyrielle_woocommerce_active_body_class' );

/**
 * Products per page.
 *
 * @return integer number of products.
 */
function lyrielle_woocommerce_products_per_page() {
	return 6;
}
add_filter( 'loop_shop_per_page', 'lyrielle_woocommerce_products_per_page' );

/**
 * Product gallery thumnbail columns.
 *
 * @return integer number of columns.
 */
function lyrielle_woocommerce_thumbnail_columns() {
	return 4;
}
add_filter( 'woocommerce_product_thumbnails_columns', 'lyrielle_woocommerce_thumbnail_columns' );

/**
 * Default loop columns on product archives.
 *
 * @return integer products per row.
 */
function lyrielle_woocommerce_loop_columns() {
	return 4;
}
add_filter( 'loop_shop_columns', 'lyrielle_woocommerce_loop_columns' );

/**
 * Related Products Args.
 *
 * @param array $args related products args.
 * @return array $args related products args.
 */
function lyrielle_woocommerce_related_products_args( $args ) {
	$defaults = array(
		'posts_per_page' => 3,
		'columns'        => 3,
	);

	$args = wp_parse_args( $defaults, $args );

	return $args;
}

add_filter( 'woocommerce_output_related_products_args', 'lyrielle_woocommerce_related_products_args' );

if ( ! function_exists( 'lyrielle_woocommerce_product_columns_wrapper' ) ) {
	/**
	 * Product columns wrapper.
	 *
	 * @return  void
	 */
	function lyrielle_woocommerce_product_columns_wrapper() {
		$columns = lyrielle_woocommerce_loop_columns();
		echo '<div class="columns-' . absint( $columns ) . '">';
	}
}
add_action( 'woocommerce_before_shop_loop', 'lyrielle_woocommerce_product_columns_wrapper', 40 );

if ( ! function_exists( 'lyrielle_woocommerce_product_columns_wrapper_close' ) ) {
	/**
	 * Product columns wrapper close.
	 *
	 * @return  void
	 */
	function lyrielle_woocommerce_product_columns_wrapper_close() {
		echo '</div>';
	}
}
add_action( 'woocommerce_after_shop_loop', 'lyrielle_woocommerce_product_columns_wrapper_close', 40 );

/**
 * Remove default WooCommerce wrapper.
 */
remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10 );
remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10 );

if ( ! function_exists( 'lyrielle_woocommerce_wrapper_before' ) ) {
	/**
	 * Before Content.
	 *
	 * Wraps all WooCommerce content in wrappers which match the theme markup.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_before() {
		?>
		<div id="primary" class="content-area container">
			<main id="main" class="site-main" role="main">
			<?php
	}
}
add_action( 'woocommerce_before_main_content', 'lyrielle_woocommerce_wrapper_before' );

if ( ! function_exists( 'lyrielle_woocommerce_wrapper_after' ) ) {
	/**
	 * After Content.
	 *
	 * Closes the wrapping divs.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_after() {
			?>
			<?php get_sidebar() ?>
			</main><!-- #main -->
		</div><!-- #primary -->
		<?php
	}
}
add_action( 'woocommerce_after_main_content', 'lyrielle_woocommerce_wrapper_after' );

/**
 * WooCommerce wrapper for Shop.
 */
if ( ! function_exists( 'lyrielle_woocommerce_wrapper_shop_loop_before' ) ) {
	/**
	 * Before Shop.
	 *
	 * Wraps Shop in wrappers which match the theme markup.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_shop_loop_before() {
		?>
		<div class="shop-wrapper">
			<div class="shop-loop">
		<?php
	}
}
add_action( 'woocommerce_before_shop_loop', 'lyrielle_woocommerce_wrapper_shop_loop_before' );

if ( ! function_exists( 'lyrielle_woocommerce_wrapper_shop_loop_after' ) ) {
	/**
	 * After Shop.
	 *
	 * Closes the wrapping divs.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_shop_loop_after() {
			?>
			</div>
		</div><!-- .shop-wrapper -->
		<?php get_sidebar(); ?>
		<?php
	}
}
add_action( 'woocommerce_after_shop_loop', 'lyrielle_woocommerce_wrapper_shop_loop_after' );

/**
 * WooCommerce wrapper for Single Product.
 */
if ( ! function_exists( 'lyrielle_woocommerce_wrapper_product_before' ) ) {
	/**
	 * Before Single product.
	 *
	 * Wraps Sibgle Product in wrappers which match the theme markup.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_product_before() {
		?>
		<div class="single-product-wrapper">
		<?php
	}
}
add_action( 'woocommerce_before_main_content', 'lyrielle_woocommerce_wrapper_product_before' );

if ( ! function_exists( 'lyrielle_woocommerce_wrapper_product_after' ) ) {
	/**
	 * After Single Product.
	 *
	 * Closes the wrapping divs.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_product_after() {
		?>
		<?php get_sidebar(); ?>
		</div><!-- .single-product-wrapper -->
		<?php
	}
}
add_action( 'woocommerce_after_main_content', 'lyrielle_woocommerce_wrapper_product_after' );

/**
 * Sample implementation of the WooCommerce Mini Cart.
 *
 * You can add the WooCommerce Mini Cart to header.php like so ...
 */
if ( function_exists( 'lyrielle_woocommerce_header_cart' ) ) {
	lyrielle_woocommerce_header_cart();
}

/**
 * WooCommerce wrapper for Shop.
 */
if ( ! function_exists( 'lyrielle_woocommerce_wrapper_cart_before' ) ) {
	/**
	 * Before Shop.
	 *
	 * Wraps Shop in wrappers which match the theme markup.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_cart_before() {
		?>
		<div class="cart-wrapper">
		<?php
	}
}
add_action( 'woocommerce_before_cart', 'lyrielle_woocommerce_wrapper_cart_before' );

if ( ! function_exists( 'lyrielle_woocommerce_wrapper_cart_after' ) ) {
	/**
	 * After Shop.
	 *
	 * Closes the wrapping divs.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_wrapper_cart_after() {
			?>
		</div><!-- .cart-wrapper -->
		<?php
	}
}
add_action( 'woocommerce_after_cart', 'lyrielle_woocommerce_wrapper_cart_after' );

if ( ! function_exists( 'lyrielle_woocommerce_cart_link_fragment' ) ) {
	/**
	 * Cart Fragments.
	 *
	 * Ensure cart contents update when products are added to the cart via AJAX.
	 *
	 * @param array $fragments Fragments to refresh via AJAX.
	 * @return array Fragments to refresh via AJAX.
	 */
	function lyrielle_woocommerce_cart_link_fragment( $fragments ) {
		ob_start();
		lyrielle_woocommerce_cart_link();
		$fragments['a.cart-contents'] = ob_get_clean();

		return $fragments;
	}
}
add_filter( 'woocommerce_add_to_cart_fragments', 'lyrielle_woocommerce_cart_link_fragment' );

if ( ! function_exists( 'lyrielle_woocommerce_cart_link' ) ) {
	/**
	 * Cart Link.
	 *
	 * Displayed a link to the cart including the number of items present and the cart total.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_cart_link() {
		?>
		<a class="cart-contents" href="<?php echo esc_url( wc_get_cart_url() ); ?>" title="<?php esc_attr_e( 'View your shopping cart', 'lyrielle' ); ?>">
			<?php
			$item_count_text = sprintf(
				/* translators: number of items in the mini cart. */
				_n( '%d item', '%d items', WC()->cart->get_cart_contents_count(), 'lyrielle' ),
				WC()->cart->get_cart_contents_count()
			);
			?>
			<span class="amount"><?php echo wp_kses_data( WC()->cart->get_cart_subtotal() ); ?></span> <span class="count"><?php echo esc_html( $item_count_text ); ?></span>
		</a>
		<?php
	}
}

if ( ! function_exists( 'lyrielle_woocommerce_header_cart' ) ) {
	/**
	 * Display Header Cart.
	 *
	 * @return void
	 */
	function lyrielle_woocommerce_header_cart() {
		if ( is_cart() ) {
			$class = 'current-menu-item';
		} else {
			$class = '';
		}
		?>
		<ul id="site-header-cart" class="site-header-cart">
			<li class="<?php echo esc_attr( $class ); ?>">
				<?php lyrielle_woocommerce_cart_link(); ?>
			</li>
			<li>
				<?php
				$instance = array(
					'title' => '',
				);

				the_widget( 'WC_Widget_Cart', $instance );
				?>
			</li>
		</ul>
		<?php
	}
}

// remove from storefront.
add_action( 'get_header', 'lyrielle_remove_storefront_sidebar' );

function lyrielle_remove_storefront_sidebar() {
    if ( is_product() ) {
        remove_action( 'storefront_sidebar', 'storefront_get_sidebar', 10 );
    }
}

/**
 * Modify review form
 */
add_filter( 'woocommerce_product_review_comment_form_args', function ( $comment_form ) {
	$comment_form['comment_field'] = '';
	if ( get_option( 'woocommerce_enable_review_rating' ) === 'yes' ) {
		$comment_form['comment_field'] = '<div class="comment-form-rating"><label for="rating">' . esc_html__( 'Your rating', 'lyrielle' ) . '</label><select name="rating" id="rating" required>
			<option value="">' . esc_html__( 'Rate&hellip;', 'lyrielle' ) . '</option>
			<option value="5">' . esc_html__( 'Perfect', 'lyrielle' ) . '</option>
			<option value="4">' . esc_html__( 'Good', 'lyrielle' ) . '</option>
			<option value="3">' . esc_html__( 'Average', 'lyrielle' ) . '</option>
			<option value="2">' . esc_html__( 'Not that bad', 'lyrielle' ) . '</option>
			<option value="1">' . esc_html__( 'Very poor', 'lyrielle' ) . '</option>
		</select></div>';
	}

	$comment_form['comment_field'] .= '<p class="comment-form-comment"><textarea id="comment" name="comment" cols="45" rows="8" placeholder="'. esc_attr_x( 'Your Review', 'woocommerce', 'lyrielle' ) .'" aria-required="true" required></textarea></p>';
	$comment_form['submit_button'] = '<button name="%1$s" type="submit" id="%2$s" class="%3$s woo-review-submit" value="%4$s" />%4$s</button>';

	$comment_form['fields']['author'] = '<p class="comment-form-author"><input id="author" name="author" type="text" value="" size="30" placeholder="'. esc_attr_x( 'Name', 'woocommerce', 'lyrielle' ) .'" aria-required="true" required /></p>';
	$comment_form['fields']['email'] = '<p class="comment-form-email"> <input id="email" name="email" type="email" value="" size="30" placeholder="'. esc_attr_x( 'Email', 'woocommerce', 'lyrielle' ) .'" aria-required="true" required /></p>';

	return $comment_form;
}, 10, 1);
