<?php
/**
 * Popular Category Posts Widget.
 */

defined( 'ABSPATH' ) || exit;

class lyrielle_Popular_Category_Posts_Widget extends WP_Widget {

	/**
	 * Register widget and shortcode.
	 */
	public function __construct() {

		$widget_ops = array(
			'classname'   => 'widget-popular-category-posts',
			'description' => esc_html( 'The widget shows posts from the Popular category.', 'lyrielle' ),
		);

		$control_ops = array(
			'id_base' => 'lyrielle_popular_category_posts_widget',
		);

		parent::__construct( 'lyrielle_popular_category_posts_widget', 'Popular Category Posts Widget', $widget_ops, $control_ops );
	}

	/**
	 * Front-end display of widget.
	 *
	 * @see WP_Widget::widget()
	 *
	 * @param array $args     Widget arguments.
	 * @param array $instance Saved values from database.
	 */
	public function widget( $args, $instance ) {
		echo $args['before_widget'];

		if ( ! empty( $instance['title'] ) ) {
			echo $args['before_title'] . apply_filters( 'widget_title', $instance['title'] ) . $args['after_title'];
		}

		echo $this->get_popular_category_posts();

		echo $args['after_widget'];
	}

	/**
	 * Get popular category posts output.
	 *
	 * @return string
	 */
	public function get_popular_category_posts( $atts = array() ) {
		ob_start();
	
		$atts = shortcode_atts(
			array(
				'title' => 'Popular Posts',
				'count' => 5,
			),
			$atts,
			'popular-category-posts'
		);
	
		// Find the most populated category
		$categories = get_categories( array(
			'orderby' => 'count',
			'order'   => 'DESC',
			'number'  => 1,
		) );
	
		if ( ! empty( $categories ) ) {
			$most_popular_category = $categories[0];
			$category_id = $most_popular_category->term_id;
			$category_name = $most_popular_category->name;
		} else {
			return '<!-- No categories found -->';
		}
	
		// Start the widget output
		?>
		<ul class="widget popular-category-posts">
			<h2 class="widget-title"><?php echo esc_html( $atts['title'] ) . ' - ' . esc_html( $category_name ); ?></h2>
			<?php 
			$query_args = array(
				'posts_per_page' => $atts['count'],
				'cat'            => $category_id, // Use the category ID directly here
				'order'          => 'DESC',
			);
	
			$the_query = new WP_Query( $query_args ); 
	
			if ( $the_query->have_posts() ) : 
				while ( $the_query->have_posts() ) : $the_query->the_post(); ?>
					<li class="widget-holder">
						<a href="<?php the_permalink(); ?>">
							<?php 
							// Check if the post has a thumbnail
							if ( has_post_thumbnail() ) {
								// Display the post thumbnail
								lyrielle_post_thumbnail();
							} else {
								// If no thumbnail is set
								echo '<span class=no-thumbnail></span>';
							}
							?>
							<div class="content">
								<div class="post-title"><?php the_title(); ?></div>
							</div>
						</a>
					</li>
				<?php endwhile; 
				wp_reset_postdata();
			else :
				echo '<!-- No posts found -->';
			endif;
			?>
		</ul>
		<?php
	
		return ob_get_clean();
	}

	/**
	 * Back-end widget form.
	 *
	 * @see WP_Widget::form()
	 *
	 * @param array $instance Previously saved values from database.
	 */
	public function form( $instance ) {

		// outputs the options form on admin
		$title = ! empty( $instance['title'] ) ? $instance['title'] : esc_html( 'Popular Posts', 'lyrielle' );
		?>
		<p>
			<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Title:', 'lyrielle' ); ?></label>
			<input class="widefat" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" type="text" value="<?php echo esc_attr( $title ); ?>">
		</p>
		<?php
	}

	/**
	 * Sanitize widget form values as they are saved.
	 *
	 * @see WP_Widget::update()
	 *
	 * @param array $new_instance Values just sent to be saved.
	 * @param array $old_instance Previously saved values from database.
	 *
	 * @return array Updated safe values to be saved.
	 */
	public function update( $new_instance, $old_instance ) {

		// processes widget options to be saved
		$instance = array();
		$instance['title'] = ( ! empty( $new_instance['title'] ) ) ? sanitize_text_field( $new_instance['title'] ) : '';

		return $instance;
	}
}

// Register Popular Category Posts Widget
function lyrielle_register_popular_category_posts_widget() {
	register_widget( 'lyrielle_Popular_Category_Posts_Widget' );
}
add_action( 'widgets_init', 'lyrielle_register_popular_category_posts_widget' );
?>