<?php
/**
 * SVG Icons class
 *
 * @package WordPress
 * @subpackage Liora
 * @since 1.0.0
 */

/**
 * This class is in charge of displaying SVG icons across the site.
 *
 * Place each <svg> source on its own array key, without adding the
 * both `width` and `height` attributes, since these are added dnamically,
 * before rendering the SVG code.
 *
 * All icons are assumed to have equal width and height, hence the option
 * to only specify a `$size` parameter in the svg methods.
 *
 * @since 1.0.0
 */
class liora_SVG_Icons {

	/**
	 * Gets the SVG code for a given icon.
	 */
	public static function get_svg( $group, $icon, $size ) {
		if ( 'ui' == $group ) {
			$arr = self::$ui_icons;
		} elseif ( 'social' == $group ) {
			$arr = self::$social_icons;
		}elseif ( 'social2' == $group ) {
			$arr = self::$social_icons;
		} else {
			$arr = array();
		}
		if ( array_key_exists( $icon, $arr ) ) {
			$repl = sprintf( '<svg width="%d" height="%d" aria-hidden="true" role="img" focusable="false" ', $size, $size );
			$svg  = preg_replace( '/^<svg /', $repl, trim( $arr[ $icon ] ) ); // Add extra attributes to SVG code.
			$svg  = preg_replace( "/([\n\t]+)/", ' ', $svg ); // Remove newlines & tabs.
			$svg  = preg_replace( '/>\s*</', '><', $svg ); // Remove white space between SVG tags.
			return $svg;
		}
		return null;
	}

	/**
	 * Detects the social network from a URL and returns the SVG code for its icon.
	 */
	public static function get_social_link_svg( $uri, $size ) {
		static $regex_map; // Only compute regex map once, for performance.
		if ( ! isset( $regex_map ) ) {
			$regex_map = array();
			$map       = &self::$social_icons_map; // Use reference instead of copy, to save memory.
			foreach ( array_keys( self::$social_icons ) as $icon ) {
				$domains            = array_key_exists( $icon, $map ) ? $map[ $icon ] : array( sprintf( '%s.com', $icon ) );
				$domains            = array_map( 'trim', $domains ); // Remove leading/trailing spaces, to prevent regex from failing to match.
				$domains            = array_map( 'preg_quote', $domains );
				$regex_map[ $icon ] = sprintf( '/(%s)/i', implode( '|', $domains ) );
			}
		}
		foreach ( $regex_map as $icon => $regex ) {
			if ( preg_match( $regex, $uri ) ) {
				return self::get_svg( 'social', $icon, $size );
				return self::get_svg( 'social2', $icon, $size );
			}
		}
		return null;
	}

	/**
	 * User Interface icons – svg sources.
	 *
	 * @var array
	 */
	static $ui_icons = array(
		'link'                   => /* material-design – link */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
    <path d="M0 0h24v24H0z" fill="none"></path>
    <path d="M3.9 12c0-1.71 1.39-3.1 3.1-3.1h4V7H7c-2.76 0-5 2.24-5 5s2.24 5 5 5h4v-1.9H7c-1.71 0-3.1-1.39-3.1-3.1zM8 13h8v-2H8v2zm9-6h-4v1.9h4c1.71 0 3.1 1.39 3.1 3.1s-1.39 3.1-3.1 3.1h-4V17h4c2.76 0 5-2.24 5-5s-2.24-5-5-5z"></path>
</svg>',

		'watch'                  => /* material-design – watch-later */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
    <defs>
        <path id="a" d="M0 0h24v24H0V0z"></path>
    </defs>
    <clipPath id="b">
        <use xlink:href="#a" overflow="visible"></use>
    </clipPath>
    <path clip-path="url(#b)" d="M12 2C6.5 2 2 6.5 2 12s4.5 10 10 10 10-4.5 10-10S17.5 2 12 2zm4.2 14.2L11 13V7h1.5v5.2l4.5 2.7-.8 1.3z"></path>
</svg>',

		'archive'                => /* material-design – folder */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
    <path d="M10 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2h-8l-2-2z"></path>
    <path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

		'tag'                    => /* material-design – local_offer */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
	<path d="M21.41 11.58l-9-9C12.05 2.22 11.55 2 11 2H4c-1.1 0-2 .9-2 2v7c0 .55.22 1.05.59 1.42l9 9c.36.36.86.58 1.41.58.55 0 1.05-.22 1.41-.59l7-7c.37-.36.59-.86.59-1.41 0-.55-.23-1.06-.59-1.42zM5.5 7C4.67 7 4 6.33 4 5.5S4.67 4 5.5 4 7 4.67 7 5.5 6.33 7 5.5 7z"></path>
	<path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

		'comment'                => /* material-design – comment */ '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <path d="M21.99 4c0-1.1-.89-2-1.99-2H4c-1.1 0-2 .9-2 2v12c0 1.1.9 2 2 2h14l4 4-.01-18z"></path>
    <path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

'comment_round'                => /* material-design – comment_round */ '
<svg xmlns="http://www.w3.org/2000/svg" class="bi bi-chat-dots" viewBox="0 0 16 16"> <path d="M5 8a1 1 0 1 1-2 0 1 1 0 0 1 2 0zm4 0a1 1 0 1 1-2 0 1 1 0 0 1 2 0zm3 1a1 1 0 1 0 0-2 1 1 0 0 0 0 2z"/> <path d="m2.165 15.803.02-.004c1.83-.363 2.948-.842 3.468-1.105A9.06 9.06 0 0 0 8 15c4.418 0 8-3.134 8-7s-3.582-7-8-7-8 3.134-8 7c0 1.76.743 3.37 1.97 4.6a10.437 10.437 0 0 1-.524 2.318l-.003.011a10.722 10.722 0 0 1-.244.637c-.079.186.074.394.273.362a21.673 21.673 0 0 0 .693-.125zm.8-3.108a1 1 0 0 0-.287-.801C1.618 10.83 1 9.468 1 8c0-3.192 3.004-6 7-6s7 2.808 7 6c0 3.193-3.004 6-7 6a8.06 8.06 0 0 1-2.088-.272 1 1 0 0 0-.711.074c-.387.196-1.24.57-2.634.893a10.97 10.97 0 0 0 .398-2z"/> </svg> ',

		'person'                 => /* material-design – person */ '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <path d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"></path>
    <path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

		'edit'                   => /* material-design – edit */ '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"></path>
    <path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

		'chevron_left'           => /* material-design – chevron_left */ '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
    <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
    <path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

		'chevron_right'          => /* material-design – chevron_right */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
    <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
    <path d="M0 0h24v24H0z" fill="none"></path>
</svg>',

		'check'                  => /* material-design – check */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
    <path d="M0 0h24v24H0z" fill="none"></path>
    <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"></path>
</svg>',

		'arrow_drop_down_circle' => /* material-design – arrow_drop_down_circle */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
	<path d="M0 0h24v24H0z" fill="none"></path>
	<path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 12l-4-4h8l-4 4z"></path>
</svg>',

		'keyboard_arrow_down'    => /* material-design – keyboard_arrow_down */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
	<path d="M7.41 8.59L12 13.17l4.59-4.58L18 10l-6 6-6-6 1.41-1.41z"></path>
	<path fill="none" d="M0 0h24v24H0V0z"></path>
</svg>',

		'keyboard_arrow_right'   => /* material-design – keyboard_arrow_right */ '
<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
	<path d="M10 17l5-5-5-5v10z"></path>
	<path fill="none" d="M0 24V0h24v24H0z"></path>
</svg>',

		'keyboard_arrow_left'   => /* material-design – keyboard_arrow_left */ '
<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24">
	<path d="M14 7l-5 5 5 5V7z"></path>
	<path fill="none" d="M24 0v24H0V0h24z"></path>
</svg>',

		'arrow_drop_down_ellipsis' => /* custom – arrow_drop_down_ellipsis */ '
<svg width="24" height="24" xmlns="http://www.w3.org/2000/svg">
    <g fill="none" fill-rule="evenodd">
        <path d="M0 0h24v24H0z"/>
        <path fill="currentColor" fill-rule="nonzero" d="M12 2c5.52 0 10 4.48 10 10s-4.48 10-10 10S2 17.52 2 12 6.48 2 12 2zM6 14a2 2 0 1 0 0-4 2 2 0 0 0 0 4zm6 0a2 2 0 1 0 0-4 2 2 0 0 0 0 4zm6 0a2 2 0 1 0 0-4 2 2 0 0 0 0 4z"/>
    </g>
</svg>',

/** Custom added icons here: **/

		'search' => 
'<svg viewBox="0 0 56.966 56.966" class="icon-search">
    <path d="M55.146,51.887L41.588,37.786c3.486-4.144,5.396-9.358,5.396-14.786c0-12.682-10.318-23-23-23s-23,10.318-23,23s10.318,23,23,23c4.761,0,9.298-1.436,13.177-4.162l13.661,14.208c0.571,0.593,1.339,0.92,2.162,0.92c0.779,0,1.518-0.297,2.079-0.837C56.255,54.982,56.293,53.08,55.146,51.887z M23.984,6c9.374,0,17,7.626,17,17s-7.626,17-17,17s-17-7.626-17-17S14.61,6,23.984,6z"></path>
</svg>',

		'close' => 
'<svg version="1.1" id="Capa_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
	 viewBox="0 0 56.966 56.966" style="enable-background:new 0 0 56.966 56.966;" xml:space="preserve">
	<path d="M55.146,51.887L41.588,37.786c3.486-4.144,5.396-9.358,5.396-14.786c0-12.682-10.318-23-23-23s-23,10.318-23,23
	s10.318,23,23,23c4.761,0,9.298-1.436,13.177-4.162l13.661,14.208c0.571,0.593,1.339,0.92,2.162,0.92
	c0.779,0,1.518-0.297,2.079-0.837C56.255,54.982,56.293,53.08,55.146,51.887z M23.984,6c9.374,0,17,7.626,17,17s-7.626,17-17,17
	s-17-7.626-17-17S14.61,6,23.984,6z"/>
</svg>',

		'menu_icon_up' => 
'<svg viewBox="0 0 284.929 284.929" class="dropdown-icon icon-up">
    <path d="M282.082,195.285L149.028,62.24c-1.901-1.903-4.088-2.856-6.562-2.856s-4.665,0.953-6.567,2.856L2.856,195.285C0.95,197.191,0,199.378,0,201.853c0,2.474,0.953,4.664,2.856,6.566l14.272,14.271c1.903,1.903,4.093,2.854,6.567,2.854c2.474,0,4.664-0.951,6.567-2.854l112.204-112.202l112.208,112.209c1.902,1.903,4.093,2.848,6.563,2.848c2.478,0,4.668-0.951,6.57-2.848l14.274-14.277c1.902-1.902,2.847-4.093,2.847-6.566C284.929,199.378,283.984,197.188,282.082,195.285z"/>
</svg>',

'menu_icon_down' => 
'<svg viewBox="0 0 129 129" class="dropdown-icon icon-down">
    <path d="m121.3,34.6c-1.6-1.6-4.2-1.6-5.8,0l-51,51.1-51.1-51.1c-1.6-1.6-4.2-1.6-5.8,0-1.6,1.6-1.6,4.2 0,5.8l53.9,53.9c0.8,0.8 1.8,1.2 2.9,1.2 1,0 2.1-0.4 2.9-1.2l53.9-53.9c1.7-1.6 1.7-4.2 0.1-5.8z"/>
</svg>',

'read_more' => 
'<svg version="1.1" viewBox="0 0 31.49 31.49">
    <path d="M21.205,5.007c-0.429-0.444-1.143-0.444-1.587,0c-0.429,0.429-0.429,1.143,0,1.571l8.047,8.047H1.111
    C0.492,14.626,0,15.118,0,15.737c0,0.619,0.492,1.127,1.111,1.127h26.554l-8.047,8.032c-0.429,0.444-0.429,1.159,0,1.587
    c0.444,0.444,1.159,0.444,1.587,0l9.952-9.952c0.444-0.429,0.444-1.143,0-1.571L21.205,5.007z"></path>
</svg>',

'play' => 
'<svg viewBox="0 0 41.999 41.999">
    <path d="M36.068,20.176l-29-20C6.761-0.035,6.363-0.057,6.035,0.114C5.706,0.287,5.5,0.627,5.5,0.999v40
    c0,0.372,0.206,0.713,0.535,0.886c0.146,0.076,0.306,0.114,0.465,0.114c0.199,0,0.397-0.06,0.568-0.177l29-20
    c0.271-0.187,0.432-0.494,0.432-0.823S36.338,20.363,36.068,20.176z"/>
</svg>',
'sun' => 
'<svg viewBox="0 0 457.32 457.32">
	<path id="XMLID_922_" d="M228.66,112.692c-63.945,0-115.968,52.022-115.968,115.967c0,63.945,52.023,115.968,115.968,115.968
		s115.968-52.023,115.968-115.968C344.628,164.715,292.605,112.692,228.66,112.692z"/>
	<path id="XMLID_1397_" d="M401.429,228.66l42.467-57.07c2.903-3.9,3.734-8.966,2.232-13.59c-1.503-4.624-5.153-8.233-9.794-9.683
		l-67.901-21.209l0.811-71.132c0.056-4.862-2.249-9.449-6.182-12.307c-3.934-2.858-9.009-3.633-13.615-2.077l-67.399,22.753
		L240.895,6.322C238.082,2.356,233.522,0,228.66,0c-4.862,0-9.422,2.356-12.235,6.322l-41.154,58.024l-67.4-22.753
		c-4.607-1.555-9.682-0.781-13.615,2.077c-3.933,2.858-6.238,7.445-6.182,12.307l0.812,71.132l-67.901,21.209
		c-4.641,1.45-8.291,5.059-9.793,9.683c-1.503,4.624-0.671,9.689,2.232,13.59l42.467,57.07l-42.467,57.07
		c-2.903,3.9-3.734,8.966-2.232,13.59c1.502,4.624,5.153,8.233,9.793,9.683l67.901,21.208l-0.812,71.132
		c-0.056,4.862,2.249,9.449,6.182,12.307c3.934,2.857,9.007,3.632,13.615,2.077l67.4-22.753l41.154,58.024
		c2.813,3.966,7.373,6.322,12.235,6.322c4.862,0,9.422-2.356,12.235-6.322l41.154-58.024l67.399,22.753
		c4.606,1.555,9.681,0.781,13.615-2.077c3.933-2.858,6.238-7.445,6.182-12.306l-0.811-71.133l67.901-21.208
		c4.641-1.45,8.291-5.059,9.794-9.683c1.502-4.624,0.671-9.689-2.232-13.59L401.429,228.66z M228.66,374.627
		c-80.487,0-145.968-65.481-145.968-145.968S148.173,82.692,228.66,82.692s145.968,65.48,145.968,145.967
		S309.147,374.627,228.66,374.627z"/>
</svg>',

'moon' => 
'<svg viewBox="0 0 24 24">
<path d="M12 22C17.5228 22 22 17.5228 22 12C22 11.5373 21.3065 11.4608 21.0672 11.8568C19.9289 13.7406 17.8615 15 15.5 15C11.9101 15 9 12.0899 9 8.5C9 6.13845 10.2594 4.07105 12.1432 2.93276C12.5392 2.69347 12.4627 2 12 2C6.47715 2 2 6.47715 2 12C2 17.5228 6.47715 22 12 22Z"/>
</svg>',

'cloud_down' =>
'<svg class="svg-cloud-2" viewBox="0 0 1428 130.87"><path class="cls-1" d="M-83,345V288s158-27,262-22l104,5,138,8,300,21s245,4,281-5c0,0-9-39,33-45,0,0,8-51,58-25,0,0,48-36,61,23,0,0,27,2,28,24,0,0,163-22,163-17v90Z" transform="translate(83 -214.13)"></path></svg>',

'bg_cloud' => 
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="2000px" height="117px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path d="M 1999.5,94.5 C 1999.5,101.833 1999.5,109.167 1999.5,116.5C 1332.83,116.5 666.167,116.5 -0.5,116.5C -0.5,99.8333 -0.5,83.1667 -0.5,66.5C 3.1949,68.8694 5.6949,72.2027 7,76.5C 8.91596,81.6494 9.91596,86.9828 10,92.5C 15.1393,94.4011 19.8059,97.0678 24,100.5C 27.7612,96.0615 31.5946,91.5615 35.5,87C 40.1191,83.7748 44.9525,80.9414 50,78.5C 50.3881,51.4542 63.5547,34.2875 89.5,27C 111.335,23.1814 128.835,30.0147 142,47.5C 144.072,50.9712 145.572,54.6379 146.5,58.5C 157.459,58.7301 166.292,63.0634 173,71.5C 191.669,53.1649 212.835,49.9983 236.5,62C 248.815,70.452 256.815,81.952 260.5,96.5C 266.39,94.9694 272.056,95.6361 277.5,98.5C 287.225,85.0156 300.559,79.0156 317.5,80.5C 317.645,56.4399 329.645,43.1066 353.5,40.5C 364.198,41.1016 373.365,45.1016 381,52.5C 399.312,41.3133 417.979,40.6467 437,50.5C 449.388,39.1976 462.888,37.6976 477.5,46C 486.115,53.5018 490.282,63.0018 490,74.5C 495.139,76.4011 499.806,79.0678 504,82.5C 507.761,78.0615 511.595,73.5615 515.5,69C 520.119,65.7748 524.952,62.9414 530,60.5C 530.388,33.4542 543.555,16.2875 569.5,9C 591.335,5.18139 608.835,12.0147 622,29.5C 624.072,32.9712 625.572,36.6379 626.5,40.5C 637.459,40.7301 646.292,45.0634 653,53.5C 671.669,35.1649 692.835,31.9983 716.5,44C 728.815,52.452 736.815,63.952 740.5,78.5C 746.39,76.9694 752.056,77.6361 757.5,80.5C 769.576,64.107 785.243,59.607 804.5,67C 814.1,72.0974 820.1,79.9307 822.5,90.5C 836.691,86.8978 849.691,89.3978 861.5,98C 864.273,100.383 866.439,103.217 868,106.5C 879.029,98.0718 891.029,96.4052 904,101.5C 905.579,99.9588 907.413,98.7922 909.5,98C 913.541,97.8265 917.541,97.3265 921.5,96.5C 925.184,86.3102 932.518,80.9769 943.5,80.5C 943.645,56.4399 955.645,43.1066 979.5,40.5C 990.198,41.1016 999.365,45.1016 1007,52.5C 1025.31,41.3133 1043.98,40.6467 1063,50.5C 1075.39,39.1976 1088.89,37.6976 1103.5,46C 1112.12,53.5018 1116.28,63.0018 1116,74.5C 1121.14,76.4011 1125.81,79.0678 1130,82.5C 1133.76,78.0615 1137.59,73.5615 1141.5,69C 1146.12,65.7748 1150.95,62.9414 1156,60.5C 1156.39,33.4542 1169.55,16.2875 1195.5,9C 1217.34,5.18139 1234.84,12.0147 1248,29.5C 1250.07,32.9712 1251.57,36.6379 1252.5,40.5C 1263.46,40.7301 1272.29,45.0634 1279,53.5C 1297.67,35.1649 1318.84,31.9983 1342.5,44C 1354.82,52.452 1362.82,63.952 1366.5,78.5C 1372.39,76.9694 1378.06,77.6361 1383.5,80.5C 1395.58,64.107 1411.24,59.607 1430.5,67C 1440.1,72.0974 1446.1,79.9307 1448.5,90.5C 1464.18,86.3516 1478.01,89.6849 1490,100.5C 1503.15,80.3523 1521.32,73.8523 1544.5,81C 1550.29,83.98 1555.46,87.8134 1560,92.5C 1572.69,81.5437 1587.53,76.2104 1604.5,76.5C 1625.75,46.4671 1654.42,35.6338 1690.5,44C 1710.87,51.0428 1726.37,63.8761 1737,82.5C 1748,76.9379 1759.5,75.6046 1771.5,78.5C 1787.57,42.3009 1814.91,29.4676 1853.5,40C 1860.1,43.2136 1866.27,47.0469 1872,51.5C 1889.95,39.3652 1907.78,39.5318 1925.5,52C 1929.06,55.0584 1932.06,58.5584 1934.5,62.5C 1962.89,60.1971 1984.56,70.8638 1999.5,94.5 Z"/></g>
</svg>',

'long_cloud' =>
'<svg version="1.0" viewBox="0 0 1280.000000 640.000000"  width="1920px" height="1920px">
<g transform="translate(0.000000,640.000000) scale(0.100000,-0.100000)">
<path d="M12450 4128 c-7 -46 -16 -90 -21 -99 -4 -8 -46 -33 -94 -54 l-86 -38
43 -25 c24 -14 45 -27 47 -29 3 -2 -11 -16 -30 -32 -70 -58 -119 -195 -89
-250 17 -34 6 -36 -33 -7 -76 58 -191 73 -272 35 -109 -51 -211 -228 -275
-477 l-11 -44 -37 9 c-78 17 -222 8 -321 -21 l-94 -27 -24 23 c-22 21 -34 23
-126 23 -116 0 -138 -8 -204 -75 -81 -80 -109 -192 -71 -281 l16 -36 -39 9
c-85 22 -158 -15 -196 -99 -21 -46 -27 -77 -31 -159 -3 -56 -2 -118 2 -138 l7
-36 648 0 c356 0 823 3 1037 7 l390 6 2 131 c1 72 5 402 8 734 4 331 10 602
13 602 3 -1 45 -9 94 -19 48 -10 87 -15 87 -10 0 5 -19 46 -41 92 l-42 83 47
83 c26 46 46 85 43 87 -2 2 -46 -2 -98 -10 l-94 -14 -63 69 c-35 38 -67 69
-72 69 -4 0 -13 -37 -20 -82z"/>
<path d="M5913 3615 c-114 -35 -218 -135 -285 -272 -38 -79 -42 -84 -55 -67
-43 62 -196 168 -263 183 -230 53 -513 -87 -737 -363 l-32 -39 -48 40 c-143
117 -291 182 -467 203 -97 11 -256 1 -313 -21 -33 -12 -29 2 17 57 l40 49 -68
3 -68 3 -38 62 -39 62 -25 -66 -24 -66 -60 -17 c-42 -12 -63 -14 -68 -6 -4 7
-16 7 -38 -1 -30 -10 -35 -8 -84 31 l-53 42 -3 -59 -3 -60 -90 -27 c-106 -31
-78 -43 -259 102 l-125 100 -7 -47 c-3 -25 -7 -110 -7 -188 l-1 -143 -155
-102 c-85 -57 -155 -105 -155 -108 0 -3 81 -32 179 -65 l179 -59 27 -103 c23
-85 26 -108 16 -130 -22 -54 -41 -169 -41 -250 l0 -83 853 0 c601 0 856 -3
864 -11 8 -8 13 -8 17 0 5 8 706 11 2467 11 l2460 0 5 30 c4 16 17 36 29 44
29 19 62 88 70 147 13 86 -43 247 -103 300 -29 26 -41 30 -70 26 -19 -4 -47
-16 -62 -29 -112 -92 -135 -109 -156 -114 -13 -3 -30 -3 -36 0 -60 30 -109 71
-140 117 -20 30 -46 66 -58 82 -25 31 -30 29 90 52 46 8 86 17 87 19 2 2 -13
18 -34 36 -21 18 -55 48 -75 66 l-37 33 11 91 c6 49 9 93 5 96 -3 3 -40 -15
-82 -40 -42 -25 -80 -46 -85 -46 -5 0 -44 16 -87 36 -43 20 -85 37 -93 38 -8
1 -39 14 -68 29 -28 15 -72 30 -96 33 -97 14 -270 -67 -429 -199 -26 -22 -49
-38 -51 -36 -1 2 -15 29 -29 61 -66 137 -237 215 -384 175 -52 -14 -53 -14
-69 11 -28 42 -66 74 -124 101 -81 37 -193 37 -272 -2 -75 -37 -175 -143 -222
-237 -20 -40 -37 -66 -40 -59 -2 8 -9 37 -15 64 -18 79 -56 148 -93 170 -28
17 -53 20 -157 22 l-124 2 -16 48 c-38 112 -127 195 -250 233 -73 23 -184 25
-248 5z m-2534 -296 c-16 -10 -23 -4 -14 10 3 6 11 8 17 5 6 -4 5 -9 -3 -15z"/>
<path d="M241 3373 c-97 -34 -172 -155 -212 -344 -20 -91 -23 -133 -23 -339 0
-205 11 -370 31 -447 l5 -23 1180 0 1179 0 -6 88 c-15 206 -114 442 -231 554
-153 145 -386 126 -539 -44 -49 -55 -111 -157 -128 -213 -6 -19 -13 -35 -17
-35 -3 0 -26 21 -50 46 -52 53 -107 84 -148 84 -38 0 -106 -45 -170 -113 -29
-32 -54 -57 -56 -57 -2 0 -6 44 -9 98 -19 310 -165 583 -368 692 -122 65 -332
90 -438 53z"/>
</g>
</svg>',

'header_cloud' =>
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="2000px" height="50px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path d="M 1999.5,26.5 C 1999.5,34.1667 1999.5,41.8333 1999.5,49.5C 1332.83,49.5 666.167,49.5 -0.5,49.5C -0.5,44.5 -0.5,39.5 -0.5,34.5C 11.1003,23.9184 24.767,20.085 40.5,23C 53.0556,26.277 62.7223,33.4437 69.5,44.5C 76.0578,42.1833 82.7244,41.1833 89.5,41.5C 98.6575,22.2596 113.657,15.4262 134.5,21C 137.971,22.9054 141.304,25.072 144.5,27.5C 158.465,19.2512 170.465,21.4178 180.5,34C 191.257,32.8942 200.757,35.7276 209,42.5C 212.871,25.9804 223.037,19.8138 239.5,24C 241.677,25.5149 243.844,27.0149 246,28.5C 256.112,22.2674 266.446,21.934 277,27.5C 293.832,17.7257 304.332,22.3924 308.5,41.5C 310.583,42.9587 312.75,44.2921 315,45.5C 318.268,39.728 323.101,36.0613 329.5,34.5C 329.28,19.9033 335.947,10.0699 349.5,5C 365.645,1.4088 377.311,6.9088 384.5,21.5C 390.057,22.4039 394.89,24.7372 399,28.5C 408.98,20.0449 420.147,18.2115 432.5,23C 441.202,27.1915 446.202,34.0248 447.5,43.5C 450.777,43.2161 453.943,43.5495 457,44.5C 465.202,35.1352 475.035,32.9686 486.5,38C 489.022,40.5218 491.522,43.0218 494,45.5C 499.585,43.1896 505.418,42.023 511.5,42C 522.55,25.8213 537.883,19.4879 557.5,23C 570.056,26.277 579.722,33.4437 586.5,44.5C 593.058,42.1833 599.724,41.1833 606.5,41.5C 615.657,22.2596 630.657,15.4262 651.5,21C 654.971,22.9054 658.304,25.072 661.5,27.5C 675.465,19.2512 687.465,21.4178 697.5,34C 708.257,32.8942 717.757,35.7276 726,42.5C 729.871,25.9804 740.037,19.8138 756.5,24C 758.677,25.5149 760.844,27.0149 763,28.5C 773.112,22.2674 783.446,21.934 794,27.5C 810.832,17.7257 821.332,22.3924 825.5,41.5C 827.583,42.9587 829.75,44.2921 832,45.5C 835.268,39.728 840.101,36.0613 846.5,34.5C 846.672,15.4972 856.172,5.49716 875,4.5C 887.483,4.49094 896.317,10.1576 901.5,21.5C 907.057,22.4039 911.89,24.7372 916,28.5C 919.453,24.8549 923.62,22.3549 928.5,21C 939.853,21.0011 951.186,21.6678 962.5,23C 971.632,24.8147 979.132,29.3147 985,36.5C 989.748,35.3651 994.581,35.1985 999.5,36C 1002.97,37.9749 1006.3,40.1416 1009.5,42.5C 1013.05,35.9602 1017.72,30.1268 1023.5,25C 1033.91,19.2884 1044.91,17.9551 1056.5,21C 1059.97,22.9054 1063.3,25.072 1066.5,27.5C 1080.46,19.2512 1092.46,21.4178 1102.5,34C 1113.26,32.8942 1122.76,35.7276 1131,42.5C 1134.87,25.9804 1145.04,19.8138 1161.5,24C 1163.68,25.5149 1165.84,27.0149 1168,28.5C 1178.11,22.2674 1188.45,21.934 1199,27.5C 1215.83,17.7257 1226.33,22.3924 1230.5,41.5C 1232.58,42.9587 1234.75,44.2921 1237,45.5C 1240.27,39.728 1245.1,36.0613 1251.5,34.5C 1251.28,19.9033 1257.95,10.0699 1271.5,5C 1287.64,1.4088 1299.31,6.9088 1306.5,21.5C 1312.06,22.4039 1316.89,24.7372 1321,28.5C 1330.98,20.0449 1342.15,18.2115 1354.5,23C 1363.2,27.1915 1368.2,34.0248 1369.5,43.5C 1372.52,43.6646 1375.52,43.498 1378.5,43C 1382.22,40.3 1386.22,37.9666 1390.5,36C 1395.17,35.3333 1399.83,35.3333 1404.5,36C 1408.24,37.8942 1411.57,40.3942 1414.5,43.5C 1418.11,42.7382 1421.78,42.2382 1425.5,42C 1436.55,25.8213 1451.88,19.4879 1471.5,23C 1484.06,26.277 1493.72,33.4437 1500.5,44.5C 1507.06,42.1833 1513.72,41.1833 1520.5,41.5C 1529.66,22.2596 1544.66,15.4262 1565.5,21C 1568.97,22.9054 1572.3,25.072 1575.5,27.5C 1589.46,19.2512 1601.46,21.4178 1611.5,34C 1622.26,32.8942 1631.76,35.7276 1640,42.5C 1643.87,25.9804 1654.04,19.8138 1670.5,24C 1672.68,25.5149 1674.84,27.0149 1677,28.5C 1686.82,22.3331 1696.82,21.9998 1707,27.5C 1709.79,25.8578 1712.62,24.3578 1715.5,23C 1731.53,19.5061 1745.53,23.1728 1757.5,34C 1758.45,34.4828 1759.45,34.6495 1760.5,34.5C 1760.67,15.4972 1770.17,5.49716 1789,4.5C 1800.24,4.53507 1808.74,9.3684 1814.5,19C 1822.29,18.654 1829.62,20.154 1836.5,23.5C 1852.78,16.9852 1866.45,20.4852 1877.5,34C 1884.27,34.1256 1890.93,34.9589 1897.5,36.5C 1901.19,35.9786 1904.86,35.312 1908.5,34.5C 1913.72,24.2465 1922.05,20.4131 1933.5,23C 1936.9,24.4476 1940.06,26.281 1943,28.5C 1953.11,22.2674 1963.45,21.934 1974,27.5C 1982.23,21.2237 1990.73,20.8903 1999.5,26.5 Z"/></g>
</svg>',
'testimonial_cloud' =>
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="250px" height="250px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path d="M -0.5,-0.5 C 82.8333,-0.5 166.167,-0.5 249.5,-0.5C 249.5,82.8333 249.5,166.167 249.5,249.5C 166.167,249.5 82.8333,249.5 -0.5,249.5C -0.5,166.167 -0.5,82.8333 -0.5,-0.5 Z M 39.5,230.5 C 40.9313,229.22 41.5979,227.553 41.5,225.5C 43.1715,217.475 44.8382,209.475 46.5,201.5C 37.4936,196.161 33.1602,188.161 33.5,177.5C 19.4917,174.492 12.4917,165.825 12.5,151.5C 12.5039,147.431 13.0039,143.431 14,139.5C 16.4854,135.023 19.6521,131.189 23.5,128C 19.9388,124.025 17.1055,119.525 15,114.5C 12.9896,99.1752 19.1562,88.8419 33.5,83.5C 29.6183,68.9456 33.9516,57.7789 46.5,50C 50.3462,48.464 54.3462,47.6307 58.5,47.5C 60.5712,39.2109 65.5712,33.3776 73.5,30C 78.4164,28.8394 83.4164,28.1728 88.5,28C 99.8883,16.7945 112.222,15.6278 125.5,24.5C 128.348,21.0688 131.348,17.7354 134.5,14.5C 152.185,5.8952 165.519,10.3952 174.5,28C 187.131,27.8693 197.131,32.8693 204.5,43C 203.978,43.9503 203.311,44.7836 202.5,45.5C 203.628,46.2621 204.795,46.9288 206,47.5C 206.251,49.433 206.751,51.2663 207.5,53C 207.167,53.3333 206.833,53.6667 206.5,54C 209.133,53.7468 211.8,53.7468 214.5,54C 222.355,62.5519 225.688,72.7186 224.5,84.5C 230.717,84.5382 234.884,87.5382 237,93.5C 240.712,104.014 240.212,114.348 235.5,124.5C 234.669,131.142 234.169,137.809 234,144.5C 230.675,152.414 224.842,157.414 216.5,159.5C 216.772,166.838 214.439,173.171 209.5,178.5C 204.751,181.383 199.751,183.383 194.5,184.5C 194.078,195.341 189.078,203.341 179.5,208.5C 173.443,210.49 167.443,210.657 161.5,209C 159.321,207.743 157.321,206.243 155.5,204.5C 152.995,206.253 150.329,207.753 147.5,209C 140.833,209.667 134.167,209.667 127.5,209C 126.892,208.13 126.059,207.63 125,207.5C 110.271,218.198 94.7708,219.698 78.5,212C 74.9218,209.311 71.9218,206.145 69.5,202.5C 67.2688,203.647 64.9355,204.647 62.5,205.5C 55.1991,215.303 47.6991,224.97 40,234.5C 39.51,233.207 39.3433,231.873 39.5,230.5 Z"/></g>
<g><path d="M 41.5,225.5 C 41.5979,227.553 40.9313,229.22 39.5,230.5C 39.4021,228.447 40.0687,226.78 41.5,225.5 Z"/></g>
</svg>',

'about_cloud' =>
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="460px" height="460px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path d="M -0.5,-0.5 C 152.833,-0.5 306.167,-0.5 459.5,-0.5C 459.5,152.833 459.5,306.167 459.5,459.5C 306.167,459.5 152.833,459.5 -0.5,459.5C -0.5,306.167 -0.5,152.833 -0.5,-0.5 Z M 220.5,9.5 C 244.902,7.29247 264.735,15.6258 280,34.5C 310.675,23.1697 336.341,30.1697 357,55.5C 363.51,65.197 367.344,75.8636 368.5,87.5C 404.856,92.6864 424.522,113.52 427.5,150C 427.652,158.407 426.319,166.574 423.5,174.5C 449.954,198.324 455.454,226.324 440,258.5C 434.503,265.994 428.336,272.827 421.5,279C 434.284,310.965 427.284,337.631 400.5,359C 391.908,364.462 382.575,367.962 372.5,369.5C 365.984,404.184 345.151,422.85 310,425.5C 300.368,425.607 291.202,423.607 282.5,419.5C 261.452,446.792 234.785,454.292 202.5,442C 193.955,437.453 186.622,431.453 180.5,424C 173.908,425.298 167.241,426.631 160.5,428C 120.647,428.316 97.3132,408.816 90.5,369.5C 58.9505,365.787 39.4505,348.453 32,317.5C 31.3333,310.5 31.3333,303.5 32,296.5C 33.4474,291.519 34.4474,286.519 35,281.5C 12.4161,262.078 5.41609,238.078 14,209.5C 15.8883,204.055 18.555,199.055 22,194.5C 26.8689,188.634 32.0356,183.134 37.5,178C 25.5929,148.97 30.9263,123.636 53.5,102C 63.1702,94.582 74.0035,89.7487 86,87.5C 93.0908,53.1297 114.091,34.463 149,31.5C 158.426,31.6066 167.426,33.6066 176,37.5C 187.387,22.7215 202.22,13.3882 220.5,9.5 Z"/></g>
</svg>',

'cloud_group' =>
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" viewBox="0 0 2066 188"  width="1920px" height="100px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path d="M 1919.5,78.5 C 1919.5,85.5 1919.5,92.5 1919.5,99.5C 1279.5,99.5 639.5,99.5 -0.5,99.5C -0.5,96.1667 -0.5,92.8333 -0.5,89.5C 0.816089,79.3271 6.48276,72.4937 16.5,69C 22.911,66.3973 29.5777,65.064 36.5,65C 47.2138,54.1783 60.2138,50.0116 75.5,52.5C 76.8671,45.9323 80.2005,40.4323 85.5,36C 89.4689,34.0156 93.4689,32.0156 97.5,30C 113.047,27.1364 128.213,28.303 143,33.5C 162.048,29.134 180.881,29.9673 199.5,36C 205.329,38.992 210.496,42.8253 215,47.5C 215.642,49.7619 216.475,51.9285 217.5,54C 234.528,53.9492 251.195,56.2825 267.5,61C 275.193,63.3595 280.193,68.3595 282.5,76C 285.656,76.9372 288.489,78.4372 291,80.5C 293.885,84.2831 296.218,88.2831 298,92.5C 299.833,89.3333 302.333,86.8333 305.5,85C 311.77,83.8482 318.103,83.1815 324.5,83C 332.558,76.1452 341.891,72.3119 352.5,71.5C 352.393,69.0968 353.393,67.2635 355.5,66C 380.489,58.8645 405.489,58.8645 430.5,66C 432.126,68.295 434.293,69.795 437,70.5C 440.768,68.4027 444.268,65.9027 447.5,63C 449.717,61.7246 452.051,60.7246 454.5,60C 461.484,59.168 468.484,58.5013 475.5,58C 485.68,51.1186 497.013,47.9519 509.5,48.5C 509.451,39.6998 513.785,34.1998 522.5,32C 526.442,31.0071 530.442,30.3404 534.5,30C 536.411,24.4213 540.077,20.4213 545.5,18C 552.167,17.6667 558.833,17.3333 565.5,17C 577.214,8.21246 588.547,8.71246 599.5,18.5C 600.747,20.323 601.414,22.323 601.5,24.5C 610.807,23.461 619.473,25.2943 627.5,30C 628.701,30.9025 629.535,32.0692 630,33.5C 630.498,36.4816 630.665,39.4816 630.5,42.5C 637.184,42.7805 642.684,45.4471 647,50.5C 648.149,53.3926 648.649,56.3926 648.5,59.5C 654.176,59.3339 659.843,59.5006 665.5,60C 668.362,60.9305 670.695,62.5971 672.5,65C 678.777,65.1231 684.944,65.9564 691,67.5C 695.363,65.8023 699.196,63.3023 702.5,60C 717.704,54.9562 732.704,55.4562 747.5,61.5C 757.007,49.4317 769.341,45.2651 784.5,49C 788.316,50.4094 791.983,52.076 795.5,54C 798.06,56.9706 800.727,59.8039 803.5,62.5C 826.169,59.2731 848.169,61.7731 869.5,70C 883.314,70.0621 896.981,71.3955 910.5,74C 914.732,75.9504 919.066,77.617 923.5,79C 927.751,82.0891 931.418,85.5891 934.5,89.5C 935.086,89.2907 935.586,88.9574 936,88.5C 935.464,81.4124 938.297,76.2458 944.5,73C 953.799,67.9036 963.799,65.2369 974.5,65C 979.564,59.8012 985.564,56.1345 992.5,54C 999.84,53.2139 1007.17,52.3805 1014.5,51.5C 1020.06,39.5504 1029.39,32.7171 1042.5,31C 1057.36,28.2344 1071.86,29.4011 1086,34.5C 1110.69,29.1855 1134.52,31.6855 1157.5,42C 1161.75,44.7331 1164.08,48.5665 1164.5,53.5C 1182.85,52.9373 1200.85,55.104 1218.5,60C 1223.49,62.1609 1228.16,64.8276 1232.5,68C 1233.31,70.0876 1234.31,72.0876 1235.5,74C 1243,76.3347 1248,81.168 1250.5,88.5C 1253.18,85.3184 1256.51,83.1517 1260.5,82C 1266.77,80.8482 1273.1,80.1815 1279.5,80C 1288.31,74.2286 1297.98,70.7286 1308.5,69.5C 1308.45,66.7081 1309.79,64.8747 1312.5,64C 1338.49,56.9308 1364.49,56.9308 1390.5,64C 1393.98,67.6916 1397.98,68.3583 1402.5,66C 1406.06,62.1124 1410.39,59.4458 1415.5,58C 1422.82,57.2231 1430.16,56.5564 1437.5,56C 1448.43,49.1003 1460.43,45.9336 1473.5,46.5C 1472.84,41.6527 1474.51,37.8194 1478.5,35C 1484.94,31.1104 1491.94,29.1104 1499.5,29C 1501.81,24.0254 1505.47,20.3587 1510.5,18C 1517.5,17.6667 1524.5,17.3333 1531.5,17C 1544.29,8.19498 1556.79,8.69498 1569,18.5C 1570.01,20.0249 1570.51,21.6916 1570.5,23.5C 1581.88,22.0686 1592.05,24.7352 1601,31.5C 1601.5,34.4816 1601.66,37.4816 1601.5,40.5C 1609.5,41.0815 1616,44.4148 1621,50.5C 1621.5,52.4727 1621.66,54.4727 1621.5,56.5C 1627.51,56.3338 1633.51,56.5005 1639.5,57C 1641.84,57.8372 1643.84,59.1706 1645.5,61C 1652.43,61.1491 1659.26,61.9824 1666,63.5C 1684.85,52.5289 1704.69,50.5289 1725.5,57.5C 1736.12,46.1304 1749.12,42.2971 1764.5,46C 1772.67,48.0023 1779.51,52.169 1785,58.5C 1797.42,57.3452 1809.92,57.1786 1822.5,58C 1833.61,59.9661 1844.61,62.2994 1855.5,65C 1874.69,64.3671 1893.36,67.0338 1911.5,73C 1914.05,75.054 1916.71,76.8874 1919.5,78.5 Z"/></g>
</svg>',

'main_cloud' =>
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="2066px" height="188px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path d="M -0.5,-0.5 C 4.5,-0.5 9.5,-0.5 14.5,-0.5C 18.4569,0.341911 22.4569,1.50858 26.5,3C 33.4656,6.48361 40.1323,10.4836 46.5,15C 50.6734,20.6679 55.0067,26.1679 59.5,31.5C 102.486,30.8645 126.486,52.0312 131.5,95C 131.331,99.7522 130.331,104.252 128.5,108.5C 136.939,108.592 144.939,110.426 152.5,114C 156.078,117.622 159.411,121.455 162.5,125.5C 163.5,125.833 164.5,126.167 165.5,126.5C 173.478,125.43 181.478,125.263 189.5,126C 192.618,126.873 195.618,128.039 198.5,129.5C 224.472,105.424 254.139,99.9237 287.5,113C 292.383,116.724 297.383,120.224 302.5,123.5C 317.568,110.931 332.068,111.598 346,125.5C 348.105,129.583 349.438,133.917 350,138.5C 366.258,133.643 380.092,137.31 391.5,149.5C 394.041,144.765 397.374,140.598 401.5,137C 421.513,122.148 443.18,119.481 466.5,129C 471.132,135.142 477.132,136.976 484.5,134.5C 490.635,113.212 503.468,97.5453 523,87.5C 554.833,75.5484 582.5,81.5484 606,105.5C 610.376,110.919 613.71,116.919 616,123.5C 623.886,103.286 638.386,90.4523 659.5,85C 685.651,79.5971 707.318,86.9304 724.5,107C 751.534,101.347 774.367,108.514 793,128.5C 795.443,133.213 798.276,137.713 801.5,142C 804.517,150.601 806.517,159.434 807.5,168.5C 817.96,165.227 827.96,166.227 837.5,171.5C 842.68,166.157 848.347,161.324 854.5,157C 870.354,150.287 883.854,153.454 895,166.5C 913.198,152.541 933.698,147.708 956.5,152C 965.384,153.333 973.217,156.833 980,162.5C 985.169,153.244 993.169,148.911 1004,149.5C 1016.15,149.666 1024.49,155.333 1029,166.5C 1029.56,167.022 1030.23,167.355 1031,167.5C 1044.96,162.719 1058.29,164.052 1071,171.5C 1076.52,165.643 1081.68,159.476 1086.5,153C 1107.72,137.426 1130.05,135.593 1153.5,147.5C 1156.33,146.067 1159.33,145.4 1162.5,145.5C 1163.8,126.744 1171.13,110.911 1184.5,98C 1193.48,91.6742 1203.15,86.6742 1213.5,83C 1248.28,76.2347 1274.95,87.7347 1293.5,117.5C 1302.5,113.475 1311.5,113.642 1320.5,118C 1323.66,119.413 1326.66,121.08 1329.5,123C 1336.48,121.664 1343.15,122.664 1349.5,126C 1352.06,129.322 1354.06,132.989 1355.5,137C 1366.27,140.617 1375.44,146.784 1383,155.5C 1383.67,156.833 1384.33,158.167 1385,159.5C 1385.7,157.968 1386.87,156.968 1388.5,156.5C 1391.19,143.636 1399.19,137.803 1412.5,139C 1422.01,128.663 1433.68,124.83 1447.5,127.5C 1467.54,86.1612 1499.54,72.8278 1543.5,87.5C 1548.37,90.8642 1553.37,94.0309 1558.5,97C 1563.36,102.525 1568.2,108.025 1573,113.5C 1576.93,120.364 1579.43,127.697 1580.5,135.5C 1584.18,135.665 1587.85,135.499 1591.5,135C 1595.83,132.333 1600.17,129.667 1604.5,127C 1637.41,117.395 1662.24,127.229 1679,156.5C 1682.76,151.705 1687.59,148.538 1693.5,147C 1698.3,146.318 1702.97,145.151 1707.5,143.5C 1710.31,145.235 1713.31,145.735 1716.5,145C 1715.02,130.642 1721.02,120.642 1734.5,115C 1740.17,114.333 1745.83,114.333 1751.5,115C 1754.33,117.248 1757.33,119.248 1760.5,121C 1761.7,121.903 1762.53,123.069 1763,124.5C 1780.67,109.286 1801.17,103.453 1824.5,107C 1841.49,109.421 1855.65,116.921 1867,129.5C 1870.29,127.8 1873.79,126.633 1877.5,126C 1885.96,125.006 1894.29,125.506 1902.5,127.5C 1909.87,114.149 1921.2,108.149 1936.5,109.5C 1931.25,78.8624 1941.58,55.3624 1967.5,39C 1974.46,35.9032 1981.62,33.4032 1989,31.5C 1995.06,30.301 2001.06,30.301 2007,31.5C 2017.38,15.2108 2031.88,4.54417 2050.5,-0.5C 2055.5,-0.5 2060.5,-0.5 2065.5,-0.5C 2065.5,62.1667 2065.5,124.833 2065.5,187.5C 1376.83,187.5 688.167,187.5 -0.5,187.5C -0.5,124.833 -0.5,62.1667 -0.5,-0.5 Z"/></g>
</svg>',

'kids1'=>
'
<svg version="1.0" xmlns="http://www.w3.org/2000/svg"
 width="905.000000pt" height="1280.000000pt" viewBox="0 0 905.000000 1280.000000"
 preserveAspectRatio="xMidYMid meet">
<g transform="translate(0.000000,1280.000000) scale(0.100000,-0.100000)"
 stroke="none">
<path d="M8075 11388 c-65 -46 -138 -126 -168 -183 -27 -52 -30 -70 -31 -140
0 -63 5 -89 22 -122 29 -55 78 -91 135 -99 40 -6 48 -4 73 21 l27 27 -7 140
c-8 168 6 230 63 280 32 28 38 30 95 26 89 -7 163 -46 257 -135 63 -61 88 -93
119 -157 138 -280 109 -600 -71 -784 -53 -55 -137 -102 -180 -102 -23 0 -32
10 -68 79 -48 93 -103 149 -223 230 -121 82 -180 103 -305 109 -170 7 -294
-30 -378 -113 -55 -55 -75 -101 -74 -172 1 -81 18 -122 68 -160 74 -57 133
-44 266 57 102 77 160 104 225 104 72 -1 120 -26 196 -106 129 -134 141 -173
64 -197 -43 -13 -83 -32 -300 -142 -157 -79 -253 -93 -350 -52 -71 29 -83 39
-143 107 -72 81 -180 155 -329 226 -149 70 -230 91 -378 97 -146 6 -255 -10
-407 -58 -251 -80 -385 -201 -429 -384 -13 -58 -15 -93 -8 -205 5 -74 8 -176
9 -226 l0 -91 -47 -56 c-76 -93 -158 -156 -304 -233 -49 -26 -107 -60 -129
-75 -72 -48 -229 -200 -370 -356 -76 -84 -140 -153 -144 -153 -3 0 -12 9 -19
20 -6 11 -39 33 -72 50 -75 38 -94 65 -87 126 9 87 73 151 265 265 204 121
384 274 457 389 69 109 115 232 128 341 l11 96 -30 59 c-40 80 -78 99 -194 99
-156 -1 -300 -67 -300 -138 0 -30 19 -87 30 -87 4 0 39 -29 77 -65 106 -97
117 -165 50 -305 -42 -87 -156 -200 -322 -318 -170 -121 -170 -120 -206 -192
-16 -33 -37 -64 -45 -67 -30 -11 -60 16 -112 99 -42 67 -68 96 -125 139 -114
86 -298 242 -344 292 -119 129 -122 277 -9 371 92 76 116 103 116 130 0 38
-26 73 -78 103 -129 76 -356 -7 -431 -158 -37 -75 -40 -136 -11 -232 55 -177
231 -352 549 -545 115 -69 216 -166 260 -247 50 -93 50 -100 2 -125 -33 -17
-48 -33 -73 -85 -20 -41 -38 -65 -49 -65 -9 0 -28 18 -42 39 -15 22 -98 113
-184 202 -223 228 -381 360 -548 454 -131 75 -161 96 -200 147 -57 73 -63 133
-28 313 21 107 14 244 -15 308 -41 91 -185 218 -315 277 -34 15 -100 40 -147
56 -79 26 -96 28 -255 28 -148 1 -181 -2 -255 -22 -227 -62 -262 -80 -509
-256 -187 -133 -211 -146 -276 -146 -66 0 -108 12 -240 70 -67 30 -148 57
-207 68 -111 23 -144 37 -171 73 -19 26 -19 27 0 70 20 45 105 126 174 166 59
34 136 55 171 48 54 -11 111 -50 146 -98 46 -64 90 -89 170 -94 85 -6 145 22
176 82 28 55 28 146 0 200 -57 110 -236 175 -444 161 -154 -9 -200 -26 -342
-122 -109 -74 -118 -83 -189 -186 -68 -99 -84 -106 -159 -68 -101 52 -173 203
-206 437 -23 166 -21 292 7 361 65 163 176 278 317 328 83 29 123 30 178 3 77
-37 100 -126 80 -309 -10 -85 3 -126 49 -156 35 -23 82 -5 149 57 127 116 122
237 -15 364 -67 62 -178 125 -259 146 -125 32 -293 -27 -407 -144 -148 -149
-234 -424 -203 -646 7 -47 19 -104 28 -126 29 -76 107 -180 224 -302 65 -68
118 -130 118 -138 0 -9 -11 -37 -24 -63 -33 -64 -46 -144 -46 -273 0 -197 63
-370 200 -551 135 -177 140 -192 150 -508 10 -285 19 -369 55 -482 46 -144
117 -266 232 -394 154 -172 443 -318 658 -331 81 -5 102 -2 200 26 123 34 180
39 213 15 30 -21 28 -36 -10 -58 -84 -48 -159 -104 -221 -165 -57 -55 -78 -86
-130 -189 -91 -181 -125 -220 -245 -281 -84 -44 -115 -68 -214 -164 -154 -149
-255 -298 -339 -498 -87 -207 -111 -317 -112 -509 -1 -228 54 -415 175 -593
62 -91 96 -125 169 -169 97 -57 185 -73 287 -52 104 21 166 46 263 105 165
100 224 170 250 296 48 232 -132 527 -349 573 -173 38 -392 -129 -392 -297 0
-96 86 -166 158 -129 17 9 62 56 102 106 39 50 86 100 103 111 106 64 263 -35
286 -181 26 -162 -87 -364 -246 -438 -55 -25 -66 -27 -203 -27 -196 1 -269 25
-373 124 -117 112 -162 283 -137 525 41 407 322 954 574 1118 33 21 62 38 65
38 4 0 3 -51 -2 -112 -12 -159 -3 -267 28 -368 60 -187 195 -323 385 -386 40
-13 84 -18 161 -17 121 1 201 15 278 49 75 34 81 32 81 -21 0 -46 -37 -153
-122 -355 -87 -205 -98 -334 -43 -490 24 -67 26 -84 21 -165 -3 -49 -13 -115
-21 -145 -8 -30 -15 -56 -15 -57 0 -2 18 15 40 37 53 53 81 59 128 26 21 -14
54 -45 74 -69 58 -70 113 -91 238 -91 101 -1 108 1 181 36 134 66 210 151 250
280 24 81 24 123 -2 200 -40 122 -137 191 -208 147 -70 -42 -81 -106 -41 -238
21 -67 21 -71 5 -130 -20 -69 -68 -132 -121 -157 -113 -54 -364 81 -364 194 0
65 338 397 530 520 30 20 117 66 195 103 158 75 172 85 343 232 68 59 191 159
275 224 161 124 245 202 296 277 48 70 124 230 161 339 18 53 36 97 40 97 4 0
17 -11 30 -25 27 -29 42 -31 60 -10 11 13 19 0 50 -84 91 -241 329 -540 545
-687 50 -34 250 -157 445 -274 426 -256 435 -261 580 -387 205 -177 256 -268
200 -358 -59 -96 -272 -160 -344 -104 -53 42 -65 86 -75 294 -9 160 -22 198
-76 221 -63 26 -169 -10 -209 -71 -53 -81 -27 -252 56 -372 56 -80 82 -107
154 -154 75 -48 145 -69 238 -69 136 0 195 29 298 148 50 58 78 57 135 -5 65
-71 93 -154 93 -273 0 -114 -24 -214 -74 -315 -77 -153 -201 -277 -326 -327
-70 -27 -235 -46 -318 -35 -123 15 -233 75 -276 151 -32 56 -38 90 -26 137 22
82 129 159 221 159 82 0 139 47 139 115 0 41 -44 117 -94 163 -43 39 -49 42
-103 42 -68 0 -97 -15 -169 -87 -88 -88 -154 -243 -154 -361 0 -137 157 -305
350 -374 86 -31 245 -36 347 -10 238 60 422 234 492 465 59 195 48 355 -36
539 -54 118 -58 164 -24 262 25 71 27 148 6 307 -19 142 -31 185 -88 319 -70
167 -114 307 -101 328 18 27 111 20 259 -21 154 -43 231 -47 329 -16 92 28
192 121 241 223 71 149 101 317 90 520 l-7 139 48 -7 c234 -32 441 -346 563
-857 49 -205 65 -431 37 -535 -20 -75 -70 -173 -116 -226 -61 -71 -192 -155
-270 -173 -229 -53 -484 47 -544 214 -35 98 -11 237 67 387 38 73 97 95 163
61 43 -23 109 -89 148 -149 55 -86 107 -124 155 -113 28 6 100 73 120 112 65
128 -155 272 -418 273 -118 0 -151 -14 -248 -108 -198 -193 -175 -416 65 -608
124 -100 302 -159 458 -152 86 4 99 7 187 51 112 55 192 133 248 242 49 93 68
164 79 293 42 466 -245 1094 -585 1281 -187 102 -220 134 -447 436 -58 77
-141 174 -186 216 l-80 76 171 6 c682 25 911 178 1091 730 24 73 56 182 70
242 65 266 79 321 87 337 8 14 18 16 53 12 39 -6 44 -4 49 17 8 31 -17 69 -45
69 -12 0 -31 -9 -43 -20 -26 -25 -44 -19 -75 22 -39 53 -32 76 54 173 89 100
153 195 197 292 75 164 84 230 54 373 -30 138 -43 281 -33 345 12 80 38 117
122 180 92 68 142 123 177 199 36 78 49 145 49 262 0 351 -172 633 -426 698
-38 10 -103 16 -170 16 -95 -1 -113 -3 -139 -22z m-1390 -1258 c272 -36 431
-101 672 -276 102 -75 120 -91 110 -101 -16 -16 -535 -135 -597 -137 -29 -1
-37 4 -55 34 -11 19 -44 58 -73 87 -69 68 -138 95 -247 97 -77 1 -83 -1 -122
-31 -77 -61 -81 -124 -14 -226 21 -33 37 -66 34 -73 -2 -6 -61 -31 -131 -54
-70 -23 -174 -66 -232 -97 -95 -50 -105 -53 -116 -38 -36 48 -45 247 -18 368
46 205 190 343 429 412 151 43 235 51 360 35z m-3977 -69 c171 -19 248 -52
352 -148 75 -70 105 -114 147 -214 46 -114 44 -418 -4 -436 -8 -3 -54 15 -103
40 -73 38 -262 113 -357 142 -29 8 -29 11 7 65 82 120 79 184 -11 246 -39 27
-50 29 -128 29 -79 0 -91 -3 -143 -33 -58 -34 -138 -113 -164 -165 -14 -27
-19 -29 -62 -25 -53 5 -223 38 -352 69 -87 21 -220 69 -220 80 0 9 118 105
214 173 127 91 302 164 441 185 71 11 253 7 383 -8z m2713 -410 c22 -22 29
-39 29 -68 0 -57 -27 -153 -55 -193 -37 -53 -137 -172 -141 -167 -2 2 8 37 22
78 22 60 28 98 32 203 4 119 7 132 28 153 31 30 49 29 85 -6z m-1627 -43 c11
-15 16 -45 16 -90 0 -96 25 -191 75 -285 24 -46 39 -83 34 -83 -5 0 -45 35
-88 78 -117 116 -179 261 -152 354 20 67 78 80 115 26z m3824 -75 c26 -35 13
-52 -60 -75 -54 -17 -54 -18 -42 54 4 22 13 39 23 42 26 7 65 -4 79 -21z
m-6012 -21 c6 -4 14 -29 18 -55 7 -44 6 -48 -11 -44 -10 3 -37 11 -60 17 -32
8 -43 16 -43 30 0 42 62 75 96 52z m2872 -1127 c22 -48 6 -109 -31 -121 -31
-10 -50 22 -44 75 7 67 53 94 75 46z m251 16 c19 -12 27 -97 11 -116 -18 -21
-57 -19 -70 4 -15 30 -12 82 7 103 18 20 31 22 52 9z m2416 -404 c-26 -9 -26
-9 -6 -23 20 -15 20 -15 -6 -44 -14 -17 -30 -30 -34 -30 -12 0 -62 88 -56 97
3 5 33 8 67 8 48 0 55 -2 35 -8z m-345 -47 c-4 -23 -10 -43 -13 -46 -3 -3 -30
-8 -61 -11 l-56 -6 0 45 c0 25 3 48 7 51 3 4 34 7 68 7 l62 0 -7 -40z m-4703
-27 c-41 -56 -67 -64 -90 -29 -17 26 -14 55 7 57 6 0 2 4 -9 9 -14 6 3 9 53 9
l74 1 -35 -47z m314 27 l60 0 -3 -42 -3 -43 -55 -3 -55 -3 -17 43 c-22 53 -22
62 -2 54 9 -3 43 -6 75 -6z m4530 -30 c24 -13 25 -30 2 -30 -26 0 -47 16 -39
29 9 14 12 14 37 1z m-4708 -57 c-7 -2 -19 -2 -25 0 -7 3 -2 5 12 5 14 0 19
-2 13 -5z m2380 -1033 c8 -91 13 -166 12 -168 -3 -4 -70 21 -84 32 -22 17 -9
198 19 264 13 29 26 49 30 45 4 -4 14 -82 23 -173z"/>
<path d="M7158 9855 c-25 -55 34 -101 76 -59 23 22 20 40 -10 63 -35 28 -52
26 -66 -4z"/>
<path d="M7002 9828 c-13 -13 -16 -47 -6 -72 7 -19 45 -21 62 -4 27 27 8 88
-28 88 -9 0 -21 -5 -28 -12z"/>
<path d="M6868 9769 c-12 -6 -18 -22 -18 -45 0 -27 5 -37 25 -46 21 -9 28 -8
45 7 24 22 25 43 4 73 -17 24 -30 27 -56 11z"/>
<path d="M6206 9571 c-18 -19 -18 -22 -2 -46 20 -30 39 -32 66 -5 15 15 17 25
11 42 -11 31 -51 35 -75 9z"/>
<path d="M6064 9515 c-19 -29 -11 -60 19 -76 17 -10 24 -8 39 11 23 29 23 45
-2 70 -26 26 -36 25 -56 -5z"/>
<path d="M5953 9443 c-3 -10 -8 -21 -10 -25 -2 -4 6 -17 17 -28 16 -16 24 -18
40 -10 25 13 26 42 2 63 -24 22 -41 21 -49 0z"/>
<path d="M1887 9802 c-26 -28 -21 -61 9 -72 49 -19 87 24 59 66 -19 29 -46 31
-68 6z"/>
<path d="M2050 9760 c-13 -24 12 -80 35 -80 36 0 59 62 33 88 -19 19 -56 14
-68 -8z"/>
<path d="M2190 9701 c-22 -42 -5 -81 35 -81 27 0 45 18 45 46 0 47 -59 73 -80
35z"/>
<path d="M2840 9530 c-18 -34 2 -70 39 -70 39 0 57 23 43 55 -7 14 -18 25 -26
25 -8 0 -21 3 -29 6 -10 3 -20 -3 -27 -16z"/>
<path d="M2990 9471 c-23 -44 1 -88 43 -77 22 5 27 12 27 39 0 54 -47 79 -70
38z"/>
<path d="M3117 9413 c-11 -10 -8 -71 3 -78 17 -11 49 5 56 27 9 30 -39 71 -59
51z"/>
<path d="M8377 11102 c-23 -25 -21 -54 4 -86 25 -32 73 -36 89 -7 37 70 -43
149 -93 93z"/>
<path d="M641 11024 c-26 -34 -27 -68 -1 -94 27 -27 36 -25 71 9 35 35 38 74
7 95 -32 23 -54 20 -77 -10z"/>
<path d="M8234 10966 c-24 -20 -28 -32 -32 -89 -4 -63 -3 -68 26 -97 72 -71
201 -14 204 90 1 34 -12 53 -66 92 -49 35 -95 37 -132 4z m84 -95 c3 -16 -1
-22 -10 -19 -7 3 -15 15 -16 27 -3 16 1 22 10 19 7 -3 15 -15 16 -27z"/>
<path d="M8496 10918 c-19 -27 -21 -69 -4 -86 19 -19 66 -15 78 7 17 32 12 79
-9 91 -30 16 -48 12 -65 -12z"/>
<path d="M745 10897 c-46 -26 -80 -84 -71 -122 10 -39 55 -84 96 -95 59 -16
105 10 125 68 19 57 19 75 -1 113 -14 28 -65 59 -96 59 -7 0 -31 -10 -53 -23z
m68 -97 c-6 -22 -33 -28 -33 -7 0 16 21 38 31 32 4 -3 5 -14 2 -25z"/>
<path d="M532 10848 c-31 -31 2 -108 46 -108 31 0 54 33 46 66 -4 16 -11 35
-17 42 -12 15 -59 16 -75 0z"/>
<path d="M8492 10718 c-17 -17 -15 -74 4 -92 32 -33 84 -8 84 40 0 46 -59 81
-88 52z"/>
<path d="M559 10647 c-42 -32 -41 -84 1 -107 21 -11 42 -3 62 25 14 19 8 55
-12 75 -23 23 -29 24 -51 7z"/>
<path d="M8410 10534 c-15 -37 7 -78 43 -82 37 -4 57 13 57 49 0 21 -7 32 -30
44 -41 21 -59 19 -70 -11z"/>
<path d="M620 10470 c-25 -25 -26 -62 -2 -83 24 -22 45 -21 65 1 29 33 26 75
-8 90 -33 15 -32 15 -55 -8z"/>
<path d="M8623 10140 c-44 -18 -83 -110 -69 -165 9 -37 63 -92 101 -105 46
-15 98 1 125 40 42 59 14 191 -45 218 -37 17 -88 22 -112 12z m69 -107 c39
-35 12 -96 -30 -67 -24 17 -29 53 -10 72 15 15 18 15 40 -5z"/>
<path d="M8007 10102 c-34 -38 -11 -106 32 -95 42 11 68 64 43 86 -29 26 -57
29 -75 9z"/>
<path d="M383 10053 c-65 -33 -93 -145 -53 -211 64 -105 230 -37 230 94 0 48
-29 98 -69 119 -37 19 -67 18 -108 -2z m85 -119 c3 -34 -19 -60 -43 -50 -24 9
-29 34 -10 62 22 34 49 28 53 -12z"/>
<path d="M7831 10051 c-18 -33 -13 -61 14 -73 22 -10 29 -9 47 10 18 18 20 26
12 49 -12 36 -57 44 -73 14z"/>
<path d="M1036 10031 c-32 -36 0 -91 54 -91 27 0 44 43 29 73 -22 42 -55 49
-83 18z"/>
<path d="M1216 9994 c-18 -48 4 -88 46 -82 17 2 24 10 26 34 3 32 -8 47 -44
58 -15 5 -24 2 -28 -10z"/>
<path d="M7656 9984 c-20 -19 -21 -58 -2 -73 24 -20 69 -9 85 21 14 24 14 28
-3 47 -23 25 -58 27 -80 5z"/>
<path d="M1388 9929 c-23 -13 -23 -51 0 -79 24 -30 67 -21 84 16 11 24 9 29
-15 50 -31 26 -43 29 -69 13z"/>
<path d="M5664 9285 c-20 -31 -10 -63 23 -76 49 -18 85 47 47 85 -23 23 -51
20 -70 -9z"/>
<path d="M3382 9250 c-26 -42 -8 -80 38 -80 51 0 69 55 28 84 -30 21 -51 20
-66 -4z"/>
<path d="M5514 9185 c-10 -40 3 -72 32 -79 45 -12 80 35 63 82 -7 18 -16 22
-48 22 -35 0 -42 -4 -47 -25z"/>
<path d="M3513 9154 c-3 -8 -1 -29 4 -47 7 -28 14 -32 46 -35 33 -3 38 0 48
25 17 46 -2 73 -51 73 -28 0 -43 -5 -47 -16z"/>
<path d="M5355 9070 c-18 -34 -9 -62 25 -80 16 -9 25 -6 45 14 29 29 32 58 9
80 -26 26 -61 20 -79 -14z"/>
<path d="M3698 9059 c-41 -23 -11 -109 38 -109 49 0 61 65 18 99 -30 24 -32
24 -56 10z"/>
<path d="M5240 8964 c-20 -23 -16 -75 5 -90 24 -18 30 -17 49 2 22 21 20 67
-2 87 -23 21 -36 21 -52 1z"/>
<path d="M3824 8925 c-24 -36 -1 -95 37 -95 38 0 53 74 21 103 -25 23 -39 21
-58 -8z"/>
<path d="M5086 8831 c-25 -28 -17 -64 17 -75 34 -12 61 1 65 33 4 26 -24 61
-48 61 -9 0 -25 -9 -34 -19z"/>
<path d="M3970 8788 c-11 -18 -20 -36 -20 -40 0 -12 37 -28 63 -28 56 0 63 65
9 87 -31 13 -33 12 -52 -19z"/>
<path d="M785 8780 c-42 -47 25 -112 79 -76 28 18 35 56 13 75 -28 23 -71 24
-92 1z"/>
<path d="M4971 8724 c-23 -29 -24 -34 -15 -58 10 -26 41 -38 64 -26 28 15 28
85 1 100 -27 14 -27 14 -50 -16z"/>
<path d="M4095 8696 c-3 -9 -7 -23 -10 -32 -9 -30 14 -55 47 -52 43 4 58 30
38 66 -18 31 -67 43 -75 18z"/>
<path d="M8260 8689 c-25 -46 23 -108 74 -95 33 8 41 38 20 81 -14 30 -21 35
-50 35 -24 0 -37 -6 -44 -21z"/>
<path d="M787 8623 c-39 -44 -25 -103 24 -103 49 0 81 87 40 110 -28 14 -46
12 -64 -7z"/>
<path d="M4882 8598 c-17 -17 -15 -63 3 -78 12 -10 21 -10 41 -1 21 9 25 17
22 49 -3 31 -7 37 -28 40 -14 2 -31 -3 -38 -10z"/>
<path d="M4187 8573 c-4 -3 -7 -21 -7 -39 0 -40 28 -61 60 -44 23 12 27 59 8
78 -13 13 -50 16 -61 5z"/>
<path d="M8252 8482 c-16 -48 15 -92 66 -92 24 0 38 16 47 52 9 36 -23 68 -69
68 -28 0 -36 -5 -44 -28z"/>
<path d="M788 8426 c-21 -16 -24 -65 -5 -90 19 -26 65 -17 91 17 19 26 19 30
5 55 -17 31 -63 40 -91 18z"/>
<path d="M8238 8279 c-26 -15 -23 -64 4 -83 30 -21 61 -20 82 3 15 17 16 25 6
55 -10 31 -16 36 -43 36 -18 0 -40 -5 -49 -11z"/>
<path d="M814 8205 c-36 -55 12 -114 72 -86 19 8 24 18 24 46 0 32 -4 38 -35
51 -44 18 -42 18 -61 -11z"/>
<path d="M8176 8064 c-18 -18 -20 -42 -7 -77 13 -33 45 -40 76 -15 45 35 25
108 -30 108 -13 0 -31 -7 -39 -16z"/>
<path d="M880 7991 c-24 -45 -6 -91 41 -106 27 -9 47 10 55 52 5 24 1 36 -15
52 -26 26 -67 27 -81 2z"/>
<path d="M8072 7884 c-26 -18 -30 -75 -6 -94 19 -16 71 -17 86 -2 20 20 13 73
-11 94 -28 22 -40 23 -69 2z"/>
<path d="M1007 7819 c-54 -31 -35 -109 26 -109 42 0 57 15 57 56 0 23 -7 38
-22 48 -26 19 -36 19 -61 5z"/>
<path d="M7916 7627 c-25 -19 -18 -85 11 -104 20 -14 24 -13 48 11 55 55 3
138 -59 93z"/>
<path d="M1152 7558 c-28 -28 3 -108 43 -108 49 0 71 53 39 98 -17 24 -62 30
-82 10z"/>
<path d="M7760 7510 c-11 -11 -20 -30 -20 -43 0 -32 27 -67 52 -67 25 0 78 31
78 45 0 19 -31 55 -60 70 -28 14 -31 14 -50 -5z"/>
<path d="M1301 7423 l-31 -37 22 -23 c26 -28 92 -32 102 -7 18 48 -7 104 -47
104 -8 0 -29 -17 -46 -37z"/>
<path d="M7537 7392 c-22 -25 -21 -44 4 -76 26 -33 63 -34 84 -2 19 28 10 55
-27 80 -29 21 -41 20 -61 -2z"/>
<path d="M7064 7369 c-25 -28 -8 -54 37 -54 40 0 51 26 23 56 -22 24 -36 24
-60 -2z"/>
<path d="M7212 7378 c-32 -32 1 -98 50 -98 39 0 61 50 36 82 -17 22 -70 32
-86 16z"/>
<path d="M7349 7356 c-16 -35 10 -76 49 -76 73 0 68 80 -5 95 -28 6 -34 3 -44
-19z"/>
<path d="M1540 7330 c-22 -22 -27 -69 -8 -88 7 -7 22 -12 34 -12 47 0 74 88
35 110 -28 14 -38 13 -61 -10z"/>
<path d="M1880 7323 c-38 -14 -51 -60 -24 -87 21 -22 50 -20 74 4 45 45 9 104
-50 83z"/>
<path d="M2019 7303 c-5 -14 -7 -33 -4 -40 7 -21 55 -12 72 13 12 19 12 24 -2
38 -25 25 -54 20 -66 -11z"/>
<path d="M1722 7304 c-12 -8 -22 -26 -22 -39 0 -25 24 -55 44 -55 21 0 56 40
56 63 0 45 -37 60 -78 31z"/>
<path d="M7236 7208 c-20 -11 -21 -56 -1 -81 17 -23 51 -15 60 14 16 49 -20
89 -59 67z"/>
<path d="M7624 7171 c-93 -58 -78 -248 22 -281 39 -13 102 -7 135 13 59 35 78
157 34 217 -31 42 -56 56 -115 64 -35 5 -52 3 -76 -13z m111 -125 c18 -13 18
-15 2 -40 -22 -32 -32 -33 -53 -1 -29 45 6 74 51 41z"/>
<path d="M1856 7141 c-26 -29 -9 -75 29 -79 32 -4 45 10 45 51 0 20 -7 30 -26
37 -32 13 -28 13 -48 -9z"/>
<path d="M1358 7082 c-35 -33 -41 -46 -45 -89 -11 -106 32 -165 125 -170 73
-5 101 10 129 67 60 121 -1 230 -127 230 -33 0 -48 -7 -82 -38z m120 -105 c5
-24 -25 -68 -41 -61 -7 2 -19 14 -26 26 -12 19 -11 24 9 40 27 22 53 20 58 -5z"/>
<path d="M7322 7058 c-29 -29 1 -88 45 -88 29 0 45 23 37 54 -10 40 -57 59
-82 34z"/>
<path d="M1762 6998 c-17 -17 -15 -77 3 -84 26 -10 61 4 75 31 12 22 11 27 -5
45 -20 22 -55 26 -73 8z"/>
<path d="M7374 6905 c-5 -18 -1 -32 10 -43 23 -23 51 -4 51 34 0 24 -5 30 -27
32 -22 3 -29 -2 -34 -23z"/>
<path d="M1717 6863 c-12 -11 -7 -50 7 -62 19 -16 56 2 56 27 0 32 -43 56 -63
35z"/>
<path d="M7451 6576 c-11 -19 -11 -28 -1 -47 22 -42 48 -50 74 -23 23 22 20
60 -6 78 -32 22 -53 20 -67 -8z"/>
<path d="M1625 6507 c-18 -27 -12 -53 17 -68 27 -15 57 -2 64 28 14 55 -50 87
-81 40z"/>
<path d="M4590 6491 c-22 -41 1 -81 45 -81 39 0 51 34 25 70 -24 34 -55 39
-70 11z"/>
<path d="M4500 6460 c-24 -24 -25 -43 -4 -64 29 -30 74 -10 74 32 0 51 -35 67
-70 32z"/>
<path d="M7616 6434 c-30 -30 -12 -90 29 -94 48 -5 55 -1 55 33 0 35 -30 77
-55 77 -7 0 -21 -7 -29 -16z"/>
<path d="M1478 6365 c-33 -44 -20 -95 23 -95 59 0 78 84 24 108 -32 15 -27 16
-47 -13z"/>
<path d="M4667 6323 c-4 -3 -7 -23 -7 -44 0 -30 6 -42 26 -56 33 -21 58 -8 62
35 5 45 -54 92 -81 65z"/>
<path d="M4430 6283 c-13 -17 -20 -38 -18 -57 2 -26 8 -32 31 -34 37 -4 57 13
57 52 0 59 -38 80 -70 39z"/>
<path d="M7706 6251 c-30 -33 -1 -101 42 -101 26 0 62 29 62 49 0 51 -73 87
-104 52z"/>
<path d="M1372 6184 c-44 -30 -18 -94 38 -94 49 0 78 61 44 94 -20 20 -53 20
-82 0z"/>
<path d="M4790 6104 c-31 -36 38 -115 74 -85 25 21 20 66 -9 85 -31 20 -49 20
-65 0z"/>
<path d="M7790 6080 c-8 -14 -7 -29 5 -55 13 -34 17 -36 52 -33 31 2 39 8 47
30 8 23 6 32 -14 53 -29 31 -75 34 -90 5z"/>
<path d="M4300 6070 c-22 -22 -25 -42 -10 -71 14 -25 31 -24 64 2 22 17 26 27
21 48 -11 44 -43 53 -75 21z"/>
<path d="M1285 6015 c-30 -33 -21 -76 19 -90 25 -8 31 -6 53 19 55 64 -14 132
-72 71z"/>
<path d="M4975 5909 c-10 -16 4 -75 21 -89 22 -19 64 3 64 32 0 29 -9 42 -37
57 -27 14 -40 14 -48 0z"/>
<path d="M4125 5864 c-29 -31 -31 -47 -9 -68 20 -21 37 -20 56 2 22 25 26 44
17 70 -11 29 -34 28 -64 -4z"/>
<path d="M7892 5868 c-6 -6 -12 -27 -12 -44 0 -59 54 -85 84 -42 19 27 20 49
4 71 -18 22 -60 31 -76 15z"/>
<path d="M6562 5793 c3 -41 7 -50 31 -61 35 -17 57 -2 57 37 0 32 -8 44 -37
60 -44 23 -54 15 -51 -36z"/>
<path d="M1207 5792 c-34 -37 -14 -92 34 -92 22 0 32 6 41 25 27 59 -34 113
-75 67z"/>
<path d="M5104 5774 c-13 -51 44 -103 85 -77 39 25 9 83 -51 97 -24 5 -28 3
-34 -20z"/>
<path d="M2560 5781 c-14 -4 -31 -19 -39 -33 -13 -23 -13 -29 1 -51 13 -19 23
-23 43 -20 31 7 47 34 43 75 -3 35 -11 39 -48 29z"/>
<path d="M3982 5737 c-40 -43 -19 -96 33 -83 29 7 62 65 52 90 -10 24 -60 20
-85 -7z"/>
<path d="M5251 5652 c-7 -12 -6 -26 2 -43 10 -19 20 -24 52 -24 36 0 40 3 43
26 2 15 -4 34 -13 43 -22 22 -72 21 -84 -2z"/>
<path d="M6655 5644 c-24 -25 -15 -72 18 -92 25 -15 30 -15 55 -2 33 19 41 59
15 82 -29 25 -69 30 -88 12z"/>
<path d="M7886 5641 c-20 -23 -20 -60 0 -89 23 -34 75 -27 88 11 20 57 -51
119 -88 78z"/>
<path d="M3838 5628 c-24 -20 -28 -44 -12 -67 23 -32 61 -27 84 11 18 30 18
34 3 50 -19 21 -53 23 -75 6z"/>
<path d="M2427 5582 c-46 -51 20 -120 75 -79 26 20 32 61 13 81 -22 21 -68 20
-88 -2z"/>
<path d="M1217 5577 c-19 -13 -32 -75 -20 -94 13 -19 56 -25 75 -11 9 7 19 27
23 45 5 25 2 36 -14 52 -24 24 -40 26 -64 8z"/>
<path d="M5468 5534 c-22 -22 -5 -72 27 -80 51 -13 85 35 48 68 -29 25 -57 30
-75 12z"/>
<path d="M7277 5533 c-4 -3 -7 -21 -7 -40 0 -45 36 -64 75 -39 29 19 32 46 7
69 -19 17 -63 23 -75 10z"/>
<path d="M3629 5489 c-27 -27 -21 -61 13 -73 33 -12 68 14 68 49 0 46 -45 60
-81 24z"/>
<path d="M1823 5474 c-20 -9 -16 -72 5 -83 43 -23 91 27 65 68 -10 16 -48 24
-70 15z"/>
<path d="M5673 5443 c-19 -7 -16 -58 5 -76 22 -20 63 -22 80 -5 24 24 14 58
-23 74 -36 15 -42 16 -62 7z"/>
<path d="M6710 5414 c-14 -35 -13 -46 6 -72 17 -25 71 -31 92 -10 17 17 15 59
-4 86 -23 33 -81 31 -94 -4z"/>
<path d="M7176 5424 c-17 -45 1 -84 39 -84 43 0 61 53 29 84 -20 21 -60 21
-68 0z"/>
<path d="M3428 5393 c-46 -29 -32 -83 22 -83 40 1 62 23 58 62 -2 27 -8 33
-28 35 -14 2 -37 -5 -52 -14z"/>
<path d="M7852 5387 c-29 -30 -28 -53 3 -73 23 -15 27 -15 50 0 14 9 28 28 31
41 13 53 -45 74 -84 32z"/>
<path d="M2366 5366 c-20 -31 -21 -70 -2 -85 8 -7 32 -11 53 -9 31 2 39 8 47
31 24 66 -60 120 -98 63z"/>
<path d="M1922 5368 c-7 -7 -12 -25 -12 -40 0 -34 41 -59 68 -41 21 12 28 59
11 79 -14 17 -51 18 -67 2z"/>
<path d="M5860 5351 c-12 -24 1 -66 25 -75 53 -20 97 12 75 54 -22 41 -82 54
-100 21z"/>
<path d="M1234 5315 c-15 -62 72 -107 98 -50 17 37 -12 75 -57 75 -29 0 -36
-4 -41 -25z"/>
<path d="M3238 5319 c-32 -18 -42 -48 -22 -70 22 -25 59 -24 84 1 25 25 25 42
2 63 -21 19 -39 21 -64 6z"/>
<path d="M7191 5257 c-28 -28 5 -97 47 -97 41 0 57 54 25 82 -24 21 -59 28
-72 15z"/>
<path d="M6030 5235 c-26 -31 21 -92 65 -81 19 5 25 13 25 35 0 49 -61 81 -90
46z"/>
<path d="M1918 5189 c-42 -31 -33 -89 13 -89 22 0 56 30 63 55 12 44 -35 65
-76 34z"/>
<path d="M3071 5189 c-30 -28 -26 -73 7 -77 55 -8 98 63 53 88 -27 14 -34 13
-60 -11z"/>
<path d="M6744 5196 c-3 -8 -4 -29 -2 -47 4 -41 48 -70 74 -49 22 19 18 76 -8
94 -28 20 -57 20 -64 2z"/>
<path d="M7716 5168 c-9 -12 -16 -29 -16 -38 0 -27 30 -60 54 -60 27 0 41 14
50 52 5 21 1 31 -20 47 -34 27 -49 26 -68 -1z"/>
<path d="M2362 5144 c-18 -12 -22 -24 -20 -51 4 -47 27 -62 62 -40 19 13 26
26 26 50 0 54 -27 70 -68 41z"/>
<path d="M7320 5125 c-14 -17 -7 -57 16 -79 22 -23 71 -20 84 3 15 29 13 40
-16 66 -30 28 -66 32 -84 10z"/>
<path d="M7497 5115 c-24 -23 -7 -78 26 -90 37 -12 61 3 65 43 3 26 -2 33 -27
46 -37 20 -46 20 -64 1z"/>
<path d="M1382 5108 c-15 -15 -16 -63 -2 -89 13 -23 53 -25 74 -3 18 18 21 56
6 85 -12 22 -59 26 -78 7z"/>
<path d="M6170 5095 c-16 -19 -5 -68 21 -91 16 -14 24 -15 44 -6 19 9 25 19
25 45 0 55 -59 89 -90 52z"/>
<path d="M1776 5059 c-21 -16 -26 -28 -24 -53 2 -25 8 -32 30 -34 39 -5 78 24
78 57 0 52 -39 66 -84 30z"/>
<path d="M2931 5044 c-38 -48 -22 -94 31 -94 32 0 48 24 48 70 0 24 -6 33 -26
40 -33 13 -29 14 -53 -16z"/>
<path d="M1596 5044 c-31 -30 -9 -84 35 -84 24 0 59 34 59 58 0 41 -63 58 -94
26z"/>
<path d="M6272 4986 c-35 -42 -9 -90 45 -84 21 2 29 9 31 26 6 40 -54 86 -76
58z"/>
<path d="M6732 4988 c-7 -7 -12 -26 -12 -43 0 -37 16 -55 47 -55 25 0 43 26
43 62 0 30 -17 48 -45 48 -12 0 -26 -5 -33 -12z"/>
<path d="M2371 4919 c-15 -44 7 -79 49 -79 25 0 30 4 36 36 7 40 -5 59 -47 69
-23 6 -28 2 -38 -26z"/>
<path d="M2848 4933 c-26 -31 -29 -42 -18 -63 20 -36 90 -16 90 26 0 48 -43
70 -72 37z"/>
<path d="M7060 4833 c-33 -23 -38 -31 -50 -75 -9 -33 -7 -45 15 -93 31 -64 62
-85 130 -85 56 0 88 28 105 91 10 37 9 50 -7 88 -21 52 -59 79 -124 86 -32 4
-51 0 -69 -12z m109 -107 c24 -29 -14 -65 -50 -46 -19 10 -26 40 -12 53 11 12
50 8 62 -7z"/>
<path d="M6742 4788 c-28 -28 -7 -93 34 -108 71 -27 92 70 23 105 -34 18 -42
18 -57 3z"/>
<path d="M1994 4776 c-69 -30 -98 -106 -69 -181 21 -56 47 -75 101 -75 84 0
144 65 144 155 0 40 -5 52 -34 81 -38 38 -85 45 -142 20z m74 -102 c23 -41
-22 -78 -53 -44 -24 27 -9 51 37 59 4 0 11 -6 16 -15z"/>
<path d="M2355 4715 c-14 -13 -25 -31 -25 -40 0 -18 35 -55 53 -55 42 0 82 74
57 105 -19 23 -58 18 -85 -10z"/>
<path d="M6826 4548 c-16 -25 -11 -71 8 -87 6 -5 28 -12 49 -16 l37 -7 0 46
c0 74 -61 115 -94 64z"/>
<path d="M2275 4494 c-11 -11 -15 -29 -12 -58 3 -36 7 -41 30 -44 38 -5 63 13
71 49 11 57 -49 92 -89 53z"/>
<path d="M6384 4482 c-53 -33 -74 -69 -74 -124 0 -60 18 -104 59 -143 27 -25
39 -29 85 -29 86 2 136 54 136 142 -1 119 -120 207 -206 154z m86 -117 c20
-10 30 -23 30 -38 0 -49 -48 -62 -87 -25 -27 26 -30 53 -5 67 23 14 28 13 62
-4z"/>
<path d="M3266 4460 c-87 -27 -160 -174 -122 -248 20 -39 35 -47 108 -61 120
-22 242 -107 268 -187 18 -55 5 -115 -33 -151 -63 -61 -146 -91 -314 -114
-224 -31 -436 75 -575 287 -85 129 -117 226 -127 374 -7 104 -7 105 -20 65
-22 -67 -25 -204 -7 -287 44 -195 222 -391 444 -489 284 -124 542 -72 685 138
82 119 113 250 82 345 -42 131 -177 290 -273 322 -50 18 -73 19 -116 6z"/>
<path d="M2663 4421 c-33 -24 -45 -41 -58 -83 -21 -73 -19 -99 14 -146 71 -98
194 -72 245 52 21 51 20 79 -4 126 -25 49 -53 68 -111 76 -38 5 -49 2 -86 -25z
m131 -113 c8 -14 5 -23 -19 -48 -16 -16 -39 -30 -51 -30 -56 0 -57 73 -1 89
42 12 57 10 71 -11z"/>
<path d="M6840 4300 c-26 -26 -25 -42 6 -79 31 -37 44 -38 72 -8 24 25 29 76
10 95 -19 19 -66 14 -88 -8z"/>
<path d="M2263 4259 c-7 -7 -13 -32 -13 -56 0 -48 22 -68 65 -57 42 10 62 77
32 103 -22 18 -71 24 -84 10z"/>
<path d="M6818 4119 c-35 -20 -12 -89 28 -89 52 0 73 56 32 84 -26 18 -36 19
-60 5z"/>
<path d="M6273 4095 c-50 -35 -47 -136 4 -156 57 -22 107 0 126 54 8 21 -13
76 -35 94 -26 22 -71 25 -95 8z"/>
<path d="M2295 4046 c-18 -27 -11 -52 17 -67 41 -22 77 15 64 67 -9 33 -59 34
-81 0z"/>
<path d="M2811 4031 c-58 -54 -33 -132 46 -147 61 -12 101 33 89 99 -14 76
-80 100 -135 48z"/>
<path d="M6704 3844 c-23 -34 -9 -75 28 -88 56 -20 95 64 46 98 -32 23 -55 20
-74 -10z"/>
<path d="M2412 3804 c-44 -31 -21 -103 34 -104 24 0 38 14 48 47 5 17 1 29
-18 48 -29 29 -34 30 -64 9z"/>
<path d="M5536 3805 c-6 -17 7 -60 22 -69 16 -10 67 10 70 28 8 41 -77 79 -92
41z"/>
<path d="M3588 3773 c-19 -5 -34 -45 -23 -62 3 -6 22 -11 41 -11 27 0 37 5 46
25 19 41 -10 62 -64 48z"/>
<path d="M5663 3683 c-3 -9 0 -28 8 -42 12 -20 23 -26 49 -26 29 0 36 4 38 23
2 13 -4 32 -14 43 -21 24 -72 25 -81 2z"/>
<path d="M6510 3670 c-36 -36 -17 -100 31 -100 25 0 59 34 59 59 0 19 -43 61
-61 61 -5 0 -18 -9 -29 -20z"/>
<path d="M3448 3646 c-22 -16 -23 -59 -3 -76 21 -17 61 1 77 35 20 45 -31 74
-74 41z"/>
<path d="M2610 3620 c-43 -43 -4 -111 55 -96 22 6 25 12 25 51 0 40 -3 47 -26
55 -33 13 -31 13 -54 -10z"/>
<path d="M5857 3603 c-24 -23 17 -93 54 -93 39 0 52 55 21 83 -19 17 -63 23
-75 10z"/>
<path d="M6030 3579 c-25 -46 12 -94 68 -87 23 3 27 8 30 40 3 32 -1 39 -27
52 -40 21 -58 20 -71 -5z"/>
<path d="M6250 3579 c-25 -46 12 -88 67 -77 26 5 31 67 7 85 -27 20 -62 16
-74 -8z"/>
<path d="M3266 3557 c-27 -20 -18 -81 12 -85 31 -5 62 25 62 58 0 39 -39 53
-74 27z"/>
<path d="M2872 3544 c-29 -20 -31 -79 -2 -94 21 -11 57 1 72 24 13 19 9 54 -8
70 -19 20 -34 20 -62 0z"/>
<path d="M3082 3532 c-17 -13 -23 -26 -20 -47 5 -42 29 -53 72 -32 36 17 45
41 30 81 -8 21 -53 20 -82 -2z"/>
</g>
</svg>
',

'village' =>
'<svg xmlns="http://www.w3.org/2000/svg" version="1.1" width="1920px" height="139px" style="shape-rendering:geometricPrecision; text-rendering:geometricPrecision; image-rendering:optimizeQuality; fill-rule:evenodd; clip-rule:evenodd" xmlns:xlink="http://www.w3.org/1999/xlink">
<g><path " d="M 1919.5,34.5 C 1919.5,68.8333 1919.5,103.167 1919.5,137.5C 1913.31,137.177 1907.31,137.51 1901.5,138.5C 1900.17,138.5 1898.83,138.5 1897.5,138.5C 1895.02,137.631 1892.35,136.964 1889.5,136.5C 1887.83,136.668 1886.5,137.335 1885.5,138.5C 1881.5,138.5 1877.5,138.5 1873.5,138.5C 1872.65,137.034 1871.32,136.2 1869.5,136C 1858.86,134.901 1848.19,134.401 1837.5,134.5C 1837.5,135.833 1837.5,137.167 1837.5,138.5C 1829.83,138.5 1822.17,138.5 1814.5,138.5C 1818.08,136.882 1822.08,135.882 1826.5,135.5C 1818.11,133.563 1809.61,132.229 1801,131.5C 1797.65,131.504 1794.81,132.671 1792.5,135C 1795.19,135.743 1797.53,136.91 1799.5,138.5C 1792.17,138.5 1784.83,138.5 1777.5,138.5C 1775.78,137.692 1774.11,136.692 1772.5,135.5C 1772.62,134.414 1773.29,133.747 1774.5,133.5C 1776.8,134.396 1779.13,134.562 1781.5,134C 1771.3,131.49 1760.97,130.99 1750.5,132.5C 1753.78,133.324 1757.12,133.824 1760.5,134C 1764.02,135.109 1764.18,136.276 1761,137.5C 1752.54,136.709 1744.04,136.209 1735.5,136C 1734.55,135.282 1734.38,134.449 1735,133.5C 1736.4,132.773 1737.9,132.273 1739.5,132C 1733.4,129.03 1727.07,128.697 1720.5,131C 1720.39,135.605 1718.06,137.438 1713.5,136.5C 1711.83,136.333 1710.17,136.167 1708.5,136C 1707.27,134.413 1707.61,133.079 1709.5,132C 1704.3,130.516 1698.97,129.682 1693.5,129.5C 1690.55,129.291 1687.88,129.958 1685.5,131.5C 1685.83,132.5 1686.5,133.167 1687.5,133.5C 1688.89,132.355 1690.22,132.355 1691.5,133.5C 1691.29,134.086 1690.96,134.586 1690.5,135C 1660.83,135.667 1631.17,135.667 1601.5,135C 1598.09,134.397 1597.76,133.231 1600.5,131.5C 1602.87,133.395 1605.53,133.895 1608.5,133C 1607.81,131.933 1607.31,130.766 1607,129.5C 1600.18,126.947 1593.35,126.947 1586.5,129.5C 1587.66,129.437 1588.99,129.77 1590.5,130.5C 1591.39,131.325 1591.72,132.325 1591.5,133.5C 1581.81,133.832 1572.14,133.498 1562.5,132.5C 1567.07,131.178 1571.74,130.345 1576.5,130C 1575.83,129.667 1575.17,129.333 1574.5,129C 1566.17,128.333 1557.83,128.333 1549.5,129C 1547.74,129.202 1546.08,129.702 1544.5,130.5C 1547.09,131.205 1549.76,131.205 1552.5,130.5C 1552.64,132.124 1551.97,133.29 1550.5,134C 1544.27,134.406 1538.1,135.24 1532,136.5C 1524.62,134.084 1524.79,132.75 1532.5,132.5C 1532.5,131.167 1532.5,129.833 1532.5,128.5C 1522.8,128.353 1513.13,128.853 1503.5,130C 1502.38,130.249 1501.38,130.749 1500.5,131.5C 1504.17,132.167 1507.83,132.833 1511.5,133.5C 1504.25,135.032 1496.92,135.699 1489.5,135.5C 1489.5,133.833 1489.5,132.167 1489.5,130.5C 1478.83,130.333 1468.16,130.5 1457.5,131C 1456.69,131.308 1456.03,131.808 1455.5,132.5C 1457.05,133.684 1458.71,134.684 1460.5,135.5C 1456.16,137.021 1451.66,137.688 1447,137.5C 1443.83,137.167 1440.67,136.833 1437.5,136.5C 1439.32,134.99 1441.32,133.657 1443.5,132.5C 1438.84,131.431 1434.18,131.431 1429.5,132.5C 1429.5,128.833 1429.5,125.167 1429.5,121.5C 1435.61,121.733 1441.61,121.066 1447.5,119.5C 1447.83,118.833 1448.17,118.167 1448.5,117.5C 1441.2,116.203 1433.86,114.87 1426.5,113.5C 1420.32,114.01 1416.65,117.344 1415.5,123.5C 1414.09,124.806 1412.42,125.64 1410.5,126C 1413.84,126.39 1417.17,126.224 1420.5,125.5C 1421.83,125.667 1423.17,125.833 1424.5,126C 1417.65,127.98 1410.65,129.647 1403.5,131C 1399.27,133.275 1395.27,135.775 1391.5,138.5C 1359.83,138.5 1328.17,138.5 1296.5,138.5C 1292.03,135.855 1287.2,133.855 1282,132.5C 1273.5,133.094 1265,133.761 1256.5,134.5C 1255.83,136.16 1254.83,137.493 1253.5,138.5C 1250.83,138.5 1248.17,138.5 1245.5,138.5C 1243.39,137.532 1241.06,137.198 1238.5,137.5C 1238.42,134.513 1239.59,133.847 1242,135.5C 1243.28,135.387 1244.11,134.72 1244.5,133.5C 1235.9,131.039 1227.23,130.873 1218.5,133C 1220.04,134.11 1220.7,135.61 1220.5,137.5C 1218.26,137.205 1216.26,137.539 1214.5,138.5C 1212.83,138.5 1211.17,138.5 1209.5,138.5C 1208.43,136.144 1207.76,133.978 1207.5,132C 1200.48,131.169 1193.48,131.336 1186.5,132.5C 1188.86,133.085 1190.86,134.252 1192.5,136C 1182.5,136.667 1172.5,136.667 1162.5,136C 1166.73,134.855 1171.06,133.855 1175.5,133C 1172.09,131.972 1168.59,131.472 1165,131.5C 1158.49,131.917 1151.99,132.417 1145.5,133C 1145.17,133.333 1144.83,133.667 1144.5,134C 1147.05,134.66 1149.55,134.493 1152,133.5C 1153.03,133.836 1153.53,134.503 1153.5,135.5C 1150.94,136.918 1148.27,137.918 1145.5,138.5C 1139.5,138.5 1133.5,138.5 1127.5,138.5C 1127.2,137.85 1126.86,137.183 1126.5,136.5C 1128.83,135.833 1131.17,135.167 1133.5,134.5C 1133.33,134.167 1133.17,133.833 1133,133.5C 1130.01,132.148 1126.84,131.481 1123.5,131.5C 1116.12,132.003 1108.79,132.837 1101.5,134C 1101.17,134.333 1100.83,134.667 1100.5,135C 1103.83,135.333 1107.17,135.667 1110.5,136C 1111.6,136.77 1111.6,137.603 1110.5,138.5C 1103.5,138.5 1096.5,138.5 1089.5,138.5C 1089.5,136.833 1089.5,135.167 1089.5,133.5C 1079.47,133.216 1069.47,133.549 1059.5,134.5C 1056.17,133.833 1052.83,133.167 1049.5,132.5C 1047.89,132.637 1046.56,133.304 1045.5,134.5C 1048.34,135.106 1050.67,136.439 1052.5,138.5C 1044.17,138.5 1035.83,138.5 1027.5,138.5C 1027.41,137.504 1027.75,136.671 1028.5,136C 1034.66,135.459 1035,134.125 1029.5,132C 1018.86,130.901 1008.19,130.401 997.5,130.5C 998.593,135.239 996.76,137.239 992,136.5C 986.508,135.917 981.008,135.417 975.5,135C 975.043,134.586 974.709,134.086 974.5,133.5C 978.5,132.833 982.5,132.167 986.5,131.5C 978.107,129.563 969.607,128.229 961,127.5C 957.647,127.504 954.813,128.671 952.5,131C 954.701,132.043 957.034,132.877 959.5,133.5C 958.451,135.112 956.951,136.112 955,136.5C 949.26,135.069 943.427,134.235 937.5,134C 935.548,133.634 933.882,132.801 932.5,131.5C 932.62,130.414 933.287,129.747 934.5,129.5C 936.801,130.396 939.134,130.562 941.5,130C 931.299,127.49 920.966,126.99 910.5,128.5C 913.785,129.324 917.118,129.824 920.5,130C 924.017,131.109 924.184,132.276 921,133.5C 912.535,132.709 904.035,132.209 895.5,132C 894.549,131.282 894.383,130.449 895,129.5C 896.399,128.773 897.899,128.273 899.5,128C 893.4,125.03 887.067,124.697 880.5,127C 880.391,131.605 878.058,133.438 873.5,132.5C 871.833,132.333 870.167,132.167 868.5,132C 867.273,130.413 867.606,129.079 869.5,128C 864.3,126.516 858.966,125.682 853.5,125.5C 850.546,125.291 847.879,125.958 845.5,127.5C 845.833,128.5 846.5,129.167 847.5,129.5C 848.891,128.355 850.224,128.355 851.5,129.5C 851.291,130.086 850.957,130.586 850.5,131C 841.5,131.333 832.5,131.667 823.5,132C 818.679,133.299 813.846,133.132 809,131.5C 795.739,131.581 782.239,131.914 768.5,132.5C 766.091,132.085 763.758,131.418 761.5,130.5C 761.833,130.167 762.167,129.833 762.5,129.5C 764.824,129.013 767.324,129.013 770,129.5C 771.833,129.167 773.667,128.833 775.5,128.5C 775.333,128.167 775.167,127.833 775,127.5C 768.18,124.947 761.347,124.947 754.5,127.5C 755.66,127.437 756.994,127.77 758.5,128.5C 759.386,129.325 759.719,130.325 759.5,131.5C 749.81,131.832 740.144,131.498 730.5,130.5C 735.071,129.178 739.738,128.345 744.5,128C 743.833,127.667 743.167,127.333 742.5,127C 734.167,126.333 725.833,126.333 717.5,127C 715.744,127.202 714.078,127.702 712.5,128.5C 715.089,129.205 717.756,129.205 720.5,128.5C 720.64,130.124 719.973,131.29 718.5,132C 712.268,132.406 706.101,133.24 700,134.5C 692.62,132.084 692.787,130.75 700.5,130.5C 700.5,129.167 700.5,127.833 700.5,126.5C 690.8,126.353 681.134,126.853 671.5,128C 670.376,128.249 669.376,128.749 668.5,129.5C 672.167,130.167 675.833,130.833 679.5,131.5C 672.249,133.032 664.916,133.699 657.5,133.5C 657.5,131.833 657.5,130.167 657.5,128.5C 646.828,128.333 636.161,128.5 625.5,129C 624.692,129.308 624.025,129.808 623.5,130.5C 625.046,131.684 626.713,132.684 628.5,133.5C 624.161,135.021 619.661,135.688 615,135.5C 611.833,135.167 608.667,134.833 605.5,134.5C 607.322,132.99 609.322,131.657 611.5,130.5C 606.844,129.431 602.178,129.431 597.5,130.5C 597.5,126.833 597.5,123.167 597.5,119.5C 603.606,119.733 609.606,119.066 615.5,117.5C 615.833,116.833 616.167,116.167 616.5,115.5C 609.198,114.203 601.865,112.87 594.5,111.5C 588.321,112.01 584.654,115.344 583.5,121.5C 582.088,122.806 580.421,123.64 578.5,124C 581.839,124.39 585.173,124.224 588.5,123.5C 589.833,123.667 591.167,123.833 592.5,124C 585.649,125.98 578.649,127.647 571.5,129C 567.5,131.333 563.5,133.667 559.5,136C 549.022,136.935 538.689,137.769 528.5,138.5C 508.5,138.5 488.5,138.5 468.5,138.5C 462.702,135.19 456.536,132.523 450,130.5C 441.502,131.094 433.002,131.761 424.5,132.5C 423.244,135.63 420.91,136.964 417.5,136.5C 413.855,135.884 410.188,135.55 406.5,135.5C 406.422,132.513 407.589,131.847 410,133.5C 411.28,133.387 412.113,132.72 412.5,131.5C 403.9,129.039 395.234,128.873 386.5,131C 388.037,132.11 388.703,133.61 388.5,135.5C 384.815,135.895 381.148,136.062 377.5,136C 376.332,133.837 375.665,131.837 375.5,130C 368.476,129.169 361.476,129.336 354.5,130.5C 356.865,131.085 358.865,132.252 360.5,134C 350.5,134.667 340.5,134.667 330.5,134C 334.727,132.855 339.061,131.855 343.5,131C 340.09,129.972 336.59,129.472 333,129.5C 326.494,129.917 319.994,130.417 313.5,131C 313.167,131.333 312.833,131.667 312.5,132C 315.047,132.66 317.547,132.493 320,131.5C 321.03,131.836 321.53,132.503 321.5,133.5C 320.711,133.783 320.044,134.283 319.5,135C 313.184,135.965 306.851,136.798 300.5,137.5C 297.927,137.547 295.927,136.547 294.5,134.5C 296.833,133.833 299.167,133.167 301.5,132.5C 301.291,131.914 300.957,131.414 300.5,131C 290.152,129.499 279.818,129.832 269.5,132C 269.167,132.333 268.833,132.667 268.5,133C 271.833,133.333 275.167,133.667 278.5,134C 279.833,134.667 279.833,135.333 278.5,136C 272.47,136.252 266.47,136.752 260.5,137.5C 259.558,137.392 258.725,137.059 258,136.5C 257.506,134.866 257.34,133.199 257.5,131.5C 247.467,131.216 237.467,131.549 227.5,132.5C 224.167,131.833 220.833,131.167 217.5,130.5C 215.893,130.637 214.559,131.304 213.5,132.5C 216.306,133.105 218.64,134.438 220.5,136.5C 217,136.833 213.5,137.167 210,137.5C 205.009,137.702 200.175,137.036 195.5,135.5C 197.149,134.012 199.149,133.012 201.5,132.5C 200.619,131.029 199.286,130.196 197.5,130C 186.86,128.901 176.194,128.401 165.5,128.5C 166.593,133.239 164.76,135.239 160,134.5C 154.508,133.917 149.008,133.417 143.5,133C 143.043,132.586 142.709,132.086 142.5,131.5C 146.5,130.833 150.5,130.167 154.5,129.5C 146.107,127.563 137.607,126.229 129,125.5C 125.647,125.504 122.813,126.671 120.5,129C 122.701,130.043 125.034,130.877 127.5,131.5C 126.451,133.112 124.951,134.112 123,134.5C 117.26,133.069 111.427,132.235 105.5,132C 103.548,131.634 101.882,130.801 100.5,129.5C 100.62,128.414 101.287,127.747 102.5,127.5C 104.801,128.396 107.134,128.562 109.5,128C 99.2994,125.49 88.966,124.99 78.5,126.5C 81.7848,127.324 85.1182,127.824 88.5,128C 92.017,129.109 92.1837,130.276 89,131.5C 80.5352,130.709 72.0352,130.209 63.5,130C 62.5494,129.282 62.3828,128.449 63,127.5C 64.3988,126.773 65.8988,126.273 67.5,126C 61.4,123.03 55.0666,122.697 48.5,125C 47.8333,126.667 47.1667,128.333 46.5,130C 43.1667,130.667 39.8333,130.667 36.5,130C 35.273,128.413 35.6063,127.079 37.5,126C 32.2995,124.516 26.9662,123.682 21.5,123.5C 18.5457,123.291 15.879,123.958 13.5,125.5C 13.8333,126.5 14.5,127.167 15.5,127.5C 16.891,126.355 18.2243,126.355 19.5,127.5C 19.2907,128.086 18.9574,128.586 18.5,129C 12.1754,129.5 5.84209,129.666 -0.5,129.5C -0.5,128.167 -0.5,126.833 -0.5,125.5C 0.5,125.5 1.5,125.5 2.5,125.5C 2.69026,115.09 3.5236,104.756 5,94.5C 10.6284,91.3978 13.4618,93.0645 13.5,99.5C 20.175,99.6663 26.8417,99.4996 33.5,99C 34.1667,98 34.8333,97 35.5,96C 28.8333,92 22.1667,88 15.5,84C 12.8333,83.6667 10.1667,83.3333 7.5,83C 4.43463,79.023 1.76797,74.8563 -0.5,70.5C -0.5,59.1667 -0.5,47.8333 -0.5,36.5C 1.67901,36.16 3.51235,35.16 5,33.5C 5.21173,30.2423 5.71173,27.0757 6.5,24C 10.1667,23.6667 13.8333,23.3333 17.5,23C 17.7941,17.87 20.4608,14.7034 25.5,13.5C 27.1161,14.1152 28.4495,15.1152 29.5,16.5C 32.5337,3.9811 40.2003,-0.185565 52.5,4C 56.7703,5.27038 59.2703,8.10371 60,12.5C 62.6277,14.1255 64.961,16.1255 67,18.5C 68.569,22.072 70.9024,25.072 74,27.5C 75.6976,30.3963 75.8643,33.3963 74.5,36.5C 83.452,35.4311 88.7853,39.4311 90.5,48.5C 100.61,51.3874 103.777,57.7207 100,67.5C 97.9715,70.0292 96.1381,72.6959 94.5,75.5C 96.5283,80.4116 96.8616,85.4116 95.5,90.5C 90.351,92.2903 85.0176,93.6236 79.5,94.5C 79.5,96.1667 79.5,97.8333 79.5,99.5C 88.5062,99.6664 97.5062,99.4998 106.5,99C 109.876,98.6929 111.21,96.8595 110.5,93.5C 112.895,92.6365 115.229,92.6365 117.5,93.5C 118.093,95.3817 118.76,97.2151 119.5,99C 127.525,99.3011 135.525,99.8011 143.5,100.5C 142.439,91.651 139.773,83.3177 135.5,75.5C 137.665,69.0035 140.998,63.1702 145.5,58C 152.077,55.6037 157.91,56.7704 163,61.5C 163.847,64.5876 163.347,67.4209 161.5,70C 163.167,70.8333 164.833,71.6667 166.5,72.5C 167.332,81.8155 167.832,91.1489 168,100.5C 172.748,101.635 177.581,101.802 182.5,101C 183.743,99.1303 184.41,96.9637 184.5,94.5C 183.934,87.5049 183.434,80.5049 183,73.5C 180.51,72.257 178.01,71.0904 175.5,70C 179,65.1667 182.5,60.3333 186,55.5C 185.794,48.4066 185.127,41.4066 184,34.5C 189.029,31.4042 194.363,30.4042 200,31.5C 200.202,33.2556 200.702,34.9223 201.5,36.5C 210.817,29.0666 221.484,26.3999 233.5,28.5C 247.175,28.758 260.842,29.258 274.5,30C 286.675,40.6769 298.675,51.5102 310.5,62.5C 308.78,63.9197 306.946,65.253 305,66.5C 304.333,76.8333 304.333,87.1667 305,97.5C 306.067,98.1905 307.234,98.6905 308.5,99C 309.364,101.071 310.697,102.738 312.5,104C 322.5,104.667 332.5,104.667 342.5,104C 341.833,102.667 341.167,101.333 340.5,100C 335.919,98.7247 331.252,97.7247 326.5,97C 323.365,93.4826 322.698,89.4826 324.5,85C 317.126,79.006 314.96,71.506 318,62.5C 319.841,59.8221 322.341,58.1555 325.5,57.5C 326.66,53.5411 326.827,49.5411 326,45.5C 325.586,45.9574 325.086,46.2907 324.5,46.5C 319.756,43.0932 314.756,40.0932 309.5,37.5C 311.021,36.0627 312.855,35.396 315,35.5C 318.296,36.0631 321.462,37.0631 324.5,38.5C 327.978,37.4223 331.478,36.4223 335,35.5C 335.945,35.0946 336.612,34.428 337,33.5C 337.333,31.5 337.667,29.5 338,27.5C 341.465,25.0374 344.631,22.2041 347.5,19C 349.631,18.4569 351.631,17.6236 353.5,16.5C 356.913,10.2378 362.246,7.57109 369.5,8.5C 380.002,8.82619 386.502,14.1595 389,24.5C 392.92,20.6327 396.92,20.6327 401,24.5C 402.544,26.4682 403.711,28.6349 404.5,31C 412.308,28.463 416.808,31.2964 418,39.5C 428.006,45.8534 431.673,54.8534 429,66.5C 427.527,70.7789 425.527,74.7789 423,78.5C 422.195,89.2316 416.861,92.565 407,88.5C 406.535,89.9308 405.701,91.0975 404.5,92C 397.748,93.8747 391.748,97.0414 386.5,101.5C 387.167,102.5 387.833,103.5 388.5,104.5C 395.803,105.497 403.137,105.831 410.5,105.5C 410.146,99.7962 412.813,97.9629 418.5,100C 419,100.5 419.5,101 420,101.5C 421.247,110.564 422.747,119.564 424.5,128.5C 430.425,125.914 436.591,124.247 443,123.5C 449.131,124.912 455.298,125.746 461.5,126C 466.814,123.368 472.147,120.701 477.5,118C 479.436,117.187 481.436,116.687 483.5,116.5C 483.261,96.3219 483.595,76.1553 484.5,56C 484.293,50.6495 483.293,45.4828 481.5,40.5C 483.05,30.2348 488.05,22.0682 496.5,16C 505.318,12.4395 513.985,12.7728 522.5,17C 534.11,24.7199 540.11,35.5532 540.5,49.5C 539.167,49.5 537.833,49.5 536.5,49.5C 536.624,67.8897 537.457,86.223 539,104.5C 543.445,102.74 547.945,101.073 552.5,99.5C 552.833,98.6667 553.167,97.8333 553.5,97C 554.777,86.1744 554.777,75.3411 553.5,64.5C 551.929,63.6906 550.262,63.3573 548.5,63.5C 549.569,56.3505 552.069,49.6838 556,43.5C 562.403,33.7602 569.57,24.5935 577.5,16C 582.5,15.3333 587.5,15.3333 592.5,16C 614.807,18.9282 636.807,23.2615 658.5,29C 673.077,40.1542 680.41,54.8209 680.5,73C 678.564,73.813 676.564,74.313 674.5,74.5C 674.5,83.1667 674.5,91.8333 674.5,100.5C 683.488,101.764 692.488,101.764 701.5,100.5C 701.289,94.9414 703.622,93.4414 708.5,96C 709.224,99.7705 711.557,101.437 715.5,101C 724.186,101.832 732.853,101.665 741.5,100.5C 742.55,98.8983 741.883,98.065 739.5,98C 740.167,97.3333 740.833,96.6667 741.5,96C 737.608,94.3599 733.608,93.0265 729.5,92C 725.918,88.263 724.085,83.763 724,78.5C 717.64,72.1921 715.973,64.8588 719,56.5C 722.234,53.9845 725.567,51.6512 729,49.5C 730.134,41.5416 734.634,38.5416 742.5,40.5C 742.335,37.4816 742.502,34.4816 743,31.5C 746.763,28.1273 749.93,24.2939 752.5,20C 756.333,20.1667 758.167,18.3333 758,14.5C 763.755,7.96337 770.922,5.79671 779.5,8C 784.415,10.9198 787.915,15.0865 790,20.5C 794.78,17.5282 798.78,18.5282 802,23.5C 802.219,24.938 802.719,26.2713 803.5,27.5C 811.3,25.2824 815.634,28.2824 816.5,36.5C 817.914,37.624 819.414,38.624 821,39.5C 821.92,42.9382 823.587,45.9382 826,48.5C 826.667,53.8333 826.667,59.1667 826,64.5C 822.48,69.235 820.48,74.5684 820,80.5C 816.867,85.8186 812.367,87.9853 806.5,87C 802.697,89.0711 799.03,91.4044 795.5,94C 792.341,94.8587 789.341,96.0254 786.5,97.5C 787.074,99.0589 787.74,100.559 788.5,102C 795.826,102.5 803.159,102.666 810.5,102.5C 810.5,100.5 810.5,98.5 810.5,96.5C 813.187,96.3359 815.854,96.5026 818.5,97C 819.126,97.7504 819.626,98.5838 820,99.5C 821.3,108.501 822.466,117.501 823.5,126.5C 827.107,127.489 830.774,127.823 834.5,127.5C 834.69,117.09 835.524,106.756 837,96.5C 842.628,93.3978 845.462,95.0645 845.5,101.5C 852.175,101.666 858.842,101.5 865.5,101C 866.167,100 866.833,99 867.5,98C 860.833,94 854.167,90 847.5,86C 844.833,85.6667 842.167,85.3333 839.5,85C 829.315,72.0924 825.815,57.5924 829,41.5C 831.488,39.2567 834.154,37.2567 837,35.5C 837.333,32.5 837.667,29.5 838,26.5C 841.745,25.703 845.579,25.203 849.5,25C 849.794,19.87 852.461,16.7034 857.5,15.5C 859.116,16.1152 860.449,17.1152 861.5,18.5C 865.948,3.95602 874.948,0.456024 888.5,8C 890.317,9.79976 891.484,11.9664 892,14.5C 894.628,16.1255 896.961,18.1255 899,20.5C 900.569,24.072 902.902,27.072 906,29.5C 907.698,32.3963 907.864,35.3963 906.5,38.5C 915.452,37.4311 920.785,41.4311 922.5,50.5C 932.61,53.3874 935.777,59.7207 932,69.5C 929.971,72.0292 928.138,74.6959 926.5,77.5C 928.528,82.4116 928.862,87.4116 927.5,92.5C 922.351,94.2903 917.018,95.6236 911.5,96.5C 911.5,98.1667 911.5,99.8333 911.5,101.5C 920.506,101.666 929.506,101.5 938.5,101C 941.876,100.693 943.21,98.8595 942.5,95.5C 944.895,94.6365 947.229,94.6365 949.5,95.5C 950.093,97.3817 950.76,99.2151 951.5,101C 959.525,101.301 967.525,101.801 975.5,102.5C 974.439,93.651 971.773,85.3177 967.5,77.5C 969.665,71.0035 972.998,65.1702 977.5,60C 984.077,57.6037 989.91,58.7704 995,63.5C 995.847,66.5876 995.347,69.4209 993.5,72C 995.167,72.8333 996.833,73.6667 998.5,74.5C 999.332,83.8155 999.832,93.1489 1000,102.5C 1004.75,103.635 1009.58,103.802 1014.5,103C 1015.74,101.13 1016.41,98.9637 1016.5,96.5C 1015.93,89.5049 1015.43,82.5049 1015,75.5C 1012.51,74.257 1010.01,73.0904 1007.5,72C 1011,67.1667 1014.5,62.3333 1018,57.5C 1017.79,50.4066 1017.13,43.4066 1016,36.5C 1021.03,33.4042 1026.36,32.4042 1032,33.5C 1032.2,35.2556 1032.7,36.9223 1033.5,38.5C 1042.82,31.0666 1053.48,28.3999 1065.5,30.5C 1079.18,30.758 1092.84,31.258 1106.5,32C 1118.68,42.6769 1130.68,53.5102 1142.5,64.5C 1140.78,65.9197 1138.95,67.253 1137,68.5C 1136.33,78.8333 1136.33,89.1667 1137,99.5C 1138.07,100.191 1139.23,100.691 1140.5,101C 1141.36,103.071 1142.7,104.738 1144.5,106C 1154.5,106.667 1164.5,106.667 1174.5,106C 1173.83,104.667 1173.17,103.333 1172.5,102C 1167.92,100.725 1163.25,99.7247 1158.5,99C 1155.37,95.4826 1154.7,91.4826 1156.5,87C 1149.13,81.006 1146.96,73.506 1150,64.5C 1151.84,61.8221 1154.34,60.1555 1157.5,59.5C 1158.66,55.5411 1158.83,51.5411 1158,47.5C 1157.59,47.9574 1157.09,48.2907 1156.5,48.5C 1151.76,45.0932 1146.76,42.0932 1141.5,39.5C 1143.02,38.0627 1144.85,37.396 1147,37.5C 1150.3,38.0631 1153.46,39.0631 1156.5,40.5C 1159.98,39.4223 1163.48,38.4223 1167,37.5C 1167.95,37.0946 1168.61,36.428 1169,35.5C 1169.33,33.5 1169.67,31.5 1170,29.5C 1173.46,27.0374 1176.63,24.2041 1179.5,21C 1181.63,20.4569 1183.63,19.6236 1185.5,18.5C 1188.91,12.2378 1194.25,9.57109 1201.5,10.5C 1212,10.8262 1218.5,16.1595 1221,26.5C 1224.92,22.6327 1228.92,22.6327 1233,26.5C 1234.54,28.4682 1235.71,30.6349 1236.5,33C 1244.31,30.463 1248.81,33.2964 1250,41.5C 1260.01,47.8534 1263.67,56.8534 1261,68.5C 1259.53,72.7789 1257.53,76.7789 1255,80.5C 1254.19,91.2316 1248.86,94.565 1239,90.5C 1238.53,91.9308 1237.7,93.0975 1236.5,94C 1229.75,95.8747 1223.75,99.0414 1218.5,103.5C 1219.17,104.5 1219.83,105.5 1220.5,106.5C 1227.8,107.497 1235.14,107.831 1242.5,107.5C 1242.15,101.796 1244.81,99.9629 1250.5,102C 1251,102.5 1251.5,103 1252,103.5C 1253.25,112.564 1254.75,121.564 1256.5,130.5C 1262.42,127.914 1268.59,126.247 1275,125.5C 1281.13,126.912 1287.3,127.746 1293.5,128C 1298.81,125.368 1304.15,122.701 1309.5,120C 1311.44,119.187 1313.44,118.687 1315.5,118.5C 1315.26,98.3219 1315.59,78.1553 1316.5,58C 1316.29,52.6495 1315.29,47.4828 1313.5,42.5C 1315.05,32.2348 1320.05,24.0682 1328.5,18C 1337.32,14.4395 1345.99,14.7728 1354.5,19C 1366.11,26.7199 1372.11,37.5532 1372.5,51.5C 1371.17,51.5 1369.83,51.5 1368.5,51.5C 1368.62,69.8897 1369.46,88.223 1371,106.5C 1375.44,104.74 1379.94,103.073 1384.5,101.5C 1384.83,100.667 1385.17,99.8333 1385.5,99C 1386.78,88.1744 1386.78,77.3411 1385.5,66.5C 1383.93,65.6906 1382.26,65.3573 1380.5,65.5C 1381.57,58.3505 1384.07,51.6838 1388,45.5C 1394.4,35.7602 1401.57,26.5935 1409.5,18C 1414.5,17.3333 1419.5,17.3333 1424.5,18C 1446.81,20.9282 1468.81,25.2615 1490.5,31C 1505.08,42.1542 1512.41,56.8209 1512.5,75C 1510.56,75.813 1508.56,76.313 1506.5,76.5C 1506.5,85.1667 1506.5,93.8333 1506.5,102.5C 1515.49,103.764 1524.49,103.764 1533.5,102.5C 1533.29,96.9414 1535.62,95.4414 1540.5,98C 1541.22,101.771 1543.56,103.437 1547.5,103C 1556.19,103.832 1564.85,103.665 1573.5,102.5C 1574.55,100.898 1573.88,100.065 1571.5,100C 1572.17,99.3333 1572.83,98.6667 1573.5,98C 1569.61,96.3599 1565.61,95.0265 1561.5,94C 1557.92,90.263 1556.08,85.763 1556,80.5C 1549.64,74.1921 1547.97,66.8588 1551,58.5C 1554.23,55.9845 1557.57,53.6512 1561,51.5C 1562.13,43.5416 1566.63,40.5416 1574.5,42.5C 1574.34,39.4816 1574.5,36.4816 1575,33.5C 1578.76,30.1273 1581.93,26.2939 1584.5,22C 1588.33,22.1667 1590.17,20.3333 1590,16.5C 1595.76,9.96337 1602.92,7.79671 1611.5,10C 1616.41,12.9198 1619.91,17.0865 1622,22.5C 1626.78,19.5282 1630.78,20.5282 1634,25.5C 1634.22,26.938 1634.72,28.2713 1635.5,29.5C 1643.3,27.2824 1647.63,30.2824 1648.5,38.5C 1649.91,39.624 1651.41,40.624 1653,41.5C 1653.92,44.9382 1655.59,47.9382 1658,50.5C 1658.67,55.8333 1658.67,61.1667 1658,66.5C 1654.48,71.235 1652.48,76.5684 1652,82.5C 1648.87,87.8186 1644.37,89.9853 1638.5,89C 1634.7,91.0711 1631.03,93.4044 1627.5,96C 1624.34,96.8587 1621.34,98.0254 1618.5,99.5C 1619.07,101.059 1619.74,102.559 1620.5,104C 1627.83,104.5 1635.16,104.666 1642.5,104.5C 1642.5,102.5 1642.5,100.5 1642.5,98.5C 1645.19,98.3359 1647.85,98.5026 1650.5,99C 1651.13,99.7504 1651.63,100.584 1652,101.5C 1653.5,111.467 1654.66,121.467 1655.5,131.5C 1661.83,131.5 1668.17,131.5 1674.5,131.5C 1674.69,121.09 1675.52,110.756 1677,100.5C 1682.63,97.3978 1685.46,99.0645 1685.5,105.5C 1692.17,105.666 1698.84,105.5 1705.5,105C 1706.17,104 1706.83,103 1707.5,102C 1700.83,98 1694.17,94 1687.5,90C 1684.83,89.6667 1682.17,89.3333 1679.5,89C 1669.31,76.0924 1665.81,61.5924 1669,45.5C 1671.49,43.2567 1674.15,41.2567 1677,39.5C 1677.33,36.5 1677.67,33.5 1678,30.5C 1681.75,29.703 1685.58,29.203 1689.5,29C 1689.79,23.87 1692.46,20.7034 1697.5,19.5C 1699.12,20.1152 1700.45,21.1152 1701.5,22.5C 1705.95,7.95602 1714.95,4.45602 1728.5,12C 1730.32,13.7998 1731.48,15.9664 1732,18.5C 1734.63,20.1255 1736.96,22.1255 1739,24.5C 1740.57,28.072 1742.9,31.072 1746,33.5C 1747.7,36.3963 1747.86,39.3963 1746.5,42.5C 1755.45,41.4311 1760.79,45.4311 1762.5,54.5C 1772.61,57.3874 1775.78,63.7207 1772,73.5C 1769.97,76.0292 1768.14,78.6959 1766.5,81.5C 1768.53,86.4116 1768.86,91.4116 1767.5,96.5C 1762.35,98.2903 1757.02,99.6236 1751.5,100.5C 1751.5,102.167 1751.5,103.833 1751.5,105.5C 1760.51,105.666 1769.51,105.5 1778.5,105C 1781.88,104.693 1783.21,102.86 1782.5,99.5C 1784.9,98.6365 1787.23,98.6365 1789.5,99.5C 1790.09,101.382 1790.76,103.215 1791.5,105C 1799.53,105.301 1807.53,105.801 1815.5,106.5C 1814.44,97.651 1811.77,89.3177 1807.5,81.5C 1809.67,75.0035 1813,69.1702 1817.5,64C 1824.08,61.6037 1829.91,62.7704 1835,67.5C 1835.85,70.5876 1835.35,73.4209 1833.5,76C 1835.17,76.8333 1836.83,77.6667 1838.5,78.5C 1839.33,87.8155 1839.83,97.1489 1840,106.5C 1844.75,107.635 1849.58,107.802 1854.5,107C 1855.74,105.13 1856.41,102.964 1856.5,100.5C 1855.93,93.5049 1855.43,86.5049 1855,79.5C 1852.51,78.257 1850.01,77.0904 1847.5,76C 1851,71.1667 1854.5,66.3333 1858,61.5C 1857.79,54.4066 1857.13,47.4066 1856,40.5C 1861.03,37.4042 1866.36,36.4042 1872,37.5C 1872.2,39.2556 1872.7,40.9223 1873.5,42.5C 1877.21,38.9758 1881.54,36.4758 1886.5,35C 1897.49,34.5002 1908.49,34.3335 1919.5,34.5 Z M 56.5,62.5 C 58.9513,64.2098 58.9513,66.2098 56.5,68.5C 55.5,68.1667 54.8333,67.5 54.5,66.5C 55.1799,65.1401 55.8466,63.8068 56.5,62.5 Z M 888.5,64.5 C 890.951,66.2098 890.951,68.2098 888.5,70.5C 887.5,70.1667 886.833,69.5 886.5,68.5C 887.18,67.1401 887.847,65.8068 888.5,64.5 Z M 760.5,65.5 C 765.383,66.7843 766.216,69.1177 763,72.5C 761.097,70.559 760.264,68.2257 760.5,65.5 Z M 1592.5,67.5 C 1597.38,68.7843 1598.22,71.1177 1595,74.5C 1593.1,72.559 1592.26,70.2257 1592.5,67.5 Z M 1728.5,68.5 C 1730.95,70.2098 1730.95,72.2098 1728.5,74.5C 1727.5,74.1667 1726.83,73.5 1726.5,72.5C 1727.18,71.1401 1727.85,69.8068 1728.5,68.5 Z M 51.5,71.5 C 52.891,75.7691 52.7244,80.1024 51,84.5C 49.2103,80.6732 48.0436,76.6732 47.5,72.5C 49.099,72.7678 50.4324,72.4345 51.5,71.5 Z M 883.5,73.5 C 884.891,77.7691 884.724,82.1024 883,86.5C 881.21,82.6732 880.044,78.6732 879.5,74.5C 881.099,74.7678 882.432,74.4345 883.5,73.5 Z M 61.5,75.5 C 65.3639,78.132 67.0306,81.632 66.5,86C 64.6547,87.128 62.6547,87.628 60.5,87.5C 60.3946,83.4517 60.7279,79.4517 61.5,75.5 Z M 767.5,75.5 C 769.714,75.2907 771.381,76.124 772.5,78C 772.035,81.0019 771.201,83.8352 770,86.5C 768.763,82.9813 767.93,79.3146 767.5,75.5 Z M 893.5,77.5 C 897.364,80.132 899.031,83.632 898.5,88C 896.655,89.128 894.655,89.628 892.5,89.5C 892.395,85.4517 892.728,81.4517 893.5,77.5 Z M 1599.5,77.5 C 1601.71,77.2907 1603.38,78.124 1604.5,80C 1604.03,83.0019 1603.2,85.8352 1602,88.5C 1600.76,84.9813 1599.93,81.3146 1599.5,77.5 Z M 1723.5,77.5 C 1724.89,81.7691 1724.72,86.1024 1723,90.5C 1721.21,86.6732 1720.04,82.6732 1719.5,78.5C 1721.1,78.7678 1722.43,78.4345 1723.5,77.5 Z M 32.5,78.5 C 37.3689,80.1822 39.5356,83.5156 39,88.5C 35.3119,86.0114 33.1452,82.6781 32.5,78.5 Z M 370.5,78.5 C 371.16,79.964 371.493,81.6307 371.5,83.5C 371.116,90.6042 370.116,90.9375 368.5,84.5C 368.611,82.2857 369.278,80.2857 370.5,78.5 Z M 756.5,79.5 C 758.419,81.4211 759.419,83.9211 759.5,87C 759.612,88.6624 758.946,89.8291 757.5,90.5C 755.246,89.8763 753.579,88.543 752.5,86.5C 753.861,84.1495 755.194,81.8162 756.5,79.5 Z M 864.5,80.5 C 869.369,82.1822 871.536,85.5156 871,90.5C 867.312,88.0114 865.145,84.6781 864.5,80.5 Z M 1202.5,80.5 C 1203.16,81.964 1203.49,83.6307 1203.5,85.5C 1203.12,92.6042 1202.12,92.9375 1200.5,86.5C 1200.61,84.2857 1201.28,82.2857 1202.5,80.5 Z M 357.5,81.5 C 358.117,81.6107 358.617,81.944 359,82.5C 359.868,85.4853 360.201,88.4853 360,91.5C 356.886,93.1786 354.719,92.3452 353.5,89C 354.644,86.3675 355.977,83.8675 357.5,81.5 Z M 1588.5,81.5 C 1590.42,83.4211 1591.42,85.9211 1591.5,89C 1591.61,90.6624 1590.95,91.8291 1589.5,92.5C 1587.25,91.8763 1585.58,90.543 1584.5,88.5C 1585.86,86.1495 1587.19,83.8162 1588.5,81.5 Z M 1733.5,81.5 C 1737.36,84.132 1739.03,87.632 1738.5,92C 1736.65,93.128 1734.65,93.628 1732.5,93.5C 1732.39,89.4517 1732.73,85.4517 1733.5,81.5 Z M 786.5,82.5 C 787.5,82.5 788.5,82.5 789.5,82.5C 788.857,86.7734 786.524,89.7734 782.5,91.5C 781.417,90.635 781.251,89.635 782,88.5C 783.277,86.2528 784.777,84.2528 786.5,82.5 Z M 1189.5,83.5 C 1190.12,83.6107 1190.62,83.944 1191,84.5C 1191.87,87.4853 1192.2,90.4853 1192,93.5C 1188.89,95.1786 1186.72,94.3452 1185.5,91C 1186.64,88.3675 1187.98,85.8675 1189.5,83.5 Z M 1618.5,84.5 C 1619.5,84.5 1620.5,84.5 1621.5,84.5C 1620.86,88.7734 1618.52,91.7734 1614.5,93.5C 1613.42,92.635 1613.25,91.635 1614,90.5C 1615.28,88.2528 1616.78,86.2528 1618.5,84.5 Z M 1704.5,84.5 C 1709.37,86.1822 1711.54,89.5156 1711,94.5C 1707.31,92.0114 1705.15,88.6781 1704.5,84.5 Z M 386.5,85.5 C 387.5,85.5 388.5,85.5 389.5,85.5C 389.13,89.5454 387.13,92.5454 383.5,94.5C 382.5,94.1667 381.833,93.5 381.5,92.5C 382.804,89.8699 384.47,87.5366 386.5,85.5 Z M 1218.5,87.5 C 1219.5,87.5 1220.5,87.5 1221.5,87.5C 1221.13,91.5454 1219.13,94.5454 1215.5,96.5C 1214.5,96.1667 1213.83,95.5 1213.5,94.5C 1214.8,91.8699 1216.47,89.5366 1218.5,87.5 Z M 71.5,89.5 C 76.4277,90.8915 76.5944,92.5582 72,94.5C 71.5065,96.1341 71.3398,97.8008 71.5,99.5C 64.8772,100.38 58.2105,100.713 51.5,100.5C 51.178,98.2776 51.8446,96.4443 53.5,95C 59.6432,93.2269 65.6432,91.3936 71.5,89.5 Z M 747.5,90.5 C 749.01,91.8361 750.677,93.0027 752.5,94C 756.896,94.7458 761.23,95.7458 765.5,97C 772.5,100.746 772.166,102.912 764.5,103.5C 759.519,102.835 754.519,102.502 749.5,102.5C 750.575,98.062 748.908,96.062 744.5,96.5C 744.615,94.0532 745.615,92.0532 747.5,90.5 Z M 903.5,91.5 C 908.428,92.8915 908.594,94.5582 904,96.5C 903.506,98.1341 903.34,99.8008 903.5,101.5C 896.877,102.38 890.211,102.713 883.5,102.5C 883.178,100.278 883.845,98.4443 885.5,97C 891.643,95.2269 897.643,93.3936 903.5,91.5 Z M 1579.5,92.5 C 1581.01,93.8361 1582.68,95.0027 1584.5,96C 1588.9,96.7458 1593.23,97.7458 1597.5,99C 1604.5,102.746 1604.17,104.912 1596.5,105.5C 1591.52,104.835 1586.52,104.502 1581.5,104.5C 1582.57,100.062 1580.91,98.062 1576.5,98.5C 1576.62,96.0532 1577.62,94.0532 1579.5,92.5 Z M 346.5,95.5 C 354.657,96.0946 362.324,98.2613 369.5,102C 367.527,102.495 365.527,102.662 363.5,102.5C 363.5,103.833 363.5,105.167 363.5,106.5C 359.085,106.695 354.751,106.195 350.5,105C 349.773,103.601 349.273,102.101 349,100.5C 345.98,99.6998 345.146,98.0331 346.5,95.5 Z M 1743.5,95.5 C 1748.43,96.8915 1748.59,98.5582 1744,100.5C 1743.51,102.134 1743.34,103.801 1743.5,105.5C 1736.88,106.38 1730.21,106.713 1723.5,106.5C 1723.18,104.278 1723.84,102.444 1725.5,101C 1731.64,99.2269 1737.64,97.3936 1743.5,95.5 Z M 1178.5,97.5 C 1186.66,98.0946 1194.32,100.261 1201.5,104C 1199.53,104.495 1197.53,104.662 1195.5,104.5C 1195.5,105.833 1195.5,107.167 1195.5,108.5C 1191.08,108.695 1186.75,108.195 1182.5,107C 1181.77,105.601 1181.27,104.101 1181,102.5C 1177.98,101.7 1177.15,100.033 1178.5,97.5 Z M 12.5,109.5 C 20.1667,109.5 27.8333,109.5 35.5,109.5C 35.5,110.833 35.5,112.167 35.5,113.5C 28.1591,113.666 20.8258,113.5 13.5,113C 12.6634,111.989 12.33,110.822 12.5,109.5 Z M 51.5,109.5 C 57.1764,109.334 62.8431,109.501 68.5,110C 69.3366,111.011 69.67,112.178 69.5,113.5C 62.8002,113.83 56.1335,113.497 49.5,112.5C 50.1925,111.482 50.8592,110.482 51.5,109.5 Z M 81.5,109.5 C 90.5326,109.394 99.5326,109.894 108.5,111C 109.451,111.718 109.617,112.551 109,113.5C 98.876,114.775 88.7094,114.775 78.5,113.5C 78.5,112.5 78.5,111.5 78.5,110.5C 79.791,110.737 80.791,110.404 81.5,109.5 Z M 713.5,111.5 C 722.839,111.334 732.173,111.5 741.5,112C 742.167,112.667 742.833,113.333 743.5,114C 738.827,115.25 733.994,116.083 729,116.5C 723.203,115.614 717.369,115.28 711.5,115.5C 711.527,113.801 712.194,112.467 713.5,111.5 Z M 751.5,111.5 C 758.206,111.448 764.872,111.948 771.5,113C 772.167,113.667 772.833,114.333 773.5,115C 766.201,116.036 758.868,116.536 751.5,116.5C 751.5,114.833 751.5,113.167 751.5,111.5 Z M 788.5,111.5 C 796.479,112.224 804.479,112.558 812.5,112.5C 812.5,113.833 812.5,115.167 812.5,116.5C 804.167,116.5 795.833,116.5 787.5,116.5C 787.366,114.708 787.699,113.042 788.5,111.5 Z M 844.5,111.5 C 852.167,111.5 859.833,111.5 867.5,111.5C 867.5,112.833 867.5,114.167 867.5,115.5C 860.159,115.666 852.826,115.5 845.5,115C 844.663,113.989 844.33,112.822 844.5,111.5 Z M 883.5,111.5 C 889.176,111.334 894.843,111.501 900.5,112C 901.337,113.011 901.67,114.178 901.5,115.5C 894.8,115.83 888.134,115.497 881.5,114.5C 882.193,113.482 882.859,112.482 883.5,111.5 Z M 913.5,111.5 C 922.533,111.394 931.533,111.894 940.5,113C 941.451,113.718 941.617,114.551 941,115.5C 930.876,116.775 920.709,116.775 910.5,115.5C 910.5,114.5 910.5,113.5 910.5,112.5C 911.791,112.737 912.791,112.404 913.5,111.5 Z M 1545.5,113.5 C 1554.84,113.334 1564.17,113.5 1573.5,114C 1574.17,114.667 1574.83,115.333 1575.5,116C 1570.83,117.25 1565.99,118.083 1561,118.5C 1555.2,117.614 1549.37,117.28 1543.5,117.5C 1543.53,115.801 1544.19,114.467 1545.5,113.5 Z M 1583.5,113.5 C 1590.21,113.448 1596.87,113.948 1603.5,115C 1604.17,115.667 1604.83,116.333 1605.5,117C 1598.2,118.036 1590.87,118.536 1583.5,118.5C 1583.5,116.833 1583.5,115.167 1583.5,113.5 Z M 1620.5,113.5 C 1628.48,114.224 1636.48,114.558 1644.5,114.5C 1644.5,115.833 1644.5,117.167 1644.5,118.5C 1636.17,118.5 1627.83,118.5 1619.5,118.5C 1619.37,116.708 1619.7,115.042 1620.5,113.5 Z M 315.5,114.5 C 323.84,114.334 332.173,114.5 340.5,115C 341.91,115.368 342.91,116.201 343.5,117.5C 332.854,118.499 322.187,118.832 311.5,118.5C 311.5,117.5 311.5,116.5 311.5,115.5C 313.099,115.768 314.432,115.434 315.5,114.5 Z M 354.5,114.5 C 360.485,115.076 366.485,115.576 372.5,116C 373.337,117.011 373.67,118.178 373.5,119.5C 366.49,119.332 359.49,118.999 352.5,118.5C 352.527,116.801 353.194,115.467 354.5,114.5 Z M 387.5,115.5 C 395.833,115.5 404.167,115.5 412.5,115.5C 412.5,116.833 412.5,118.167 412.5,119.5C 404.167,119.5 395.833,119.5 387.5,119.5C 387.5,118.167 387.5,116.833 387.5,115.5 Z M 1684.5,115.5 C 1692.17,115.5 1699.83,115.5 1707.5,115.5C 1707.5,116.833 1707.5,118.167 1707.5,119.5C 1700.16,119.666 1692.83,119.5 1685.5,119C 1684.66,117.989 1684.33,116.822 1684.5,115.5 Z M 1723.5,115.5 C 1729.18,115.334 1734.84,115.501 1740.5,116C 1741.34,117.011 1741.67,118.178 1741.5,119.5C 1734.8,119.83 1728.13,119.497 1721.5,118.5C 1722.19,117.482 1722.86,116.482 1723.5,115.5 Z M 1753.5,115.5 C 1762.53,115.394 1771.53,115.894 1780.5,117C 1781.45,117.718 1781.62,118.551 1781,119.5C 1770.88,120.775 1760.71,120.775 1750.5,119.5C 1750.5,118.5 1750.5,117.5 1750.5,116.5C 1751.79,116.737 1752.79,116.404 1753.5,115.5 Z M 1147.5,116.5 C 1155.84,116.334 1164.17,116.5 1172.5,117C 1173.91,117.368 1174.91,118.201 1175.5,119.5C 1164.85,120.499 1154.19,120.832 1143.5,120.5C 1143.5,119.5 1143.5,118.5 1143.5,117.5C 1145.1,117.768 1146.43,117.434 1147.5,116.5 Z M 1186.5,116.5 C 1192.49,117.076 1198.49,117.576 1204.5,118C 1205.34,119.011 1205.67,120.178 1205.5,121.5C 1198.49,121.332 1191.49,120.999 1184.5,120.5C 1184.53,118.801 1185.19,117.467 1186.5,116.5 Z M 1219.5,117.5 C 1227.83,117.5 1236.17,117.5 1244.5,117.5C 1244.5,118.833 1244.5,120.167 1244.5,121.5C 1236.17,121.5 1227.83,121.5 1219.5,121.5C 1219.5,120.167 1219.5,118.833 1219.5,117.5 Z M 788.5,126.5 C 796.611,126.065 804.611,126.565 812.5,128C 803.833,128.667 795.167,128.667 786.5,128C 787.416,127.722 788.082,127.222 788.5,126.5 Z M 1620.5,128.5 C 1627.84,128.334 1635.17,128.5 1642.5,129C 1645.07,130.397 1644.9,131.564 1642,132.5C 1641.05,131.978 1640.22,131.311 1639.5,130.5C 1638.05,131.617 1636.38,132.284 1634.5,132.5C 1631.88,131.846 1629.38,131.846 1627,132.5C 1624.72,129.936 1622.72,130.27 1621,133.5C 1620.53,132.069 1619.7,130.903 1618.5,130C 1619.42,129.722 1620.08,129.222 1620.5,128.5 Z"/></g>
<g><path style="opacity:0.451" fill="#4caf4f" d="M 30.5,126.5 C 32.3841,126.712 33.7174,127.712 34.5,129.5C 31.5812,130.932 30.2479,129.932 30.5,126.5 Z"/></g>
<g><path style="opacity:0.426" fill="#4caf4f" d="M 26.5,127.5 C 27.9953,127.415 28.9953,128.081 29.5,129.5C 27.6043,130.639 26.6043,129.973 26.5,127.5 Z"/></g>
<g><path style="opacity:0.342" fill="#4caf4f" d="M 57.5,127.5 C 59.337,127.639 59.6704,128.306 58.5,129.5C 57.7025,129.043 57.3691,128.376 57.5,127.5 Z"/></g>
<g><path style="opacity:0.39" fill="#4caf4f" d="M 52.5,128.5 C 53.9778,128.238 55.3112,128.571 56.5,129.5C 55.6798,131.111 54.3464,131.778 52.5,131.5C 52.5,130.5 52.5,129.5 52.5,128.5 Z"/></g>
<g><path style="opacity:0.298" fill="#4caf4f" d="M 100.5,129.5 C 100.728,131.355 100.061,132.188 98.5,132C 97.1404,129.192 97.8071,128.359 100.5,129.5 Z"/></g>
<g><path style="opacity:0.451" fill="#4caf4f" d="M 862.5,128.5 C 864.384,128.712 865.717,129.712 866.5,131.5C 863.581,132.932 862.248,131.932 862.5,128.5 Z"/></g>
<g><path style="opacity:0.433" fill="#4caf4f" d="M 722.5,129.5 C 724.048,129.821 724.715,130.821 724.5,132.5C 721.752,132.598 721.086,131.598 722.5,129.5 Z"/></g>
<g><path style="opacity:0.426" fill="#4caf4f" d="M 858.5,129.5 C 859.995,129.415 860.995,130.081 861.5,131.5C 859.604,132.639 858.604,131.973 858.5,129.5 Z"/></g>
<g><path style="opacity:0.342" fill="#4caf4f" d="M 889.5,129.5 C 891.337,129.639 891.67,130.306 890.5,131.5C 889.702,131.043 889.369,130.376 889.5,129.5 Z"/></g>
<g><path style="opacity:0.39" fill="#4caf4f" d="M 884.5,130.5 C 885.978,130.238 887.311,130.571 888.5,131.5C 887.68,133.111 886.346,133.778 884.5,133.5C 884.5,132.5 884.5,131.5 884.5,130.5 Z"/></g>
<g><path style="opacity:0.298" fill="#4caf4f" d="M 932.5,131.5 C 932.728,133.355 932.061,134.188 930.5,134C 929.14,131.192 929.807,130.359 932.5,131.5 Z"/></g>
<g><path style="opacity:0.274" fill="#4caf4f" d="M 1595.5,130.5 C 1597.33,131.14 1597.67,132.14 1596.5,133.5C 1595.61,132.675 1595.28,131.675 1595.5,130.5 Z"/></g>
<g><path style="opacity:0.433" fill="#4caf4f" d="M 1554.5,131.5 C 1556.05,131.821 1556.71,132.821 1556.5,134.5C 1553.75,134.598 1553.09,133.598 1554.5,131.5 Z"/></g>
<g><path style="opacity:0.347" fill="#4caf4f" d="M 321.5,133.5 C 323.76,132.114 324.594,132.781 324,135.5C 322.187,136.609 321.354,135.942 321.5,133.5 Z"/></g>
<g><path style="opacity:0.53" fill="#4caf4f" d="M 367.5,132.5 C 369.912,132.694 370.912,134.027 370.5,136.5C 369.5,136.5 368.5,136.5 367.5,136.5C 367.5,135.167 367.5,133.833 367.5,132.5 Z"/></g>
<g><path style="opacity:0.49" fill="#4caf4f" d="M 390.5,132.5 C 395.276,132.671 395.609,133.671 391.5,135.5C 389.197,135.124 388.863,134.124 390.5,132.5 Z"/></g>
<g><path style="opacity:0.385" fill="#4caf4f" d="M 396.5,132.5 C 399.018,133.759 398.852,135.093 396,136.5C 395.421,135.072 395.588,133.738 396.5,132.5 Z"/></g>
<g><path style="opacity:0.451" fill="#4caf4f" d="M 1702.5,132.5 C 1704.38,132.712 1705.72,133.712 1706.5,135.5C 1703.58,136.932 1702.25,135.932 1702.5,132.5 Z"/></g>
<g><path style="opacity:0.347" fill="#4caf4f" d="M 630.5,133.5 C 635.796,134.688 635.796,135.522 630.5,136C 629.402,135.23 629.402,134.397 630.5,133.5 Z"/></g>
<g><path style="opacity:0.426" fill="#4caf4f" d="M 1698.5,133.5 C 1700,133.415 1701,134.081 1701.5,135.5C 1699.6,136.639 1698.6,135.973 1698.5,133.5 Z"/></g>
<g><path style="opacity:0.342" fill="#4caf4f" d="M 1729.5,133.5 C 1731.34,133.639 1731.67,134.306 1730.5,135.5C 1729.7,135.043 1729.37,134.376 1729.5,133.5 Z"/></g>
<g><path style="opacity:0.347" fill="#4caf4f" d="M 1155.5,138.5 C 1154.83,138.5 1154.17,138.5 1153.5,138.5C 1153.5,137.5 1153.5,136.5 1153.5,135.5C 1156.4,134.506 1157.07,135.506 1155.5,138.5 Z"/></g>
<g><path style="opacity:0.53" fill="#4caf4f" d="M 1202.5,138.5 C 1201.5,138.5 1200.5,138.5 1199.5,138.5C 1199.24,137.022 1199.57,135.689 1200.5,134.5C 1202.11,135.32 1202.78,136.654 1202.5,138.5 Z"/></g>
<g><path style="opacity:0.49" fill="#4caf4f" d="M 1222.5,134.5 C 1227.28,134.671 1227.61,135.671 1223.5,137.5C 1221.2,137.124 1220.86,136.124 1222.5,134.5 Z"/></g>
<g><path style="opacity:0.385" fill="#4caf4f" d="M 1228.5,138.5 C 1228.17,138.5 1227.83,138.5 1227.5,138.5C 1227.27,135.188 1228.27,134.355 1230.5,136C 1229.94,137.011 1229.28,137.844 1228.5,138.5 Z"/></g>
<g><path style="opacity:0.39" fill="#4caf4f" d="M 1724.5,134.5 C 1725.98,134.238 1727.31,134.571 1728.5,135.5C 1727.68,137.111 1726.35,137.778 1724.5,137.5C 1724.5,136.5 1724.5,135.5 1724.5,134.5 Z"/></g>
<g><path style="opacity:0.298" fill="#4caf4f" d="M 1772.5,135.5 C 1772.5,136.5 1772.5,137.5 1772.5,138.5C 1771.83,138.5 1771.17,138.5 1770.5,138.5C 1768.93,135.506 1769.6,134.506 1772.5,135.5 Z"/></g>
<g><path style="opacity:0.486" fill="#4caf4f" d="M 190.5,135.5 C 191.978,135.238 193.311,135.571 194.5,136.5C 193.833,136.833 193.167,137.167 192.5,137.5C 191.287,137.253 190.62,136.586 190.5,135.5 Z"/></g>
<g><path style="opacity:0.347" fill="#4caf4f" d="M 1465.5,138.5 C 1464.5,138.5 1463.5,138.5 1462.5,138.5C 1461.22,136.894 1461.72,135.894 1464,135.5C 1466.28,135.894 1466.78,136.894 1465.5,138.5 Z"/></g>
<g><path style="opacity:0.381" fill="#4caf4f" d="M 234.5,138.5 C 232.833,138.5 231.167,138.5 229.5,138.5C 231.167,135.833 232.833,135.833 234.5,138.5 Z"/></g>
<g><path style="opacity:0.439" fill="#4caf4f" d="M 1022.5,138.5 C 1023.83,137.167 1025.17,137.167 1026.5,138.5C 1025.17,138.5 1023.83,138.5 1022.5,138.5 Z"/></g>
</svg>',

'butterfly' =>
'<svg viewBox="0 0 905.000000 1280.000000"> 
<g transform="translate(0.000000,1280.000000) scale(0.100000,-0.100000)">
<path d="M8075 11388 c-65 -46 -138 -126 -168 -183 -27 -52 -30 -70 -31 -140
0 -63 5 -89 22 -122 29 -55 78 -91 135 -99 40 -6 48 -4 73 21 l27 27 -7 140
c-8 168 6 230 63 280 32 28 38 30 95 26 89 -7 163 -46 257 -135 63 -61 88 -93
119 -157 138 -280 109 -600 -71 -784 -53 -55 -137 -102 -180 -102 -23 0 -32
10 -68 79 -48 93 -103 149 -223 230 -121 82 -180 103 -305 109 -170 7 -294
-30 -378 -113 -55 -55 -75 -101 -74 -172 1 -81 18 -122 68 -160 74 -57 133
-44 266 57 102 77 160 104 225 104 72 -1 120 -26 196 -106 129 -134 141 -173
64 -197 -43 -13 -83 -32 -300 -142 -157 -79 -253 -93 -350 -52 -71 29 -83 39
-143 107 -72 81 -180 155 -329 226 -149 70 -230 91 -378 97 -146 6 -255 -10
-407 -58 -251 -80 -385 -201 -429 -384 -13 -58 -15 -93 -8 -205 5 -74 8 -176
9 -226 l0 -91 -47 -56 c-76 -93 -158 -156 -304 -233 -49 -26 -107 -60 -129
-75 -72 -48 -229 -200 -370 -356 -76 -84 -140 -153 -144 -153 -3 0 -12 9 -19
20 -6 11 -39 33 -72 50 -75 38 -94 65 -87 126 9 87 73 151 265 265 204 121
384 274 457 389 69 109 115 232 128 341 l11 96 -30 59 c-40 80 -78 99 -194 99
-156 -1 -300 -67 -300 -138 0 -30 19 -87 30 -87 4 0 39 -29 77 -65 106 -97
117 -165 50 -305 -42 -87 -156 -200 -322 -318 -170 -121 -170 -120 -206 -192
-16 -33 -37 -64 -45 -67 -30 -11 -60 16 -112 99 -42 67 -68 96 -125 139 -114
86 -298 242 -344 292 -119 129 -122 277 -9 371 92 76 116 103 116 130 0 38
-26 73 -78 103 -129 76 -356 -7 -431 -158 -37 -75 -40 -136 -11 -232 55 -177
231 -352 549 -545 115 -69 216 -166 260 -247 50 -93 50 -100 2 -125 -33 -17
-48 -33 -73 -85 -20 -41 -38 -65 -49 -65 -9 0 -28 18 -42 39 -15 22 -98 113
-184 202 -223 228 -381 360 -548 454 -131 75 -161 96 -200 147 -57 73 -63 133
-28 313 21 107 14 244 -15 308 -41 91 -185 218 -315 277 -34 15 -100 40 -147
56 -79 26 -96 28 -255 28 -148 1 -181 -2 -255 -22 -227 -62 -262 -80 -509
-256 -187 -133 -211 -146 -276 -146 -66 0 -108 12 -240 70 -67 30 -148 57
-207 68 -111 23 -144 37 -171 73 -19 26 -19 27 0 70 20 45 105 126 174 166 59
34 136 55 171 48 54 -11 111 -50 146 -98 46 -64 90 -89 170 -94 85 -6 145 22
176 82 28 55 28 146 0 200 -57 110 -236 175 -444 161 -154 -9 -200 -26 -342
-122 -109 -74 -118 -83 -189 -186 -68 -99 -84 -106 -159 -68 -101 52 -173 203
-206 437 -23 166 -21 292 7 361 65 163 176 278 317 328 83 29 123 30 178 3 77
-37 100 -126 80 -309 -10 -85 3 -126 49 -156 35 -23 82 -5 149 57 127 116 122
237 -15 364 -67 62 -178 125 -259 146 -125 32 -293 -27 -407 -144 -148 -149
-234 -424 -203 -646 7 -47 19 -104 28 -126 29 -76 107 -180 224 -302 65 -68
118 -130 118 -138 0 -9 -11 -37 -24 -63 -33 -64 -46 -144 -46 -273 0 -197 63
-370 200 -551 135 -177 140 -192 150 -508 10 -285 19 -369 55 -482 46 -144
117 -266 232 -394 154 -172 443 -318 658 -331 81 -5 102 -2 200 26 123 34 180
39 213 15 30 -21 28 -36 -10 -58 -84 -48 -159 -104 -221 -165 -57 -55 -78 -86
-130 -189 -91 -181 -125 -220 -245 -281 -84 -44 -115 -68 -214 -164 -154 -149
-255 -298 -339 -498 -87 -207 -111 -317 -112 -509 -1 -228 54 -415 175 -593
62 -91 96 -125 169 -169 97 -57 185 -73 287 -52 104 21 166 46 263 105 165
100 224 170 250 296 48 232 -132 527 -349 573 -173 38 -392 -129 -392 -297 0
-96 86 -166 158 -129 17 9 62 56 102 106 39 50 86 100 103 111 106 64 263 -35
286 -181 26 -162 -87 -364 -246 -438 -55 -25 -66 -27 -203 -27 -196 1 -269 25
-373 124 -117 112 -162 283 -137 525 41 407 322 954 574 1118 33 21 62 38 65
38 4 0 3 -51 -2 -112 -12 -159 -3 -267 28 -368 60 -187 195 -323 385 -386 40
-13 84 -18 161 -17 121 1 201 15 278 49 75 34 81 32 81 -21 0 -46 -37 -153
-122 -355 -87 -205 -98 -334 -43 -490 24 -67 26 -84 21 -165 -3 -49 -13 -115
-21 -145 -8 -30 -15 -56 -15 -57 0 -2 18 15 40 37 53 53 81 59 128 26 21 -14
54 -45 74 -69 58 -70 113 -91 238 -91 101 -1 108 1 181 36 134 66 210 151 250
280 24 81 24 123 -2 200 -40 122 -137 191 -208 147 -70 -42 -81 -106 -41 -238
21 -67 21 -71 5 -130 -20 -69 -68 -132 -121 -157 -113 -54 -364 81 -364 194 0
65 338 397 530 520 30 20 117 66 195 103 158 75 172 85 343 232 68 59 191 159
275 224 161 124 245 202 296 277 48 70 124 230 161 339 18 53 36 97 40 97 4 0
17 -11 30 -25 27 -29 42 -31 60 -10 11 13 19 0 50 -84 91 -241 329 -540 545
-687 50 -34 250 -157 445 -274 426 -256 435 -261 580 -387 205 -177 256 -268
200 -358 -59 -96 -272 -160 -344 -104 -53 42 -65 86 -75 294 -9 160 -22 198
-76 221 -63 26 -169 -10 -209 -71 -53 -81 -27 -252 56 -372 56 -80 82 -107
154 -154 75 -48 145 -69 238 -69 136 0 195 29 298 148 50 58 78 57 135 -5 65
-71 93 -154 93 -273 0 -114 -24 -214 -74 -315 -77 -153 -201 -277 -326 -327
-70 -27 -235 -46 -318 -35 -123 15 -233 75 -276 151 -32 56 -38 90 -26 137 22
82 129 159 221 159 82 0 139 47 139 115 0 41 -44 117 -94 163 -43 39 -49 42
-103 42 -68 0 -97 -15 -169 -87 -88 -88 -154 -243 -154 -361 0 -137 157 -305
350 -374 86 -31 245 -36 347 -10 238 60 422 234 492 465 59 195 48 355 -36
539 -54 118 -58 164 -24 262 25 71 27 148 6 307 -19 142 -31 185 -88 319 -70
167 -114 307 -101 328 18 27 111 20 259 -21 154 -43 231 -47 329 -16 92 28
192 121 241 223 71 149 101 317 90 520 l-7 139 48 -7 c234 -32 441 -346 563
-857 49 -205 65 -431 37 -535 -20 -75 -70 -173 -116 -226 -61 -71 -192 -155
-270 -173 -229 -53 -484 47 -544 214 -35 98 -11 237 67 387 38 73 97 95 163
61 43 -23 109 -89 148 -149 55 -86 107 -124 155 -113 28 6 100 73 120 112 65
128 -155 272 -418 273 -118 0 -151 -14 -248 -108 -198 -193 -175 -416 65 -608
124 -100 302 -159 458 -152 86 4 99 7 187 51 112 55 192 133 248 242 49 93 68
164 79 293 42 466 -245 1094 -585 1281 -187 102 -220 134 -447 436 -58 77
-141 174 -186 216 l-80 76 171 6 c682 25 911 178 1091 730 24 73 56 182 70
242 65 266 79 321 87 337 8 14 18 16 53 12 39 -6 44 -4 49 17 8 31 -17 69 -45
69 -12 0 -31 -9 -43 -20 -26 -25 -44 -19 -75 22 -39 53 -32 76 54 173 89 100
153 195 197 292 75 164 84 230 54 373 -30 138 -43 281 -33 345 12 80 38 117
122 180 92 68 142 123 177 199 36 78 49 145 49 262 0 351 -172 633 -426 698
-38 10 -103 16 -170 16 -95 -1 -113 -3 -139 -22z m-1390 -1258 c272 -36 431
-101 672 -276 102 -75 120 -91 110 -101 -16 -16 -535 -135 -597 -137 -29 -1
-37 4 -55 34 -11 19 -44 58 -73 87 -69 68 -138 95 -247 97 -77 1 -83 -1 -122
-31 -77 -61 -81 -124 -14 -226 21 -33 37 -66 34 -73 -2 -6 -61 -31 -131 -54
-70 -23 -174 -66 -232 -97 -95 -50 -105 -53 -116 -38 -36 48 -45 247 -18 368
46 205 190 343 429 412 151 43 235 51 360 35z m-3977 -69 c171 -19 248 -52
352 -148 75 -70 105 -114 147 -214 46 -114 44 -418 -4 -436 -8 -3 -54 15 -103
40 -73 38 -262 113 -357 142 -29 8 -29 11 7 65 82 120 79 184 -11 246 -39 27
-50 29 -128 29 -79 0 -91 -3 -143 -33 -58 -34 -138 -113 -164 -165 -14 -27
-19 -29 -62 -25 -53 5 -223 38 -352 69 -87 21 -220 69 -220 80 0 9 118 105
214 173 127 91 302 164 441 185 71 11 253 7 383 -8z m2713 -410 c22 -22 29
-39 29 -68 0 -57 -27 -153 -55 -193 -37 -53 -137 -172 -141 -167 -2 2 8 37 22
78 22 60 28 98 32 203 4 119 7 132 28 153 31 30 49 29 85 -6z m-1627 -43 c11
-15 16 -45 16 -90 0 -96 25 -191 75 -285 24 -46 39 -83 34 -83 -5 0 -45 35
-88 78 -117 116 -179 261 -152 354 20 67 78 80 115 26z m3824 -75 c26 -35 13
-52 -60 -75 -54 -17 -54 -18 -42 54 4 22 13 39 23 42 26 7 65 -4 79 -21z
m-6012 -21 c6 -4 14 -29 18 -55 7 -44 6 -48 -11 -44 -10 3 -37 11 -60 17 -32
8 -43 16 -43 30 0 42 62 75 96 52z m2872 -1127 c22 -48 6 -109 -31 -121 -31
-10 -50 22 -44 75 7 67 53 94 75 46z m251 16 c19 -12 27 -97 11 -116 -18 -21
-57 -19 -70 4 -15 30 -12 82 7 103 18 20 31 22 52 9z m2416 -404 c-26 -9 -26
-9 -6 -23 20 -15 20 -15 -6 -44 -14 -17 -30 -30 -34 -30 -12 0 -62 88 -56 97
3 5 33 8 67 8 48 0 55 -2 35 -8z m-345 -47 c-4 -23 -10 -43 -13 -46 -3 -3 -30
-8 -61 -11 l-56 -6 0 45 c0 25 3 48 7 51 3 4 34 7 68 7 l62 0 -7 -40z m-4703
-27 c-41 -56 -67 -64 -90 -29 -17 26 -14 55 7 57 6 0 2 4 -9 9 -14 6 3 9 53 9
l74 1 -35 -47z m314 27 l60 0 -3 -42 -3 -43 -55 -3 -55 -3 -17 43 c-22 53 -22
62 -2 54 9 -3 43 -6 75 -6z m4530 -30 c24 -13 25 -30 2 -30 -26 0 -47 16 -39
29 9 14 12 14 37 1z m-4708 -57 c-7 -2 -19 -2 -25 0 -7 3 -2 5 12 5 14 0 19
-2 13 -5z m2380 -1033 c8 -91 13 -166 12 -168 -3 -4 -70 21 -84 32 -22 17 -9
198 19 264 13 29 26 49 30 45 4 -4 14 -82 23 -173z"/>
<path d="M7158 9855 c-25 -55 34 -101 76 -59 23 22 20 40 -10 63 -35 28 -52
26 -66 -4z"/>
<path d="M7002 9828 c-13 -13 -16 -47 -6 -72 7 -19 45 -21 62 -4 27 27 8 88
-28 88 -9 0 -21 -5 -28 -12z"/>
<path d="M6868 9769 c-12 -6 -18 -22 -18 -45 0 -27 5 -37 25 -46 21 -9 28 -8
45 7 24 22 25 43 4 73 -17 24 -30 27 -56 11z"/>
<path d="M6206 9571 c-18 -19 -18 -22 -2 -46 20 -30 39 -32 66 -5 15 15 17 25
11 42 -11 31 -51 35 -75 9z"/>
<path d="M6064 9515 c-19 -29 -11 -60 19 -76 17 -10 24 -8 39 11 23 29 23 45
-2 70 -26 26 -36 25 -56 -5z"/>
<path d="M5953 9443 c-3 -10 -8 -21 -10 -25 -2 -4 6 -17 17 -28 16 -16 24 -18
40 -10 25 13 26 42 2 63 -24 22 -41 21 -49 0z"/>
<path d="M1887 9802 c-26 -28 -21 -61 9 -72 49 -19 87 24 59 66 -19 29 -46 31
-68 6z"/>
<path d="M2050 9760 c-13 -24 12 -80 35 -80 36 0 59 62 33 88 -19 19 -56 14
-68 -8z"/>
<path d="M2190 9701 c-22 -42 -5 -81 35 -81 27 0 45 18 45 46 0 47 -59 73 -80
35z"/>
<path d="M2840 9530 c-18 -34 2 -70 39 -70 39 0 57 23 43 55 -7 14 -18 25 -26
25 -8 0 -21 3 -29 6 -10 3 -20 -3 -27 -16z"/>
<path d="M2990 9471 c-23 -44 1 -88 43 -77 22 5 27 12 27 39 0 54 -47 79 -70
38z"/>
<path d="M3117 9413 c-11 -10 -8 -71 3 -78 17 -11 49 5 56 27 9 30 -39 71 -59
51z"/>
<path d="M8377 11102 c-23 -25 -21 -54 4 -86 25 -32 73 -36 89 -7 37 70 -43
149 -93 93z"/>
<path d="M641 11024 c-26 -34 -27 -68 -1 -94 27 -27 36 -25 71 9 35 35 38 74
7 95 -32 23 -54 20 -77 -10z"/>
<path d="M8234 10966 c-24 -20 -28 -32 -32 -89 -4 -63 -3 -68 26 -97 72 -71
201 -14 204 90 1 34 -12 53 -66 92 -49 35 -95 37 -132 4z m84 -95 c3 -16 -1
-22 -10 -19 -7 3 -15 15 -16 27 -3 16 1 22 10 19 7 -3 15 -15 16 -27z"/>
<path d="M8496 10918 c-19 -27 -21 -69 -4 -86 19 -19 66 -15 78 7 17 32 12 79
-9 91 -30 16 -48 12 -65 -12z"/>
<path d="M745 10897 c-46 -26 -80 -84 -71 -122 10 -39 55 -84 96 -95 59 -16
105 10 125 68 19 57 19 75 -1 113 -14 28 -65 59 -96 59 -7 0 -31 -10 -53 -23z
m68 -97 c-6 -22 -33 -28 -33 -7 0 16 21 38 31 32 4 -3 5 -14 2 -25z"/>
<path d="M532 10848 c-31 -31 2 -108 46 -108 31 0 54 33 46 66 -4 16 -11 35
-17 42 -12 15 -59 16 -75 0z"/>
<path d="M8492 10718 c-17 -17 -15 -74 4 -92 32 -33 84 -8 84 40 0 46 -59 81
-88 52z"/>
<path d="M559 10647 c-42 -32 -41 -84 1 -107 21 -11 42 -3 62 25 14 19 8 55
-12 75 -23 23 -29 24 -51 7z"/>
<path d="M8410 10534 c-15 -37 7 -78 43 -82 37 -4 57 13 57 49 0 21 -7 32 -30
44 -41 21 -59 19 -70 -11z"/>
<path d="M620 10470 c-25 -25 -26 -62 -2 -83 24 -22 45 -21 65 1 29 33 26 75
-8 90 -33 15 -32 15 -55 -8z"/>
<path d="M8623 10140 c-44 -18 -83 -110 -69 -165 9 -37 63 -92 101 -105 46
-15 98 1 125 40 42 59 14 191 -45 218 -37 17 -88 22 -112 12z m69 -107 c39
-35 12 -96 -30 -67 -24 17 -29 53 -10 72 15 15 18 15 40 -5z"/>
<path d="M8007 10102 c-34 -38 -11 -106 32 -95 42 11 68 64 43 86 -29 26 -57
29 -75 9z"/>
<path d="M383 10053 c-65 -33 -93 -145 -53 -211 64 -105 230 -37 230 94 0 48
-29 98 -69 119 -37 19 -67 18 -108 -2z m85 -119 c3 -34 -19 -60 -43 -50 -24 9
-29 34 -10 62 22 34 49 28 53 -12z"/>
<path d="M7831 10051 c-18 -33 -13 -61 14 -73 22 -10 29 -9 47 10 18 18 20 26
12 49 -12 36 -57 44 -73 14z"/>
<path d="M1036 10031 c-32 -36 0 -91 54 -91 27 0 44 43 29 73 -22 42 -55 49
-83 18z"/>
<path d="M1216 9994 c-18 -48 4 -88 46 -82 17 2 24 10 26 34 3 32 -8 47 -44
58 -15 5 -24 2 -28 -10z"/>
<path d="M7656 9984 c-20 -19 -21 -58 -2 -73 24 -20 69 -9 85 21 14 24 14 28
-3 47 -23 25 -58 27 -80 5z"/>
<path d="M1388 9929 c-23 -13 -23 -51 0 -79 24 -30 67 -21 84 16 11 24 9 29
-15 50 -31 26 -43 29 -69 13z"/>
<path d="M5664 9285 c-20 -31 -10 -63 23 -76 49 -18 85 47 47 85 -23 23 -51
20 -70 -9z"/>
<path d="M3382 9250 c-26 -42 -8 -80 38 -80 51 0 69 55 28 84 -30 21 -51 20
-66 -4z"/>
<path d="M5514 9185 c-10 -40 3 -72 32 -79 45 -12 80 35 63 82 -7 18 -16 22
-48 22 -35 0 -42 -4 -47 -25z"/>
<path d="M3513 9154 c-3 -8 -1 -29 4 -47 7 -28 14 -32 46 -35 33 -3 38 0 48
25 17 46 -2 73 -51 73 -28 0 -43 -5 -47 -16z"/>
<path d="M5355 9070 c-18 -34 -9 -62 25 -80 16 -9 25 -6 45 14 29 29 32 58 9
80 -26 26 -61 20 -79 -14z"/>
<path d="M3698 9059 c-41 -23 -11 -109 38 -109 49 0 61 65 18 99 -30 24 -32
24 -56 10z"/>
<path d="M5240 8964 c-20 -23 -16 -75 5 -90 24 -18 30 -17 49 2 22 21 20 67
-2 87 -23 21 -36 21 -52 1z"/>
<path d="M3824 8925 c-24 -36 -1 -95 37 -95 38 0 53 74 21 103 -25 23 -39 21
-58 -8z"/>
<path d="M5086 8831 c-25 -28 -17 -64 17 -75 34 -12 61 1 65 33 4 26 -24 61
-48 61 -9 0 -25 -9 -34 -19z"/>
<path d="M3970 8788 c-11 -18 -20 -36 -20 -40 0 -12 37 -28 63 -28 56 0 63 65
9 87 -31 13 -33 12 -52 -19z"/>
<path d="M785 8780 c-42 -47 25 -112 79 -76 28 18 35 56 13 75 -28 23 -71 24
-92 1z"/>
<path d="M4971 8724 c-23 -29 -24 -34 -15 -58 10 -26 41 -38 64 -26 28 15 28
85 1 100 -27 14 -27 14 -50 -16z"/>
<path d="M4095 8696 c-3 -9 -7 -23 -10 -32 -9 -30 14 -55 47 -52 43 4 58 30
38 66 -18 31 -67 43 -75 18z"/>
<path d="M8260 8689 c-25 -46 23 -108 74 -95 33 8 41 38 20 81 -14 30 -21 35
-50 35 -24 0 -37 -6 -44 -21z"/>
<path d="M787 8623 c-39 -44 -25 -103 24 -103 49 0 81 87 40 110 -28 14 -46
12 -64 -7z"/>
<path d="M4882 8598 c-17 -17 -15 -63 3 -78 12 -10 21 -10 41 -1 21 9 25 17
22 49 -3 31 -7 37 -28 40 -14 2 -31 -3 -38 -10z"/>
<path d="M4187 8573 c-4 -3 -7 -21 -7 -39 0 -40 28 -61 60 -44 23 12 27 59 8
78 -13 13 -50 16 -61 5z"/>
<path d="M8252 8482 c-16 -48 15 -92 66 -92 24 0 38 16 47 52 9 36 -23 68 -69
68 -28 0 -36 -5 -44 -28z"/>
<path d="M788 8426 c-21 -16 -24 -65 -5 -90 19 -26 65 -17 91 17 19 26 19 30
5 55 -17 31 -63 40 -91 18z"/>
<path d="M8238 8279 c-26 -15 -23 -64 4 -83 30 -21 61 -20 82 3 15 17 16 25 6
55 -10 31 -16 36 -43 36 -18 0 -40 -5 -49 -11z"/>
<path d="M814 8205 c-36 -55 12 -114 72 -86 19 8 24 18 24 46 0 32 -4 38 -35
51 -44 18 -42 18 -61 -11z"/>
<path d="M8176 8064 c-18 -18 -20 -42 -7 -77 13 -33 45 -40 76 -15 45 35 25
108 -30 108 -13 0 -31 -7 -39 -16z"/>
<path d="M880 7991 c-24 -45 -6 -91 41 -106 27 -9 47 10 55 52 5 24 1 36 -15
52 -26 26 -67 27 -81 2z"/>
<path d="M8072 7884 c-26 -18 -30 -75 -6 -94 19 -16 71 -17 86 -2 20 20 13 73
-11 94 -28 22 -40 23 -69 2z"/>
<path d="M1007 7819 c-54 -31 -35 -109 26 -109 42 0 57 15 57 56 0 23 -7 38
-22 48 -26 19 -36 19 -61 5z"/>
<path d="M7916 7627 c-25 -19 -18 -85 11 -104 20 -14 24 -13 48 11 55 55 3
138 -59 93z"/>
<path d="M1152 7558 c-28 -28 3 -108 43 -108 49 0 71 53 39 98 -17 24 -62 30
-82 10z"/>
<path d="M7760 7510 c-11 -11 -20 -30 -20 -43 0 -32 27 -67 52 -67 25 0 78 31
78 45 0 19 -31 55 -60 70 -28 14 -31 14 -50 -5z"/>
<path d="M1301 7423 l-31 -37 22 -23 c26 -28 92 -32 102 -7 18 48 -7 104 -47
104 -8 0 -29 -17 -46 -37z"/>
<path d="M7537 7392 c-22 -25 -21 -44 4 -76 26 -33 63 -34 84 -2 19 28 10 55
-27 80 -29 21 -41 20 -61 -2z"/>
<path d="M7064 7369 c-25 -28 -8 -54 37 -54 40 0 51 26 23 56 -22 24 -36 24
-60 -2z"/>
<path d="M7212 7378 c-32 -32 1 -98 50 -98 39 0 61 50 36 82 -17 22 -70 32
-86 16z"/>
<path d="M7349 7356 c-16 -35 10 -76 49 -76 73 0 68 80 -5 95 -28 6 -34 3 -44
-19z"/>
<path d="M1540 7330 c-22 -22 -27 -69 -8 -88 7 -7 22 -12 34 -12 47 0 74 88
35 110 -28 14 -38 13 -61 -10z"/>
<path d="M1880 7323 c-38 -14 -51 -60 -24 -87 21 -22 50 -20 74 4 45 45 9 104
-50 83z"/>
<path d="M2019 7303 c-5 -14 -7 -33 -4 -40 7 -21 55 -12 72 13 12 19 12 24 -2
38 -25 25 -54 20 -66 -11z"/>
<path d="M1722 7304 c-12 -8 -22 -26 -22 -39 0 -25 24 -55 44 -55 21 0 56 40
56 63 0 45 -37 60 -78 31z"/>
<path d="M7236 7208 c-20 -11 -21 -56 -1 -81 17 -23 51 -15 60 14 16 49 -20
89 -59 67z"/>
<path d="M7624 7171 c-93 -58 -78 -248 22 -281 39 -13 102 -7 135 13 59 35 78
157 34 217 -31 42 -56 56 -115 64 -35 5 -52 3 -76 -13z m111 -125 c18 -13 18
-15 2 -40 -22 -32 -32 -33 -53 -1 -29 45 6 74 51 41z"/>
<path d="M1856 7141 c-26 -29 -9 -75 29 -79 32 -4 45 10 45 51 0 20 -7 30 -26
37 -32 13 -28 13 -48 -9z"/>
<path d="M1358 7082 c-35 -33 -41 -46 -45 -89 -11 -106 32 -165 125 -170 73
-5 101 10 129 67 60 121 -1 230 -127 230 -33 0 -48 -7 -82 -38z m120 -105 c5
-24 -25 -68 -41 -61 -7 2 -19 14 -26 26 -12 19 -11 24 9 40 27 22 53 20 58 -5z"/>
<path d="M7322 7058 c-29 -29 1 -88 45 -88 29 0 45 23 37 54 -10 40 -57 59
-82 34z"/>
<path d="M1762 6998 c-17 -17 -15 -77 3 -84 26 -10 61 4 75 31 12 22 11 27 -5
45 -20 22 -55 26 -73 8z"/>
<path d="M7374 6905 c-5 -18 -1 -32 10 -43 23 -23 51 -4 51 34 0 24 -5 30 -27
32 -22 3 -29 -2 -34 -23z"/>
<path d="M1717 6863 c-12 -11 -7 -50 7 -62 19 -16 56 2 56 27 0 32 -43 56 -63
35z"/>
<path d="M7451 6576 c-11 -19 -11 -28 -1 -47 22 -42 48 -50 74 -23 23 22 20
60 -6 78 -32 22 -53 20 -67 -8z"/>
<path d="M1625 6507 c-18 -27 -12 -53 17 -68 27 -15 57 -2 64 28 14 55 -50 87
-81 40z"/>
<path d="M4590 6491 c-22 -41 1 -81 45 -81 39 0 51 34 25 70 -24 34 -55 39
-70 11z"/>
<path d="M4500 6460 c-24 -24 -25 -43 -4 -64 29 -30 74 -10 74 32 0 51 -35 67
-70 32z"/>
<path d="M7616 6434 c-30 -30 -12 -90 29 -94 48 -5 55 -1 55 33 0 35 -30 77
-55 77 -7 0 -21 -7 -29 -16z"/>
<path d="M1478 6365 c-33 -44 -20 -95 23 -95 59 0 78 84 24 108 -32 15 -27 16
-47 -13z"/>
<path d="M4667 6323 c-4 -3 -7 -23 -7 -44 0 -30 6 -42 26 -56 33 -21 58 -8 62
35 5 45 -54 92 -81 65z"/>
<path d="M4430 6283 c-13 -17 -20 -38 -18 -57 2 -26 8 -32 31 -34 37 -4 57 13
57 52 0 59 -38 80 -70 39z"/>
<path d="M7706 6251 c-30 -33 -1 -101 42 -101 26 0 62 29 62 49 0 51 -73 87
-104 52z"/>
<path d="M1372 6184 c-44 -30 -18 -94 38 -94 49 0 78 61 44 94 -20 20 -53 20
-82 0z"/>
<path d="M4790 6104 c-31 -36 38 -115 74 -85 25 21 20 66 -9 85 -31 20 -49 20
-65 0z"/>
<path d="M7790 6080 c-8 -14 -7 -29 5 -55 13 -34 17 -36 52 -33 31 2 39 8 47
30 8 23 6 32 -14 53 -29 31 -75 34 -90 5z"/>
<path d="M4300 6070 c-22 -22 -25 -42 -10 -71 14 -25 31 -24 64 2 22 17 26 27
21 48 -11 44 -43 53 -75 21z"/>
<path d="M1285 6015 c-30 -33 -21 -76 19 -90 25 -8 31 -6 53 19 55 64 -14 132
-72 71z"/>
<path d="M4975 5909 c-10 -16 4 -75 21 -89 22 -19 64 3 64 32 0 29 -9 42 -37
57 -27 14 -40 14 -48 0z"/>
<path d="M4125 5864 c-29 -31 -31 -47 -9 -68 20 -21 37 -20 56 2 22 25 26 44
17 70 -11 29 -34 28 -64 -4z"/>
<path d="M7892 5868 c-6 -6 -12 -27 -12 -44 0 -59 54 -85 84 -42 19 27 20 49
4 71 -18 22 -60 31 -76 15z"/>
<path d="M6562 5793 c3 -41 7 -50 31 -61 35 -17 57 -2 57 37 0 32 -8 44 -37
60 -44 23 -54 15 -51 -36z"/>
<path d="M1207 5792 c-34 -37 -14 -92 34 -92 22 0 32 6 41 25 27 59 -34 113
-75 67z"/>
<path d="M5104 5774 c-13 -51 44 -103 85 -77 39 25 9 83 -51 97 -24 5 -28 3
-34 -20z"/>
<path d="M2560 5781 c-14 -4 -31 -19 -39 -33 -13 -23 -13 -29 1 -51 13 -19 23
-23 43 -20 31 7 47 34 43 75 -3 35 -11 39 -48 29z"/>
<path d="M3982 5737 c-40 -43 -19 -96 33 -83 29 7 62 65 52 90 -10 24 -60 20
-85 -7z"/>
<path d="M5251 5652 c-7 -12 -6 -26 2 -43 10 -19 20 -24 52 -24 36 0 40 3 43
26 2 15 -4 34 -13 43 -22 22 -72 21 -84 -2z"/>
<path d="M6655 5644 c-24 -25 -15 -72 18 -92 25 -15 30 -15 55 -2 33 19 41 59
15 82 -29 25 -69 30 -88 12z"/>
<path d="M7886 5641 c-20 -23 -20 -60 0 -89 23 -34 75 -27 88 11 20 57 -51
119 -88 78z"/>
<path d="M3838 5628 c-24 -20 -28 -44 -12 -67 23 -32 61 -27 84 11 18 30 18
34 3 50 -19 21 -53 23 -75 6z"/>
<path d="M2427 5582 c-46 -51 20 -120 75 -79 26 20 32 61 13 81 -22 21 -68 20
-88 -2z"/>
<path d="M1217 5577 c-19 -13 -32 -75 -20 -94 13 -19 56 -25 75 -11 9 7 19 27
23 45 5 25 2 36 -14 52 -24 24 -40 26 -64 8z"/>
<path d="M5468 5534 c-22 -22 -5 -72 27 -80 51 -13 85 35 48 68 -29 25 -57 30
-75 12z"/>
<path d="M7277 5533 c-4 -3 -7 -21 -7 -40 0 -45 36 -64 75 -39 29 19 32 46 7
69 -19 17 -63 23 -75 10z"/>
<path d="M3629 5489 c-27 -27 -21 -61 13 -73 33 -12 68 14 68 49 0 46 -45 60
-81 24z"/>
<path d="M1823 5474 c-20 -9 -16 -72 5 -83 43 -23 91 27 65 68 -10 16 -48 24
-70 15z"/>
<path d="M5673 5443 c-19 -7 -16 -58 5 -76 22 -20 63 -22 80 -5 24 24 14 58
-23 74 -36 15 -42 16 -62 7z"/>
<path d="M6710 5414 c-14 -35 -13 -46 6 -72 17 -25 71 -31 92 -10 17 17 15 59
-4 86 -23 33 -81 31 -94 -4z"/>
<path d="M7176 5424 c-17 -45 1 -84 39 -84 43 0 61 53 29 84 -20 21 -60 21
-68 0z"/>
<path d="M3428 5393 c-46 -29 -32 -83 22 -83 40 1 62 23 58 62 -2 27 -8 33
-28 35 -14 2 -37 -5 -52 -14z"/>
<path d="M7852 5387 c-29 -30 -28 -53 3 -73 23 -15 27 -15 50 0 14 9 28 28 31
41 13 53 -45 74 -84 32z"/>
<path d="M2366 5366 c-20 -31 -21 -70 -2 -85 8 -7 32 -11 53 -9 31 2 39 8 47
31 24 66 -60 120 -98 63z"/>
<path d="M1922 5368 c-7 -7 -12 -25 -12 -40 0 -34 41 -59 68 -41 21 12 28 59
11 79 -14 17 -51 18 -67 2z"/>
<path d="M5860 5351 c-12 -24 1 -66 25 -75 53 -20 97 12 75 54 -22 41 -82 54
-100 21z"/>
<path d="M1234 5315 c-15 -62 72 -107 98 -50 17 37 -12 75 -57 75 -29 0 -36
-4 -41 -25z"/>
<path d="M3238 5319 c-32 -18 -42 -48 -22 -70 22 -25 59 -24 84 1 25 25 25 42
2 63 -21 19 -39 21 -64 6z"/>
<path d="M7191 5257 c-28 -28 5 -97 47 -97 41 0 57 54 25 82 -24 21 -59 28
-72 15z"/>
<path d="M6030 5235 c-26 -31 21 -92 65 -81 19 5 25 13 25 35 0 49 -61 81 -90
46z"/>
<path d="M1918 5189 c-42 -31 -33 -89 13 -89 22 0 56 30 63 55 12 44 -35 65
-76 34z"/>
<path d="M3071 5189 c-30 -28 -26 -73 7 -77 55 -8 98 63 53 88 -27 14 -34 13
-60 -11z"/>
<path d="M6744 5196 c-3 -8 -4 -29 -2 -47 4 -41 48 -70 74 -49 22 19 18 76 -8
94 -28 20 -57 20 -64 2z"/>
<path d="M7716 5168 c-9 -12 -16 -29 -16 -38 0 -27 30 -60 54 -60 27 0 41 14
50 52 5 21 1 31 -20 47 -34 27 -49 26 -68 -1z"/>
<path d="M2362 5144 c-18 -12 -22 -24 -20 -51 4 -47 27 -62 62 -40 19 13 26
26 26 50 0 54 -27 70 -68 41z"/>
<path d="M7320 5125 c-14 -17 -7 -57 16 -79 22 -23 71 -20 84 3 15 29 13 40
-16 66 -30 28 -66 32 -84 10z"/>
<path d="M7497 5115 c-24 -23 -7 -78 26 -90 37 -12 61 3 65 43 3 26 -2 33 -27
46 -37 20 -46 20 -64 1z"/>
<path d="M1382 5108 c-15 -15 -16 -63 -2 -89 13 -23 53 -25 74 -3 18 18 21 56
6 85 -12 22 -59 26 -78 7z"/>
<path d="M6170 5095 c-16 -19 -5 -68 21 -91 16 -14 24 -15 44 -6 19 9 25 19
25 45 0 55 -59 89 -90 52z"/>
<path d="M1776 5059 c-21 -16 -26 -28 -24 -53 2 -25 8 -32 30 -34 39 -5 78 24
78 57 0 52 -39 66 -84 30z"/>
<path d="M2931 5044 c-38 -48 -22 -94 31 -94 32 0 48 24 48 70 0 24 -6 33 -26
40 -33 13 -29 14 -53 -16z"/>
<path d="M1596 5044 c-31 -30 -9 -84 35 -84 24 0 59 34 59 58 0 41 -63 58 -94
26z"/>
<path d="M6272 4986 c-35 -42 -9 -90 45 -84 21 2 29 9 31 26 6 40 -54 86 -76
58z"/>
<path d="M6732 4988 c-7 -7 -12 -26 -12 -43 0 -37 16 -55 47 -55 25 0 43 26
43 62 0 30 -17 48 -45 48 -12 0 -26 -5 -33 -12z"/>
<path d="M2371 4919 c-15 -44 7 -79 49 -79 25 0 30 4 36 36 7 40 -5 59 -47 69
-23 6 -28 2 -38 -26z"/>
<path d="M2848 4933 c-26 -31 -29 -42 -18 -63 20 -36 90 -16 90 26 0 48 -43
70 -72 37z"/>
<path d="M7060 4833 c-33 -23 -38 -31 -50 -75 -9 -33 -7 -45 15 -93 31 -64 62
-85 130 -85 56 0 88 28 105 91 10 37 9 50 -7 88 -21 52 -59 79 -124 86 -32 4
-51 0 -69 -12z m109 -107 c24 -29 -14 -65 -50 -46 -19 10 -26 40 -12 53 11 12
50 8 62 -7z"/>
<path d="M6742 4788 c-28 -28 -7 -93 34 -108 71 -27 92 70 23 105 -34 18 -42
18 -57 3z"/>
<path d="M1994 4776 c-69 -30 -98 -106 -69 -181 21 -56 47 -75 101 -75 84 0
144 65 144 155 0 40 -5 52 -34 81 -38 38 -85 45 -142 20z m74 -102 c23 -41
-22 -78 -53 -44 -24 27 -9 51 37 59 4 0 11 -6 16 -15z"/>
<path d="M2355 4715 c-14 -13 -25 -31 -25 -40 0 -18 35 -55 53 -55 42 0 82 74
57 105 -19 23 -58 18 -85 -10z"/>
<path d="M6826 4548 c-16 -25 -11 -71 8 -87 6 -5 28 -12 49 -16 l37 -7 0 46
c0 74 -61 115 -94 64z"/>
<path d="M2275 4494 c-11 -11 -15 -29 -12 -58 3 -36 7 -41 30 -44 38 -5 63 13
71 49 11 57 -49 92 -89 53z"/>
<path d="M6384 4482 c-53 -33 -74 -69 -74 -124 0 -60 18 -104 59 -143 27 -25
39 -29 85 -29 86 2 136 54 136 142 -1 119 -120 207 -206 154z m86 -117 c20
-10 30 -23 30 -38 0 -49 -48 -62 -87 -25 -27 26 -30 53 -5 67 23 14 28 13 62
-4z"/>
<path d="M3266 4460 c-87 -27 -160 -174 -122 -248 20 -39 35 -47 108 -61 120
-22 242 -107 268 -187 18 -55 5 -115 -33 -151 -63 -61 -146 -91 -314 -114
-224 -31 -436 75 -575 287 -85 129 -117 226 -127 374 -7 104 -7 105 -20 65
-22 -67 -25 -204 -7 -287 44 -195 222 -391 444 -489 284 -124 542 -72 685 138
82 119 113 250 82 345 -42 131 -177 290 -273 322 -50 18 -73 19 -116 6z"/>
<path d="M2663 4421 c-33 -24 -45 -41 -58 -83 -21 -73 -19 -99 14 -146 71 -98
194 -72 245 52 21 51 20 79 -4 126 -25 49 -53 68 -111 76 -38 5 -49 2 -86 -25z
m131 -113 c8 -14 5 -23 -19 -48 -16 -16 -39 -30 -51 -30 -56 0 -57 73 -1 89
42 12 57 10 71 -11z"/>
<path d="M6840 4300 c-26 -26 -25 -42 6 -79 31 -37 44 -38 72 -8 24 25 29 76
10 95 -19 19 -66 14 -88 -8z"/>
<path d="M2263 4259 c-7 -7 -13 -32 -13 -56 0 -48 22 -68 65 -57 42 10 62 77
32 103 -22 18 -71 24 -84 10z"/>
<path d="M6818 4119 c-35 -20 -12 -89 28 -89 52 0 73 56 32 84 -26 18 -36 19
-60 5z"/>
<path d="M6273 4095 c-50 -35 -47 -136 4 -156 57 -22 107 0 126 54 8 21 -13
76 -35 94 -26 22 -71 25 -95 8z"/>
<path d="M2295 4046 c-18 -27 -11 -52 17 -67 41 -22 77 15 64 67 -9 33 -59 34
-81 0z"/>
<path d="M2811 4031 c-58 -54 -33 -132 46 -147 61 -12 101 33 89 99 -14 76
-80 100 -135 48z"/>
<path d="M6704 3844 c-23 -34 -9 -75 28 -88 56 -20 95 64 46 98 -32 23 -55 20
-74 -10z"/>
<path d="M2412 3804 c-44 -31 -21 -103 34 -104 24 0 38 14 48 47 5 17 1 29
-18 48 -29 29 -34 30 -64 9z"/>
<path d="M5536 3805 c-6 -17 7 -60 22 -69 16 -10 67 10 70 28 8 41 -77 79 -92
41z"/>
<path d="M3588 3773 c-19 -5 -34 -45 -23 -62 3 -6 22 -11 41 -11 27 0 37 5 46
25 19 41 -10 62 -64 48z"/>
<path d="M5663 3683 c-3 -9 0 -28 8 -42 12 -20 23 -26 49 -26 29 0 36 4 38 23
2 13 -4 32 -14 43 -21 24 -72 25 -81 2z"/>
<path d="M6510 3670 c-36 -36 -17 -100 31 -100 25 0 59 34 59 59 0 19 -43 61
-61 61 -5 0 -18 -9 -29 -20z"/>
<path d="M3448 3646 c-22 -16 -23 -59 -3 -76 21 -17 61 1 77 35 20 45 -31 74
-74 41z"/>
<path d="M2610 3620 c-43 -43 -4 -111 55 -96 22 6 25 12 25 51 0 40 -3 47 -26
55 -33 13 -31 13 -54 -10z"/>
<path d="M5857 3603 c-24 -23 17 -93 54 -93 39 0 52 55 21 83 -19 17 -63 23
-75 10z"/>
<path d="M6030 3579 c-25 -46 12 -94 68 -87 23 3 27 8 30 40 3 32 -1 39 -27
52 -40 21 -58 20 -71 -5z"/>
<path d="M6250 3579 c-25 -46 12 -88 67 -77 26 5 31 67 7 85 -27 20 -62 16
-74 -8z"/>
<path d="M3266 3557 c-27 -20 -18 -81 12 -85 31 -5 62 25 62 58 0 39 -39 53
-74 27z"/>
<path d="M2872 3544 c-29 -20 -31 -79 -2 -94 21 -11 57 1 72 24 13 19 9 54 -8
70 -19 20 -34 20 -62 0z"/>
<path d="M3082 3532 c-17 -13 -23 -26 -20 -47 5 -42 29 -53 72 -32 36 17 45
41 30 81 -8 21 -53 20 -82 -2z"/>
</g>
</svg>
'

	);

	/**
	 * Social Icons – domain mappings.
	 *
	 * By default, each Icon ID is matched against a .com TLD. To override this behavior,
	 * specify all the domains it covers (including the .com TLD too, if applicable).
	 *
	 * @var array
	 */
	static $social_icons_map = array(
		'amazon'      => array(
			'amazon.com',
			'amazon.cn',
			'amazon.in',
			'amazon.fr',
			'amazon.de',
			'amazon.it',
			'amazon.nl',
			'amazon.es',
			'amazon.co',
			'amazon.ca',
		),
		'apple'       => array(
			'apple.com',
			'itunes.com',
		),
		'behance'     => array(
			'behance.net',
		),
		'codepen'     => array(
			'codepen.io',
		),
		'facebook'    => array(
			'facebook.com',
			'fb.me',
		),
		'tiktok'    => array(
			'tiktok.com',
		),
		'feed'        => array(
			'feed',
		),
		'google-plus' => array(
			'plus.google.com',
		),
		'lastfm'      => array(
			'last.fm',
		),
		'mail'        => array(
			'mailto:',
		),
		'slideshare'  => array(
			'slideshare.net',
		),
		'pocket'      => array(
			'getpocket.com',
		),
		'twitch'      => array(
			'twitch.tv',
		),
		'wordpress'   => array(
			'wordpress.com',
			'wordpress.org',
		),
	);

	/**
	 * Social Icons – svg sources.
	 *
	 * @var array
	 */
	static $social_icons = array(
		'500px'       => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M6.94026,15.1412c.00437.01213.108.29862.168.44064a6.55008,6.55008,0,1,0,6.03191-9.09557,6.68654,6.68654,0,0,0-2.58357.51467A8.53914,8.53914,0,0,0,8.21268,8.61344L8.209,8.61725V3.22948l9.0504-.00008c.32934-.0036.32934-.46353.32934-.61466s0-.61091-.33035-.61467L7.47248,2a.43.43,0,0,0-.43131.42692v7.58355c0,.24466.30476.42131.58793.4819.553.11812.68074-.05864.81617-.2457l.018-.02481A10.52673,10.52673,0,0,1,9.32258,9.258a5.35268,5.35268,0,1,1,7.58985,7.54976,5.417,5.417,0,0,1-3.80867,1.56365,5.17483,5.17483,0,0,1-2.69822-.74478l.00342-4.61111a2.79372,2.79372,0,0,1,.71372-1.78792,2.61611,2.61611,0,0,1,1.98282-.89477,2.75683,2.75683,0,0,1,1.95525.79477,2.66867,2.66867,0,0,1,.79656,1.909,2.724,2.724,0,0,1-2.75849,2.748,4.94651,4.94651,0,0,1-.86254-.13719c-.31234-.093-.44519.34058-.48892.48349-.16811.54966.08453.65862.13687.67489a3.75751,3.75751,0,0,0,1.25234.18375,3.94634,3.94634,0,1,0-2.82444-6.742,3.67478,3.67478,0,0,0-1.13028,2.584l-.00041.02323c-.0035.11667-.00579,2.881-.00644,3.78811l-.00407-.00451a6.18521,6.18521,0,0,1-1.0851-1.86092c-.10544-.27856-.34358-.22925-.66857-.12917-.14192.04372-.57386.17677-.47833.489Zm4.65165-1.08338a.51346.51346,0,0,0,.19513.31818l.02276.022a.52945.52945,0,0,0,.3517.18416.24242.24242,0,0,0,.16577-.0611c.05473-.05082.67382-.67812.73287-.738l.69041.68819a.28978.28978,0,0,0,.21437.11032.53239.53239,0,0,0,.35708-.19486c.29792-.30419.14885-.46821.07676-.54751l-.69954-.69975.72952-.73469c.16-.17311.01874-.35708-.12218-.498-.20461-.20461-.402-.25742-.52855-.14083l-.7254.72665-.73354-.73375a.20128.20128,0,0,0-.14179-.05695.54135.54135,0,0,0-.34379.19648c-.22561.22555-.274.38149-.15656.5059l.73374.7315-.72942.73072A.26589.26589,0,0,0,11.59191,14.05782Zm1.59866-9.915A8.86081,8.86081,0,0,0,9.854,4.776a.26169.26169,0,0,0-.16938.22759.92978.92978,0,0,0,.08619.42094c.05682.14524.20779.531.50006.41955a8.40969,8.40969,0,0,1,2.91968-.55484,7.87875,7.87875,0,0,1,3.086.62286,8.61817,8.61817,0,0,1,2.30562,1.49315.2781.2781,0,0,0,.18318.07586c.15529,0,.30425-.15253.43167-.29551.21268-.23861.35873-.4369.1492-.63538a8.50425,8.50425,0,0,0-2.62312-1.694A9.0177,9.0177,0,0,0,13.19058,4.14283ZM19.50945,18.6236h0a.93171.93171,0,0,0-.36642-.25406.26589.26589,0,0,0-.27613.06613l-.06943.06929A7.90606,7.90606,0,0,1,7.60639,18.505a7.57284,7.57284,0,0,1-1.696-2.51537,8.58715,8.58715,0,0,1-.5147-1.77754l-.00871-.04864c-.04939-.25873-.28755-.27684-.62981-.22448-.14234.02178-.5755.088-.53426.39969l.001.00712a9.08807,9.08807,0,0,0,15.406,4.99094c.00193-.00192.04753-.04718.0725-.07436C19.79425,19.16234,19.87422,18.98728,19.50945,18.6236Z"></path>
</svg>',

		'amazon'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M13.582,8.182C11.934,8.367,9.78,8.49,8.238,9.166c-1.781,0.769-3.03,2.337-3.03,4.644 c0,2.953,1.86,4.429,4.253,4.429c2.02,0,3.125-0.477,4.685-2.065c0.516,0.747,0.685,1.109,1.629,1.894 c0.212,0.114,0.483,0.103,0.672-0.066l0.006,0.006c0.567-0.505,1.599-1.401,2.18-1.888c0.231-0.188,0.19-0.496,0.009-0.754 c-0.52-0.718-1.072-1.303-1.072-2.634V8.305c0-1.876,0.133-3.599-1.249-4.891C15.23,2.369,13.422,2,12.04,2 C9.336,2,6.318,3.01,5.686,6.351C5.618,6.706,5.877,6.893,6.109,6.945l2.754,0.298C9.121,7.23,9.308,6.977,9.357,6.72 c0.236-1.151,1.2-1.706,2.284-1.706c0.584,0,1.249,0.215,1.595,0.738c0.398,0.584,0.346,1.384,0.346,2.061V8.182z M13.049,14.088 c-0.451,0.8-1.169,1.291-1.967,1.291c-1.09,0-1.728-0.83-1.728-2.061c0-2.42,2.171-2.86,4.227-2.86v0.615 C13.582,12.181,13.608,13.104,13.049,14.088z M20.683,19.339C18.329,21.076,14.917,22,11.979,22c-4.118,0-7.826-1.522-10.632-4.057 c-0.22-0.199-0.024-0.471,0.241-0.317c3.027,1.762,6.771,2.823,10.639,2.823c2.608,0,5.476-0.541,8.115-1.66 C20.739,18.62,21.072,19.051,20.683,19.339z M21.336,21.043c-0.194,0.163-0.379,0.076-0.293-0.139 c0.284-0.71,0.92-2.298,0.619-2.684c-0.301-0.386-1.99-0.183-2.749-0.092c-0.23,0.027-0.266-0.173-0.059-0.319 c1.348-0.946,3.555-0.673,3.811-0.356C22.925,17.773,22.599,19.986,21.336,21.043z"></path>
</svg>',

		'apple'       => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M20.07,17.586a10.874,10.874,0,0,1-1.075,1.933,9.822,9.822,0,0,1-1.385,1.674,2.687,2.687,0,0,1-1.78.784,4.462,4.462,0,0,1-1.644-.393,4.718,4.718,0,0,0-1.77-.391,4.878,4.878,0,0,0-1.82.391A4.9,4.9,0,0,1,9.021,22a2.53,2.53,0,0,1-1.82-.8A10.314,10.314,0,0,1,5.752,19.46,11.987,11.987,0,0,1,4.22,16.417a11.143,11.143,0,0,1-.643-3.627,6.623,6.623,0,0,1,.87-3.465A5.1,5.1,0,0,1,6.268,7.483a4.9,4.9,0,0,1,2.463-.695,5.8,5.8,0,0,1,1.9.443,6.123,6.123,0,0,0,1.511.444,9.04,9.04,0,0,0,1.675-.523,5.537,5.537,0,0,1,2.277-.4,4.835,4.835,0,0,1,3.788,1.994,4.213,4.213,0,0,0-2.235,3.827,4.222,4.222,0,0,0,1.386,3.181,4.556,4.556,0,0,0,1.385.909q-.167.483-.353.927ZM16.211,2.4a4.267,4.267,0,0,1-1.094,2.8,3.726,3.726,0,0,1-3.1,1.528A3.114,3.114,0,0,1,12,6.347a4.384,4.384,0,0,1,1.16-2.828,4.467,4.467,0,0,1,1.414-1.061A4.215,4.215,0,0,1,16.19,2a3.633,3.633,0,0,1,.021.4Z"></path>
</svg>',

		'bandcamp'    => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M15.27 17.289 3 17.289 8.73 6.711 21 6.711 15.27 17.289"></path>
</svg>',

		'behance'     => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M7.799,5.698c0.589,0,1.12,0.051,1.606,0.156c0.482,0.102,0.894,0.273,1.241,0.507c0.344,0.235,0.612,0.546,0.804,0.938 c0.188,0.387,0.281,0.871,0.281,1.443c0,0.619-0.141,1.137-0.421,1.551c-0.284,0.413-0.7,0.751-1.255,1.014 c0.756,0.218,1.317,0.601,1.689,1.146c0.374,0.549,0.557,1.205,0.557,1.975c0,0.623-0.12,1.161-0.359,1.612 c-0.241,0.457-0.569,0.828-0.973,1.114c-0.408,0.288-0.876,0.5-1.399,0.637C9.052,17.931,8.514,18,7.963,18H2V5.698H7.799 M7.449,10.668c0.481,0,0.878-0.114,1.192-0.345c0.311-0.228,0.463-0.603,0.463-1.119c0-0.286-0.051-0.523-0.152-0.707 C8.848,8.315,8.711,8.171,8.536,8.07C8.362,7.966,8.166,7.894,7.94,7.854c-0.224-0.044-0.457-0.06-0.697-0.06H4.709v2.874H7.449z M7.6,15.905c0.267,0,0.521-0.024,0.759-0.077c0.243-0.053,0.457-0.137,0.637-0.261c0.182-0.12,0.332-0.283,0.441-0.491 C9.547,14.87,9.6,14.602,9.6,14.278c0-0.633-0.18-1.084-0.533-1.357c-0.356-0.27-0.83-0.404-1.413-0.404H4.709v3.388L7.6,15.905z M16.162,15.864c0.367,0.358,0.897,0.538,1.583,0.538c0.493,0,0.92-0.125,1.277-0.374c0.354-0.248,0.571-0.514,0.654-0.79h2.155 c-0.347,1.072-0.872,1.838-1.589,2.299C19.534,18,18.67,18.23,17.662,18.23c-0.701,0-1.332-0.113-1.899-0.337 c-0.567-0.227-1.041-0.544-1.439-0.958c-0.389-0.415-0.689-0.907-0.904-1.484c-0.213-0.574-0.32-1.21-0.32-1.899 c0-0.666,0.11-1.288,0.329-1.863c0.222-0.577,0.529-1.075,0.933-1.492c0.406-0.42,0.885-0.751,1.444-0.994 c0.558-0.241,1.175-0.363,1.857-0.363c0.754,0,1.414,0.145,1.98,0.44c0.563,0.291,1.026,0.686,1.389,1.181 c0.363,0.493,0.622,1.057,0.783,1.69c0.16,0.632,0.217,1.292,0.171,1.983h-6.428C15.557,14.84,15.795,15.506,16.162,15.864 M18.973,11.184c-0.291-0.321-0.783-0.496-1.384-0.496c-0.39,0-0.714,0.066-0.973,0.2c-0.254,0.132-0.461,0.297-0.621,0.491 c-0.157,0.197-0.265,0.405-0.328,0.628c-0.063,0.217-0.101,0.413-0.111,0.587h3.98C19.478,11.969,19.265,11.509,18.973,11.184z M15.057,7.738h4.985V6.524h-4.985L15.057,7.738z"></path>
</svg>',

		'chain'       => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M19.647,16.706a1.134,1.134,0,0,0-.343-.833l-2.549-2.549a1.134,1.134,0,0,0-.833-.343,1.168,1.168,0,0,0-.883.392l.233.226q.2.189.264.264a2.922,2.922,0,0,1,.184.233.986.986,0,0,1,.159.312,1.242,1.242,0,0,1,.043.337,1.172,1.172,0,0,1-1.176,1.176,1.237,1.237,0,0,1-.337-.043,1,1,0,0,1-.312-.159,2.76,2.76,0,0,1-.233-.184q-.073-.068-.264-.264l-.226-.233a1.19,1.19,0,0,0-.4.895,1.134,1.134,0,0,0,.343.833L15.837,19.3a1.13,1.13,0,0,0,.833.331,1.18,1.18,0,0,0,.833-.318l1.8-1.789a1.12,1.12,0,0,0,.343-.821Zm-8.615-8.64a1.134,1.134,0,0,0-.343-.833L8.163,4.7a1.134,1.134,0,0,0-.833-.343,1.184,1.184,0,0,0-.833.331L4.7,6.473a1.12,1.12,0,0,0-.343.821,1.134,1.134,0,0,0,.343.833l2.549,2.549a1.13,1.13,0,0,0,.833.331,1.184,1.184,0,0,0,.883-.38L8.728,10.4q-.2-.189-.264-.264A2.922,2.922,0,0,1,8.28,9.9a.986.986,0,0,1-.159-.312,1.242,1.242,0,0,1-.043-.337A1.172,1.172,0,0,1,9.254,8.079a1.237,1.237,0,0,1,.337.043,1,1,0,0,1,.312.159,2.761,2.761,0,0,1,.233.184q.073.068.264.264l.226.233a1.19,1.19,0,0,0,.4-.895ZM22,16.706a3.343,3.343,0,0,1-1.042,2.488l-1.8,1.789a3.536,3.536,0,0,1-4.988-.025l-2.525-2.537a3.384,3.384,0,0,1-1.017-2.488,3.448,3.448,0,0,1,1.078-2.561l-1.078-1.078a3.434,3.434,0,0,1-2.549,1.078,3.4,3.4,0,0,1-2.5-1.029L3.029,9.794A3.4,3.4,0,0,1,2,7.294,3.343,3.343,0,0,1,3.042,4.806l1.8-1.789A3.384,3.384,0,0,1,7.331,2a3.357,3.357,0,0,1,2.5,1.042l2.525,2.537a3.384,3.384,0,0,1,1.017,2.488,3.448,3.448,0,0,1-1.078,2.561l1.078,1.078a3.551,3.551,0,0,1,5.049-.049l2.549,2.549A3.4,3.4,0,0,1,22,16.706Z"></path>
</svg>',

		'codepen'     => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M22.016,8.84c-0.002-0.013-0.005-0.025-0.007-0.037c-0.005-0.025-0.008-0.048-0.015-0.072 c-0.003-0.015-0.01-0.028-0.013-0.042c-0.008-0.02-0.015-0.04-0.023-0.062c-0.007-0.015-0.013-0.028-0.02-0.042 c-0.008-0.02-0.018-0.037-0.03-0.057c-0.007-0.013-0.017-0.027-0.025-0.038c-0.012-0.018-0.023-0.035-0.035-0.052 c-0.01-0.013-0.02-0.025-0.03-0.037c-0.015-0.017-0.028-0.032-0.043-0.045c-0.01-0.012-0.022-0.023-0.035-0.035 c-0.015-0.015-0.032-0.028-0.048-0.04c-0.012-0.01-0.025-0.02-0.037-0.03c-0.005-0.003-0.01-0.008-0.015-0.012l-9.161-6.096 c-0.289-0.192-0.666-0.192-0.955,0L2.359,8.237C2.354,8.24,2.349,8.245,2.344,8.249L2.306,8.277 c-0.017,0.013-0.033,0.027-0.048,0.04C2.246,8.331,2.234,8.342,2.222,8.352c-0.015,0.015-0.028,0.03-0.042,0.047 c-0.012,0.013-0.022,0.023-0.03,0.037C2.139,8.453,2.125,8.471,2.115,8.488C2.107,8.501,2.099,8.514,2.09,8.526 C2.079,8.548,2.069,8.565,2.06,8.585C2.054,8.6,2.047,8.613,2.04,8.626C2.032,8.648,2.025,8.67,2.019,8.69 c-0.005,0.013-0.01,0.027-0.013,0.042C1.999,8.755,1.995,8.778,1.99,8.803C1.989,8.817,1.985,8.828,1.984,8.84 C1.978,8.879,1.975,8.915,1.975,8.954v6.093c0,0.037,0.003,0.075,0.008,0.112c0.002,0.012,0.005,0.025,0.007,0.038 c0.005,0.023,0.008,0.047,0.015,0.072c0.003,0.015,0.008,0.028,0.013,0.04c0.007,0.022,0.013,0.042,0.022,0.063 c0.007,0.015,0.013,0.028,0.02,0.04c0.008,0.02,0.018,0.038,0.03,0.058c0.007,0.013,0.015,0.027,0.025,0.038 c0.012,0.018,0.023,0.035,0.035,0.052c0.01,0.013,0.02,0.025,0.03,0.037c0.013,0.015,0.028,0.032,0.042,0.045 c0.012,0.012,0.023,0.023,0.035,0.035c0.015,0.013,0.032,0.028,0.048,0.04l0.038,0.03c0.005,0.003,0.01,0.007,0.013,0.01 l9.163,6.095C11.668,21.953,11.833,22,12,22c0.167,0,0.332-0.047,0.478-0.144l9.163-6.095l0.015-0.01 c0.013-0.01,0.027-0.02,0.037-0.03c0.018-0.013,0.035-0.028,0.048-0.04c0.013-0.012,0.025-0.023,0.035-0.035 c0.017-0.015,0.03-0.032,0.043-0.045c0.01-0.013,0.02-0.025,0.03-0.037c0.013-0.018,0.025-0.035,0.035-0.052 c0.008-0.013,0.018-0.027,0.025-0.038c0.012-0.02,0.022-0.038,0.03-0.058c0.007-0.013,0.013-0.027,0.02-0.04 c0.008-0.022,0.015-0.042,0.023-0.063c0.003-0.013,0.01-0.027,0.013-0.04c0.007-0.025,0.01-0.048,0.015-0.072 c0.002-0.013,0.005-0.027,0.007-0.037c0.003-0.042,0.007-0.079,0.007-0.117V8.954C22.025,8.915,22.022,8.879,22.016,8.84z M12.862,4.464l6.751,4.49l-3.016,2.013l-3.735-2.492V4.464z M11.138,4.464v4.009l-3.735,2.494L4.389,8.954L11.138,4.464z M3.699,10.562L5.853,12l-2.155,1.438V10.562z M11.138,19.536l-6.749-4.491l3.015-2.011l3.735,2.492V19.536z M12,14.035L8.953,12 L12,9.966L15.047,12L12,14.035z M12.862,19.536v-4.009l3.735-2.492l3.016,2.011L12.862,19.536z M20.303,13.438L18.147,12 l2.156-1.438L20.303,13.438z"></path>
</svg>',

		'deviantart'  => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M 18.19 5.636 18.19 2 18.188 2 14.553 2 14.19 2.366 12.474 5.636 11.935 6 5.81 6 5.81 10.994 9.177 10.994 9.477 11.357 5.81 18.363 5.81 22 5.811 22 9.447 22 9.81 21.634 11.526 18.364 12.065 18 18.19 18 18.19 13.006 14.823 13.006 14.523 12.641 18.19 5.636z"></path>
</svg>',

		'digg'        => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M4.5,5.4h2.2V16H1V8.5h3.5V5.4L4.5,5.4z M4.5,14.2v-4H3.2v4H4.5z M7.6,8.5V16h2.2V8.5C9.8,8.5,7.6,8.5,7.6,8.5z M7.6,5.4 v2.2h2.2V5.4C9.8,5.4,7.6,5.4,7.6,5.4z M10.7,8.5h5.7v10.1h-5.7v-1.8h3.5V16h-3.5C10.7,16,10.7,8.5,10.7,8.5z M14.2,14.2v-4h-1.3v4 H14.2z M17.3,8.5H23v10.1h-5.7v-1.8h3.5V16h-3.5C17.3,16,17.3,8.5,17.3,8.5z M20.8,14.2v-4h-1.3v4H20.8z"></path>
</svg>',

		'dribbble'    => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12,22C6.486,22,2,17.514,2,12S6.486,2,12,2c5.514,0,10,4.486,10,10S17.514,22,12,22z M20.434,13.369 c-0.292-0.092-2.644-0.794-5.32-0.365c1.117,3.07,1.572,5.57,1.659,6.09C18.689,17.798,20.053,15.745,20.434,13.369z M15.336,19.876c-0.127-0.749-0.623-3.361-1.822-6.477c-0.019,0.006-0.038,0.013-0.056,0.019c-4.818,1.679-6.547,5.02-6.701,5.334 c1.448,1.129,3.268,1.803,5.243,1.803C13.183,20.555,14.311,20.313,15.336,19.876z M5.654,17.724 c0.193-0.331,2.538-4.213,6.943-5.637c0.111-0.036,0.224-0.07,0.337-0.102c-0.214-0.485-0.448-0.971-0.692-1.45 c-4.266,1.277-8.405,1.223-8.778,1.216c-0.003,0.087-0.004,0.174-0.004,0.261C3.458,14.207,4.29,16.21,5.654,17.724z M3.639,10.264 c0.382,0.005,3.901,0.02,7.897-1.041c-1.415-2.516-2.942-4.631-3.167-4.94C5.979,5.41,4.193,7.613,3.639,10.264z M9.998,3.709 c0.236,0.316,1.787,2.429,3.187,5c3.037-1.138,4.323-2.867,4.477-3.085C16.154,4.286,14.17,3.471,12,3.471 C11.311,3.471,10.641,3.554,9.998,3.709z M18.612,6.612C18.432,6.855,17,8.69,13.842,9.979c0.199,0.407,0.389,0.821,0.567,1.237 c0.063,0.148,0.124,0.295,0.184,0.441c2.842-0.357,5.666,0.215,5.948,0.275C20.522,9.916,19.801,8.065,18.612,6.612z"></path>
</svg>',

		'dropbox'     => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12,6.134L6.069,9.797L2,6.54l5.883-3.843L12,6.134z M2,13.054l5.883,3.843L12,13.459L6.069,9.797L2,13.054z M12,13.459 l4.116,3.439L22,13.054l-4.069-3.257L12,13.459z M22,6.54l-5.884-3.843L12,6.134l5.931,3.663L22,6.54z M12.011,14.2l-4.129,3.426 l-1.767-1.153v1.291l5.896,3.539l5.897-3.539v-1.291l-1.769,1.153L12.011,14.2z"></path>
</svg>',

		'etsy'        => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M9.16033,4.038c0-.27174.02717-.43478.48913-.43478h6.22283c1.087,0,1.68478.92391,2.11957,2.663l.35326,1.38587h1.05978C19.59511,3.712,19.75815,2,19.75815,2s-2.663.29891-4.23913.29891h-7.962L3.29076,2.163v1.1413L4.731,3.57609c1.00543.19022,1.25.40761,1.33152,1.33152,0,0,.08152,2.71739.08152,7.20109s-.08152,7.17391-.08152,7.17391c0,.81522-.32609,1.11413-1.33152,1.30435l-1.44022.27174V22l4.2663-.13587h7.11957c1.60326,0,5.32609.13587,5.32609.13587.08152-.97826.625-5.40761.70652-5.89674H19.7038L18.644,18.52174c-.84239,1.90217-2.06522,2.038-3.42391,2.038H11.1712c-1.3587,0-2.01087-.54348-2.01087-1.712V12.65217s3.0163,0,3.99457.08152c.76087.05435,1.22283.27174,1.46739,1.33152l.32609,1.413h1.16848l-.08152-3.55978.163-3.587H15.02989l-.38043,1.57609c-.24457,1.03261-.40761,1.22283-1.46739,1.33152-1.38587.13587-4.02174.1087-4.02174.1087Z"></path>
</svg>',

		'facebook'    => '
<svg viewBox="0 0 320 512" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M279.14 288l14.22-92.66h-88.91v-60.13c0-25.35 12.42-50.06 52.24-50.06h40.42V6.26S260.43 0 225.36 0c-73.22 0-121.08 44.38-121.08 124.72v70.62H22.89V288h81.39v224h100.17V288z"/></path>
</svg>',

		'feed'        => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M2,8.667V12c5.515,0,10,4.485,10,10h3.333C15.333,14.637,9.363,8.667,2,8.667z M2,2v3.333 c9.19,0,16.667,7.477,16.667,16.667H22C22,10.955,13.045,2,2,2z M4.5,17C3.118,17,2,18.12,2,19.5S3.118,22,4.5,22S7,20.88,7,19.5 S5.882,17,4.5,17z"></path>
</svg>',

		'flickr'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M6.5,7c-2.75,0-5,2.25-5,5s2.25,5,5,5s5-2.25,5-5S9.25,7,6.5,7z M17.5,7c-2.75,0-5,2.25-5,5s2.25,5,5,5s5-2.25,5-5 S20.25,7,17.5,7z"></path>
</svg>',

		'foursquare'  => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M17.573,2c0,0-9.197,0-10.668,0S5,3.107,5,3.805s0,16.948,0,16.948c0,0.785,0.422,1.077,0.66,1.172 c0.238,0.097,0.892,0.177,1.285-0.275c0,0,5.035-5.843,5.122-5.93c0.132-0.132,0.132-0.132,0.262-0.132h3.26 c1.368,0,1.588-0.977,1.732-1.552c0.078-0.318,0.692-3.428,1.225-6.122l0.675-3.368C19.56,2.893,19.14,2,17.573,2z M16.495,7.22 c-0.053,0.252-0.372,0.518-0.665,0.518c-0.293,0-4.157,0-4.157,0c-0.467,0-0.802,0.318-0.802,0.787v0.508 c0,0.467,0.337,0.798,0.805,0.798c0,0,3.197,0,3.528,0s0.655,0.362,0.583,0.715c-0.072,0.353-0.407,2.102-0.448,2.295 c-0.04,0.193-0.262,0.523-0.655,0.523c-0.33,0-2.88,0-2.88,0c-0.523,0-0.683,0.068-1.033,0.503 c-0.35,0.437-3.505,4.223-3.505,4.223c-0.032,0.035-0.063,0.027-0.063-0.015V4.852c0-0.298,0.26-0.648,0.648-0.648 c0,0,8.228,0,8.562,0c0.315,0,0.61,0.297,0.528,0.683L16.495,7.22z"></path>
</svg>',

		'goodreads'   => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M17.3,17.5c-0.2,0.8-0.5,1.4-1,1.9c-0.4,0.5-1,0.9-1.7,1.2C13.9,20.9,13.1,21,12,21c-0.6,0-1.3-0.1-1.9-0.2 c-0.6-0.1-1.1-0.4-1.6-0.7c-0.5-0.3-0.9-0.7-1.2-1.2c-0.3-0.5-0.5-1.1-0.5-1.7h1.5c0.1,0.5,0.2,0.9,0.5,1.2 c0.2,0.3,0.5,0.6,0.9,0.8c0.3,0.2,0.7,0.3,1.1,0.4c0.4,0.1,0.8,0.1,1.2,0.1c1.4,0,2.5-0.4,3.1-1.2c0.6-0.8,1-2,1-3.5v-1.7h0 c-0.4,0.8-0.9,1.4-1.6,1.9c-0.7,0.5-1.5,0.7-2.4,0.7c-1,0-1.9-0.2-2.6-0.5C8.7,15,8.1,14.5,7.7,14c-0.5-0.6-0.8-1.3-1-2.1 c-0.2-0.8-0.3-1.6-0.3-2.5c0-0.9,0.1-1.7,0.4-2.5c0.3-0.8,0.6-1.5,1.1-2c0.5-0.6,1.1-1,1.8-1.4C10.3,3.2,11.1,3,12,3 c0.5,0,0.9,0.1,1.3,0.2c0.4,0.1,0.8,0.3,1.1,0.5c0.3,0.2,0.6,0.5,0.9,0.8c0.3,0.3,0.5,0.6,0.6,1h0V3.4h1.5V15 C17.6,15.9,17.5,16.7,17.3,17.5z M13.8,14.1c0.5-0.3,0.9-0.7,1.3-1.1c0.3-0.5,0.6-1,0.8-1.6c0.2-0.6,0.3-1.2,0.3-1.9 c0-0.6-0.1-1.2-0.2-1.9c-0.1-0.6-0.4-1.2-0.7-1.7c-0.3-0.5-0.7-0.9-1.3-1.2c-0.5-0.3-1.1-0.5-1.9-0.5s-1.4,0.2-1.9,0.5 c-0.5,0.3-1,0.7-1.3,1.2C8.5,6.4,8.3,7,8.1,7.6C8,8.2,7.9,8.9,7.9,9.5c0,0.6,0.1,1.3,0.2,1.9C8.3,12,8.6,12.5,8.9,13 c0.3,0.5,0.8,0.8,1.3,1.1c0.5,0.3,1.1,0.4,1.9,0.4C12.7,14.5,13.3,14.4,13.8,14.1z"></path>
</svg>',

		'google-plus' => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M8,11h6.61c0.06,0.35,0.11,0.7,0.11,1.16c0,4-2.68,6.84-6.72,6.84c-3.87,0-7-3.13-7-7s3.13-7,7-7 c1.89,0,3.47,0.69,4.69,1.83l-1.9,1.83C10.27,8.16,9.36,7.58,8,7.58c-2.39,0-4.34,1.98-4.34,4.42S5.61,16.42,8,16.42 c2.77,0,3.81-1.99,3.97-3.02H8V11L8,11z M23,11h-2V9h-2v2h-2v2h2v2h2v-2h2"></path>
</svg>',

		'google'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12.02,10.18v3.72v0.01h5.51c-0.26,1.57-1.67,4.22-5.5,4.22c-3.31,0-6.01-2.75-6.01-6.12s2.7-6.12,6.01-6.12 c1.87,0,3.13,0.8,3.85,1.48l2.84-2.76C16.99,2.99,14.73,2,12.03,2c-5.52,0-10,4.48-10,10s4.48,10,10,10c5.77,0,9.6-4.06,9.6-9.77 c0-0.83-0.11-1.42-0.25-2.05H12.02z"></path>
</svg>',

		'github'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12,2C6.477,2,2,6.477,2,12c0,4.419,2.865,8.166,6.839,9.489c0.5,0.09,0.682-0.218,0.682-0.484 c0-0.236-0.009-0.866-0.014-1.699c-2.782,0.602-3.369-1.34-3.369-1.34c-0.455-1.157-1.11-1.465-1.11-1.465 c-0.909-0.62,0.069-0.608,0.069-0.608c1.004,0.071,1.532,1.03,1.532,1.03c0.891,1.529,2.341,1.089,2.91,0.833 c0.091-0.647,0.349-1.086,0.635-1.337c-2.22-0.251-4.555-1.111-4.555-4.943c0-1.091,0.39-1.984,1.03-2.682 C6.546,8.54,6.202,7.524,6.746,6.148c0,0,0.84-0.269,2.75,1.025C10.295,6.95,11.15,6.84,12,6.836 c0.85,0.004,1.705,0.114,2.504,0.336c1.909-1.294,2.748-1.025,2.748-1.025c0.546,1.376,0.202,2.394,0.1,2.646 c0.64,0.699,1.026,1.591,1.026,2.682c0,3.841-2.337,4.687-4.565,4.935c0.359,0.307,0.679,0.917,0.679,1.852 c0,1.335-0.012,2.415-0.012,2.741c0,0.269,0.18,0.579,0.688,0.481C19.138,20.161,22,16.416,22,12C22,6.477,17.523,2,12,2z"></path>
</svg>',

		'instagram'   => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12,4.622c2.403,0,2.688,0.009,3.637,0.052c0.877,0.04,1.354,0.187,1.671,0.31c0.42,0.163,0.72,0.358,1.035,0.673 c0.315,0.315,0.51,0.615,0.673,1.035c0.123,0.317,0.27,0.794,0.31,1.671c0.043,0.949,0.052,1.234,0.052,3.637 s-0.009,2.688-0.052,3.637c-0.04,0.877-0.187,1.354-0.31,1.671c-0.163,0.42-0.358,0.72-0.673,1.035 c-0.315,0.315-0.615,0.51-1.035,0.673c-0.317,0.123-0.794,0.27-1.671,0.31c-0.949,0.043-1.233,0.052-3.637,0.052 s-2.688-0.009-3.637-0.052c-0.877-0.04-1.354-0.187-1.671-0.31c-0.42-0.163-0.72-0.358-1.035-0.673 c-0.315-0.315-0.51-0.615-0.673-1.035c-0.123-0.317-0.27-0.794-0.31-1.671C4.631,14.688,4.622,14.403,4.622,12 s0.009-2.688,0.052-3.637c0.04-0.877,0.187-1.354,0.31-1.671c0.163-0.42,0.358-0.72,0.673-1.035 c0.315-0.315,0.615-0.51,1.035-0.673c0.317-0.123,0.794-0.27,1.671-0.31C9.312,4.631,9.597,4.622,12,4.622 M12,3 C9.556,3,9.249,3.01,8.289,3.054C7.331,3.098,6.677,3.25,6.105,3.472C5.513,3.702,5.011,4.01,4.511,4.511 c-0.5,0.5-0.808,1.002-1.038,1.594C3.25,6.677,3.098,7.331,3.054,8.289C3.01,9.249,3,9.556,3,12c0,2.444,0.01,2.751,0.054,3.711 c0.044,0.958,0.196,1.612,0.418,2.185c0.23,0.592,0.538,1.094,1.038,1.594c0.5,0.5,1.002,0.808,1.594,1.038 c0.572,0.222,1.227,0.375,2.185,0.418C9.249,20.99,9.556,21,12,21s2.751-0.01,3.711-0.054c0.958-0.044,1.612-0.196,2.185-0.418 c0.592-0.23,1.094-0.538,1.594-1.038c0.5-0.5,0.808-1.002,1.038-1.594c0.222-0.572,0.375-1.227,0.418-2.185 C20.99,14.751,21,14.444,21,12s-0.01-2.751-0.054-3.711c-0.044-0.958-0.196-1.612-0.418-2.185c-0.23-0.592-0.538-1.094-1.038-1.594 c-0.5-0.5-1.002-0.808-1.594-1.038c-0.572-0.222-1.227-0.375-2.185-0.418C14.751,3.01,14.444,3,12,3L12,3z M12,7.378 c-2.552,0-4.622,2.069-4.622,4.622S9.448,16.622,12,16.622s4.622-2.069,4.622-4.622S14.552,7.378,12,7.378z M12,15 c-1.657,0-3-1.343-3-3s1.343-3,3-3s3,1.343,3,3S13.657,15,12,15z M16.804,6.116c-0.596,0-1.08,0.484-1.08,1.08 s0.484,1.08,1.08,1.08c0.596,0,1.08-0.484,1.08-1.08S17.401,6.116,16.804,6.116z"></path>
</svg>',

		'lastfm'    => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M10.5002,0 C4.7006,0 0,4.70109753 0,10.4998496 C0,16.2989526 4.7006,21 10.5002,21 C16.299,21 21,16.2989526 21,10.4998496 C21,4.70109753 16.299,0 10.5002,0 Z M14.69735,14.7204413 C13.3164,14.7151781 12.4346,14.0870017 11.83445,12.6859357 L11.6816001,12.3451305 L10.35405,9.31011397 C9.92709997,8.26875064 8.85260001,7.57120012 7.68010001,7.57120012 C6.06945001,7.57120012 4.75925001,8.88509738 4.75925001,10.5009524 C4.75925001,12.1164565 6.06945001,13.4303036 7.68010001,13.4303036 C8.77200001,13.4303036 9.76514999,12.827541 10.2719501,11.8567047 C10.2893,11.8235214 10.3239,11.8019673 10.36305,11.8038219 C10.4007,11.8053759 10.43535,11.8287847 10.4504,11.8631709 L10.98655,13.1045863 C11.0016,13.1389726 10.9956,13.17782 10.97225,13.2068931 C10.1605001,14.1995341 8.96020001,14.7683115 7.68010001,14.7683115 C5.33305,14.7683115 3.42340001,12.8535563 3.42340001,10.5009524 C3.42340001,8.14679459 5.33300001,6.23203946 7.68010001,6.23203946 C9.45720002,6.23203946 10.8909,7.19074535 11.6138,8.86359341 C11.6205501,8.88018505 12.3412,10.5707777 12.97445,12.0190621 C13.34865,12.8739575 13.64615,13.3959676 14.6288,13.4291508 C15.5663001,13.4612814 16.25375,12.9121534 16.25375,12.1484869 C16.25375,11.4691321 15.8320501,11.3003585 14.8803,10.98216 C13.2365,10.4397989 12.34495,9.88605929 12.34495,8.51817658 C12.34495,7.1809207 13.26665,6.31615054 14.692,6.31615054 C15.62875,6.31615054 16.3155,6.7286858 16.79215,7.5768142 C16.80495,7.60062396 16.8079001,7.62814302 16.8004001,7.65420843 C16.7929,7.68027384 16.7748,7.70212868 16.7507001,7.713808 L15.86145,8.16900031 C15.8178001,8.19200805 15.7643,8.17807308 15.73565,8.13847371 C15.43295,7.71345711 15.0956,7.52513451 14.6423,7.52513451 C14.05125,7.52513451 13.6220001,7.92899802 13.6220001,8.48649708 C13.6220001,9.17382194 14.1529001,9.34144259 15.0339,9.61923972 C15.14915,9.65578139 15.26955,9.69397731 15.39385,9.73432853 C16.7763,10.1865133 17.57675,10.7311301 17.57675,12.1836251 C17.57685,13.629654 16.3389,14.7204413 14.69735,14.7204413 Z"></path>
</svg>',

		'linkedin'    => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M19.7,3H4.3C3.582,3,3,3.582,3,4.3v15.4C3,20.418,3.582,21,4.3,21h15.4c0.718,0,1.3-0.582,1.3-1.3V4.3 C21,3.582,20.418,3,19.7,3z M8.339,18.338H5.667v-8.59h2.672V18.338z M7.004,8.574c-0.857,0-1.549-0.694-1.549-1.548 c0-0.855,0.691-1.548,1.549-1.548c0.854,0,1.547,0.694,1.547,1.548C8.551,7.881,7.858,8.574,7.004,8.574z M18.339,18.338h-2.669 v-4.177c0-0.996-0.017-2.278-1.387-2.278c-1.389,0-1.601,1.086-1.601,2.206v4.249h-2.667v-8.59h2.559v1.174h0.037 c0.356-0.675,1.227-1.387,2.526-1.387c2.703,0,3.203,1.779,3.203,4.092V18.338z"></path>
</svg>',

		'mail'        => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M20,4H4C2.895,4,2,4.895,2,6v12c0,1.105,0.895,2,2,2h16c1.105,0,2-0.895,2-2V6C22,4.895,21.105,4,20,4z M20,8.236l-8,4.882 L4,8.236V6h16V8.236z"></path>
</svg>',

		'meetup'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M19.24775,14.722a3.57032,3.57032,0,0,1-2.94457,3.52073,3.61886,3.61886,0,0,1-.64652.05634c-.07314-.0008-.10187.02846-.12507.09547A2.38881,2.38881,0,0,1,13.49453,20.094a2.33092,2.33092,0,0,1-1.827-.50716.13635.13635,0,0,0-.19878-.00408,3.191,3.191,0,0,1-2.104.60248,3.26309,3.26309,0,0,1-3.00324-2.71993,2.19076,2.19076,0,0,1-.03512-.30865c-.00156-.08579-.03413-.1189-.11608-.13493a2.86421,2.86421,0,0,1-1.23189-.56111,2.945,2.945,0,0,1-1.166-2.05749,2.97484,2.97484,0,0,1,.87524-2.50774.112.112,0,0,0,.02091-.16107,2.7213,2.7213,0,0,1-.36648-1.48A2.81256,2.81256,0,0,1,6.57673,7.58838a.35764.35764,0,0,0,.28869-.22819,4.2208,4.2208,0,0,1,6.02892-1.90111.25161.25161,0,0,0,.22023.0243,3.65608,3.65608,0,0,1,3.76031.90678A3.57244,3.57244,0,0,1,17.95918,8.626a2.97339,2.97339,0,0,1,.01829.57356.10637.10637,0,0,0,.0853.12792,1.97669,1.97669,0,0,1,1.27939,1.33733,2.00266,2.00266,0,0,1-.57112,2.12652c-.05284.05166-.04168.08328-.01173.13489A3.51189,3.51189,0,0,1,19.24775,14.722Zm-6.35959-.27836a1.6984,1.6984,0,0,0,1.14556,1.61113,3.82039,3.82039,0,0,0,1.036.17935,1.46888,1.46888,0,0,0,.73509-.12255.44082.44082,0,0,0,.26057-.44274.45312.45312,0,0,0-.29211-.43375.97191.97191,0,0,0-.20678-.063c-.21326-.03806-.42754-.0701-.63973-.11215a.54787.54787,0,0,1-.50172-.60926,2.75864,2.75864,0,0,1,.1773-.901c.1763-.535.414-1.045.64183-1.55913A12.686,12.686,0,0,0,15.85,10.47863a1.58461,1.58461,0,0,0,.04861-.87208,1.04531,1.04531,0,0,0-.85432-.83981,1.60658,1.60658,0,0,0-1.23654.16594.27593.27593,0,0,1-.36286-.03413c-.085-.0747-.16594-.15379-.24918-.23055a.98682.98682,0,0,0-1.33577-.04933,6.1468,6.1468,0,0,1-.4989.41615.47762.47762,0,0,1-.51535.03566c-.17448-.09307-.35512-.175-.53531-.25665a1.74949,1.74949,0,0,0-.56476-.2016,1.69943,1.69943,0,0,0-1.61654.91787,8.05815,8.05815,0,0,0-.32952.80126c-.45471,1.2557-.82507,2.53825-1.20838,3.81639a1.24151,1.24151,0,0,0,.51532,1.44389,1.42659,1.42659,0,0,0,1.22008.17166,1.09728,1.09728,0,0,0,.66994-.69764c.44145-1.04111.839-2.09989,1.25981-3.14926.11581-.28876.22792-.57874.35078-.86438a.44548.44548,0,0,1,.69189-.19539.50521.50521,0,0,1,.15044.43836,1.75625,1.75625,0,0,1-.14731.50453c-.27379.69219-.55265,1.38236-.82766,2.074a2.0836,2.0836,0,0,0-.14038.42876.50719.50719,0,0,0,.27082.57722.87236.87236,0,0,0,.66145.02739.99137.99137,0,0,0,.53406-.532q.61571-1.20914,1.228-2.42031.28423-.55863.57585-1.1133a.87189.87189,0,0,1,.29055-.35253.34987.34987,0,0,1,.37634-.01265.30291.30291,0,0,1,.12434.31459.56716.56716,0,0,1-.04655.1915c-.05318.12739-.10286.25669-.16183.38156-.34118.71775-.68754,1.43273-1.02568,2.152A2.00213,2.00213,0,0,0,12.88816,14.44366Zm4.78568,5.28972a.88573.88573,0,0,0-1.77139.00465.8857.8857,0,0,0,1.77139-.00465Zm-14.83838-7.296a.84329.84329,0,1,0,.00827-1.68655.8433.8433,0,0,0-.00827,1.68655Zm10.366-9.43673a.83506.83506,0,1,0-.0091,1.67.83505.83505,0,0,0,.0091-1.67Zm6.85014,5.22a.71651.71651,0,0,0-1.433.0093.71656.71656,0,0,0,1.433-.0093ZM5.37528,6.17908A.63823.63823,0,1,0,6.015,5.54483.62292.62292,0,0,0,5.37528,6.17908Zm6.68214,14.80843a.54949.54949,0,1,0-.55052.541A.54556.54556,0,0,0,12.05742,20.98752Zm8.53235-8.49689a.54777.54777,0,0,0-.54027.54023.53327.53327,0,0,0,.532.52293.51548.51548,0,0,0,.53272-.5237A.53187.53187,0,0,0,20.58977,12.49063ZM7.82846,2.4715a.44927.44927,0,1,0,.44484.44766A.43821.43821,0,0,0,7.82846,2.4715Zm13.775,7.60492a.41186.41186,0,0,0-.40065.39623.40178.40178,0,0,0,.40168.40168A.38994.38994,0,0,0,22,10.48172.39946.39946,0,0,0,21.60349,10.07642ZM5.79193,17.96207a.40469.40469,0,0,0-.397-.39646.399.399,0,0,0-.396.405.39234.39234,0,0,0,.39939.389A.39857.39857,0,0,0,5.79193,17.96207Z"></path>
</svg>',

		'medium'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M20.962,7.257l-5.457,8.867l-3.923-6.375l3.126-5.08c0.112-0.182,0.319-0.286,0.527-0.286c0.05,0,0.1,0.008,0.149,0.02 c0.039,0.01,0.078,0.023,0.114,0.041l5.43,2.715l0.006,0.003c0.004,0.002,0.007,0.006,0.011,0.008 C20.971,7.191,20.98,7.227,20.962,7.257z M9.86,8.592v5.783l5.14,2.57L9.86,8.592z M15.772,17.331l4.231,2.115 C20.554,19.721,21,19.529,21,19.016V8.835L15.772,17.331z M8.968,7.178L3.665,4.527C3.569,4.479,3.478,4.456,3.395,4.456 C3.163,4.456,3,4.636,3,4.938v11.45c0,0.306,0.224,0.669,0.498,0.806l4.671,2.335c0.12,0.06,0.234,0.088,0.337,0.088 c0.29,0,0.494-0.225,0.494-0.602V7.231C9,7.208,8.988,7.188,8.968,7.178z"></path>
</svg>',

		'pinterest'   => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12.289,2C6.617,2,3.606,5.648,3.606,9.622c0,1.846,1.025,4.146,2.666,4.878c0.25,0.111,0.381,0.063,0.439-0.169 c0.044-0.175,0.267-1.029,0.365-1.428c0.032-0.128,0.017-0.237-0.091-0.362C6.445,11.911,6.01,10.75,6.01,9.668 c0-2.777,2.194-5.464,5.933-5.464c3.23,0,5.49,2.108,5.49,5.122c0,3.407-1.794,5.768-4.13,5.768c-1.291,0-2.257-1.021-1.948-2.277 c0.372-1.495,1.089-3.112,1.089-4.191c0-0.967-0.542-1.775-1.663-1.775c-1.319,0-2.379,1.309-2.379,3.059 c0,1.115,0.394,1.869,0.394,1.869s-1.302,5.279-1.54,6.261c-0.405,1.666,0.053,4.368,0.094,4.604 c0.021,0.126,0.167,0.169,0.25,0.063c0.129-0.165,1.699-2.419,2.142-4.051c0.158-0.59,0.817-2.995,0.817-2.995 c0.43,0.784,1.681,1.446,3.013,1.446c3.963,0,6.822-3.494,6.822-7.833C20.394,5.112,16.849,2,12.289,2"></path>
</svg>',

		'pocket'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M21.927,4.194C21.667,3.48,20.982,3,20.222,3h-0.01h-1.721H3.839C3.092,3,2.411,3.47,2.145,4.17 C2.066,4.378,2.026,4.594,2.026,4.814v6.035l0.069,1.2c0.29,2.73,1.707,5.115,3.899,6.778c0.039,0.03,0.079,0.059,0.119,0.089 l0.025,0.018c1.175,0.859,2.491,1.441,3.91,1.727c0.655,0.132,1.325,0.2,1.991,0.2c0.615,0,1.232-0.057,1.839-0.17 c0.073-0.014,0.145-0.028,0.219-0.044c0.02-0.004,0.042-0.012,0.064-0.023c1.359-0.297,2.621-0.864,3.753-1.691l0.025-0.018 c0.04-0.029,0.08-0.058,0.119-0.089c2.192-1.664,3.609-4.049,3.898-6.778l0.069-1.2V4.814C22.026,4.605,22,4.398,21.927,4.194z M17.692,10.481l-4.704,4.512c-0.266,0.254-0.608,0.382-0.949,0.382c-0.342,0-0.684-0.128-0.949-0.382l-4.705-4.512 C5.838,9.957,5.82,9.089,6.344,8.542c0.524-0.547,1.392-0.565,1.939-0.04l3.756,3.601l3.755-3.601 c0.547-0.524,1.415-0.506,1.939,0.04C18.256,9.089,18.238,9.956,17.692,10.481z"></path>
</svg>',

		'reddit'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M22,11.816c0-1.256-1.021-2.277-2.277-2.277c-0.593,0-1.122,0.24-1.526,0.614c-1.481-0.965-3.455-1.594-5.647-1.69 l1.171-3.702l3.18,0.748c0.008,1.028,0.846,1.862,1.876,1.862c1.035,0,1.877-0.842,1.877-1.878c0-1.035-0.842-1.877-1.877-1.877 c-0.769,0-1.431,0.466-1.72,1.13l-3.508-0.826c-0.203-0.047-0.399,0.067-0.46,0.261l-1.35,4.268 c-2.316,0.038-4.411,0.67-5.97,1.671C5.368,9.765,4.853,9.539,4.277,9.539C3.021,9.539,2,10.56,2,11.816 c0,0.814,0.433,1.523,1.078,1.925c-0.037,0.221-0.061,0.444-0.061,0.672c0,3.292,4.011,5.97,8.941,5.97s8.941-2.678,8.941-5.97 c0-0.214-0.02-0.424-0.053-0.632C21.533,13.39,22,12.661,22,11.816z M18.776,4.394c0.606,0,1.1,0.493,1.1,1.1s-0.493,1.1-1.1,1.1 s-1.1-0.494-1.1-1.1S18.169,4.394,18.776,4.394z M2.777,11.816c0-0.827,0.672-1.5,1.499-1.5c0.313,0,0.598,0.103,0.838,0.269 c-0.851,0.676-1.477,1.479-1.812,2.36C2.983,12.672,2.777,12.27,2.777,11.816z M11.959,19.606c-4.501,0-8.164-2.329-8.164-5.193 S7.457,9.22,11.959,9.22s8.164,2.329,8.164,5.193S16.46,19.606,11.959,19.606z M20.636,13.001c-0.326-0.89-0.948-1.701-1.797-2.384 c0.248-0.186,0.55-0.301,0.883-0.301c0.827,0,1.5,0.673,1.5,1.5C21.223,12.299,20.992,12.727,20.636,13.001z M8.996,14.704 c-0.76,0-1.397-0.616-1.397-1.376c0-0.76,0.637-1.397,1.397-1.397c0.76,0,1.376,0.637,1.376,1.397 C10.372,14.088,9.756,14.704,8.996,14.704z M16.401,13.328c0,0.76-0.616,1.376-1.376,1.376c-0.76,0-1.399-0.616-1.399-1.376 c0-0.76,0.639-1.397,1.399-1.397C15.785,11.931,16.401,12.568,16.401,13.328z M15.229,16.708c0.152,0.152,0.152,0.398,0,0.55 c-0.674,0.674-1.727,1.002-3.219,1.002c-0.004,0-0.007-0.002-0.011-0.002c-0.004,0-0.007,0.002-0.011,0.002 c-1.492,0-2.544-0.328-3.218-1.002c-0.152-0.152-0.152-0.398,0-0.55c0.152-0.152,0.399-0.151,0.55,0 c0.521,0.521,1.394,0.775,2.669,0.775c0.004,0,0.007,0.002,0.011,0.002c0.004,0,0.007-0.002,0.011-0.002 c1.275,0,2.148-0.253,2.669-0.775C14.831,16.556,15.078,16.556,15.229,16.708z"></path>
</svg>',

		'skype'       => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M10.113,2.699c0.033-0.006,0.067-0.013,0.1-0.02c0.033,0.017,0.066,0.033,0.098,0.051L10.113,2.699z M2.72,10.223 c-0.006,0.034-0.011,0.069-0.017,0.103c0.018,0.032,0.033,0.064,0.051,0.095L2.72,10.223z M21.275,13.771 c0.007-0.035,0.011-0.071,0.018-0.106c-0.018-0.031-0.033-0.064-0.052-0.095L21.275,13.771z M13.563,21.199 c0.032,0.019,0.065,0.035,0.096,0.053c0.036-0.006,0.071-0.011,0.105-0.017L13.563,21.199z M22,16.386 c0,1.494-0.581,2.898-1.637,3.953c-1.056,1.057-2.459,1.637-3.953,1.637c-0.967,0-1.914-0.251-2.75-0.725 c0.036-0.006,0.071-0.011,0.105-0.017l-0.202-0.035c0.032,0.019,0.065,0.035,0.096,0.053c-0.543,0.096-1.099,0.147-1.654,0.147 c-1.275,0-2.512-0.25-3.676-0.743c-1.125-0.474-2.135-1.156-3.002-2.023c-0.867-0.867-1.548-1.877-2.023-3.002 c-0.493-1.164-0.743-2.401-0.743-3.676c0-0.546,0.049-1.093,0.142-1.628c0.018,0.032,0.033,0.064,0.051,0.095L2.72,10.223 c-0.006,0.034-0.011,0.069-0.017,0.103C2.244,9.5,2,8.566,2,7.615c0-1.493,0.582-2.898,1.637-3.953 c1.056-1.056,2.46-1.638,3.953-1.638c0.915,0,1.818,0.228,2.622,0.655c-0.033,0.007-0.067,0.013-0.1,0.02l0.199,0.031 c-0.032-0.018-0.066-0.034-0.098-0.051c0.002,0,0.003-0.001,0.004-0.001c0.586-0.112,1.187-0.169,1.788-0.169 c1.275,0,2.512,0.249,3.676,0.742c1.124,0.476,2.135,1.156,3.002,2.024c0.868,0.867,1.548,1.877,2.024,3.002 c0.493,1.164,0.743,2.401,0.743,3.676c0,0.575-0.054,1.15-0.157,1.712c-0.018-0.031-0.033-0.064-0.052-0.095l0.034,0.201 c0.007-0.035,0.011-0.071,0.018-0.106C21.754,14.494,22,15.432,22,16.386z M16.817,14.138c0-1.331-0.613-2.743-3.033-3.282 l-2.209-0.49c-0.84-0.192-1.807-0.444-1.807-1.237c0-0.794,0.679-1.348,1.903-1.348c2.468,0,2.243,1.696,3.468,1.696 c0.645,0,1.209-0.379,1.209-1.031c0-1.521-2.435-2.663-4.5-2.663c-2.242,0-4.63,0.952-4.63,3.488c0,1.221,0.436,2.521,2.839,3.123 l2.984,0.745c0.903,0.223,1.129,0.731,1.129,1.189c0,0.762-0.758,1.507-2.129,1.507c-2.679,0-2.307-2.062-3.743-2.062 c-0.645,0-1.113,0.444-1.113,1.078c0,1.236,1.501,2.886,4.856,2.886C15.236,17.737,16.817,16.199,16.817,14.138z"></path>
</svg>',

		'slideshare'  => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M11.738,10.232a2.142,2.142,0,0,1-.721,1.619,2.556,2.556,0,0,1-3.464,0,2.183,2.183,0,0,1,0-3.243,2.572,2.572,0,0,1,3.464,0A2.136,2.136,0,0,1,11.738,10.232Zm5.7,0a2.15,2.15,0,0,1-.715,1.619,2.563,2.563,0,0,1-3.469,0,2.183,2.183,0,0,1,0-3.243,2.58,2.58,0,0,1,3.469,0A2.144,2.144,0,0,1,17.439,10.232Zm2.555,2.045V4.7a2.128,2.128,0,0,0-.363-1.4,1.614,1.614,0,0,0-1.261-.415H5.742a1.656,1.656,0,0,0-1.278.386A2.246,2.246,0,0,0,4.129,4.7v7.643a8.212,8.212,0,0,0,1,.454q.516.193.92.318a6.847,6.847,0,0,0,.92.21q.516.085.806.125a6.615,6.615,0,0,0,.795.045l.665.006q.16,0,.642-.023t.506-.023a1.438,1.438,0,0,1,1.079.307,1.134,1.134,0,0,0,.114.1,7.215,7.215,0,0,0,.693.579q.079-1.033,1.34-.988.057,0,.415.017l.488.023q.13.006.517.011t.6-.011l.619-.051a5.419,5.419,0,0,0,.693-.1l.7-.153a5.353,5.353,0,0,0,.761-.221q.345-.131.766-.307a8.727,8.727,0,0,0,.818-.392Zm1.851-.057a10.4,10.4,0,0,1-4.225,2.862,6.5,6.5,0,0,1-.261,5.281,3.524,3.524,0,0,1-2.078,1.681,2.452,2.452,0,0,1-2.067-.17,1.915,1.915,0,0,1-.931-1.863l-.011-3.7V16.3l-.279-.068q-.188-.045-.267-.057l-.011,3.839a1.9,1.9,0,0,1-.943,1.863,2.481,2.481,0,0,1-2.078.17,3.519,3.519,0,0,1-2.067-1.7,6.546,6.546,0,0,1-.25-5.258A10.4,10.4,0,0,1,2.152,12.22a.56.56,0,0,1-.045-.715q.238-.3.681.011l.125.079a.767.767,0,0,1,.125.091V3.8a1.987,1.987,0,0,1,.534-1.4,1.7,1.7,0,0,1,1.295-.579H19.141a1.7,1.7,0,0,1,1.295.579,1.985,1.985,0,0,1,.534,1.4v7.882l.238-.17q.443-.307.681-.011a.56.56,0,0,1-.045.715Z"></path>
</svg>',

		'snapchat'    => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12.065,2a5.526,5.526,0,0,1,3.132.892A5.854,5.854,0,0,1,17.326,5.4a5.821,5.821,0,0,1,.351,2.33q0,.612-.117,2.487a.809.809,0,0,0,.365.091,1.93,1.93,0,0,0,.664-.176,1.93,1.93,0,0,1,.664-.176,1.3,1.3,0,0,1,.729.234.7.7,0,0,1,.351.6.839.839,0,0,1-.41.7,2.732,2.732,0,0,1-.9.41,3.192,3.192,0,0,0-.9.378.728.728,0,0,0-.41.618,1.575,1.575,0,0,0,.156.56,6.9,6.9,0,0,0,1.334,1.953,5.6,5.6,0,0,0,1.881,1.315,5.875,5.875,0,0,0,1.042.3.42.42,0,0,1,.365.456q0,.911-2.852,1.341a1.379,1.379,0,0,0-.143.507,1.8,1.8,0,0,1-.182.605.451.451,0,0,1-.429.241,5.878,5.878,0,0,1-.807-.085,5.917,5.917,0,0,0-.833-.085,4.217,4.217,0,0,0-.807.065,2.42,2.42,0,0,0-.82.293,6.682,6.682,0,0,0-.755.5q-.351.267-.755.527a3.886,3.886,0,0,1-.989.436A4.471,4.471,0,0,1,11.831,22a4.307,4.307,0,0,1-1.256-.176,3.784,3.784,0,0,1-.976-.436q-.4-.26-.749-.527a6.682,6.682,0,0,0-.755-.5,2.422,2.422,0,0,0-.807-.293,4.432,4.432,0,0,0-.82-.065,5.089,5.089,0,0,0-.853.1,5,5,0,0,1-.762.1.474.474,0,0,1-.456-.241,1.819,1.819,0,0,1-.182-.618,1.411,1.411,0,0,0-.143-.521q-2.852-.429-2.852-1.341a.42.42,0,0,1,.365-.456,5.793,5.793,0,0,0,1.042-.3,5.524,5.524,0,0,0,1.881-1.315,6.789,6.789,0,0,0,1.334-1.953A1.575,1.575,0,0,0,6,12.9a.728.728,0,0,0-.41-.618,3.323,3.323,0,0,0-.9-.384,2.912,2.912,0,0,1-.9-.41.814.814,0,0,1-.41-.684.71.71,0,0,1,.338-.593,1.208,1.208,0,0,1,.716-.241,1.976,1.976,0,0,1,.625.169,2.008,2.008,0,0,0,.69.169.919.919,0,0,0,.416-.091q-.117-1.849-.117-2.474A5.861,5.861,0,0,1,6.385,5.4,5.516,5.516,0,0,1,8.625,2.819,7.075,7.075,0,0,1,12.062,2Z"></path>
</svg>',

		'soundcloud'  => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M8.9,16.1L9,14L8.9,9.5c0-0.1,0-0.1-0.1-0.1c0,0-0.1-0.1-0.1-0.1c-0.1,0-0.1,0-0.1,0.1c0,0-0.1,0.1-0.1,0.1L8.3,14l0.1,2.1 c0,0.1,0,0.1,0.1,0.1c0,0,0.1,0.1,0.1,0.1C8.8,16.3,8.9,16.3,8.9,16.1z M11.4,15.9l0.1-1.8L11.4,9c0-0.1,0-0.2-0.1-0.2 c0,0-0.1,0-0.1,0s-0.1,0-0.1,0c-0.1,0-0.1,0.1-0.1,0.2l0,0.1l-0.1,5c0,0,0,0.7,0.1,2v0c0,0.1,0,0.1,0.1,0.1c0.1,0.1,0.1,0.1,0.2,0.1 c0.1,0,0.1,0,0.2-0.1c0.1,0,0.1-0.1,0.1-0.2L11.4,15.9z M2.4,12.9L2.5,14l-0.2,1.1c0,0.1,0,0.1-0.1,0.1c0,0-0.1,0-0.1-0.1L2.1,14 l0.1-1.1C2.2,12.9,2.3,12.9,2.4,12.9C2.3,12.9,2.4,12.9,2.4,12.9z M3.1,12.2L3.3,14l-0.2,1.8c0,0.1,0,0.1-0.1,0.1 c-0.1,0-0.1,0-0.1-0.1L2.8,14L3,12.2C3,12.2,3,12.2,3.1,12.2C3.1,12.2,3.1,12.2,3.1,12.2z M3.9,11.9L4.1,14l-0.2,2.1 c0,0.1,0,0.1-0.1,0.1c-0.1,0-0.1,0-0.1-0.1L3.5,14l0.2-2.1c0-0.1,0-0.1,0.1-0.1C3.9,11.8,3.9,11.8,3.9,11.9z M4.7,11.9L4.9,14 l-0.2,2.1c0,0.1-0.1,0.1-0.1,0.1c-0.1,0-0.1,0-0.1-0.1L4.3,14l0.2-2.2c0-0.1,0-0.1,0.1-0.1C4.7,11.7,4.7,11.8,4.7,11.9z M5.6,12 l0.2,2l-0.2,2.1c0,0.1-0.1,0.1-0.1,0.1c0,0-0.1,0-0.1,0c0,0,0-0.1,0-0.1L5.1,14l0.2-2c0,0,0-0.1,0-0.1s0.1,0,0.1,0 C5.5,11.9,5.5,11.9,5.6,12L5.6,12z M6.4,10.7L6.6,14l-0.2,2.1c0,0,0,0.1,0,0.1c0,0-0.1,0-0.1,0c-0.1,0-0.1-0.1-0.2-0.2L5.9,14 l0.2-3.3c0-0.1,0.1-0.2,0.2-0.2c0,0,0.1,0,0.1,0C6.4,10.7,6.4,10.7,6.4,10.7z M7.2,10l0.2,4.1l-0.2,2.1c0,0,0,0.1,0,0.1 c0,0-0.1,0-0.1,0c-0.1,0-0.2-0.1-0.2-0.2l-0.1-2.1L6.8,10c0-0.1,0.1-0.2,0.2-0.2c0,0,0.1,0,0.1,0S7.2,9.9,7.2,10z M8,9.6L8.2,14 L8,16.1c0,0.1-0.1,0.2-0.2,0.2c-0.1,0-0.2-0.1-0.2-0.2L7.5,14l0.1-4.4c0-0.1,0-0.1,0.1-0.1c0,0,0.1-0.1,0.1-0.1c0.1,0,0.1,0,0.1,0.1 C8,9.6,8,9.6,8,9.6z M11.4,16.1L11.4,16.1L11.4,16.1z M9.7,9.6L9.8,14l-0.1,2.1c0,0.1,0,0.1-0.1,0.2s-0.1,0.1-0.2,0.1 c-0.1,0-0.1,0-0.1-0.1s-0.1-0.1-0.1-0.2L9.2,14l0.1-4.4c0-0.1,0-0.1,0.1-0.2s0.1-0.1,0.2-0.1c0.1,0,0.1,0,0.2,0.1S9.7,9.5,9.7,9.6 L9.7,9.6z M10.6,9.8l0.1,4.3l-0.1,2c0,0.1,0,0.1-0.1,0.2c0,0-0.1,0.1-0.2,0.1c-0.1,0-0.1,0-0.2-0.1c0,0-0.1-0.1-0.1-0.2L10,14 l0.1-4.3c0-0.1,0-0.1,0.1-0.2c0,0,0.1-0.1,0.2-0.1c0.1,0,0.1,0,0.2,0.1S10.6,9.7,10.6,9.8z M12.4,14l-0.1,2c0,0.1,0,0.1-0.1,0.2 c-0.1,0.1-0.1,0.1-0.2,0.1c-0.1,0-0.1,0-0.2-0.1c-0.1-0.1-0.1-0.1-0.1-0.2l-0.1-1l-0.1-1l0.1-5.5v0c0-0.1,0-0.2,0.1-0.2 c0.1,0,0.1-0.1,0.2-0.1c0,0,0.1,0,0.1,0c0.1,0,0.1,0.1,0.1,0.2L12.4,14z M22.1,13.9c0,0.7-0.2,1.3-0.7,1.7c-0.5,0.5-1.1,0.7-1.7,0.7 h-6.8c-0.1,0-0.1,0-0.2-0.1c-0.1-0.1-0.1-0.1-0.1-0.2V8.2c0-0.1,0.1-0.2,0.2-0.3c0.5-0.2,1-0.3,1.6-0.3c1.1,0,2.1,0.4,2.9,1.1 c0.8,0.8,1.3,1.7,1.4,2.8c0.3-0.1,0.6-0.2,1-0.2c0.7,0,1.3,0.2,1.7,0.7C21.8,12.6,22.1,13.2,22.1,13.9L22.1,13.9z"></path>
</svg>',

		'spotify'     => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12,2C6.477,2,2,6.477,2,12c0,5.523,4.477,10,10,10c5.523,0,10-4.477,10-10C22,6.477,17.523,2,12,2 M16.586,16.424 c-0.18,0.295-0.563,0.387-0.857,0.207c-2.348-1.435-5.304-1.76-8.785-0.964c-0.335,0.077-0.67-0.133-0.746-0.469 c-0.077-0.335,0.132-0.67,0.469-0.746c3.809-0.871,7.077-0.496,9.713,1.115C16.673,15.746,16.766,16.13,16.586,16.424 M17.81,13.7 c-0.226,0.367-0.706,0.482-1.072,0.257c-2.687-1.652-6.785-2.131-9.965-1.166C6.36,12.917,5.925,12.684,5.8,12.273 C5.675,11.86,5.908,11.425,6.32,11.3c3.632-1.102,8.147-0.568,11.234,1.328C17.92,12.854,18.035,13.335,17.81,13.7 M17.915,10.865 c-3.223-1.914-8.54-2.09-11.618-1.156C5.804,9.859,5.281,9.58,5.131,9.086C4.982,8.591,5.26,8.069,5.755,7.919 c3.532-1.072,9.404-0.865,13.115,1.338c0.445,0.264,0.59,0.838,0.327,1.282C18.933,10.983,18.359,11.129,17.915,10.865"></path>
</svg>',

		'stumbleupon' => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12,4.294c-2.469,0-4.471,2.002-4.471,4.471v6.353c0,0.585-0.474,1.059-1.059,1.059c-0.585,0-1.059-0.474-1.059-1.059 v-2.824H2v2.941c0,2.469,2.002,4.471,4.471,4.471c2.469,0,4.471-2.002,4.471-4.471V8.765c0-0.585,0.474-1.059,1.059-1.059 s1.059,0.474,1.059,1.059v1.294l1.412,0.647l2-0.647V8.765C16.471,6.296,14.469,4.294,12,4.294z M13.059,12.353v2.882 c0,2.469,2.002,4.471,4.471,4.471S22,17.704,22,15.235v-2.824h-3.412v2.824c0,0.585-0.474,1.059-1.059,1.059 c-0.585,0-1.059-0.474-1.059-1.059v-2.882l-2,0.647L13.059,12.353z"></path>
</svg>',

		'tumblr'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M16.749,17.396c-0.357,0.17-1.041,0.319-1.551,0.332c-1.539,0.041-1.837-1.081-1.85-1.896V9.847h3.861V6.937h-3.847V2.039 c0,0-2.77,0-2.817,0c-0.046,0-0.127,0.041-0.138,0.144c-0.165,1.499-0.867,4.13-3.783,5.181v2.484h1.945v6.282 c0,2.151,1.587,5.206,5.775,5.135c1.413-0.024,2.982-0.616,3.329-1.126L16.749,17.396z"></path>
</svg>',

		'twitch'      => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M16.499,8.089h-1.636v4.91h1.636V8.089z M12,8.089h-1.637v4.91H12V8.089z M4.228,3.178L3,6.451v13.092h4.499V22h2.456 l2.454-2.456h3.681L21,14.636V3.178H4.228z M19.364,13.816l-2.864,2.865H12l-2.453,2.453V16.68H5.863V4.814h13.501V13.816z"></path>
</svg>',
		'tiktok'      => '
<svg viewBox="0 0 16 16" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M9 0h1.98c.144.715.54 1.617 1.235 2.512C12.895 3.389 13.797 4 15 4v2c-1.753 0-3.07-.814-4-1.829V11a5 5 0 1 1-5-5v2a3 3 0 1 0 3 3V0Z"/></path>
</svg>',

		'twitter'     => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M22.23,5.924c-0.736,0.326-1.527,0.547-2.357,0.646c0.847-0.508,1.498-1.312,1.804-2.27 c-0.793,0.47-1.671,0.812-2.606,0.996C18.324,4.498,17.257,4,16.077,4c-2.266,0-4.103,1.837-4.103,4.103 c0,0.322,0.036,0.635,0.106,0.935C8.67,8.867,5.647,7.234,3.623,4.751C3.27,5.357,3.067,6.062,3.067,6.814 c0,1.424,0.724,2.679,1.825,3.415c-0.673-0.021-1.305-0.206-1.859-0.513c0,0.017,0,0.034,0,0.052c0,1.988,1.414,3.647,3.292,4.023 c-0.344,0.094-0.707,0.144-1.081,0.144c-0.264,0-0.521-0.026-0.772-0.074c0.522,1.63,2.038,2.816,3.833,2.85 c-1.404,1.1-3.174,1.756-5.096,1.756c-0.331,0-0.658-0.019-0.979-0.057c1.816,1.164,3.973,1.843,6.29,1.843 c7.547,0,11.675-6.252,11.675-11.675c0-0.178-0.004-0.355-0.012-0.531C20.985,7.47,21.68,6.747,22.23,5.924z"></path>
</svg>',

		'vimeo'       => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M22.396,7.164c-0.093,2.026-1.507,4.799-4.245,8.32C15.322,19.161,12.928,21,10.97,21c-1.214,0-2.24-1.119-3.079-3.359 c-0.56-2.053-1.119-4.106-1.68-6.159C5.588,9.243,4.921,8.122,4.206,8.122c-0.156,0-0.701,0.328-1.634,0.98L1.594,7.841 c1.027-0.902,2.04-1.805,3.037-2.708C6.001,3.95,7.03,3.327,7.715,3.264c1.619-0.156,2.616,0.951,2.99,3.321 c0.404,2.557,0.685,4.147,0.841,4.769c0.467,2.121,0.981,3.181,1.542,3.181c0.435,0,1.09-0.688,1.963-2.065 c0.871-1.376,1.338-2.422,1.401-3.142c0.125-1.187-0.343-1.782-1.401-1.782c-0.498,0-1.012,0.115-1.541,0.341 c1.023-3.35,2.977-4.977,5.862-4.884C21.511,3.066,22.52,4.453,22.396,7.164z"></path>
</svg>',

		'vk'          => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M22,7.1c0.2,0.4-0.4,1.5-1.6,3.1c-0.2,0.2-0.4,0.5-0.7,0.9c-0.5,0.7-0.9,1.1-0.9,1.4c-0.1,0.3-0.1,0.6,0.1,0.8 c0.1,0.1,0.4,0.4,0.8,0.9h0l0,0c1,0.9,1.6,1.7,2,2.3c0,0,0,0.1,0.1,0.1c0,0.1,0,0.1,0.1,0.3c0,0.1,0,0.2,0,0.4 c0,0.1-0.1,0.2-0.3,0.3c-0.1,0.1-0.4,0.1-0.6,0.1l-2.7,0c-0.2,0-0.4,0-0.6-0.1c-0.2-0.1-0.4-0.1-0.5-0.2l-0.2-0.1 c-0.2-0.1-0.5-0.4-0.7-0.7s-0.5-0.6-0.7-0.8c-0.2-0.2-0.4-0.4-0.6-0.6C14.8,15,14.6,15,14.4,15c0,0,0,0-0.1,0c0,0-0.1,0.1-0.2,0.2 c-0.1,0.1-0.2,0.2-0.2,0.3c-0.1,0.1-0.1,0.3-0.2,0.5c-0.1,0.2-0.1,0.5-0.1,0.8c0,0.1,0,0.2,0,0.3c0,0.1-0.1,0.2-0.1,0.2l0,0.1 c-0.1,0.1-0.3,0.2-0.6,0.2h-1.2c-0.5,0-1,0-1.5-0.2c-0.5-0.1-1-0.3-1.4-0.6s-0.7-0.5-1.1-0.7s-0.6-0.4-0.7-0.6l-0.3-0.3 c-0.1-0.1-0.2-0.2-0.3-0.3s-0.4-0.5-0.7-0.9s-0.7-1-1.1-1.6c-0.4-0.6-0.8-1.3-1.3-2.2C2.9,9.4,2.5,8.5,2.1,7.5C2,7.4,2,7.3,2,7.2 c0-0.1,0-0.1,0-0.2l0-0.1c0.1-0.1,0.3-0.2,0.6-0.2l2.9,0c0.1,0,0.2,0,0.2,0.1S5.9,6.9,5.9,7L6,7c0.1,0.1,0.2,0.2,0.3,0.3 C6.4,7.7,6.5,8,6.7,8.4C6.9,8.8,7,9,7.1,9.2l0.2,0.3c0.2,0.4,0.4,0.8,0.6,1.1c0.2,0.3,0.4,0.5,0.5,0.7s0.3,0.3,0.4,0.4 c0.1,0.1,0.3,0.1,0.4,0.1c0.1,0,0.2,0,0.3-0.1c0,0,0,0,0.1-0.1c0,0,0.1-0.1,0.1-0.2c0.1-0.1,0.1-0.3,0.1-0.5c0-0.2,0.1-0.5,0.1-0.8 c0-0.4,0-0.8,0-1.3c0-0.3,0-0.5-0.1-0.8c0-0.2-0.1-0.4-0.1-0.5L9.6,7.6C9.4,7.3,9.1,7.2,8.7,7.1C8.6,7.1,8.6,7,8.7,6.9 C8.9,6.7,9,6.6,9.1,6.5c0.4-0.2,1.2-0.3,2.5-0.3c0.6,0,1,0.1,1.4,0.1c0.1,0,0.3,0.1,0.3,0.1c0.1,0.1,0.2,0.1,0.2,0.3 c0,0.1,0.1,0.2,0.1,0.3s0,0.3,0,0.5c0,0.2,0,0.4,0,0.6c0,0.2,0,0.4,0,0.7c0,0.3,0,0.6,0,0.9c0,0.1,0,0.2,0,0.4c0,0.2,0,0.4,0,0.5 c0,0.1,0,0.3,0,0.4s0.1,0.3,0.1,0.4c0.1,0.1,0.1,0.2,0.2,0.3c0.1,0,0.1,0,0.2,0c0.1,0,0.2,0,0.3-0.1c0.1-0.1,0.2-0.2,0.4-0.4 s0.3-0.4,0.5-0.7c0.2-0.3,0.5-0.7,0.7-1.1c0.4-0.7,0.8-1.5,1.1-2.3c0-0.1,0.1-0.1,0.1-0.2c0-0.1,0.1-0.1,0.1-0.1l0,0l0.1,0 c0,0,0,0,0.1,0s0.2,0,0.2,0l3,0c0.3,0,0.5,0,0.7,0S21.9,7,21.9,7L22,7.1z"></path>
</svg>',

		'wordpress'   => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12.158,12.786L9.46,20.625c0.806,0.237,1.657,0.366,2.54,0.366c1.047,0,2.051-0.181,2.986-0.51 c-0.024-0.038-0.046-0.079-0.065-0.124L12.158,12.786z M3.009,12c0,3.559,2.068,6.634,5.067,8.092L3.788,8.341 C3.289,9.459,3.009,10.696,3.009,12z M18.069,11.546c0-1.112-0.399-1.881-0.741-2.48c-0.456-0.741-0.883-1.368-0.883-2.109 c0-0.826,0.627-1.596,1.51-1.596c0.04,0,0.078,0.005,0.116,0.007C16.472,3.904,14.34,3.009,12,3.009 c-3.141,0-5.904,1.612-7.512,4.052c0.211,0.007,0.41,0.011,0.579,0.011c0.94,0,2.396-0.114,2.396-0.114 C7.947,6.93,8.004,7.642,7.52,7.699c0,0-0.487,0.057-1.029,0.085l3.274,9.739l1.968-5.901l-1.401-3.838 C9.848,7.756,9.389,7.699,9.389,7.699C8.904,7.67,8.961,6.93,9.446,6.958c0,0,1.484,0.114,2.368,0.114 c0.94,0,2.397-0.114,2.397-0.114c0.485-0.028,0.542,0.684,0.057,0.741c0,0-0.488,0.057-1.029,0.085l3.249,9.665l0.897-2.996 C17.841,13.284,18.069,12.316,18.069,11.546z M19.889,7.686c0.039,0.286,0.06,0.593,0.06,0.924c0,0.912-0.171,1.938-0.684,3.22 l-2.746,7.94c2.673-1.558,4.47-4.454,4.47-7.771C20.991,10.436,20.591,8.967,19.889,7.686z M12,22C6.486,22,2,17.514,2,12 C2,6.486,6.486,2,12,2c5.514,0,10,4.486,10,10C22,17.514,17.514,22,12,22z"></path>
</svg>',

		'yelp'        => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M12.271,16.718v1.417q-.011,3.257-.067,3.4a.707.707,0,0,1-.569.446,4.637,4.637,0,0,1-2.024-.424A4.609,4.609,0,0,1,7.8,20.565a.844.844,0,0,1-.19-.4.692.692,0,0,1,.044-.29,3.181,3.181,0,0,1,.379-.524q.335-.412,2.019-2.409.011,0,.669-.781a.757.757,0,0,1,.44-.274.965.965,0,0,1,.552.039.945.945,0,0,1,.418.324.732.732,0,0,1,.139.468Zm-1.662-2.8a.783.783,0,0,1-.58.781l-1.339.435q-3.067.981-3.257.981a.711.711,0,0,1-.6-.4,2.636,2.636,0,0,1-.19-.836,9.134,9.134,0,0,1,.011-1.857,3.559,3.559,0,0,1,.335-1.389.659.659,0,0,1,.625-.357,22.629,22.629,0,0,1,2.253.859q.781.324,1.283.524l.937.379a.771.771,0,0,1,.4.34A.982.982,0,0,1,10.609,13.917Zm9.213,3.313a4.467,4.467,0,0,1-1.021,1.8,4.559,4.559,0,0,1-1.512,1.417.671.671,0,0,1-.7-.078q-.156-.112-2.052-3.2l-.524-.859a.761.761,0,0,1-.128-.513.957.957,0,0,1,.217-.513.774.774,0,0,1,.926-.29q.011.011,1.327.446,2.264.736,2.7.887a2.082,2.082,0,0,1,.524.229.673.673,0,0,1,.245.68Zm-7.5-7.049q.056,1.137-.6,1.361-.647.19-1.272-.792L6.237,4.08a.7.7,0,0,1,.212-.691,5.788,5.788,0,0,1,2.314-1,5.928,5.928,0,0,1,2.5-.352.681.681,0,0,1,.547.5q.034.2.245,3.407T12.327,10.181Zm7.384,1.2a.679.679,0,0,1-.29.658q-.167.112-3.67.959-.747.167-1.015.257l.011-.022a.769.769,0,0,1-.513-.044.914.914,0,0,1-.413-.357.786.786,0,0,1,0-.971q.011-.011.836-1.137,1.394-1.908,1.673-2.275a2.423,2.423,0,0,1,.379-.435A.7.7,0,0,1,17.435,8a4.482,4.482,0,0,1,1.372,1.489,4.81,4.81,0,0,1,.9,1.868v.034Z"></path>
</svg>',

		'youtube'     => '
<svg viewBox="0 0 24 24" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
	<path d="M21.8,8.001c0,0-0.195-1.378-0.795-1.985c-0.76-0.797-1.613-0.801-2.004-0.847c-2.799-0.202-6.997-0.202-6.997-0.202 h-0.009c0,0-4.198,0-6.997,0.202C4.608,5.216,3.756,5.22,2.995,6.016C2.395,6.623,2.2,8.001,2.2,8.001S2,9.62,2,11.238v1.517 c0,1.618,0.2,3.237,0.2,3.237s0.195,1.378,0.795,1.985c0.761,0.797,1.76,0.771,2.205,0.855c1.6,0.153,6.8,0.201,6.8,0.201 s4.203-0.006,7.001-0.209c0.391-0.047,1.243-0.051,2.004-0.847c0.6-0.607,0.795-1.985,0.795-1.985s0.2-1.618,0.2-3.237v-1.517 C22,9.62,21.8,8.001,21.8,8.001z M9.935,14.594l-0.001-5.62l5.404,2.82L9.935,14.594z"></path>
</svg>',

		

	);

}


/**
 * SVG icons related functions
 *
 * @package WordPress
 * @subpackage Twenty_Nineteen
 * @since 1.0.0
 */

/**
 * Gets the SVG code for a given icon.
 */
function liora_get_icon_svg( $icon, $size = 24 ) {
	return liora_SVG_Icons::get_svg( 'ui', $icon, $size );
}

/**
 * Gets the SVG code for a given social icon.
 */
function liora_get_social_icon_svg( $icon, $size = 24 ) {
	return liora_SVG_Icons::get_svg( 'social', $icon, $size );
}

/**
 * Detects the social network from a URL and returns the SVG code for its icon.
 */
function liora_get_social_link_svg( $uri, $size = 24 ) {
	return liora_SVG_Icons::get_social_link_svg( $uri, $size );
}

/**
 * Display SVG icons in social links menu.
 *
 * @param  string  $item_output The menu item output.
 * @param  WP_Post $item        Menu item object.
 * @param  int     $depth       Depth of the menu.
 * @param  array   $args        wp_nav_menu() arguments.
 * @return string  $item_output The menu item output with social icon.
 */
function liora_nav_menu_social_icons( $item_output, $item, $depth, $args ) {
	// Change SVG icon inside social links menu if there is supported URL.
	if ( 'social' === $args->theme_location ) {
		$svg = liora_get_social_link_svg( $item->url, 26 );
		if ( empty( $svg ) ) {
			$svg = liora_get_icon_svg( 'link' );
		}
		$item_output = str_replace( $args->link_after, '</span>' . $svg, $item_output );
	}
	if ( 'social2' === $args->theme_location ) {
		$svg = liora_get_social_link_svg( $item->url, 26 );
		if ( empty( $svg ) ) {
			$svg = liora_get_icon_svg( 'link' );
		}
		$item_output = str_replace( $args->link_after, '</span>' . $svg, $item_output );
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'liora_nav_menu_social_icons', 10, 4 );
