<?php
/**
 * Journal Blog functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Journal_Blog
 */

if ( ! function_exists( 'journal_blog_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function journal_blog_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Journal Blog, use a find and replace
		 * to change 'journal-blog' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'journal-blog', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		add_image_size('journal-featured', 331, 331, array('center','top') );


		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'menu-1' => esc_html__( 'Primary', 'journal-blog' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'journal_blog_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		$args = array(
			'width'         => 1000,
			'height'        => 1000,
			'flex-width'    => true,
			'flex-height'    => true,
			'default-image' => JOURNAL_DIRECTORY_URI . '/assets/images/girl-2907463_1920.jpg',
		);
		add_theme_support( 'custom-header', $args );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support( 'custom-logo', array(
			'height'      => 60,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => false,
		) );
	}
endif;
add_action( 'after_setup_theme', 'journal_blog_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function journal_blog_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'journal_blog_content_width', 640 );
}
add_action( 'after_setup_theme', 'journal_blog_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function journal_blog_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'journal-blog' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here.', 'journal-blog' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
}
add_action( 'widgets_init', 'journal_blog_widgets_init' );


// Excerpt Length
function journal_blog_excerpt_length( $length ) {
	if ( is_admin() ) 
	{
		return $length;
	}

	$length =  20;

	return $length;
}
add_filter( 'excerpt_length', 'journal_blog_excerpt_length', 999 );

// Excerpt More
function journal_blog_excerpt_more( $link ) {
	if ( is_admin() ) {
		return $link;
	}

	$link = sprintf( '<a href="%1$s" class="btn btn-readmore">%2$s</a>',
		esc_url( get_permalink( get_the_ID() ) ),
		/* translators: %s: Name of current post */
		sprintf( __( 'Read More <span class="screen-reader-text"> "%s"</span>', 'journal-blog' ), get_the_title( get_the_ID() ) )
	);
	return ' &hellip; ' . $link;
}
add_filter( 'excerpt_more', 'journal_blog_excerpt_more' );


function journal_blog_social_cb(){ 	
	$facebook = get_theme_mod('journal_blog_social_facebook','');
	$twitter = get_theme_mod('journal_blog_social_twitter','');
	$gplus = get_theme_mod('journal_blog_social_gplus','');
	$youtube = get_theme_mod('journal_blog_social_youtube','');
	$instagram = get_theme_mod('journal_blog_social_instagram','');
	$pinterest = get_theme_mod('journal_blog_social_pinterest','');
	$rss = get_theme_mod('journal_blog_social_rssfeed','');
	?>
	<div class="social">
		<ul>
		<?php if (!empty($facebook)) { ?>
			<li><a href="<?php echo esc_url($facebook)?>" class="facebook" title="Facebook" target="_blank"><i class="fa fa-facebook"></i></a></li>
		<?php 
		} 
		if (!empty($twitter)) {
		?>		
			<li><a href="<?php echo esc_url($twitter)?>" class="twitter" title="Twitter" target="_blank"><i class="fa fa-twitter"></i></a></li>
			<?php 
		} 
		if (!empty($gplus)) {
		?>	
			<li><a href="<?php echo esc_url($gplus)?>" class="gplus" title="Google Plus" target="_blank"><i class="fa fa-google-plus"></i></a></li>
			<?php 
		} 
		if (!empty($youtube)) {
		?>	
			<li><a href="<?php echo esc_url($youtube)?>" class="youtube" title="Youtube" target="_blank"><i class="fa fa-youtube"></i></a></li>
		<?php 
		} 
		if (!empty($instagram)) {
		?>	
			<li><a href="<?php echo esc_url($instagram)?>" class="instagram" title="instagram" target="_blank"><i class="fa fa-instagram"></i></a></li>
			<?php 
		} 
		if (!empty($pinterest)) {
		?>	
			<li><a href="<?php echo esc_url($pinterest)?>" class="pinterest" title="pinterest" target="_blank"><i class="fa fa-pinterest"></i></a></li>
			<?php 
		}
		if (!empty($rss)) {
		?>	
			<li><a href="<?php echo esc_url($rss)?>" class="rss" title="RSS" target="_blank"><i class="fa fa-rss"></i></a></li>
		<?php 
		} 
		?>	
		</ul>
	</div>
	<?php 
} 
add_action( 'journal_blog_social_links', 'journal_blog_social_cb' );
