<?php
/**
 * Functions which enhance the theme by hooking into WordPress core actions/hooks
 *
 * @package IT_Zone
 */
if ( ! function_exists( 'it_zone_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_setup() {
		/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on IT_Zone, use a find and replace
		* to change 'it-zone' to the name of your theme in all the template files.
		*/
		load_theme_textdomain( 'it-zone', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
		add_theme_support( 'title-tag' );

		/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus(
			array(
				'primary'     => esc_html__( 'Primary', 'it-zone' ),
				'secondary'   => esc_html__( 'Secondary', 'it-zone' ),
				'tertiary'    => esc_html__( 'Tertiary', 'it-zone' ),
				'footer-menu' => esc_html__( 'Footer Menu', 'it-zone' ),
			)
		);

		/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
		add_theme_support(
			'html5',
			array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'style',
				'script',
			)
		);

		// Set up the WordPress core custom background feature.
		add_theme_support(
			'custom-background',
			apply_filters(
				'it_zone_custom_background_args',
				array(
					'default-color' => 'ffffff',
					'default-image' => '',
				)
			)
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );
		add_theme_support( 'align-wide' );
		add_theme_support( 'appearance-tools' );
		add_theme_support( 'border' );
		add_theme_support( 'custom-spacing' );
		add_theme_support( 'custom-line-height' );
		add_theme_support( 'link-color' );
		add_theme_support( 'responsive-embeds' );
		add_theme_support( 'editor-styles' );

		// Add support for wp-block-styles
		add_theme_support( 'wp-block-styles' );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support(
			'custom-logo',
			array(
				'height'      => 250,
				'width'       => 250,
				'flex-width'  => true,
				'flex-height' => true,
			)
		);
		/**
		 * Registering new image sizes
		 */
		add_image_size( 'blog_card_image', 384, 480, true );

		// Remove core block patterns
		remove_theme_support( 'core-block-patterns' );
		/**
		 * Set up the WordPress core custom header feature.
		 *
		 * @uses it_zone_header_style()
		 */
		add_theme_support(
			'custom-header',
			apply_filters(
				'it_zone_custom_header_args',
				array(
					'video'              => false,
					'default-text-color' => '000000',
					'width'              => 1920,
					'height'             => 720,
					'flex-height'        => true,
					'wp-head-callback'   => 'it_zone_header_style',
					'header-text'        => true, // show the checkbox to display/hide site ttile and tagline
				)
			)
		);

		/**
		 * Set the content width in pixels, based on the theme's design and stylesheet.
		 *
		 * Priority 0 to make it available to lower priority callbacks.
		 *
		 * @global int $content_width
		 */
		$GLOBALS['content_width'] = apply_filters( 'it_zone_content_width', 800 );
	}
endif;
add_action( 'after_setup_theme', 'it_zone_setup' );

if ( ! function_exists( 'it_zone_scripts' ) ) :
	/**
	 * Enqueue scripts and styles.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_scripts() {
		// Use minified libraries if SCRIPT_DEBUG is false
		$build  = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? 'unminify/' : '';
		$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		wp_enqueue_style( 'it-zone-google-fonts', it_zone_google_fonts_url(), array(), null );
		wp_style_add_data( 'it-zone-style', 'rtl', 'replace' );
		wp_enqueue_style( 'it-zone-styles', get_template_directory_uri() . '/assets/css/' . $build . 'style' . $suffix . '.css', array(), IT_ZONE_VERSION, false );

		wp_enqueue_style( 'it-zone-style', get_stylesheet_uri(), array(), IT_ZONE_VERSION );

		if ( it_zone_woo_boolean() ) {
			wp_enqueue_style( 'it-zone-woo', get_template_directory_uri() . '/assets/css/' . $build . 'woo' . $suffix . '.css', array(), IT_ZONE_VERSION );
		}
		wp_enqueue_script( 'it-zone-navigation', get_template_directory_uri() . '/js/' . $build . 'navigation' . $suffix . '.js', array( 'jquery' ), IT_ZONE_VERSION, true );

		// Enqueue the custom script first
		wp_enqueue_script(
			'it-zone-custom',
			get_template_directory_uri() . '/assets/js/' . $build . 'custom' . $suffix . '.js',
			array( 'jquery' ),
			IT_ZONE_VERSION,
			true // Load in the footer
		);

		wp_enqueue_script(
			'it-zone-modal-accessibility', 
			get_template_directory_uri() . '/js/' . $build . 'modal-accessibility' . $suffix . '.js', 
			array('jquery'), 
			IT_ZONE_VERSION,
			true
		);

		wp_localize_script(
			'it-zone-custom',
			'it_zone_data',
			array(
				'rtl' => is_rtl(),
			)
		);

		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}
endif;
add_action( 'wp_enqueue_scripts', 'it_zone_scripts' );

if ( ! function_exists( 'it_zone_enqueue_backend_styles' ) ) :
	/**
	 * Enqueuing styles and scripts for Backend.
	 *
	 * @since 1.0.0
	 * @param string $hook The current admin page hook.
	 * @return void
	 */
	function it_zone_enqueue_backend_styles( $hook ) {
		// Use minified libraries if SCRIPT_DEBUG is false
		$build  = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? 'unminify/' : '';
		$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		wp_enqueue_style(
			'it-zone-editor-styles',
			get_template_directory_uri() . '/assets/css/' . $build . 'editor-style' . $suffix . '.css',
			array(),
			IT_ZONE_VERSION,
			false
		);

		if ( $hook == "themes.php") {
			wp_enqueue_script(
				'it-zone-admin-notice-scripts',
				get_template_directory_uri() . '/assets/js/unminify/admin-notice.js',
				array('jquery'),
				IT_ZONE_VERSION,
				true
			);
			wp_enqueue_style(
				'it-zone-admin-notice-style',
				get_template_directory_uri() . '/assets/css/unminify/admin-notice.css',
				array(),
				IT_ZONE_VERSION,
				false
			);
			wp_localize_script(
				'it-zone-admin-notice-scripts',
				'itZoneAdminData',
				array(
					'ajax_url' => admin_url('admin-ajax.php'),
					'nonce'    => wp_create_nonce('dismiss_notice_nonce'),
				)
			);
		}
	}
endif;
add_action( 'admin_enqueue_scripts', 'it_zone_enqueue_backend_styles' );

if ( ! function_exists( 'it_zone_body_classes' ) ) :
	/**
	 * Adds custom classes to the array of body classes.
	 *
	 * @since 1.0.0
	 * @param array $classes Classes for the body element.
	 * @return array Modified array of body classes.
	 */
	function it_zone_body_classes( $classes ) {
		// Adds a class of hfeed to non-singular pages.
		if ( ! is_singular() ) {
			$classes[] = 'hfeed';
		}
		// Adds a class of gl-full-wrap when there is no sidebar present.
		if ( ! is_active_sidebar( 'primary-sidebar' ) ) {
			$classes[] = 'gl-full-wrap';
		}
		if ( is_singular() || is_archive() || is_search() || is_home() ) {
			$classes[] = it_zone_sidebar_layout();
		}

		return $classes;
	}
endif;
add_filter( 'body_class', 'it_zone_body_classes' );


if ( ! function_exists( 'it_zone_widgets_init' ) ) :
	/**
	 * Register widget area.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_widgets_init() {
		$sidebars = array(
			'primary-sidebar' => array(
				'name'        => esc_html__( 'Primary Sidebar', 'it-zone' ),
				'id'          => 'primary-sidebar',
				'description' => esc_html__( 'Primary Sidebar', 'it-zone' ),
			),
			'footer-one'      => array(
				'name'        => esc_html__( 'Footer One', 'it-zone' ),
				'id'          => 'footer-one',
				'description' => esc_html__( 'Add footer icons widgets here.', 'it-zone' ),
			),
			'footer-two'      => array(
				'name'        => esc_html__( 'Footer Two', 'it-zone' ),
				'id'          => 'footer-two',
				'description' => esc_html__( 'Add footer two widgets here.', 'it-zone' ),
			),
			'footer-three'    => array(
				'name'        => esc_html__( 'Footer Three', 'it-zone' ),
				'id'          => 'footer-three',
				'description' => esc_html__( 'Add footer three widgets here.', 'it-zone' ),
			),
			'footer-four'     => array(
				'name'        => esc_html__( 'Footer Four', 'it-zone' ),
				'id'          => 'footer-four',
				'description' => esc_html__( 'Add footer four widgets here.', 'it-zone' ),
			),
		);
		foreach ( $sidebars as $sidebar ) {
			register_sidebar(
				array(
					'name'          => esc_html( $sidebar['name'] ),
					'id'            => esc_attr( $sidebar['id'] ),
					'description'   => esc_html( $sidebar['description'] ),
					'before_widget' => '<section id="%1$s" class="widget %2$s">',
					'after_widget'  => '</section>',
					'before_title'  => '<h5 class="widget-title" itemprop="name">',
					'after_title'   => '</h5>',
				)
			);
		}
	}
endif;
add_action( 'widgets_init', 'it_zone_widgets_init' );

if ( ! function_exists( 'it_zone_archive_title_prefix' ) ) {
	/**
	 * Function to modify the archive title.
	 *
	 * @since 1.0.0
	 * @param string $title The archive title.
	 * @return string Modified archive title.
	 */
	function it_zone_archive_title_prefix( $title ) {
		$archive_prefix = get_theme_mod( 'itz_archive_prefix', true );
		if ( is_post_type_archive( 'product' ) ) {
			$title = get_the_title( get_option( 'woocommerce_shop_page_id' ) );
		} elseif ( $archive_prefix ) {
			if ( is_category() ) {
				$title = single_cat_title( '', false );
			} elseif ( is_tag() ) {
				$title = single_tag_title( '', false );
			} elseif ( is_author() ) {
				$author = get_queried_object();
				$title = '<span class="vcard">' . esc_html( $author->display_name ) . '</span>';
			} elseif ( is_year() ) {
				$title = get_the_date( __( 'Y', 'it-zone' ) );
			} elseif ( is_month() ) {
				$title = get_the_date( __( 'F Y', 'it-zone' ) );
			} elseif ( is_day() ) {
				$title = get_the_date( __( 'F j, Y', 'it-zone' ) );
			} elseif ( is_post_type_archive() ) {
				$title = post_type_archive_title( '', false );
			} elseif ( is_tax() ) {
				$tax   = get_taxonomy( get_queried_object()->taxonomy );
				$title = single_term_title( '', false );
			}
		}
		return $title;
	}
}
add_filter( 'get_the_archive_title', 'it_zone_archive_title_prefix' );

if ( ! function_exists( 'it_zone_pingback_header' ) ) :
	/**
	 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_pingback_header() {
		if ( is_singular() && pings_open() ) {
			printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
		}
	}
endif;
add_action( 'wp_head', 'it_zone_pingback_header' );

if ( ! function_exists( 'wp_body_open' ) ) :
	/**
	 * Shim for sites older than 5.2.
	 *
	 * @link https://core.trac.wordpress.org/ticket/12563
	 * @since 1.0.0
	 * @return void
	 */
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
endif;

if ( ! function_exists( 'it_zone_primary_nagivation' ) ) :
	/**
	 * Primary Navigation.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_primary_nagivation() {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'primary' ) ) {
			echo '<nav class="main-navigation"><div id="site-navigation" class="menu-container">';
			wp_nav_menu(
				array(
					'theme_location' => 'primary',
					'container'      => 'ul',
					'menu_id'        => 'primary-menu',
					'menu_class'     => 'menu nav-menu',
					'fallback_cb'    => 'it_zone_navigation_menu_fallback',
				)
			);
			echo '</div></nav>';
		}
	}
endif;

if ( ! function_exists( 'it_zone_secondary_nagivation' ) ) :
	/**
	 * Secondary Navigation.
	 *
	 * @since 1.0.0
	 * @param string $classname CSS class name for the menu.
	 * @return void
	 */
	function it_zone_secondary_nagivation( $classname = 'menu' ) {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'secondary' ) ) { ?>
			<nav class="main-navigation secondary-menu">
				<?php
					wp_nav_menu(
						array(
							'theme_location' => 'secondary',
							'container'      => 'ul',
							'menu_class'     => $classname,
							'fallback_cb'    => 'it_zone_navigation_menu_fallback',
						)
					);
				?>
			</nav>
			<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_tertiary_nagivation' ) ) :
	/**
	 * Tertiary Navigation.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_tertiary_nagivation() {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'tertiary' ) ) {
			?>
			<nav class="navigation tertiary-menu">
				<?php
					wp_nav_menu(
						array(
							'theme_location' => 'tertiary',
							'container'      => 'ul',
							'menu_class'     => 'tertiary-menu',
							'fallback_cb'    => 'it_zone_navigation_menu_fallback',
						)
					);
				?>
			</nav>
			<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_footer_navigation' ) ) :
	/**
	 * Footer Navigation Menu.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_footer_navigation() {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'footer-menu' ) ) {
			?>
			<div class="footer-bottom-menu">
				<div class="menu-footer-container">
					<?php
						wp_nav_menu(
							array(
								'theme_location' => 'footer-menu',
								'container'      => 'ul',
								'menu_class'     => 'footer-bottom-links',
								'fallback_cb'    => 'it_zone_navigation_menu_fallback',
							)
						);
					?>
				</div>
			</div>
			<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_apply_theme_shortcode' ) ) :
	/**
	 * Apply theme shortcodes to a string.
	 *
	 * @since 1.0.0
	 * @param string $string The string to process.
	 * @return string Processed string with shortcodes replaced.
	 */
	function it_zone_apply_theme_shortcode( $string ) {
		if ( empty( $string ) ) {
			return $string;
		}
		$search  = array( '[the-year]', '[the-site-link]' );
		$replace = array(
			date_i18n( esc_html__( 'Y', 'it-zone' ) ),
			'<a href="' . esc_url( home_url( '/' ) ) . '">' . esc_html( get_bloginfo( 'name', 'display' ) ) . '</a>',
		);
		$string  = str_replace( $search, $replace, $string );
		return $string;
	}
endif;

if ( ! function_exists( 'it_zone_pagination' ) ) :
	/**
	 * Pagination function.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_pagination() {
		if ( is_singular() ) {
			// Display wp_link_pages for paginated content within posts/pages
			wp_link_pages(
				array(
					'before'      => '<div class="page-links"><span class="page-links-title">' . __( 'Pages:', 'it-zone' ) . '</span>',
					'after'       => '</div>',
					'link_before' => '<span class="page-number">',
					'link_after'  => '</span>',
					'separator'   => '',
				)
			);

			$post_type = get_post_type();
			$next_post = get_next_post( array( 'post_type' => $post_type ) );
			$prev_post = get_previous_post( array( 'post_type' => $post_type ) );

			if ( $next_post || $prev_post ) {
				?>
				<nav class="post-navigation">
					<div class="nav-links">
						<?php if ( $prev_post ) { ?>
							<div class="nav-previous">
								<a href="<?php the_permalink( $prev_post->ID ); ?>" rel="prev">
									<div class="pagination-details">
										<span class="meta-nav"><?php echo esc_html__( 'Previous Post', 'it-zone' ); ?></span>
										<header class="entry-header">
											<h4 class="entry-title"><?php echo esc_html( get_the_title( $prev_post->ID ) ); ?> </h4>
										</header>
									</div>
								</a>
							</div>
							<?php
						}
						if ( $next_post ) {
							?>
							<div class="nav-next">
								<a href="<?php the_permalink( $next_post->ID ); ?>" rel="next">
									<article class="post">
										<div class="pagination-details">
											<span class="meta-nav"><?php echo esc_html__( 'Next Post', 'it-zone' ); ?></span>
											<header class="entry-header">
												<h4 class="entry-title"><?php echo esc_html( get_the_title( $next_post->ID ) ); ?> </h4>
											</header>
										</div>
									</article>
								</a>
							</div>
						<?php } ?>
					</div>	
				</nav>
				<?php
			}
		} else {
			$pagination = get_theme_mod( 'pagination_type', 'numbered' );

			switch ( $pagination ) {
				case 'default': // Default Pagination
					the_posts_navigation();

					break;

				case 'numbered': // Numbered Pagination
					echo '<div class="numbered">';
					the_posts_pagination(
						array(
							'prev_text'          => __( 'Previous', 'it-zone' ),
							'next_text'          => __( 'Next', 'it-zone' ),
							'before_page_number' => '<span class="meta-nav screen-reader-text">' . __( 'Page', 'it-zone' ) . ' </span>',
						)
					);
					echo '</div>';

					break;

				default:
					the_posts_navigation();
					break;
			}
		}
	}
endif;

if ( ! function_exists( 'it_zone_comment' ) ) :
	/**
	 * Display comments template.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_comment() {
		// If comments are open or we have at least one comment, load up the comment template.
		if ( comments_open() || get_comments_number() ) {
			comments_template();
		}
	}
endif;

if ( ! function_exists( 'it_zone_navigation_menu_fallback' ) ) :
	/**
	 * Navigation menu fallback container.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_navigation_menu_fallback() {
		if ( current_user_can( 'manage_options' ) ) {
			?>
			<ul class="menu">
				<li>
					<a href="<?php echo esc_url( admin_url( 'nav-menus.php' ) ); ?>"><?php echo esc_html__( 'Click here to add a menu', 'it-zone' ); ?></a>
				</li>
			</ul>
			<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_get_comment_count' ) ) :
	/**
	 * Display comment count with title.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_get_comment_count() {
		$comment_count = get_comments_number();
		$post_title    = get_the_title();
		echo '<h2 class="comments-title">';
		printf(
			_nx( '%1$s Comment on "%2$s"', '%1$s Comments on "%2$s"', $comment_count, 'comments count', 'it-zone' ),
			number_format_i18n( $comment_count ),
			$post_title
		);
		echo '</h2>';
	}
endif;

if ( ! function_exists( 'it_zone_move_comment_field_to_bottom' ) ) :
	/**
	 * Move comment field from the current position to before the cookies field.
	 *
	 * @since 1.0.0
	 * @param array $fields Comment form fields.
	 * @return array Modified comment form fields.
	 */
	function it_zone_move_comment_field_to_bottom( $fields ) {
		if ( isset( $fields['comment'] ) || isset( $fields['cookies'] ) ) {
			$comment_field = ( isset( $fields['comment'] ) && ! empty( $fields['comment'] ) ) ? $fields['comment'] : '';
			$cookies       = ( isset( $fields['cookies'] ) && ! empty( $fields['cookies'] ) ) ? $fields['cookies'] : '';

			unset( $fields['comment'] );
			unset( $fields['cookies'] );

			// Append the comment field before the cookies field
			if ( $comment_field ) {
				$fields['comment'] = $comment_field;
			}
			if ( $cookies ) {
				$fields['cookies'] = $cookies;
			}
		}
		return $fields;
	}
	add_filter( 'comment_form_fields', 'it_zone_move_comment_field_to_bottom' );
endif;

if ( ! function_exists( 'it_zone_admin_notice' ) ) :
	/**
	 * Show the Admin Notice in the dashboard.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_admin_notice(){
		$currentScreen = get_current_screen();

		if ($currentScreen->id === "themes") {

			// check whether user has already dismissed the notice
			$dismissed = get_option('itz_dismissed_custom_admin_notice');

			if (true === (bool) $dismissed) return;
		?>
			<div id="gl_admin_notice" class="notice notice-info is-dismissible gl-custom-admin-notice">
				<div class="col-left">
					<h2><?php esc_html_e('Build Your Dream Site with GLThemes', 'it-zone'); ?></h2>
					<p><?php esc_html_e('Revolutionize your website with GLThemes, elegant and incredibly fast WordPress themes to help you augment your business to the next level.', 'it-zone'); ?></p>
					<ul>
						<li><?php esc_html_e('Theme Installation Service', 'it-zone'); ?></li>
						<li><?php esc_html_e('WordPress Maintenance Service', 'it-zone'); ?></li>
						<li><?php esc_html_e('SEO Services', 'it-zone'); ?></li>
						<li><?php esc_html_e('WordPress Web Hosting', 'it-zone'); ?></li>
						<li><?php esc_html_e('Theme Customization Service', 'it-zone'); ?></li>
						<li><?php esc_html_e('Hire A WordPress Developer', 'it-zone'); ?></li>
					</ul>
					<a target="_blank" rel="nofollow noreferrer" class="button button-primary" href="<?php echo esc_url("https://glthemes.com"); ?>"><?php esc_html_e('Learn More', 'it-zone'); ?>
					</a>
				</div>
				<div class="col-right">
					<img src="<?php echo esc_url(get_template_directory_uri() . '/assets/images/admin-notice.png'); ?>">
				</div>
			</div>
		<?php 	
		}
	}
add_action('admin_notices', 'it_zone_admin_notice');
endif;

if ( ! function_exists( 'it_zone_dismiss_admin_notice' ) ) :
	/**
	 * Function to permanently dismiss the admin notice once clicked.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_dismiss_admin_notice(){
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'dismiss_notice_nonce' ) ) {
			wp_die( __( 'Invalid nonce. Access denied.', 'it-zone' ) );
		}
		update_option('itz_dismissed_custom_admin_notice', true);

		wp_send_json_success();
	}
	add_action('wp_ajax_itz_dismiss_admin_notice', 'it_zone_dismiss_admin_notice');
endif;