<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package IT_Zone
 */

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
if ( ! function_exists( 'it_zone_doctype' ) ) :
	/**
	 * Display DOCTYPE declaration.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_doctype() { ?>
		<!DOCTYPE html>
		<html <?php language_attributes(); ?>>
	<?php }
endif;
add_action( 'it_zone_doctype', 'it_zone_doctype' );

if ( ! function_exists( 'it_zone_head' ) ) :
	/**
	 * Display head meta tags.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_head() {
		?>
		<meta charset="<?php bloginfo( 'charset' ); ?>">
		<meta name="viewport" content="width=device-width, initial-scale=1">
		<link rel="profile" href="https://gmpg.org/xfn/11">
	<?php }
endif;
add_action( 'it_zone_before_wp_head', 'it_zone_head' );

if ( ! function_exists( 'it_zone_page_start' ) ) :
	/**
	 * Display page start wrapper.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_page_start() {
		?>
		<div id="page" class="site">
			<a class="skip-link screen-reader-text" href="#gl-container"><?php esc_html_e( 'Skip to content', 'it-zone' ); ?></a>
	<?php }
endif;
add_action( 'it_zone_before_header', 'it_zone_page_start', 20 );

if ( ! function_exists( 'it_zone_header_inclusion' ) ) :
	/**
	 * Display header markup.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_header_inclusion() {
		?>
		<header class="site-header style-one" itemscope itemtype="http://schema.org/WPHeader">
			<div class="desktop-header">
				<?php it_zone_top_header(); ?>
				<?php it_zone_bottom_header(); ?>
			</div>
			<?php it_zone_mobile_header(); ?>
		</header>
		<span id="overlay"></span>
		<?php
	}
endif;
add_action( 'it_zone_header', 'it_zone_header_inclusion', 10 );

if ( ! function_exists( 'it_zone_background_header' ) ) :
	/**
	 * Display breadcrumb section with background.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_background_header() {
		$breadcrumb_toggle = get_theme_mod( 'breadcrumb_toggle', true );

		if ( ! is_front_page() && $breadcrumb_toggle ) {
			$breadcrumb_image    = get_theme_mod( 'breadcrumb_image' );
			$breadcrumb_bg_color = get_theme_mod( 'breadcrumb_bg_color', '#0F5299' );
			$background_style    = $breadcrumb_bg_color;

			if ( $breadcrumb_image ) {
				$background_style = 'background-image: url(' . esc_url( wp_get_attachment_image_url( $breadcrumb_image, 'full' ) ) . ')';
				$class            = 'background-image-wrapper';
			} else {
				$background_style = 'background: ' . $breadcrumb_bg_color . ';';
				$class            = '';
			}
			?>
			<!-- breadcrumb start -->
			<div class="breadcrumb-wrapper <?php echo esc_attr( $class ); ?>" style="<?php echo esc_attr( $background_style ); ?>">
				<div class="container">
					<div class="breadcrumb-text"> 
						<?php it_zone_breadcrumbs(); ?>
						<header class="entry-header">
							<?php it_zone_header_title(); ?>	
						</header>
					</div>
				</div>
			</div>
			<!-- breadcrumb end -->
		<?php
		}
	}
endif;
add_action( 'it_zone_after_header', 'it_zone_background_header', 10 );

if ( ! function_exists( 'it_zone_entry_header' ) ) :
	/**
	 * Display entry header with meta information.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_entry_header() {
		if ( is_page() ) {
			return;
		}
		if ( is_single() ) {
			$ed_single_posted_date       = get_theme_mod( 'ed_single_posted_date', true );
			$ed_single_posted_author     = get_theme_mod( 'ed_single_posted_author', true );
			$ed_single_meta_reading_time = get_theme_mod( 'ed_single_meta_reading_time', true );

			if ( $ed_single_posted_author || $ed_single_posted_date || $ed_single_meta_reading_time ) : ?>
				<div class="entry-meta">
					<?php if ( $ed_single_posted_author || $ed_single_posted_date || $ed_single_meta_reading_time ) : ?>
						<div class="header-post-meta">
							<?php if ( $ed_single_posted_author ) {
								it_zone_posted_by();
							} ?>
							<?php if ( $ed_single_posted_date || $ed_single_meta_reading_time ) : ?>
								<div class="header-meta-time">
									<?php if ( $ed_single_posted_date ) {
										it_zone_posted_on();
									} ?>
									<?php if ( $ed_single_meta_reading_time ) : ?>
										<span class="reading-time">
											<?php it_zone_get_post_reading_time_info( get_the_ID() ); ?>
										</span>
									<?php endif; ?>
								</div>
							<?php endif; ?>
						</div>
					<?php endif; ?>
				</div>
			<?php endif;
		} else {
			$ed_archive_post_category = get_theme_mod( 'ed_archive_post_category', true );
			?>
			<div class="blog__info">
				<?php if ( $ed_archive_post_category ) : ?>
					<?php it_zone_category(); ?>
				<?php endif; ?>
				<h4>
					<a href="<?php the_permalink(); ?>" title="<?php the_title_attribute(); ?>">
						<?php the_title(); ?>
					</a>
				</h4>
			</div>
		<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_post_thumbnail' ) ) :
	/**
	 * Display post thumbnail.
	 *
	 * Wraps the post thumbnail in an anchor element on index views, or a div
	 * element when on single views.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_post_thumbnail() {
		if ( ! is_singular() ) {
			$ed_posted_date = get_theme_mod( 'ed_archive_posted_date', true );
			echo '<figure class="blog__img"><a href="' . esc_url( get_permalink() ) . '" class="post-thumbnail">';
			if ( has_post_thumbnail() ) {
				the_post_thumbnail( 'blog_card_image' );
			} else {
				it_zone_get_fallback_svg( 'blog_card_image' );
			}
			echo '</a>';
			if ( $ed_posted_date ) {
				it_zone_posted_on();
			}
			echo '</figure>';
		} else {
			if ( ! has_post_thumbnail() ) {
				return;
			}
			?>
			<div class="post-frontmatter">
				<div class="post-thumbnail">
					<?php
					$thumbnail_id = get_post_thumbnail_id();
					// Get all the images metas
					$caption = wp_get_attachment_caption( $thumbnail_id );

					echo '<div class=img-date-wrap>';
					the_post_thumbnail( 'full', );
					echo '</div>';
					if ( $caption ) {
						echo '<figcaption>' . esc_html( $caption ) . '</figcaption>';
					}
					?>
				</div>
			</div>
		<?php
		}
	}
endif;

/**
 * For the single post, entry header is first and then post thumbnail.
 * While for the blog card like archive post, post thumbnail is first and then entry header.
 */
if ( ! function_exists( 'it_zone_display_post_layout' ) ) :
	/**
	 * Display post layout based on context.
	 *
	 * For single posts: Header first, then Thumbnail.
	 * For archive posts: Thumbnail first, then Header.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_display_post_layout() {
		if ( is_single() ) {
			// On single post: Header first, then Thumbnail
			it_zone_entry_header();
			it_zone_post_thumbnail();
		} else {
			// Default Card Layout: Thumbnail first, then Header
			it_zone_post_thumbnail();
			it_zone_entry_header();
		}
	}
endif;

add_action( 'it_zone_before_post_entry_content', 'it_zone_display_post_layout' );

if ( ! function_exists( 'it_zone_entry_content_wrapper_start' ) ) :
	/**
	 * Start post content wrapper.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_entry_content_wrapper_start() {
		if ( is_singular() ) {
			echo '<div class="content-wrap">';
		}
	}
endif;
add_action( 'it_zone_before_post_entry_content', 'it_zone_entry_content_wrapper_start', 30 );

if ( ! function_exists( 'it_zone_entry_content' ) ) :
	/**
	 * Display entry content.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_entry_content() {
		if ( is_singular() ) {
			echo '<div class="entry-content" itemprop="text">';
			the_content();
			echo '</div>';
		}
	}
endif;
add_action( 'it_zone_post_entry_content', 'it_zone_entry_content', 40 );

if ( ! function_exists( 'it_zone_entry_footer' ) ) :
	/**
	 * Display entry footer with edit link.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_entry_footer() {
		edit_post_link(
			sprintf(
				wp_kses(
					/* translators: %s: Name of current post. Only visible to screen readers */
					__( 'Edit <span class="screen-reader-text">%s</span>', 'it-zone' ),
					array(
						'span' => array(
							'class' => array(),
						),
					)
				),
				wp_kses_post( get_the_title() )
			),
			'<span class="edit-link">',
			'</span>'
		);
	}
endif;
add_action( 'it_zone_post_entry_content', 'it_zone_entry_footer', 50 );

if ( ! function_exists( 'it_zone_entry_content_wrapper_end' ) ) :
	/**
	 * End post content wrapper.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_entry_content_wrapper_end() {
		if ( is_singular() ) {
			echo '</div>';
		} // End content-wrap
	}
endif;
add_action( 'it_zone_entry_content_wrapper_end', 'it_zone_entry_content_wrapper_end', 60 );

if ( ! function_exists( 'it_zone_content_wrapper_start' ) ) :
	/**
	 * Start content wrapper.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_content_wrapper_start() {
		?>
		<div class="content-area" id="primary">
			<div class="container">
				<?php if ( is_author() ) {
					it_zone_author_box();
				} if ( is_404() ) {
					return false;
				}
				if ( is_search() && have_posts() ) {
					$ed_search_block         = get_theme_mod( 'ed_search_block', true );
					$search_heading_text     = get_theme_mod( 'search_heading_text', __( 'New Search:', 'it-zone' ) );
					$search_description_text = get_theme_mod( 'search_description_text', __( 'Please conduct another search if you are not satisfied with the outcomes below.', 'it-zone' ) );

					if ( $search_heading_text ) { ?>
						<h4 class="search-heading">
							<?php echo esc_html( $search_heading_text ); ?>
						</h4>
					<?php } ?>
					<?php if ( $ed_search_block ) { ?>
						<section class="widget widget_block widget_search">
							<?php echo get_search_form(); ?>
						</section>
					<?php } ?>
					<?php if ( $search_description_text ) { ?>
						<p class="description">
							<?php echo wp_kses_post( $search_description_text ); ?>
						</p>
					<?php } ?>
				<?php } ?>
				<div class="page-grid">
					<div class="site-main" id="main">
						<?php
						if ( is_author() ) {
							echo '<h2 class="arthur-collection-title">' . esc_html__( 'Collection of Author\'s Best Work', 'it-zone' ) . '</h2>';
						}
						if ( ! is_singular() ) {
							echo '<div class="grid-layout">';
						}
	}
endif;
add_action( 'it_zone_before_posts_content', 'it_zone_content_wrapper_start' );

if ( ! function_exists( 'it_zone_single_entry_footer_sections' ) ) :
	/**
	 * Display single entry footer sections.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_single_entry_footer_sections() {
		$post_type = get_post_type( get_the_ID() );
		$mp_author = get_theme_mod( 'mp_author', true );

		if ( is_singular( 'post' ) ) {
			if ( $mp_author ) {
				it_zone_post_author_box();
			}
			it_zone_post_footer_meta();
			it_zone_pagination();
			it_zone_comment();
			it_zone_related_posts( $post_type, 'blog_card_image' );
		}
	}
endif;
add_action( 'it_zone_after_posts_content', 'it_zone_single_entry_footer_sections', 5 );

if ( ! function_exists( 'it_zone_content_wrapper_end' ) ) :
	/**
	 * End content wrapper.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_content_wrapper_end() {
		if ( ! is_singular() ) {
			echo '</div>';
		} //End grid-layout
		if ( is_archive() || is_home() ) {
			it_zone_pagination();
		}
		?>
						</div> <!-- End main -->
					<?php
					if ( ! is_404() && ! is_search() ) {
						get_sidebar();
					}
					?>
				</div> <!-- End page-grid -->
			</div> <!-- End container -->
		</div> <!-- End content area -->
	<?php }
endif;
add_action( 'it_zone_after_posts_content', 'it_zone_content_wrapper_end', 10 );

if ( ! function_exists( 'it_zone_footer_start' ) ) :
	/**
	 * Start footer markup.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_footer_start() {
		?>
		</div><!-- End gl-container -->
		<footer id="colophon" class="site-footer" itemscope itemtype="http://schema.org/WPFooter">
		<?php
	}
endif;
add_action( 'it_zone_footer', 'it_zone_footer_start', 20 );

if ( ! function_exists( 'it_zone_footer_main' ) ) :
	/**
	 * Display footer main section with widgets.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_footer_main() {
		$toggle_newsletter = get_theme_mod( 'toggle_footer_newsletter', false );
		$footer_one        = 'footer-one';
		$footer_two        = 'footer-two';
		$footer_three      = 'footer-three';
		$footer_four       = 'footer-four';

		if ( $toggle_newsletter || is_active_sidebar( $footer_one ) || is_active_sidebar( $footer_four ) || is_active_sidebar( $footer_two ) || is_active_sidebar( $footer_three ) ) {
			?>
			<div class="main-footer">
				<div class="container">
					<div class="main-footer__wrapper">
						<?php if ( $toggle_newsletter || is_active_sidebar( $footer_one ) ) : ?>
							<div class="footer-left">
								<?php if ( $toggle_newsletter ) {
									it_zone_footer_newsletter();
								} if ( is_active_sidebar( $footer_one ) ) { ?>
									<div class="store-icon">
										<?php dynamic_sidebar( 'footer-one' ); ?>
									</div>
								<?php } ?>                      
							</div>
						<?php endif; ?>
						<?php if ( is_active_sidebar( $footer_four ) || is_active_sidebar( $footer_two ) || is_active_sidebar( $footer_three ) ) : ?>
							<div class="footer-right">
								<?php if ( is_active_sidebar( $footer_two ) || is_active_sidebar( $footer_three ) ) : ?>
									<div class="footer-links">
										<?php if ( is_active_sidebar( $footer_two ) ) { ?>
											<div class="footer-group">
												<?php dynamic_sidebar( 'footer-two' ); ?>
											</div>
										<?php } if ( is_active_sidebar( $footer_three ) ) { ?>
											<div class="footer-group">
												<?php dynamic_sidebar( 'footer-three' ); ?>
											</div>
										<?php } ?>                             
									</div>
								<?php endif; ?>
								<?php if ( is_active_sidebar( $footer_four ) ) { ?>
									<div class="footer-group">
										<?php dynamic_sidebar( 'footer-four' ); ?>
									</div>
								<?php } ?>    
							</div>
						<?php endif; ?>
					</div>
				</div>
			</div>
		<?php
		}
	}
endif;
add_action( 'it_zone_footer', 'it_zone_footer_main', 30 );

if ( ! function_exists( 'it_zone_footer_bottom' ) ) :
	/**
	 * Footer Bottom has footer left and right
	 * -Left: Calling functions footer_copyright, toggle_author_link and toggle_wp_link
	 * -Right: Calling functions footer_navigation and privacy_policy_link
	 *
	 * Contains copyright, author link, WordPress link, navigation, and privacy policy.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_footer_bottom() {
		if ( ( current_user_can( 'manage_options' ) || has_nav_menu( 'footer-menu' ) ) || function_exists( 'the_privacy_policy_link' ) ) {
			?>
			<div class="footer-bottom">
				<div class="container">
					<div class="footer-bottom__wrapper">
						<div class="footer-bottom__left">
							<?php it_zone_footer_copyright(); ?>
							<?php it_zone_toggle_author_link(); ?>
							<?php it_zone_toggle_wp_link(); ?>
						</div>
						<?php if ( ( current_user_can( 'manage_options' ) || has_nav_menu( 'footer-menu' ) ) || function_exists( 'the_privacy_policy_link' ) ) { ?>			
							<div class="footer-bottom__right">
								<?php if ( current_user_can( 'manage_options' ) || has_nav_menu( 'footer-menu' ) ) {
									it_zone_footer_navigation();
								} ?>
								<?php if ( function_exists( 'the_privacy_policy_link' ) ) {
									the_privacy_policy_link();
								} ?>
							</div>
						<?php } ?>
					</div>
				</div>
			</div>
		<?php
		}
	}
endif;
add_action( 'it_zone_footer', 'it_zone_footer_bottom', 40 );

if ( ! function_exists( 'it_zone_footer_end' ) ) :
	/**
	 * End footer markup.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_footer_end() {
		?>
		</footer>
		<?php
	}
endif;
add_action( 'it_zone_footer', 'it_zone_footer_end', 50 );

if ( ! function_exists( 'it_zone_page_end' ) ) :
	/**
	 * End page markup.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_page_end() {
		?>
			<span id="overlay"></span>
		</div><!-- #page -->
		<?php
	}
endif;
add_action( 'it_zone_after_footer', 'it_zone_page_end', 10 );

if ( ! function_exists( 'it_zone_posted_on' ) ) :
	/**
	 * Display post date/time information.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_posted_on() {
		$time_string = '<time class="entry-date published updated" datetime="%3$s" itemprop="dateModified">%4$s</time>';

		$time_string = sprintf(
			$time_string,
			esc_attr( get_the_date( 'j M, Y' ) ),
			esc_html( get_the_date( 'j M, Y' ) ),
			esc_attr( get_the_modified_date( 'j M, Y' ) ),
			esc_html( get_the_modified_date( 'j M, Y' ) )
		);

		$posted_on = $time_string;
		echo '<span class="blog-date">' . wp_kses_post( $posted_on ) . '</span>'; // WPCS: XSS OK.
	}
endif;

if ( ! function_exists( 'it_zone_posted_by' ) ) :
	/**
	 * Display post author information.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_posted_by() {
		$ed_posted_author = get_theme_mod( 'ed_single_posted_author', true );

		// Get author ID and user data.
		$author_id    = get_the_author_meta( 'ID' );
		$first_name   = get_the_author_meta( 'first_name', $author_id );
		$display_name = get_the_author_meta( 'display_name', $author_id );

		// Prefer first name, fallback to display name.
		$author_name = ! empty( $first_name ) ? $first_name : $display_name;

		if ( $ed_posted_author ) {
			echo '<span class="author-details">';
			$byline = sprintf(
				/* translators: %s: post author. */
				esc_html_x( 'By %s', 'post author', 'it-zone' ),
				'<span><a href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( $author_name ) . '</a></span>'
			);
			echo '<span class="byline"> ' . wp_kses_post( $byline ) . '</span>';
			echo '</span>';
		}
	}
endif;

if ( ! function_exists( 'it_zone_category' ) ) :
	/**
	 * Display post categories.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_category() {
		$categories = get_the_category( get_the_ID() );
		if ( is_array( $categories ) && ! empty( $categories ) ) {
			?>
			<div class="blog__top">
				<?php foreach ( $categories as $category ) {
					$categories_url = get_category_link( $category->term_id );
					?>
					<span class="category-list">
						<a href="<?php echo esc_url( $categories_url ); ?>">
							<?php echo esc_html( $category->name ); ?>
						</a>
					</span>
				<?php } ?>
			</div>
		<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_author_box' ) ) :
	/**
	 * Display author information box.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_author_box() {
		$author_id          = get_the_author_meta( 'ID' );
		$author_description = get_the_author_meta( 'description' );
		$first_name         = get_the_author_meta( 'first_name', $author_id );
		$author_avatar      = get_avatar( $author_id, 135 );
		$noofposts          = count_user_posts( $author_id );

		$author_name = get_the_author_meta( 'display_name', $author_id );

		// Determine the correct text for the number of blogs published
		$blogs_published = sprintf(
			_nx(
				'%s Post',
				'%s Posts',
				$noofposts,
				'post count',
				'it-zone'
			),
			number_format_i18n( $noofposts )
		);

		// Only display the author section if there is a description
		if ( $author_description ) {
			?>
			<div class="author-section">
				<div class="author-wrapper">
					<div class="author-top">
						<?php if ( $author_avatar ) { ?>
							<figure class="author-img">
								<?php echo wp_kses_post( $author_avatar ); ?>
							</figure>
						<?php } ?>
						<div class="author-wrap">
							<?php if ( $author_name ) { ?>
								<div class="author-meta">
									<h5 class="author-name">
										<?php echo esc_html( $author_name ); ?>
									</h5>
									<span class="author-count">(<?php echo esc_html( $blogs_published ); ?>)</span>
								</div>
							<?php } ?>
						</div>
					</div>
					<div class="author-details">
						<div class="author-content">
							<p><?php echo wp_kses_post( $author_description ); ?></p>
						</div>
					</div>
				</div>
			</div>
		<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_post_author_box' ) ) :
	/**
	 * Display post author box.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_post_author_box() {
		$author_id          = get_the_author_meta( 'ID' );
		$author_description = get_the_author_meta( 'description' );
		$author_avatar      = get_avatar( $author_id, 135 );

		$author_name = get_the_author_meta( 'display_name', $author_id );

		// Only display the author section if there is a description
		if ( $author_description ) {
			?>
			<div class="author-section">
				<div class="author-wrapper">
					<?php if ( $author_avatar ) { ?>
						<div class="author-top">
							<figure class="author-img">
								<?php echo wp_kses_post( $author_avatar ); ?>
							</figure>
						</div>
					<?php } ?>
					<div class="author-details">
						<?php if ( $author_name ) { ?>
							<div class="author-wrap">
								<div class="author-meta">
									<h5 class="author-name">
										<a href="<?php echo esc_url( get_author_posts_url( $author_id ) ); ?>" class="url" rel="author">
											<?php echo esc_html( $author_name ); ?>
										</a>
									</h5>
								</div>
							</div>
						<?php } ?>
						<div class="author-content">
							<p><?php echo wp_kses_post( $author_description ); ?></p>
						</div>
					</div>
				</div>
			</div>
		<?php
		}
	}
endif;

if ( ! function_exists( 'it_zone_post_footer_meta' ) ) :
	/**
	 * Display post footer meta information.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_post_footer_meta() {
		?>
		<footer class="entry-footer">
			<?php it_zone_tags(); ?>
		</footer>
		<?php
	}
endif;

if ( ! function_exists( 'it_zone_footer_newsletter' ) ) :
	/**
	 * Display footer newsletter section.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function it_zone_footer_newsletter() {
		$toggle_newsletter     = get_theme_mod( 'toggle_footer_newsletter', false );
		$footer_subheading     = get_theme_mod( 'footer_subheading', __( 'Newsletter', 'it-zone' ) );
		$footer_heading        = get_theme_mod( 'footer_heading', __( 'Subscribe Us & Get Weekly Notifications', 'it-zone' ) );
		$footer_form_shortcode = get_theme_mod( 'footer_contactform_shortcode' );

		if ( $toggle_newsletter && ( $footer_subheading || $footer_heading || $footer_form_shortcode ) ) { ?>
			<?php if ( $footer_subheading ) { ?>
				<span class="footer-left-heading"><?php echo esc_html( $footer_subheading ); ?></span>
			<?php } if ( $footer_heading ) { ?>
				<h2><?php echo esc_html( $footer_heading ); ?></h2>
			<?php } if ( $footer_form_shortcode ) { ?>
				<?php echo do_shortcode( $footer_form_shortcode ); ?>
			<?php
			}
		}
	}
endif;