/**
 * Custom JavaScript for IT Zone Theme
 *
 * Handles mobile navigation, search functionality, notifications, and other interactive elements.
 *
 * @package IT_Zone
 * @since 1.0.0
 */

(function ($) {
	'use strict';

	// Constants
	const CONSTANTS = {
		OPACITY: {
			VISIBLE: 1,
			HIDDEN: 0
		},
		VISIBILITY: {
			VISIBLE: 'visible',
			HIDDEN: 'hidden'
		},
		POSITION: {
			OPEN: '0',
			CLOSED: '-100%'
		},
		HEIGHT: {
			CLOSED: '0px'
		}
	};

	/**
	 * Initialize mobile header functionality
	 */
	function initMobileHeader() {
		const sideBar = document.getElementById('mobileSideMenu');
		const hambutton = document.getElementById('sideMenuOpener');
		const overlay = document.getElementById('overlay');
		const closeButton = document.querySelector('.close-sidebar-btn');

		if (!sideBar || !hambutton || !overlay || !closeButton) {
			return;
		}

		/**
		 * Show mobile sidebar
		 */
		function showSidebar() {
			overlay.style.opacity = CONSTANTS.OPACITY.VISIBLE;
			overlay.style.visibility = CONSTANTS.VISIBILITY.VISIBLE;
			sideBar.style.opacity = CONSTANTS.OPACITY.VISIBLE;
			sideBar.style.visibility = CONSTANTS.VISIBILITY.VISIBLE;
			sideBar.style.left = CONSTANTS.POSITION.OPEN;

			sideBar.classList.add('is-open');
			// Disable body scrolling
			document.body.style.overflow = 'hidden';
		}

		/**
		 * Hide mobile sidebar
		 */
		function hideSidebar() {
			overlay.style.opacity = CONSTANTS.OPACITY.HIDDEN;
			overlay.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
			sideBar.style.opacity = CONSTANTS.OPACITY.HIDDEN;
			sideBar.style.visibility = CONSTANTS.VISIBILITY.HIDDEN;
			sideBar.style.left = CONSTANTS.POSITION.CLOSED;

			sideBar.classList.remove('is-open');
			// Re-enable body scrolling
			document.body.style.overflow = '';
		}

		hambutton.addEventListener('click', showSidebar);
		overlay.addEventListener('click', hideSidebar);
		closeButton.addEventListener('click', hideSidebar);
	}

	/**
	 * Initialize mobile navigation menu functionality
	 */
	function initMobileNavigation() {
		const parentMenus = document.querySelectorAll('#mobileSideMenu .main-navigation .menu-item.menu-item-has-children');

		parentMenus.forEach(function (menu) {
			const aTag = menu.querySelector('a');
			const subMenu = menu.querySelector('.sub-menu');

			if (!aTag || !subMenu) {
				return;
			}

			aTag.addEventListener('click', function (e) {
				e.preventDefault();
				aTag.classList.toggle('open');

				let totalHeight = 0;
				const allSubMenus = menu.querySelectorAll('.sub-menu');

				allSubMenus.forEach(function (ul) {
					totalHeight += ul.scrollHeight;
				});

				if (!subMenu.style.maxHeight || subMenu.style.maxHeight === CONSTANTS.HEIGHT.CLOSED || subMenu.style.maxHeight === '0') {
					subMenu.style.maxHeight = totalHeight + 'px';
				} else {
					subMenu.style.maxHeight = '0';
				}
			});
		});
	}

	function subMenuAddButton() {
		// Mobile Side Menu 
		const mobileMenu = document.querySelector('#mobileSideMenu .nav-menu');
		if (mobileMenu) {
			const menuItems = mobileMenu.querySelectorAll('.menu-item.menu-item-has-children');
			menuItems.forEach((item) => {
				const button = document.createElement('button');
				button.classList.add('angle-down');
				button.textContent = 'Angle down';
				const submenu = item.querySelector('.sub-menu');
				item.insertBefore(button, submenu)
			});
		}
	}

	/**
	 * Initialize submenu open close
	 */
	function subMenuOpenClose() {
		const submenuOpener = document.querySelectorAll('.angle-down');
		if (submenuOpener) {
			submenuOpener.forEach(opener => {
				opener.addEventListener('click', () => handleOpeningSubmenu(opener));
			});
		}
	}
	/**
	 * Handle opening submenu
	 * @param {HTMLElement} opener - The element that opens the submenu
	 */
	const handleOpeningSubmenu = (opener) => {
		let totalHeight = 0; // Initialize totalHeight to 0
		const submenu = opener.nextSibling;
		totalHeight = submenu.scrollHeight
		const innerSubMenu = submenu.querySelectorAll('ul');
		innerSubMenu.forEach(menu => {
			totalHeight += menu.scrollHeight;
		})
		if (submenu.style.visibility === "visible") {
			submenu.style.visibility = "hidden";
			submenu.style.maxHeight = "0";
		} else {
			submenu.style.visibility = "visible";
			submenu.style.opacity = "1";
			submenu.style.maxHeight = totalHeight + "px";
		}
		opener.classList.toggle('active')
		totalHeight = 0;
	}

	/**
	 * Initialize all functionality when DOM is ready
	 */
	function init() {
		initMobileHeader();
		initMobileNavigation();
		subMenuAddButton();
		subMenuOpenClose();
	}

	// Initialize when DOM is ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', init);
	} else {
		init();
	}

})(jQuery);