<?php
/**
 * Imagined functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Imagined
 */

if ( ! function_exists( 'imagined_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function imagined_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Imagined, use a find and replace
	 * to change 'imagined' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'imagined', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_editor_style();
	add_theme_support( 'post-thumbnails' );
	add_image_size( 'imagined-featured-thumbnail',  990, 450, true );
	add_image_size( 'imagined-widget-post-thumb',  80, 80, true );
	add_theme_support( 'custom-logo', array(
		'height'      => 100,
		'width'       => 400,
		'flex-height' => true,
		'flex-width'  => true,
		'header-text' => array( 'site-title', 'site-description' ),
	) );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary', 'imagined' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'comment-list',
		'gallery',
		'caption',
	) );


	// Set up the WordPress core custom background feature.
	add_theme_support( 'custom-background', apply_filters( 'imagined_custom_background_args', array(
		'default-color' => 'ffffff',
		'default-image' => '',
	) ) );
}
endif;
add_action( 'after_setup_theme', 'imagined_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function imagined_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'imagined_content_width', 640 );
}
add_action( 'after_setup_theme', 'imagined_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function imagined_widgets_init() {
	
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'imagined' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here.', 'imagined' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );
	
	register_sidebar( array(
		'name' => __( 'Footer One', 'imagined' ),
		'id' => 'imagined-footer-one-widget',
		'before_widget' => '<div id="footer-one" class="widget footer-widget">',
		'after_widget' => "</div>",
		'before_title' => '<h1 class="widget-title">',
		'after_title' => '</h1>',
	) );
	
	register_sidebar( array(
		'name' => __( 'Footer Two', 'imagined' ),
		'id' => 'imagined-footer-two-widget',
		'before_widget' => '<div id="footer-two" class="widget footer-widget">',
		'after_widget' => "</div>",
		'before_title' => '<h1 class="widget-title">',
		'after_title' => '</h1>',
	) );
	
	register_sidebar( array(
		'name' => __( 'Footer Three', 'imagined' ),
		'id' => 'imagined-footer-three-widget',
		'before_widget' => '<div id="footer-three" class="widget footer-widget">',
		'after_widget' => "</div>",
		'before_title' => '<h1 class="widget-title">',
		'after_title' => '</h1>',
	) );
	
}
add_action( 'widgets_init', 'imagined_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function imagined_scripts() {
	wp_enqueue_style( 'bootstrap', get_template_directory_uri() .'/assets/css/bootstrap.css' );
	wp_enqueue_style( 'font-awesome', get_template_directory_uri() .'/assets/font-awesome/css/font-awesome.min.css' );
	wp_enqueue_style( 'prettyPhoto', get_template_directory_uri() .'/assets/css/prettyPhoto.css' );
	wp_enqueue_style( 'imagined-google-fonts', '//fonts.googleapis.com/css?family=Playfair+Display:400,400italic,700,700italic,900,900italic|Lato:400,900,700,400italic,700italic,300');
	wp_enqueue_style( 'imagined-ie', get_stylesheet_directory_uri() . "/assets/css/ie.css", array()  );
    wp_style_add_data( 'imagined-ie', 'conditional', 'IE' );
	wp_enqueue_style( 'imagined-style', get_stylesheet_uri() );
	
	wp_enqueue_script( 'bootstrap', get_template_directory_uri() . '/js/bootstrap.js', array('jquery') );
	wp_enqueue_script( 'prettyPhoto', get_template_directory_uri() . '/js/jquery.prettyPhoto.min.js', array('jquery'));
	wp_enqueue_script( 'imagined-custom-js', get_template_directory_uri() . '/js/custom.js', array('jquery-masonry', 'imagesloaded') );
	wp_enqueue_script( 'imagined-ie-responsive-js', get_template_directory_uri() . '/js/ie-responsive.min.js', array() );
	wp_script_add_data( 'imagined-ie-responsive-js', 'conditional', 'lt IE 9' );
	wp_enqueue_script( 'imagined-ie-shiv', get_template_directory_uri() . "/js/html5shiv.min.js");
	wp_script_add_data( 'imagined-ie-shiv', 'conditional', 'lt IE 9' );

	wp_enqueue_script( 'imagined-navigation', get_template_directory_uri() . '/js/navigation.js', array(), '20151215', true );
	wp_enqueue_script( 'imagined-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20151215', true );
	
	$imagined_slider_speed = 6000;
	if ( get_theme_mod( 'imagined_slider_speed_setting' ) ) {
		$imagined_slider_speed = esc_attr( get_theme_mod( 'imagined_slider_speed_setting' ) ) ;
	}
	wp_localize_script( "imagined-custom-js", "imagined_slider_speed", array('vars' => $imagined_slider_speed) );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'imagined_scripts' );

/**
 * Limit post excerpt
 */
function imagined_custom_excerpt_length( $length ) {
return 30;
}
add_filter( 'excerpt_length', 'imagined_custom_excerpt_length', 999 );

/**
 * Implement the Custom Widget.
 */
require get_template_directory() . '/inc/widget.php';
/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';
/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
require get_template_directory() . '/inc/jetpack.php';
