<?php
/**
 * Contains various functions
 *
 * @package Illuminate
 */

// Set content width
if ( ! isset( $content_width ) ) : 
	$content_width = 640;
endif;

// Sets up theme defaults and registers support for various WordPress features
if ( ! function_exists( 'illuminate_setup' ) ) :

function illuminate_setup() {
	// Make theme available for translation. Translations should be put into the /languages/ directory.
	load_theme_textdomain( 'illuminate', get_template_directory() . '/languages' );
	
	// Add RSS feeds automatically
	add_theme_support( 'automatic-feed-links' );
	
	// Enable featured images/post thumbnails
	add_theme_support( 'post-thumbnails' );
	
	// This theme only uses one menu
	register_nav_menus( array(
		'primary' => __( 'Primary Menu', 'illuminate' )
	) );
	
	// Enable support for various post formats
	add_theme_support( 'post-formats', array( 'aside', 'image', 'video', 'quote', 'link' ) );	
}
endif;
add_action( 'after_setup_theme', 'illuminate_setup' );

// Register widget areas
function illuminate_widgets_init() {
	register_sidebar( array(
		'name'          => __( 'Primary', 'illuminate' ),
		'description'   => __( 'This sidebar appears in the right-hand column.', 'illuminate' ),
		'id'            => 'primary',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>'
	) );
}
add_action( 'widgets_init', 'illuminate_widgets_init' );

// Enqueue various stylesheets and scripts
function illuminate_scripts() {
	// Call in Foundation CSS
	wp_enqueue_style( 'illuminate-foundation', get_template_directory_uri() . '/inc/foundation.css' );
	
	// Call in Genericons icons
	wp_enqueue_style( 'illuminate-genericons', get_template_directory_uri() . '/inc/genericons/genericons.css' );
	
	// Call in main stylesheet. This comes after the Foundation CSS and Genericons so that styles will cascade correctly
	wp_enqueue_style( 'illuminate-style', get_stylesheet_uri() );
	
	// Call in Google Web Fonts if desired
	// Fill in the names of the desired fonts after "css?", with |'s between font names and replacing spaces with +'s
	// wp_enqueue_style( 'illuminate-google-fonts', 'http://fonts.googleapis.com/css?' );
	
	// Call in Modernizr JS
	wp_enqueue_script( 'illuminate-modernizr', get_template_directory_uri() . '/js/modernizr.custom.94162.js', array( 'jquery' ), null, false );
	
	// Call in Foundation JS
	wp_enqueue_script( 'illuminate-foundation-js', get_template_directory_uri() . '/js/foundation.min.js', array( 'jquery' ), null, true );
	
	wp_enqueue_script( 'illuminate-navigation', get_template_directory_uri() . '/js/navigation.js', array(), '20120206', true );

	wp_enqueue_script( 'illuminate-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20130115', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'illuminate_scripts' );

// Implement custom editor styling
function illuminate_add_editor_style() {
	add_editor_style( 'inc/editor-style.css' );
}
add_action( 'init', 'illuminate_add_editor_style' );

// Call in custom template tags
require get_template_directory() . '/inc/template-tags.php';

// Filters wp_title() so something useful happens
function illuminate_wp_title( $title, $sep ) {
	global $page, $paged;

	if ( is_feed() ) {
		return $title;
	}

	// Add the blog name
	$title .= get_bloginfo( 'name' );

	// Add the blog description for the home/front page.
	$site_description = get_bloginfo( 'description', 'display' );
	if ( $site_description && ( is_home() || is_front_page() ) ) {
		$title .= " $sep $site_description";
	}

	// Add a page number if necessary:
	if ( $paged >= 2 || $page >= 2 ) {
		$title .= " $sep " . sprintf( __( 'Page %s', 'illuminate' ), max( $paged, $page ) );
	}

	return $title;
}
add_filter( 'wp_title', 'illuminate_wp_title', 10, 2 );

// Strip out all HTML tags except for ones that are explicitly allowed
function illuminate_strip_tags( $commentdata ) {
	$commentdata[ 'comment_content' ] = wp_kses_data( $commentdata[ 'comment_content' ] );
	
	return $commentdata;	
}
add_filter( 'preprocess_comment', 'illuminate_strip_tags', 10, 1 );

// Turn the comment reply link into a Foundation button
function illuminate_comment_reply_foundation_button( $string ) {
	return str_replace( 'comment-reply-link', 'button radius small comment-reply-link', $string );
}
add_filter( 'comment_reply_link', 'illuminate_comment_reply_foundation_button', 10 );

// Turn the password submit button into a Foundation button
function illuminate_password_foundation_button( $string ) {
	return str_replace( 'type="submit"', 'input class="button radius small" type="submit"', $string );
}
add_filter( 'the_password_form', 'illuminate_password_foundation_button', 10 );

// Add author's first name to comment reply link
// Thanks to Raam Dev
// http://raamdev.com/2013/personalizing-the-wordpress-comment-reply-link/
function illuminate_comment_author_to_reply_link( $link, $args, $comment ) {
	$comment = get_comment( $comment );
 
	// If no comment author is blank, use 'Anonymous'
	if ( empty( $comment->comment_author ) ) :
        	if ( ! empty( $comment->user_id ) ) :
			$user = get_userdata( $comment->user_id );
			$author = $user->user_login;
		else :
			$author = __( 'Anonymous', 'illuminate' );
		endif;
    else :
        $author = $comment->comment_author;
    endif;
 
    // If the user provided more than a first name, use only first name
    if ( strpos( $author, ' ' ) ) :
        $author = substr( $author, 0, strpos( $author, ' ' ) );
    endif;
 
    // Replace Reply Link with "Reply to (Author First Name)"
    $reply_link_text = $args[ 'reply_text' ];
    $link = str_replace( $reply_link_text, 'Reply to ' . $author, $link );
 
    return $link;
}
add_filter( 'comment_reply_link', 'illuminate_comment_author_to_reply_link', 10, 3 );

// Display the pagination links a bit differently
function illuminate_foundation_pagination( $args ) {
	$defaults = array(
		'before'         => '<ul class="pagination"><li class="arrow unavailable">&laquo;</li>',
		'after'          => '<li class="arrow unavailable">&raquo;</ul>',
		'next_or_number' => 'number',
		'pagelink'       => '%',
		'text_before'    => ' ',
		'text_after'     => ' '
	);
	
	$output = '';
	global $page, $numpages, $multipage, $more, $pagenow;
	
	if ( $multipage ) :
		if ( 'number' == $defaults[ 'next_or_number' ] ) :
			$output .= $defaults[ 'before' ] ;
			for ( $idx = 1; $idx < ( $numpages + 1 ); $idx = $idx + 1 ) {
				if ( is_single() && ( ( $idx == $page) || ( ( ! $more ) && ( $page == 1 ) ) ) ) :
					$output .= '<li class="current">' . _wp_link_page( $idx ) . $idx . '</a></li>';
				else :
					$output .= '<li>' . _wp_link_page( $idx ) . $idx . '</a></li>';
				endif;
			}
		endif;
		
		$output .= $defaults[ 'after' ];
	endif;
	
	return $output;
}
add_filter( 'wp_link_pages', 'illuminate_foundation_pagination', 10 );