<?php

/**
 * Functions which enhance the theme by hooking into WordPress core actions/hooks
 *
 * @package Home Experts
 */
if ( ! function_exists( 'home_experts_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function home_experts_setup() {
		/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on home_experts, use a find and replace
		* to change 'home-experts' to the name of your theme in all the template files.
		*/
		load_theme_textdomain( 'home-experts', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
		add_theme_support( 'title-tag' );

		/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
		add_theme_support( 'post-thumbnails' );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus(
			array(
				'primary'     => esc_html__( 'Primary', 'home-experts' ),
				'secondary'   => esc_html__( 'Secondary', 'home-experts' ),
				'tertiary'    => esc_html__( 'Tertiary', 'home-experts' ),
				'footer-menu' => esc_html__( 'Footer Menu', 'home-experts' ),
			)
		);

		/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
		add_theme_support(
			'html5',
			array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'style',
				'script',
			)
		);

		// Set up the WordPress core custom background feature.
		add_theme_support(
			'custom-background',
			apply_filters(
				'home_experts_custom_background_args',
				array(
					'default-color' => 'ffffff',
					'default-image' => '',
				)
			)
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );
		add_theme_support( 'align-wide' );
		add_theme_support( 'appearance-tools' );
		add_theme_support( 'border' );
		add_theme_support( 'custom-spacing' );
		add_theme_support( 'custom-line-height' );
		add_theme_support( 'link-color' );
		add_theme_support( 'responsive-embeds' );
		add_theme_support( 'editor-styles' );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support(
			'custom-logo',
			array(
				'height'      => 250,
				'width'       => 250,
				'flex-width'  => true,
				'flex-height' => true,
			)
		);
		/**
		 * Registering new image sizes
		 */
		add_image_size( 'blog_card_image', 384, 480, true );

		// Remove core block patterns
		remove_theme_support( 'core-block-patterns' );
		/**
		 * Set up the WordPress core custom header feature.
		 *
		 * @uses home_experts_header_style()
		 */
		add_theme_support(
			'custom-header',
			apply_filters(
				'home_experts_custom_header_args',
				array(
					'default-image'      => esc_url( get_template_directory_uri() . '/assets/images/static-banner.jpg' ),
					'video'              => true,
					'default-text-color' => '000000',
					'width'              => 1920,
					'height'             => 720,
					'flex-height'        => true,
					'wp-head-callback'   => 'home_experts_header_style',
					'header-text'        => true, // show the checkbox to display/hide site ttile and tagline
				)
			)
		);

		/**
		 * Set the content width in pixels, based on the theme's design and stylesheet.
		 *
		 * Priority 0 to make it available to lower priority callbacks.
		 *
		 * @global int $content_width
		 */
		$GLOBALS['content_width'] = apply_filters( 'home_experts_content_width', 800 );
	}
endif;
add_action( 'after_setup_theme', 'home_experts_setup' );

if ( ! function_exists( 'home_experts_scripts' ) ) :
	/**
	 * Enqueue scripts and styles.
	 */
	function home_experts_scripts() {
		// Use minified libraries if SCRIPT_DEBUG is false
		$build  = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? 'unminify/' : '';
		$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		wp_style_add_data( 'home-experts-style', 'rtl', 'replace' );

		wp_enqueue_style( 'home-experts-google-fonts', home_experts_google_fonts_url(), array(), null );
				
		wp_enqueue_style( 'home-experts-styles', get_template_directory_uri() . '/assets/css/' . $build . 'style' . $suffix . '.css', array(), HOME_EXPERTS_VERSION, false );

		wp_enqueue_style( 'home-experts-style', get_stylesheet_uri(), array(), HOME_EXPERTS_VERSION );

		if ( home_experts_woo_boolean() ) {
			wp_enqueue_style( 'home-experts-woo', get_template_directory_uri() . '/assets/css/' . $build . 'woo' . $suffix . '.css', array(), HOME_EXPERTS_VERSION );
		}
		
		// Only load navigation script if menu exists
		if ( has_nav_menu( 'primary' ) || current_user_can( 'manage_options' ) ) {
			wp_enqueue_script( 'home-experts-navigation', get_template_directory_uri() . '/js/' . $build . 'navigation' . $suffix . '.js', array( 'jquery' ), HOME_EXPERTS_VERSION, true );
		}

		// Enqueue the custom script first
		wp_enqueue_script(
			'home-experts-custom',
			get_template_directory_uri() . '/assets/js/' . $build . 'custom' . $suffix . '.js',
			array( 'jquery' ),
			HOME_EXPERTS_VERSION,
			true // Load in the footer
		);
		
		wp_enqueue_script(
			'home-experts-modal-accessibility', 
			get_template_directory_uri() . '/assets/js/' . $build . 'modal-accessibility' . $suffix . '.js', 
			array('jquery'), 
			HOME_EXPERTS_VERSION,
			true
		);

		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}
endif;
add_action( 'wp_enqueue_scripts', 'home_experts_scripts' );

if ( ! function_exists( 'home_experts_enqueue_backend_styles' ) ) :
	/**
	 * Enqueuing styles and scripts for Backend
	 *
	 * @param string $hook The current admin page.
	 * @return void
	 */
	function home_experts_enqueue_backend_styles( $hook ) {
		// Use minified libraries if SCRIPT_DEBUG is false
		$build  = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? 'unminify/' : '';
		$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		wp_enqueue_style( 
			'home-experts-editor-styles', 
			get_template_directory_uri() . '/assets/css/' . $build . 'editor-style' . $suffix . '.css', 
			array(), 
			HOME_EXPERTS_VERSION, 
			false 
		);

		if ( $hook == "themes.php") {
			wp_enqueue_script(
				'home-experts-admin-scripts',
				get_template_directory_uri() . '/assets/js/' . $build . 'admin-notice' . $suffix . '.js',
				array( 'jquery' ),
				HOME_EXPERTS_VERSION,
				true
			);
			// This contain the static path to the icons, so this file is not minified.
			wp_enqueue_style(
				'home-experts-admin-notice-style',
				get_template_directory_uri() . '/assets/css/unminify/admin-notice.css',
				array(),
				HOME_EXPERTS_VERSION,
				false
			);
			wp_localize_script(
				'home-experts-admin-scripts',
				'home_experts_admin_data',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'nonce'    => wp_create_nonce( 'dismiss_notice_nonce' ),
				)
			);
		}
	}
endif;
add_action( 'admin_enqueue_scripts', 'home_experts_enqueue_backend_styles' );

if ( ! function_exists( 'home_experts_body_classes' ) ) :
	/**
	 * Adds custom classes to the array of body classes.
	 *
	 * @param  array $classes Classes for the body element.
	 * @return array
	 */
	function home_experts_body_classes( $classes ) {
		// Adds a class of hfeed to non-singular pages.
		if ( ! is_singular() ) {
			$classes[] = 'hfeed';
		}
		// Adds a class of gl-full-wrap when there is no sidebar present.
		if ( ! is_active_sidebar( 'primary-sidebar' ) ) {
			$classes[] = 'gl-full-wrap';
		}
		if ( is_singular() || is_archive() || is_search() || is_home() ) {
			$classes[] = home_experts_sidebar_layout();
		}

		return $classes;
	}
endif;
add_filter( 'body_class', 'home_experts_body_classes' );


if ( ! function_exists( 'home_experts_widgets_init' ) ) :
	/**
	 * Register widget area.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
	 */
	function home_experts_widgets_init() {
		$sidebars = array(
			'primary-sidebar' => array(
				'name'        => esc_html__( 'Primary Sidebar', 'home-experts' ),
				'id'          => 'primary-sidebar',
				'description' => esc_html__( 'Primary Sidebar', 'home-experts' ),
			),
			'footer-one'      => array(
				'name'        => esc_html__( 'Footer One', 'home-experts' ),
				'id'          => 'footer-one',
				'description' => esc_html__( 'Add footer icons widgets here.', 'home-experts' ),
			),
			'footer-two'      => array(
				'name'        => esc_html__( 'Footer Two', 'home-experts' ),
				'id'          => 'footer-two',
				'description' => esc_html__( 'Add footer two widgets here.', 'home-experts' ),
			),
			'footer-three'    => array(
				'name'        => esc_html__( 'Footer Three', 'home-experts' ),
				'id'          => 'footer-three',
				'description' => esc_html__( 'Add footer three widgets here.', 'home-experts' ),
			),
			'footer-four'     => array(
				'name'        => esc_html__( 'Footer Four', 'home-experts' ),
				'id'          => 'footer-four',
				'description' => esc_html__( 'Add footer four widgets here.', 'home-experts' ),
			),
			'footer-five'     => array(
				'name'        => esc_html__( 'Footer Five', 'home-experts' ),
				'id'          => 'footer-five',
				'description' => esc_html__( 'Add footer five widgets here.', 'home-experts' ),
			),
		);
		foreach ( $sidebars as $sidebar ) {
			register_sidebar(
				array(
					'name'          => esc_html( $sidebar['name'] ),
					'id'            => esc_attr( $sidebar['id'] ),
					'description'   => esc_html( $sidebar['description'] ),
					'before_widget' => '<section id="%1$s" class="widget %2$s">',
					'after_widget'  => '</section>',
					'before_title'  => '<h5 class="widget-title" itemprop="name">',
					'after_title'   => '</h5>',
				)
			);
		}
	}
endif;
add_action( 'widgets_init', 'home_experts_widgets_init' );

function home_experts_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
	}
}
add_action( 'wp_head', 'home_experts_pingback_header' );

if ( ! function_exists( 'wp_body_open' ) ) :
	/**
	 * Shim for sites older than 5.2.
	 *
	 * @link https://core.trac.wordpress.org/ticket/12563
	 */
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
endif;

if ( ! function_exists( 'home_experts_primary_navigation' ) ) :
	/**
	 * Primary Navigation.
	 */
	function home_experts_primary_navigation() {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'primary' ) ) {
			echo '<nav class="main-navigation"><div class="menu-container">';
			wp_nav_menu(
				array(
					'theme_location' => 'primary',
					'container'      => 'ul',
					'menu_id'        => 'primary-menu',
					'menu_class'     => 'menu',
					'fallback_cb'    => 'home_experts_navigation_menu_fallback',
				)
			);
			echo '</div></nav>';
		}
	}
endif;

if ( ! function_exists( 'home_experts_secondary_navigation' ) ) :
	/**
	 * Secondary Navigation.
	 */
	function home_experts_secondary_navigation( $classname = 'menu' ) {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'secondary' ) ) { ?>
			<nav class="main-navigation secondary-menu">
				<?php
				wp_nav_menu(
					array(
						'theme_location' => 'secondary',
						'container'      => 'ul',
						'menu_class'     => $classname,
						'fallback_cb'    => 'home_experts_navigation_menu_fallback',
					)
				);
				?>
			</nav>
			<?php
		}
	}
endif;

if ( ! function_exists( 'home_experts_tertiary_nagivation' ) ) :
	/**
	 * Secondary Navigation.
	 */
	function home_experts_tertiary_nagivation() {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'tertiary' ) ) {
			?>
			<nav class="navigation tertiary-menu">
				<?php
				wp_nav_menu(
					array(
						'theme_location' => 'tertiary',
						'container'      => 'ul',
						'menu_class'     => 'tertiary-menu',
						'fallback_cb'    => 'home_experts_navigation_menu_fallback',
					)
				);
				?>
			</nav>
			<?php
		}
	}
endif;

if ( ! function_exists( 'home_experts_footer_navigation' ) ) :
	/**
	 * Footer Navigation Menu
	 *
	 * @return void
	 */
	function home_experts_footer_navigation() {
		if ( current_user_can( 'manage_options' ) || has_nav_menu( 'footer-menu' ) ) {
			?>
			<div class="footer-bottom-menu">
				<div class="menu-footer-container">
					<?php
					wp_nav_menu(
						array(
							'theme_location' => 'footer-menu',
							'container'      => 'ul',
							'menu_class'     => 'footer-bottom-links',
							'fallback_cb'    => 'home_experts_navigation_menu_fallback',
						)
					);
					?>
				</div>
			</div>
			<?php
		}
	}
endif;

if ( ! function_exists( 'home_experts_apply_theme_shortcode' ) ) :
	/**
	 * Footer Shortcode
	 */
	function home_experts_apply_theme_shortcode( $string ) {
		if ( empty( $string ) ) {
			return $string;
		}
		$search  = array( '[the-year]', '[the-site-link]' );
		$replace = array(
			date_i18n( esc_html__( 'Y', 'home-experts' ) ),
			'<a href="' . esc_url( home_url( '/' ) ) . '">' . esc_html( get_bloginfo( 'name', 'display' ) ) . '</a>',
		);
		$string  = str_replace( $search, $replace, $string );
		return $string;
	}
endif;

if ( ! function_exists( 'home_experts_pagination' ) ) :
	/**
	 * Pagination function
	 *
	 * @return void
	 */
	function home_experts_pagination() {
		if ( is_singular() ) {
			$post_type = get_post_type();
			$next_post = get_next_post();
			$prev_post = get_previous_post();

			if ( $next_post || $prev_post ) {
				?>
				<nav class="posts-navigation">
					<div class="nav-links">
						<?php if ( $prev_post ) { ?>
							<div class="nav-previous">
								<a href="<?php the_permalink( $prev_post->ID ); ?>" rel="prev">
									<?php echo esc_html__( 'Previous Post', 'home-experts' ); ?>
								</a>
							</div>
							<?php
						}
						if ( $next_post ) {
							?>
							<div class="nav-next">
								<a href="<?php the_permalink( $next_post->ID ); ?>" rel="next">
									<?php echo esc_html__( 'Next Post', 'home-experts' ); ?>
								</a>
							</div>
						<?php } ?>
					</div>
				</nav>
				<?php
			}
		} else {
			$pagination = get_theme_mod( 'pagination_type', 'default' );
			switch ( $pagination ) {
				case 'default':
					the_posts_navigation();
					break;

				case 'numbered': // Numbered Pagination
					echo '<div class="numbered">';
						the_posts_pagination(
							array(
								'prev_text'          => __( 'Previous', 'home-experts' ),
								'next_text'          => __( 'Next', 'home-experts' ),
								'before_page_number' => '<span class="meta-nav screen-reader-text">' . __( 'Page', 'home-experts' ) . ' </span>',
							)
						);
					echo '</div>';
					break;
				default:
					the_posts_navigation();
					break;
			}
		}
	}
endif;

if ( ! function_exists( 'home_experts_numbered_pagination' ) ) :
    /**
     * Numbered Pagination for paginated posts
     */
    function home_experts_numbered_pagination() {
        // Only run on singular posts or pages
        if ( ! is_singular() ) {
            return;
        }

        global $page, $numpages, $multipage;

        // Only output pagination if the post is actually paginated
        if ( $multipage && $numpages > 1 ) : ?>
            <div class="numbered">
                <div class="page-links navigation pagination">
                    <?php
                    wp_link_pages(
                        array(
                            'before'      => '<div class="nav-links">',
                            'after'       => '</div>',
                            'link_before' => '<span class="meta-nav screen-reader-text">' . esc_html__( 'Page ', 'home-experts' ) . '</span>',
                            'link_after'  => '',
                            'pagelink'    => '%',
                            'separator'   => '',
                        )
                    );
                    ?>
                </div>
            </div>
        <?php endif;
    }
endif;

if ( ! function_exists( 'home_experts_comment' ) ) :
	/**
	 * Comment
	 */
	function home_experts_comment() {
		// If comments are open or we have at least one comment, load up the comment template.
		if ( comments_open() || get_comments_number() ) {
			comments_template();
		}
	}
endif;

if ( ! function_exists( 'home_experts_navigation_menu_fallback' ) ) :
	/**
	 * Navigation menu fallback container
	 *
	 * @return void
	 */
	function home_experts_navigation_menu_fallback() {
		if ( current_user_can( 'manage_options' ) ) {
			?>
			<ul class="menu">
				<li>
					<a href="<?php echo esc_url( admin_url( 'nav-menus.php' ) ); ?>"><?php echo esc_html__( 'Click here to add a menu', 'home-experts' ); ?></a>
				</li>
			</ul>
			<?php
		}
	}
endif;

if ( ! function_exists( 'home_experts_get_comment_count' ) ) :
	/**
	 * Comment Count
	 */
	function home_experts_get_comment_count() {
		$comment_count = number_format_i18n( get_comments_number() );
		$post_title    = get_the_title();
		echo '<h2 class="comments-title">';
		printf(
			_nx( '%1$s Comment on "%2$s"', '%1$s Comments on "%2$s"', $comment_count, 'comments count', 'home-experts' ),
			esc_html( number_format_i18n( $comment_count ) ),
			esc_html( $post_title )
		);
		echo '</h2>';
	}
endif;


if ( ! function_exists( 'home_experts_reorder_comment_fields' ) ) :
	/**
	 * Move comment filed from the current position to before the cookies field
	 *
	 * @param  array $fields
	 * @return array
	 */
	function home_experts_reorder_comment_fields( $fields ) {
		if ( isset( $fields['comment'] ) || isset( $fields['cookies'] ) ) {
			$comment_field = ( isset( $fields['comment'] ) && ! empty( $fields['comment'] ) ) ? $fields['comment'] : '';
			$cookies       = ( isset( $fields['cookies'] ) && ! empty( $fields['cookies'] ) ) ? $fields['cookies'] : '';

			unset( $fields['comment'] );
			unset( $fields['cookies'] );

			// Append the comment field before the cookies field
			if ( $comment_field ) {
				$fields['comment'] = $comment_field;
			}
			if ( $cookies ) {
				$fields['cookies'] = $cookies;
			}
		}
		return $fields;
	}
	add_filter( 'comment_form_fields', 'home_experts_reorder_comment_fields' );
endif;

if ( ! function_exists( 'home_experts_admin_notice' ) ) :
	/**
	 * Show the Admin Notice in the dashboard
	 *
	 * @return void
	 */
	function home_experts_admin_notice() {
		$currentScreen = get_current_screen();

		if ( $currentScreen->id === 'themes' ) {
			// check whether user has already dismissed the notice
			$dismissed = get_option( 'he_dismissed_gl_admin_notice' );

			if ( true === (bool) $dismissed ) {
				return;
			}
			?>
			<div id="gl_admin_notice" class="notice notice-info is-dismissible gl-custom-admin-notice">
				<div class="col-left">
					<h2><?php esc_html_e( 'Build Your Dream Site with GLThemes', 'home-experts' ); ?></h2>
					<p><?php esc_html_e( 'Revolutionize your website with GLThemes, elegant and incredibly fast WordPress themes to help you augment your business to the next level.', 'home-experts' ); ?></p>
					<ul>
						<li><?php esc_html_e( 'Theme Installation Service', 'home-experts' ); ?></li>
						<li><?php esc_html_e( 'WordPress Maintainance Service', 'home-experts' ); ?></li>
						<li><?php esc_html_e( 'SEO Services', 'home-experts' ); ?></li>
						<li><?php esc_html_e( 'WordPress Web Hosting', 'home-experts' ); ?></li>
						<li><?php esc_html_e( 'Theme Customization Service', 'home-experts' ); ?></li>
						<li><?php esc_html_e( 'Hire A WordPress Developer', 'home-experts' ); ?></li>
					</ul>
					<a target="_blank" rel="nofollow noreferrer" class="button button-primary" href="<?php echo esc_url( 'https://glthemes.com' ); ?>"><?php esc_html_e( 'Learn More', 'home-experts' ); ?>
					</a>
				</div>
				<div class="col-right">
					<img src="<?php echo esc_url( get_template_directory_uri() . '/assets/images/admin-notice.png' ); ?>" alt="<?php esc_attr_e( 'Good Looking Themes Admin Notice Image', 'home-experts' ); ?>">
				</div>
			</div>
			<?php
		}
	}
add_action('admin_notices', 'home_experts_admin_notice');
endif;

if ( ! function_exists( 'home_experts_admin_notice_dismiss' ) ) :
	/**
	 * Function to permanently dismiss the admin notice once clicked
	 *
	 * @return void
	 */
	function home_experts_admin_notice_dismiss() {
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'dismiss_notice_nonce' ) ) {
			wp_send_json_error();
		}
		update_option( 'he_dismissed_gl_admin_notice', true );

		wp_send_json_success();
	}
	add_action( 'wp_ajax_he_dismiss_admin_notice', 'home_experts_admin_notice_dismiss' );
endif;

// Enable excerpt support for pages, with post type check
function home_experts_add_excerpt_to_pages() {
	if ( post_type_exists( 'page' ) ) {
		add_post_type_support( 'page', 'excerpt' );
	}
}
add_action( 'init', 'home_experts_add_excerpt_to_pages' );


if ( ! function_exists( 'home_experts_breadcrumb_inline_styles' ) ) :
	/**
	 * Add breadcrumb inline styles to head.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	function home_experts_breadcrumb_inline_styles() {
		// Only load breadcrumb CSS if breadcrumbs are enabled and not on front page
		if ( is_front_page() ) {
			return;
		}
		
		$style = '
			.breadcrumbs .trail-browse,
			.breadcrumbs .trail-items,
			.breadcrumbs .trail-items li {
				display:     inline-block;
				margin:      0;
				padding:     0;
				border:      none;
				background:  transparent;
				text-indent: 0;
			}

			.breadcrumbs .trail-browse {
				font-size:   inherit;
				font-style:  inherit;
				font-weight: inherit;
				color:       inherit;
			}

			.breadcrumbs .trail-items {
				list-style: none;
			}

			.trail-items li::after {
				content: "\002F";
				padding: 0 0.5em;
			}

			.trail-items li:last-of-type::after {
				display: none;
			}';

		$style = apply_filters( 'breadcrumb_trail_inline_style', trim( str_replace( array( "\r", "\n", "\t", '  ' ), '', $style ) ) );

		if ( $style ) {
			printf( "\n" . '<style type="text/css" id="home-experts-breadcrumbs-css">%s</style>' . "\n", esc_html( $style ) );
		}
	}
endif;

add_action( 'wp_head', 'home_experts_breadcrumb_inline_styles' );

/**
 * Home Experts Typography Related Functions
 *
 * @package Home Experts
 */

if ( ! function_exists( 'home_experts_google_fonts_url' ) ) :
	/**
	 * Get Google Fonts URL for the theme.
	 *
	 * @since 1.0.0
	 * @return string Google Fonts URL.
	 */
	function home_experts_google_fonts_url() {
		$fonts_url = '';
		// Ignore any non-Google fonts.

		$opensans_font = _x( 'on', 'Open Sans font: on or off', 'home-experts' );

		if ( 'off' !== $opensans_font ) {
			$font_families[] = 'DM Sans:400,400i,500,500i,600,600i,700,700i';
	
			$query_args = array(
				'family'  => urlencode( implode( '|', $font_families ) ),
				'subset'  => urlencode( 'latin,latin-ext' ),
				'display' => urlencode( 'fallback' ),
			);

			$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );

			$toggle_localgoogle_fonts = get_theme_mod( 'toggle_localgoogle_fonts', true );

			if ( $toggle_localgoogle_fonts ) {
				$font_families = array(
					'DM+Sans:ital,wght@0,400;0,500;0,600;0,700;1,400;1,500;1,600;1,700',
				);

				$fonts_url = add_query_arg(
					array(
						'family'  => implode( '&family=', $font_families ),
						'display' => 'swap',
					),
					'https://fonts.googleapis.com/css2'
				);

				$fonts_url = home_experts_get_webfont_url( esc_url_raw( $fonts_url ) );
			} else {
				$fonts_url = $fonts_url;
			}
		}
		return esc_url( $fonts_url );
	}
endif;