<?php

/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package Home Experts
 */

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */

if ( ! function_exists( 'home_experts_doctype' ) ) :
	/**
	 * Doctype Declaration
	 */
	function home_experts_doctype() {
		?>
		<!DOCTYPE html>
		<html <?php language_attributes(); ?>>
		<?php
	}
endif;
add_action( 'home_experts_doctype', 'home_experts_doctype' );

if ( ! function_exists( 'home_experts_head' ) ) :
	/**
	 * Before wp_head
	 */
	function home_experts_head() {
		?>
		<meta charset="<?php bloginfo( 'charset' ); ?>">
		<meta name="viewport" content="width=device-width, initial-scale=1">
		<link rel="profile" href="https://gmpg.org/xfn/11">
		<?php
	}
endif;
add_action( 'home_experts_before_wp_head', 'home_experts_head' );

if ( ! function_exists( 'home_experts_page_start' ) ) :
	/**
	 * Page Start
	 */
	function home_experts_page_start() {
		?>
		<div id="page" class="site">
			<a class="skip-link screen-reader-text" href="#primary"><?php esc_html_e( 'Skip to content', 'home-experts' ); ?></a>
			<?php
	}
endif;
add_action( 'home_experts_before_header', 'home_experts_page_start', 20 );

if ( ! function_exists( 'home_experts_header_inclusion' ) ) :
	/**
	 * Header Function
	 */
	function home_experts_header_inclusion() {
		$toggle_header = get_theme_mod( 'topbar_toggle', false );
		?>
		<header class="site-header style-one">
			<div class="desktop-header">
				<?php if ( $toggle_header ) : ?>
					<div class="header-top">
						<div class="header-top-wrapper">
							<div class="container">
								<div class="header-social-info">
									<div class="header-location-mail-wrap">
										<?php home_experts_header_location_info(); ?>
										<?php home_experts_header_contact_info(); ?> 
									</div>
									<div class="header-phone-social-wrap">
										<?php home_experts_social_media_repeater( 'header-one-social' ); ?> 
									</div>
								</div>
							</div>  
						</div>
					</div>
				<?php endif; ?>
				<div class="header-bottom">
					<div class="container">
						<div class="header-wrapper">
							<?php home_experts_site_branding( true ); ?>
							<?php home_experts_primary_navigation(); ?>
                            <?php
                            $header_button_text = get_theme_mod( 'header_btn_text' );
                            $header_button_link = get_theme_mod( 'header_btn_link' );
                            if ( $header_button_text && $header_button_link ) : ?>
								<div class="header-bottom-right">
									<?php home_experts_front_header_get_started( 'btn-secondary' ); ?>
								</div>
                            <?php endif; ?>
						</div>
					</div>
				</div>
			</div>
			<?php home_experts_mobile_header(); ?>
		</header>
		<span id="overlay"></span>
	<?php
	}
endif;
add_action( 'home_experts_header', 'home_experts_header_inclusion', 10 );

if( ! function_exists( 'home_experts_background_header' ) ) :
    /**
     * Breadcrumbs section
     *
     * @return void
     */
    function home_experts_background_header(){
        if( ! is_front_page() ){ 
            ?>
            <!-- breadcrumb start -->
            <div class="breadcrumb-wrapper background-color-wrapper">
                <div class="container">
                    <div class="breadcrumb-text"> 
						<?php home_experts_breadcrumbs(); ?>
                        <div class="breadcrumb-header">
							<?php home_experts_header_title(); ?>	
							<?php home_experts_header_excerpt(); ?>
						</div>
                    </div>
                </div>
            </div>
            <!-- breadcrumb end -->
        <?php 
        }
    }
endif;
add_action( 'home_experts_after_header', 'home_experts_background_header', 10 );

if ( ! function_exists( 'home_experts_post_thumbnail' ) ) :
	/**
	 * Displays an optional post thumbnail.
	 *
	 * Wraps the post thumbnail in an anchor element on index views, or a div
	 * element when on single views.
	 */
	function home_experts_post_thumbnail() {
		if ( ! is_singular() ) {
			echo '<figure class="blog__img" itemscope itemtype="https://schema.org/ImageObject">';
			echo '<a href="' . esc_url( get_permalink() ) . '" class="post-thumbnail" itemprop="contentUrl">';
			if ( has_post_thumbnail() ) {
				the_post_thumbnail(
					'blog_card_image',
					array(
						'itemprop' => 'thumbnail',
						'alt'      => the_title_attribute( array( 'echo' => false ) ),
						'loading'  => 'lazy',
					)
				);
			} else {
				home_experts_get_fallback_svg( 'blog_card_image' );
			}
			echo '</a>';
			echo '</figure>';
		} else {
			if ( ! has_post_thumbnail() ) {
				return;
			}

			$thumbnail_id = get_post_thumbnail_id();
			$caption      = wp_get_attachment_caption( $thumbnail_id );

			echo '<figure class="post-thumbnail" itemscope itemtype="https://schema.org/ImageObject">';
			echo wp_get_attachment_image(
				$thumbnail_id,
				'full',
				false,
				array(
					'itemprop' => 'contentUrl',
					'alt'      => the_title_attribute( array( 'echo' => false ) ),
					'loading'  => 'lazy',
				)
			);

			if ( $caption ) {
				echo '<figcaption class="wp-caption-text" itemprop="caption">' . esc_html( $caption ) . '</figcaption>';
			}

			echo '</figure>';
		}
	}
endif;
add_action( 'home_experts_before_post_entry_content', 'home_experts_post_thumbnail', 10 );

if ( ! function_exists( 'home_experts_entry_content_wrapper_start' ) ) :
	/**
	 * Post Content wrapper Starts
	 */
	function home_experts_entry_content_wrapper_start() {
		if ( is_single() ) {
			$class = 'single-content-wrap';
		} else {
			$class = 'card-content-wrap';
		}
		echo '<div class="content-wrap ' . esc_attr( $class ) . '">';
	}
endif;
add_action( 'home_experts_before_post_entry_content', 'home_experts_entry_content_wrapper_start', 20 );

if ( ! function_exists( 'home_experts_entry_header_meta' ) ) :
	/**
	 * Meta Details
	 */
	function home_experts_entry_header_meta() {
		if( is_page() ) return;
		
		$ed_single_posted_date   = get_theme_mod( 'ed_single_posted_date', true );
		$ed_single_posted_author = get_theme_mod( 'ed_single_posted_author', true );

		if ( $ed_single_posted_author || $ed_single_posted_date ) :
			?>
				<div class="entry-meta">
					<?php if ( $ed_single_posted_author || $ed_single_posted_date ) : ?>
						<div class="header-post-meta">
							<?php if ( $ed_single_posted_date ) : ?>
								<?php home_experts_posted_on(); ?>
							<?php endif; ?>
							<?php if ( $ed_single_posted_author ) : ?>
								<?php home_experts_posted_by(); ?>
							<?php endif; ?>
						</div>
					<?php endif; ?>
				</div>
			<?php
		endif;
	}
endif;
add_action( 'home_experts_before_post_entry_content', 'home_experts_entry_header_meta', 30 );

if ( ! function_exists( 'home_experts_entry_header_title' ) ) :
	/**
	 * Meta Details with SEO Itemprop
	 */
	function home_experts_entry_header_title() {
		if( is_page() ) return;
		
		if ( is_single() ) {
			the_title( '<h2 class="entry-title" itemprop="headline">', '</h2>' );
		} else {
			the_title( '<h2 class="entry-title" itemprop="headline"><a href="' . esc_url( get_permalink() ) . '" rel="bookmark">', '</a></h2>' );
		}
	}
endif;
add_action( 'home_experts_before_post_entry_content', 'home_experts_entry_header_title', 40 );

if ( ! function_exists( 'home_experts_entry_content' ) ) :
	/**
	 * Entry Content
	 */
	function home_experts_entry_content() {
		if ( is_singular() ) {
			echo '<div class="entry-content" itemprop="text">';
				the_content();
			echo '</div>';
		} else {
			echo '<div class="card-content" itemprop="text">';
			if ( has_excerpt() ) {
				the_excerpt();
			} else {
				echo wp_kses_post( wpautop( wp_trim_words( get_the_content(), 20, __( '...', 'home-experts' ) ) ) );
			}
			echo '</div>';
		}
	}
endif;
add_action( 'home_experts_post_entry_content', 'home_experts_entry_content', 10 );

if ( ! function_exists( 'home_experts_entry_footer' ) ) :
	/**
	 * Prints HTML with meta information for the categories, tags and comments.
	 */
	function home_experts_entry_footer() {
		edit_post_link(
			sprintf(
				wp_kses(
					/* translators: %s: Name of current post. Only visible to screen readers */
					__( 'Edit <span class="screen-reader-text">%s</span>', 'home-experts' ),
					array(
						'span' => array(
							'class' => array(),
						),
					)
				),
				wp_kses_post( get_the_title() )
			),
			'<span class="edit-link">',
			'</span>'
		);
	}
endif;
add_action( 'home_experts_post_entry_content', 'home_experts_entry_footer', 20 );

if ( ! function_exists( 'home_experts_entry_content_wrapper_end' ) ) :
	/**
	 * Post Content wrapper Starts
	 */
	function home_experts_entry_content_wrapper_end() {
		home_experts_numbered_pagination();
		echo '</div>'; // End single-content-wrap/card-content-wrap
	}
endif;
add_action( 'home_experts_post_entry_content', 'home_experts_entry_content_wrapper_end', 30 );

if ( ! function_exists( 'home_experts_content_wrapper_start' ) ) :
	/**
	 * Content Wrapper
	 *
	 * @return void
	 */
	function home_experts_content_wrapper_start() {
		?>
			<div class="content-area" id="primary">
				<div class="container">
					<?php if ( is_author() ) : ?>
						<?php home_experts_author_box(); ?>
					<?php endif; ?>
					<?php if ( is_404() ) : ?>
						<?php return false; ?>
					<?php endif; ?>
					<?php
					if ( is_search() && have_posts() ) :
						$ed_search_block = get_theme_mod( 'ed_search_block', true );

						$search_query = get_search_query();
						global $wp_query;
						$total_posts = $wp_query->found_posts;
						
						$result_text = sprintf(
							_n(
								'%s result for %s',
								'%s results for %s',
								$total_posts,
								'home-experts'
							),
							number_format_i18n( $total_posts ),
							'<span class="search-term">"'. esc_html( $search_query ) . '"</span>'
						);
						?>
						<h4 class="search-heading">
							<?php echo wp_kses_post( $result_text ); ?>
						</h4>
						<p class="description">
							<?php esc_html_e( 'Please conduct another search if you are not satisfied with the outcomes below.', 'home-experts' ); ?>
						</p>
						<?php if ( $ed_search_block ) : ?>
							<section class="widget widget_block widget_search">
								<?php echo get_search_form(); ?>
							</section>
						<?php endif; ?>
					<?php endif; ?>
					<div class="page-grid">
						<div class="site-main" id="main">
							<?php if( is_author() && have_posts() ) : ?>
								<h2><?php esc_html_e( "Author's top blogs collection", "home-experts" ); ?></h2>
							<?php endif; ?>
						<?php
						if ( ! is_singular() ) :
							echo '<div class="grid-layout">';
						endif;
	}
endif;
add_action( 'home_experts_before_posts_content', 'home_experts_content_wrapper_start' );

if ( ! function_exists( 'home_experts_single_entry_footer_sections' ) ) :
	/**
	 * Entry Footer
	 */
	function home_experts_single_entry_footer_sections() {
		$post_type = get_post_type( get_the_ID() );
		$show_author_box = get_theme_mod( 'mp_author', true );

		if ( is_singular( 'post' ) ) {
			home_experts_post_footer_meta();
			if ( $show_author_box ) :
				home_experts_author_box();
			endif;
			home_experts_pagination();
			home_experts_comment();
			home_experts_related_posts( $post_type, 'blog_card_image' );
		}
	}
endif;
add_action( 'home_experts_after_posts_content', 'home_experts_single_entry_footer_sections', 5 );

if ( ! function_exists( 'home_experts_content_wrapper_end' ) ) :
	/**
	 * Content Wrapper
	 */
	function home_experts_content_wrapper_end() {
		if ( ! is_singular() ) :
			echo '</div>'; // End grid-layout
		endif;
		if ( is_archive() || is_home() || is_search() ) :
			home_experts_pagination();
		endif;
		?>
		</div> <!-- End main -->
		<?php
		if ( ! is_404() ) :
			get_sidebar();
		endif;
		?>
				</div> <!-- End page-grid -->
			</div> <!-- End container -->
		</div> <!-- End content area -->
		<?php
	}
endif;
add_action( 'home_experts_after_posts_content', 'home_experts_content_wrapper_end', 10 );

// Footer Functions
if ( ! function_exists( 'home_experts_footer_start' ) ) {
	/**
	 * Footer Start
	 */
	function home_experts_footer_start() {
		?>
		<footer id="colophon" class="site-footer" itemscope itemtype="http://schema.org/WPFooter">
		<?php
	}
}
add_action( 'home_experts_footer', 'home_experts_footer_start', 20 );

if ( ! function_exists( 'home_experts_footer_main' ) ) {
	/**
	 * Footer Middle Widgets Section
	 *
	 * @return void
	 */
	function home_experts_footer_main() {
		$toggle_newsletter = get_theme_mod( 'toggle_footer_newsletter', false );
		$footer_one        = 'footer-one';
		$footer_two        = 'footer-two';
		$footer_three      = 'footer-three';
		$footer_four       = 'footer-four';
		$footer_five       = 'footer-five';
		if ( $toggle_newsletter || is_active_sidebar( $footer_one ) || is_active_sidebar( $footer_four ) || is_active_sidebar( $footer_two ) || is_active_sidebar( $footer_three ) || is_active_sidebar( $footer_five ) ) {
			?>
			<div class="main-footer">
				<div class="container">
					<div class="main-footer__top">
						<?php if ( $toggle_newsletter ) : ?>
							<div class="main-footer__top-left">
								<?php home_experts_footer_newsletter(); ?>
							</div>
						<?php endif; ?>
						<div class="main-footer__top-right">
							<?php if ( is_active_sidebar( $footer_one ) ) : ?>
								<div class="footer-group">
									<?php dynamic_sidebar( 'footer-one' ); ?>
								</div>
							<?php endif; ?>
						</div>
					</div>
					<?php if ( is_active_sidebar( $footer_two ) || is_active_sidebar( $footer_three ) || is_active_sidebar( $footer_four ) || is_active_sidebar( $footer_five ) ) : ?>
						<div class="main-footer__wrapper">
							<div class="footer-right">
								<div class="footer-links">
									<?php if ( is_active_sidebar( $footer_two ) ) : ?>
										<div class="footer-group">
											<?php dynamic_sidebar( 'footer-two' ); ?>
										</div>
									<?php endif; ?>
									<?php if ( is_active_sidebar( $footer_three ) ) : ?>
										<div class="footer-group">
											<?php dynamic_sidebar( 'footer-three' ); ?>
										</div>
									<?php endif; ?>
									<?php if ( is_active_sidebar( $footer_four ) ) : ?>
										<div class="footer-group">
											<?php dynamic_sidebar( 'footer-four' ); ?>
										</div>
									<?php endif; ?>
									<?php if ( is_active_sidebar( $footer_five ) ) : ?>
										<div class="footer-group">
											<?php dynamic_sidebar( 'footer-five' ); ?>
										</div>
									<?php endif; ?>
								</div>
							</div>
						</div>
					<?php endif; ?>
				</div>
			</div>
			<?php
		}
	}
}
add_action( 'home_experts_footer', 'home_experts_footer_main', 30 );

if ( ! function_exists( 'home_experts_footer_bottom' ) ) {
	/**
	 * Footer Bottom has footer left and right
	 * -Left: Calling functions footer_copyright
	 * -Right: Calling functions footer_navigation and privacy_policy_link
	 *
	 * @return void
	 */
	function home_experts_footer_bottom() { ?>
		<div class="footer-bottom">
			<div class="container">
				<div class="footer-bottom__wrapper">
					<?php home_experts_footer_copyright(); ?>
					<?php home_experts_toggle_author_link(); ?>
					<?php home_experts_toggle_wp_link(); ?>
				</div>
			</div>
		</div>
		<?php
	}
}
add_action( 'home_experts_footer', 'home_experts_footer_bottom', 40 );

if ( ! function_exists( 'home_experts_footer_end' ) ) {
	/**
	 * Footer end
	 */
	function home_experts_footer_end() {
		?>
		</footer>
		<?php
	}
}
add_action( 'home_experts_footer', 'home_experts_footer_end', 50 );

if ( ! function_exists( 'home_experts_page_end' ) ) {
	/**
	 * Page End
	 *
	 * @return void
	 */
	function home_experts_page_end() {
		?>
			<span id="overlay"></span>
		</div><!-- #page -->
		<?php
	}
}
add_action( 'home_experts_after_footer', 'home_experts_page_end', 10 );

if ( ! function_exists( 'home_experts_posted_on' ) ) :
	/**
	 * Prints HTML with meta information for the current post-date/time.
	 */
	function home_experts_posted_on() {
		$time_string = '<time class="entry-date published updated" datetime="%3$s" itemprop="dateModified">%4$s</time>';

		$time_string = sprintf(
			$time_string,
			esc_attr( get_the_date( 'j M' ) ),
			esc_html( get_the_date( 'j M' ) ),
			esc_attr( get_the_modified_date( 'j M' ) ),
			esc_html( get_the_modified_date( 'j M' ) )
		);

		$posted_on = $time_string;
		echo '<span class="blog-date">' . wp_kses_post( $posted_on ) . '</span>';
	}
endif;

if ( ! function_exists( 'home_experts_get_author_name' ) ) :
	/**
	 * Get formatted author name.
	 *
	 * @param int $author_id Author ID.
	 * @return string Formatted author name.
	 */
	function home_experts_get_author_name( $author_id ) {
		$first_name   = get_the_author_meta( 'first_name', $author_id );
		$last_name    = get_the_author_meta( 'last_name', $author_id );
		$display_name = get_the_author_meta( 'display_name', $author_id );

		if ( ! empty( $first_name ) && ! empty( $last_name ) ) {
			return $first_name . ' ' . $last_name;
		} elseif ( ! empty( $first_name ) ) {
			return $first_name;
		}

		return $display_name;
	}
endif;

if ( ! function_exists( 'home_experts_posted_by' ) ) :
	/**
	 * Prints HTML with meta information for the current author.
	 */
	function home_experts_posted_by() {
		$ed_posted_author = get_theme_mod( 'ed_single_posted_author', true );

		// Get author ID and user data.
		$author_id = get_the_author_meta( 'ID' );
		$author_name = home_experts_get_author_name( $author_id );

		if ( $ed_posted_author ) {
			echo '<span class="author-details">';
			$byline = sprintf(
				/* translators: %s: post author. */
				esc_html_x( '%s', 'post author', 'home-experts' ),
				'<span><a href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( $author_name ) . '</a></span>'
			);
			echo '<span class="byline"> ' . $byline . '</span>'; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo '</span>';
		}
	}
endif;

if ( ! function_exists( 'home_experts_category' ) ) :
	/**
	 * Show categories
	 */
	function home_experts_category() {
		$categories = get_the_category( get_the_ID() );
		if ( is_array( $categories ) && ! empty( $categories ) ) {
			?>
			<div class="blog__top">
			<?php
			foreach ( $categories as $category ) {
				$categories_url = get_category_link( $category->term_id );
				?>
					<span class="category-list">
						<a href="<?php echo esc_url( $categories_url ); ?>">
									<?php echo esc_html( $category->name ); ?>
						</a>
					</span>
				<?php } ?>
			</div>
			<?php
		}
	}
endif;

if ( ! function_exists( 'home_experts_author_box' ) ) :
	/**
	 * Author Section
	 */
	function home_experts_author_box() {
		$author_id = get_the_author_meta( 'ID' );
		
		// Get all author data in one call for better performance
		$author_data = get_userdata( $author_id );
		if ( ! $author_data ) {
			return;
		}
		
		$author_description = isset($author_data->description) ? $author_data->description : '';
		if ( empty( $author_description ) ) {
			return;
		}
		
		$author_name   = home_experts_get_author_name( $author_id );
		$author_avatar = get_avatar( $author_id, 150 );
		
		// Cache post count with transient
		$post_count = count_user_posts( $author_id );
		?>
		<div class="author-section">
			<div class="author-wrapper">
				<div class="author-top">
					<?php if ( $author_avatar ) : ?>
						<figure class="author-img">
								<?php echo wp_kses_post( $author_avatar ); ?>
						</figure>
					<?php endif; ?>
				</div>
				<div class="author-details">
					<?php if ( $author_name ) : ?>
						<div class="author-wrap">
							<div class="author-meta">
								<h5 class="author-name">
									<a href="<?php echo esc_url( get_author_posts_url( $author_id ) ); ?>" class="url" rel="author">
										<?php echo esc_html( $author_name ); ?>
									</a>
								</h5>
								<?php
								if ( $post_count > 0 ) {
									$post_count_text = sprintf(
										_n(
											'(%s blog published)',
											'(%s blogs published)',
											$post_count,
											'home-experts'
										),
										number_format_i18n( $post_count )
									);
									echo '<span class="author-post-count">' . esc_html( $post_count_text ) . '</span>';
								}
								?>
							</div>
						</div>
					<?php endif; ?>
					<div class="author-content">
						<p><?php echo wp_kses_post( $author_description ); ?></p>
					</div>
				</div>
			</div>
		</div>
	<?php
	}
endif;

if ( ! function_exists( 'home_experts_post_footer_meta' ) ) :
	/**
	 * Author Section
	 */
	function home_experts_post_footer_meta() {
		if ( !('post' === get_post_type() && has_tag() ) ) return;
		?>
		<footer class="entry-footer">
			<?php home_experts_tags(); ?>
		</footer>
		<?php
	}
endif;

if ( ! function_exists( 'home_experts_header_contact_info' ) ) {
	/**
	 * Header Information i.e email only
	 */
	function home_experts_header_contact_info( $container_class = '' ) {
		$email = get_theme_mod( 'email', sanitize_email( 'dolores@example.com' ) );

		if ( $email ) {
			?>
			<div class="<?php echo esc_attr( $container_class ); ?>">
				<p>
					<a class="header-email" href="<?php echo esc_url( 'mailto:' . sanitize_email( $email ) ); ?>">
						<?php echo wp_kses( home_experts_handle_all_svgs( 'email' ), home_experts_get_kses_extended_ruleset() ); ?>
						<?php echo esc_html( sanitize_email( $email ) ); ?>
					</a>
				</p>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'home_experts_header_location_info' ) ) {
	/**
	 * Header Information i.e location only
	 */
	function home_experts_header_location_info( $container_class = '' ) {
		$location = get_theme_mod( 'location', __( '3517 W., Pennsylvania 57867', 'home-experts' ) );

		if ( $location ) {
			?>
			<div class="<?php echo esc_attr( $container_class ); ?>">
				<p>
					<span class="header-location">
						<?php echo wp_kses( home_experts_handle_all_svgs( 'location' ), home_experts_get_kses_extended_ruleset() ); ?>
						<?php echo esc_html( $location ); ?>
					</span>
				</p>
			</div>
			<?php
		}
	}
}

if ( ! function_exists( 'home_experts_footer_newsletter' ) ) :
	/**
	 * Footer Newsletter
	 *
	 * @return void
	 */
	function home_experts_footer_newsletter() {
		$toggle_newsletter = get_theme_mod( 'toggle_footer_newsletter', false );
		$footer_subheading = get_theme_mod( 'footer_subheading', __( 'Feel free to send us your questions or request a free consultation.', 'home-experts' ) );
		$footer_heading    = get_theme_mod( 'footer_heading', __( 'Do you have any questions?', 'home-experts' ) );

		if ( $toggle_newsletter && ( $footer_subheading || $footer_heading ) ) {
			?>
			<?php if ( $footer_heading ) { ?>
				<h2><?php echo esc_html( $footer_heading ); ?></h2>
				<?php
			}
			if ( $footer_subheading ) {
				?>
				<span class="footer-left-heading"><?php echo esc_html( $footer_subheading ); ?></span>
				<?php
			}
		}
	}
endif;
