(function ($) {
    'use strict';

    // Wait for DOM to be ready
    $(document).ready(function () {
        // Constants
        const $sidebar = $('#mobileSideMenu');
        const $hamBarEl = $('#sideMenuOpener');
        const $mobileSideMenuClose = $('#mobileSidebarClose');

        // State variables
        let isFocusInsideSidebar = false;
        let isShiftTab = false;
        let runOnce = 0;

        // ============================================
        // Helper Functions
        // ============================================

        /**
         * Check if element is visible
         */
        function isElementVisible($el) {
            return $el.is(':visible') && $el.css('visibility') !== 'hidden';
        }

        /**
         * Check if submenu is closed
         */
        function isSubmenuClosed($submenu) {
            if (!$submenu.length) return true;

            const $parentItem = $submenu.closest('.menu-item-has-children');
            const $toggleButton = $parentItem.find('.angle-down');

            // Check if toggle button is not active
            if ($toggleButton.length && !$toggleButton.hasClass('active')) {
                return true;
            }

            // Check if submenu has maxHeight 0
            const maxHeight = $submenu.css('maxHeight');
            return maxHeight === '0px' || maxHeight === '0';
        }

        /**
         * Get focusable elements from a container
         */
        function getFocusableElements($container, excludeSelector) {
            return $container.find('a, button, input, select, textarea, [tabindex]:not([tabindex="-1"])')
                .filter(function () {
                    const $el = $(this);

                    // Exclude by selector if provided
                    if (excludeSelector && $el.closest(excludeSelector).length) {
                        return false;
                    }

                    // Only include visible elements
                    if (!isElementVisible($el)) {
                        return false;
                    }

                    // Exclude if tabindex is -1
                    const tabindex = $el.attr('tabindex');
                    if (tabindex === '-1') {
                        return false;
                    }

                    return true;
                });
        }

        /**
         * Filter sidebar focusable elements (exclude closed submenus)
         */
        function filterSidebarFocusable($elements) {
            return $elements.filter(function () {
                const $el = $(this);
                const $submenu = $el.closest('.sub-menu');

                // If inside a submenu, check if it's closed
                if ($submenu.length > 0 && isSubmenuClosed($submenu)) {
                    return false;
                }

                return true;
            });
        }

        /**
         * Open submenu
         */
        function openSubmenu($submenu, $item) {
            if (!$submenu.length) return;

            // Calculate total height of all submenus
            let totalHeight = 0;
            const allSubmenus = $item.find('.sub-menu');
            allSubmenus.each(function () {
                const $ul = $(this);
                if ($ul.length) {
                    totalHeight += $ul[0].scrollHeight;
                }
            });

            $submenu.css({
                maxHeight: totalHeight + 'px',
                visibility: 'visible',
                opacity: '1'
            });

            // Set tabindex to 0 for all links inside the submenu
            $submenu.find('a').attr('tabindex', '0');
            $submenu.find('.sub-menu').attr('tabindex', '0');
        }

        /**
         * Close submenu
         */
        function closeSubmenu($submenu) {
            if (!$submenu.length) return;

            $submenu.css('maxHeight', 0);

            // Hide nested submenus
            $submenu.find('.sub-menu').css('maxHeight', 0);

            // Remove active class from nested buttons
            $submenu.find('.angle-down').removeClass('active')
                .attr('aria-expanded', 'false')
                .attr('tabindex', '1');

            // Set tabindex to -1 for all links in submenus
            $submenu.find('a').attr('tabindex', '-1');
            $submenu.find('.sub-menu').attr('tabindex', '-1');
        }

        /**
         * Handle tab navigation within a list of focusable elements
         */
        function handleTabNavigation(e, $focusableElements, isReverse) {
            if ($focusableElements.length === 0) return;

            const currentIndex = $focusableElements.index(document.activeElement);

            // If current element is not in list, focus first
            if (currentIndex === -1) {
                e.preventDefault();
                $focusableElements.first().focus();
                return;
            }

            e.preventDefault();

            if (isReverse) {
                // Shift+Tab: move to previous element
                if (currentIndex === 0) {
                    // At first element, wrap to last
                    $focusableElements.last().focus();
                } else {
                    // Normal backward navigation
                    $focusableElements.eq(currentIndex - 1).focus();
                }
            } else {
                // Tab: move to next element
                if (currentIndex === $focusableElements.length - 1) {
                    // At last element, wrap to first
                    $focusableElements.first().focus();
                } else {
                    // Normal forward navigation
                    $focusableElements.eq(currentIndex + 1).focus();
                }
            }
        }
        /**
         * Handle Shift+Tab on menu items with children
         */
        function handleShiftTabOnMenuItem(e) {
            if (!e.shiftKey || e.key !== 'Tab' || isFocusInsideSidebar) {
                return;
            }

            const $target = $(document.activeElement);

            // Find the closest parent menu item
            const $menuItem = $target.closest('li.menu-item');
            if (!$menuItem.length) return;

            // Check if this menu item has a submenu
            const $submenu = $menuItem.children('.sub-menu');
            if ($submenu.length && $submenu.css('visibility') === 'hidden') {
                // Open submenu immediately
                $submenu.css({ visibility: 'visible', opacity: 1 });

                // Move focus directly to the last link in the submenu
                const $lastLink = $submenu.find('a').last();

                if ($lastLink.length) {
                    // Temporarily make hidden links focusable
                    if ($lastLink.is(':hidden')) {
                        $lastLink.attr('tabindex', -1);
                    }

                    $lastLink.focus();

                    if ($lastLink.attr('tabindex') == -1) {
                        $lastLink.removeAttr('tabindex');
                    }

                    return; // exit early because we already moved focus
                }
            }

            // If submenu is already open or no submenu, move backward in the menu
            const $allLinks = $('#primary-menu li.menu-item a').get().reverse(); // reverse array for backward tab
            const currentIndex = $allLinks.indexOf($target.get(0));
            if (currentIndex === -1) return;

            e.preventDefault(); // stop native backward tab

            const prevIndex = currentIndex + 1; // reversed array
            if (prevIndex < $allLinks.length) {
                const $prev = $($allLinks[prevIndex]);

                if ($prev.is(':hidden')) {
                    $prev.attr('tabindex', -1);
                }

                $prev.focus();

                if ($prev.attr('tabindex') == -1) {
                    $prev.removeAttr('tabindex');
                }
            }
        }

        /**
         * Handle sidebar keyboard navigation in keyup
         */
        function handleSidebarKeyupNavigation(e) {
            if (!isFocusInsideSidebar) return;

            const $focusableElements = getFocusableElements($sidebar);
            const lastElement = $focusableElements[$focusableElements.length - 1];
            const firstElement = $focusableElements[0];

            // Last focus inside sidebar when the first element is focused
            if (document.activeElement === firstElement && e.shiftKey && e.key === 'Tab') {
                e.preventDefault();
                setTimeout(() => {
                    $(lastElement).focus();
                }, 0);
            }

            // Focus previous link when shift + tab key is pressed on inactive dropdown
            const dropDownNotActive = $(document.activeElement).is('.angle-down:not(.active)');
            if (e.key === "Tab" && e.shiftKey && dropDownNotActive) {
                const $currentLi = $(document.activeElement).closest('li');
                const $previousSibling = $currentLi.prev('.menu-item-has-children');
                if ($previousSibling.length) {
                    e.preventDefault();
                    $previousSibling.find('button').focus();
                }
            }
        }

        /**
         * Handle close button keyboard activation
         */
        function handleCloseButtonKeyboard(e) {
            if (document.activeElement === $mobileSideMenuClose[0] &&
                (e.code.toLowerCase() === "enter" || e.code.toLowerCase() === "space")) {
                $mobileSideMenuClose.trigger('click');
                setTimeout(() => {
                    $hamBarEl.focus();
                }, 0);
            }
        }

        $(document).on('keydown', function (e) {
            if (e.shiftKey && e.key === 'Tab') {
                isShiftTab = true;
                handleShiftTabOnMenuItem(e);
            }
        });
        // Keyup event handler
        $(document).on('keyup', function (e) {
            handleSidebarKeyupNavigation(e);
            handleCloseButtonKeyboard(e);
        });



        /**
         * Show submenu when menu item gets focus
         */
        function handleMenuFocusIn(e) {
            if (isFocusInsideSidebar) return;

            const $menuItem = $(e.target).closest('.menu-item-has-children');
            const $submenu = $menuItem.children('.sub-menu');

            if ($submenu.length) {
                $submenu.css({
                    visibility: 'visible',
                    opacity: '1'
                });
            }
        }

        /**
         * Hide submenus when focus leaves menu
         */
        function handleMenuFocusOut() {
            setTimeout(() => {
                const focused = document.activeElement;

                if (!$(focused).closest('.menu-item-has-children').length && !isFocusInsideSidebar) {
                    $('.sub-menu').css({
                        visibility: 'hidden',
                        opacity: '0'
                    });
                    runOnce = 0;
                }
            }, 0);
        }

        // Focus event handlers
        $(document).on('focusin', handleMenuFocusIn);
        $(document).on('focusout', handleMenuFocusOut);

        /**
         * Close all dropdowns in the sidebar
         */
        function closeAllDropdowns() {
            const $activeButtons = $sidebar.find('.angle-down.active');

            $activeButtons.each(function () {
                const $button = $(this);
                const $wrapper = $button.closest('.menu-link-wrapper');
                const $submenu = $wrapper.next('.sub-menu');

                $button.removeClass('active').attr('aria-expanded', 'false');
                closeSubmenu($submenu);
            });
        }

        /**
         * Create toggle button for submenu
         */
        function createSubmenuToggleButton($item, $mainLink, $submenu) {
            const $button = $('<a>')
                .addClass('angle-down')
                .attr('href', '#')
                .text('Angle down')
                .attr('aria-expanded', 'false')
                .attr('aria-label', 'Toggle submenu');

            const $wrapper = $('<div>').addClass('menu-link-wrapper');
            $mainLink.appendTo($wrapper);
            $button.appendTo($wrapper);
            $submenu.css('maxHeight', 0);
            $submenu.before($wrapper);

            // Toggle submenu on click
            $button.on('click', function (e) {
                e.preventDefault();
                const isActive = $button.hasClass('active');
                const $currentSubmenu = $wrapper.next('.sub-menu');

                if (isActive) {
                    $button.removeClass('active');
                    closeSubmenu($currentSubmenu);
                } else {
                    $button.addClass('active');
                    openSubmenu($currentSubmenu, $item);
                }
            });

            // Handle keyboard navigation
            $button.on('keydown', function (e) {
                if (e.key === "Enter" || e.key === " ") {
                    e.preventDefault();
                    $button.trigger('click');
                }
            });
        }

        /**
         * Initialize mobile submenu buttons
         */
        function subMenuAddButton() {
            const $mobileMenu = $('#mobileSideMenu .main-navigation #primary-menu');

            // Set tabindex to -1 for all links inside submenus initially
            $mobileMenu.find('.sub-menu a').attr('tabindex', '-1');
            $mobileMenu.find('.sub-menu').attr('tabindex', '-1');

            if (!$mobileMenu.length) return;

            const $menuItems = $mobileMenu.find('.menu-item.menu-item-has-children');

            $menuItems.each(function () {
                const $item = $(this);

                // Skip if wrapper already exists
                if ($item.find('.menu-link-wrapper').length) {
                    return;
                }

                const $mainLink = $item.find('> a').first();
                const $submenu = $item.find('.sub-menu').first();

                if ($mainLink.length && $submenu.length) {
                    createSubmenuToggleButton($item, $mainLink, $submenu);
                }
            });
        }

        /**
         * Handle sidebar state changes (open/close)
         */
        function handleSidebarStateChange() {
            const hasActive = $sidebar.hasClass('active');

            if (!hasActive) {
                isFocusInsideSidebar = false;
                closeAllDropdowns();
            } else {
                isFocusInsideSidebar = true;
            }
        }

        /**
         * Initialize sidebar observer
         */
        function initSidebarObserver() {
            const observer = new MutationObserver(function (mutations) {
                mutations.forEach(mutation => {
                    if (mutation.attributeName === "class") {
                        handleSidebarStateChange();
                    }
                });
            });

            if ($sidebar.length > 0) {
                observer.observe($sidebar[0], { attributes: true });
            }
        }

        // ============================================
        // Initialization
        // ============================================

        // Initialize mobile submenu buttons
        subMenuAddButton();

        // Close all dropdowns when sidebar close button is clicked
        $mobileSideMenuClose.on('click', closeAllDropdowns);

        // Initialize sidebar observer
        initSidebarObserver();


        /**
         * Handle body tab navigation (wrap focus from last to first)
         */
        function handleBodyTabNavigation(e) {
            if (e.key !== "Tab" || e.shiftKey || isFocusInsideSidebar) {
                return;
            }

            const $bodyFocusable = getFocusableElements($('body'), '#mobileSideMenu');

            if ($bodyFocusable.length === 0) return;

            const lastElement = $bodyFocusable[$bodyFocusable.length - 1];
            const firstElement = $bodyFocusable[0];

            // Wrap focus from last element to first element
            if (document.activeElement === lastElement && !isShiftTab) {
                e.preventDefault();
                setTimeout(() => {
                    $(firstElement).focus();
                }, 0);
            }

            // Shift+Tab: wrap focus from first element to last element
            if (e.shiftKey && document.activeElement === firstElement) {
                e.preventDefault();
                setTimeout(() => {
                    $(lastElement).focus();
                }, 0);
            }
        }

        /**
         * Handle sidebar tab navigation (trap focus inside sidebar)
         */
        function handleSidebarTabNavigation(e) {
            if (e.key !== 'Tab' || !isFocusInsideSidebar) {
                return;
            }

            const $sidebarFocusable = filterSidebarFocusable(
                getFocusableElements($sidebar)
            );

            if ($sidebarFocusable.length === 0) return;

            handleTabNavigation(e, $sidebarFocusable, e.shiftKey);
        }

        // ============================================
        // Event Handlers
        // ============================================

        // Trap focus inside sidebar - prevent Tab/Shift+Tab from escaping
        $(document).on('keydown', function (e) {
            handleBodyTabNavigation(e);
            handleSidebarTabNavigation(e);
        });
    });

})(jQuery);
