<?php
/**
 * Custom Functions
 * @package Health Care Center
 * @since 1.0.0
 */

if( !function_exists('health_care_center_site_logo') ):

    /**
     * Logo & Description
     */
    /**
     * Displays the site logo, either text or image.
     *
     * @param array $health_care_center_args Arguments for displaying the site logo either as an image or text.
     * @param boolean $health_care_center_echo Echo or return the HTML.
     *
     * @return string $health_care_center_html Compiled HTML based on our arguments.
     */
    function health_care_center_site_logo( $health_care_center_args = array(), $health_care_center_echo = true ){
        $health_care_center_logo = get_custom_logo();
        $health_care_center_site_title = get_bloginfo('name');
        $health_care_center_contents = '';
        $health_care_center_classname = '';
        $health_care_center_defaults = array(
            'logo' => '%1$s<span class="screen-reader-text">%2$s</span>',
            'logo_class' => 'site-logo site-branding',
            'title' => '<a href="%1$s" class="custom-logo-name">%2$s</a>',
            'title_class' => 'site-title',
            'home_wrap' => '<h1 class="%1$s">%2$s</h1>',
            'single_wrap' => '<div class="%1$s">%2$s</div>',
            'condition' => (is_front_page() || is_home()) && !is_page(),
        );
        $health_care_center_args = wp_parse_args($health_care_center_args, $health_care_center_defaults);
        /**
         * Filters the arguments for `health_care_center_site_logo()`.
         *
         * @param array $health_care_center_args Parsed arguments.
         * @param array $health_care_center_defaults Function's default arguments.
         */
        $health_care_center_args = apply_filters('health_care_center_site_logo_args', $health_care_center_args, $health_care_center_defaults);
        
        $health_care_center_show_logo  = get_theme_mod('health_care_center_display_logo', false);
        $health_care_center_show_title = get_theme_mod('health_care_center_display_title', true);

        if ( has_custom_logo() && $health_care_center_show_logo ) {
            $health_care_center_contents .= sprintf($health_care_center_args['logo'], $health_care_center_logo, esc_html($health_care_center_site_title));
            $health_care_center_classname = $health_care_center_args['logo_class'];
        }

        if ( $health_care_center_show_title ) {
            $health_care_center_contents .= sprintf($health_care_center_args['title'], esc_url(get_home_url(null, '/')), esc_html($health_care_center_site_title));
            // If logo isn't shown, fallback to title class for wrapper.
            if ( !$health_care_center_show_logo ) {
                $health_care_center_classname = $health_care_center_args['title_class'];
            }
        }

        // If nothing is shown (logo or title both disabled), exit early
        if ( empty($health_care_center_contents) ) {
            return;
        }

        $health_care_center_wrap = $health_care_center_args['condition'] ? 'home_wrap' : 'single_wrap';
        // $health_care_center_wrap = 'home_wrap';
        $health_care_center_html = sprintf($health_care_center_args[$health_care_center_wrap], $health_care_center_classname, $health_care_center_contents);
        /**
         * Filters the arguments for `health_care_center_site_logo()`.
         *
         * @param string $health_care_center_html Compiled html based on our arguments.
         * @param array $health_care_center_args Parsed arguments.
         * @param string $health_care_center_classname Class name based on current view, home or single.
         * @param string $health_care_center_contents HTML for site title or logo.
         */
        $health_care_center_html = apply_filters('health_care_center_site_logo', $health_care_center_html, $health_care_center_args, $health_care_center_classname, $health_care_center_contents);
        if (!$health_care_center_echo) {
            return $health_care_center_html;
        }
        echo $health_care_center_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    }

endif;

if( !function_exists('health_care_center_site_description') ):

    /**
     * Displays the site description.
     *
     * @param boolean $health_care_center_echo Echo or return the html.
     *
     * @return string $health_care_center_html The HTML to display.
     */
    function health_care_center_site_description($health_care_center_echo = true){

        if ( get_theme_mod('health_care_center_display_header_text', false) == true ) :
        $health_care_center_description = get_bloginfo('description');
        if (!$health_care_center_description) {
            return;
        }
        $health_care_center_wrapper = '<div class="site-description"><span>%s</span></div><!-- .site-description -->';
        $health_care_center_html = sprintf($health_care_center_wrapper, esc_html($health_care_center_description));
        /**
         * Filters the html for the site description.
         *
         * @param string $health_care_center_html The HTML to display.
         * @param string $health_care_center_description Site description via `bloginfo()`.
         * @param string $health_care_center_wrapper The format used in case you want to reuse it in a `sprintf()`.
         * @since 1.0.0
         *
         */
        $health_care_center_html = apply_filters('health_care_center_site_description', $health_care_center_html, $health_care_center_description, $health_care_center_wrapper);
        if (!$health_care_center_echo) {
            return $health_care_center_html;
        }
        echo $health_care_center_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        endif;
    }

endif;

if( !function_exists('health_care_center_posted_on') ):

    /**
     * Prints HTML with meta information for the current post-date/time.
     */
    function health_care_center_posted_on( $health_care_center_icon = true, $animation_class = '' ){

        $health_care_center_default = health_care_center_get_default_theme_options();
        $health_care_center_post_date = absint( get_theme_mod( 'health_care_center_post_date',$health_care_center_default['health_care_center_post_date'] ) );

        if( $health_care_center_post_date ){

            $health_care_center_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
            if (get_the_time('U') !== get_the_modified_time('U')) {
                $health_care_center_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
            }

            $health_care_center_time_string = sprintf($health_care_center_time_string,
                esc_attr(get_the_date(DATE_W3C)),
                esc_html(get_the_date()),
                esc_attr(get_the_modified_date(DATE_W3C)),
                esc_html(get_the_modified_date())
            );

            $health_care_center_year = get_the_date('Y');
            $health_care_center_month = get_the_date('m');
            $health_care_center_day = get_the_date('d');
            $health_care_center_link = get_day_link($health_care_center_year, $health_care_center_month, $health_care_center_day);

            $health_care_center_posted_on = '<a href="' . esc_url($health_care_center_link) . '" rel="bookmark">' . $health_care_center_time_string . '</a>';

            echo '<div class="entry-meta-item entry-meta-date">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $animation_class ).'">';

            if( $health_care_center_icon ){

                echo '<span class="entry-meta-icon calendar-icon"> ';
                health_care_center_the_theme_svg('calendar');
                echo '</span>';

            }

            echo '<span class="posted-on">' . $health_care_center_posted_on . '</span>'; // WPCS: XSS OK.
            echo '</div>';
            echo '</div>';

        }

    }

endif;

if( !function_exists('health_care_center_posted_by') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function health_care_center_posted_by( $health_care_center_icon = true, $animation_class = '' ){   

        $health_care_center_default = health_care_center_get_default_theme_options();
        $health_care_center_post_author = absint( get_theme_mod( 'health_care_center_post_author',$health_care_center_default['health_care_center_post_author'] ) );

        if( $health_care_center_post_author ){

            echo '<div class="entry-meta-item entry-meta-author">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $animation_class ).'">';

            if( $health_care_center_icon ){
            
                echo '<span class="entry-meta-icon author-icon"> ';
                health_care_center_the_theme_svg('user');
                echo '</span>';
                
            }

            $health_care_center_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';
            echo '<span class="byline"> ' . $health_care_center_byline . '</span>'; // WPCS: XSS OK.
            echo '</div>';
            echo '</div>';

        }

    }

endif;


if( !function_exists('health_care_center_posted_by_avatar') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function health_care_center_posted_by_avatar( $health_care_center_date = false ){

        $health_care_center_default = health_care_center_get_default_theme_options();
        $health_care_center_post_author = absint( get_theme_mod( 'health_care_center_post_author',$health_care_center_default['health_care_center_post_author'] ) );

        if( $health_care_center_post_author ){



            echo '<div class="entry-meta-left">';
            echo '<div class="entry-meta-item entry-meta-avatar">';
            echo wp_kses_post( get_avatar( get_the_author_meta( 'ID' ) ) );
            echo '</div>';
            echo '</div>';

            echo '<div class="entry-meta-right">';

            $health_care_center_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';

            echo '<div class="entry-meta-item entry-meta-byline"> ' . $health_care_center_byline . '</div>';

            if( $health_care_center_date ){
                health_care_center_posted_on($health_care_center_icon = false);
            }
            echo '</div>';

        }

    }

endif;

if( !function_exists('health_care_center_entry_footer') ):

    /**
     * Prints HTML with meta information for the categories, tags and comments.
     */
    function health_care_center_entry_footer( $health_care_center_cats = true, $health_care_center_tags = true, $health_care_center_edits = true){   

        $health_care_center_default = health_care_center_get_default_theme_options();
        $health_care_center_post_category = absint( get_theme_mod( 'health_care_center_post_category',$health_care_center_default['health_care_center_post_category'] ) );
        $health_care_center_post_tags = absint( get_theme_mod( 'health_care_center_post_tags',$health_care_center_default['health_care_center_post_tags'] ) );

        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            if( $health_care_center_cats && $health_care_center_post_category ){

                /* translators: used between list items, there is a space after the comma */
                $health_care_center_categories = get_the_category();
                if ($health_care_center_categories) {
                    echo '<div class="entry-meta-item entry-meta-categories">';
                    echo '<div class="entry-meta-wrapper">';
                
                    /* translators: 1: list of categories. */
                    echo '<span class="cat-links">';
                    foreach( $health_care_center_categories as $health_care_center_category ){

                        $health_care_center_cat_name = $health_care_center_category->name;
                        $health_care_center_cat_slug = $health_care_center_category->slug;
                        $health_care_center_cat_url = get_category_link( $health_care_center_category->term_id );
                        ?>

                        <a class="twp_cat_<?php echo esc_attr( $health_care_center_cat_slug ); ?>" href="<?php echo esc_url( $health_care_center_cat_url ); ?>" rel="category tag"><?php echo esc_html( $health_care_center_cat_name ); ?></a>

                    <?php }
                    echo '</span>'; // WPCS: XSS OK.
                    echo '</div>';
                    echo '</div>';
                }

            }

            if( $health_care_center_tags && $health_care_center_post_tags ){
                /* translators: used between list items, there is a space after the comma */
                $health_care_center_tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'health-care-center'));
                if( $health_care_center_tags_list ){

                    echo '<div class="entry-meta-item entry-meta-tags">';
                    echo '<div class="entry-meta-wrapper">';

                    /* translators: 1: list of tags. */
                    echo '<span class="tags-links">';
                    echo wp_kses_post($health_care_center_tags_list) . '</span>'; // WPCS: XSS OK.
                    echo '</div>';
                    echo '</div>';

                }

            }

            if( $health_care_center_edits ){

                edit_post_link(
                    sprintf(
                        wp_kses(
                        /* translators: %s: Name of current post. Only visible to screen readers */
                            __('Edit <span class="screen-reader-text">%s</span>', 'health-care-center'),
                            array(
                                'span' => array(
                                    'class' => array(),
                                ),
                            )
                        ),
                        get_the_title()
                    ),
                    '<span class="edit-link">',
                    '</span>'
                );
            }

        }
    }

endif;

if ( ! function_exists( 'health_care_center_post_thumbnail' ) ) :

    /**
     * Displays an optional post thumbnail.
     *
     * Shows background style image with height class on archive/search/front,
     * and a normal inline image on single post/page views.
     */
    function health_care_center_post_thumbnail( $health_care_center_image_size = 'medium' ) {

        if ( post_password_required() || is_attachment() ) {
            return;
        }

        // Fallback image path
        $health_care_center_default_image = get_template_directory_uri() . '/assets/images/default.png';

        // Image size → height class map
        $health_care_center_size_class_map = array(
            'full'      => 'data-bg-large',
            'large'     => 'data-bg-big',
            'medium'    => 'data-bg-medium',
            'small'     => 'data-bg-small',
            'xsmall'    => 'data-bg-xsmall',
            'thumbnail' => 'data-bg-thumbnail',
        );

        $health_care_center_class = isset( $health_care_center_size_class_map[ $health_care_center_image_size ] )
            ? $health_care_center_size_class_map[ $health_care_center_image_size ]
            : 'data-bg-medium';

        if ( is_singular() ) {
            the_post_thumbnail();
        } else {
            // 🔵 On archives → use background image style
            $health_care_center_image = has_post_thumbnail()
                ? wp_get_attachment_image_src( get_post_thumbnail_id(), $health_care_center_image_size )
                : array( $health_care_center_default_image );

            $health_care_center_bg_image = isset( $health_care_center_image[0] ) ? $health_care_center_image[0] : $health_care_center_default_image;
            ?>
            <div class="post-thumbnail data-bg <?php echo esc_attr( $health_care_center_class ); ?>"
                 data-background="<?php echo esc_url( $health_care_center_bg_image ); ?>">
                <a href="<?php the_permalink(); ?>" class="theme-image-responsive" tabindex="0"></a>
            </div>
            <?php
        }
    }

endif;

if( !function_exists('health_care_center_is_comment_by_post_author') ):

    /**
     * Comments
     */
    /**
     * Check if the specified comment is written by the author of the post commented on.
     *
     * @param object $health_care_center_comment Comment data.
     *
     * @return bool
     */
    function health_care_center_is_comment_by_post_author($health_care_center_comment = null){

        if (is_object($health_care_center_comment) && $health_care_center_comment->user_id > 0) {
            $health_care_center_user = get_userdata($health_care_center_comment->user_id);
            $post = get_post($health_care_center_comment->comment_post_ID);
            if (!empty($health_care_center_user) && !empty($post)) {
                return $health_care_center_comment->user_id === $post->post_author;
            }
        }
        return false;
    }

endif;

if( !function_exists('health_care_center_breadcrumb') ) :

    /**
     * Health Care Center Breadcrumb
     */
    function health_care_center_breadcrumb($health_care_center_comment = null){

        echo '<div class="entry-breadcrumb">';
        health_care_center_breadcrumb_trail();
        echo '</div>';

    }

endif;