<?php
/**
 * Glow Slider Component
 * 
 * Displays a carousel of featured posts with hero images and left-aligned content overlay
 * 
 * @package Graceful Glow Blog
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Helper function to validate slide data
 * 
 * @param array $slide_data The slide data to validate
 * @return bool True if valid, false otherwise
 */
function glow_slider_validate_slide_data( $slide_data ) {
	// Check required fields
	$required_fields = array( 'id', 'title', 'permalink', 'featured_image' );
	
	foreach ( $required_fields as $field ) {
		if ( ! isset( $slide_data[ $field ] ) || empty( $slide_data[ $field ] ) ) {
			return false;
		}
	}
	
	// Validate URL fields
	if ( ! filter_var( $slide_data['featured_image'], FILTER_VALIDATE_URL ) ) {
		return false;
	}
	
	if ( ! filter_var( $slide_data['permalink'], FILTER_VALIDATE_URL ) ) {
		return false;
	}
	
	// Validate ID is numeric
	if ( ! is_numeric( $slide_data['id'] ) || $slide_data['id'] <= 0 ) {
		return false;
	}
	
	return true;
}

/**
 * Calculate reading time for a post
 * 
 * @param int $post_id Post ID
 * @return string Reading time text
 */
function glow_slider_get_reading_time( $post_id ) {
	$content = get_post_field( 'post_content', $post_id );
	$word_count = str_word_count( strip_tags( $content ) );
	$reading_time = ceil( $word_count / 200 ); // Average reading speed: 200 words per minute
	
	if ( $reading_time < 1 ) {
		$reading_time = 1;
	}
	
	return $reading_time . ' ' . esc_html__( 'min read', 'graceful-glow-blog' );
}

/**
 * Display Glow Slider
 * 
 * Main function to render the Glow Slider component
 */
function glow_slider_display() {
	// Check if slider is enabled
	$slider_enabled = get_theme_mod( 'glow_slider_enabled', true );
	if ( ! $slider_enabled ) {
		return;
	}
	
	// Get customizer settings
	$post_count = absint( get_theme_mod( 'glow_slider_post_count', 3 ) );
	
	// Ensure post count is within valid range
	if ( $post_count < 1 ) {
		$post_count = 1;
	} elseif ( $post_count > 5 ) {
		$post_count = 5;
	}
	
	$show_category = (bool) get_theme_mod( 'glow_slider_show_category', true );
	$show_reading_time = (bool) get_theme_mod( 'glow_slider_show_reading_time', true );
	$show_button = (bool) get_theme_mod( 'glow_slider_show_button', true );
	$show_arrows = (bool) get_theme_mod( 'glow_slider_show_arrows', true );
	$show_dots = (bool) get_theme_mod( 'glow_slider_show_dots', true );
	$background_color = sanitize_hex_color( get_theme_mod( 'glow_slider_background_color', '#e5ded0' ) );
	$button_text = sanitize_text_field( get_theme_mod( 'glow_slider_button_text', esc_html__( 'Read More', 'graceful-glow-blog' ) ) );
	
	// Fallback for empty button text
	if ( empty( $button_text ) ) {
		$button_text = esc_html__( 'Read More', 'graceful-glow-blog' );
	}
	
	// Query latest posts with featured images
	$query_args = array(
		'posts_per_page' => $post_count,
		'post_status' => 'publish',
		'meta_query' => array(
			array(
				'key' => '_thumbnail_id',
				'compare' => 'EXISTS',
			),
		),
		'orderby' => 'date',
		'order' => 'DESC',
		'no_found_rows' => true,
		'update_post_meta_cache' => false,
		'update_post_term_cache' => false,
	);
	
	$posts_query = new WP_Query( $query_args );
	
	// Check for errors
	if ( is_wp_error( $posts_query ) ) {
		echo '<div class="glow-slider-error">' . esc_html__( 'Error loading posts for slider.', 'graceful-glow-blog' ) . '</div>';
		return;
	}
	
	// Handle empty results
	if ( ! $posts_query->have_posts() ) {
		echo '<div class="glow-slider-empty">' . esc_html__( 'No featured posts available.', 'graceful-glow-blog' ) . '</div>';
		wp_reset_postdata();
		return;
	}
	
	// Prepare slide data
	$slides_data = array();
	
	while ( $posts_query->have_posts() ) {
		$posts_query->the_post();
		
		// Get featured image
		$featured_image = get_the_post_thumbnail_url( get_the_ID(), 'large' );
		
		if ( ! $featured_image ) {
			$featured_image = get_the_post_thumbnail_url( get_the_ID(), 'medium_large' );
		}
		if ( ! $featured_image ) {
			$featured_image = get_the_post_thumbnail_url( get_the_ID(), 'medium' );
		}
		
		if ( ! $featured_image ) {
			continue;
		}
		
		$post_title = get_the_title();
		$post_permalink = get_permalink();
		
		if ( empty( $post_title ) || empty( $post_permalink ) ) {
			continue;
		}
		
		// Get first category
		$categories = get_the_category();
		$category_name = '';
		if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
			$category_name = esc_html( $categories[0]->name );
		}
		
		// Get reading time
		$reading_time = glow_slider_get_reading_time( get_the_ID() );
		
		$slide_data = array(
			'id' => get_the_ID(),
			'title' => $post_title,
			'permalink' => $post_permalink,
			'featured_image' => $featured_image,
			'category' => $category_name,
			'reading_time' => $reading_time,
		);
		
		if ( glow_slider_validate_slide_data( $slide_data ) ) {
			$slides_data[] = $slide_data;
		}
	}
	
	wp_reset_postdata();
	
	if ( empty( $slides_data ) ) {
		echo '<div class="glow-slider-empty">' . esc_html__( 'No featured posts available.', 'graceful-glow-blog' ) . '</div>';
		return;
	}

	// Render slider HTML
	?>
	<style>
		.glow-slider-wrapper::before {
			background-color: <?php echo esc_attr( $background_color ); ?> !important;
		}
	</style>
	<div class="glow-slider-wrapper">
		<div class="glow-slider-container">
			<div class="owl-carousel glow-slider-carousel">
				<?php foreach ( $slides_data as $slide ) : ?>
					<div class="glow-slider-item" style="background-image: url('<?php echo esc_url( $slide['featured_image'] ); ?>');">
						<div class="glow-slider-overlay"></div>
						<div class="glow-slider-content">
							<?php if ( $show_category && ! empty( $slide['category'] ) ) : ?>
								<div class="glow-slider-category">
									<span><?php echo esc_html( strtoupper( $slide['category'] ) ); ?></span>
								</div>
							<?php endif; ?>
							
							<h2 class="glow-slider-title">
								<a href="<?php echo esc_url( $slide['permalink'] ); ?>" title="<?php echo esc_attr( $slide['title'] ); ?>">
									<?php echo esc_html( $slide['title'] ); ?>
								</a>
							</h2>
							
							<div class="glow-slider-meta">
								<?php if ( $show_button ) : ?>
								<a href="<?php echo esc_url( $slide['permalink'] ); ?>" class="glow-slider-button">
									<?php echo esc_html( $button_text ); ?>
								</a>
								<?php endif; ?>
								
								<?php if ( $show_reading_time ) : ?>
									<span class="glow-slider-reading-time">
										<svg class="glow-slider-clock-icon" width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg">
											<circle cx="8" cy="8" r="7" stroke="currentColor" stroke-width="1.5"/>
											<path d="M8 4V8L11 11" stroke="currentColor" stroke-width="1.5" stroke-linecap="round"/>
										</svg>
										<?php echo esc_html( $slide['reading_time'] ); ?>
									</span>
								<?php endif; ?>
							</div>
						</div>
					</div>
				<?php endforeach; ?>
			</div>
			
			<?php if ( $show_arrows ) : ?>
			<!-- Navigation Arrows -->
			<button class="glow-slider-nav prev" type="button" aria-label="<?php esc_attr_e( 'Previous slide', 'graceful-glow-blog' ); ?>">
				<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
					<path d="M15 18L9 12L15 6" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round"/>
				</svg>
			</button>
			<button class="glow-slider-nav next" type="button" aria-label="<?php esc_attr_e( 'Next slide', 'graceful-glow-blog' ); ?>">
				<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
					<path d="M9 18L15 12L9 6" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round"/>
				</svg>
			</button>
			<?php endif; ?>
			
			<?php if ( $show_dots ) : ?>
			<!-- Dots Navigation -->
			<div class="glow-slider-dots"></div>
			<?php endif; ?>
		</div>
	</div>
	<?php
}
