<?php

/**
 * This source file is subject to the GNU GENERAL PUBLIC LICENSE (GPL 3.0)
 * It is also available at this URL: http://www.gnu.org/licenses/gpl-3.0.txt
 */

define( 'GIORNALE_MIN_PHP_VERSION', '5.3' );

/*-----------------------------------------------------------------------------------*/
/* Switches back to the previous theme if the minimum PHP version is not met */
/*-----------------------------------------------------------------------------------*/

if ( ! function_exists( 'giornale_check_php_version' ) ) {

	function giornale_check_php_version() {

		if ( version_compare( PHP_VERSION, GIORNALE_MIN_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', 'giornale_min_php_not_met_notice' );
			switch_theme( get_option( 'theme_switched' ));
			return false;

		};
	}

	add_action( 'after_switch_theme', 'giornale_check_php_version' );

}

/*-----------------------------------------------------------------------------------*/
/* An error notice that can be displayed if the Minimum PHP version is not met */
/*-----------------------------------------------------------------------------------*/

if ( ! function_exists( 'giornale_min_php_not_met_notice' ) ) {

	function giornale_min_php_not_met_notice() {
		?>
		<div class="notice notice-error is_dismissable">
			<p>
				<?php esc_html_e('You need to update your PHP version to run this theme.', 'giornale' ); ?><br />
				<?php
				printf(
					esc_html__( 'Actual version is: %1$s, required version is: %2$s.', 'giornale' ),
					PHP_VERSION,
					GIORNALE_MIN_PHP_VERSION
				);
				?>
			</p>
		</div>
		<?php

	}

}

/*-----------------------------------------------------------------------------------*/
/* WooCommerce is active */
/*-----------------------------------------------------------------------------------*/

if ( ! function_exists( 'giornale_is_woocommerce_active' ) ) {

	function giornale_is_woocommerce_active( $type = '' ) {

        global $woocommerce;

        if ( isset( $woocommerce ) ) {

			if ( !$type || call_user_func($type) ) {

				return true;

			}

		}

	}

}

/*-----------------------------------------------------------------------------------*/
/* Get post comments */
/*-----------------------------------------------------------------------------------*/

if ( ! function_exists( 'giornale_comments' ) ) {

	function giornale_comments($comment, $args, $depth) {

	?>

		<li <?php comment_class(); ?> id="li-comment-<?php comment_ID() ?>">

		<div id="comment-<?php comment_ID(); ?>" class="comment-container">

			<div class="comment-avatar">
				<?php echo get_avatar($comment->comment_author_email, 80 ); ?>
			</div>

			<div class="comment-text">

            	<header class="comment-author">

                    <span class="author"><cite><?php printf( esc_html__('%s has written:','giornale'), get_comment_author_link());?> </cite></span>

                    <time datetime="<?php echo esc_attr(get_comment_date())?>" class="comment-date">

                        <a href="<?php echo esc_url( get_comment_link( $comment->comment_ID ) ) ?>">
                            <?php printf(esc_html__('%1$s at %2$s','giornale'), get_comment_date(),  get_comment_time()) ?>
                        </a>

                        <?php

							comment_reply_link(
								array_merge(
									$args,
									array(
										'depth' => $depth,
										'max_depth' => $args['max_depth']
									)
								)
							);

							edit_comment_link(esc_html__('(Edit)','giornale'));

						?>

                    </time>

				</header>

				<?php if ($comment->comment_approved == '0') : ?>

                    <p><em><?php esc_html_e('Your comment is awaiting approval.','giornale') ?></em></p>

				<?php endif; ?>

				<?php comment_text() ?>

			</div>

			<div class="clear"></div>

		</div>

	<?php

	}

}

/*-----------------------------------------------------------------------------------*/
/* Get archive title */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_get_the_archive_title')) {

	function giornale_get_archive_title() {

		if ( is_category() ) {
			$title = sprintf( esc_html__( 'Category: %s', 'giornale' ), single_cat_title( '', false ) );
		} elseif ( is_tag() ) {
			$title = sprintf( esc_html__( 'Tag: %s', 'giornale' ), single_tag_title( '', false ) );
		} elseif ( is_author() ) {
			$title = sprintf( esc_html__( 'Author: %s', 'giornale' ), '<span class="vcard">' . get_the_author() . '</span>' );
		} elseif ( is_year() ) {
			$title = sprintf( esc_html__( 'Year: %s', 'giornale' ), get_the_date( esc_html_x( 'Y', 'yearly archives date format', 'giornale' ) ) );
		} elseif ( is_month() ) {
			$title = sprintf( esc_html__( 'Month: %s', 'giornale' ), get_the_date( esc_html_x( 'F Y', 'monthly archives date format', 'giornale' ) ) );
		} elseif ( is_day() ) {
			$title = sprintf( esc_html__( 'Day: %s', 'giornale' ), get_the_date( esc_html_x( 'F j, Y', 'daily archives date format', 'giornale' ) ) );
		} elseif ( is_tax( 'post_format' ) ) {
			if ( is_tax( 'post_format', 'post-format-aside' ) ) {
				$title = esc_html_x( 'Asides', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-gallery' ) ) {
				$title = esc_html_x( 'Galleries', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-image' ) ) {
				$title = esc_html_x( 'Images', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-video' ) ) {
				$title = esc_html_x( 'Videos', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-quote' ) ) {
				$title = esc_html_x( 'Quotes', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-link' ) ) {
				$title = esc_html_x( 'Links', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-status' ) ) {
				$title = esc_html_x( 'Statuses', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-audio' ) ) {
				$title = esc_html_x( 'Audio', 'post format archive title', 'giornale' );
			} elseif ( is_tax( 'post_format', 'post-format-chat' ) ) {
				$title = esc_html_x( 'Chats', 'post format archive title', 'giornale' );
			}
		} elseif ( is_post_type_archive() ) {
			$title = sprintf( esc_html__( 'Archives: %s', 'giornale' ), post_type_archive_title( '', false ) );
		} elseif ( is_tax() ) {
			$tax = get_taxonomy( get_queried_object()->taxonomy );
			$title = sprintf( esc_html__( '%1$s: %2$s', 'giornale' ), $tax->labels->singular_name, single_term_title( '', false ) );
		}

		if ( isset($title) )  :
			return $title;
		else:
			return false;
		endif;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Check if is single page */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_is_single')) {

	function giornale_is_single() {

		if ( is_single() || is_page() ) :

			return true;

		endif;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Get theme setting */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_setting')) {

	function giornale_setting($id, $default = FALSE ) {

		return get_theme_mod($id, $default);

	}

}

/*-----------------------------------------------------------------------------------*/
/* Get post meta */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_postmeta')) {

	function giornale_postmeta( $id, $default = '' ) {

		global $post, $wp_query;

		if (giornale_is_woocommerce_active('is_shop')) :

			$content_ID = get_option('woocommerce_shop_page_id');

		else :

			$content_ID = (isset($post->ID)) ? $post->ID : false;

		endif;

		$val = get_post_meta( $content_ID , $id, TRUE);

		if ( !empty($val) ) :

			return $val;

		else:

			return $default;

		endif;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Responsive embed */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_embed_html')) {

	function giornale_embed_html( $html ) {
		return '<div class="embed-container">' . $html . '</div>';
	}

	add_filter( 'embed_oembed_html', 'giornale_embed_html', 10, 3 );
	add_filter( 'video_embed_html', 'giornale_embed_html' );

}

/*-----------------------------------------------------------------------------------*/
/* Content template */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_template')) {

	function giornale_template($id) {

		$template = array (

			'full' => 'col-md-12' ,
			'left-sidebar' => 'col-md-8' ,
			'right-sidebar' => 'col-md-8'

		);

		$span = $template['right-sidebar'];
		$sidebar =  'right-sidebar';

		if (
			giornale_is_woocommerce_active('is_woocommerce') &&
			giornale_postmeta('giornale_template') &&
			is_search()
		) {

			$span = $template[esc_attr(giornale_postmeta('giornale_template'))];
			$sidebar =  esc_attr(giornale_postmeta('giornale_template'));

		} elseif (
			!is_attachment() &&
			giornale_postmeta('giornale_template') &&
			(is_page() || is_single() || giornale_is_woocommerce_active('is_shop'))
		) {

			$span = $template[esc_attr(giornale_postmeta('giornale_template'))];
			$sidebar =  esc_attr(giornale_postmeta('giornale_template'));

		} elseif (
			!giornale_is_woocommerce_active('is_woocommerce') &&
			( is_category() || is_tag() || is_month() ) &&
			giornale_setting('giornale_category_layout')
		) {

			$span = $template[esc_attr(giornale_setting('giornale_category_layout'))];
			$sidebar =  esc_attr(giornale_setting('giornale_category_layout'));

		} elseif (
			is_home() &&
			giornale_setting('giornale_home_layout')
		) {

			$span = $template[esc_attr(giornale_setting('giornale_home_layout'))];
			$sidebar =  esc_attr(giornale_setting('giornale_home_layout'));

		} else if (
			!giornale_is_woocommerce_active('is_woocommerce') &&
			is_search() &&
			giornale_setting('giornale_search_layout')
		) {

			$span = $template[esc_attr(giornale_setting('giornale_search_layout'))];
			$sidebar =  esc_attr(giornale_setting('giornale_search_layout'));

		} else if (
			giornale_is_woocommerce_active('is_woocommerce') &&
			( giornale_is_woocommerce_active('is_product_category') || giornale_is_woocommerce_active('is_product_tag') ) &&
			giornale_setting('giornale_woocommerce_category_layout')
		) {

			$span = $template[esc_attr(giornale_setting('giornale_woocommerce_category_layout'))];
			$sidebar =  esc_attr(giornale_setting('giornale_woocommerce_category_layout'));

		} elseif ( is_attachment() ) {

			$span = $template['full'];
			$sidebar =  'full';

		}

		return ${$id};

	}

}

/*-----------------------------------------------------------------------------------*/
/* Is archive */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_is_archive')) {

	function giornale_is_archive() {

		if (
			is_category() ||
			is_tag() ||
			is_author() ||
			is_year() ||
			is_month()

		) {

			return true;

		} else {

			return false;

		}

	}

}

/*-----------------------------------------------------------------------------------*/
/* BODY CLASSES */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_body_classes_function')) {

	function giornale_body_classes_function($classes) {

		global $wp_customize;

		if ( isset( $wp_customize ) ) :

			$classes[] = 'is_customizer_panel';

		endif;

		if (
			( is_home() ) ||
			( giornale_is_archive() ) ||
			( is_search() )
		) :

			$classes[] = 'is_blog_section';

		endif;

		if (
			( 
				!get_background_image() &&
				get_theme_mod('background_color','ffffff') == 'ffffff'
			)
		) :

			$classes[] = 'is_minimal_layout';

		endif;

		if ( giornale_setting('giornale_topbar_rightcolumn') == 'secondary_menu') :

			$classes[] = 'is_fullwidth_topbar';

		endif;

		if ( giornale_setting('giornale_underline_below_section_titles', false) == true ) :

			$classes[] = 'hide_underline_below_section_titles';

		endif;

		if ( 
			(
				giornale_is_archive() && 
				giornale_setting('giornale_category_layout') == 'full'
			) || 
			(
				is_search() && 
				giornale_setting('giornale_search_layout') == 'full'
			)
		) :

			$classes[] = 'top-section-fullwidth-layout';
		
		endif;
		
		return $classes;

	}

	add_filter('body_class', 'giornale_body_classes_function');

}

/*-----------------------------------------------------------------------------------*/
/* Post class */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_post_class')) {

	function giornale_post_class($classes) {

		$masonry  = 'post-container masonry-item col-md-4';
		$standard = 'post-container col-md-12';

		if ( !giornale_is_single() ) {

			if ( is_home() ) {

				if ( 
					giornale_setting('giornale_home_layout') == 'col-md-4'
				) {

					$classes[] = $masonry;

				} else {

					$classes[] = $standard;

				}

			} else if ( is_archive() && !giornale_is_woocommerce_active('is_shop') ) {

				if ( giornale_setting('giornale_category_layout') == 'col-md-4' ) {

					$classes[] = $masonry;

				} else {

					$classes[] = $standard;

				}

			} else if ( is_search() ) {

				if ( giornale_setting('giornale_search_layout') == 'col-md-4' ) {

					$classes[] = $masonry;

				} else {

					$classes[] = $standard;

				}

			}

		} else if ( giornale_is_single() && giornale_is_woocommerce_active('is_cart') ) {

			$classes[] = 'post-container col-md-12 woocommerce_cart_page';

		} else if ( giornale_is_single() && !giornale_is_woocommerce_active('is_product') ) {

			$classes[] = 'post-container col-md-12';

		} else if ( is_page() ) {

			$classes[] = 'full';

		}

		if (
			( is_home() ) ||
			( giornale_is_archive() ) ||
			( is_search() )
		) :

			$classes[] = 'post-container-wrap';

		endif;

		return $classes;

	}

	add_filter('post_class', 'giornale_post_class');

}

/*-----------------------------------------------------------------------------------*/
/* Get paged */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_paged')) {

	function giornale_paged() {

		if ( get_query_var('paged') ) {
			$paged = get_query_var('paged');
		} elseif ( get_query_var('page') ) {
			$paged = get_query_var('page');
		} else {
			$paged = 1;
		}

		return $paged;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Swipebox post gallery */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_swipebox')) {

	function giornale_swipebox($html, $id, $size, $permalink, $icon, $text) {

		if ( !$permalink )
			return str_replace( '<a', '<a class="swipebox"', $html );
		else
			return $html;
	}

	add_filter( 'wp_get_attachment_link', 'giornale_swipebox', 10, 6);

}

/*-----------------------------------------------------------------------------------*/
/* Get link url  */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_get_link_url')) {

	function giornale_get_link_url() {

		$content = get_the_content();
		$has_url = get_url_in_content( $content );
		return ( $has_url ) ? $has_url : apply_filters( 'the_permalink', get_permalink() );

	}

}

/*-----------------------------------------------------------------------------------*/
/* Excerpt more */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_hide_excerpt_more')) {

	function giornale_hide_excerpt_more() {
		return '';
	}

	add_filter('the_content_more_link', 'giornale_hide_excerpt_more');
	add_filter('excerpt_more', 'giornale_hide_excerpt_more');

}

/*-----------------------------------------------------------------------------------*/
/* Customize excerpt more */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_customize_excerpt_more')) {

	function giornale_customize_excerpt_more( $excerpt ) {

		global $post;

		if ( giornale_is_single() ) :

			return $excerpt;

		else:

			$allowed = array(
				'span' => array(
					'class' => array(),
				),
			);

			$class = 'button ';
			$button = esc_html__('Read More','giornale');
			$container = 'class="read-more"';

			if ( giornale_setting('giornale_readmore_layout') == "default" || !giornale_setting('giornale_readmore_layout') ) :

				$class = 'button ' . esc_attr(giornale_setting('giornale_readmore_layout'));
				$button = esc_html__('Read More','giornale');
				$container = 'class="read-more"';

			else :

				$class = 'nobutton';
				$button = ' [&hellip;] ';
				$container = '';

			endif;

			if (
				( $pos=strpos($post->post_content, '<!--more-->') ) &&
				!has_excerpt($post->ID)
			):

				$content = apply_filters( 'the_content', get_the_content());

			else:

				$content = $excerpt;

			endif;

			return $content . '<div class="read-more-wrapper"><a '. wp_kses($container, $allowed) . ' href="' . esc_url(get_permalink($post->ID)) . '" title="'.esc_attr__('Read More','giornale').'"> <span class="'.esc_attr($class).'">'.$button.'</span></a></div>';

		endif;


	}

	add_filter( 'get_the_excerpt', 'giornale_customize_excerpt_more' );

}

/*-----------------------------------------------------------------------------------*/
/* Recent posts more tag */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_recent_post_more_tag')) {

    function giornale_recent_post_more_tag($postID) {

        $excerpt = wp_trim_excerpt('', $postID);

        if (!empty($excerpt)) {
            $more = ' <a href="' . esc_url(get_permalink($postID)) . '">[&hellip;]</a>';
            return $excerpt . $more;
        }

        return $excerpt;
    }

}
	
/*-----------------------------------------------------------------------------------*/
/* Get post icon */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_posticon')) {

	function giornale_posticon() {
		
		$html = '';

		$icons = array (
			'video' => 'fa fa-play' , 
			'gallery' => 'fa fa-camera' , 
			'audio' => 'fa fa-volume-up' , 
			'chat' => 'fa fa-users', 
			'status' => 'fa fa-keyboard-o', 
			'image' => 'fa fa-picture-o' ,
			'quote' => 'fa fa-quote-left', 
			'link' => 'fa fa-external-link', 
			'aside' => 'fa fa-file-text-o', 
		);

		if ( get_post_format() ) { 
		
			$html .= '<i class="'.esc_attr($icons[get_post_format()]).'"></i> '; 
			$html .= ucfirst( strtolower( esc_html(get_post_format()) )); 
		
		} else {
		
			$html .= '<i class="fa fa-pencil-square"></i> '; 
			$html .= esc_html__( 'Article','giornale'); 
		
		}

		return $html;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Enqueue script and styles */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_scripts_styles')) {

	function giornale_scripts_styles() {

		$googleFontsArgs = array(
			'family' =>	str_replace('|', '%7C','Work+Sans:100,200,300,400,500,600,700,800,900|Roboto+Slab:100,200,300,400,500,600,700,800,900'),
			'subset' =>	'latin,latin-ext'
		);

		wp_enqueue_style('dashicons');
		wp_enqueue_style('google-fonts', add_query_arg ( $googleFontsArgs, "https://fonts.googleapis.com/css" ), array(), '1.0.0' );
		wp_enqueue_style('bootstrap', get_template_directory_uri() . '/assets/css/bootstrap.css', array(), '3.3.7' );
		wp_enqueue_style('font-awesome', get_template_directory_uri() . '/assets/css/font-awesome.css', array(), '4.7.0' );
		wp_enqueue_style('swipebox', get_template_directory_uri() . '/assets/css/swipebox.css', array(), '1.3.0' );
		wp_enqueue_style('giornale-style', get_stylesheet_uri(), array() );
		wp_enqueue_style('giornale-woocommerce', get_template_directory_uri() . '/assets/css/giornale-woocommerce.css', array(), '1.0.0' );

		wp_enqueue_style(
			'giornale-' . esc_attr(get_theme_mod('giornale_skin', 'orange')),
			get_template_directory_uri() . '/assets/skins/' . esc_attr(get_theme_mod('giornale_skin', 'orange')) . '.css',
			array( 'giornale-style' ),
			'1.0.0'
		);

		wp_enqueue_script( 'giornale-navigation', get_template_directory_uri() . '/assets/js/navigation.js' , array('jquery'), '1.0', TRUE );

		wp_localize_script( 'giornale-navigation', 'accessibleNavigationScreenReaderText', array(
			'expandMain'   => __( 'Open the main menu', 'giornale' ),
			'collapseMain' => __( 'Close the main menu', 'giornale' ),
			'expandChild'   => __( 'expand submenu', 'giornale' ),
			'collapseChild' => __( 'collapse submenu', 'giornale' ),
		));

		wp_enqueue_script(
			'jquery.ticker',
			get_template_directory_uri() . '/assets/js/jquery.ticker.js',
			array('jquery'),
			'1.2.1',
			TRUE
		);

		wp_enqueue_script('jquery-easing', get_template_directory_uri() . '/assets/js/jquery.easing.js' , array('jquery'), '1.3', TRUE );
		wp_enqueue_script('jquery-nicescroll', get_template_directory_uri() . '/assets/js/jquery.nicescroll.js' , array('jquery'), '3.7.6', TRUE );
		wp_enqueue_script('jquery-swipebox', get_template_directory_uri() . '/assets/js/jquery.swipebox.js' , array('jquery'), '1.4.4', TRUE );
		wp_enqueue_script('jquery-touchSwipe', get_template_directory_uri() . '/assets/js/jquery.touchSwipe.js' , array('jquery'), '1.6.18', TRUE );
		wp_enqueue_script('fitvids', get_template_directory_uri() . '/assets/js/jquery.fitvids.js' , array('jquery'), '1.1', TRUE );
		wp_enqueue_script('giornale-template',get_template_directory_uri() . '/assets/js/giornale-template.js',array('jquery', 'imagesloaded', 'masonry'), '1.0.0', TRUE );

		if ( is_singular() ) wp_enqueue_script( 'comment-reply' );

		wp_enqueue_script('html5shiv', get_template_directory_uri().'/assets/scripts/html5shiv.js', FALSE, '3.7.3');
		wp_script_add_data('html5shiv', 'conditional', 'IE 8' );
		wp_enqueue_script('selectivizr', get_template_directory_uri().'/assets/scripts/selectivizr.js', FALSE, '1.0.3b');
		wp_script_add_data('selectivizr', 'conditional', 'IE 8' );

	}

	add_action( 'wp_enqueue_scripts', 'giornale_scripts_styles' );

}

/*-----------------------------------------------------------------------------------*/
/* Theme setup */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_setup')) {

	function giornale_setup() {

		global $content_width;

		if ( !isset($content_width) )
			$content_width = 1170;

		load_theme_textdomain( 'giornale', get_template_directory() . '/languages');

		add_theme_support( 'post-formats', array( 'aside','gallery','quote','video','audio','link','status','chat','image' ) );
		add_theme_support( 'automatic-feed-links' );
		add_theme_support( 'post-thumbnails' );
		add_theme_support( 'woocommerce' );
		add_theme_support( 'title-tag' );
		add_theme_support( 'custom-logo');

		add_theme_support( 'custom-background', array(
			'default-color' => 'ffffff',
		));

		add_image_size( 'giornale_blog_thumbnail', 790, 550, TRUE );
		add_image_size( 'giornale_recent_post_image', 650, 450, TRUE );
		add_image_size( 'giornale_trending_image', 140, 140, TRUE );

		register_nav_menu( 'main-menu', esc_attr__( 'Main Menu', 'giornale' ) );
		register_nav_menu( 'secondary-menu', esc_attr__( 'Secondary Menu', 'giornale' ) );

		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/aside-format.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/default-format.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/image-format.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/link-format.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/page-format.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/product-format.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/post-formats/quote-format.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/post/related-post.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/templates/after-content.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/before-content.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/masonry.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/media.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/news-ticker.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/pagination.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/post-author.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/post-formats.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/related-posts.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/social_links.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/title.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/templates/top-section.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/sidebars/bottom-sidebar.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/sidebars/footer-sidebar.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/sidebars/header-sidebar.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/sidebars/scroll-sidebar.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/sidebars/side-sidebar.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/includes/class-customize.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/includes/class-metaboxes.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/includes/class-notice.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/includes/class-plugin-activation.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/includes/class-welcome-page.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/admin/customize/customize.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/functions/function-required-plugins.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/functions/function-style.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/functions/function-widgets.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/functions/function-woocommerce.php' );

		require_once( trailingslashit( get_template_directory() ) . '/core/metaboxes/page.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/metaboxes/post.php' );
		require_once( trailingslashit( get_template_directory() ) . '/core/metaboxes/product.php' );


	}

	add_action( 'after_setup_theme', 'giornale_setup' );

}

/*-----------------------------------------------------------------------------------*/
/* Exclude featured posts on homepage */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_exclude_recent_posts_on_home')) {

	function giornale_exclude_recent_posts_on_home(&$query) {

		if (
			$query->is_home() &&
			$query->is_main_query() &&
			(
				giornale_setting('giornale_enable_recent_posts', true) == true &&
				giornale_setting('giornale_exclude_recent_posts_on_loop', false) == true
			)
		) {

			$offset = 4;

			$ppp = get_option('posts_per_page');

			if ( $query->is_paged ) {

				$page_offset = $offset + ( ($query->query_vars['paged']-1) * $ppp );
				$query->set('offset', $page_offset );

			}
			else {
				$query->set('offset',$offset);
			}

		}

	}

	add_action('pre_get_posts', 'giornale_exclude_recent_posts_on_home', 1 );

}

/*-----------------------------------------------------------------------------------*/
/* Adjust offset pagination */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_adjust_offset_pagination')) {

	function giornale_adjust_offset_pagination($found_posts, $query) {

		$offset = 4;

		if (
			$query->is_home() &&
			$query->is_main_query() &&
			(
				giornale_setting('giornale_enable_recent_posts', true) == true &&
				giornale_setting('giornale_exclude_recent_posts_on_loop', false) == true
			)
		) {
			return $found_posts - $offset;
		}

		return $found_posts;

	}

	add_filter('found_posts', 'giornale_adjust_offset_pagination', 1, 2 );

}

/*-----------------------------------------------------------------------------------*/
/* Get categories */
/*-----------------------------------------------------------------------------------*/ 

if (!function_exists('giornale_get_categories')) {

	function giornale_get_categories() {

		$args = array(
			'taxonomy' => 'category',
			'hide_empty' => true,
		);

		$return['all'] = esc_html__( 'All categories', 'giornale' );

		foreach ( get_terms($args) as $cat) {
			$return[$cat->term_id] = $cat->name;
		}
		
		return $return;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Get current date */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_get_current_date')) {

	function giornale_get_current_date() {

			$date_format = esc_attr(giornale_setting('giornale_topbar_date_format', 'l, F j Y'));

			if(strtotime(wp_date($date_format))) {
				$current_date = wp_date($date_format);
			} else {
				$current_date = wp_date('l, F j Y');
			}

			return $current_date;

	}

}

/*-----------------------------------------------------------------------------------*/
/* Get topbar right column */
/*-----------------------------------------------------------------------------------*/

if (!function_exists('giornale_get_topbar_right_column')) {

	function giornale_get_topbar_right_column() {

			$html = '';

			$right_column = giornale_setting('giornale_topbar_rightcolumn', 'topbar_text');

			switch ($right_column) {

				case 'social_icons':

					$html = giornale_get_social_icons();
				
				break;
				
				case 'secondary_menu':

					$html = '<div id="secondary-menu-wrapper">';

						$html .= '<button class="menu-toggle" aria-controls="top-menu" aria-expanded="false" type="button">';
							$html .= '<span aria-hidden="true">'.esc_html__( 'Menu', 'giornale' ).'</span>';
							$html .= '<span class="dashicons" aria-hidden="true"></span>';
						$html .= '</button>';

					$html .= '<nav id="secondary-menu" class="header-menu" >';
						$html .= wp_nav_menu(array('theme_location' => 'secondary-menu', 'container' => false, 'echo' => false));
					$html .= '</nav>';
					
					$html .= '</div>';

				break;
				
				default :
				case 'topbar_text':

					$html = '<div id="topbar-text">';

						$html .= esc_html(giornale_setting('giornale_topbar_text', __( 'Lorem ipsum dolor sit amet, consectetur adipiscing elit.','giornale')));
					
					$html .= '</div>';

				break;
			
			}

			$htmlLength = strlen($html);

			return ($htmlLength > 0) ? '<div class="topbar-right-column">' . $html . '</div>' : '';

	}

}

?>