<?php
/**
 * Reading Time
 *
 * @package Fred
 */

/**
 * Get Post Reading Time from Post Meta
 *
 * @param int $post_id The post ID.
 */
function vs_get_post_reading_time( $post_id = null ) {
  if ( ! $post_id ) {
    $post_id = get_the_ID();
  }

  // Get existing post meta.
  $reading_time = get_post_meta( $post_id, '_vs_reading_time', true );

  // Calculate and save reading time, if there's no existing post meta.
  if ( ! $reading_time ) {
    $reading_time = vs_calculate_post_reading_time( $post_id );
    update_post_meta( $post_id, '_vs_reading_time', $reading_time );
  }

  return $reading_time;
}

/**
 * Calculate Post Reading Time in Minutes
 *
 * @param int $post_id The post ID.
 */
function vs_calculate_post_reading_time( $post_id = null ) {
  if ( ! $post_id ) {
    $post_id = get_the_ID();
  }
  $post_content     = get_post_field( 'post_content', $post_id );
  $strip_shortcodes = strip_shortcodes( $post_content );
  $strip_tags       = strip_tags( $strip_shortcodes );
  $locale           = vs_get_locale();

  if ( 'ru_RU' === $locale ) {
    $word_count = count( preg_split( '/\s+/', $strip_tags ) );
  } else {
    $word_count = str_word_count( $strip_tags );
  }

  $reading_time = intval( ceil( $word_count / 265 ) );

  // Filter for Reading Time.
  if ( function_exists( 'iconv_strlen') ) {
    $reading_time = apply_filters( 'vs_calculate_reading_time', $reading_time, iconv_strlen( $strip_tags ) );
  } else {
    $reading_time = apply_filters( 'vs_calculate_reading_time', $reading_time, mb_strlen( $strip_tags ) );
  }

  return $reading_time;
}

/**
 * Get locale in uniform format.
 */
function vs_get_locale() {

  $locale = get_locale();

  if ( preg_match( '#^[a-z]{2}\-[A-Z]{2}$#', $locale ) ) {
    $locale = str_replace( '-', '_', $locale );
  } elseif ( preg_match( '#^[a-z]{2}$#', $locale ) ) {
    if ( function_exists( 'mb_strtoupper' ) ) {
      $locale .= '_' . mb_strtoupper( $locale, 'UTF-8' );
    } else {
      $locale .= '_' . mb_strtoupper( $locale );
    }
  }

  if ( empty( $locale ) ) {
    $locale = 'en_US';
  }

  return apply_filters( 'vs_locale', $locale );
}

/**
 * Update Post Reading Time on Post Save
 *
 * @param int  $post_id The post ID.
 * @param post $post    The post object.
 * @param bool $update  Whether this is an existing post being updated or not.
 */
function update_post_reading_time( $post_id, $post, $update ) {
  if ( ! $post_id ) {
    $post_id = get_the_ID();
  }
  $reading_time = vs_calculate_post_reading_time( $post_id );

  update_post_meta( $post_id, '_vs_reading_time', $reading_time );
}
add_action( 'save_post', 'update_post_reading_time', 10, 3 );
