<?php
/**
 * Custom template tags for this theme.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package Expert Adventure Guide
 */

if ( ! function_exists( 'expert_adventure_guide_posted_on' ) ) :
/**
 * Prints HTML with meta information for the current post-date/time and author.
 */
function expert_adventure_guide_posted_on() {
	$expert_adventure_guide_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
	if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
		$expert_adventure_guide_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
	}

	$expert_adventure_guide_time_string = sprintf( $expert_adventure_guide_time_string,
		esc_attr( get_the_date( 'c' ) ),
		esc_html( get_the_date() ),
		esc_attr( get_the_modified_date( 'c' ) ),
		esc_html( get_the_modified_date() )
	);

	$expert_adventure_guide_posted_on = sprintf(
		/* translators: %s: Date. */
		esc_html_x( 'Posted on %s', 'post date', 'expert-adventure-guide' ),
		'<a href="' . esc_url( get_permalink() ) . '" rel="bookmark">' . $expert_adventure_guide_time_string . '</a>'
	);

	$expert_adventure_guide_byline = sprintf(
		/* translators: %s: by. */
		esc_html_x( 'by %s', 'post author', 'expert-adventure-guide' ),
		'<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">' . esc_html( get_the_author() ) . '</a></span>'
	);
	echo '<span class="posted-on">' . $expert_adventure_guide_posted_on . '</span><span class="byline"> ' . $expert_adventure_guide_byline . '</span>'; // WPCS: XSS OK.
}
endif;


if ( ! function_exists( 'expert_adventure_guide_entry_footer' ) ) :
/**
 * Prints HTML with meta information for the categories, tags and comments.
 */
function expert_adventure_guide_entry_footer() {
	// Hide category and tag text for pages.
	if ( 'post' === get_post_type() ) {
		/* translators: used between list items, there is a space after the comma */
		$expert_adventure_guide_categories_list = get_the_category_list( esc_html__( ', ', 'expert-adventure-guide' ) );
		if ( $expert_adventure_guide_categories_list && expert_adventure_guide_categorized_blog() ) {
			/* translators: %1$s: Posted. */
			printf( '<span class="cat-links">' . esc_html__( 'Posted in %1$s', 'expert-adventure-guide' ) . '</span>', $expert_adventure_guide_categories_list ); // WPCS: XSS OK.
		}

		/* translators: used between list items, there is a space after the comma */
		$expert_adventure_guide_tags_list = get_the_tag_list( '', esc_html__( ', ', 'expert-adventure-guide' ) );
		if ( $expert_adventure_guide_tags_list ) {
			/* translators: %1$s: Tagged. */
			printf( '<span class="tags-links">' . esc_html__( 'Tagged %1$s', 'expert-adventure-guide' ) . '</span>', $expert_adventure_guide_tags_list ); // WPCS: XSS OK.
		}
	}

	if ( ! is_single() && ! post_password_required() && ( comments_open() || get_comments_number() ) ) {
		echo '<span class="comments-link">';
		/* translators: %s: post title */
		comments_popup_link( sprintf( wp_kses( __( 'Leave a Comment<span class="screen-reader-text"> on %s</span>', 'expert-adventure-guide' ), array( 'span' => array( 'class' => array() ) ) ), get_the_title() ) );
		echo '</span>';
	}

	edit_post_link(
		sprintf(
			/* translators: %s: Name of current post */
			esc_html__( 'Edit %s', 'expert-adventure-guide' ),
			the_title( '<span class="screen-reader-text">"', '"</span>', false )
		),
		'<span class="edit-link">',
		'</span>'
	);
}
endif;

/**
 * Returns true if a blog has more than 1 category.
 *
 * @return bool
 */
function expert_adventure_guide_categorized_blog() {
	if ( false === ( $expert_adventure_guide_all_the_cool_cats = get_transient( 'expert_adventure_guide_categories' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$expert_adventure_guide_all_the_cool_cats = get_categories( array(
			'fields'     => 'ids',
			'hide_empty' => 1,
			// We only need to know if there is more than one category.
			'number'     => 2,
		) );

		// Count the number of categories that are attached to the posts.
		$expert_adventure_guide_all_the_cool_cats = count( $expert_adventure_guide_all_the_cool_cats );

		set_transient( 'expert_adventure_guide_categories', $expert_adventure_guide_all_the_cool_cats );
	}

	if ( $expert_adventure_guide_all_the_cool_cats > 1 ) {
		// This blog has more than 1 category so expert_adventure_guide_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so expert_adventure_guide_categorized_blog should return false.
		return false;
	}
}

/**
 * Flush out the transients used in expert_adventure_guide_categorized_blog.
 */
function expert_adventure_guide_category_transient_flusher() {
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}
	// Like, beat it. Dig?
	delete_transient( 'expert_adventure_guide_categories' );
}
add_action( 'edit_category', 'expert_adventure_guide_category_transient_flusher' );
add_action( 'save_post',     'expert_adventure_guide_category_transient_flusher' );

/**
 * Register Google fonts.
 */
function expert_adventure_guide_google_font() {
	$font_url      = '';
	$expert_adventure_guide_font_family   = array(
		'Playfair Display:ital,wght@0,400;0,500;0,600;0,700;0,800;0,900;1,400;1,500;1,600;1,700;1,800;1,900',
		'Poppins:ital,wght@0,100;0,200;0,300;0,400;0,500;0,600;0,700;0,800;0,900;1,100;1,200;1,300;1,400;1,500;1,600;1,700;1,800;1,900'
	);

	$expert_adventure_guide_fonts_url = add_query_arg( array(
		'family' => implode( '&family=', $expert_adventure_guide_font_family ),
		'display' => 'swap',
	), 'https://fonts.googleapis.com/css2' );

	$expert_adventure_guide_contents = wptt_get_webfont_url( esc_url_raw( $expert_adventure_guide_fonts_url ) );
	return $expert_adventure_guide_contents;
}

function expert_adventure_guide_scripts_styles() {
    wp_enqueue_style( 'expert-adventure-guide-fonts', expert_adventure_guide_google_font(), array(), null );
}
add_action( 'wp_enqueue_scripts', 'expert_adventure_guide_scripts_styles' );

/**
 * Register Breadcrumb for Multiple Variation
 */
function expert_adventure_guide_breadcrumbs_style() {
	get_template_part('./template-parts/sections/section','breadcrumb');
}

/**
 * This Function Check whether Sidebar active or Not
 */
if(!function_exists( 'expert_adventure_guide_post_layout' )) :
function expert_adventure_guide_post_layout(){
	if(is_active_sidebar('expert-adventure-guide-sidebar-primary'))
		{ echo 'col-lg-8'; } 
	else 
		{ echo 'col-lg-12'; }  
} endif;


// offer Meta
function expert_adventure_guide_bn_custom_meta_offer() {
    add_meta_box( 'bn_meta', __( 'Trip Offer Meta Feilds', 'expert-adventure-guide' ), 'expert_adventure_guide_meta_callback_trip_offer', 'post', 'normal', 'high' );
}
/* Hook things in for admin*/
if (is_admin()){
  add_action('admin_menu', 'expert_adventure_guide_bn_custom_meta_offer');
}

function expert_adventure_guide_meta_callback_trip_offer( $post ) {
    wp_nonce_field( basename( __FILE__ ), 'expert_adventure_guide_offer_trip_meta_nonce' );
    $expert_adventure_guide_trip_amount = get_post_meta( $post->ID, 'expert_adventure_guide_trip_amount', true );
    $expert_adventure_guide_trip_days = get_post_meta( $post->ID, 'expert_adventure_guide_trip_days', true );

    // Set default values if no value is found
    $expert_adventure_guide_trip_amount = !empty( $expert_adventure_guide_trip_amount ) ? $expert_adventure_guide_trip_amount : '$399';
    $expert_adventure_guide_trip_days = !empty( $expert_adventure_guide_trip_days ) ? $expert_adventure_guide_trip_days : '7 days 6 nights';
    ?>
    <div id="testimonials_custom_stuff">
        <table id="list">
            <tbody id="the-list" data-wp-lists="list:meta">
                <tr id="meta-8">
                    <td class="left">
                        <?php esc_html_e( 'Package Amount', 'expert-adventure-guide' )?>
                    </td>
                    <td class="left">
                        <input type="text" name="expert_adventure_guide_trip_amount" id="expert_adventure_guide_trip_amount" value="<?php echo esc_attr( $expert_adventure_guide_trip_amount ); ?>" />
                    </td>
                </tr>
                <tr id="meta-8">
                    <td class="left">
                        <?php esc_html_e( 'Package Days', 'expert-adventure-guide' )?>
                    </td>
                    <td class="left">
                        <input type="text" name="expert_adventure_guide_trip_days" id="expert_adventure_guide_trip_days" value="<?php echo esc_attr( $expert_adventure_guide_trip_days ); ?>" />
                    </td>
                </tr>
            </tbody>
        </table>
    </div>
    <?php
}

/* Saves the custom meta input */
function expert_adventure_guide_bn_metadesig_save( $post_id ) {
    if (!isset($_POST['expert_adventure_guide_offer_trip_meta_nonce']) || !wp_verify_nonce( strip_tags( wp_unslash( $_POST['expert_adventure_guide_offer_trip_meta_nonce']) ), basename(__FILE__))) {
        return;
    }

    if (!current_user_can('edit_post', $post_id)) {
        return;
    }

    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }

    // Save Trip Amount
    if( isset( $_POST[ 'expert_adventure_guide_trip_amount' ] ) ) {
        update_post_meta( $post_id, 'expert_adventure_guide_trip_amount', strip_tags( wp_unslash( $_POST[ 'expert_adventure_guide_trip_amount' ]) ) );
    }
    // Save Trip Days
    if( isset( $_POST[ 'expert_adventure_guide_trip_days' ] ) ) {
        update_post_meta( $post_id, 'expert_adventure_guide_trip_days', strip_tags( wp_unslash( $_POST[ 'expert_adventure_guide_trip_days' ]) ) );
    }
}
add_action( 'save_post', 'expert_adventure_guide_bn_metadesig_save' );