<?php 

// UPDATED 2022 04 24
// NOTE: All html MUST return sanitized content

// These functions are theme related template functions
// Calculates and fetches WordPress elements to template actions and filters
// The global $post object must to be setup correctly
// As every developer should do working with output post and page data
// Following standard query setup and reset post data, 
// you can use this functions anywhere in plugins, widgets or modules
// inside or outside 'the loop'
// As we pass the '$post' as '$obj' namespace between the calls

// This file also contains template setup functions and some global theme variables
// Globals the right way are not evil

function entex_theme_fn_get_main_id(){
	$page_id = get_queried_object_id();
	if($page_id) return $page_id;
		else return 0;
}

// Consider ... get_post_object...
// All are post setup related, make shure the obj chain never breaks
function entex_theme_fn_get_object($obj = null){
	if(!$obj){
		global $post;
		if(!empty($post)) $obj = get_post($post);
	} else $obj = get_post($obj);
	if(!$obj) return null;
		else return $obj;
}



/* -------------------- */
// ARTICLE IS CURRENT ID
/* In use when filter out meta information on each entry in a loop */
/* We dont need to show category info like 'News' on listing posts */
/* If we are in the category (archive) 'News' */
/* However, if the post belongs to a child category of 'News' */
/* The category meta information will show as normal */

/* relative from singular pageload or archive pageload */
function entex_theme_fn_is_current_id($id = null){
	if($id == null) $id = get_the_ID();
	$page_id = get_queried_object_id();
	if($id == $page_id) return ' ua-is-current-id';
		else return '';
}

/* relative from archive pageload */
function entex_theme_fn_is_current_term_id($id = null){ 
	if($id == null) $id = get_the_ID();
	$parent_id = entex_theme_fn_get_parent_id($id);
	if(!$parent_id) return '';
	$term_id = get_queried_object_id();
	if($parent_id == $term_id) return ' ua-is-current-id';
		else return '';
}


/* -------------------- */
// HOME

function entex_theme_fn_get_blogname(){
	return trim(strip_tags(get_bloginfo('name')));
}


function entex_theme_fn_get_home_link(){
	return '<a rel="home" href="'. esc_url(home_url('/')) .'">'. get_bloginfo('name') .'</a>';
}







/* -------------------- */
// TITLES

/* current page-load title */
/* argumnent not in use but passed anyway */
function entex_theme_fn_get_the_current_title($title = ''){ 

	/* contextual core */
	$current = entex_theme_fn_get_is();
	
	if(is_author()) $current['filtered-title'] = '<span class="vcard">'. esc_html($current['filtered-title']) .'</span>';
		else $current['filtered-title'] = '<span>'. esc_html($current['filtered-title']) .'</span>';
	$current['extended-title-above'] = $current['extended-title-above'] ? '<em>'. esc_html($current['extended-title-above']) .'</em> ' : '';
	$current['extended-title-below'] = $current['extended-title-below'] ? ' <em>'. esc_html($current['extended-title-below']) .'</em>' : '';
	$title = $current['extended-title-above'].$current['filtered-title'].$current['extended-title-below'];
	
	return $title;
}

/* entries and titles outside the loop compatible */
/* arguments should be passed but not necessary, as this mostly is inside a loop */
/* arguments are in the filter the_title order, as it can be use as a global filter as well */
function entex_theme_fn_get_the_entry_title($title = '', $post_id = null){
	
	if(!$post_id) $post_id = get_the_ID();
	if(!$title){
		$title = get_the_title($post_id);
		if(!$title) return _x('ERR: Doing it wrong - No title saved in post', 'DO NOT TRANSLATE', 'entex');
	}
	
	/* module || plugin || manually filtered in */
	if(!function_exists('entex_theme_get_extended_main_title')) return '<span>'.esc_html($title).'</span>';

	$current['extended-title-above'] = '';
	$current['filtered-title'] = '';
	$current['extended-title-below'] = '';
	
	$position = entex_theme_get_extended_main_title_position($post_id);
	if($position) $current['extended-title-above'] = entex_theme_get_extended_main_title($post_id);
		else $current['extended-title-below'] = entex_theme_get_extended_main_title($post_id);
	
	$current['filtered-title'] = '<span>'.esc_html($title).'</span>';
	$current['extended-title-above'] = $current['extended-title-above'] ? '<em>'. esc_html($current['extended-title-above']) .'</em> ' : '';
	$current['extended-title-below'] = $current['extended-title-below'] ? ' <em>'. esc_html($current['extended-title-below']) .'</em>' : '';
	$title = $current['extended-title-above'].$current['filtered-title'].$current['extended-title-below'];
	
	return $title;

}

/* returns the extended title only */
function entex_theme_fn_get_extended_title($post_id = null){
	
	if(!$post_id) $post_id = get_the_ID();

	/* module */
	if(!function_exists('entex_theme_get_extended_main_title')) return '';

	$extended = entex_theme_get_extended_main_title($post_id);
	if(!$extended) return '';
		else return esc_html($extended);
}



/* -------------------- */
// AJAX

function entex_theme_fn_get_share_link_hover(){

	$postType = get_post_type_object(get_post_type());
	if(isset($postType->labels->singular_name)) $label = $postType->labels->singular_name;
		else $label = $postType->name;
	/* translators: */
	return esc_html(sprintf(_x('Share this %s', 'Meta: Hover meta share link LIKE: Share this [ post type ]', 'entex'), $label));
}

function entex_theme_fn_get_share_link(){
	$content = '';
	$title = entex_theme_fn_get_share_link_hover();
	$content .= '<a href="'. esc_url(get_permalink()) .'" class="ua-ajax-link ua-ajax-share" title="'. esc_attr($title) .'" data-post-id="'. esc_attr(get_the_ID()) .'">'. esc_html_x('Share this', 'Meta: Share this label', 'entex') .'</a>';
	return $content;
}



/* -------------------- */
// AUTHOR

function entex_theme_fn_get_author_link_hover(){
	/* translators: */
	return esc_html(sprintf(_x('View all entries posted by %s', 'Meta: Hover author link', 'entex'), get_the_author()));
}

function entex_theme_fn_get_author_link(){
	$content = '';
	$title = entex_theme_fn_get_author_link_hover();
	$content .= '<span class="ua-author author vcard">';
	$content .= '<a href="'. esc_url(get_author_posts_url(get_the_author_meta('ID'))) .'" class="url fn n" title="'. esc_attr($title) .'">'. get_the_author() .'</a>';
	$content .= '</span>';
	return $content;
}

function entex_theme_fn_get_author_avatar($size = 0, $link = false, $args = null){
	if(!is_array($args)){
		$args = array();
		if($size){
			if(!is_numeric($size)){
				$args = array(
					'ua_size' => $size
				);
			}
		} else $size = 96;
	} else $size = 96;
	$avatar = get_avatar(get_the_author_meta('user_email'), $size, '', get_the_author(), $args);
	
	if($avatar && $link){
		$title = entex_theme_fn_get_author_link_hover();
		$avatar = '<a href="'. esc_url(get_author_posts_url(get_the_author_meta('ID'))) .'" class="url fn n" title="'. esc_attr($title) .'">'. $avatar .'</a>';
	}
	return $avatar;
}






/* -------------------- */
// PREAMBLE
/* returns the occurence of preamble class attribute of block content */
/* but only if preamble is the first (not empty) block */
/* Note, as default only the text content is returned, no styles or markup */
/* In sync with entex-setup-gutenberg.php (entex_theme_render_block_core()) */

function entex_theme_fn_get_preamble($element = true){
	
	if(is_admin()) return '';
	if(!function_exists('parse_blocks')) return '';
	if(defined('ENTEX_CLASSIC_EDITOR_INTERFACE')) return '';
	
	$html = '';
	global $post;

	$GLOBALS['ENTEX-SIDEBAR-BLOCKS-PARSING'] = true;
	$blocks = parse_blocks($post->post_content);
	if(!empty($blocks)){
		foreach($blocks as $block){
			/* do not accept left overs or empty paragraphs from the editor */
			if((!$block['blockName'] && !$block['innerHTML']) || trim($block['innerHTML']) == '<p></p>') continue;
			if($block['blockName'] == 'core/paragraph'){
				if(isset($block['attrs']['className']) && $block['attrs']['className']){
					if(strpos($block['attrs']['className'], 'is-style-ua-preamble') !== FALSE){
						$block_content = $block['innerHTML'];
						if($block_content && trim($block_content)){
							if($element) $html = $block_content;
								else $html = strip_tags($block_content);
						}
					}
				}
			}
			/* bail if first accepted block not preamble */
			break;
		}
	}
	$GLOBALS['ENTEX-SIDEBAR-BLOCKS-PARSING'] = false;
	return $html;
}






/* -------------------- */
// ARCHIVES

function entex_theme_fn_get_archive_image($size = 'post-thumbnail', $link = false){
	
	$image_id = entex_theme_get_meta_archive_image();
	if(!$image_id) return '';
	$image = wp_get_attachment_image($image_id, $size);
	if($image) return $image;
		else return '';
}



/* -------------------- */
// EDIT POST LINK

function entex_theme_fn_get_edit_post_link(){
	$url = get_edit_post_link();
	if(!$url) return '';
	$_TARGET = '';
	if(!defined('ENTEX_THEME_EDIT_POST_LINK_TARGET_FALSE')){
		$_TARGET = ' target="_blank"';
	}
	$link = '<a'. $_TARGET .' class="post-edit-link" href="'. esc_url($url) .'">'. esc_html_x('Edit', 'Post edit link', 'entex') .'</a>';
	return $link;
}



/* -------------------- */
// EDIT COMMENT LINK

function entex_theme_fn_get_edit_comment_link(){
	$url = get_edit_comment_link();
	if(!$url) return '';
	$_TARGET = '';
	if(!defined('ENTEX_THEME_EDIT_COMMENT_LINK_TARGET_FALSE')){
		$_TARGET = ' target="_comments"';
	}
	$link = '<a'. $_TARGET .' class="post-edit-link" href="'. esc_url($url) .'">'. esc_html_x('Edit', 'Post edit link', 'entex') .'</a>';
	return $link;
}



/* -------------------- */
// POST THUMBNAIL
/* This only validate */

function entex_theme_fn_show_post_thumbnail($post_id = null, $paged = false, $password = true, $meta = true){
	
	if(!$post_id) $post_id = get_the_ID();
	if(!$post_id) return '';
	
	if(!has_post_thumbnail($post_id)) return '';
	if($password && post_password_required()) return '';
	
	if($paged){
		$args = entex_get_multipage();
		if($args['page'] > 1) return '';
	}
	
	/* Theme custom field */
	if($meta && function_exists('entex_theme_get_meta_post_thumbnail_hide')){
		if(entex_theme_get_meta_post_thumbnail_hide()) return '';
	}
	
	return true;
}


/* -------------------- */
// ATTACHMENT
// sanitized by the get function

function entex_theme_fn_get_post_thumbnail_meta(){
	$attachment_id = get_post_thumbnail_id();
	$meta = wp_get_attachment_metadata($attachment_id);
	return $meta;
}

function entex_theme_fn_get_attachment_mime(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-mime'];
}

function entex_theme_fn_get_attachment_media_summary(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-media-summary'];
}

function entex_theme_fn_get_attachment_summary(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-summary'];
}

function entex_theme_fn_get_attachment_filename(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-filename'];
}

function entex_theme_fn_get_attachment_extension(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-extension'];
}

function entex_theme_fn_get_attachment_filesize(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-filesize'];
}

function entex_theme_fn_get_attachment_dimension(){
	global $post;
	/* cached */
	$attr = entex_fn_extract_attachment($post->ID, $max = 600);
	return $attr['data-file-dimension'];
}

function entex_theme_fn_get_attachment_parent(){
	global $post;
	if($post->post_parent) $content = get_the_title($post->post_parent);
		else $content = '';
	return $content;
}

function entex_theme_fn_get_attachment_parent_link(){
	global $post;
	if($post->post_parent){
		$content = get_the_title($post->post_parent);
		$content = '<a href="'.esc_url(get_permalink($post->post_parent)).'">'.$content.'</a>';
	} else $content = '';
	return $content;
}

function entex_theme_fn_get_post_thumbnail_meta_latlng(){
	global $post;
	if($post->post_type != 'attachment'){
		$attachment_id = get_post_thumbnail_id();
		if(!$attachment_id) return '';
	} else {
		$attachment_id = $post->ID;
	}
	/* cached */
	$attr = entex_fn_extract_attachment($attachment_id, $max = 600);
	return $attr['data-file-location'];
}



/* -------------------- */
// POST THUMBNAIL WRAPPER
/* The 'post thumbnail' wrapper, not the image element */
function entex_theme_fn_get_post_thumbnail_classes($size = null){
	
	global $post;
	$classes[] = 'post-thumbnail';
	if($size) $classes[] = 'ua-size-'.$size;
	
	/* validate featured image size */
	if($size && has_post_thumbnail()){
		$post_thumbnail_id = get_post_thumbnail_id();
		if($post_thumbnail_id){
			/* cached */
			/* note: the attachment post thumbnail id */
			$attr = entex_fn_extract_attachment($post_thumbnail_id, $max = apply_filters('entex_theme_filter_extract_attachment_max_value', 600));
			if(!isset($attr['data-file-size-'.$size])){
				$classes[] = 'ua-attachment-size-false';
			} else {
				$classes[] = 'ua-attachment-size-'.$size;
			}
		}
	}
	
	if(get_post_type() == 'attachment'){
		
		/* cached */
		/* note: get_the_ID() - attachment id - current 'post object' id */
		$attr = entex_fn_extract_attachment(get_the_ID(), $max = apply_filters('entex_theme_filter_extract_attachment_max_value', 600));
		
		if($size && !has_post_thumbnail() && $attr['data-file-media-type'] == 'image'){
			if(!isset($attr['data-file-size-'.$size])){
				$classes[] = 'ua-attachment-size-false';
			} else {
				$classes[] = 'ua-attachment-size-'.$size;
			}
		}
		/* same as has_post_thumbnail() */
		if($attr['data-file-post-thumbnail']) $classes[] = 'ua-attachment-post-thumbnail';
		
		if($attr['data-file-media-type'] == 'image'){
			$classes[] = 'ua-attachment-image';
		} else if($attr['data-file-media-type'] == 'video'){
			$classes[] = 'ua-attachment-video';
		} else if($attr['data-file-media-type'] == 'audio'){
			$classes[] = 'ua-attachment-audio';
		} else {
			$classes[] = 'ua-attachment-media';
		}
	} else {
		$classes[] = 'ua-attachment-'.get_post_type();
	}
	return $classes;
}




/* -------------------- */
// ATTACHMENT SUMMARY VARIABLE
// -- ATTENTION: This must recive the attachment id, not the 'post' id -- //
// This function asume that the current $post object is attachment 
/* $max 60 Backend (table thumbnail), $max 600 Frontend (main area min-width) */
/* available filter 'entex_theme_filter_extract_attachment_max_value' */
/* cached */
function entex_fn_extract_attachment($post_id = null, $max = 60){
	
	$output = '';
	if(!$post_id) $id = get_the_ID();
		else $id = $post_id;
		
	if($id){

		static $cache;
		if(isset($cache[$id][$max])){
			return $cache[$id][$max];
		} 
		
		/* do not use global $post object from here */
		
		$_MIME = get_post_mime_type($id);
		
		$image = false;
		if(in_array($_MIME, array('image/gif', 'image/png', 'image/jpeg', 'image/jpg', 'image/webp'))) $image = true;
		$mime = $_MIME;
		
		$url = wp_get_attachment_url($id);
		$file = basename($url);
		$filepath = str_replace($file, '', $url);
		$extension = substr(strrchr(basename($file), '.'), 1);
		$filename = $file;
		$filesize = '';

		$width = '';
		$height = '';
		$dimension = '';
		
		$meta = wp_get_attachment_metadata($id, true);
		if(isset($meta['filesize'])) $filesize = $meta['filesize'];
		if($image){
			
			$media = 'image';
			$media_type = 'image';
			$icon = '';
			$background = '';
			$repeat = 'no-repeat';
		
			if(has_post_thumbnail($id)){
				$post_thumbnail = get_the_post_thumbnail_url($id, 'post-thumbnail');
			} else $post_thumbnail = '';
					
			/* note: wp creates sizes if one of the dimensions fit */
			/* note: this thumbnail might have strange dimensions */
			if(isset($meta['sizes']['thumbnail']['file'])){
				$thumbnail = $meta['sizes']['thumbnail']['file'];
				$thumbnail = $filepath.$thumbnail;
			} else {
				$thumbnail = $url;
			}
			
			/* get true existsing sizes */
			/* current theme sizes must match */
			/* otherwise reported as not existsing */
			/* core function in use */
			/* the result ($images) is only merged with returnvalue if not empty */
			
			$image_sizes = entex_get_image_sizes();

			foreach($image_sizes as $this_size => $this_arr){
				if(isset($meta['sizes'][$this_size]['file'])){
					$valid = true;
					/* registered sizes might keep '0' or bool */
					if($this_arr['width'] && $this_arr['width'] != $meta['sizes'][$this_size]['width']){
						$valid = false;
					}
					if($this_arr['height'] && $this_arr['height'] != $meta['sizes'][$this_size]['height']){
						$valid = false;
					}
					if($valid){
						$this_image = $meta['sizes'][$this_size]['file'];
						$images['data-file-size-'.$this_size] = esc_url($filepath.$this_image);
					} else $placeholder = true;
				} else {
					$sniffer = '';
				}
			}
					
			if(isset($meta['width'])){
				if($max == 60){
					if($meta['width'] < $max || $meta['height'] < $max){
						$background = $url;
						if($meta['width'] < 5){
							$repeat = 'repeat';
						} else {
							if(strpos($filename, 'bg') !== false){
								$repeat = 'repeat';
							} else if(strpos($filename, 'background') !== false){
								$repeat = 'repeat';
							}
						}
						$placeholder = true;
					}
				} else {
					if($meta['width'] < $max){
						$background = $url;
						if($meta['width'] < 5){
							$repeat = 'repeat';
						} else {
							if(strpos($filename, 'bg') !== false){
								$repeat = 'repeat';
							} else if(strpos($filename, 'background') !== false){
								$repeat = 'repeat';
							}
						}
						$placeholder = true;
					}
				}
			}
					
			if(isset($meta['width'])){
				$width = $meta['width'];
				$height = $meta['height'];
				$dimension = $meta['width'] .' x '. $meta['height'];
			}
					
			/* entex theme */
			if(isset($meta['sizes']['normal']['file'])){
				$expanded = $meta['sizes']['normal']['file'];
				$expanded = $filepath.$expanded;
			} else if(isset($meta['sizes']['medium_large']['file'])){
				$expanded = $meta['sizes']['medium_large']['file'];
				$expanded = $filepath.$expanded;
			} else $expanded = $url;

			/* filter */ // GLOBAL
			$expanded = apply_filters('entex_image_data_expanded_url', $expanded);
			
			if(isset($meta['image_meta']['latlng'])) $location = $meta['image_meta']['latlng'];
				else $location = '';
				
			
				
		} else {
			
			$media = 'media';
			$media_type = strtok($_MIME, '/');
			
			$icon = wp_mime_type_icon($id);
			$background = $icon;
			$repeat = 'no-repeat';
			
			if(has_post_thumbnail($id)){
				$post_thumbnail = get_the_post_thumbnail_url($id, 'post-thumbnail');
				$thumbnail = get_the_post_thumbnail_url($id, 'thumbnail');
			} else {
				$post_thumbnail = '';
				$thumbnail = $icon;
				$placeholder = true;
			}
			
			$expanded = $icon;
			$location = '';
		}
		
		/* not in use yet */
		if(!$post_thumbnail && isset($placeholder)){
			$attr['data-file-placeholder'] = '';
		}
		
		if(!$filesize) $filesize = filesize(get_attached_file($id));
		$broken = false;
		if(!$filesize) $broken = true;
		
		$protected = '';
		$meta = get_post_meta($id, 'entex_protected_file', true);
		if($meta) $protected = true;
		
		if(current_user_can('edit_posts')) $edit = admin_url('/post.php').'?post='.$id.'&action=edit';
			else $edit = '';
		
		$summary = $extension.' / '.entex_formatBytes($filesize);
		
		
		// -- BEGIN COLLECT -- //
		
		$_DATA = '';
		
		/*
		$attr['data-file-parent'] = wp_get_post_parent_id($id);
		$attr['data-file-caption'] = esc_attr(strip_tags(get_the_excerpt($id)));
		$attr['data-file-description'] = esc_attr(strip_tags(get_the_content($id)));
		*/
		
		$attr['data-file-broken'] = $broken; // bool
		$attr['data-file-id'] = $id;
		$attr['data-file-media'] = $media;
		$attr['data-file-mime'] = $mime;
		$attr['data-file-media-type'] = $media_type;
		$attr['data-file-media-summary'] = $media_type .'/'. $extension;
		$attr['data-file-summary'] = esc_attr($summary);
		$attr['data-file-url'] = esc_url($url);
		$attr['data-file-expanded'] = esc_url($expanded);
		$attr['data-file-extension'] = $extension;
		$attr['data-file-filesize'] = entex_formatBytes($filesize);
		$attr['data-file-filename'] = esc_attr($filename);
		$attr['data-file-edit'] = $edit;
		$attr['data-file-location'] = $location;
		$attr['data-file-protected'] = $protected; // bool
		
		$attr['data-file-icon'] = esc_url($icon);
		$attr['data-file-thumbnail'] = esc_url($thumbnail);
		$attr['data-file-post-thumbnail'] = esc_url($post_thumbnail);
		$attr['data-file-background'] = $background;
		$attr['data-file-repeat'] = $repeat;
		
		$attr['data-file-width'] = $width;
		$attr['data-file-height'] = $height;
		$attr['data-file-dimension'] = $dimension;
		
		if(!empty($images)) $attr = array_merge($images, $attr);
		
		// -- END COLLECT -- //
		
		$cache[$id][$max] = $attr;
		return $attr;
		
	}
	return $output;
}



/* -------------------- */
// DATE

function entex_theme_fn_is_today(){ 
	if(current_time('Y-m-d') == get_the_date('Y-m-d')) return true;
		else return false;
}

function entex_theme_fn_is_future(){ 
	$time = get_the_date('Y-m-d').' '.get_the_time('H:i:s');
	if(strtotime($time) > strtotime(wp_date('Y-m-d H:i:s'))) return true;
		else return false;
}

function entex_theme_fn_is_event(){ 
	global $post;
	$published = strtotime($post->post_date);
	$filtered = strtotime(get_the_date('Y-m-d').' '.get_the_time('H:i:s'));
	if($filtered > $published) return true;
		else return false;
}

function entex_theme_fn_is_yesterday(){ 
	$date = new DateTime(current_time('Y-m-d'));
	$date->modify('yesterday');
	$yesterday = $date->format('Y-m-d');
	if($yesterday == get_the_date('Y-m-d')) return true;
		else return false;
}

function entex_theme_fn_is_tomorrow(){ 
	$date = new DateTime(current_time('Y-m-d'));
	$date->modify('tomorrow');
	$tomorrow = $date->format('Y-m-d');
	if($tomorrow == get_the_date('Y-m-d')) return true;
		else return false;
}

function entex_theme_fn_is_less_than($backwards = 365){ 
	$unix_posted_date = strtotime(get_the_date('Y-m-d').' '.get_the_time('H:i:s'));
	if((strtotime(wp_date('Y-m-d H:i:s')) - (86400 * $backwards)) < $unix_posted_date) return true;
		else return false;
}

function entex_theme_fn_is_older_than($backwards = 365){ 
	$unix_posted_date = strtotime(get_the_date('Y-m-d').' '.get_the_time('H:i:s'));
	if((strtotime(wp_date('Y-m-d H:i:s')) - (86400 * $backwards)) < $unix_posted_date) return false;
		else return true;
}

function entex_theme_fn_is_updated(){
	global $post;
	if($post->post_modified_gmt !== $post->post_date_gmt) return true;
		else return false;
}

/* unfiltered date and time */
function entex_theme_fn_get_datetime($modified = false, $format = 'c'){
	global $post;
	if($modified) $data = $post->post_modified_gmt;
		else $data = $post->post_date_gmt;
	$format = apply_filters('entex_theme_fn_get_datetime_format', $format, $modified);
	return wp_date($format, strtotime($data));
}

function entex_theme_fn_get_ago(){ 
	$unix_posted_date = strtotime(get_the_date('Y-m-d').' '.get_the_time('H:i:s'));
	$ago = human_time_diff($unix_posted_date, strtotime(wp_date('Y-m-d H:i:s')));
	return $ago;
}

function entex_theme_fn_get_within(){ 
	$unix_posted_date = strtotime(get_the_date('Y-m-d').' '.get_the_time('H:i:s'));
	$within = human_time_diff($unix_posted_date, strtotime(wp_date('Y-m-d H:i:s')));
	return $within;
}


function entex_theme_fn_get_time_ago(){ 

	$future = entex_theme_fn_is_future();
	
	if($future){
		/* translators: */
		$text = _x('Within %s', 'Time LIKE: Within 10 days', 'entex');
		/* filter */ // - placeholder
		$text = apply_filters('entex_theme_fn_get_time_within_text', $text);
		$text = sprintf($text, entex_theme_fn_get_within());
		
		/* translators: */
		return sprintf('<time class="ua-ago ua-time-within" datetime="%1$s">%2$s</time>', esc_attr(entex_theme_fn_get_datetime()), esc_html($text));
		
	} else {
		/* translators: */
		$text = _x('%s ago', 'Time LIKE: 10 days ago', 'entex');
		/* filter */ // - placeholder
		$text = apply_filters('entex_theme_fn_get_time_ago_text', $text, $future);
		$text = sprintf($text, entex_theme_fn_get_ago());
		
		/* translators: */
		return sprintf('<time class="ua-ago ua-time-ago" datetime="%1$s">%2$s</time>', esc_attr(entex_theme_fn_get_datetime()), esc_html($text));
	}

}

function entex_theme_fn_get_time(){
	
	$event = entex_theme_fn_is_event();
	
	if($event && get_the_time('H:i:s') == '00:00:01') $hour = _x('During day', 'Time LIKE: No time exists for date', 'entex');
		else $hour = get_the_time(_x('G : i a', 'Time LIKE: 10 : 15 am', 'entex'));
		
	/* filter */
	$text = apply_filters('entex_theme_fn_get_time_text', $hour, $event);
	
	/* translators: */
	if($text) return sprintf('<time class="ua-hour ua-time-hour'. ($event ? ' ua-time-event' : '') .'" datetime="%1$s">%2$s</time>', esc_attr(entex_theme_fn_get_datetime()), esc_html($text));
		else return '';
}

function entex_theme_fn_get_today(){ 

	$future = entex_theme_fn_is_future();
	$event = entex_theme_fn_is_event();
	
	/* filter */
	$text = apply_filters('entex_theme_fn_get_today_text', _x('Today', 'Time LIKE: Posted Today', 'entex'), $future, $event);
	/* translators: */
	return sprintf('<time class="ua-hour" datetime="%1$s">%2$s</time>', esc_attr(entex_theme_fn_get_datetime()), esc_html($text));
}

function entex_theme_fn_get_time_link(){ 

	$event = entex_theme_fn_is_event();

	$url = get_day_link(get_the_date('Y'), get_the_date('m/'), get_the_date('d/'));
	$date = get_the_date(_x('F j, Y', 'daily archives date format', 'entex'));
	
	/* The title attribute only provide the date information, NO TIME */
	
	/* translators: */
	if(get_post_format() == 'status') $hover = _x('View all updates posted on %s', 'Time LIKE: Posted on September 1, 2017', 'entex');
		/* translators: */
		else $hover = _x('View all entries posted on %s', 'Time LIKE: Posted on September 1, 2017', 'entex');
		
	/* override if event */
	/* translators: */
	if($event) $hover = _x('View all entries during %s', 'Time LIKE: View all entries during September 1, 2017', 'entex');
		
	/* filter */ // - placeholder
	$hover = apply_filters('entex_theme_fn_get_time_link_hover', $hover, $event);
	$hover = sprintf($hover, $date);

	$hour = entex_theme_fn_get_time();
	if($hour) return '<a href="'. esc_url(add_query_arg('by', 'status', $url)) .'" title="'. esc_attr($hover) .'">'. esc_html($hour) .'</a>';
		else return '';
}

function entex_theme_fn_get_date($set_today = true, $set_updated = false){ 

	if($set_updated){
		if(entex_theme_fn_is_updated()){
			/* translators: */
			$placeholder = '<time class="entry-date published ua-hidden" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
		} else {
			/* translators: */
			$placeholder = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
		}
	} else {
		if(entex_theme_fn_is_updated()){
			/* translators: */
			$placeholder = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated ua-hidden" datetime="%3$s">%4$s</time>';
		} else {
			/* translators: */
			$placeholder = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
		}
	}
	
	$text = get_the_date();
	if($set_today){
		if(entex_theme_fn_is_today()) $text = _x('Today', 'Time LIKE: Posted Today', 'entex');
		if(entex_theme_fn_is_yesterday()) $text = _x('Yesterday', 'Time LIKE: Posted Yesterday', 'entex');
		if(entex_theme_fn_is_tomorrow()) $text = _x('Tomorrow', 'Time LIKE: Posted Tomorrow', 'entex');
	}
	
	/* filter */
	$text = apply_filters('entex_theme_fn_get_date_text', $text, $set_today, $set_updated);
	
	/* translators: */
	$content = sprintf($placeholder,
		esc_attr(entex_theme_fn_get_datetime()),
		esc_html($text),
		esc_attr(entex_theme_fn_get_datetime(true)),
		esc_html(get_the_modified_date())
	);
	
	return $content;
}

function entex_theme_fn_get_date_link($set_today = true, $set_updated = false){ 
	
	$event = entex_theme_fn_is_event();
	
	$content = entex_theme_fn_get_date($set_today, $set_updated);
	$url = get_month_link(get_the_date('Y'), get_the_date('m/'));
	/* translators: */
	$hover = _x('View all entries during %s', 'Date LIKE: View all entries during November 2010', 'entex');
	
	/* filter */
	$hover = apply_filters('entex_theme_fn_get_date_link_hover', $hover, $set_today, $set_updated, $event);
	$hover = sprintf($hover, get_the_date(_x('F Y', 'Date LIKE: November 2010', 'entex')));
	
	$content = '<a href="'. esc_url($url) .'" title="'. esc_attr($hover) .'">'. $content .'</a>';
	return $content;
}




/* helper */
function entex_theme_fn_cleanup_url($string){
	return strtolower(trim(preg_replace('~[^0-9a-z]+~i', '-', html_entity_decode(preg_replace('~&([a-z]{1,2})(?:acute|cedil|circ|grave|lig|orn|ring|slash|th|tilde|uml);~i', '$1', htmlentities($string, ENT_QUOTES, 'UTF-8')), ENT_QUOTES, 'UTF-8')), '-'));
}




/* -------------------- */
// COMMENTS

function entex_theme_fn_get_comments_link(){
	/* translators: */
	return entex_get_comments_popup_link(_x('No comments', 'Comments: Count', 'entex'), _x('1 Comment', 'Comments: Count', 'entex'), _x('% Comments', 'Comments: Count', 'entex'));
}

/* NOTE: This function returns the comment object */
function entex_theme_fn_get_comments_last(){ 
	
	$output = '';
	$number = 1;
	
	/* forked from widget */
	$comments = get_comments(
		array(
			'number'      => $number,
			'status'      => 'approve',
			'post_status' => 'publish'
		)
	);

	if(is_array($comments) && $comments){
		// Prime cache for associated posts. (Prime post term cache if we need it for permalinks.)
		$post_ids = array_unique(wp_list_pluck($comments, 'comment_post_ID'));
		_prime_post_caches($post_ids, strpos(get_option('permalink_structure'), '%category%'), false);

		foreach((array) $comments as $comment){
			break;
		}
	}
	return $comment;
}

function entex_theme_fn_get_comments_last_link($label = 'title'){
	
	/* get the last as object */
	$comment = entex_theme_fn_get_comments_last();
	if(!isset($comment->comment_date)) return '';
	
	$date = $comment->comment_date;
	list($date, $time) = explode(' ', $date);
	
	if($label == 'date'){
		$label = esc_html($date);
		/* translators: */
		$hover = sprintf(_x('By %1$s on %2$s', 'Comments meta: By Author name on Post title', 'entex'), $comment->comment_author, get_the_title($comment->comment_post_ID));
	}
	if($label == 'author'){
		$label = get_comment_author_link($comment);
		/* translators: */
		$hover = sprintf(_x('%1$s on %2$s', 'Comments meta: Date on Post title', 'entex'), $date, get_the_title($comment->comment_post_ID));
	}
	if($label == 'title'){
		$label = wp_kses_post(get_the_title($comment->comment_post_ID));
		/* translators: */
		$hover = sprintf(_x('%1$s by %2$s', 'Comments meta: Date by Author name', 'entex'), $date, $comment->comment_author);
	}
	
	$link = '<a title="'. esc_attr($hover) .'" href="'. esc_url(get_comment_link($comment)) .'">'. $label .'</a>';
	return $link;
}


/* singular - current post last comment */
/* NOTE: This function returns the comment object */
function entex_theme_fn_get_comment_last(){ 
	
	$output = '';
	$number = 1;
	
	/* forked from widget */
	$comments = get_comments(
		array(
			'post_id' => get_the_ID(),
			'number' => $number,
			'status' => 'approve',
			'post_status' => 'publish'
		)
	);

	if(is_array($comments) && $comments){
		// Prime cache for associated posts. (Prime post term cache if we need it for permalinks.)
		$post_ids = array_unique(wp_list_pluck($comments, 'comment_post_ID'));
		_prime_post_caches($post_ids, strpos(get_option('permalink_structure'), '%category%'), false);

		foreach((array) $comments as $comment){
			break;
		}
	}
	if(!isset($comment)) $comment = '';
	return $comment;
}

function entex_theme_fn_get_comment_last_link($label = 'date'){
	
	$comment = entex_theme_fn_get_comment_last();
	if(!isset($comment->comment_date)) return '';
	
	$date = $comment->comment_date;
	list($date, $time) = explode(' ', $date);
	
	if($label == 'date'){
		$label = esc_html($date);
		/* translators: */
		$hover = sprintf(_x('By %1$s on %2$s', 'Comments: By Author name on Post title', 'entex'), $comment->comment_author, get_the_title($comment->comment_post_ID));
	}
	if($label == 'author'){
		$label = get_comment_author_link($comment);
		/* translators: */
		$hover = sprintf(_x('%1$s on %2$s', 'Comments: Date on Post title', 'entex'), $date, get_the_title($comment->comment_post_ID));
	}
	if($label == 'title'){
		$label = wp_kses_post(get_the_title($comment->comment_post_ID));
		/* translators: */
		$hover = sprintf(_x('%1$s by %2$s', 'Comments: Date by Author name', 'entex'), $date, $comment->comment_author);
	}
	
	$link = '<a title="'. esc_attr($hover) .'" href="'. esc_url(get_comment_link($comment)) .'">'. $label .'</a>';
	return $link;
}



/* -------------------- */
// EXTENDED COUNTED POSTS

function entex_theme_fn_get_counted_posts(){
	
	global $wp_query;
	$ppage = var_export($wp_query->query_vars['posts_per_page'], true);
	$currpage = var_export($wp_query->query_vars['paged'], true); // MAYBE: $paged = (get_query_var('paged')) ? get_query_var('paged') : 1;
	$nr = $wp_query->max_num_pages; 
	$counted = $wp_query->post_count;
	if($counted == $ppage) $counted = $nr * $ppage;
	if($currpage == $nr && $nr > 1) $counted = $nr * $ppage;
	
	/* defaults */
	$singular = _x('entry', 'Counted posts LIKE: One entry', 'entex');
	$plural = _x('entries', 'Counted posts LIKE: Many entries', 'entex');
	
	$current = entex_theme_fn_get_is();
	if(!empty($current)){
		$label = $current['labels'];
		$singular = entex_fn_lowercase($label['item']);
		$plural = entex_fn_lowercase($label['items']);
	}
	
	/* translators: */
	$counted_txt = sprintf( _x('More then %1$s %2$s found', 'Counted posts: More then [ count ] [ items ] found', 'entex'), $counted, $plural );
	/* translators: */
	if($counted < $ppage) $counted_txt = sprintf( _x('%1$s %2$s found', 'Counted posts: [ count ] [ items ] found', 'entex'), $counted, $plural );
	
	/* translation alternatives */
	/* translators: */
	if($counted == 1) $counted_txt = sprintf( _x('Only %1$s %2$s found', 'Counted posts: Only 1 [ item ] found', 'entex'), $counted, $singular );
	/* translators: */
	if($counted == 1) $counted_txt = sprintf( _x('Only one %1$s found', 'Counted posts: Only one [ item ] found', 'entex'), $singular );
	
	return esc_html($counted_txt);
}




/* -------------------- */
// Check if post type has hierarchical_taxonomy
// Returns first found ( == true ) else empty string ( == false )
function entex_theme_fn_has_hierarchical_taxonomy($obj){
	
	$obj = entex_theme_fn_get_object($obj);
	if(!$obj) return '';
	$taxonomy = '';
	
	$taxonomies = get_object_taxonomies($obj);
	if(!empty($taxonomies)){
		foreach($taxonomies as $taxonomy){
			if(is_taxonomy_hierarchical($taxonomy)) break;
				else $taxonomy = '';
		}
	}
	return $taxonomy;
}





/* -------------------- */
// PARENT FROM POST
// Consider ... get_post_parent_object...
/* Retruns first closest 'deepest' parent on hiarchial taxonomies */
/* Concider add primary term functionality */

function entex_theme_fn_get_parent_object($obj = null, $all = false){
	
	$obj = entex_theme_fn_get_object($obj);
	if(!$obj) return '';
	
	$parent = array();
	$parents = array();

	if(isset($QQ) && $all && entex_is_endpoint()){
		$parents['id'][] = $obj->ID;
		$parents['title'][] = $obj->post_title;
		$parents['url'][] = get_permalink($obj->ID);
	}

	if(isset($QQ) && !$all && entex_is_endpoint()){
		$parent['id'] = $obj->ID;
		$parent['title'] = $obj->post_title;
		$parent['url'] = get_permalink($obj->ID);
	} else {
		if(isset($obj->post_parent) && 0 == $obj->post_parent || $obj->post_type == 'attachment'){
			$taxonomies = get_the_taxonomies($obj); 
			if(!empty($taxonomies)){
				$taxonomy = '';
				/* filter */
				$taxonomy = apply_filters('entex_theme_taxonomy_post_parent', $taxonomy, $obj->ID, $obj);
				if(!$taxonomy){
					if(array_key_exists('category', $taxonomies)){
						$taxonomy = 'category';
					} else {
						/* Accepts the first hierarchical taxonomy */
						foreach($taxonomies as $taxonomy => $whatever){
							if(is_taxonomy_hierarchical($taxonomy)) break;
								else $taxonomy = '';
						}
					}
				}

				if($taxonomy){
						
					$terms = get_the_terms($obj, $taxonomy);
					if(is_array($terms) && $terms !== array()){
						
						if(isset($terms[(count($terms) - 1)])){
							$deepest_term = $terms[(count($terms) - 1)];
						} else {
							
							
							return array();
							
							
							
							foreach($terms as $term){ 
								$tids[] = $term->term_id;
							}
							$closest = get_terms(
								array(
									'taxonomy' => $taxonomy,
									'order' => 'DESC',
									'orderby' => 'term_id',
									'include' => implode(',',$tids),
									'childless' => true
								)
							);
							if(!empty($closest)) $terms = $closest;
							$deepest_term = $terms[0];
							
							$closest_all = get_terms(
								array(
									'taxonomy' => $taxonomy,
									'order' => 'DESC',
									'orderby' => 'term_id',
									'include' => implode(',',$tids)
								)
							);
							if(!empty($closest_all)) $terms = $closest_all;
						}
						
						$parent['id'] = $deepest_term->term_id;
						$parent['title'] = $deepest_term->name;
						$parent['url'] = get_term_link($deepest_term->term_id);
						
						foreach($terms as $term){
							$parents['id'][] = $term->term_id;
							$parents['title'][] = $term->name;
							$parents['url'][] = get_term_link($term->term_id);
						}
					}
				}
			}

		} else {

			$ancestors = array();
			if(isset($obj->ancestors)){
				if(is_array($obj->ancestors)){
					$ancestors = array_values($obj->ancestors);
				} else {
					$ancestors = array($obj->ancestors );
				}
			} else if(isset($obj->post_parent)){
				$ancestors = array($obj->post_parent);
			}
			
			/* ev filter in here */
			if(!is_array($ancestors)) $ancestors = (array) $ancestors;
			
			$parent['id'] = $ancestors[0];
			$parent['title'] = get_the_title($ancestors[0]);
			$parent['url'] = get_permalink($ancestors[0]);
			
			$ancestors = array_reverse( $ancestors );
			if(is_array($ancestors) && $ancestors !== array()){
				foreach($ancestors as $ancestor){
					$parents['id'][] = $ancestor;
					$parents['title'][] = get_the_title($ancestor);
					$parents['url'][] = get_permalink($ancestor);
				}
			}
		}	
	}

	if($all) return $parents;
		else return $parent;
}

function entex_theme_fn_get_parent_id($obj = null){ 
	$parent = entex_theme_fn_get_parent_object($obj);
	if(empty($parent)) return '';
		else return absint($parent['id']);
}

function entex_theme_fn_get_parent($obj = null){ 
	$parent = entex_theme_fn_get_parent_object($obj);
	if(empty($parent)) return '';
		else return esc_html($parent['title']);
}

function entex_theme_fn_get_parent_link($obj = null){
	$parent = entex_theme_fn_get_parent_object($obj);
	if(empty($parent)) return '';
		else return '<a href="'. esc_url($parent['url']) .'" rel="category tag">'. esc_html($parent['title']) .'</a>';
}

function entex_theme_fn_get_parents($obj = null){ 
	$parents = entex_theme_fn_get_parent_object($obj, true);
	return $parents;
}

function entex_theme_fn_get_parents_link($obj = null, $sep = ', '){ 
	$parents = entex_theme_fn_get_parents($obj, true);
	if(empty($parents)) return '';
	for($i = 0; $i < count($parents); $i++){
		if($parents['url'][$i]) $links[] = '<a href="'. esc_url($parents['url'][$i]) .'" rel="category tag">'. esc_html($parents['title'][$i]) .'</a>';
	}
	return implode($sep, $links);
}

/* -------------------- */
// POST FORMAT parents

function entex_theme_fn_get_format($obj = null){
	$obj = entex_theme_fn_get_object($obj);
	if(!$obj) return array();
	$format = get_post_format();
	if(!$format) return array();
	if(current_theme_supports('post-formats', $format)){
		$label = get_post_format_string($format);
		if(!$label){
			return array();
		} else {
			$f['url'] = esc_url(get_post_format_link($format));
			$f['title'] = $label;
			return $f;
		}
	} else return array();
}

function entex_theme_fn_get_format_link($obj = null){
	$format = entex_theme_fn_get_format($obj);
	if(empty($format)) return '';
	$link = '<a href="'. esc_url($format['url']) .'">'. esc_html($format['title']) .'</a>';
	return $link;
}

function entex_theme_fn_get_post_tag_link($sep = ', ', $link = true){
	return entex_theme_fn_get_post_taxonomy_terms(null, 'post_tag', $sep, $link);
}



/* -------------------- */
// ANY TAXONOMIES 

function entex_theme_fn_get_post_taxonomy_terms($post_id = null, $taxonomy = 'category', $sep = ', ', $link = true){
	if(!taxonomy_exists($taxonomy)) return '';
	if(!$post_id) $post_id = get_the_ID();
	$terms = get_the_term_list($post_id, $taxonomy, '', $sep, '');
	if(!$terms) return '';
	if(!$link) return strip_tags($terms);
		else return $terms;
}

/* -------------------- */
// TAXONOMY PARENT or POST PARENT (priority the latter) ? link or ids

function entex_theme_fn_get_term_parents($term_id = null){
	if($term_id){
		$taxonomy = 'category';
		$current = entex_theme_fn_get_is();
		if($current['has-taxonomies']){
			$taxonomies = $current['has-taxonomies'];
			if($taxonomies[1]){
				foreach($taxonomies as $taxonomy){
					if(is_taxonomy_hierarchical($taxonomy)){
						break;
					}
				}
			} else $taxonomy = $taxonomies[0];
		} else return false;
	} else {
		if(!empty(get_queried_object()) && isset(get_queried_object()->taxonomy)){
			$taxonomy = get_queried_object()->taxonomy;
			$term_id = get_queried_object_id();
		} else return false;
	}
	if(!is_taxonomy_hierarchical($taxonomy)) return false;
	$parents = get_ancestors($term_id, $taxonomy, 'taxonomy');
	return $parents;
}



function entex_theme_fn_kses_excerpt($content) { 
	$content = wp_kses_no_null($content);
	$content = wp_kses_normalize_entities($content);
	return $content;
}


/* -------------------- */
// EXCERPT
/* 
* Returns excerpt from the post object
* If no excerpt, build from content
* Based on charater count, not words
* Default Google description rule with maximum 320 characters
*/

function entex_theme_fn_get_excerpt($obj = null, $indication = true){
	
	if(!$obj){
		$obj = get_the_ID();
		if(!$obj) return '';
	}
	$obj = get_post($obj);
	if(!isset($obj->post_excerpt)) return '';
	
	if(defined('ENTEX_EXCERPT_CHAR_LENGTH')) $max_if_excerpt = ENTEX_EXCERPT_CHAR_LENGTH;
		else $max_if_excerpt = 320;
	if(defined('ENTEX_EXCERPT_CHAR_LENGTH_FROM_CONTENT')) $max_if_content = ENTEX_EXCERPT_CHAR_LENGTH_FROM_CONTENT;
		else $max_if_content = 156;
		
	
	$e = $obj->post_excerpt;
	if($e){
		$has_excerpt = true;
		$limit = $max_if_excerpt;
	} else {
		$has_excerpt = false;
		$limit = $max_if_content;
	}
	
	/* filter */
	$length = apply_filters('entex_theme_excerpt_length', $limit, $has_excerpt, $obj->ID, $obj);
	
	/* global */ 
	// - passed from functions or whereever it might needed
	// MAKE SHURE to reset the global after each entry
	if(isset($GLOBALS['ENTEX-EXCERPT-CHAR-LENGTH'])){
		/* A true means only return the raw excerpt as it is - or nothing */
		if($GLOBALS['ENTEX-EXCERPT-CHAR-LENGTH'] === true){
			if(!$has_excerpt) return '';
			$original = $e;
			/* circut */
			return apply_filters('entex_theme_excerpt', $e, $original, $obj->ID, $obj);
		} else {
			$length = $GLOBALS['ENTEX-EXCERPT-CHAR-LENGTH'];
		}
	}
	
	if($length) $length = intval($length);
	
	
	if(!$e){
		$e = $obj->post_content;
		if($e){
			$e = entex_fn_strip_shortcode($e);
			$e = entex_theme_fn_kses_excerpt($e);
			$html = wpautop($e);
			$html = preg_replace("/<([a-z][a-z0-9]*)[^>]*?(\/?)>/i",'<$1$2>', $html);
			preg_match_all('!<p>(.+?)</p>!sim', $html, $matches, PREG_PATTERN_ORDER);
			$html_array = array_map('strip_tags', array_filter($matches[1]));
			if(!empty($html_array)) $html = implode(' ', $html_array);
				else $html = '';
			$e = $html;
		}
	} else {
		/* default excerpt */
		$e = trim(strip_tags(strip_shortcodes($e)));
	}
	
	// NOTE: Less 10 chars not valid
	if($e && mb_strlen($e) < 10) $e = '';
	
	if($e){
		
		$original = $e;
		$cut = mb_substr($e, 0, $length, 'UTF-8');
		
		$counted = mb_strlen($cut);
		$counted_original = mb_strlen($original);
		$clone = mb_strlen(html_entity_decode($cut, ENT_COMPAT, 'UTF-8'));

		if($counted > $clone){
			$diff = ($counted - $clone);
			$diff = (($diff / 4) * 3);
			$length = ($length + $diff);
			$counted = ($counted + $diff);
			$counted_original = ($counted_original + $diff);
		}

		/* The stripper! */
		$e = mb_substr($e, 0, $length, 'UTF-8');
	} else {
		$e = '';
		$original = $e;
		/* filter */
		$e = apply_filters('entex_theme_excerpt_no_excerpt', $e, $obj->ID, $obj);
	}
	
	/* filter */
	return wp_kses(apply_filters('entex_theme_excerpt', $e, $original, $obj->ID, $obj), entex_wp_kses());
}




/* -------------------- */
// CONTENT

function entex_theme_fn_get_more(){
	global $post;
	if(isset($post->post_content)){
		$content = $post->post_content;
		if(strpos($content, '<!--more') === FALSE) return '';
		$more = get_extended($content);
		if($more['extended']) return $more['main'];
	}
	return '';
}

function entex_theme_fn_has_more(){
	global $post;
	if(isset($post->post_content)){
		if(strpos($post->post_content, '<!--more') !== FALSE) return true;
	}
	return false;
}



/* -------------------- */
// CONTEXTUAL
// Fetch archive meta contextual subtitle
// LIKE $content = sprintf('Content published at %1$s', entex_theme_fn_get_home_link());
// This is a contextual scope and more easy to maintain (translations etc) if they are all together
// Thats why wee pass it on here

function entex_theme_fn_get_archive_meta_contextual(){
	$content = '';
	/* filter */ // Populated in entex-contextual-filters.php
	$content = apply_filters('entex_theme_filter_get_archive_meta_contextual', $content);
	return wp_kses($content, entex_wp_kses());
}



/* -------------------- */
// MULTIPAGE
/* Note: entex_is_multipage() is Entex function */
/* Returns 'No' as false or paged string */

function entex_theme_fn_get_entex_is_multipage(){
	
	$multipage = 'No';
	if(entex_is_multipage()){
		$args = entex_get_multipage();
		if(is_singular()){
			/* translators: */
			$multipage = sprintf(_x('Part %1$s of %2$s', 'Multipage singular LIKE: Part 1 of 8', 'entex'), $args['page'], $args['numpages']);
		} else {
			/* translators: */
			$multipage = sprintf(_x('Page %1$s of %2$s', 'Multipage archive LIKE: Page 1 of 8', 'entex'), $args['page'], $args['numpages']);
		}
	}
	/* filter */
	return esc_html(apply_filters('entex_theme_fn_get_entex_is_multipage', $multipage));
}


/* -------------------- */
// GET LOGIN / LOGOUT URL

function entex_theme_fn_get_login_logout_url($redirect = ''){
	
	if(isset($_SERVER['REQUEST_URI'])) $URI = esc_url_raw(wp_unslash($_SERVER['REQUEST_URI']));
		else $URI = '';
	if(!$redirect) $redirect = $URI;
	if(is_admin()) $redirect = home_url();
	if(is_user_logged_in()) $url = wp_logout_url($redirect);
		else $url = wp_login_url($redirect);
	return $url;
}


/* -------------------- */
// IF IS WP-LOGIN.PHP

function entex_is_wp_login(){
	static $cache = false;
	if($cache) return $cache;
	$ABSPATH_MY = str_replace(array('\\','/'), DIRECTORY_SEPARATOR, ABSPATH);
	$cache = (
		(in_array($ABSPATH_MY.'wp-login.php', get_included_files())	|| in_array($ABSPATH_MY.'wp-register.php', get_included_files())) || 
		(isset($_GLOBALS['pagenow']) && $GLOBALS['pagenow'] === 'wp-login.php') || 
		(isset($_SERVER['PHP_SELF']) && $_SERVER['PHP_SELF'] == '/wp-login.php')
	);

	return $cache;
}






/* -------------------- */
// MENUES

/* populate the fallback and all rules to all menues */
/* called from inside: entex_theme_setup */
function entex_wp_page_menu_args($args) {
	$args['fallback_cb'] = 'entex_fallback_menues';
	if($args['theme_location'] == 'footer') $args['depth'] = 1;
	return $args;
}

/* fallback output for menues */
/* called from inside: entex_theme_setup -> entex_wp_page_menu_args */
function entex_fallback_menues($args){
	
	$location = $args['theme_location'];

	if(!$location){
		$location = _x('Menu not set in backend', 'Data', 'entex');
		echo '<ul><li><a href="#">'. esc_html($location) .'</a></li></ul>';
		return;
	} else {
		/* translators: */
		$location = sprintf(_x('Menu %s not set', 'Data', 'entex'), $args['theme_location']);
	}
	
	$txt_link = _x('Link', 'Data', 'entex');
	$txt_parent_page = _x('Parent page', 'Data', 'entex');
	$txt_subpage = _x('Link to subpage', 'Data', 'entex');
	
	if($args['theme_location'] == 'primary'){
		echo '<ul>';
		echo '<li><a href="#">'. esc_html($location) .'</a></li>';
		echo '<li><a href="#">'. esc_html($txt_link) .'</a></li><li><a href="#">'. esc_html($txt_link) .'</a></li><li><a href="#">'. esc_html($txt_link) .'</a></li><li><a href="#">'. esc_html($txt_parent_page) .'</a><ul><li><a href="#">'. esc_html($txt_subpage) .'</a></li><li><a href="#">'. esc_html($txt_subpage) .'</a></li><li><a href="#">'. esc_html($txt_subpage) .'</a></li></ul></li>';
		echo '</ul>'; 
	} else echo '<ul><li><a href="#">'. esc_html($location) .'</a></li><li><a href="#">'. esc_html($txt_link) .'</a></li></ul>'; 
	echo "\n";
}





/* -------------------- */
// SYNCRONIZED DATA
/* get YOAST settings or filtered in */

function entex_theme_fn_get_social_url($media = '', $array = false){
	if(!$media && empty($array)) return '';
	$url = '';
	$label = entex_fn_titlecase($media);
	$media = mb_strtolower($media);
	$social_data = get_option('wpseo_social');
	if(isset($social_data['facebook_site'])){
		if($media == 'facebook') $KEY = 'facebook_site';
		if($media == 'twitter') $KEY = 'twitter_site';
		if(!isset($KEY)) $KEY = $media.'_url';
		if(isset($social_data[$KEY]) && $social_data[$KEY] && trim($social_data[$KEY])) $url = trim($social_data[$KEY]);
	}
	$url = apply_filters('entex_theme_filter_'. $media .'_url', $url);
	if(!$array) return $url;
	if(isset($KEY)) unset($KEY);
	$socials = array('facebook_site', 'instagram_url', 'linkedin_url', 'myspace_url', 'pinterest_url', 'twitter_site', 'youtube_url', 'wikipedia_url');
	foreach($socials as $KEY){
		$url = '';
		list($media, $d) = explode('_', $KEY, 2);
		if(isset($social_data[$KEY]) && $social_data[$KEY] && trim($social_data[$KEY])) $url = trim($social_data[$KEY]);
		$url = apply_filters('entex_theme_filter_'. $media .'_url', $url);
		if($url) $data[$media] = $url;
	}
	return $data;
}

// -- END FILE -- //