<?php 

// ENTEX CORE FUNCTIONS
// This file extends PHP and WordPress functions that not exists
// That is to keep the vocabulary within the same context

// UPDATED 2022 09 25


/* -------------------- */
// RETURN BOOL as function :
/* Makes filters easier to trace or unhook filers and actions */

function entex_return_true(){
	return true;
}

function entex_return_false(){
	return false;
}




/* -------------------- */
// IS GUTENBERG IN USE
/* must be called after correct initial */

function entex_is_gutenberg(){
	if(defined('ENTEX_CLASSIC_EDITOR_INTERFACE')) return false;
	$current_screen = get_current_screen();
	if(method_exists($current_screen, 'is_block_editor') && $current_screen->is_block_editor()){
		return true;
	} else return false;
}



/* -------------------- */
// return sanitized simple $_GET

function entex_request_get($var){
	$value = filter_input( 
		INPUT_GET,
		$var,
		FILTER_SANITIZE_STRING 
	);
	return $value;
}


/* -------------------- */
// return sanitized simple $_POST

function entex_request_post($var){
	$value = filter_input( 
		INPUT_POST,
		$var,
		FILTER_SANITIZE_STRING 
	);
	return $value;
}


/* -------------------- */
// return the TRUE current url

function entex_wp_current_url($query_string = true){
	if($query_string) $arr = array($_GET);
		else $arr = array();
	global $wp;
	if('' === get_option('permalink_structure')){
		return home_url(add_query_arg($arr, $wp->request));
	} else {
		$url = trailingslashit(trailingslashit(home_url()).$wp->request);
		return add_query_arg($arr, $url);
	}
}


/* -------------------- */
// remove individual stylesheets

function entex_wp_remove_styles(){
	
	if(is_admin()) return;
	$handles = apply_filters('entex_add_remove_style_handles', array());
	if(empty($handles) || !is_array($handles)) return;
	
	global $wp_styles;
	foreach($handles as $handle){
		if($handle){
			if(!wp_style_is($handle, 'enqueued')) continue;
			wp_styles()->remove($handle);
			wp_styles()->dequeue($handle);
		}
	}
}
add_action('wp_enqueue_scripts', 'entex_wp_remove_styles', 60);
add_action('wp_print_footer_scripts', 'entex_wp_remove_styles', -5);





/* -------------------- */
// WordPress on LOCALHOST problem:
/* cURL error 60: SSL certificate problem: self signed certificate */
/* On Localhost install, the Theme customizer font layout control might need this */
/* Other then this Theme related functionality on localhost, */
/* This solves a lot of other issues as well on WordPress update services */

function entex_install_localhost_http_request_args($r, $url){
	if(defined('ENTEX_SSL_VERIFY_FALSE_ENABLE')){
		if(strpos(home_url(), 'https://localhost') !== FALSE){
			$r['sslverify'] = false;
		}
	}
	return $r;
}
add_filter('http_request_args', 'entex_install_localhost_http_request_args', 10, 2);




/* -------------------- */
// WP_KSES
/* Form elements accepted, but not the form element */

function entex_wp_kses(){
	
	/* default */
	$allowed_tags = wp_kses_allowed_html('post');

	/* add <time> element */
	$allowed_tags['time'] = $allowed_tags['span'];
	$allowed_tags['time']['datetime'] = true;
	
	/* adjust <video> element */
	$allowed_tags['video']['src'] = true;
	
	/* form elements */
	$allowed_tags['input'] = $allowed_tags['span'];
	$allowed_tags['input']['type'] = true;
	$allowed_tags['input']['name'] = true;
	$allowed_tags['input']['value'] = true;
	$allowed_tags['input']['checked'] = true;
	$allowed_tags['input']['selected'] = true;
	$allowed_tags['input']['disabled'] = true;
	$allowed_tags['input']['required'] = true;
	$allowed_tags['input']['placeholder'] = true;
	$allowed_tags['input']['min'] = true;
	$allowed_tags['input']['max'] = true;
	$allowed_tags['input']['onclick'] = true;
	
	$allowed_tags['select'] = $allowed_tags['input'];
	$allowed_tags['option'] = $allowed_tags['input'];
	
	/* add <source> element */
	$allowed_tags['source'] = $allowed_tags['input'];
	$allowed_tags['source']['src'] = true;
	$allowed_tags['source']['media'] = true;
	$allowed_tags['source']['sizes'] = true;
	
	$allowed_tags['meta']['charset'] = true;
	$allowed_tags['meta']['name'] = true;
	$allowed_tags['meta']['content'] = true;
	$allowed_tags['meta']['property'] = true;
	$allowed_tags['meta']['id'] = true;
	$allowed_tags['meta']['lang'] = true;
	
	$allowed_tags['link']['stylesheet'] = true;
	$allowed_tags['link']['rel'] = true;
	$allowed_tags['link']['id'] = true;
	$allowed_tags['link']['href'] = true;
	$allowed_tags['link']['type'] = true;
	$allowed_tags['link']['media'] = true;
	$allowed_tags['link']['lang'] = true;
	
	$allowed_tags['nav']['tabindex'] = true;

	return $allowed_tags;
}

add_filter('safe_style_css', function($styles){
    $styles[] = 'display';
	$styles[] = 'opacity';
	$styles[] = 'filter';
    return $styles;
});






/* -------------------- */
// PHP DEBUGGER (var_dump)
/* Used by Entex theme contextual engine */
/* Also when learning WordPress and examinate any variable in our templates */
/* in a sanitized and safe way */

function entex_populateDump($data, $entities = true){ 
	if($data === null){
		$data = 'null (No argument passed to Dump)';
		$entities = false;
	}
	if($entities){
		$data = map_deep($data, 'htmlspecialchars');
		$data = map_deep($data, 'trim');
	}
	add_action('shutdown', function() use ($data){
		if(is_admin() || is_customize_preview()) echo wp_kses_post('<pre style="padding: 60px">');
			else echo wp_kses_post('<pre>');
		var_dump($data);
		echo wp_kses_post('</pre>');
	});
}

/* alias */
function entex_PD($data = '', $entities = true){
	entex_populateDump($data, $entities);
}






/* -------------------- */
// GET IMAGE SIZES
/* return array of all existsing image sizes in current theme */
/* incuding WordPress default ones */
/* if size is passed, returns only that / or null if not exists */
/* wp might create size as array of width and height on icons etc etc */
/* But our templates needs 'true' existing sizes */

function entex_get_image_sizes($exists = ''){
	
	/* additional size does not exists */
	if($exists == 'full' || (!empty($exists) && is_array($exists))){
		return null;
	}
	
	foreach(get_intermediate_image_sizes() as $size){
		$image_sizes[$size]['width'] = intval(get_option("{$size}_size_w"));
		$image_sizes[$size]['height'] = intval(get_option("{$size}_size_h"));
		$image_sizes[$size]['crop'] = get_option("{$size}_crop") ? get_option("{$size}_crop") : false;
	}

	global $_wp_additional_image_sizes;
    if(isset($_wp_additional_image_sizes) && count($_wp_additional_image_sizes)){
        $image_sizes = array_merge($image_sizes, $_wp_additional_image_sizes);
    }
	
	if($exists){
		if(isset($image_sizes[$exists])) return $image_sizes[$exists];
			else return null;
	} else return $image_sizes;
}






/* -------------------- */
// PAGE MESSAGE - sanitized by output function
/* This is in use as instructions on front end */
/* After customizer, notices or warnings for administrators */
/* The model is also in use for user messages, if applied by */
/* definition or custom fields, or entex plugins */
/* This Theme function makes shure all notices on front end */
/* appears at the same place, above the main title or crumbs */
/* As we dont want popups, modal windows, clutter the presentation/ theme layout */

function entex_page_message($message){
	if(!is_string($message)) return;
	add_action('entex_template_before_main_content', function() use ($message){
		entex_template_output_page_message($indent = 0, $message);
	}, 10);
}
if(defined('ENTEX_GLOBAL_PAGE_MESSAGE')) entex_page_message(ENTEX_GLOBAL_PAGE_MESSAGE);
	
/* alias */
function entex_PM($message){
	entex_page_message($message);
}






/* -------------------- */
// SOURCECODE CONTEXT :
/* This is a main theme and template function */
/* That might filter in information as sourcecode comments */
/* Where the current template comes from or going back to */
/* Custom plugins or functions can trace the flow inside the Entex Theme */
function entex_backtrace($pointer = 0){
	return apply_filters('entex_filter_backtrace', '', $pointer);
}





/* -------------------- */
// PHP THEME OUTPUT FUNCTIONS
/* All original Theme functions passing here */
/* Linebreaks or comments */
/* No HTML should be rendered 'directly' */
/* By using this port, sanitized output is secured */


function entex_IND($nr = 0, $add = 0, $ct = ''){
	$nr = (int) $nr;
	$add = (int) $add;
	if(!$ct) $ct = " ";
	if($add) $nr = ($nr + $add);
	if($nr > 0) return str_repeat($ct, $nr);
		else return '';
}


/* All collected HTML already sanitized */
function entex_OUT($output = '', $trim = true){
	if($output){
		if($trim) echo rtrim($output)."\r\n"; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			else echo $output."\r\n"; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
	}
}

function entex_LBR(){
	$br = "\r\n";
	return $br;
}

function entex_CMN($nr = 0, $note = ''){
	return entex_IND($nr, 0) .'<!-- '. $note .' -->'. entex_LBR();
}

function entex_eCMN($nr = 0, $note = ''){
	echo wp_kses_post(entex_CMN($nr, $note));
}

function entex_CMD($note = ''){
	return entex_CMDh($note) . entex_LBR();
}

function entex_CML(){
	return entex_CMLh() . entex_LBR();
}


/* helper */
// Comment Dot
function entex_CMDh($note){
	return '<!-- ::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::: -->'. ($note ? '<!-- '. $note .' -->' : '');
}

/* helper */
// Comment Line
function entex_CMLh(){
	return '<!-- ——————————————————————————————————————————————————————————————————————————————————————————————————————————————— -->';
}

// TEMPLATE STYLE
/* Some original templates makes small CSS adjustments if conditionals are met */
/* As direct style tags not allowed, we use the built in WordPress approach */
/* But we dont wanna add the same style more then once */
/* This end-output function takes care of that */

function entex_STL($add = '', $file, $handle = 'entex-template-inline'){
	if(!$add) return; 
	if(isset($GLOBALS['ENTEX-OUTPUT-TEMPLATE-STYLES'][$handle][$file])) return;
	$GLOBALS['ENTEX-OUTPUT-TEMPLATE-STYLES'][$handle][$file] = true;
	if(!wp_script_is($handle, 'enqueued')){
		wp_register_style($handle, false);
		wp_enqueue_style($handle);
	}

	wp_add_inline_style($handle, $add);
}

function entex_SCR($add = '', $file, $handle = 'entex-template-inline'){
	if(!$add) return;
	if(isset($GLOBALS['ENTEX-OUTPUT-TEMPLATE-SCRIPTS'][$handle][$file])) return;
	$GLOBALS['ENTEX-OUTPUT-TEMPLATE-SCRIPTS'][$handle][$file] = true;
	if(!wp_script_is($handle, 'enqueued')){
		wp_register_script($handle, false);
		wp_enqueue_script($handle);
	}
	wp_add_inline_script($handle, $add);
}





/* -------------------- */
// GLOBAL PHP FUNCTIONS 
/* In use by the theme */

/* keep pluggable */
if(!function_exists('entex_mb_ucfirst')){
	function entex_mb_ucfirst($str, $encoding = "UTF-8", $lower_str_end = false) {
		$first_letter = mb_strtoupper(mb_substr($str, 0, 1, $encoding), $encoding);
		$str_end = "";
		if ($lower_str_end) {
			$str_end = mb_strtolower(mb_substr($str, 1, mb_strlen($str, $encoding), $encoding), $encoding);
		} else {
			$str_end = mb_substr($str, 1, mb_strlen($str, $encoding), $encoding);
		}
		$str = $first_letter . $str_end;
		return $str;
	}
}


/* keep pluggable */
if(!function_exists('entex_formatBytes')){
	function entex_formatBytes($bytes, $precision = 2) { 
		$units = array('B', 'kB', 'mB', 'GB', 'TB'); 
		$bytes = max($bytes, 0); 
		$pow = floor(($bytes ? log($bytes) : 0) / log(1024)); 
		$pow = min($pow, count($units) - 1); 
		$bytes /= (1 << (10 * $pow)); 
		return round($bytes, $precision) . ' ' . $units[$pow]; 
	} 
}

/* helpers */

function entex_fn_lowercase($str){
	return mb_strtolower($str, 'UTF-8');
}

function entex_fn_uppercase($str){
	return mb_strtoupper($str, 'UTF-8');
}

function entex_fn_hypencase($str){
	$str = entex_fn_lowercase(entex_fn_trim_nbsp($str));
	$str = str_replace('_', ' ', $str);
	$str = trim(entex_fn_clean_whitespace($str));
	return str_replace(' ', '-', $str);
}

function entex_fn_undercase($str){
	$str = entex_fn_lowercase(entex_fn_trim_nbsp($str));
	$str = str_replace('-', ' ', $str);
	$str = trim(entex_fn_clean_whitespace($str));
	return str_replace(' ', '_', $str);
}

function entex_fn_titlecase($str, $hyphens = true){
	if($hyphens) $str = str_replace('-', ' ', $str);
	return entex_mb_ucfirst(str_replace('_', ' ', $str));
}

function entex_fn_extensioncase($str){
	return substr($str, 0, (strrpos($str, '.')));
}

function entex_fn_filecase($str){
	return pathinfo($str, PATHINFO_FILENAME);
}

function entex_fn_clean_whitespace($string){
	return trim(preg_replace('/\s+/', ' ', $string));
}

function entex_fn_strip_shortcode($text){
	return preg_replace('#\[[^\]]+\]#', '', $text);
}

function entex_fn_trim_nbsp($string){
	$find = array('&nbsp;', chr( 0xC2 ) . chr( 0xA0 ));
	$string = str_replace($find, ' ', $string);
	$string = trim($string);
	return $string;
}





/* -------------------- */
// CORE
/*
 * entex_is_multipage() ouside the loop
 * More dev: https://wordpress.stackexchange.com/questions/202709/how-to-give-paged-links-custom-title
 */
 
if(!function_exists('entex_is_multipage')):
	function entex_is_multipage($obj = null){
		$args = entex_get_multipage($obj);
		if($args['multipage'] > 0) return true;
			else return false;
	}
endif;


if(!function_exists('entex_on_multipage')):
	function entex_on_multipage(){
		if(!is_singular()){
			global $wp_query;
			if((get_query_var('paged') ? get_query_var('paged') : 1) > 1) return get_query_var('paged');
				else return false;
		} else {
			if((get_query_var('page') ? get_query_var('page') : 1) > 1) return get_query_var('page');
				else return false;
		}
	}
endif;

if(!function_exists('entex_get_multipage')):
	function entex_get_multipage($obj = null){
		
		static $cache = false;
		if(!empty($cache)) return $cache;
		
		$args['multipage'] = 0;
		$args['numpages'] = 1;
		$args['page'] = 1;
		
		if(!is_singular()){
			
			global $wp_query;
			$args['numpages'] = $wp_query->max_num_pages;
			$args['page'] = get_query_var('paged') ? get_query_var('paged') : 1;
			if($args['numpages'] > 1) $args['multipage'] = $args['numpages'];

		} else {

			if(!$obj) $obj = $GLOBALS['post'];
			if(isset($obj->post_content)){
				$args['numpages'] = substr_count($obj->post_content, '<!--nextpage-->') + 1;
				$args['page'] = get_query_var('page') ? get_query_var('page') : 1;
				if($args['numpages'] > 1) $args['multipage'] = $args['numpages'];
			}
		}
		
		$cache = $args;
		return $args;
	}
endif;







// CORE 
/*
 * Extending WordPress as function is missing 
 * Returns comments_popup_link() as string instead of echo directly
 * Copy of comments_popup_link(), but modified to return as string
 * WordPress core translations should work
 * Thanks to Jonas Lundman
 * https://developer.wordpress.org/reference/functions/comments_popup_link/
 */
if(!function_exists('entex_get_comments_popup_link')){
	function entex_get_comments_popup_link( $zero = false, $one = false, $more = false, $css_class = '', $none = false ) {

		$id = get_the_ID();
		$title = get_the_title();
		$number = get_comments_number( $id );
	
		$html = '';
 
		if ( false === $zero ) {
			/* translators: %s: post title */
			$zero = sprintf( __( 'No Comments<span class="screen-reader-text"> on %s</span>', 'entex' ), $title );
		}
 
		if ( false === $one ) {
		    /* translators: %s: post title */
		    $one = sprintf( __( '1 Comment<span class="screen-reader-text"> on %s</span>', 'entex' ), $title );
		}
 
		if ( false === $more ) {
			/* translators: 1: Number of comments 2: post title */
    		$more = _n( '%1$s Comment<span class="screen-reader-text"> on %2$s</span>', '%1$s Comments<span class="screen-reader-text"> on %2$s</span>', $number, 'entex' );
		    $more = sprintf( $more, number_format_i18n( $number ), $title );
		}
 
		if ( false === $none ) {
		    /* translators: %s: post title */
		    $none = sprintf( __( 'Comments Off<span class="screen-reader-text"> on %s</span>', 'entex' ), $title );
		}

		if ( 0 == $number && !comments_open() && !pings_open() ) { 
		    $html .=  '<span' . ((!empty($css_class)) ? ' class="' . esc_attr( $css_class ) . '"' : '') . '>' . $none . '</span>';
		    return $html;
		}
 
		if ( post_password_required() ) {
		    $html .= '<span class="screen-reader-text">'. __( 'Enter your password to view comments.', 'entex' ) .'</span>';
		    return $html;
		}
 
		$html .=  '<a href="';
		if ( 0 == $number ) {
		    $respond_link = get_permalink() . '#respond';
		    $html .=  apply_filters( 'respond_link', $respond_link, $id );
		} else { 
			$hash = get_comments_number( $id ) ? '#comments' : '#respond';
			$comments_link = get_permalink( $id ) . $hash;
			$html .= apply_filters( 'get_comments_link', $comments_link, $id );
		}
		$html .=  '"';
 
		if ( !empty( $css_class ) ) {
    		$html .=  ' class="'.$css_class.'" ';
		}
 
		$attributes = '';
		$html .=  apply_filters( 'comments_popup_link_attributes', $attributes );
 
		$html .=  '>';
		$html .= get_comments_number_text( $zero, $one, $more );
		$html .=  '</a>';
		
		return wp_kses($html, entex_wp_kses());
	}
}






// CORE 
/*
 * Extending WordPress as function is missing 
 * Using entex_is_endpoint() as WordPress conditionals context
 * Used by Theme breadcrumbs and contextual engine
 * https://codex.wordpress.org/Conditional_Tags
 */
 
function entex_is_endpoint($specific = false, $populated = false){
	return entex_get_endpoint($specific, $populated) ? true : false;
}

function entex_get_endpoint($specific = false, $populated = false, $fallback = false){
	
	static $cache = false;
	static $endpoint = false;
	static $value = false;
	
	if($cache){
		if($populated){
			if(!$specific && $value) return $value;
				else if($specific == $endpoint && $value) return $value;
			if($fallback) return $endpoint;
		} else {
			if(!$specific) return $endpoint;
				else if($specific == $endpoint) return $endpoint;
		}
	}
	$cache = true;
	
	if(!get_option('permalink_structure')) return false;
	
	global $wp;
	if(!$wp->request) return false;
	$r = $wp->request;
	$parts = explode('/', $r);
	if(empty($parts)) return false;
	
	
	// https://codex.wordpress.org/Reserved_Terms
	$reserved = array(
		'revision',
		'nav_menu_item',
		'custom_css',
		'customize_changeset',
		'action',
		'attachment',
		'attachment_id',
		'author',
		'author_name',
		'calendar',
		'cat',
		'category',
		'category__and',
		'category__in',
		'category__not_in',
		'category_name',
		'comments_per_page',
		'comments_popup',
		'custom',
		'customize_messenger_channel',
		'customized',
		'cpage',
		'day',
		'debug',
		'embed',
		'error',
		'exact',
		'feed',
		'hour',
		'link_category',
		'm',
		'minute',
		'monthnum',
		'more',
		'name',
		'nav_menu',
		'nonce',
		'nopaging',
		'offset',
		'order',
		'orderby',
		'p',
		'page',
		'page_id',
		'paged',
		'pagename',
		'pb',
		'perm',
		'post',
		'post__in',
		'post__not_in',
		'post_format',
		'post_mime_type',
		'post_status',
		'post_tag',
		'post_type',
		'posts',
		'posts_per_archive_page',
		'posts_per_page',
		'preview',
		'robots',
		's',
		'search',
		'second',
		'sentence',
		'showposts',
		'static',
		'subpost',
		'subpost_id',
		'tag',
		'tag__and',
		'tag__in',
		'tag__not_in',
		'tag_id',
		'tag_slug__and',
		'tag_slug__in',
		'taxonomy',
		'tb',
		'term',
		'terms',
		'theme',
		'title',
		'type',
		'w',
		'withcomments',
		'withoutcomments',
		'year'
	);

	global $wp_the_query;
	foreach($parts as $maybe_endpoint){
		if(!$maybe_endpoint || !trim($maybe_endpoint)) continue;
		$maybe_endpoint = trim($maybe_endpoint);
		if(post_type_exists($maybe_endpoint)) continue;
		if(in_array($maybe_endpoint, $reserved)) continue;
		if(isset($wp_the_query->query_vars[$maybe_endpoint])) {
			
			/* cached */
			$endpoint = trim($maybe_endpoint);
			$value = get_query_var($endpoint);
			
			if($populated){
				if(!$specific && $value) return $value;
					else if($specific == $endpoint && $value) return $value;
				if($fallback) return $endpoint;
			} else {
				if(!$specific) return $endpoint;
					else if($specific == $endpoint) return $endpoint;
			}
			break;
		}
	}
	return false;
}


// -- END FILE -- //