<?php if(!defined('ABSPATH')) die();
/**
 * Entex Theme functions
 *
 * Set up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme you can override certain functions (those wrapped
 * in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before
 * the parent theme's file, so the child theme functions would be used.
 *
 * @link https://codex.wordpress.org/Theme_Development
 * @link https://codex.wordpress.org/Child_Themes
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters,
 * {@link https://codex.wordpress.org/Plugin_API}
 *
 * @package Entex
 * @subpackage Entex
 * @since entex 1.0
 * @version 2.20415
 *
 *
 * Documentation: https://entex.info
 * Author: Jonas Lundman
 *
**/




/* -------------------- */
// THEME SETUP

if(!function_exists('entex_theme_setup')):
/* Create your own function to override in a child theme. */

/**
 * Sets up theme defaults and registers support for various WordPress features.
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * REMARK: This setup is not complete for this theme.
 * Child themes can partially override diffrent setups that fits purpose and adjustments of context
 * All theme related configurations are found inside the 'inc/setup/' folder
 * Each file is requierd by the included 'inc/core/entex-core-includes.php' 
 * By define the constant in your child theme, you can override loading this file and completely re-include your own setups
 * 
 */
 
function entex_theme_setup() {

	load_theme_textdomain('entex', get_template_directory() .'/languages');
	
	/* remove widgets block interface */
	/* the main includes file has more filters applied */
	remove_theme_support('widgets-block-editor');
	
	/* Images and post-thumbnail */
	/* inc/setup/entex-setup-images.php */
	
	/* Widgets CSS filters */
	/* inc/setup/entex-setup-widgets.php */
	
	/* Stylesheets and scripts */
	add_action('wp_enqueue_scripts', 'entex_theme_wp_enqueue_scripts');

	/* Feed is added but optionally filtered out by cleanup module */
	add_theme_support('automatic-feed-links');
	add_theme_support('responsive-embeds');
	
	add_theme_support('title-tag');
	
	/* Activate excerpts for pages to use schema.org seo content */
	add_post_type_support('page', 'excerpt');
	add_post_type_support('attachment', 'thumbnail');

	/* Remove excerpt filter, we use our own */
	remove_filter('get_the_excerpt', 'wp_trim_excerpt');
	add_filter('get_the_excerpt', 'entex_theme_filter_wp_trim_excerpt', 30, 2);

	/* Do not translate */
	register_nav_menus(
		array(
			'primary' => __('Primary location', 'entex'),
			'secondary' => __('Secondary location', 'entex'),
			'top-primary'  => __('Top navigation location', 'entex'),
			'top-secondary'  => __('Top navigation secondary location', 'entex'),
			'header-primary'  => __('Header navigation location', 'entex'),
			'social-primary'  => __('Social navigation location', 'entex'),
			'social-secondary'  => __('Social navigation location secondary', 'entex'),
			'footer' => __('Footer location', 'entex')
		)
	);
	
	/* menu fallbacks and rules */
	add_filter('wp_nav_menu_args', 'entex_wp_page_menu_args', 10, 1);
	
	add_theme_support('html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	));

	add_theme_support('post-formats', array(
		'gallery',
		'status',
		'link'
	));
	
	// Media
	/* add iOS Quicktime video support */
	add_filter('wp_video_extensions', function($exts){
		$exts[] = 'mov';
		return $exts;
	});

}
endif; // END Pluggable
add_action('after_setup_theme', 'entex_theme_setup');

/* ******************** */



/* -------------------- */
// STYLES SCRIPT SETUP

if(!function_exists('entex_theme_wp_enqueue_scripts')) :
/* Create your own function to override in a child theme. */

function entex_theme_wp_enqueue_scripts(){

	/* Google fonts */
	$fonts_url = entex_theme_fonts_url();
	if($fonts_url) wp_enqueue_style('entex-theme-fonts', $fonts_url, array(), null);
	
	$fonts_url = entex_theme_customize_fonts_url();
	if($fonts_url) wp_enqueue_style('entex-theme-fonts-customizer', $fonts_url, array(), null);

	/* Entex Theme stylesheet */
	$version = filemtime(get_template_directory().'/assets/ua-css/ua-default-stylesheet.min.css');
	wp_enqueue_style('entex-theme-stylesheet', get_template_directory_uri() .'/assets/ua-css/ua-default-stylesheet.min.css', array(), $version);
	
	/* temporary fix since 5.9 */
	wp_add_inline_style('entex-theme-stylesheet', '.wp-block-image.is-resized img { max-width: none; } .wp-block-gallery:not(.has-nested-images):last-child { margin-bottom: -1em; }');
	wp_add_inline_style('entex-theme-stylesheet', '.post-thumbnail > .post-thumbnail-meta > * { position: relative; overflow: hidden; }');
	
	/* Theme stylesheet */
	add_action('get_footer', function() use ($version){ wp_enqueue_style('entex', get_stylesheet_uri(), array(), filemtime(get_stylesheet_directory().'/style.css')); });
	
	/* Theme scripts */
	wp_enqueue_script('entex-theme-javascript', get_template_directory_uri() .'/assets/ua-js/ua-default-stylesheet.js', array('jquery'), $version, true);
	
	/* Schema placeholder */
	wp_register_script('entex-schema', get_template_directory_uri() .'/assets/ua-js/entex-schema.js', array(), $version, true);
	wp_enqueue_script('entex-schema');
	
	if(is_singular() && comments_open() && get_option('thread_comments')){
		add_action('get_footer', function(){ wp_enqueue_script('comment-reply'); });
	}
}
endif; // END Pluggable

/* ******************** */



/* -------------------- */
// GOOGLE FONTS

if(!function_exists('entex_theme_fonts_url')) :
/* Create your own function to override in a child theme. */

function entex_theme_fonts_url() {
	
	$fonts_url = '';
	if(apply_filters('entex_theme_google_initial_fonts_disable', get_theme_mod('entex_theme_google_fonts_disable', 0))) return $fonts_url;
	
	/* filter */
	$subsets = apply_filters('entex_theme_fonts_subsets', 'latin,latin-ext');

	// NOTE:
	/* 
	 * If there are characters in your language that are not supported by this theme choosen fonts below, 
	 * translate the target font to 'off'. Do not translate into your own language.
     * Default fallback is Arial, Helvetica, Sans-serif.
	 * This CSS layout can be changed within customizer, however, 
	 * those are always loaded if not using 'entex_theme_google_fonts_disable'
	**/
	 
	/* H4, Menus, crumbs, meta and smaller text */ 
	if('off' !== _x('on', 'Source Sans Pro: on or off', 'entex')){
		$fonts[] = 'Source+Sans+Pro:400,600';
	}
	
	/* Site title, H1, Entry Headings and layered captions */
	if('off' !== _x('on', 'Oswald: on or off', 'entex')){
		$fonts[] = 'Oswald:500,600';
	}
	
	/* H2, H3 and preamble */
	if('off' !== _x('on', 'Roboto: on or off', 'entex')){
		$fonts[] = 'Roboto:500,500i,700';
	}
	
	/* Site description */
	if('off' !== _x('on', 'Roboto condensed: on or off', 'entex')){
		$fonts[] = 'Roboto+Condensed:400';
	}
	
	if(isset($fonts)){
		$fonts_url = add_query_arg( array(
			'family' => implode('|', $fonts),
			'subset' => $subsets,
		), 'https://fonts.googleapis.com/css');
	}

	return $fonts_url;
}
endif; // END Pluggable

/* ******************** */



/* -------------------- */
// GOOGLE FONTS from CUSTOMIZER

function entex_theme_customize_fonts_url(){
	$fonts_url = '';
	if(!is_customize_preview()){
		if(function_exists('entex_theme_customize_google_output')){
			$fonts_url = entex_theme_customize_google_output();
		}
	}
	return $fonts_url;
}





/* -- NO MORE CUSTOM DEFINITIONS FROM HERE PLEASE -- */
/* ------------------------------------------------------------- */
// INCLUSIONS

require get_template_directory() . '/inc/core/entex-core-includes.php';

/* ------------------------------------------------------------- */

// -- END FILE -- //