<?php
/**
 * Typography control class.
 *
 * @since  1.0.0
 * @access public
 */

class Ecommerce_Watch_Store_Control_Typography extends WP_Customize_Control {

	/**
	 * The type of customize control being rendered.
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $type = 'ecommerce-watch-store-typography';

	/**
	 * Array 
	 *
	 * @since  1.0.0
	 * @access public
	 * @var    string
	 */
	public $l10n = array();

	/**
	 * Set up our control.
	 *
	 * @since  1.0.0
	 * @access public
	 * @param  object  $manager
	 * @param  string  $id
	 * @param  array   $args
	 * @return void
	 */
	public function __construct( $manager, $id, $args = array() ) {

		// Let the parent class do its thing.
		parent::__construct( $manager, $id, $args );

		// Make sure we have labels.
		$this->l10n = wp_parse_args(
			$this->l10n,
			array(
				'color'       => esc_html__( 'Font Color', 'ecommerce-watch-store' ),
				'family'      => esc_html__( 'Font Family', 'ecommerce-watch-store' ),
				'size'        => esc_html__( 'Font Size',   'ecommerce-watch-store' ),
				'weight'      => esc_html__( 'Font Weight', 'ecommerce-watch-store' ),
				'style'       => esc_html__( 'Font Style',  'ecommerce-watch-store' ),
				'line_height' => esc_html__( 'Line Height', 'ecommerce-watch-store' ),
				'letter_spacing' => esc_html__( 'Letter Spacing', 'ecommerce-watch-store' ),
			)
		);
	}

	/**
	 * Enqueue scripts/styles.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function enqueue() {
		wp_enqueue_script( 'ecommerce-watch-store-ctypo-customize-controls' );
		wp_enqueue_style(  'ecommerce-watch-store-ctypo-customize-controls' );
	}

	/**
	 * Add custom parameters to pass to the JS via JSON.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function to_json() {
		parent::to_json();

		// Loop through each of the settings and set up the data for it.
		foreach ( $this->settings as $setting_key => $setting_id ) {

			$this->json[ $setting_key ] = array(
				'link'  => $this->get_link( $setting_key ),
				'value' => $this->value( $setting_key ),
				'label' => isset( $this->l10n[ $setting_key ] ) ? $this->l10n[ $setting_key ] : ''
			);

			if ( 'family' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_families();

			elseif ( 'weight' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_weight_choices();

			elseif ( 'style' === $setting_key )
				$this->json[ $setting_key ]['choices'] = $this->get_font_style_choices();
		}
	}

	/**
	 * Underscore JS template to handle the control's output.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return void
	 */
	public function content_template() { ?>

		<# if ( data.label ) { #>
			<span class="customize-control-title">{{ data.label }}</span>
		<# } #>

		<# if ( data.description ) { #>
			<span class="description customize-control-description">{{{ data.description }}}</span>
		<# } #>

		<ul>

		<# if ( data.family && data.family.choices ) { #>

			<li class="typography-font-family">

				<# if ( data.family.label ) { #>
					<span class="customize-control-title">{{ data.family.label }}</span>
				<# } #>

				<select {{{ data.family.link }}}>

					<# _.each( data.family.choices, function( label, choice ) { #>
						<option value="{{ choice }}" <# if ( choice === data.family.value ) { #> selected="selected" <# } #>>{{ label }}</option>
					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.weight && data.weight.choices ) { #>

			<li class="typography-font-weight">

				<# if ( data.weight.label ) { #>
					<span class="customize-control-title">{{ data.weight.label }}</span>
				<# } #>

				<select {{{ data.weight.link }}}>

					<# _.each( data.weight.choices, function( label, choice ) { #>

						<option value="{{ choice }}" <# if ( choice === data.weight.value ) { #> selected="selected" <# } #>>{{ label }}</option>

					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.style && data.style.choices ) { #>

			<li class="typography-font-style">

				<# if ( data.style.label ) { #>
					<span class="customize-control-title">{{ data.style.label }}</span>
				<# } #>

				<select {{{ data.style.link }}}>

					<# _.each( data.style.choices, function( label, choice ) { #>

						<option value="{{ choice }}" <# if ( choice === data.style.value ) { #> selected="selected" <# } #>>{{ label }}</option>

					<# } ) #>

				</select>
			</li>
		<# } #>

		<# if ( data.size ) { #>

			<li class="typography-font-size">

				<# if ( data.size.label ) { #>
					<span class="customize-control-title">{{ data.size.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.size.link }}} value="{{ data.size.value }}" />

			</li>
		<# } #>

		<# if ( data.line_height ) { #>

			<li class="typography-line-height">

				<# if ( data.line_height.label ) { #>
					<span class="customize-control-title">{{ data.line_height.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.line_height.link }}} value="{{ data.line_height.value }}" />

			</li>
		<# } #>

		<# if ( data.letter_spacing ) { #>

			<li class="typography-letter-spacing">

				<# if ( data.letter_spacing.label ) { #>
					<span class="customize-control-title">{{ data.letter_spacing.label }} (px)</span>
				<# } #>

				<input type="number" min="1" {{{ data.letter_spacing.link }}} value="{{ data.letter_spacing.value }}" />

			</li>
		<# } #>

		</ul>
	<?php }

	/**
	 * Returns the available fonts.  Fonts should have available weights, styles, and subsets.
	 *
	 * @todo Integrate with Google fonts.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_fonts() { return array(); }

	/**
	 * Returns the available font families.
	 *
	 * @todo Pull families from `get_fonts()`.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	function get_font_families() {

		return array(
			'' => __( 'No Fonts', 'ecommerce-watch-store' ),       	
       	'Abril Fatface' => __( 'Abril Fatface', 'ecommerce-watch-store' ),
        'Acme' => __( 'Acme', 'ecommerce-watch-store' ),
        'Anton' => __( 'Anton', 'ecommerce-watch-store' ),
        'Architects Daughter' => __( 'Architects Daughter', 'ecommerce-watch-store' ),
        'Arimo' => __( 'Arimo', 'ecommerce-watch-store' ),
        'Arsenal' => __( 'Arsenal', 'ecommerce-watch-store' ),
        'Arvo' => __( 'Arvo', 'ecommerce-watch-store' ),
        'Alegreya' => __( 'Alegreya', 'ecommerce-watch-store' ),
        'Alfa Slab One' => __( 'Alfa Slab One', 'ecommerce-watch-store' ),
        'Averia Serif Libre' => __( 'Averia Serif Libre', 'ecommerce-watch-store' ),
        'Bangers' => __( 'Bangers', 'ecommerce-watch-store' ),
        'Boogaloo' => __( 'Boogaloo', 'ecommerce-watch-store' ),
        'Bad Script' => __( 'Bad Script', 'ecommerce-watch-store' ),
        'Bitter' => __( 'Bitter', 'ecommerce-watch-store' ),
        'Bree Serif' => __( 'Bree Serif', 'ecommerce-watch-store' ),
        'BenchNine' => __( 'BenchNine', 'ecommerce-watch-store' ),
        'Cabin' => __( 'Cabin', 'ecommerce-watch-store' ),
        'Cardo' => __( 'Cardo', 'ecommerce-watch-store' ),
        'Courgette' => __( 'Courgette', 'ecommerce-watch-store' ),
        'Cherry Swash' => __( 'Cherry Swash', 'ecommerce-watch-store' ),
        'Cormorant Garamond' => __( 'Cormorant Garamond', 'ecommerce-watch-store' ),
        'Crimson Text' => __( 'Crimson Text', 'ecommerce-watch-store' ),
        'Cuprum' => __( 'Cuprum', 'ecommerce-watch-store' ),
        'Cookie' => __( 'Cookie', 'ecommerce-watch-store' ),
        'Chewy' => __( 'Chewy', 'ecommerce-watch-store' ),
        'Days One' => __( 'Days One', 'ecommerce-watch-store' ),
        'Dosis' => __( 'Dosis', 'ecommerce-watch-store' ),
        'Droid Sans' => __( 'Droid Sans', 'ecommerce-watch-store' ),
        'Economica' => __( 'Economica', 'ecommerce-watch-store' ),
        'Fredoka One' => __( 'Fredoka One', 'ecommerce-watch-store' ),
        'Fjalla One' => __( 'Fjalla One', 'ecommerce-watch-store' ),
        'Francois One' => __( 'Francois One', 'ecommerce-watch-store' ),
        'Frank Ruhl Libre' => __( 'Frank Ruhl Libre', 'ecommerce-watch-store' ),
        'Gloria Hallelujah' => __( 'Gloria Hallelujah', 'ecommerce-watch-store' ),
        'Great Vibes' => __( 'Great Vibes', 'ecommerce-watch-store' ),
        'Handlee' => __( 'Handlee', 'ecommerce-watch-store' ),
        'Hammersmith One' => __( 'Hammersmith One', 'ecommerce-watch-store' ),
        'Inconsolata' => __( 'Inconsolata', 'ecommerce-watch-store' ),
        'Indie Flower' => __( 'Indie Flower', 'ecommerce-watch-store' ),
        'IM Fell English SC' => __( 'IM Fell English SC', 'ecommerce-watch-store' ),
        'Julius Sans One' => __( 'Julius Sans One', 'ecommerce-watch-store' ),
        'Josefin Slab' => __( 'Josefin Slab', 'ecommerce-watch-store' ),
        'Josefin Sans' => __( 'Josefin Sans', 'ecommerce-watch-store' ),
        'Kanit' => __( 'Kanit', 'ecommerce-watch-store' ),
        'Lobster' => __( 'Lobster', 'ecommerce-watch-store' ),
        'Lato' => __( 'Lato', 'ecommerce-watch-store' ),
        'Lora' => __( 'Lora', 'ecommerce-watch-store' ),
        'Libre Baskerville' => __( 'Libre Baskerville', 'ecommerce-watch-store' ),
        'Lobster Two' => __( 'Lobster Two', 'ecommerce-watch-store' ),
        'Merriweather' => __( 'Merriweather', 'ecommerce-watch-store' ),
        'Monda' => __( 'Monda', 'ecommerce-watch-store' ),
        'Montserrat' => __( 'Montserrat', 'ecommerce-watch-store' ),
        'Muli' => __( 'Muli', 'ecommerce-watch-store' ),
        'Marck Script' => __( 'Marck Script', 'ecommerce-watch-store' ),
        'Noto Serif' => __( 'Noto Serif', 'ecommerce-watch-store' ),
        'Open Sans' => __( 'Open Sans', 'ecommerce-watch-store' ),
        'Overpass' => __( 'Overpass', 'ecommerce-watch-store' ),
        'Overpass Mono' => __( 'Overpass Mono', 'ecommerce-watch-store' ),
        'Oxygen' => __( 'Oxygen', 'ecommerce-watch-store' ),
        'Orbitron' => __( 'Orbitron', 'ecommerce-watch-store' ),
        'Patua One' => __( 'Patua One', 'ecommerce-watch-store' ),
        'Pacifico' => __( 'Pacifico', 'ecommerce-watch-store' ),
        'Padauk' => __( 'Padauk', 'ecommerce-watch-store' ),
        'Playball' => __( 'Playball', 'ecommerce-watch-store' ),
        'Playfair Display' => __( 'Playfair Display', 'ecommerce-watch-store' ),
        'PT Sans' => __( 'PT Sans', 'ecommerce-watch-store' ),
        'Philosopher' => __( 'Philosopher', 'ecommerce-watch-store' ),
        'Permanent Marker' => __( 'Permanent Marker', 'ecommerce-watch-store' ),
        'Poiret One' => __( 'Poiret One', 'ecommerce-watch-store' ),
        'Quicksand' => __( 'Quicksand', 'ecommerce-watch-store' ),
        'Quattrocento Sans' => __( 'Quattrocento Sans', 'ecommerce-watch-store' ),
        'Raleway' => __( 'Raleway', 'ecommerce-watch-store' ),
        'Rubik' => __( 'Rubik', 'ecommerce-watch-store' ),
        'Rokkitt' => __( 'Rokkitt', 'ecommerce-watch-store' ),
        'Russo One' => __( 'Russo One', 'ecommerce-watch-store' ),
        'Righteous' => __( 'Righteous', 'ecommerce-watch-store' ),
        'Slabo' => __( 'Slabo', 'ecommerce-watch-store' ),
        'Source Sans Pro' => __( 'Source Sans Pro', 'ecommerce-watch-store' ),
        'Shadows Into Light Two' => __( 'Shadows Into Light Two', 'ecommerce-watch-store'),
        'Shadows Into Light' => __( 'Shadows Into Light', 'ecommerce-watch-store' ),
        'Sacramento' => __( 'Sacramento', 'ecommerce-watch-store' ),
        'Shrikhand' => __( 'Shrikhand', 'ecommerce-watch-store' ),
        'Tangerine' => __( 'Tangerine', 'ecommerce-watch-store' ),
        'Ubuntu' => __( 'Ubuntu', 'ecommerce-watch-store' ),
        'VT323' => __( 'VT323', 'ecommerce-watch-store' ),
        'Varela Round' => __( 'Varela Round', 'ecommerce-watch-store' ),
        'Vampiro One' => __( 'Vampiro One', 'ecommerce-watch-store' ),
        'Vollkorn' => __( 'Vollkorn', 'ecommerce-watch-store' ),
        'Volkhov' => __( 'Volkhov', 'ecommerce-watch-store' ),
        'Yanone Kaffeesatz' => __( 'Yanone Kaffeesatz', 'ecommerce-watch-store' )
		);
	}

	/**
	 * Returns the available font weights.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_font_weight_choices() {

		return array(
			'' => esc_html__( 'No Fonts weight', 'ecommerce-watch-store' ),
			'100' => esc_html__( 'Thin',       'ecommerce-watch-store' ),
			'300' => esc_html__( 'Light',      'ecommerce-watch-store' ),
			'400' => esc_html__( 'Normal',     'ecommerce-watch-store' ),
			'500' => esc_html__( 'Medium',     'ecommerce-watch-store' ),
			'700' => esc_html__( 'Bold',       'ecommerce-watch-store' ),
			'900' => esc_html__( 'Ultra Bold', 'ecommerce-watch-store' ),
		);
	}

	/**
	 * Returns the available font styles.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return array
	 */
	public function get_font_style_choices() {

		return array(
			'' => esc_html__( 'No Fonts Style', 'ecommerce-watch-store' ),
			'normal'  => esc_html__( 'Normal', 'ecommerce-watch-store' ),
			'italic'  => esc_html__( 'Italic', 'ecommerce-watch-store' ),
			'oblique' => esc_html__( 'Oblique', 'ecommerce-watch-store' )
		);
	}
}
