<?php
/**
 * Custom functions that act independently of the theme templates.
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 * @package eco_store
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function eco_store_body_classes( $classes ) {
  global $eco_store_post;
  
    if( !is_page_template( 'template-home.php' ) ){
        $classes[] = 'inner';
        // Adds a class of group-blog to blogs with more than 1 published author.
    }

    if ( is_multi_author() ) {
        $classes[] = 'group-blog ';
    }

    // Adds a class of custom-background-image to sites with a custom background image.
    if ( get_background_image() ) {
        $classes[] = 'custom-background-image';
    }
    
    // Adds a class of custom-background-color to sites with a custom background color.
    if ( get_background_color() != 'ffffff' ) {
        $classes[] = 'custom-background-color';
    }
    

    if( eco_store_woocommerce_activated() && ( is_shop() || is_product_category() || is_product_tag() || 'product' === get_post_type() ) && ! is_active_sidebar( 'shop-sidebar' ) ){
        $classes[] = 'full-width';
    }    

    // Adds a class of hfeed to non-singular pages.
    if ( ! is_page() ) {
        $classes[] = 'hfeed ';
    }
  
    if( is_404() ||  is_search() ){
        $classes[] = 'full-width';
    }
  
    if( ! is_active_sidebar( 'right-sidebar' ) ) {
        $classes[] = 'full-width'; 
    }

    return $classes;
}
add_filter( 'body_class', 'eco_store_body_classes' );

 /**
 * 
 * @link http://www.altafweb.com/2011/12/remove-specific-tag-from-php-string.html
 */
function eco_store_strip_single( $tag, $string ){
    $string=preg_replace('/<'.$tag.'[^>]*>/i', '', $string);
    $string=preg_replace('/<\/'.$tag.'>/i', '', $string);
    return $string;
}

if ( ! function_exists( 'eco_store_excerpt_more' ) ) :
/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... * 
 */
function eco_store_excerpt_more($more) {
  return is_admin() ? $more : ' &hellip; ';
}
endif;
add_filter( 'excerpt_more', 'eco_store_excerpt_more' );

if( ! function_exists( 'eco_store_footer_credit' ) ):
    /**
     * Footer Credits
    */
    function eco_store_footer_credit() {

        // Check if footer copyright is enabled
        $eco_store_show_footer_copyright = get_theme_mod( 'eco_store_footer_setting', true );

        if ( ! $eco_store_show_footer_copyright ) {
            return; // Exit if disabled
        }

        $eco_store_copyright_text = get_theme_mod('eco_store_footer_copyright_text');
    
        $eco_store_text = '<div class="site-info"><div class="container"><span class="copyright">';
        if ($eco_store_copyright_text) {
            $eco_store_text .= wp_kses_post($eco_store_copyright_text); 
        } else {
            $eco_store_text .= esc_html__('&copy; ', 'eco-store') . date_i18n(esc_html__('Y', 'eco-store')); 
            $eco_store_text .= ' <a href="' . esc_url(home_url('/')) . '">' . esc_html(get_bloginfo('name')) . '</a>' . esc_html__('. All Rights Reserved.', 'eco-store');
        }
        $eco_store_text .= '</span>';
        $eco_store_text .= '<span class="by"> <a href="' . esc_url('https://www.themeignite.com/products/eco-store') . '" rel="nofollow" target="_blank">' . ECO_STORE_THEME_NAME . '</a>' . esc_html__(' By ', 'eco-store') . '<a href="' . esc_url('https://themeignite.com/') . '" rel="nofollow" target="_blank">' . esc_html__('Themeignite', 'eco-store') . '</a>.';
        $eco_store_text .= sprintf(esc_html__(' Powered By %s', 'eco-store'), '<a href="' . esc_url(__('https://wordpress.org/', 'eco-store')) . '" target="_blank">WordPress</a>.');
        if (function_exists('the_privacy_policy_link')) {
            $eco_store_text .= get_the_privacy_policy_link();
        }
        $eco_store_text .= '</span></div></div>';
        echo apply_filters('eco_store_footer_text', $eco_store_text);
    }
    add_action('eco_store_footer', 'eco_store_footer_credit');
    endif;

/**
 * Is Woocommerce activated
*/
if ( ! function_exists( 'eco_store_woocommerce_activated' ) ) {
  function eco_store_woocommerce_activated() {
    if ( class_exists( 'woocommerce' ) ) { return true; } else { return false; }
  }
}

if( ! function_exists( 'eco_store_change_comment_form_default_fields' ) ) :
/**
 * Change Comment form default fields i.e. author, email & url.
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function eco_store_change_comment_form_default_fields( $fields ){    
    // get the current commenter if available
    $eco_store_commenter = wp_get_current_commenter();
 
    // core functionality
    $req      = get_option( 'require_name_email' );
    $eco_store_aria_req = ( $req ? " aria-required='true'" : '' );
    $eco_store_required = ( $req ? " required" : '' );
    $eco_store_author   = ( $req ? __( 'Name*', 'eco-store' ) : __( 'Name', 'eco-store' ) );
    $eco_store_email    = ( $req ? __( 'Email*', 'eco-store' ) : __( 'Email', 'eco-store' ) );
 
    // Change just the author field
    $fields['author'] = '<p class="comment-form-author"><label class="screen-reader-text" for="author">' . esc_html__( 'Name', 'eco-store' ) . '<span class="required">*</span></label><input id="author" name="author" placeholder="' . esc_attr( $eco_store_author ) . '" type="text" value="' . esc_attr( $eco_store_commenter['comment_author'] ) . '" size="30"' . $eco_store_aria_req . $eco_store_required . ' /></p>';
    
    $fields['email'] = '<p class="comment-form-email"><label class="screen-reader-text" for="email">' . esc_html__( 'Email', 'eco-store' ) . '<span class="required">*</span></label><input id="email" name="email" placeholder="' . esc_attr( $eco_store_email ) . '" type="text" value="' . esc_attr(  $eco_store_commenter['comment_author_email'] ) . '" size="30"' . $eco_store_aria_req . $eco_store_required. ' /></p>';
    
    $fields['url'] = '<p class="comment-form-url"><label class="screen-reader-text" for="url">' . esc_html__( 'Website', 'eco-store' ) . '</label><input id="url" name="url" placeholder="' . esc_attr__( 'Website', 'eco-store' ) . '" type="text" value="' . esc_attr( $eco_store_commenter['comment_author_url'] ) . '" size="30" /></p>'; 
    
    return $fields;    
}
endif;
add_filter( 'comment_form_default_fields', 'eco_store_change_comment_form_default_fields' );

if( ! function_exists( 'eco_store_change_comment_form_defaults' ) ) :
/**
 * Change Comment Form defaults
 * https://blog.josemcastaneda.com/2016/08/08/copy-paste-hurting-theme/
*/
function eco_store_change_comment_form_defaults( $defaults ){    
    $defaults['comment_field'] = '<p class="comment-form-comment"><label class="screen-reader-text" for="comment">' . esc_html__( 'Comment', 'eco-store' ) . '</label><textarea id="comment" name="comment" placeholder="' . esc_attr__( 'Comment', 'eco-store' ) . '" cols="45" rows="8" aria-required="true" required></textarea></p>';
    
    return $defaults;    
}
endif;
add_filter( 'comment_form_defaults', 'eco_store_change_comment_form_defaults' );

if( ! function_exists( 'eco_store_escape_text_tags' ) ) :
/**
 * Remove new line tags from string
 *
 * @param $text
 * @return string
 */
function eco_store_escape_text_tags( $text ) {
    return (string) str_replace( array( "\r", "\n" ), '', strip_tags( $text ) );
}
endif;

if( ! function_exists( 'wp_body_open' ) ) :
/**
 * Fire the wp_body_open action.
 * Added for backwards compatibility to support pre 5.2.0 WordPress versions.
*/
function wp_body_open() {
    /**
     * Triggered after the opening <body> tag.
    */
    do_action( 'wp_body_open' );
}
endif;

if ( ! function_exists( 'eco_store_get_fallback_svg' ) ) :    
/**
 * Get Fallback SVG
*/
function eco_store_get_fallback_svg( $eco_store_post_thumbnail ) {
    if( ! $eco_store_post_thumbnail ){
        return;
    }
    
    $eco_store_image_size = eco_store_get_image_sizes( $eco_store_post_thumbnail );
     
    if( $eco_store_image_size ){ ?>
        <div class="svg-holder">
             <svg class="fallback-svg" viewBox="0 0 <?php echo esc_attr( $eco_store_image_size['width'] ); ?> <?php echo esc_attr( $eco_store_image_size['height'] ); ?>" preserveAspectRatio="none">
                    <rect width="<?php echo esc_attr( $eco_store_image_size['width'] ); ?>" height="<?php echo esc_attr( $eco_store_image_size['height'] ); ?>" style="fill:#dedddd;"></rect>
            </svg>
        </div>
        <?php
    }
}
endif;

function eco_store_enqueue_google_fonts() {

    require get_template_directory() . '/inc/wptt-webfont-loader.php';

    wp_enqueue_style(
        'google-fonts-tenor-sans',
        eco_store_wptt_get_webfont_url( 'https://fonts.googleapis.com/css2?family=Tenor+Sans&display=swap' ),
        array(),
        '1.0'
    );

    wp_enqueue_style(
        'google-fonts-inter-sans',
        eco_store_wptt_get_webfont_url( 'https://fonts.googleapis.com/css2?family=Inter:ital,opsz,wght@0,14..32,100..900;1,14..32,100..900&display=swap' ),
        array(),
        '1.0'
    );
}
add_action( 'wp_enqueue_scripts', 'eco_store_enqueue_google_fonts' );


if( ! function_exists( 'eco_store_site_branding' ) ) :
/**
 * Site Branding
*/
function eco_store_site_branding(){
    $eco_store_logo_site_title = get_theme_mod( 'eco_store_header_site_title', 0 );
    $eco_store_tagline = get_theme_mod( 'eco_store_header_tagline', false );
    $eco_store_eco_store_logo_width = get_theme_mod('eco_store_logo_width', 100); // Retrieve the logo width setting

    ?>
    <div class="site-branding" style="max-width: <?php echo esc_attr(get_theme_mod('eco_store_logo_width', '-1'))?>px;">
        <?php 
        // Check if custom logo is set and display it
        if (function_exists('has_custom_logo') && has_custom_logo()) {
            the_custom_logo();
        }
        if ($eco_store_logo_site_title):
             if (is_front_page()): ?>
            <h1 class="site-title" style="font-size: <?php echo esc_attr(get_theme_mod('eco_store_site_title_size', '30')); ?>px;">
            <a href="<?php echo esc_url(home_url('/')); ?>" rel="home"><?php bloginfo('name'); ?></a>
          </h1>
            <?php else: ?>
                <p class="site-title" itemprop="name">
                    <a href="<?php echo esc_url(home_url('/')); ?>" rel="home"><?php bloginfo('name'); ?></a>
                </p>
            <?php endif; ?>
        <?php endif; 
    
        if ($eco_store_tagline) :
            $eco_store_description = get_bloginfo('description', 'display');
            if ($eco_store_description || is_customize_preview()) :
        ?>
                <p class="site-description" itemprop="description"><?php echo $eco_store_description; ?></p>
            <?php endif;
        endif;
        ?>
    </div>
    <?php
}
endif;
if( ! function_exists( 'eco_store_navigation' ) ) :
    /**
     * Site Navigation
    */
    function eco_store_navigation(){
        ?>
        <nav class="main-navigation" id="site-navigation" role="navigation">
            <?php 
            wp_nav_menu( array( 
                'theme_location' => 'primary', 
                'menu_id' => 'primary-menu' 
            ) ); 
            ?>
        </nav>

        <div class="mobile-nav">
            <button class="toggle-button" data-toggle-target=".main-menu-modal" data-toggle-body-class="showing-main-menu-modal" aria-expanded="false" data-set-focus=".close-main-nav-toggle">
                <span class="toggle-bar"></span>
                <span class="toggle-bar"></span>
                <span class="toggle-bar"></span>
            </button>
            <div class="mobile-nav-wrap">
                <nav class="main-navigation" id="mobile-navigation"  role="navigation">
                    <div class="primary-menu-list main-menu-modal cover-modal" data-modal-target-string=".main-menu-modal">
                        <button class="close close-main-nav-toggle" data-toggle-target=".main-menu-modal" data-toggle-body-class="showing-main-menu-modal" aria-expanded="false" data-set-focus=".main-menu-modal"></button>
                        <div class="mobile-menu" aria-label="<?php esc_attr_e( 'Mobile', 'eco-store' ); ?>">
                            <?php
                                wp_nav_menu( array(
                                    'theme_location' => 'primary',
                                    'menu_id'        => 'mobile-primary-menu',
                                    'menu_class'     => 'nav-menu main-menu-modal',
                                ) );
                            ?>
                        </div>
                    </div>
                </nav>
            </div>
        </div>

        <?php
    }
endif;

if ( ! function_exists( 'eco_store_header_info' ) ) :
    function eco_store_header_info() {
        $eco_store_social_icon = get_theme_mod( 'eco_store_social_icon_setting', false);
        ?>
        <div class="header_info">
            <div class="social-links d-flex align-items-center gap-2 justify-content-end">
                <!-- Search Icon -->
                <?php if ( get_theme_mod('eco_store_show_hide_search', false) != '' ) : ?>
                    <div class="search-body text-center text-md-center">
                        <button type="button" class="search-show">
                            <i class="<?php echo esc_attr(get_theme_mod('eco_store_search_icon','fas fa-search')); ?>"></i>
                        </button>
                    </div>
                <?php endif; ?>
                <!-- Social Icon -->
                <?php if ( $eco_store_social_icon ){?>
                   
                        <?php 
                        $eco_store_social_link1 = get_theme_mod( 'eco_store_social_link_1' );
                        $eco_store_social_link2 = get_theme_mod( 'eco_store_social_link_2' );
                        $eco_store_social_link3 = get_theme_mod( 'eco_store_social_link_3' );
                        $eco_store_social_link4 = get_theme_mod( 'eco_store_social_link_4' );

                        if ( ! empty( $eco_store_social_link1 ) ) {
                            echo '<a class="social1" href="' . esc_url( $eco_store_social_link1 ) . '" target="_blank"><i class="fab fa-facebook-f"></i></a>';
                        }
                        if ( ! empty( $eco_store_social_link2 ) ) {
                            echo '<a class="social2" href="' . esc_url( $eco_store_social_link2 ) . '" target="_blank"><i class="fab fa-youtube"></i></a>';
                        } 
                        if ( ! empty( $eco_store_social_link3 ) ) {
                            echo '<a class="social3" href="' . esc_url( $eco_store_social_link3 ) . '" target="_blank"><i class="fab fa-instagram"></i></a>';
                        }
                        if ( ! empty( $eco_store_social_link4 ) ) {
                            echo '<a class="social4" href="' . esc_url( $eco_store_social_link4 ) . '" target="_blank"><i class="fab fa-snapchat"></i></a>';
                        }
                        ?>

                <?php } ?>
            </div>
            <!-- Search Form -->
            <?php if ( get_theme_mod('eco_store_show_hide_search', false) != '' ) : ?>
                <div class="searchform-inner">
                    <?php get_search_form(); ?>
                    <button type="button" class="close" aria-label="<?php esc_attr_e('Close', 'eco-store'); ?>">
                        <span aria-hidden="true">X</span>
                    </button>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
endif;

if( ! function_exists( 'eco_store_header' ) ) :
    /**
     * Header Start
    */
    function eco_store_header(){
        $eco_store_header_image = get_header_image();
        $eco_store_sticky_header = get_theme_mod('eco_store_sticky_header');?>
        <div id="page-site-header" class="main-header">
            <header id="masthead" class="site-header header-inner" role="banner">
                        <div class="theme-menu head_bg" data-sticky="<?php echo esc_attr( $eco_store_sticky_header ); ?>">
                            <div class="container">
                                <div class="row header_bg" <?php echo $eco_store_header_image != '' ? 'style="background-image: url(' . esc_url( $eco_store_header_image ) . '); background-repeat: no-repeat; background-size: 100% 100%"': ""; ?>>
                                    <div class="col-xl-3 col-lg-3 col-md-4 align-self-center">
                                        <?php eco_store_site_branding(); ?>
                                    </div>
                                    <div class="col-xl-7 col-lg-6 col-md-3 align-self-center">
                                       <?php eco_store_navigation(); ?> 
                                    </div>
                                    <div class="col-xl-2 col-lg-3 col-md-5 align-self-center">
                                        <?php eco_store_header_info(); ?>
                                    </div>
                                </div>
                            </div>
                        </div>
                    
            </header>
        </div>
        <?php
    }
endif;
add_action( 'eco_store_header', 'eco_store_header', 20 );
