<?php

/**
 * dhaara functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package dhaara
 */

/**
 * ============================================================================
 * dhaara only works in WordPress 6.0 or later.
 * ============================================================================
 */
if (version_compare($GLOBALS['wp_version'], '6.0', '<')) {
    require get_template_directory() . '/inc/back-compat.php';
}

/**
 * ============================================================================
 * Define Constants
 * ============================================================================
 */
define('DHAARA_THEME_ROOT',  get_template_directory_uri());
if (!function_exists('dhaara_setup')) :
    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function dhaara_setup()
    {
        /*
         * Make theme available for translation.
         * Translations can be filed in the /languages/ directory.
         * If you're building a theme based on dhaara, use a find and replace
         * to change 'dhaara' to the name of your theme in all the template files.
        */
        load_theme_textdomain('dhaara', get_template_directory() . '/languages');

        // Add default posts and comments RSS feed links to head.
        add_theme_support('automatic-feed-links');

        /*
         * Let WordPress manage the document title.
         * By adding theme support, we declare that this theme does not use a
         * hard-coded <title> tag in the document head, and expect WordPress to
         * provide it for us.
        */
        add_theme_support('title-tag');

        /*
         * Enable support for Post Thumbnails on posts and pages.
         *
         * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
        */
        add_theme_support('post-thumbnails');
        add_image_size('dhaara-featured-image', 2000, 1200, true);
        add_image_size('dhaara-thumbnail-avatar', 100, 100, true);

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus(array(
            'primary-menu' => esc_html__('Primary Menu', 'dhaara'),
            'footer-quick-links' => __('Footer Quick Links', 'dhaara'),
            'footer-services'    => __('Footer Services', 'dhaara'),
            'footer-bottom'      => __('Footer Bottom Menu', 'dhaara')
        ));

        add_filter('walker_nav_menu_start_el', 'dhaara_parent_menu_dropdown', 10, 4);


        add_filter('nav_menu_link_attributes', 'add_link_atts');
        /*
         * Switch default core markup for search form, comment form, and comments
         * to output valid HTML5.
        */
        add_theme_support('html5', array(
            'search-form',
            'comment-form',
            'comment-list',
            'gallery',
            'caption',
        ));

        /*
         * Enable support for Post Formats.
         * See http://codex.wordpress.org/Post_Formats
        */
        add_theme_support('post-formats', array(
            'aside',
            'image',
            'video',
            'audio',
            'quote',
            'link',
            'gallery',
        ));

        // Set up the WordPress core custom background feature.
        add_theme_support('custom-background', apply_filters('dhaara_custom_background_args', array(
            'default-color' => 'ffffff',
            'default-image' => ''
        )));

        // Add theme support for selective refresh for widgets.
        add_theme_support('customize-selective-refresh-widgets');

        /**
         * Add support for core custom logo.
         *
         * @link https://codex.wordpress.org/Theme_Logo
         */
        add_theme_support('custom-logo', array(
            'height' => 250,
            'width' => 250,
            'flex-width' => true,
            'flex-height' => true,
        ));
    }
endif;
add_action('after_setup_theme', 'dhaara_setup');


function dhaara_parent_menu_dropdown($item_output, $item, $depth, $args)
{
    // Apply filter to both primary and side menus
    if (
        !empty($args->theme_location) && in_array($args->theme_location, ['primary-menu']) || empty($args->theme_location)
    ) {
        if (!empty($item->classes) && in_array('menu-item-has-children', $item->classes)) {
            $menu_output = '   
                <span class="main-text "> 
                    <span class="text ">
                    <span class="nav_link">' . esc_html($item->title) . '</span>
                    </span>
                    <span class="the-arrow ">
                     <i class="fa-solid fa-angle-down"></i>
                    </span>
                </span>';
            $item_output = str_replace($item->title, $menu_output, $item_output);
        }
    }
    return $item_output;
}

function add_link_atts($atts)
{
    $atts['class'] = "nav-link";
    return $atts;
}
/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function dhaara_content_width()
{
    $GLOBALS['content_width'] = apply_filters('dhaara_content_width', 640);
}
add_action('after_setup_theme', 'dhaara_content_width', 0);

/**
 * ============================================================================
 * Register widget area
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 * ============================================================================
 */

function dhaara_widgets_init()
{
    register_sidebar(array(
        'name' => esc_html__('Blog Sidebar', 'dhaara'),
        'id' => 'blog-sidebar',
        'description' => esc_html__('Widgets in this area will be shown in the sidebar.', 'dhaara'),
        'before_widget' => '<aside id="%1$s" class="widget %2$s">',
        'after_widget' => '</aside>',
        'before_title' => '<h3 class="widget-title">',
        'after_title' => '</h3>',
    ));

    register_sidebar(array(
        'name' => esc_html__('Page Sidebar', 'dhaara'),
        'id' => 'page-sidebar',
        'description' => esc_html__('Widgets in this area will be shown in the sidebar of any page.', 'dhaara'),
        'before_widget' => '<aside id="%1$s" class="widget %2$s">',
        'after_widget' => '</aside>',
        'before_title' => '<h3 class="widget-title">',
        'after_title' => '</h3>',
    ));
}

add_action('widgets_init', 'dhaara_widgets_init');

/**
 * ============================================================================
 * Enqueue scripts and styles.
 * ===========================================================================
 */
function dhaara_scripts()
{

    $dependencies = array();
    if (did_action('elementor/loaded')) {
        $dependencies = array('elementor-frontend');
    }
    wp_enqueue_style('fontaweasome-style', DHAARA_THEME_ROOT . '/assets/font/fontawesome/fontawesome.min.css');
    wp_enqueue_style('dhaara-style', DHAARA_THEME_ROOT . '/style.css');
    wp_enqueue_style(
        'bootstrap-css',
        DHAARA_THEME_ROOT . '/assets/css/bootstrap-grid.css',
         $dependencies,
        '1.0.0'
    );
    wp_enqueue_style(
        'main-css',
        DHAARA_THEME_ROOT . '/assets/css/main.css',
        $dependencies,
        '1.0.0'
    );
    wp_enqueue_script(
        'theme-js',
        DHAARA_THEME_ROOT . '/assets/js/theme.js',
        array('jquery'),
        wp_get_theme()->get('Version'),
        true
    );


    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
}
add_action('wp_enqueue_scripts', 'dhaara_scripts');

function dhaara_enqueue_custom_colors()
{
    // Get theme customizer values
    $primary_color = get_theme_mod('theme_primary_color', '#FFB600');
    $secondary_color = get_theme_mod('theme_secondary_color', '#FF8C00');

    // Create the CSS
    $custom_css = "
        :root {
            --primary: " . esc_attr($primary_color) . ";
            --secondary: " . esc_attr($secondary_color) . ";
        }
    ";
    
    // Add inline styles to main stylesheet
    wp_add_inline_style('main-css', $custom_css);
}

/**
 * Modifies tag cloud widget arguments to display all tags in the same font size
 * and use list format for better accessibility.
 *
 *
 * @param array $args Arguments for tag cloud widget.
 * @return array The filtered arguments for tag cloud widget.
 */
function dhaara_widget_tag_cloud_args($args)
{
    $args['largest'] = 1;
    $args['smallest'] = 1;
    $args['unit'] = 'em';
    $args['format'] = 'list';

    return $args;
}
add_filter('widget_tag_cloud_args', 'dhaara_widget_tag_cloud_args');

/**
 * ============================================================================
 * Enqueue  font
 * ===========================================================================
 */
function dhaara_load_google_fonts()
{
    wp_enqueue_style(
        'mytheme-fonts',
        get_template_directory_uri() . '/assets/font/jost/jost-fonts.css',
        array(),
        null
    );
}
add_action('wp_enqueue_scripts', 'dhaara_load_google_fonts');
/**
 * ============================================================================
 * Enqueue the Dashicons script
 * ===========================================================================
 */
add_action('wp_enqueue_scripts', 'dhaara_load_dashicons_front_end');
function dhaara_load_dashicons_front_end()
{
    wp_enqueue_style('dashicons');
}

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-helper.php';
/**
 * Customizer Functions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if (defined('JETPACK__VERSION')) {
    require get_template_directory() . '/inc/jetpack.php';
}

require_once get_template_directory() . '/inc/variables.php';