<?php
/**
 * Theme Mode Switcher (Light/Dark/Auto)
 *
 * Implements color mode switching with system preference detection.
 *
 * @package Developry_Nexus
 * @since 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add theme mode CSS variables and switcher script.
 */
function developry_nexus_theme_mode_head() {
	$theme_mode = get_theme_mod( 'developry_nexus_theme_mode', 'auto' );
	?>
	<style id="developry-nexus-theme-mode-styles">
		/* Theme mode toggle button */
		.developry-nexus-theme-toggle {
			position: fixed;
			bottom: 20px;
			right: 20px;
			width: 50px;
			height: 50px;
			border-radius: 50%;
			background: var(--wp--preset--color--primary);
			border: 2px solid var(--wp--preset--color--contrast);
			box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
			cursor: pointer;
			display: flex;
			align-items: center;
			justify-content: center;
			z-index: 9999;
			transition: all 0.3s ease;
		}

		.developry-nexus-theme-toggle:hover {
			background: var(--wp--preset--color--secondary);
			transform: scale(1.1);
		}

		.developry-nexus-theme-toggle svg {
			width: 24px;
			height: 24px;
			fill: var(--wp--preset--color--contrast);
		}

		/* Smooth transition for theme changes */
		body,
		.wp-site-blocks {
			transition: background-color 0.3s ease, color 0.3s ease;
		}
	</style>

	<script>
	(function() {
		// Load dark mode style variation CSS
		function loadDarkStyleCSS() {
			var darkStyleLink = document.getElementById('developry-nexus-dark-style');
			if (!darkStyleLink) {
				darkStyleLink = document.createElement('link');
				darkStyleLink.id = 'developry-nexus-dark-style';
				darkStyleLink.rel = 'stylesheet';
				darkStyleLink.href = '<?php echo esc_url( get_theme_file_uri( 'styles/dark.json' ) ); ?>'.replace('.json', '.css');

				// Generate CSS from dark.json
				var darkCSS = `
					:root {
						--wp--preset--color--base: #1a1a1a !important;
						--wp--preset--color--contrast: #ffffff !important;
						--wp--preset--color--primary: #6dd5ed !important;
						--wp--preset--color--secondary: #9ca3af !important;
						--wp--preset--color--tertiary: #2d2d2d !important;
						--wp--preset--color--success: #4ade80 !important;
						--wp--preset--color--warning: #fbbf24 !important;
						--wp--preset--color--danger: #f87171 !important;
						--wp--preset--color--info: #60a5fa !important;
						--wp--preset--color--light: #374151 !important;
						--wp--preset--color--dark: #111827 !important;
					}
				`;

				var styleElement = document.createElement('style');
				styleElement.id = 'developry-nexus-dark-style';
				styleElement.textContent = darkCSS;
				document.head.appendChild(styleElement);

				return styleElement;
			}
			return darkStyleLink;
		}

		function removeDarkStyleCSS() {
			var darkStyle = document.getElementById('developry-nexus-dark-style');
			if (darkStyle) {
				darkStyle.remove();
			}
		}

		// Get theme mode from customizer or default to 'auto'
		var themeModePreference = '<?php echo esc_js( $theme_mode ); ?>';
		var currentMode = themeModePreference;

		// Check for user's saved preference in localStorage
		if (themeModePreference === 'auto') {
			var savedMode = localStorage.getItem('developry_nexus_theme_mode');
			if (savedMode && (savedMode === 'light' || savedMode === 'dark')) {
				currentMode = savedMode;
			} else if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
				currentMode = 'dark';
			}
		}

		// Apply theme mode immediately (prevent flash)
		function applyThemeMode(mode) {
			if (mode === 'dark') {
				loadDarkStyleCSS();
				document.documentElement.setAttribute('data-theme-mode', 'dark');
			} else {
				removeDarkStyleCSS();
				document.documentElement.setAttribute('data-theme-mode', 'light');
			}
		}

		// Apply on page load
		applyThemeMode(currentMode);

		// Wait for DOM to be ready
		if (document.readyState === 'loading') {
			document.addEventListener('DOMContentLoaded', initThemeToggle);
		} else {
			initThemeToggle();
		}

		function initThemeToggle() {
			var toggle = document.querySelector('.developry-nexus-theme-toggle');
			if (!toggle) return;

			toggle.addEventListener('click', function() {
				var currentDataMode = document.documentElement.getAttribute('data-theme-mode');
				var newMode = currentDataMode === 'dark' ? 'light' : 'dark';

				applyThemeMode(newMode);
				localStorage.setItem('developry_nexus_theme_mode', newMode);

				// Update icon
				updateToggleIcon(toggle, newMode);
			});

			// Set initial icon
			var initialMode = document.documentElement.getAttribute('data-theme-mode') || 'light';
			updateToggleIcon(toggle, initialMode);
		}

		function updateToggleIcon(toggle, mode) {
			var iconHTML = mode === 'dark'
				? '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor" stroke="currentColor" stroke-width="2"><circle cx="12" cy="12" r="5"/><line x1="12" y1="1" x2="12" y2="3"/><line x1="12" y1="21" x2="12" y2="23"/><line x1="4.22" y1="4.22" x2="5.64" y2="5.64"/><line x1="18.36" y1="18.36" x2="19.78" y2="19.78"/><line x1="1" y1="12" x2="3" y2="12"/><line x1="21" y1="12" x2="23" y2="12"/><line x1="4.22" y1="19.78" x2="5.64" y2="18.36"/><line x1="18.36" y1="5.64" x2="19.78" y2="4.22"/></svg>'
				: '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor"><path d="M21 12.79A9 9 0 1 1 11.21 3 7 7 0 0 0 21 12.79z"/></svg>';
			toggle.innerHTML = iconHTML;
		}

		// Listen for system preference changes (auto mode only)
		if (themeModePreference === 'auto' && window.matchMedia) {
			window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', function(e) {
				var savedMode = localStorage.getItem('developry_nexus_theme_mode');
				if (!savedMode) {
					var newMode = e.matches ? 'dark' : 'light';
					applyThemeMode(newMode);
					var toggle = document.querySelector('.developry-nexus-theme-toggle');
					if (toggle) {
						updateToggleIcon(toggle, newMode);
					}
				}
			});
		}
	})();
	</script>
	<?php
}
add_action( 'wp_head', 'developry_nexus_theme_mode_head', 1 );

/**
 * Add theme mode toggle button to footer.
 */
function developry_nexus_theme_mode_toggle() {
	$theme_mode = get_theme_mod( 'developry_nexus_theme_mode', 'auto' );

	// Only show toggle if not in customizer.
	if ( is_customize_preview() ) {
		return;
	}
	?>
	<button class="developry-nexus-theme-toggle" aria-label="<?php esc_attr_e( 'Toggle theme mode', 'developry-nexus' ); ?>" title="<?php esc_attr_e( 'Switch between light and dark mode', 'developry-nexus' ); ?>">
		1
		<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" fill="currentColor">
			<path d="M20.354 15.354A9 9 0 018.646 3.646 9.003 9.003 0 0012 21a9.003 9.003 0 008.354-5.646z"/>
		</svg>
	</button>
	<?php
}
add_action( 'wp_footer', 'developry_nexus_theme_mode_toggle' );

/**
 * Add theme mode body class.
 *
 * @param array $classes Body classes.
 * @return array Modified classes.
 */
function developry_nexus_theme_mode_body_class( $classes ) {
	$theme_mode = get_theme_mod( 'developry_nexus_theme_mode', 'auto' );
	$classes[]  = 'developry-nexus-theme-mode-' . $theme_mode;
	return $classes;
}
add_filter( 'body_class', 'developry_nexus_theme_mode_body_class' );
