<?php
/**
 * Pro Upgrade CTAs and Notices
 *
 * Functions to promote upgrading to Nexus Pro.
 *
 * @package Developry_Nexus
 * @since 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Check if Pro plugin is active
 *
 * @return bool
 */
function developry_nexus_is_pro_active() {
	return defined( 'DEVRY_NEXUS_PRO_VERSION' ) || class_exists( 'Devry_Nexus_Pro' );
}

/**
 * Add Pro upgrade admin notice
 */
function developry_nexus_pro_upgrade_notice() {
	// Don't show if Pro is active.
	if ( developry_nexus_is_pro_active() ) {
		return;
	}

	// Don't show on certain pages.
	$screen = get_current_screen();
	if ( ! $screen || in_array( $screen->base, array( 'plugins', 'update-core' ), true ) ) {
		return;
	}

	// Check if notice was dismissed permanently.
	$user_id          = get_current_user_id();
	$notice_dismissed = get_user_meta( $user_id, 'developry_nexus_pro_notice_dismissed', true );

	if ( $notice_dismissed ) {
		return;
	}

	?>
	<div class="notice notice-info developry-nexus-pro-admin-notice is-dismissible" data-notice="developry_nexus_pro_notice">
		<div class="developry-nexus-pro-notice-content">
			<div class="developry-nexus-pro-notice-icon">
				<span class="dashicons dashicons-star-filled"></span>
			</div>
			<div class="developry-nexus-pro-notice-text">
				<h3><?php esc_html_e( 'Unlock the Full Power of Nexus Pro!', 'developry-nexus' ); ?></h3>
				<p><?php esc_html_e( 'Get advanced SEO features, AI-powered content analysis, entity templates, and priority support.', 'developry-nexus' ); ?></p>
				<p>
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-primary" target="_blank" rel="noopener">
						<?php esc_html_e( 'Upgrade to Pro', 'developry-nexus' ); ?>
					</a>
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-secondary" target="_blank" rel="noopener">
						<?php esc_html_e( 'View Features', 'developry-nexus' ); ?>
					</a>
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener" style="margin-left: 10px;">
						<?php esc_html_e( 'Try Live Demo', 'developry-nexus' ); ?>
					</a>
				</p>
			</div>
		</div>
	</div>
	<style>
		.developry-nexus-pro-admin-notice {
			border-left: 4px solid #f59e0b;
			padding: 15px;
		}
		.developry-nexus-pro-notice-content {
			display: flex;
			align-items: flex-start;
			gap: 15px;
		}
		.developry-nexus-pro-notice-icon .dashicons {
			font-size: 48px;
			width: 48px;
			height: 48px;
			color: #f59e0b;
		}
		.developry-nexus-pro-notice-text h3 {
			margin: 0 0 10px 0;
			font-size: 16px;
		}
		.developry-nexus-pro-notice-text p {
			margin: 8px 0;
		}
	</style>
	<?php
}
add_action( 'admin_notices', 'developry_nexus_pro_upgrade_notice' );

/**
 * Handle notice dismissal
 */
function developry_nexus_dismiss_pro_notice() {
	check_ajax_referer( 'developry-nexus-dismiss-notice', 'nonce' );

	if ( isset( $_POST['notice'] ) ) {
		$notice = sanitize_text_field( wp_unslash( $_POST['notice'] ) );

		if ( 'developry_nexus_pro_notice' === $notice ) {
			$user_id = get_current_user_id();

			// Permanently dismiss the notice for this user.
			update_user_meta( $user_id, 'developry_nexus_pro_notice_dismissed', true );

			wp_send_json_success( array( 'message' => 'Notice dismissed permanently' ) );
		}
	}

	wp_send_json_error( array( 'message' => 'Invalid notice type' ) );
}
add_action( 'wp_ajax_developry_nexus_dismiss_pro_notice', 'developry_nexus_dismiss_pro_notice' );
add_action( 'wp_ajax_developry_nexus_dismiss_notice', 'developry_nexus_dismiss_notice' );

/**
 * Add Pro upgrade menu item with highlight
 */
function developry_nexus_add_pro_menu() {
	if ( developry_nexus_is_pro_active() ) {
		return;
	}

	add_theme_page(
		esc_html__( 'Upgrade to Pro', 'developry-nexus' ),
		'<span style="color: #f59e0b;">⭐ ' . esc_html__( 'Upgrade to Pro', 'developry-nexus' ) . '</span>',
		'manage_options',
		'developry-nexus-upgrade-pro',
		'developry_nexus_render_pro_page'
	);
}
add_action( 'admin_menu', 'developry_nexus_add_pro_menu', 99 );

/**
 * Render Pro upgrade page
 */
function developry_nexus_render_pro_page() {
	?>
	<div class="wrap developry-nexus-pro-page">
		<div class="developry-nexus-pro-hero">
			<h1 style="display: flex; align-items: center; justify-content: center; gap: 10px;">
				<span class="dashicons dashicons-star-filled"></span> 
				<?php esc_html_e( 'Upgrade to Nexus Pro', 'developry-nexus' ); ?>
			</h1>
			<p class="developry-nexus-pro-tagline">
				<?php esc_html_e( 'Unlock the full potential of your WordPress site with premium SEO and AI features.', 'developry-nexus' ); ?>
			</p>
		</div>

		<div class="developry-nexus-pro-comparison">
			<div class="developry-nexus-comparison-grid">

				<!-- Free Column -->
				<div class="developry-nexus-comparison-column">
					<div class="developry-nexus-comparison-header developry-nexus-free-header">
						<h2><?php esc_html_e( 'Free Theme', 'developry-nexus' ); ?></h2>
						<p class="developry-nexus-price"><?php esc_html_e( '$0', 'developry-nexus' ); ?></p>
					</div>
					<ul class="developry-nexus-feature-list">
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( '6 Unique Style Variants', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( '8 Ready-to-use Page Templates', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( '6 Default Block Patterns', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( 'Core Schema Markup', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( 'Basic Readability Score', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( 'Responsive Design', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-no"></span> <?php esc_html_e( 'SEO & Schema Settings', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-no"></span> <?php esc_html_e( 'Performance Optimization', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-no"></span> <?php esc_html_e( 'Breadcrumb Navigation', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-no"></span> <?php esc_html_e( 'Advanced AI Features', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-no"></span> <?php esc_html_e( 'Entity Templates', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-no"></span> <?php esc_html_e( 'Priority Support', 'developry-nexus' ); ?></li>
					</ul>
				</div>

				<!-- Pro Column -->
				<div class="developry-nexus-comparison-column developry-nexus-featured-column">
					<div class="developry-nexus-featured-badge"><?php esc_html_e( 'Most Popular', 'developry-nexus' ); ?></div>
					<div class="developry-nexus-comparison-header developry-nexus-pro-header">
						<h2><?php esc_html_e( 'Pro Plugin', 'developry-nexus' ); ?></h2>
						<p class="developry-nexus-price">
							<?php esc_html_e( '$59', 'developry-nexus' ); ?>
							<span class="developry-nexus-price-period"><?php esc_html_e( '/year', 'developry-nexus' ); ?></span>
						</p>
					</div>
					<ul class="developry-nexus-feature-list">
						<li><span class="dashicons dashicons-yes"></span> <?php esc_html_e( 'Everything in Free', 'developry-nexus' ); ?></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( '12 Exclusive Pro Style Variants', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( '16+ Premium Page Templates', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( '20+ Advanced Block Patterns', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( '7 Full Template Page Patterns', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Additional Header & Footer Variations', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Performance Optimization', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Breadcrumb Navigation', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Advanced Readability Analysis', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'AI-Powered SEO Recommendations', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'AI Snippet Preview Generator', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Entity Templates (Person, Product, Place)', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'FAQ & Citation Block Patterns', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'JSON AI Sitemap', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Content Provenance Tracking', 'developry-nexus' ); ?></strong></li>
						<li><span class="dashicons dashicons-yes"></span> <strong><?php esc_html_e( 'Priority Support & Updates', 'developry-nexus' ); ?></strong></li>
					</ul>
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-primary button-hero" target="_blank" rel="noopener">
						<?php esc_html_e( 'Upgrade Now', 'developry-nexus' ); ?>
					</a>
				</div>

			</div>
		</div>

		<div class="developry-nexus-pro-features-detail">
			<h2><?php esc_html_e( 'Pro Features in Detail', 'developry-nexus' ); ?></h2>

			<div class="developry-nexus-features-grid">

				<?php foreach ( developry_nexus_get_pro_features_detailed() as $feature ) : ?>
					<div class="developry-nexus-feature-card">
						<span class="dashicons <?php echo esc_attr( $feature['icon'] ); ?>"></span>
						<h3><?php echo esc_html( $feature['title'] ); ?></h3>
						<p><?php echo esc_html( $feature['description'] ); ?></p>
					</div>
				<?php endforeach; ?>

			</div>
		</div>

		<div class="developry-nexus-pro-testimonials">
			<h2><?php esc_html_e( 'What Our Pro Users Say', 'developry-nexus' ); ?></h2>
			<div class="developry-nexus-testimonials-grid">
				<div class="developry-nexus-testimonial">
					<p class="developry-nexus-testimonial-text">"Nexus Pro helped us improve our organic traffic by 150% in just 3 months. The AI recommendations are incredibly accurate!"</p>
					<p class="developry-nexus-testimonial-author">— Sarah M., Digital Marketing Agency</p>
				</div>
				<div class="developry-nexus-testimonial">
					<p class="developry-nexus-testimonial-text">"The entity templates saved us hours of work. Now we can create rich, schema-ready pages in minutes."</p>
					<p class="developry-nexus-testimonial-author">— John D., E-commerce Owner</p>
				</div>
				<div class="developry-nexus-testimonial">
					<p class="developry-nexus-testimonial-text">"Best SEO theme for WordPress. The Pro features are worth every penny!"</p>
					<p class="developry-nexus-testimonial-author">— Mike T., Blogger</p>
				</div>
			</div>
		</div>

		<div class="developry-nexus-pro-cta-footer">
			<h2><?php esc_html_e( 'Ready to Upgrade?', 'developry-nexus' ); ?></h2><br />
			<p><?php esc_html_e( 'Join thousands of satisfied Pro users and take your SEO to the next level.', 'developry-nexus' ); ?></p>
			<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-primary button-hero" target="_blank" rel="noopener">
				<?php esc_html_e( 'Get Nexus Pro Now', 'developry-nexus' ); ?>
			</a>
			<p class="developry-nexus-money-back" style="opacity: 0.7; font-size: 14px; margin-top: 10px;">
				<em><?php esc_html_e( '30-Day Money-Back Guarantee', 'developry-nexus' ); ?></em>
			</p>
		</div>
	</div>

	<style>
		.developry-nexus-pro-page {
			max-width: 1200px;
			margin: 20px auto;
		}

		.developry-nexus-pro-hero {
			text-align: center;
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			color: white;
			padding: 60px 40px;
			border-radius: 12px;
			margin-bottom: 40px;
		}

		.developry-nexus-pro-hero h1 {
			font-size: 42px;
			margin: 0 0 15px 0;
			color: white;
		}

		.developry-nexus-pro-hero h1 .dashicons {
			color: #fbbf24;
			font-size: 42px;
			width: 42px;
			height: 42px;
		}

		.developry-nexus-pro-tagline {
			font-size: 18px;
			opacity: 0.9;
			margin: 0;
		}

		.developry-nexus-comparison-grid {
			display: grid;
			grid-template-columns: 1fr 1fr;
			gap: 30px;
			margin-bottom: 60px;
		}

		.developry-nexus-comparison-column {
			background: white;
			border: 2px solid #e5e7eb;
			border-radius: 12px;
			overflow: hidden;
		}

		.developry-nexus-featured-column {
			border-color: #f59e0b;
			position: relative;
			transform: scale(1.05);
			box-shadow: 0 10px 30px rgba(0,0,0,0.1);
		}

		.developry-nexus-featured-badge {
			position: absolute;
			top: 20px;
			right: -35px;
			background: #f59e0b;
			color: white;
			padding: 5px 40px;
			transform: rotate(45deg);
			font-size: 12px;
			font-weight: 600;
			text-transform: uppercase;
		}

		.developry-nexus-comparison-header {
			padding: 30px;
			text-align: center;
		}

		.developry-nexus-free-header {
			background: #f9fafb;
		}

		.developry-nexus-pro-header {
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			color: white;
		}

		.developry-nexus-comparison-header h2 {
			margin: 0 0 10px 0;
			font-size: 28px;
		}

		.developry-nexus-pro-header h2 {
			color: white;
		}

		.developry-nexus-price {
			font-size: 48px;
			font-weight: 700;
			color: #1f2937;
		}

		.developry-nexus-pro-header .developry-nexus-price {
			color: white;
		}

		.developry-nexus-price-period {
			font-size: 18px;
			font-weight: 400;
		}

		.developry-nexus-feature-list {
			list-style: none;
			margin: 0;
			padding: 30px;
		}

		.developry-nexus-feature-list li {
			display: flex;
			align-items: center;
			gap: 10px;
			margin-bottom: 12px;
			font-size: 15px;
		}

		.developry-nexus-feature-list .dashicons {
			font-size: 20px;
			width: 20px;
			height: 20px;
			flex-shrink: 0;
		}

		.developry-nexus-feature-list .dashicons-yes {
			color: #10b981;
		}

		.developry-nexus-feature-list .dashicons-no {
			color: #ef4444;
		}

		.developry-nexus-featured-column .button-hero {
			margin: 0 30px 30px 30px;
			width: calc(100% - 60px);
			text-align: center;
			background: #fbbf24;
			border-color: #f59e0b;
			color: #1f2937;
			font-weight: 600;
		}

		.developry-nexus-featured-column .button-hero:hover {
			background: #f59e0b;
		}

		.developry-nexus-pro-features-detail h2 {
			text-align: center;
			font-size: 32px;
			margin-bottom: 40px;
		}

		.developry-nexus-features-grid {
			display: grid;
			grid-template-columns: repeat(3, 1fr);
			gap: 30px;
			margin-bottom: 60px;
		}

		.developry-nexus-feature-card {
			background: white;
			padding: 30px;
			border: 1px solid #e5e7eb;
			border-radius: 8px;
			text-align: center;
		}

		.developry-nexus-feature-card .dashicons {
			font-size: 48px;
			width: 48px;
			height: 48px;
			color: #667eea;
			margin-bottom: 15px;
		}

		.developry-nexus-feature-card h3 {
			font-size: 18px;
			margin: 0 0 10px 0;
		}

		.developry-nexus-feature-card p {
			margin: 0;
			color: #6b7280;
			font-size: 14px;
			line-height: 1.6;
		}

		.developry-nexus-pro-testimonials {
			margin-bottom: 60px;
		}

		.developry-nexus-pro-testimonials h2 {
			text-align: center;
			font-size: 32px;
			margin-bottom: 40px;
		}

		.developry-nexus-testimonials-grid {
			display: grid;
			grid-template-columns: repeat(3, 1fr);
			gap: 30px;
		}

		.developry-nexus-testimonial {
			background: #f9fafb;
			padding: 30px;
			border-radius: 8px;
			border-left: 4px solid #667eea;
		}

		.developry-nexus-testimonial-text {
			font-style: italic;
			margin-bottom: 15px;
			color: #1f2937;
		}

		.developry-nexus-testimonial-author {
			margin: 0;
			font-weight: 600;
			color: #667eea;
			font-size: 14px;
		}

		.developry-nexus-pro-cta-footer {
			text-align: center;
			background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
			color: white;
			padding: 60px 40px;
			border-radius: 12px;
		}

		.developry-nexus-pro-cta-footer h2 {
			font-size: 36px;
			margin: 0 0 15px 0;
			color: white;
		}

		.developry-nexus-pro-cta-footer p {
			font-size: 18px;
			margin: 0 0 30px 0;
		}

		.developry-nexus-pro-cta-footer .button-hero {
			background: #fbbf24;
			border-color: #f59e0b;
			color: #1f2937;
			font-weight: 600;
		}

		.developry-nexus-pro-cta-footer .button-hero:hover {
			background: #f59e0b;
		}

		.developry-nexus-money-back {
			margin-top: 15px;
			font-size: 14px;
			opacity: 0.9;
		}

		@media (max-width: 782px) {
			.developry-nexus-comparison-grid,
			.developry-nexus-features-grid,
			.developry-nexus-testimonials-grid {
				grid-template-columns: 1fr;
			}

			.developry-nexus-featured-column {
				transform: none;
			}
		}
	</style>
	<?php
}

/**
 * Update dashboard widget with Pro CTA
 */
function developry_nexus_dashboard_widget_with_pro() {
	wp_add_dashboard_widget(
		'developry_nexus_dashboard_widget',
		esc_html__( 'Nexus Theme Info', 'developry-nexus' ),
		'developry_nexus_dashboard_widget_content_with_pro'
	);
}
add_action( 'wp_dashboard_setup', 'developry_nexus_dashboard_widget_with_pro' );

/**
 * Dashboard widget content with Pro CTA
 */
function developry_nexus_dashboard_widget_content_with_pro() {
	?>
	<div class="developry-nexus-dashboard-widget">
		<h4><?php esc_html_e( 'Theme Version:', 'developry-nexus' ); ?> <?php echo esc_html( DEVELOPRY_NEXUS_VERSION ); ?></h4>
		<p><?php esc_html_e( 'Nexus is an AI-Ready SEO WordPress theme built for performance and search visibility.', 'developry-nexus' ); ?></p>

		<?php if ( ! developry_nexus_is_pro_active() ) : ?>
			<div class="developry-nexus-dashboard-pro-cta" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); padding: 15px; border-radius: 6px; margin: 15px 0;">
				<h4 style="color: white; margin: 0 0 10px 0;">
					<span class="dashicons dashicons-star-filled" style="color: #fbbf24;"></span>
					<?php esc_html_e( 'Upgrade to Pro', 'developry-nexus' ); ?>
				</h4>
				<p style="color: rgba(255,255,255,0.9); margin: 0 0 10px 0; font-size: 13px;">
					<?php esc_html_e( 'Unlock advanced SEO features, AI-powered analysis, and premium support.', 'developry-nexus' ); ?>
				</p>
				<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-primary" target="_blank" rel="noopener" style="background: #fbbf24; border-color: #f59e0b; color: #1f2937;">
					<?php esc_html_e( 'Learn More', 'developry-nexus' ); ?>
				</a>
			</div>
		<?php endif; ?>

		<ul>
			<li><a href="<?php echo esc_url( admin_url( 'customize.php' ) ); ?>"><?php esc_html_e( 'Customize Theme', 'developry-nexus' ); ?></a></li>
			<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Documentation', 'developry-nexus' ); ?></a></li>
			<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Support', 'developry-nexus' ); ?></a></li>
		</ul>
	</div>
	<?php
}
