<?php
/**
 * Pro Features Gating
 *
 * Handle gating of Pro-only styles, templates, and patterns.
 *
 * @package Developry_Nexus
 * @since 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Pass Pro plugin status and Pro-only style titles to the Site Editor.
 */
function developry_nexus_enqueue_hide_pro_styles() {
	// Only load in Site Editor.
	if ( ! is_admin() ) {
		return;
	}

	// Enqueue the pro features script.
	wp_enqueue_script(
		'developry-nexus-pro-features',
		get_template_directory_uri() . '/assets/js/pro-features.js',
		array( 'wp-edit-site' ),
		DEVELOPRY_NEXUS_VERSION,
		array( 'strategy' => 'defer' )
	);

	// Localize script with Pro status and style list.
	wp_localize_script(
		'developry-nexus-pro-features',
		'DEVELOPRY_NEXUS_PRO',
		array(
			'active'    => developry_nexus_is_pro_active(),
			'proStyles' => developry_nexus_get_pro_styles(),
		)
	);
}
add_action( 'enqueue_block_editor_assets', 'developry_nexus_enqueue_hide_pro_styles' );

/**
 * Filter templates to add title, descriptions and hide Pro-only templates
 *
 * @param array $templates Available templates.
 * @return array Filtered templates.
 */
function developry_nexus_filter_pro_templates( $templates ) {
	// Get template descriptions from centralized config.
	$template_descriptions = array_merge(
		developry_nexus_get_free_template_descriptions(),
		developry_nexus_get_pro_template_descriptions()
	);

	// Add categories and descriptions to all templates and filter Pro templates if not active.
	foreach ( $templates as $key => $template ) {
		if ( ! isset( $template->slug ) ) {
			continue;
		}

		// Add title & description, if available for templates.
		if ( isset( $template_descriptions[ $template->slug ] ) ) {
			if ( 0 === strpos( $template->slug, 'pro-' ) ) { // Format titles for Pro templates based on the slug.
				$templates[ $key ]->title = ucwords( str_replace( array( '-', '_' ), ' ', str_replace( 'pro', '', $template->slug ) ) ) . ' (Pro)';
			}

			$templates[ $key ]->description = $template_descriptions[ $template->slug ];
		}

		// If Pro is not active, filter out Pro templates.
		if ( ! developry_nexus_is_pro_active() && 0 === strpos( $template->slug, 'pro-' ) ) {
			unset( $templates[ $key ] );
		}
	}

	return $templates;
}
add_filter( 'get_block_templates', 'developry_nexus_filter_pro_templates', 10, 1 );

/**
 * Add Pro patterns category
 */
function developry_nexus_add_pro_pattern_category() {
	// Register Pro patterns category.
	register_block_pattern_category(
		'developry-nexus-pro',
		array(
			'label'       => developry_nexus_is_pro_active() ? __( 'Nexus Pro', 'developry-nexus' ) : __( '⭐ Upgrade for Nexus Pro', 'developry-nexus' ),
			'description' => developry_nexus_is_pro_active() ? __( 'Professional patterns with advanced features.', 'developry-nexus' ) : __( 'Premium patterns available with Nexus Pro. Upgrade to unlock advanced templates and features.', 'developry-nexus' ),
		)
	);

	// Register placeholder pattern when Pro is not active.
	if ( ! developry_nexus_is_pro_active() ) {
		register_block_pattern(
			'developry-nexus-pro/pro-upgrade-required',
			array(
				'title'       => __( '⭐ Unlock 21 Pro Patterns', 'developry-nexus' ),
				'description' => __( 'Upgrade to Nexus Pro to access 21 premium patterns including pricing tables, testimonials, team grids, and more.', 'developry-nexus' ),
				'categories'  => array( 'developry-nexus-pro' ),
				'keywords'    => array( 'pro', 'premium', 'upgrade' ),
				'content'     => '<!-- wp:group {"align":"full","style":{"spacing":{"padding":{"top":"var:preset|spacing|x-large","bottom":"var:preset|spacing|x-large","left":"var:preset|spacing|medium","right":"var:preset|spacing|medium"}},"border":{"radius":"8px","width":"2px"}},"borderColor":"primary","backgroundColor":"tertiary","layout":{"type":"constrained","contentSize":"600px"}} -->
<div class="wp-block-group alignfull has-border-color has-primary-border-color has-tertiary-background-color has-background" style="border-width:2px;border-radius:8px;padding-top:var(--wp--preset--spacing--x-large);padding-right:var(--wp--preset--spacing--medium);padding-bottom:var(--wp--preset--spacing--x-large);padding-left:var(--wp--preset--spacing--medium)">
	<!-- wp:heading {"textAlign":"center","level":2} -->
	<h2 class="wp-block-heading has-text-align-center">' . esc_html__( '⭐ Unlock Nexus Pro Patterns', 'developry-nexus' ) . '</h2>
	<!-- /wp:heading -->

	<!-- wp:paragraph {"align":"center"} -->
	<p class="has-text-align-center">' . esc_html__( 'Get access to 21 professional patterns including:', 'developry-nexus' ) . '</p>
	<!-- /wp:paragraph -->

	<!-- wp:list {"className":"has-text-align-center"} -->
	<ul class="has-text-align-center">
		<li>' . esc_html__( 'Pricing Tables', 'developry-nexus' ) . '</li>
		<li>' . esc_html__( 'Testimonial Grids', 'developry-nexus' ) . '</li>
		<li>' . esc_html__( 'Team Member Cards', 'developry-nexus' ) . '</li>
		<li>' . esc_html__( 'Feature Comparisons', 'developry-nexus' ) . '</li>
		<li>' . esc_html__( 'Statistics Counters', 'developry-nexus' ) . '</li>
		<li>' . esc_html__( 'And 16 more...', 'developry-nexus' ) . '</li>
	</ul>
	<!-- /wp:list -->

	<!-- wp:buttons {"layout":{"type":"flex","justifyContent":"center"}} -->
	<div class="wp-block-buttons">
		<!-- wp:button {"className":"is-style-fill"} -->
		<div class="wp-block-button is-style-fill"><a class="wp-block-button__link wp-element-button" href="https://developrythemes.com" target="_blank" rel="noopener">' . esc_html__( 'Upgrade to Pro', 'developry-nexus' ) . '</a></div>
		<!-- /wp:button -->
	</div>
	<!-- /wp:buttons -->

	<!-- wp:paragraph {"align":"center","fontSize":"small","style":{"spacing":{"margin":{"top":"var:preset|spacing|medium"}}}} -->
	<p class="has-text-align-center has-small-font-size" style="margin-top:var(--wp--preset--spacing--medium)">' . esc_html__( '✓ Priority Support • ✓ Regular Updates • ✓ 30-Day Money Back', 'developry-nexus' ) . '</p>
	<!-- /wp:paragraph -->
</div>
<!-- /wp:group -->',
			)
		);
	}
}
add_action( 'init', 'developry_nexus_add_pro_pattern_category', 9 );
