<?php
/**
 * Admin Functions
 *
 * Admin-specific functionality and customizations.
 *
 * @package Developry_Nexus
 * @since 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Enqueue admin scripts and styles.
 */
function developry_nexus_admin_enqueue_scripts() {
	// Enqueue admin JavaScript
	wp_enqueue_script(
		'developry-nexus-admin',
		DEVELOPRY_NEXUS_THEME_URI . '/assets/js/admin.js',
		array( 'jquery' ),
		DEVELOPRY_NEXUS_VERSION,
		true
	);

	// Localize script with AJAX URL and nonce
	wp_localize_script(
		'developry-nexus-admin',
		'ajaxVars',
		array(
			'ajaxUrl' => admin_url( 'admin-ajax.php' ),
			'nonce'   => wp_create_nonce( 'developry-nexus-dismiss-notice' ),
		)
	);
}
add_action( 'admin_enqueue_scripts', 'developry_nexus_admin_enqueue_scripts' );

/**
 * Add custom admin styles.
 */
function developry_nexus_admin_styles() {
	echo '<style>
		.developry-nexus-admin-notice {
			padding: 15px;
			background: #fff;
			border-left: 4px solid #0066cc;
		}
		.developry-nexus-admin-notice h3 {
			margin-top: 0;
		}
	</style>';
}

add_action( 'admin_head', 'developry_nexus_admin_styles' );

/**
 * Add theme welcome notice.
 */
function developry_nexus_welcome_notice() {
	// Only show on theme pages.
	$screen = get_current_screen();
	if ( 'themes' !== $screen->base ) {
		return;
	}

	// Check if notice was dismissed.
	if ( get_user_meta( get_current_user_id(), 'developry_nexus_welcome_dismissed', true ) ) {
		return;
	}

	?>
	<div class="notice notice-info developry-nexus-admin-notice is-dismissible" data-notice="developry_nexus_welcome">
		<h3><?php esc_html_e( 'Welcome to Nexus!', 'developry-nexus' ); ?></h3>
		<p><?php esc_html_e( 'Thank you for choosing Nexus - The AI-Ready SEO WordPress Theme. Get started by customizing your site in the Theme Customizer.', 'developry-nexus' ); ?></p>
		<p>
			<a href="<?php echo esc_url( admin_url( 'customize.php' ) ); ?>" class="button button-primary">
				<?php esc_html_e( 'Customize Your Site', 'developry-nexus' ); ?>
			</a>
			<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button" target="_blank" rel="noopener">
				<?php esc_html_e( 'View Documentation', 'developry-nexus' ); ?>
			</a>
		</p>
	</div>
	<?php
}

add_action( 'admin_notices', 'developry_nexus_welcome_notice' );

/**
 * Handle notice dismissal via AJAX.
 */
function developry_nexus_dismiss_notice() {
	check_ajax_referer( 'developry-nexus-dismiss-notice', 'nonce' );

	if ( isset( $_POST['notice'] ) ) {
		$notice = sanitize_text_field( wp_unslash( $_POST['notice'] ) );
		update_user_meta( get_current_user_id(), $notice . '_dismissed', true );
		wp_send_json_success();
	}

	wp_send_json_error();
}

add_action( 'wp_ajax_developry_nexus_dismiss_notice', 'developry_nexus_dismiss_notice' );

/**
 * Add editor styles.
 */
function developry_nexus_add_editor_styles() {
	add_editor_style( 'style.css' );
}

add_action( 'after_setup_theme', 'developry_nexus_add_editor_styles' );

/**
 * Add theme dashboard widget.
 */
function developry_nexus_dashboard_widget() {
	wp_add_dashboard_widget(
		'developry_nexus_dashboard_widget',
		esc_html__( 'Nexus Theme Info', 'developry-nexus' ),
		'developry_nexus_dashboard_widget_content'
	);
}

add_action( 'wp_dashboard_setup', 'developry_nexus_dashboard_widget' );

/**
 * Dashboard widget content.
 */
function developry_nexus_dashboard_widget_content() {
	?>
	<div class="developry-nexus-dashboard-widget">
		<h4><?php esc_html_e( 'Theme Version:', 'developry-nexus' ); ?> <?php echo esc_html( DEVELOPRY_NEXUS_VERSION ); ?></h4>
		<p><?php esc_html_e( 'Nexus is an AI-Ready SEO WordPress theme built for performance and search visibility.', 'developry-nexus' ); ?></p>
		<ul>
			<li><a href="<?php echo esc_url( admin_url( 'customize.php' ) ); ?>"><?php esc_html_e( 'Customize Theme', 'developry-nexus' ); ?></a></li>
			<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Documentation', 'developry-nexus' ); ?></a></li>
			<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Support', 'developry-nexus' ); ?></a></li>
		</ul>
	</div>
	<?php
}

/**
 * Add admin menu item for theme settings.
 */
function developry_nexus_admin_menu() {
	add_theme_page(
		esc_html__( 'Theme Options', 'developry-nexus' ),
		esc_html__( 'Theme Options', 'developry-nexus' ),
		'manage_options',
		'developry-nexus-settings',
		'developry_nexus_settings_page'
	);
}
add_action( 'admin_menu', 'developry_nexus_admin_menu' );

/**
 * Theme settings page content.
 */
function developry_nexus_settings_page() {
	?>
	<div class="wrap">
		<h1><?php esc_html_e( 'Theme Settings', 'developry-nexus' ); ?></h1>
		<p><?php esc_html_e( 'Welcome to Nexus - The AI-Ready SEO WordPress Theme. Configure your theme settings using the buttons below.', 'developry-nexus' ); ?></p>

		<div class="developry-nexus-settings-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-top: 30px;">

			<!-- Theme Options Card -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #667eea; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-admin-settings" style="color: #667eea;"></span>
					<?php esc_html_e( 'Theme Options', 'developry-nexus' ); ?>
				</h2>
				<p><?php esc_html_e( 'Access all theme settings including SEO, performance, readability, and more.', 'developry-nexus' ); ?></p>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[panel]=developry_nexus_theme_options' ) ); ?>" class="button button-primary">
					<?php esc_html_e( 'Open Theme Options', 'developry-nexus' ); ?>
				</a>
			</div>

			<!-- Accessibility Card -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #ef4444; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-universal-access" style="color: #ef4444;"></span>
					<?php esc_html_e( 'Accessibility', 'developry-nexus' ); ?>
				</h2>
				<p><?php esc_html_e( 'Configure accessibility features for WCAG 2.2 compliance.', 'developry-nexus' ); ?></p>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_accessibility_section' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'Configure Accessibility', 'developry-nexus' ); ?>
				</a>
			</div>

			<!-- Content Readability Card -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #0548b4; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-book-alt" style="color: #0548b4;"></span>
					<?php esc_html_e( 'Content Readability', 'developry-nexus' ); ?>
				</h2>
				<p><?php esc_html_e( 'Enable real-time readability analysis for content creators.', 'developry-nexus' ); ?></p>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_readability_section' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'Configure Readability', 'developry-nexus' ); ?>
				</a>
			</div>

			<!-- SEO & Schema Card -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #10b981; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-search" style="color: #10b981;"></span>
					<?php esc_html_e( 'SEO & Schema', 'developry-nexus' ); ?>
					<sup style="font-size: 10px"><?php esc_html_e( 'PRO', 'developry-nexus' ); ?></sup>
				</h2>
				<p><?php esc_html_e( 'Configure structured data and SEO settings for better search visibility.', 'developry-nexus' ); ?></p>
				<?php if ( developry_nexus_is_pro_active() ) : ?>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_seo_section' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'Configure SEO', 'developry-nexus' ); ?>
				</a>
				<?php endif; ?>
			</div>

			<!-- AI Features Card -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #8b5cf6; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-chart-line" style="color: #8b5cf6;"></span>
					<?php esc_html_e( 'AI & Structured Summary', 'developry-nexus' ); ?>
					<sup style="font-size: 10px"><?php esc_html_e( 'PRO', 'developry-nexus' ); ?></sup>
				</h2>
				<p><?php esc_html_e( 'Configure AI-ready features for better search engine understanding.', 'developry-nexus' ); ?></p>
				<?php if ( developry_nexus_is_pro_active() ) : ?>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_ai_section' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'Configure AI Features', 'developry-nexus' ); ?>
				</a>
				<?php endif; ?>
			</div>

			<!-- Performance Card -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #f59e0b; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-performance" style="color: #f59e0b;"></span>
					<?php esc_html_e( 'Performance', 'developry-nexus' ); ?>
					<sup style="font-size: 10px"><?php esc_html_e( 'PRO', 'developry-nexus' ); ?></sup>
				</h2>
				<p><?php esc_html_e( 'Optimize Core Web Vitals and site speed for better user experience.', 'developry-nexus' ); ?></p>
				<?php if ( developry_nexus_is_pro_active() ) : ?>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_performance_section' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'Optimize Performance', 'developry-nexus' ); ?>
				</a>
				<?php endif; ?>
			</div>

			<!-- AI Snippet Preview -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #000000; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-welcome-view-site" style="color: #000000;"></span>
					<?php esc_html_e( 'AI Snippet Preview', 'developry-nexus' ); ?>
					<sup style="font-size: 10px"><?php esc_html_e( 'PRO', 'developry-nexus' ); ?></sup>
				</h2>
				<p><?php esc_html_e( 'Preview how AI might summarize and display your content.', 'developry-nexus' ); ?></p>
				<?php if ( developry_nexus_is_pro_active() ) : ?>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_pro_ai_preview' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'AI Snippet Preview', 'developry-nexus' ); ?>
				</a>
				<?php endif; ?>
			</div>

			<!-- SERP Preview -->
			<div class="developry-nexus-settings-card" style="background: white; padding: 24px; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.05);">
				<h2 style="margin-top: 0; color: #ef44db; display: flex; align-items: center; gap: 5px;">
					<span class="dashicons dashicons-google" style="color: #ef44db;"></span>
					<?php esc_html_e( 'SERP Preview', 'developry-nexus' ); ?>
					<sup style="font-size: 10px"><?php esc_html_e( 'PRO', 'developry-nexus' ); ?></sup>
				</h2>
				<p><?php esc_html_e( 'Preview how your content appears in Google search results.', 'developry-nexus' ); ?></p>
				<?php if ( developry_nexus_is_pro_active() ) : ?>
				<a href="<?php echo esc_url( admin_url( 'customize.php?autofocus[section]=developry_nexus_pro_serp_preview' ) ); ?>" class="button button-secondary">
					<?php esc_html_e( 'SERP Preview', 'developry-nexus' ); ?>
				<?php endif; ?>
				</a>
			</div>
		</div>

		<?php if ( ! developry_nexus_is_pro_active() ) : ?>
			<!-- Pro Upgrade Banner -->
			<div style="margin-top: 40px; padding: 40px; background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); border-radius: 12px; text-align: center;">
				<h2 style="color: white; margin-top: 0; font-size: 32px;">
					<span class="dashicons dashicons-star-filled" style="color: #fbbf24; font-size: 32px; width: 32px; height: 32px;"></span>
					<?php esc_html_e( 'Upgrade to Nexus Pro', 'developry-nexus' ); ?>
				</h2>
				<p style="color: rgba(255,255,255,0.9); font-size: 18px; margin-bottom: 25px;">
					<?php esc_html_e( 'Unlock advanced SEO features, AI-powered content analysis, and premium support.', 'developry-nexus' ); ?>
				</p>
				<div style="display: flex; gap: 15px; justify-content: center; flex-wrap: wrap;">
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-primary button-hero" target="_blank" rel="noopener" style="background: #fbbf24; border-color: rgba(245, 158, 11, 0.2); color: #1f2937; font-weight: 600;">
						<?php esc_html_e( 'Upgrade Now', 'developry-nexus' ); ?>
					</a>
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-secondary button-hero" target="_blank" rel="noopener" style="background-color: transparent; color: white; border-color: rgba(0, 0, 0, 0.2);">
						<?php esc_html_e( 'View All Features', 'developry-nexus' ); ?>
					</a>
					<a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" class="button button-secondary button-hero" target="_blank" rel="noopener" style="background: transparent; color: white; border-color: rgba(0, 0, 0, 0.2);">
						<?php esc_html_e( 'Try Live Demo', 'developry-nexus' ); ?>
					</a>
				</div>
				<p style="color: rgba(255,255,255,0.8); margin-top: 20px; font-size: 14px;">
					<?php esc_html_e( 'Starting at $59/year • 30-Day Money-Back Guarantee', 'developry-nexus' ); ?>
				</p>
			</div>
		<?php endif; ?>

		<div style="margin-top: 40px; padding: 20px; background: #f9fafb; border-left: 4px solid #667eea; border-radius: 4px;">
			<h3 style="margin-top: 0;"><?php esc_html_e( 'Quick Links', 'developry-nexus' ); ?></h3>
			<ul>
				<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Documentation', 'developry-nexus' ); ?></a></li>
				<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Support', 'developry-nexus' ); ?></a></li>
				<li><a href="<?php echo esc_url( apply_filters( 'developry_nexus_url', 'https://developrythemes.com' ) ); ?>" target="_blank" rel="noopener"><?php esc_html_e( 'Changelog', 'developry-nexus' ); ?></a></li>
			</ul>
		</div>
	</div>
	<?php
}
