<?php
/**
 * Accessibility Enhancement Functions
 *
 * WCAG 2.2 compliance and accessibility improvements.
 *
 * @package Developry_Nexus
 * @since 1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Add skip link to header.
 */
function developry_nexus_skip_link() {
	// Check if user enabled skip links in customizer.
	if ( ! get_theme_mod( 'developry_nexus_enable_skip_links', true ) ) {
		return;
	}

	echo '<a class="skip-link screen-reader-text" href="#main-content">' . esc_html__( 'Skip to content', 'developry-nexus' ) . '</a>';
}

add_action( 'wp_body_open', 'developry_nexus_skip_link', 5 );

/**
 * Add ARIA landmarks to body classes.
 *
 * @param array $classes Body classes.
 * @return array Modified body classes.
 */
function developry_nexus_body_aria_landmarks( $classes ) {
	$classes[] = 'developry-nexus-accessible';
	return $classes;
}

add_filter( 'body_class', 'developry_nexus_body_aria_landmarks' );

/**
 * Add ARIA label to search form.
 *
 * @param string $form Search form HTML.
 * @return string Modified search form HTML.
 */
function developry_nexus_search_form_aria( $form ) {
	$form = str_replace( '<form', '<form role="search" aria-label="' . esc_attr__( 'Search', 'developry-nexus' ) . '"', $form );
	return $form;
}

add_filter( 'get_search_form', 'developry_nexus_search_form_aria' );

/**
 * Improve menu accessibility.
 *
 * @param array $args Nav menu args.
 * @return array Modified nav menu args.
 */
function developry_nexus_nav_menu_aria( $args ) {
	if ( ! isset( $args['container_aria_label'] ) && isset( $args['theme_location'] ) ) {
		switch ( $args['theme_location'] ) {
			case 'primary':
				$args['container_aria_label'] = esc_attr__( 'Primary Navigation', 'developry-nexus' );
				break;
			case 'footer':
				$args['container_aria_label'] = esc_attr__( 'Footer Navigation', 'developry-nexus' );
				break;
		}
	}
	return $args;
}

add_filter( 'wp_nav_menu_args', 'developry_nexus_nav_menu_aria' );

/**
 * Add screen reader text for pagination.
 *
 * @param string $template Pagination template.
 * @return string Modified pagination template.
 */
function developry_nexus_pagination_aria( $template ) {
	$template = str_replace( '<nav ', '<nav aria-label="' . esc_attr__( 'Pagination', 'developry-nexus' ) . '" ', $template );
	return $template;
}

add_filter( 'navigation_markup_template', 'developry_nexus_pagination_aria' );

/**
 * Add ARIA labels to post navigation.
 *
 * @param array $args Post navigation args.
 * @return array Modified args.
 */
function developry_nexus_post_navigation_aria( $args ) {
	$args['aria_label'] = esc_attr__( 'Post Navigation', 'developry-nexus' );
	return $args;
}

add_filter( 'wp_post_navigation_args', 'developry_nexus_post_navigation_aria' );

/**
 * Improve comment form accessibility.
 *
 * @param array $args Comment form args.
 * @return array Modified comment form args.
 */
function developry_nexus_comment_form_aria( $args ) {
	$args['title_reply_before'] = '<h3 id="reply-title" class="comment-reply-title">';
	$args['title_reply_after']  = '</h3>';

	// Add ARIA labels to form fields.
	$args['comment_field'] = str_replace(
		'<textarea',
		'<textarea aria-label="' . esc_attr__( 'Comment', 'developry-nexus' ) . '"',
		$args['comment_field']
	);

	return $args;
}

add_filter( 'comment_form_defaults', 'developry_nexus_comment_form_aria' );

/**
 * Add focus indicators to interactive elements.
 */
function developry_nexus_focus_indicators() {
	// Check if user enabled enhanced focus indicators in customizer.
	if ( ! get_theme_mod( 'developry_nexus_enhanced_focus', true ) ) {
		return;
	}

	?>
	<style>
		:focus-visible {
			outline: 2px solid var(--wp--preset--color--primary);
			outline-offset: 2px;
		}
		.skip-link:focus {
			clip: auto;
			height: auto;
			width: auto;
			position: absolute;
			top: 0;
			left: 0;
			z-index: 100000;
			padding: 1em;
			background: var(--wp--preset--color--primary);
			color: var(--wp--preset--color--base);
			text-decoration: none;
		}
		/* Screen reader text */
		.screen-reader-text {
			border: 0;
			clip: rect(1px, 1px, 1px, 1px);
			clip-path: inset(50%);
			height: 1px;
			margin: -1px;
			overflow: hidden;
			padding: 0;
			position: absolute !important;
			width: 1px;
			word-wrap: normal !important;
		}
		.screen-reader-text:focus {
			background-color: var(--wp--preset--color--tertiary);
			clip: auto !important;
			clip-path: none;
			color: var(--wp--preset--color--contrast);
			display: block;
			font-size: 1em;
			height: auto;
			left: 5px;
			line-height: normal;
			padding: 15px 23px 14px;
			text-decoration: none;
			top: 5px;
			width: auto;
			z-index: 100000;
		}
	</style>
	<?php
}

add_action( 'wp_head', 'developry_nexus_focus_indicators', 100 );

/**
 * Add lang attribute to images without it.
 *
 * @param string $html Image HTML.
 * @return string Modified image HTML.
 */
function developry_nexus_image_alt_accessibility( $html ) {
	// Ensure all images have alt attributes.
	if ( strpos( $html, 'alt=' ) === false ) {
		$html = str_replace( '<img', '<img alt=""', $html );
	}
	return $html;
}

add_filter( 'post_thumbnail_html', 'developry_nexus_image_alt_accessibility' );
add_filter( 'get_avatar', 'developry_nexus_image_alt_accessibility' );
