/**
 * Nexus Theme - Main JavaScript
 *
 * @format
 * @package Developry_Nexus
 * @since 1.0.0
 */

(function () {
	"use strict";

	/**
	 * Initialize when DOM is ready
	 */
	function init() {
		// Add smooth scrolling for anchor links
		initSmoothScroll();

		// Add keyboard navigation enhancements
		initKeyboardNav();

		// Initialize skip link focus fix
		initSkipLinkFocus();
	}

	/**
	 * Smooth scroll for anchor links
	 */
	function initSmoothScroll() {
		const links = document.querySelectorAll('a[href^="#"]');

		links.forEach((link) => {
			link.addEventListener("click", function (e) {
				const href = this.getAttribute("href");

				// Skip if it's just "#"
				if (href === "#") {
					return;
				}

				const target = document.querySelector(href);

				if (target) {
					e.preventDefault();

					target.scrollIntoView({
						behavior: "smooth",
						block: "start"
					});

					// Update URL without jumping
					if (history.pushState) {
						history.pushState(null, null, href);
					}

					// Focus the target element for accessibility
					target.focus();
				}
			});
		});
	}

	/**
	 * Enhanced keyboard navigation
	 */
	function initKeyboardNav() {
		// Add keyboard support for navigation menus
		const navMenus = document.querySelectorAll(".wp-block-navigation");

		navMenus.forEach((menu) => {
			const menuItems = menu.querySelectorAll("a");

			menuItems.forEach((item, index) => {
				item.addEventListener("keydown", function (e) {
					let targetItem;

					switch (e.key) {
						case "ArrowDown":
						case "ArrowRight":
							e.preventDefault();
							targetItem = menuItems[index + 1];
							break;
						case "ArrowUp":
						case "ArrowLeft":
							e.preventDefault();
							targetItem = menuItems[index - 1];
							break;
						case "Home":
							e.preventDefault();
							targetItem = menuItems[0];
							break;
						case "End":
							e.preventDefault();
							targetItem = menuItems[menuItems.length - 1];
							break;
					}

					if (targetItem) {
						targetItem.focus();
					}
				});
			});
		});
	}

	/**
	 * Fix skip link focus in WebKit browsers
	 */
	function initSkipLinkFocus() {
		const skipLink = document.querySelector(".skip-link");

		if (!skipLink) {
			return;
		}

		skipLink.addEventListener("click", function (e) {
			const target = document.querySelector(this.getAttribute("href"));

			if (target) {
				e.preventDefault();

				// If target can't receive focus, add tabindex
				if (!target.hasAttribute("tabindex")) {
					target.setAttribute("tabindex", "-1");
				}

				target.focus();

				// Scroll to target
				target.scrollIntoView({
					behavior: "smooth",
					block: "start"
				});
			}
		});
	}

	/**
	 * Debounce function for performance
	 */
	function debounce(func, wait) {
		let timeout;
		return function executedFunction(...args) {
			const later = () => {
				clearTimeout(timeout);
				func(...args);
			};
			clearTimeout(timeout);
			timeout = setTimeout(later, wait);
		};
	}

	/**
	 * Check if reduced motion is preferred
	 */
	function prefersReducedMotion() {
		return window.matchMedia("(prefers-reduced-motion: reduce)").matches;
	}

	// Initialize on DOM ready
	if (document.readyState === "loading") {
		document.addEventListener("DOMContentLoaded", init);
	} else {
		init();
	}
})();
