/**
 * Nexus Editor Sidebar
 *
 * Custom sidebar panel for post/page settings in the block editor.
 *
 * @format
 */

(function (wp) {
	const { registerPlugin } = wp.plugins;
	const { PluginSidebar, PluginSidebarMoreMenuItem } = wp.editPost;
	const { PanelBody, TextareaControl, ToggleControl } = wp.components;
	const { useSelect, useDispatch } = wp.data;
	const { __ } = wp.i18n;
	const { createElement: el } = wp.element;

	/**
	 * Content Readability Panel Component (Limited - Free Version)
	 */
	const ReadabilityPanel = function () {
		const content = useSelect(function (select) {
			return select("core/editor").getEditedPostContent();
		}, []);

		const config = window.developryNexusEditorSidebar || {};
		const isProActive = config.isProActive;

		// Calculate basic stats
		const text = content
			.replace(/<[^>]*>/g, "")
			.replace(/\s+/g, " ")
			.trim();
		const words = text.split(/\s+/).filter(function (w) {
			return w.length > 0;
		});
		const sentences = text.split(/[.!?]+/).filter(function (s) {
			return s.trim().length > 0;
		});
		const wordCount = words.length;
		const sentenceCount = Math.max(1, sentences.length);
		const avgWordsPerSentence = wordCount > 0 ? wordCount / sentenceCount : 0;

		// Calculate syllables
		let syllableCount = 0;
		words.forEach(function (word) {
			syllableCount += Math.max(1, word.toLowerCase().match(/[aeiouy]+/g)?.length || 1);
		});
		const avgSyllablesPerWord = wordCount > 0 ? syllableCount / wordCount : 0;

		// Flesch Reading Ease: 206.835 - 1.015 * (words/sentences) - 84.6 * (syllables/words)
		const fleschReading = Math.max(
			0,
			Math.min(100, 206.835 - 1.015 * avgWordsPerSentence - 84.6 * avgSyllablesPerWord)
		);

		// Flesch-Kincaid Grade Level: 0.39 * (words/sentences) + 11.8 * (syllables/words) - 15.59
		const fleschGrade = Math.max(0, 0.39 * avgWordsPerSentence + 11.8 * avgSyllablesPerWord - 15.59);

		// If Pro is active, don't show this panel (Pro has its own advanced version)
		if (isProActive) {
			return null;
		}

		return el(
			PanelBody,
			{ title: __("Content Readability", "developry-nexus"), initialOpen: true },
			// Basic Stats
			el(
				"div",
				{ style: { marginBottom: "15px" } },
				el(
					"div",
					{
						style: {
							display: "flex",
							justifyContent: "space-between",
							marginBottom: "10px",
							paddingBottom: "10px",
							borderBottom: "1px solid #e0e0e0"
						}
					},
					el("span", { style: { fontWeight: "500" } }, __("Reading Ease:", "developry-nexus")),
					el(
						"span",
						{
							style: {
								fontSize: "16px",
								fontWeight: "bold",
								color: fleschReading >= 60 ? "#10b981" : fleschReading >= 40 ? "#f59e0b" : "#d63638"
							}
						},
						fleschReading.toFixed(1)
					)
				),
				el(
					"div",
					{
						style: {
							display: "flex",
							justifyContent: "space-between",
							marginBottom: "10px",
							paddingBottom: "10px",
							borderBottom: "1px solid #e0e0e0"
						}
					},
					el("span", { style: { fontWeight: "500" } }, __("Grade Level:", "developry-nexus")),
					el("span", { style: { fontSize: "16px", fontWeight: "bold" } }, fleschGrade.toFixed(1))
				),
				el(
					"div",
					{
						style: {
							display: "flex",
							justifyContent: "space-between",
							marginBottom: "10px",
							paddingBottom: "10px",
							borderBottom: "1px solid #e0e0e0"
						}
					},
					el("span", { style: { fontWeight: "500" } }, __("Word Count:", "developry-nexus")),
					el(
						"span",
						{ style: { fontSize: "16px", fontWeight: "bold", color: wordCount >= 300 ? "#10b981" : "#f59e0b" } },
						wordCount
					)
				),
				el(
					"div",
					{
						style: {
							display: "flex",
							justifyContent: "space-between",
							marginBottom: "10px",
							paddingBottom: "10px",
							borderBottom: "1px solid #e0e0e0"
						}
					},
					el("span", { style: { fontWeight: "500" } }, __("Sentences:", "developry-nexus")),
					el("span", { style: { fontSize: "16px", fontWeight: "bold" } }, sentenceCount)
				),
				el(
					"div",
					{
						style: {
							display: "flex",
							justifyContent: "space-between",
							paddingBottom: "10px"
						}
					},
					el("span", { style: { fontWeight: "500" } }, __("Avg Words/Sentence:", "developry-nexus")),
					el("span", { style: { fontSize: "16px", fontWeight: "bold" } }, avgWordsPerSentence.toFixed(1))
				)
			),

			// Pro Upgrade Notice
			el(
				"div",
				{
					style: {
						padding: "16px",
						background: "linear-gradient(135deg, #667eea 0%, #764ba2 100%)",
						borderRadius: "8px",
						color: "white",
						textAlign: "center"
					}
				},
				el(
					"p",
					{ style: { marginTop: 0, fontWeight: "600", fontSize: "14px" } },
					__("View Advanced Analysis", "developry-nexus")
				),
				el(
					"ul",
					{
						style: {
							margin: "0 0 10px 0",
							color: "white",
							fontSize: "12px",
							lineHeight: "1.6",
							textAlign: "left"
						}
					},
					el("li", null, "✓ " + __("Real-time Flesch Reading Ease score", "developry-nexus")),
					el("li", null, "✓ " + __("Grade level analysis", "developry-nexus")),
					el("li", null, "✓ " + __("Sentence complexity metrics", "developry-nexus")),
					el("li", null, "✓ " + __("Improvement suggestions", "developry-nexus"))
				),
				el(
					"a",
					{
						href: "https://developrythemes.com",
						target: "_blank",
						rel: "noopener noreferrer",
						style: {
							display: "inline-block",
							padding: "8px 20px",
							background: "#fbbf24",
							color: "#1f2937",
							borderRadius: "4px",
							textDecoration: "none",
							fontWeight: "600",
							fontSize: "14px"
						}
					},
					__("Learn More", "developry-nexus")
				)
			)
		);
	};

	/**
	 * Pro Features Panel Component
	 */
	const ProFeaturesPanel = function () {
		const config = window.developryNexusEditorSidebar || {};
		const isProActive = config.isProActive;

		if (isProActive) {
			const meta = useSelect(function (select) {
				return select("core/editor").getEditedPostAttribute("meta");
			}, []);

			const { editPost } = useDispatch("core/editor");
			const contentProvenance =
				meta && meta._developry_nexus_content_provenance ? meta._developry_nexus_content_provenance : "";
			const isAIGenerated = meta && meta._developry_nexus_ai_generated ? meta._developry_nexus_ai_generated : false;

			return el(
				PanelBody,
				{ title: __("Content Provenance", "developry-nexus"), initialOpen: false },
				el(ToggleControl, {
					label: __("AI Generated Content", "developry-nexus"),
					help: __("Mark this content as AI-generated for transparency", "developry-nexus"),
					checked: isAIGenerated,
					onChange: function (value) {
						editPost({ meta: { _developry_nexus_ai_generated: value } });
					}
				}),
				el(TextareaControl, {
					label: __("Provenance Notes", "developry-nexus"),
					help: __("Add notes about content sources, AI tools used, or human editing", "developry-nexus"),
					value: contentProvenance,
					onChange: function (value) {
						editPost({ meta: { _developry_nexus_content_provenance: value } });
					},
					rows: 3
				})
			);
		}

		// Show upgrade CTA if Pro is not active
		return el(
			PanelBody,
			{ title: __("⭐ Pro Features", "developry-nexus"), initialOpen: true },
			el(
				"div",
				{
					style: {
						padding: "16px",
						background: "linear-gradient(135deg, #667eea 0%, #764ba2 100%)",
						borderRadius: "8px",
						color: "white",
						textAlign: "center"
					}
				},
				el(
					"p",
					{ style: { marginTop: 0, fontWeight: "600", fontSize: "14px" } },
					__("Unlock Premium Features", "developry-nexus")
				),
				el(
					"ul",
					{
						style: {
							margin: "0 0 10px 0",
							color: "white",
							fontSize: "12px",
							lineHeight: "1.6",
							textAlign: "left"
						}
					},
					el("li", null, "✓ " + __("SEO & Schema Settings", "developry-nexus")),
					el("li", null, "✓ " + __("AI & Structured Summary", "developry-nexus")),
					el("li", null, "✓ " + __("Content Provenance Tracking", "developry-nexus")),
					el("li", null, "✓ " + __("Advanced Readability Analysis", "developry-nexus")),
					el("li", null, "✓ " + __("AI SEO Suggestions", "developry-nexus")),
					el("li", null, "✓ " + __("12 Pro Styles, 16+ Templates, 20+ Patterns", "developry-nexus"))
				),
				el(
					"a",
					{
						href: config.upgradeUrl || "https://developrythemes.com",
						target: "_blank",
						rel: "noopener noreferrer",
						style: {
							display: "inline-block",
							padding: "8px 20px",
							background: "#fbbf24",
							color: "#1f2937",
							borderRadius: "4px",
							textDecoration: "none",
							fontWeight: "600",
							fontSize: "14px"
						}
					},
					__("Upgrade to Pro", "developry-nexus")
				)
			)
		);
	};

	/**
	 * Main Sidebar Component
	 */
	const DevelopryNexusSidebar = function () {
		const { Slot } = wp.components;

		return el(
			wp.element.Fragment,
			null,
			el(
				PluginSidebarMoreMenuItem,
				{
					target: "developry-nexus-sidebar",
					icon: "admin-generic"
				},
				__("DevelopryNexus Settings", "developry-nexus")
			),
			el(
				PluginSidebar,
				{
					name: "developry-nexus-sidebar",
					title: __("DevelopryNexus Settings", "developry-nexus"),
					icon: "admin-generic"
				},
				el(ReadabilityPanel),
				el(Slot, { name: "DevelopryNexusSidebarPanels" }),
				el(ProFeaturesPanel)
			)
		);
	};

	// Register the plugin
	registerPlugin("developry-nexus-sidebar", {
		render: DevelopryNexusSidebar
	});
})(window.wp);
