<?php

/**
 * Widgets Extensions
 *
 * @package DeciduousLibrary
 * @subpackage WidgetsExtensions
 */
 
/**
 * This is a replacement function for the WordPress `get_sidebar()` function. The reason for this function
 * over the core function is because the core function does not provide the functionality needed to properly
 * implement what's needed, particularly the ability to add sidebar templates to a sub-directory.
 *
 * This function maintains compatibility with the core `get_sidebar()` function.  It does so in two ways:
 * 1) The `get_sidebar` hook is properly fired and 2) The core naming convention of sidebar templates
 * (`sidebar-$name.php` and `sidebar.php`) is preserved and given a higher priority than custom templates.
 *
 * @link http://core.trac.wordpress.org/ticket/15086
 * @link http://core.trac.wordpress.org/ticket/18676
 *
 */
function deciduous_get_sidebar( $name = '' ) {
	do_action( 'get_sidebar', $name ); // Core WordPress hook
	$templates = array();
	if ( $name ) {
		$templates[] = "sidebar-{$name}.php";
		$templates[] = "widget-areas/{$name}.php";
	}
	$templates[] = 'sidebar.php';
	$templates[] = 'widget-areas/sidebar.php';
	locate_template( $templates, true, false );
}

/**
 * Defines the array for creating and displaying the widgetized areas
 * in the WP-Admin and front-end of the site.
 * 
 * Filter: deciduous_f_widget_areas
 *
 * @uses deciduous_before_widget()
 * @uses deciduous_after_widget()
 * @uses deciduous_before_title()
 * @uses deciduous_after_title()
 * @return array
 */
function deciduous_widgets_array() {
	$deciduous_widget_areas = array(
		'Primary Aside' => array(
			'admin_menu_order' => 100,
			'args' => array (
				'name' => __( 'Primary Aside', 'deciduous' ),
				'id' => 'primary-aside',
                'description' => __('The primary widget area, most often used as a sidebar.', 'deciduous'),
				'before_widget' => deciduous_before_widget(),
				'after_widget' => deciduous_after_widget(),
				'before_title' => deciduous_before_title(),
				'after_title' => deciduous_after_title(),
				),
			'priority'		=> 10,
			),
		'Secondary Aside' => array(
			'admin_menu_order' => 200,
			'args' => array (
				'name' => __( 'Secondary Aside', 'deciduous' ),
				'id' => 'secondary-aside',
                'description' => __('The secondary widget area, most often used as a sidebar.', 'deciduous'),
				'before_widget' => deciduous_before_widget(),
				'after_widget' => deciduous_after_widget(),
				'before_title' => deciduous_before_title(),
				'after_title' => deciduous_after_title(),
				),
			'priority'		=> 10,
			),
		'1st Subsidiary Aside' => array(
			'admin_menu_order' => 300,
			'args' => array (
				'name' => __( '1st Subsidiary Aside', 'deciduous' ),
				'id' => '1st-subsidiary-aside',
                'description' => __('The 1st widget area in the footer.', 'deciduous'),
				'before_widget' => deciduous_before_widget(),
				'after_widget' => deciduous_after_widget(),
				'before_title' => deciduous_before_title(),
				'after_title' => deciduous_after_title(),
				),
			'priority'		=> 10,
			),
		'2nd Subsidiary Aside' => array(
			'admin_menu_order' => 400,
			'args' => array (
				'name' => __( '2nd Subsidiary Aside', 'deciduous' ),
				'id' => '2nd-subsidiary-aside',
                'description' => __('The 2nd widget area in the footer.', 'deciduous'),
				'before_widget' => deciduous_before_widget(),
				'after_widget' => deciduous_after_widget(),
				'before_title' => deciduous_before_title(),
				'after_title' => deciduous_after_title(),
				),
			'priority'		=> 10,
			),
		'3rd Subsidiary Aside' => array(
			'admin_menu_order' => 500,
			'args' => array (
				'name' => __( '3rd Subsidiary Aside', 'deciduous' ),
				'id' => '3rd-subsidiary-aside',
                'description' => __('The 3rd widget area in the footer.', 'deciduous'),
				'before_widget' => deciduous_before_widget(),
				'after_widget' => deciduous_after_widget(),
				'before_title' => deciduous_before_title(),
				'after_title' => deciduous_after_title(),
				),
			'priority'		=> 10,
		),
	);
	
	return apply_filters('deciduous_f_widget_areas', $deciduous_widget_areas);
	
}

/**
 * Registers Widget Areas(Sidebars) 
 *
 */
function deciduous_register_widgetareas() {
	$deciduous_widgetized_areas = deciduous_widgets_array();
	foreach ( $deciduous_widgetized_areas as $key => $value ) {
			register_sidebar( $deciduous_widgetized_areas[$key]['args'] );
	}  

}

add_action( 'widgets_init', 'deciduous_register_widgetareas' );


/**
 * Filters the order in which the Widget Areas (Sidebars) will be listed in the WP-Admin/Widgets
 * 
 * It sorts the array generated by deciduous_widgetized_areas() by [admin_menu_order]
 * allowing for child themes to filter deciduous_widgetized_areas to control
 * the sidebar list order in the WP-Admin/Widgets.
 * 
 * @see deciduous_widgetized_areas
 * @param array
 * @return array
 */
function deciduous_sort_widgetized_areas($content) {
	asort($content);
	return $content;
}
add_filter('deciduous_f_widget_areas', 'deciduous_sort_widgetized_areas', 100);



/**
 * Check Layout before displaying maain asides
 *
 * 
 */
function deciduous_main_asides_switch() {
	
	$current_layout = apply_filters( 'deciduous_f_current_theme_layout', deciduous_get_theme_opt( 'layout' ) );
	
	if ( in_array( $current_layout, deciduous_available_layout_slugs() ) && 'full-width' == $current_layout ) {
		$show = false;
	} else {
		$show = true;
	}
	
	$show = apply_filters( 'deciduous_f_main_asides_switch', $show );
	
	return $show;
}


/**
 * Returns the opening CSS markup before the widget area
 *
 * Filter: deciduous_before_widget_area
 * 
 * @param $hook determines the markup specific to the widget area
 * @return string 
 */
function deciduous_before_widget_area($hook) {
	$content =  "\n\t\t\t";
	if ( $hook == 'primary-aside' ) {
		$content .= '<aside id="primary" class="primary-aside aside main-aside" role="complementary">' . "\n\n";
	} elseif ( $hook == 'secondary-aside' ) {
		$content .= '<aside id="secondary" class="secondary-aside aside main-aside" role="complementary">' . "\n\n";
	} elseif ( $hook == '1st-subsidiary-aside' ) {
		$content .= '<aside id="first" class="first-sub-aside aside sub-aside" role="complementary">' . "\n\n";
	} elseif ( $hook == '2nd-subsidiary-aside' ) {
		$content .= '<aside id="second" class="second-sub-aside aside sub-aside" role="complementary">' . "\n\n";
	} elseif ( $hook == '3rd-subsidiary-aside' ) {
		$content .= '<aside id="third" class="third-sub-aside aside sub-aside" role="complementary">' . "\n\n";
	} else {
		$content .= '<aside id="' . $hook . '" class="' . $hook . ' aside" role="complementary">' ."\n";
	}
	$content .= "\t\t\t\t\t" . '<div class="inner xoxo">'. "\n\n\t\t\t\t";
	return apply_filters( 'deciduous_f_before_widget_area', $content, $hook );
}

/**
 * Returns the closing CSS markup before the widget area
 *
 * Filter: deciduous_after_widget_area
 * 
 * @param $hook determines the markup specific to the widget area
 * @return string 
 */
function deciduous_after_widget_area($hook) {
	$content = "\n\t\t\t\t" . '</div><!-- .inner -->' . "\n\t\t";
	if ( $hook == 'primary-aside' ) {
		$content .= '</aside><!-- .primary-aside -->' ."\n\n";
	} elseif ( $hook == 'secondary-aside' ) {
		$content .= '</aside><!-- .secondary-aside -->' ."\n\n";
	} elseif ( $hook == '1st-subsidiary-aside' ) {
		$content .= '</aside><!-- .first-sub-aside -->' ."\n\n";
	} elseif ( $hook == '2nd-subsidiary-aside' ) {
		$content .= '</aside><!-- .second-sub-aside -->' ."\n\n";
	} elseif ( $hook == '3rd-subsidiary-aside' ) {
		$content .= '</aside><!-- .third-sub-aside -->' ."\n\n";
	} else {
		$content .=  "\t\t\t" . '</aside><!-- .' . $hook . ' .aside -->' ."\n\n";
	} 
	return apply_filters( 'deciduous_f_after_widget_area', $content, $hook );
}




/**
 * Markup before the widget
 */
function deciduous_before_widget() {
	$content = "\t\t" . '<section id="%1$s" class="widget %2$s">'  . "\n\t\t\t\t\t\t\t";
	return apply_filters('deciduous_f_before_widget', $content);
}


/**
 * Markup after the widget
 */
function deciduous_after_widget() {
	$content = "\t\t\t\t" . '</section>' .  "\n\n";
	return apply_filters('deciduous_f_after_widget', $content);
}



/**
 * Markup before the widget title
 */
function deciduous_before_title() {
	$content =  "\n\t\t\t\t\t\t\t" . '<h3 class="widgettitle">';
	return apply_filters('deciduous_f_before_widget_title', $content);
}


/**
 * Markup after the widget title
 */
function deciduous_after_title() {
	$content = "</h3>\n\t\t\t\t";
	return apply_filters('deciduous_f_after_widget_title', $content);
}


/**
 * Enforce title on search widget
 * 
 * @since 2.0.0
 * 
 * @param  string  $title    The current widget title
 * @param  object  $instance The current widget instance
 * @param  string  $id_base  The registered id of current widget
 */
function deciduous_filter_search_widget( $title, $instance = '', $id_base = '' ) {
	
	if ( 'search' == $id_base ) {
		$title = apply_filters( 'widget_title', empty( $instance['title'] ) ? __( 'Search', 'deciduous' ) : $instance['title'] );
		$title = '<label for="s">' . $title . '</label>';
	}
	
	return $title;
}
add_filter( 'widget_title', 'deciduous_filter_search_widget', 10, 3 );



?>