<?php
/**
 * Custom Conary template tags
 *
 * Eventually, some of the functionality here could be replaced by core features.
 *
 *
 * @package Conary
 * @since Conary 1.0
 */

if ( ! function_exists( 'conary_entry_meta' ) ) :
/**
 * Prints HTML with meta information for the categories, tags.
 *
 * Create your own conary_entry_meta() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_entry_meta() {

	if ( 'post' === get_post_type() ) {
		$author_avatar_size = apply_filters( 'conary_author_avatar_size', 49 );
		printf( '<span class="byline"><span class="author vcard">%1$s<span>%2$s </span><a class="url fn n" href="%3$s">%4$s</a></span></span>',
			get_avatar( get_the_author_meta( 'user_email' ), $author_avatar_size ),
			esc_html_x( 'By', 'Used before post author name.', 'conary' ),
			esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
			get_the_author()
		);
	}

	if ( in_array( get_post_type(), array( 'post', 'attachment' ) ) ) {
		conary_entry_date();
	}

	if ( ( comments_open() || get_comments_number() ) ) {
		echo '<span class="comments-link">';
		echo conary_svg_icon('comment');
		comments_popup_link(
			sprintf(
				wp_kses(
					/* translators: %s: post title that hidden in a span with class "screen-reader-text" */
					__( '0 Comment<span class="screen-reader-text"> on %s</span>', 'conary' ),
					array(
						'span' => array( "class" => array() )
					)
				),
				get_the_title()
			)
		);
		echo '</span>';
	}
}
endif;

if ( ! function_exists( 'conary_entry_date' ) ) :
/**
 * Prints HTML with date information for current post.
 *
 * Create your own conary_entry_date() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_entry_date() {
	$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';

	if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
		$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
	}

	$time_string = sprintf( $time_string,
		esc_attr( get_the_date( 'c' ) ),
		get_the_date(),
		esc_attr( get_the_modified_date( 'c' ) ),
		get_the_modified_date()
	);

	printf( '<span class="posted-on"><span>%1$s </span><a href="%2$s" rel="bookmark">%3$s</a></span>',
		esc_html_x( 'On', 'Used before publish date.', 'conary' ),
		esc_url( get_permalink() ),
		$time_string
	);
}
endif;

if ( ! function_exists( 'conary_entry_taxonomies' ) ) :
/**
 * Prints HTML with category and tags for current post.
 *
 * Create your own conary_entry_taxonomies() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_entry_taxonomies() {
	$categories_list = get_the_category_list( esc_html_x( ', ', 'Used between list items, there is a space after the comma.', 'conary' ) );
	if ( $categories_list && conary_categorized_blog() ) {
		printf( '<span class="cat-links"><span class="screen-reader-text">%1$s </span>%2$s</span>',
			esc_html_x( 'Categories', 'Used before category names.', 'conary' ),
			$categories_list
		);
	}
}
endif;

if ( ! function_exists( 'conary_entry_tags' ) ) :
/**
 * Prints HTML with tags for current post.
 *
 * Create your own conary_entry_tags() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_entry_tags() {
	$tags_list = get_the_tag_list( '', esc_html_x( ', ', 'Used between list items, there is a space after the comma.', 'conary' ) );
	if ( $tags_list ) {
		printf( '<span class="tags-links"><span>%1$s </span>%2$s</span>',
			esc_html_x( 'Tags', 'Used before tag names.', 'conary' ),
			$tags_list
		);
	}

}
endif;

if ( ! function_exists('conary_edit_post_link') ) :
/**
 * Prints HTML edit post link.
 *
 * Create your own conary_edit_post_link() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_edit_post_link() {

	edit_post_link(
		sprintf(
			wp_kses(
				/* translators: %s is post title that hidden in a span with class "screen-reader-text" */
				__( 'Edit<span class="screen-reader-text"> "%s"</span>', 'conary' ),
				array (
					'span' => array (
						'class' => array()
					)
				)
			),
			get_the_title()
		),
		'<span class="edit-link">',
		'</span>'
	);

}
endif;

if ( ! function_exists( 'conary_post_thumbnail' ) ) :
/**
 * Displays an optional post thumbnail.
 *
 * Wraps the post thumbnail in an anchor element on index views, or a div
 * element when on single views.
 *
 * Create your own conary_post_thumbnail() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_post_thumbnail() {
	if ( post_password_required() || is_attachment() || ! has_post_thumbnail() ) {
		return;
	}

	if ( is_singular() ) :
	?>

	<div class="post-thumbnail">
		<?php the_post_thumbnail(); ?>
	</div><!-- .post-thumbnail -->

	<?php else : ?>

	<a class="post-thumbnail" href="<?php the_permalink(); ?>" aria-hidden="true">
		<?php the_post_thumbnail( 'post-thumbnail', array( 'alt' => the_title_attribute( 'echo=0' ) ) ); ?>
	</a>

	<?php endif; // End is_singular()
}
endif;

if ( ! function_exists( 'conary_excerpt' ) ) :
	/**
	 * Displays the optional excerpt.
	 *
	 * Wraps the excerpt in a div element.
	 *
	 * Create your own conary_excerpt() function to override in a child theme.
	 *
	 * @since Conary 1.0
	 *
	 * @param string $class Optional. Class string of the div element. Defaults to 'entry-summary'.
	 */
	function conary_excerpt( $class = 'entry-summary' ) {
		$class = esc_attr( $class );

		if ( has_excerpt() || is_search() ) : ?>
			<div class="<?php echo esc_attr( $class ); ?>">
				<?php the_excerpt(); ?>
			</div>
		<?php endif;
	}
endif;

if ( ! function_exists( 'conary_excerpt_more' ) && ! is_admin() ) :
/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... and
 * a 'Read more' link.
 *
 * Create your own conary_excerpt_more() function to override in a child theme.
 *
 * @since Conary 1.0
 *
 * @return string 'Read More' link prepended with an ellipsis.
 */
function conary_excerpt_more() {
	$link = sprintf( '<a href="%1$s" class="more-link">%2$s</a>',
		esc_url( get_permalink( get_the_ID() ) ),
		sprintf(
			wp_kses(
				/* translators: %s: post title that hidden in a span with class "screen-reader-text" */
				__( 'Read More<span class="screen-reader-text"> "%s"</span>', 'conary' ),
				array (
					'span' => array (
						'class' => array()
					)
				)
			),
			get_the_title( get_the_ID() )
		)
	);
	return ' &hellip; ' . $link;
}
add_filter( 'excerpt_more', 'conary_excerpt_more' );
endif;

if ( ! function_exists( 'conary_excerpt_length' ) && ! is_admin() ) :
/**
 * Change the excerpt number
 *
 * @since Conary 1.0
 *
 * @return number Number for excerpt.
 */
function conary_excerpt_length( $length ) {
	return 30;
}
add_filter( 'excerpt_length', 'conary_excerpt_length', 999 );
endif;

if ( ! function_exists( 'conary_categorized_blog' ) ) :
/**
 * Determines whether blog/site has more than one category.
 *
 * Create your own conary_categorized_blog() function to override in a child theme.
 *
 * @since Conary 1.0
 *
 * @return bool True if there is more than one category, false otherwise.
 */
function conary_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'conary_categories' ) ) ) {
		// Create an array of all the categories that are attached to posts.
		$all_the_cool_cats = get_categories( array(
			'fields'     => 'ids',
			// We only need to know if there is more than one category.
			'number'     => 2,
		) );

		// Count the number of categories that are attached to the posts.
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'conary_categories', $all_the_cool_cats );
	}

	if ( $all_the_cool_cats > 1 ) {
		// This blog has more than 1 category so conary_categorized_blog should return true.
		return true;
	} else {
		// This blog has only 1 category so conary_categorized_blog should return false.
		return false;
	}
}
endif;

/**
 * Flushes out the transients used in conary_categorized_blog().
 *
 * @since Conary 1.0
 */
function conary_category_transient_flusher() {
	if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
		return;
	}
	// Like, beat it. Dig?
	delete_transient( 'conary_categories' );
}
add_action( 'edit_category', 'conary_category_transient_flusher' );
add_action( 'save_post',     'conary_category_transient_flusher' );

if ( ! function_exists( 'conary_the_custom_logo' ) ) :
/**
 * Displays the optional custom logo.
 *
 * Does nothing if the custom logo is not available.
 *
 * @since Conary 1.0
 */
function conary_the_custom_logo() {
	if ( function_exists( 'the_custom_logo' ) ) {
		the_custom_logo();
	}
}
endif;

/**
 * Print markup for SVG icon.
 *
 * @since Conary 1.0
 * @param string $icon keyword for icon name
 */
function conary_svg_icon ( $icon ) {
	$icon = esc_attr( $icon );
	$symbol = '<svg class="icon icon-' . $icon . '"><use xlink:href="' . get_template_directory_uri() . '/icons/symbol-defs.svg#icon-' . $icon . '"></use></svg>';

	return $symbol;
}

/**
 * Custom function to retrieve the archive title based on the queried object.
 *
 * @since Conary 1.0
 *
 * @return string Archive title.
 */
function conary_archive_title() {
	if ( is_category() ) {
		$title = sprintf(
			'<div class="archive-title-pretext entry-meta">%1$s</div><h1 class="page-title">%2$s</h1><div class="taxonomy-description">%3$s</div>',
			esc_html__( 'Under Category', 'conary' ),
			single_cat_title( '', false ),
			term_description()
		);
	} elseif ( is_tag() ) {
		$title = sprintf(
			'<div class="archive-title-pretext entry-meta">%1$s</div><h1 class="page-title">%2$s</h1><div class="taxonomy-description">%3$s</div>',
			esc_html__( 'Tagged As', 'conary' ),
			single_tag_title( '', false ),
			term_description()
		);
	} elseif ( is_author() ) {
		$title = sprintf(
			'<div class="archive-title-pretext entry-meta">%1$s</div><h1 class="page-title">%2$s</h1><div class="author-avatar">%3$s</div><div class="taxonomy-description">%4$s</div>',
			esc_html__( 'Author by', 'conary' ),
			get_the_author(),
			get_avatar( get_the_author_meta( 'user_email' ), 80 ),
			get_the_author_meta( 'description' )
		);
	} elseif ( is_year() ) {
		$title = sprintf(
			'<div class="archive-title-pretext entry-meta">%1$s</div><h1 class="page-title">%2$s</h1>',
			esc_html__( 'Posted in year', 'conary' ),
			get_the_date( esc_html_x( 'Y', 'yearly archives date format', 'conary' ) )
		);
	} elseif ( is_month() ) {
		$title = sprintf(
			'<div class="archive-title-pretext entry-meta">%1$s</div><h1 class="page-title">%2$s</h1>',
			esc_html__( 'Posted in month', 'conary' ),
			get_the_date( esc_html_x( 'F Y', 'monthly archives date format', 'conary' ) )
		);
	} elseif ( is_day() ) {
		$title = sprintf(
			'<div class="archive-title-pretext entry-meta">%1$s</div><h1 class="page-title">%2$s</h1>',
			esc_html__( 'Posted in', 'conary' ),
			get_the_date( esc_html_x( 'F j, Y', 'daily archives date format', 'conary' ) )
		);
	} elseif ( is_tax( 'post_format' ) ) {
		if ( is_tax( 'post_format', 'post-format-aside', 'conary' ) ) {
			$title = esc_html_x( 'Asides', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-gallery', 'conary' ) ) {
			$title = esc_html_x( 'Galleries', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-image', 'conary' ) ) {
			$title = esc_html_x( 'Images', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-video', 'conary' ) ) {
			$title = esc_html_x( 'Videos', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-quote', 'conary' ) ) {
			$title = esc_html_x( 'Quotes', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-link', 'conary' ) ) {
			$title = esc_html_x( 'Links', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-status', 'conary' ) ) {
			$title = esc_html_x( 'Statuses', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-audio', 'conary' ) ) {
			$title = esc_html_x( 'Audio', 'post format archive title', 'conary' );
		} elseif ( is_tax( 'post_format', 'post-format-chat', 'conary' ) ) {
			$title = esc_html_x( 'Chats', 'post format archive title', 'conary' );
		}
	} elseif ( is_post_type_archive() ) {
		/* translators: 1: post type title */
		$title = sprintf( esc_html__( 'Archives: %s', 'conary' ), post_type_archive_title( '', false ) );
	} elseif ( is_tax() ) {
		$tax = get_taxonomy( get_queried_object()->taxonomy );
		/* translators: 1: Taxonomy singular name, 2: Current taxonomy term */
		$title = sprintf( esc_html__( '%1$s: %2$s', 'conary' ), $tax->labels->singular_name, single_term_title( '', false ) );
	} else {
		$title = esc_html__( 'Archives', 'conary' );
	}

	echo apply_filters( 'conary_archive_title', $title );
}

/**
 * Breadcrumbs
 *
 * @since Conary 1.0
 */
function conary_breadcrumbs() {
	if ( !is_front_page() ) {
		echo '<nav class="site-breadcrumbs" ><div> <a href="';
		echo home_url();
		echo '">';
		esc_html_e('Home', 'conary');
		echo "</a> ";
	}

	if ( (is_category() || is_single()) && !is_attachment() ) {
		$category = get_the_category();
		if (count($category) > 0){
			$ID = $category[0]->cat_ID;
			if ( $ID ) {
				echo get_category_parents($ID, TRUE, ' ', FALSE );
			}
		}
	}

	if(is_single() || is_page()) {
		if ( !is_front_page() ){
			echo '<span>';
			echo wp_kses( get_the_title(), array() );
			echo '</span>';
		}
	}
	echo '<span>';
	if(is_tag()){ esc_html_e('Tag: ', 'conary') . single_tag_title('',FALSE); }
	if(is_404()){ esc_html_e('404 - Page not Found', 'conary') ; }
	if(is_search()){ esc_html_e('Search', 'conary'); }
	if(is_year()){ echo get_the_time('Y'); }
	if(is_month()){ echo get_the_time('F Y'); }
	if (is_author()){ esc_html_e('Posts by ', 'conary') . get_the_author(); }
	echo '</span>';

	if ( !is_front_page() ) {
		echo "</div></nav>";
	}
}

/**
 * Render the footer credit, print from the footer_credit options, or default.
 *
 * @since Conary 1.0
 *
 * @return string
 */
function conary_footer_credit( $echo = false ) {

	if ( $footer_credit = get_theme_mod( 'footer_credit', false ) ) {
		$footer_credit = conary_sanitize_footer_credit( $footer_credit );
	}else{

		$footer_credit = conary_sanitize_footer_credit(
			sprintf(
				/* translators: 1. Site title, 2. WordPress dot org link, 3. Designer link */
				esc_html_x('%1$s Powered by %2$s Designed by %3$s.', '%1$s for homepage link, %2$s for wordpress.org link, %3$s for theme designer link', 'conary'),
				'<a href="' . esc_url( home_url('/') ) .'" rel="home">' . get_bloginfo('name') . '</a>',
				'<a href="' . esc_url( 'https://wordpress.org/' ) .'">' . esc_html__('WordPress', 'conary') . '</a>',
				'<a href="' . esc_url( 'https://fancythemes.com/' ) .'">' . esc_html__('FancyThemes', 'conary') . '</a>'
			)
		);
	}

	if ( $echo ) echo $footer_credit;
	return $footer_credit;
}
