<?php
/**
 * Conary functions and definitions
 *
 * Set up the theme and provides some helper functions, which are used in the
 * theme as custom template tags. Others are attached to action and filter
 * hooks in WordPress to change core functionality.
 *
 * When using a child theme you can override certain functions (those wrapped
 * in a function_exists() call) by defining them first in your child theme's
 * functions.php file. The child theme's functions.php file is included before
 * the parent theme's file, so the child theme functions would be used.
 *
 * @link https://codex.wordpress.org/Theme_Development
 * @link https://codex.wordpress.org/Child_Themes
 *
 * Functions that are not pluggable (not wrapped in function_exists()) are
 * instead attached to a filter or action hook.
 *
 * For more information on hooks, actions, and filters,
 * {@link https://codex.wordpress.org/Plugin_API}
 *
 *
 * @package Conary
 * @since Conary 1.0
 */

if ( ! function_exists( 'conary_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 *
 * Create your own conary_setup() function to override in a child theme.
 *
 * @since Conary 1.0
 */
function conary_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed at WordPress.org. See: https://translate.wordpress.org/projects/wp-themes/conary
	 * If you're building a theme based on Conary, use a find and replace
	 * to change 'conary' to the name of your theme in all the template files
	 */
	load_theme_textdomain( 'conary' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for custom logo.
	 *
	 *  @since Conary 1.0
	 */
	add_theme_support( 'custom-logo', array(
		'height'      => 240,
		'width'       => 240,
		'flex-height' => true,
	) );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link http://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 1200, 9999 );

	/*
	 * Add a new image size, square 300px
	 */
	add_image_size( 'square-300', 300, 300, true );

	// This theme uses wp_nav_menu() in two locations.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu', 'conary' ),
		'social'  => esc_html__( 'Social Links Menu', 'conary' ),
		'footer-menu'  => esc_html__( 'Footer Menu', 'conary' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		//'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	/*
	 * Enable support for Post Formats.
	 *
	 * See: https://codex.wordpress.org/Post_Formats
	 */
	add_theme_support( 'post-formats', array(
		'video',
		'gallery',
		'audio',
	) );

	/*
	 * This theme styles the visual editor to resemble the theme style,
	 * specifically font, colors, icons, and column width.
	 */
	add_editor_style( array( 'css/editor-style.css', conary_fonts_url() ) );

	// Indicate widget sidebars can use selective refresh in the Customizer.
	add_theme_support( 'customize-selective-refresh-widgets' );
}
endif; // conary_setup
add_action( 'after_setup_theme', 'conary_setup' );

/**
 * Sets the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 *
 * @since Conary 1.0
 */
function conary_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'conary_content_width', 840 );
}
add_action( 'after_setup_theme', 'conary_content_width', 0 );

/**
 * Registers a widget area.
 *
 * @link https://developer.wordpress.org/reference/functions/register_sidebar/
 *
 * @since Conary 1.0
 */
function conary_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'conary' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'conary' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar 2 (Left Sidebar)', 'conary' ),
		'id'            => 'sidebar-2',
		'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'conary' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Full Width Feature Widget', 'conary' ),
		'id'            => 'feature-widget-full-width',
		'before_widget' => '<section id="%1$s" class="widget feature-widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title"><span>',
		'after_title'   => '</span></h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Full Width Footer Widget', 'conary' ),
		'id'            => 'footer-widget-full-width',
		'before_widget' => '<section id="%1$s" class="widget footer-widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget 1', 'conary' ),
		'id'            => 'footer-widget-1',
		'before_widget' => '<section id="%1$s" class="widget footer-widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget 2', 'conary' ),
		'id'            => 'footer-widget-2',
		'before_widget' => '<section id="%1$s" class="widget footer-widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget 3', 'conary' ),
		'id'            => 'footer-widget-3',
		'before_widget' => '<section id="%1$s" class="widget footer-widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	register_sidebar( array(
		'name'          => esc_html__( 'Footer Widget 4', 'conary' ),
		'id'            => 'footer-widget-4',
		'before_widget' => '<section id="%1$s" class="widget footer-widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

}
add_action( 'widgets_init', 'conary_widgets_init' );

if ( ! function_exists( 'conary_fonts_url' ) ) :
/**
 * Register Google fonts for Conary.
 *
 * Create your own conary_fonts_url() function to override in a child theme.
 *
 * @since Conary 1.0
 *
 * @return string Google fonts URL for the theme.
 */
function conary_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	/* translators: If there are characters in your language that are not supported by Source Sans Pro, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== esc_html_x( 'on', 'Source Sans Pro font: on or off', 'conary' ) ) {
		$fonts[] = 'Source Sans Pro:400,400i,600,600i';
	}

	/* translators: If there are characters in your language that are not supported by Source Serif Serif, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== esc_html_x( 'on', 'Source Serif Serif font: on or off', 'conary' ) ) {
		$fonts[] = 'Source Serif Pro:400,700';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
		), '//fonts.googleapis.com/css' );
	}

	return $fonts_url;
}
endif;

/**
 * Handles JavaScript detection.
 *
 * Adds a `js` class to the root `<html>` element when JavaScript is detected.
 *
 * @since Conary 1.0
 */
function conary_javascript_detection() {
	echo "<script>(function(html){html.className = html.className.replace(/\bno-js\b/,'js')})(document.documentElement);</script>\n";
}
add_action( 'wp_head', 'conary_javascript_detection', 0 );

/**
 * Enqueues scripts and styles.
 *
 * @since Conary 1.0
 */
function conary_scripts() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'conary-fonts', conary_fonts_url(), array(), null );

	// Theme stylesheet.
	wp_enqueue_style( 'conary-style', get_stylesheet_uri() );

	// Load the Internet Explorer specific stylesheet.
	wp_enqueue_style( 'conary-ie', get_template_directory_uri() . '/css/ie.css', array( 'conary-style' ), '20160816' );
	wp_style_add_data( 'conary-ie', 'conditional', 'lt IE 10' );

	// Load the Internet Explorer 8 specific stylesheet.
	wp_enqueue_style( 'conary-ie8', get_template_directory_uri() . '/css/ie8.css', array( 'conary-style' ), '20160816' );
	wp_style_add_data( 'conary-ie8', 'conditional', 'lt IE 9' );

	// Load the Internet Explorer 7 specific stylesheet.
	wp_enqueue_style( 'conary-ie7', get_template_directory_uri() . '/css/ie7.css', array( 'conary-style' ), '20160816' );
	wp_style_add_data( 'conary-ie7', 'conditional', 'lt IE 8' );

	// Load the html5 shiv.
	wp_enqueue_script( 'conary-html5', get_template_directory_uri() . '/js/html5.js', array(), '3.7.3' );
	wp_script_add_data( 'conary-html5', 'conditional', 'lt IE 9' );

	wp_enqueue_script( 'conary-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), '20160816', true );

	wp_enqueue_script( 'svgxuse', get_template_directory_uri() . '/js/svgxuse.min.js', array(), '1.2.4' );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

	if ( is_singular() && wp_attachment_is_image() ) {
		wp_enqueue_script( 'conary-keyboard-image-navigation', get_template_directory_uri() . '/js/keyboard-image-navigation.js', array( 'jquery' ), '20160816' );
	}

	wp_enqueue_script( 'conary-script', get_template_directory_uri() . '/js/functions.js', array( 'jquery' ), '20160816', true );

	wp_localize_script( 'conary-script', 'screenReaderText', array(
		'expand'   => esc_html__( 'expand child menu', 'conary' ),
		'collapse' => esc_html__( 'collapse child menu', 'conary' ),
		'loadMoreText' => esc_html__( 'Load More', 'conary' ),
		'loadingText'  => esc_html__( 'Loading...', 'conary' ),

	) );

	wp_enqueue_script( 'flexslider', get_template_directory_uri() . '/js/jquery.flexslider-min.js', array( 'jquery' ), '20160816', true );

	$featured_posts_type =  get_theme_mod('featured_posts_type', 'carousel');
	wp_localize_script( 'conary-script', 'sliderOptions', array(
		'slideType'     => $featured_posts_type,
		'slideshow'			=> get_theme_mod( 'featured_posts_auto_slide', false ),
		'slideshowSpeed'	=> get_theme_mod( 'featured_posts_auto_slide_time', 7000 ),
		'prevText'			=> '<span class="screen-reader-text">'. esc_html__('Previous', 'conary') .'</span>' . conary_svg_icon('arrow-prev') ,
		'nextText'			=> '<span class="screen-reader-text">'. esc_html__('Next', 'conary') .'</span>' . conary_svg_icon('arrow-next'),
		'itemWidth'			=> ( $featured_posts_type == 'carousel' ) ? 300 : 0,
		'minItems'			=> ( $featured_posts_type == 'carousel' ) ? 3 : 1,
		'maxItems'			=> ( $featured_posts_type == 'carousel' ) ? 3 : 0,
	) );

	wp_enqueue_script( 'conary-fitvids-script', get_template_directory_uri() . '/js/jquery.fitvids.js', array( 'jquery' ), '20160816', true );
}
add_action( 'wp_enqueue_scripts', 'conary_scripts' );

/**
 * Enqueues admin scripts and styles.
 *
 * @since Conary 1.0
 */
function conary_admin_enqueue_scripts( $hook ) {
	if ( $hook == 'widgets.php' ) {
		wp_enqueue_style( 'conary-admin', get_template_directory_uri() . '/css/admin.css' );
	}
}
add_action( 'admin_enqueue_scripts', 'conary_admin_enqueue_scripts' );



/**
 * Adds custom classes to the array of body classes.
 *
 * @since Conary 1.0
 *
 * @param array $classes Classes for the body element.
 * @return array (Maybe) filtered body classes.
 */
function conary_body_classes( $classes ) {
	// Adds a class of custom-background-image to sites with a custom background image.
	if ( get_background_image() ) {
		$classes[] = 'custom-background-image';
	}

	// Adds a class of group-blog to sites with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}
	$layout_class = '';
	if ( is_active_sidebar( 'sidebar-2' ) && ( is_home() || is_page_template('blog3-page.php' ) ) ) {
		$layout_class .= 'right-';
	}

	if ( is_active_sidebar( 'sidebar-1' ) ) {
		$layout_class .= 'left-';
	}

	if ( ! is_active_sidebar( 'sidebar-1' ) && ! is_active_sidebar( 'sidebar-2' ) ) {
		$layout_class .= 'no-';
	}

	$layout_class .= 'sidebar';

	$classes[] = $layout_class;

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	return $classes;
}
add_filter( 'body_class', 'conary_body_classes' );

/**
 * Converts a HEX value to RGB.
 *
 * @since Conary 1.0
 *
 * @param string $color The original color, in 3- or 6-digit hexadecimal form.
 * @return array Array containing RGB (red, green, and blue) values for the given
 *               HEX code, empty array otherwise.
 */
function conary_hex2rgb( $color ) {
	$color = trim( $color, '#' );

	if ( strlen( $color ) === 3 ) {
		$r = hexdec( substr( $color, 0, 1 ).substr( $color, 0, 1 ) );
		$g = hexdec( substr( $color, 1, 1 ).substr( $color, 1, 1 ) );
		$b = hexdec( substr( $color, 2, 1 ).substr( $color, 2, 1 ) );
	} else if ( strlen( $color ) === 6 ) {
		$r = hexdec( substr( $color, 0, 2 ) );
		$g = hexdec( substr( $color, 2, 2 ) );
		$b = hexdec( substr( $color, 4, 2 ) );
	} else {
		return array();
	}

	return array( 'red' => $r, 'green' => $g, 'blue' => $b );
}

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/sanitize-callbacks.php';
require get_template_directory() . '/inc/customizer-simple.php';
require get_template_directory() . '/inc/customizer.php';

/**
 * Posts Widget.
 */
require get_template_directory() . '/inc/widgets/recent-posts.php';

/**
 * Add custom image sizes attribute to enhance responsive image functionality
 * for content images
 *
 * @since Conary 1.0
 *
 * @param string $sizes A source size value for use in a 'sizes' attribute.
 * @param array  $size  Image size. Accepts an array of width and height
 *                      values in pixels (in that order).
 * @return string A source size value for use in a content image 'sizes' attribute.
 */
function conary_content_image_sizes_attr( $sizes, $size ) {
	$width = $size[0];

	840 <= $width && $sizes = '(max-width: 709px) 85vw, (max-width: 909px) 67vw, (max-width: 1362px) 62vw, 840px';

	if ( 'page' === get_post_type() ) {
		840 > $width && $sizes = '(max-width: ' . $width . 'px) 85vw, ' . $width . 'px';
	} else {
		840 > $width && 600 <= $width && $sizes = '(max-width: 709px) 85vw, (max-width: 909px) 67vw, (max-width: 984px) 61vw, (max-width: 1362px) 45vw, 600px';
		600 > $width && $sizes = '(max-width: ' . $width . 'px) 85vw, ' . $width . 'px';
	}

	return $sizes;
}
add_filter( 'wp_calculate_image_sizes', 'conary_content_image_sizes_attr', 10 , 2 );

/**
 * Add custom image sizes attribute to enhance responsive image functionality
 * for post thumbnails
 *
 * @since Conary 1.0
 *
 * @param array $attr Attributes for the image markup.
 * @param int   $attachment Image attachment ID.
 * @param array $size Registered image size or flat array of height and width dimensions.
 * @return string A source size value for use in a post thumbnail 'sizes' attribute.
 */
function conary_post_thumbnail_sizes_attr( $attr, $attachment, $size ) {
	if ( 'post-thumbnail' === $size ) {
		is_active_sidebar( 'sidebar-1' ) && $attr['sizes'] = '(max-width: 709px) 85vw, (max-width: 909px) 67vw, (max-width: 984px) 60vw, (max-width: 1362px) 62vw, 840px';
		! is_active_sidebar( 'sidebar-1' ) && $attr['sizes'] = '(max-width: 709px) 85vw, (max-width: 909px) 67vw, (max-width: 1362px) 88vw, 1200px';
	}
	return $attr;
}
add_filter( 'wp_get_attachment_image_attributes', 'conary_post_thumbnail_sizes_attr', 10 , 3 );

/**
 * Modifies tag cloud widget arguments to have all tags in the widget same font size.
 *
 * @since Conary 1.0
 *
 * @param array $args Arguments for tag cloud widget.
 * @return array A new modified arguments.
 */
function conary_widget_tag_cloud_args( $args ) {
	$args['largest'] = 1;
	$args['smallest'] = 1;
	$args['unit'] = 'em';
	return $args;
}
add_filter( 'widget_tag_cloud_args', 'conary_widget_tag_cloud_args' );

/**
 * Adding extra markup on widget title, wrap the last word with a <span>
 *
 * @since Conary 1.0
 *
 */
function conary_widget_title( $title, $instance, $id_base) {

	$array_word = explode(' ', $title );
	$array_word[ count($array_word) - 1 ] = '<span>' . $array_word[ count($array_word) - 1 ] . '</span>';
	$title = implode(' ', $array_word);

	return $title;
}
//add_filter('widget_title', 'conary_widget_title', 10, 3);

/**
 * Setting a new excerpt length
 *
 * @since Conary 1.0
 *
 */
function conary_excerpt_length( $length ) {
	return 20;
}
add_filter( 'excerpt_length', 'conary_excerpt_length', 999 );

/**
 * Replace the string 'icon_replace' on SVG use xlink:href attribute from wp_nav_menu's link_before argument by the escaped domain name from link url
 * the dot(.) on domain replaced by dash(-), eg. plus.google.com -> plus-google-com
 * so for the menu with URL linked to google plus domain will have SVG icon with use tag like
 * <use xlink:href="http://your-domain/wp-content/themes/conary/icons/symbol-defs.svg#icon-social-plus-google-com"></use>
 *
 * see also function conary_svg_icon() in the template-tags.php
 * see also the declaration of wp_nav_menu for theme location "social" in the header.php and footer.php
 *
 * @since Conary 1.0
 *
 * @param string $item_output The menu item's starting HTML output.
 * @param object $item		Menu item data object.
 * @param int	$depth	   Depth of menu item. Used for padding.
 * @param array  $args		An array of arguments. @see wp_nav_menu()
 */
function conary_social_menu_item_output ( $item_output, $item, $depth, $args ) {
	$parsed_url = parse_url( $item->url);
	$url_host = ! empty( $parsed_url['host'] ) ? str_replace( '.', '-', $parsed_url['host'] ) : '';
	$output = str_replace('icon_replace', 'social-' . $url_host, $item_output);
	return $output;
}

/**
 * Add featured image as background image to post navigation elements.
 *
 * @since Conary 1.0
 *
 * @see wp_add_inline_style()
 */
function conary_post_nav_image() {
	if ( ! is_single() ) {
		return;
	}

	$previous = ( is_attachment() ) ? get_post( get_post()->post_parent ) : get_adjacent_post( false, '', true );
	$next     = get_adjacent_post( false, '', false );
	$css      = '';

	if ( is_attachment() && 'attachment' == $previous->post_type ) {
		return;
	}

	if ( $previous &&  has_post_thumbnail( $previous->ID ) ) {
		$prevthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $previous->ID ), 'thumbnail' );
		$css .= '
			.post-navigation .nav-previous { background-image: url(' . esc_url( $prevthumb[0] ) . '); background-size: 50px; background-repeat: no-repeat; background-position: 0 40px; }
			.post-navigation .nav-previous a { padding-left: 90px; }
		';
	}

	if ( $next && has_post_thumbnail( $next->ID ) ) {
		$nextthumb = wp_get_attachment_image_src( get_post_thumbnail_id( $next->ID ), 'thumbnail' );
		$css .= '
			.post-navigation .nav-next { background-image: url(' . esc_url( $nextthumb[0] ) . '); border-top: 0; background-size: 50px; background-repeat: no-repeat; background-position: 100% 40px; }
			.post-navigation .nav-next a { padding-right: 90px; }
		';
	}

	wp_add_inline_style( 'conary-style', $css );
}
add_action( 'wp_enqueue_scripts', 'conary_post_nav_image' );

/**
 * Setup a font controls & settings for Easy Google Fonts plugin (if installed)
 *
 * @since Conary 1.0
 *
 * @param array $options Default control list by the plugin.
 * @return array Modified $options parameter to applied in filter 'tt_font_get_option_parameters'.
 */
function conary_easy_google_fonts($options) {

	// Just replace all the plugin default font control

	unset(  $options['tt_default_body'],
			$options['tt_default_heading_2'],
			$options['tt_default_heading_3'],
			$options['tt_default_heading_4'],
			$options['tt_default_heading_5'],
			$options['tt_default_heading_6'],
			$options['tt_default_heading_1']
		);

	$new_options = array(

		'conary_default_body' => array(
			'name'        => 'conary_default_body',
			'title'       => esc_html__( 'Body & Paragraphs', 'conary' ),
			'description' => esc_html__( "Please select a font for the theme's body and paragraph text", 'conary' ),
			'properties'  => array( 'selector' => apply_filters( 'conary_default_body_font', 'body, input, select, textarea, blockquote cite, .entry-footer, .site-main div.sharedaddy h3.sd-title' ) ),
		),

		'conary_default_menu' => array(
			'name'        => 'conary_default_menu',
			'title'       => esc_html__( 'Menu', 'conary' ),
			'description' => esc_html__( "Please select a font for the theme's menu styles", 'conary' ),
			'properties'  => array( 'selector' => apply_filters( 'conary_default_heading', '.main-navigation' ) ),
		),

		'conary_default_entry_title' => array(
			'name'        => 'conary_default_entry_title',
			'title'       => esc_html__( 'Entry Title', 'conary' ),
			'description' => esc_html__( "Please select a font for the theme's Entry title styles", 'conary' ),
			'properties'  => array( 'selector' => apply_filters( 'conary_default_menu_font', '.site-title, .entry-title, .post-navigation .post-title, .comment-reply-title, .comment-meta .fn, .page-title, .site-main #jp-relatedposts .jp-relatedposts-items-visual h4.jp-relatedposts-post-title, .site-main #jp-relatedposts h3.jp-relatedposts-headline' ) ),
		),

		'conary_default_entry_meta' => array(
			'name'        => 'conary_default_entry_meta',
			'title'       => esc_html__( 'Entry Meta', 'conary' ),
			'description' => esc_html__( "Please select a font for the theme's Entry meta styles", 'conary' ),
			'properties'  => array( 'selector' => apply_filters( 'conary_default_meta_font', '.entry-meta, .site-info, .site-breadcrumbs, .posted-on, .post-navigation .meta-nav, .comment-metadata, .pingback .edit-link, .comment-reply-link, .site-content #jp-relatedposts .jp-relatedposts-items .jp-relatedposts-post .jp-relatedposts-post-date, .site-content #jp-relatedposts .jp-relatedposts-items .jp-relatedposts-post .jp-relatedposts-post-context' ) ),
		),

		'conary_default_widget_title' => array(
			'name'        => 'conary_default_widget_title',
			'title'       => esc_html__( 'Widget Title', 'conary' ),
			'description' => esc_html__( "Please select a font for the theme's Widget title styles", 'conary' ),
			'properties'  => array( 'selector' => apply_filters( 'conary_default_widget_title_font', '.widget .widget-title, .button, input[type="button"], input[type="reset"], input[type="submit"], .widget-recent-posts .tab-control a span, .load-more a, .comments-title, .comment-reply-title, #page .site-main #jp-relatedposts h3.jp-relatedposts-headline, .site-main #jp-relatedposts h3.jp-relatedposts-headline em  ' ) ),
		),


	);

	return array_merge( $options, $new_options);
}
add_filter( 'tt_font_get_option_parameters', 'conary_easy_google_fonts', 10 , 1 );

function conary_pattern_wave() {
	?>
	<svg class="conary-patterns">
		<defs>
			<path id="wave" d="M5,0c1.035,0,2.07,0.383,2.857,1.15C8.447,1.731,9.223,2.021,10,2.022V3C8.92,3,7.902,2.587,7.146,1.846 c-0.59-0.582-1.366-0.873-2.143-0.873s-1.556,0.291-2.15,0.873C2.098,2.587,1.081,3,0,3V2.022c0.776,0,1.553-0.291,2.142-0.873 C2.93,0.383,3.965,0,5,0z" width="10" height="3" />
			<pattern id="wave-content" x="0" y="0" width="10" height="3" patternUnits="userSpaceOnUse">
				<use xlink:href="#wave"></use>
			</pattern>
			<pattern id="wave-footer" x="0" y="0" width="10" height="3" patternUnits="userSpaceOnUse">
				<use xlink:href="#wave"></use>
			</pattern>
		</defs>
	</svg>
	<?php
}
//add_action('wp_head', 'conary_pattern_wave', 100);
