<?php
/**
 * Clickright Lite functions and definitions
 *
 * @link https://www.themebounce.com/themes/clickright/
 *
 * @package WordPress
 * @subpackage Clickright Lite
 * @since 1.0
 */

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function clickright_lite_setup(){	
	/*
	 * Make theme available for translation.
	 */
	load_theme_textdomain( 'clickright-lite-full-thumb' );
	
	// Add default posts and comments RSS feed links to head.
	add_theme_support('automatic-feed-links');

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( "title-tag" );

	/* 
	 * Custom Background
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded tags.
	 */
	add_theme_support( "custom-background" );	

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support('post-thumbnails');
	add_image_size('clickright-lite-full-thumb', 940, 0, true);
	
	// Set the default content width.
	$GLOBALS['content_width'] = 940;

	// Theme wp_nav_menu().
	register_nav_menus( array(
		'main-menu' => __('Primary Menu', 'clickright-lite' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	// Add theme support for Custom Logo.
	add_theme_support( 'custom-logo', array(
	   'height'      => 250,
	   'width'       => 250,
	   'flex-width'  => true,
	) );
}

add_action('after_setup_theme', 'clickright_lite_setup');

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function clickright_lite_widgets() {
	register_sidebar(array(
		'name' => __( 'Sidebar', 'clickright-lite' ),
		'id'            => 'sidebar',
		'description'   => __( 'Add widgets here to appear in your sidebar', 'clickright-lite' ),
		'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget' => '</div>',
		'before_title' => '<h4 class="widget-title">',
		'after_title' => '</h4>',
	));
	
	register_sidebar(array(
		'name' => __( 'Footer Area 1', 'clickright-lite' ),
		'id' => 'footer1',
		'description' => __( 'Widgets in this area will be shown on footer1.', 'clickright-lite' ),
      	'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget' => '</div><!-- widget end -->',
		'before_title' => '<h3 class="widget-title">',
		'after_title' => '</h3>',

    ) );
register_sidebar(array(
		'name' => __( 'Footer Area 2', 'clickright-lite' ),
		'id' => 'footer2',
		'description' => __( 'Widgets in this area will be shown on footer2.', 'clickright-lite' ),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget' => '</div><!-- widget end -->',
		'before_title' => '<h3 class="widget-title">',
		'after_title' => '</h3>',
    ) );
register_sidebar(array(
		'name' => __( 'Footer Area 3', 'clickright-lite' ),
		'id' => 'footer3',
		'description' => __( 'Widgets in this area will be shown on footer3.' , 'clickright-lite'),
        'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget' => '</div><!-- widget end -->',
		'before_title' => '<h3 class="widget-title">',
		'after_title' => '</h3>',
    ) );
}

add_action('widgets_init', 'clickright_lite_widgets');

/**
 * Replaces "[...]" (appended to automatically generated excerpts) with ... and
 * a 'Continue reading' link.
 *
 * @since Clickright Lite 1.0
 *
 * @param string $link Link to single post/page.
 * @return &hellip.
 */
function clickright_lite_excerpt_more( $link ) {
	if ( is_admin() ) {
		return $link;
	}

	$link = sprintf( '<p class="link-more"><a href="%1$s" class="more-link">%2$s</a></p>',
		esc_url( get_permalink( get_the_ID() ) ),
		/* translators: %s: Name of current post */
		sprintf( __( 'Continue reading<span class="screen-reader-text"> "%s"</span>', 'clickright-lite' ), get_the_title( get_the_ID() ) )
	);
	return ' &hellip; ' ;
}
add_filter( 'excerpt_more', 'clickright_lite_excerpt_more' );

/**
 * Register Google fonts.
 */
function clickright_lite_fonts_url() {
		$fonts_url = '';
		$fonts     = array();
		$subsets   = 'latin,latin-ext';

		if ( 'off' !== _x( 'on', 'Lato: on or off', 'clickright-lite' ) ) {
			$fonts[] = 'Lato:400,700';
		}	
		if ( 'off' !== _x( 'on', 'Source Sans Pro: on or off', 'clickright-lite' ) ) {
			$fonts[] = 'Source Sans Pro:400,700';
		}
		if ( 'off' !== _x( 'on', 'Open Sans: on or off', 'clickright-lite' ) ) {
			$fonts[] = 'Open Sans:700';
		}			

		if ( $fonts ) {
			$fonts_url = add_query_arg( array(
				'family' => urlencode( implode( '|', $fonts ) ),
				'subset' => urlencode( $subsets ),
			), 'https://fonts.googleapis.com/css' );
		}

		return $fonts_url;
	}

/**
 * Enqueue scripts and styles.
 */
function clickright_lite_load_scripts()
{	
	// Theme stylesheet
	wp_enqueue_style('clickright-lite-style', get_stylesheet_uri());
	
	wp_enqueue_style( 'clickright-lite-fonts', clickright_lite_fonts_url(), array(), null );
	
	wp_enqueue_script('clickright-lite-scripts', get_template_directory_uri() . '/assets/js/clickright.js', array( 'jquery' ) );

	if (is_singular() && get_option('thread_comments')) wp_enqueue_script('comment-reply');
}
	add_action('wp_enqueue_scripts', 'clickright_lite_load_scripts');

/**
 * COMMENTS LAYOUT.
 */
function clickright_lite_comments($comment, $args, $depth)
{
	$GLOBALS['comment'] = $comment; // WPCS: override ok.
?>
		<li <?php
	comment_class(); ?> id="comment-<?php
	comment_ID() ?>">
			
			<div class="thecomment">
						
				<div class="author-img">
					<?php
	echo get_avatar($comment, $args['avatar_size']); ?>
				</div>
				
				<div class="comment-text">
					<span class="reply">
						<?php
	comment_reply_link(array_merge($args, array(
		'reply_text' => __('Reply', 'clickright-lite') ,
		'depth' => $depth,
		'max_depth' => $args['max_depth']
	)) , $comment->comment_ID); ?>
						<?php
	edit_comment_link(__('Edit', 'clickright-lite')); ?>
					</span>
					<span class="author"><?php
	echo get_comment_author_link(); ?></span>
					<span class="date"><?php
					/* translators: 1: Comment date, 2: Comment time. */
	printf( esc_html__('%1$s at %2$s', 'clickright-lite') , get_comment_date() , get_comment_time()) ?></span>
					<?php
	if ($comment->comment_approved == '0'): ?>
						<em><i class="icon-info-sign"></i> <?php
		esc_html_e('Comment awaiting approval', 'clickright-lite'); ?></em>
						<br />
					<?php
	endif; ?>
					<?php
	comment_text(); ?>
				</div>
						
			</div>
			
			
		</li>

		<?php
}
/**
 * FEATURED CATEGORY.
 */
function clickright_lite_category($separator)
{
	if (get_theme_mod('clickright_lite_featured_cat_hide') == true) {
		$excluded_cat = esc_attr(get_theme_mod('clickright_lite_featured_cat'));
		$first_time = 1;
		foreach((get_the_category()) as $category) {
			if ($category->cat_ID != $excluded_cat) {
				if ($first_time == 1) {				
					/* translators: used category keyword */
					echo wp_kses_post('<a href="' . get_category_link($category->term_id) . '" title="' . sprintf(__("View all posts in %s", "clickright-lite") , $category->name) . '" ' . '>' . $category->name . '</a>');
					$first_time = 0;
				}
				else {
					/* translators: used category keyword */
					echo wp_kses_post($separator . '<a href="' . get_category_link($category->term_id) . '" title="' . sprintf(__("View all posts in %s", "clickright-lite") , $category->name) . '" ' . '>' . $category->name . '</a>');
				}
			}
		}
	}
	else {
		$first_time = 1;
		foreach((get_the_category()) as $category) {
			if ($first_time == 1) {
				/* translators: used category keyword */
				echo wp_kses_post('<a href="' . get_category_link($category->term_id) . '" title="' . sprintf(__("View all posts in %s", "clickright-lite") , $category->name) . '" ' . '>' . $category->name . '</a>');
				$first_time = 0;
			}
			else {
				/* translators: used category keyword */
				echo wp_kses_post($separator . '<a href="' . get_category_link($category->term_id) . '" title="' . sprintf(__("View all posts in %s", "clickright-lite") , $category->name) . '" ' . '>' . $category->name . '</a>');
			}
		}
	}
}