<?php
/**
 * Custom Functions
 * @package Carpenter Shop
 * @since 1.0.0
 */

if( !function_exists('carpenter_shop_site_logo') ):

    /**
     * Logo & Description
     */
    /**
     * Displays the site logo, either text or image.
     *
     * @param array $carpenter_shop_args Arguments for displaying the site logo either as an image or text.
     * @param boolean $carpenter_shop_echo Echo or return the HTML.
     *
     * @return string $carpenter_shop_html Compiled HTML based on our arguments.
     */
    function carpenter_shop_site_logo( $carpenter_shop_args = array(), $carpenter_shop_echo = true ){
        $carpenter_shop_logo = get_custom_logo();
        $carpenter_shop_site_title = get_bloginfo('name');
        $carpenter_shop_contents = '';
        $carpenter_shop_classname = '';
        $carpenter_shop_defaults = array(
            'logo' => '%1$s<span class="screen-reader-text">%2$s</span>',
            'logo_class' => 'site-logo site-branding',
            'title' => '<a href="%1$s" class="custom-logo-name">%2$s</a>',
            'title_class' => 'site-title',
            'home_wrap' => '<h1 class="%1$s">%2$s</h1>',
            'single_wrap' => '<div class="%1$s">%2$s</div>',
            'condition' => (is_front_page() || is_home()) && !is_page(),
        );
        $carpenter_shop_args = wp_parse_args($carpenter_shop_args, $carpenter_shop_defaults);
        /**
         * Filters the arguments for `carpenter_shop_site_logo()`.
         *
         * @param array $carpenter_shop_args Parsed arguments.
         * @param array $carpenter_shop_defaults Function's default arguments.
         */
        $carpenter_shop_args = apply_filters('carpenter_shop_site_logo_args', $carpenter_shop_args, $carpenter_shop_defaults);
        
        $carpenter_shop_show_logo  = get_theme_mod('carpenter_shop_display_logo', false);
        $carpenter_shop_show_title = get_theme_mod('carpenter_shop_display_title', true);

        if ( has_custom_logo() && $carpenter_shop_show_logo ) {
            $carpenter_shop_contents .= sprintf($carpenter_shop_args['logo'], $carpenter_shop_logo, esc_html($carpenter_shop_site_title));
            $carpenter_shop_classname = $carpenter_shop_args['logo_class'];
        }

        if ( $carpenter_shop_show_title ) {
            $carpenter_shop_contents .= sprintf($carpenter_shop_args['title'], esc_url(get_home_url(null, '/')), esc_html($carpenter_shop_site_title));
            // If logo isn't shown, fallback to title class for wrapper.
            if ( !$carpenter_shop_show_logo ) {
                $carpenter_shop_classname = $carpenter_shop_args['title_class'];
            }
        }

        // If nothing is shown (logo or title both disabled), exit early
        if ( empty($carpenter_shop_contents) ) {
            return;
        }

        $carpenter_shop_wrap = $carpenter_shop_args['condition'] ? 'home_wrap' : 'single_wrap';
        // $carpenter_shop_wrap = 'home_wrap';
        $carpenter_shop_html = sprintf($carpenter_shop_args[$carpenter_shop_wrap], $carpenter_shop_classname, $carpenter_shop_contents);
        /**
         * Filters the arguments for `carpenter_shop_site_logo()`.
         *
         * @param string $carpenter_shop_html Compiled html based on our arguments.
         * @param array $carpenter_shop_args Parsed arguments.
         * @param string $carpenter_shop_classname Class name based on current view, home or single.
         * @param string $carpenter_shop_contents HTML for site title or logo.
         */
        $carpenter_shop_html = apply_filters('carpenter_shop_site_logo', $carpenter_shop_html, $carpenter_shop_args, $carpenter_shop_classname, $carpenter_shop_contents);
        if (!$carpenter_shop_echo) {
            return $carpenter_shop_html;
        }
        echo $carpenter_shop_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

    }

endif;

if( !function_exists('carpenter_shop_site_description') ):

    /**
     * Displays the site description.
     *
     * @param boolean $carpenter_shop_echo Echo or return the html.
     *
     * @return string $carpenter_shop_html The HTML to display.
     */
    function carpenter_shop_site_description($carpenter_shop_echo = true){

        if ( get_theme_mod('carpenter_shop_display_header_text', false) == true ) :
        $carpenter_shop_description = get_bloginfo('description');
        if (!$carpenter_shop_description) {
            return;
        }
        $carpenter_shop_wrapper = '<div class="site-description"><span>%s</span></div><!-- .site-description -->';
        $carpenter_shop_html = sprintf($carpenter_shop_wrapper, esc_html($carpenter_shop_description));
        /**
         * Filters the html for the site description.
         *
         * @param string $carpenter_shop_html The HTML to display.
         * @param string $carpenter_shop_description Site description via `bloginfo()`.
         * @param string $carpenter_shop_wrapper The format used in case you want to reuse it in a `sprintf()`.
         * @since 1.0.0
         *
         */
        $carpenter_shop_html = apply_filters('carpenter_shop_site_description', $carpenter_shop_html, $carpenter_shop_description, $carpenter_shop_wrapper);
        if (!$carpenter_shop_echo) {
            return $carpenter_shop_html;
        }
        echo $carpenter_shop_html; //phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
        endif;
    }

endif;

if( !function_exists('carpenter_shop_posted_on') ):

    /**
     * Prints HTML with meta information for the current post-date/time.
     */
    function carpenter_shop_posted_on( $carpenter_shop_icon = true, $carpenter_shop_animation_class = '' ){

        $carpenter_shop_default = carpenter_shop_get_default_theme_options();
        $carpenter_shop_post_date = absint( get_theme_mod( 'carpenter_shop_post_date',$carpenter_shop_default['carpenter_shop_post_date'] ) );

        if( $carpenter_shop_post_date ){

            $carpenter_shop_time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
            if (get_the_time('U') !== get_the_modified_time('U')) {
                $carpenter_shop_time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
            }

            $carpenter_shop_time_string = sprintf($carpenter_shop_time_string,
                esc_attr(get_the_date(DATE_W3C)),
                esc_html(get_the_date()),
                esc_attr(get_the_modified_date(DATE_W3C)),
                esc_html(get_the_modified_date())
            );

            $carpenter_shop_year = get_the_date('Y');
            $carpenter_shop_month = get_the_date('m');
            $carpenter_shop_day = get_the_date('d');
            $carpenter_shop_link = get_day_link($carpenter_shop_year, $carpenter_shop_month, $carpenter_shop_day);

            $carpenter_shop_posted_on = '<a href="' . esc_url($carpenter_shop_link) . '" rel="bookmark">' . $carpenter_shop_time_string . '</a>';

            echo '<div class="entry-meta-item entry-meta-date">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $carpenter_shop_animation_class ).'">';

            if( $carpenter_shop_icon ){

                echo '<span class="entry-meta-icon calendar-icon"> ';
                carpenter_shop_the_theme_svg('calendar');
                echo '</span>';

            }

            echo '<span class="posted-on">' . $carpenter_shop_posted_on . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;

if( !function_exists('carpenter_shop_posted_by') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function carpenter_shop_posted_by( $carpenter_shop_icon = true, $carpenter_shop_animation_class = '' ){   

        $carpenter_shop_default = carpenter_shop_get_default_theme_options();
        $carpenter_shop_post_author = absint( get_theme_mod( 'carpenter_shop_post_author',$carpenter_shop_default['carpenter_shop_post_author'] ) );

        if( $carpenter_shop_post_author ){

            echo '<div class="entry-meta-item entry-meta-author">';
            echo '<div class="entry-meta-wrapper '.esc_attr( $carpenter_shop_animation_class ).'">';

            if( $carpenter_shop_icon ){
            
                echo '<span class="entry-meta-icon author-icon"> ';
                carpenter_shop_the_theme_svg('user');
                echo '</span>';
                
            }

            $carpenter_shop_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';
            echo '<span class="byline"> ' . $carpenter_shop_byline . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
            echo '</div>';
            echo '</div>';

        }

    }

endif;


if( !function_exists('carpenter_shop_posted_by_avatar') ) :

    /**
     * Prints HTML with meta information for the current author.
     */
    function carpenter_shop_posted_by_avatar( $carpenter_shop_date = false ){

        $carpenter_shop_default = carpenter_shop_get_default_theme_options();
        $carpenter_shop_post_author = absint( get_theme_mod( 'carpenter_shop_post_author',$carpenter_shop_default['carpenter_shop_post_author'] ) );

        if( $carpenter_shop_post_author ){



            echo '<div class="entry-meta-left">';
            echo '<div class="entry-meta-item entry-meta-avatar">';
            echo wp_kses_post( get_avatar( get_the_author_meta( 'ID' ) ) );
            echo '</div>';
            echo '</div>';

            echo '<div class="entry-meta-right">';

            $carpenter_shop_byline = '<span class="author vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta('ID') ) ) . '">' . esc_html(get_the_author()) . '</a></span>';

            echo '<div class="entry-meta-item entry-meta-byline"> ' . $carpenter_shop_byline . '</div>';

            if( $carpenter_shop_date ){
                carpenter_shop_posted_on($carpenter_shop_icon = false);
            }
            echo '</div>';

        }

    }

endif;

if( !function_exists('carpenter_shop_entry_footer') ):

    /**
     * Prints HTML with meta information for the categories, tags and comments.
     */
    function carpenter_shop_entry_footer( $carpenter_shop_cats = true, $carpenter_shop_tags = true, $carpenter_shop_edits = true){   

        $carpenter_shop_default = carpenter_shop_get_default_theme_options();
        $carpenter_shop_post_category = absint( get_theme_mod( 'carpenter_shop_post_category',$carpenter_shop_default['carpenter_shop_post_category'] ) );
        $carpenter_shop_post_tags = absint( get_theme_mod( 'carpenter_shop_post_tags',$carpenter_shop_default['carpenter_shop_post_tags'] ) );

        // Hide category and tag text for pages.
        if ('post' === get_post_type()) {

            if( $carpenter_shop_cats && $carpenter_shop_post_category ){

                /* translators: used between list items, there is a space after the comma */
                $carpenter_shop_categories = get_the_category();
                if ($carpenter_shop_categories) {
                    echo '<div class="entry-meta-item entry-meta-categories">';
                    echo '<div class="entry-meta-wrapper">';
                
                    /* translators: 1: list of categories. */
                    echo '<span class="cat-links">';
                    foreach( $carpenter_shop_categories as $carpenter_shop_category ){

                        $carpenter_shop_cat_name = $carpenter_shop_category->name;
                        $carpenter_shop_cat_slug = $carpenter_shop_category->slug;
                        $carpenter_shop_cat_url = get_category_link( $carpenter_shop_category->term_id );
                        ?>

                        <a class="twp_cat_<?php echo esc_attr( $carpenter_shop_cat_slug ); ?>" href="<?php echo esc_url( $carpenter_shop_cat_url ); ?>" rel="category tag"><?php echo esc_html( $carpenter_shop_cat_name ); ?></a>

                    <?php }
                    echo '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';
                }

            }

            if( $carpenter_shop_tags && $carpenter_shop_post_tags ){
                /* translators: used between list items, there is a space after the comma */
                $carpenter_shop_tags_list = get_the_tag_list('', esc_html_x(', ', 'list item separator', 'carpenter-shop'));
                if( $carpenter_shop_tags_list ){

                    echo '<div class="entry-meta-item entry-meta-tags">';
                    echo '<div class="entry-meta-wrapper">';

                    /* translators: 1: list of tags. */
                    echo '<span class="tags-links">';
                    echo wp_kses_post($carpenter_shop_tags_list) . '</span>'; // phpcs:ignore Standard.Category.SniffName.ErrorCode
                    echo '</div>';
                    echo '</div>';

                }

            }

            if( $carpenter_shop_edits ){

                edit_post_link(
                    sprintf(
                        wp_kses(
                        /* translators: %s: Name of current post. Only visible to screen readers */
                            __('Edit <span class="screen-reader-text">%s</span>', 'carpenter-shop'),
                            array(
                                'span' => array(
                                    'class' => array(),
                                ),
                            )
                        ),
                        get_the_title()
                    ),
                    '<span class="edit-link">',
                    '</span>'
                );
            }

        }
    }

endif;

if ( ! function_exists( 'carpenter_shop_post_thumbnail' ) ) :

    /**
     * Displays an optional post thumbnail.
     *
     * Shows background style image with height class on archive/search/front,
     * and a normal inline image on single post/page views.
     */
    function carpenter_shop_post_thumbnail( $carpenter_shop_image_size = 'medium' ) {

        if ( post_password_required() || is_attachment() ) {
            return;
        }

        // Fallback image path
        $carpenter_shop_default_image = get_template_directory_uri() . '/inc/homepage-setup/assets/homepage-setup-images/slider-img1.png';

        // Image size → height class map
        $carpenter_shop_size_class_map = array(
            'full'      => 'data-bg-large',
            'large'     => 'data-bg-big',
            'medium'    => 'data-bg-medium',
            'small'     => 'data-bg-small',
            'xsmall'    => 'data-bg-xsmall',
            'thumbnail' => 'data-bg-thumbnail',
        );

        $carpenter_shop_class = isset( $carpenter_shop_size_class_map[ $carpenter_shop_image_size ] )
            ? $carpenter_shop_size_class_map[ $carpenter_shop_image_size ]
            : 'data-bg-medium';

        if ( is_singular() ) {
            the_post_thumbnail();
        } else {
            // 🔵 On archives → use background image style
            $carpenter_shop_image = has_post_thumbnail()
                ? wp_get_attachment_image_src( get_post_thumbnail_id(), $carpenter_shop_image_size )
                : array( $carpenter_shop_default_image );

            $carpenter_shop_bg_image = isset( $carpenter_shop_image[0] ) ? $carpenter_shop_image[0] : $carpenter_shop_default_image;
            ?>
            <div class="post-thumbnail data-bg <?php echo esc_attr( $carpenter_shop_class ); ?>"
                 data-background="<?php echo esc_url( $carpenter_shop_bg_image ); ?>">
                <a href="<?php the_permalink(); ?>" class="theme-image-responsive" tabindex="0"></a>
            </div>
            <?php
        }
    }

endif;

if( !function_exists('carpenter_shop_is_comment_by_post_author') ):

    /**
     * Comments
     */
    /**
     * Check if the specified comment is written by the author of the post commented on.
     *
     * @param object $carpenter_shop_comment Comment data.
     *
     * @return bool
     */
    function carpenter_shop_is_comment_by_post_author($carpenter_shop_comment = null){

        if (is_object($carpenter_shop_comment) && $carpenter_shop_comment->user_id > 0) {
            $carpenter_shop_user = get_userdata($carpenter_shop_comment->user_id);
            $post = get_post($carpenter_shop_comment->comment_post_ID);
            if (!empty($carpenter_shop_user) && !empty($post)) {
                return $carpenter_shop_comment->user_id === $post->post_author;
            }
        }
        return false;
    }

endif;

if( !function_exists('carpenter_shop_breadcrumb') ) :

    /**
     * Carpenter Shop Breadcrumb
     */
    function carpenter_shop_breadcrumb($carpenter_shop_comment = null){

        echo '<div class="entry-breadcrumb">';
        carpenter_shop_breadcrumb_trail();
        echo '</div>';

    }

endif;