<?php
/**
 * Wizard
 *
 * @package Carpenter_Shop_Whizzie
 * @author Catapult Themes
 * @since 1.0.0
 */

class Carpenter_Shop_Whizzie {
	
	protected $version = '1.1.0';
	
	/** @var string Current theme name, used as namespace in actions. */
	protected $carpenter_shop_theme_name = '';
	protected $carpenter_shop_theme_title = '';
	
	/** @var string Wizard page slug and title. */
	protected $carpenter_shop_page_slug = '';
	protected $carpenter_shop_page_title = '';
	
	/** @var array Wizard steps set by user. */
	protected $config_steps = array();
	
	/**
	 * Relative plugin url for this plugin folder
	 * @since 1.0.0
	 * @var string
	 */
	protected $carpenter_shop_plugin_url = '';

	public $carpenter_shop_plugin_path;
	public $parent_slug;
	
	/**
	 * TGMPA instance storage
	 *
	 * @var object
	 */
	protected $tgmpa_instance;
	
	/**
	 * TGMPA Menu slug
	 *
	 * @var string
	 */
	protected $tgmpa_menu_slug = 'tgmpa-install-plugins';
	
	/**
	 * TGMPA Menu url
	 *
	 * @var string
	 */
	protected $tgmpa_url = 'themes.php?page=tgmpa-install-plugins';
	
	/**
	 * Constructor
	 *
	 * @param $config	Our config parameters
	 */
	public function __construct( $config ) {
		$this->set_vars( $config );
		$this->init();
	}
	
	/**
	 * Set some settings
	 * @since 1.0.0
	 * @param $config	Our config parameters
	 */
	public function set_vars( $config ) {
	
		require_once trailingslashit( WHIZZIE_DIR ) . 'tgm/class-tgm-plugin-activation.php';
		require_once trailingslashit( WHIZZIE_DIR ) . 'tgm/tgm.php';

		if( isset( $config['carpenter_shop_page_slug'] ) ) {
			$this->carpenter_shop_page_slug = esc_attr( $config['carpenter_shop_page_slug'] );
		}
		if( isset( $config['carpenter_shop_page_title'] ) ) {
			$this->carpenter_shop_page_title = esc_attr( $config['carpenter_shop_page_title'] );
		}
		if( isset( $config['steps'] ) ) {
			$this->config_steps = $config['steps'];
		}
		
		$this->carpenter_shop_plugin_path = trailingslashit( dirname( __FILE__ ) );
		$relative_url = str_replace( get_template_directory(), '', $this->carpenter_shop_plugin_path );
		$this->carpenter_shop_plugin_url = trailingslashit( get_template_directory_uri() . $relative_url );
		$carpenter_shop_current_theme = wp_get_theme();
		$this->carpenter_shop_theme_title = $carpenter_shop_current_theme->get( 'Name' );
		$this->carpenter_shop_theme_name = strtolower( preg_replace( '#[^a-zA-Z]#', '', $carpenter_shop_current_theme->get( 'Name' ) ) );
		$this->carpenter_shop_page_slug = apply_filters( $this->carpenter_shop_theme_name . '_theme_setup_wizard_carpenter_shop_page_slug', $this->carpenter_shop_theme_name . '-wizard' );
		$this->parent_slug = apply_filters( $this->carpenter_shop_theme_name . '_theme_setup_wizard_parent_slug', '' );

	}
	
	/**
	 * Hooks and filters
	 * @since 1.0.0
	 */	
	public function init() {
		
		if ( class_exists( 'TGM_Plugin_Activation' ) && isset( $GLOBALS['tgmpa'] ) ) {
			add_action( 'init', array( $this, 'get_tgmpa_instance' ), 30 );
			add_action( 'init', array( $this, 'set_tgmpa_url' ), 40 );
		}
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
		add_action( 'admin_menu', array( $this, 'menu_page' ) );
		add_action( 'admin_init', array( $this, 'get_plugins' ), 30 );
		add_filter( 'tgmpa_load', array( $this, 'tgmpa_load' ), 10, 1 );
		add_action( 'wp_ajax_setup_plugins', array( $this, 'setup_plugins' ) );
		add_action( 'wp_ajax_carpenter_shop_setup_widgets', array( $this, 'carpenter_shop_setup_widgets' ) );
		
	}
	
	public function enqueue_scripts() {
		wp_enqueue_style( 'carpenter-shop-homepage-setup-style', get_template_directory_uri() . '/inc/homepage-setup/assets/css/homepage-setup-style.css');
		wp_register_script( 'carpenter-shop-homepage-setup-script', get_template_directory_uri() . '/inc/homepage-setup/assets/js/homepage-setup-script.js', array( 'jquery' ), time() );
		wp_localize_script( 
			'carpenter-shop-homepage-setup-script',
			'whizzie_params',
			array(
				'ajaxurl' 		=> admin_url( 'admin-ajax.php' ),
				'wpnonce' 		=> wp_create_nonce( 'whizzie_nonce' ),
				'verify_text'	=> esc_html( 'verifying', 'carpenter-shop' )
			)
		);
		wp_enqueue_script( 'carpenter-shop-homepage-setup-script' );
	}
	
	public static function get_instance() {
		if ( ! self::$instance ) {
			self::$instance = new self;
		}
		return self::$instance;
	}
	
	public function tgmpa_load( $status ) {
		return is_admin() || current_user_can( 'install_themes' );
	}
			
	/**
	 * Get configured TGMPA instance
	 *
	 * @access public
	 * @since 1.1.2
	 */
	public function get_tgmpa_instance() {
		$this->tgmpa_instance = call_user_func( array( get_class( $GLOBALS['tgmpa'] ), 'get_instance' ) );
	}
	
	/**
	 * Update $tgmpa_menu_slug and $tgmpa_parent_slug from TGMPA instance
	 *
	 * @access public
	 * @since 1.1.2
	 */
	public function set_tgmpa_url() {
		$this->tgmpa_menu_slug = ( property_exists( $this->tgmpa_instance, 'menu' ) ) ? $this->tgmpa_instance->menu : $this->tgmpa_menu_slug;
		$this->tgmpa_menu_slug = apply_filters( $this->carpenter_shop_theme_name . '_theme_setup_wizard_tgmpa_menu_slug', $this->tgmpa_menu_slug );
		$tgmpa_parent_slug = ( property_exists( $this->tgmpa_instance, 'parent_slug' ) && $this->tgmpa_instance->parent_slug !== 'themes.php' ) ? 'admin.php' : 'themes.php';
		$this->tgmpa_url = apply_filters( $this->carpenter_shop_theme_name . '_theme_setup_wizard_tgmpa_url', $tgmpa_parent_slug . '?page=' . $this->tgmpa_menu_slug );
	}
	
	/**
	 * Make a modal screen for the wizard
	 */
	public function menu_page() {
		add_theme_page( esc_html( $this->carpenter_shop_page_title ), esc_html( $this->carpenter_shop_page_title ), 'manage_options', $this->carpenter_shop_page_slug, array( $this, 'wizard_page' ) );
	}
	
	/**
	 * Make an interface for the wizard
	 */
	public function wizard_page() { 
		tgmpa_load_bulk_installer();

		if ( ! class_exists( 'TGM_Plugin_Activation' ) || ! isset( $GLOBALS['tgmpa'] ) ) {
			die( esc_html__( 'Failed to find TGM', 'carpenter-shop' ) );
		}

		$url = wp_nonce_url( add_query_arg( array( 'plugins' => 'go' ) ), 'whizzie-setup' );
		$method = '';
		$fields = array_keys( $_POST );

		if ( false === ( $creds = request_filesystem_credentials( esc_url_raw( $url ), $method, false, false, $fields ) ) ) {
			return true;
		}

		if ( ! WP_Filesystem( $creds ) ) {
			request_filesystem_credentials( esc_url_raw( $url ), $method, true, false, $fields );
			return true;
		}

		$carpenter_shop_theme = wp_get_theme();
		$carpenter_shop_theme_title = $carpenter_shop_theme->get( 'Name' );
		$carpenter_shop_theme_version = $carpenter_shop_theme->get( 'Version' );

		?>
		<div class="wrap">
			<?php
				printf( '<h1>%s %s</h1>', esc_html( $carpenter_shop_theme_title ), esc_html( '(Version :- ' . $carpenter_shop_theme_version . ')' ) );
			?>
			<div class="homepage-setup">
				<div class="homepage-setup-theme-bundle">
					<div class="homepage-setup-theme-bundle-one">
						<h1><?php echo esc_html__( 'WP Theme Bundle', 'carpenter-shop' ); ?></h1>
						<p><?php echo wp_kses_post( 'Get <span>15% OFF</span> on all WordPress themes! Use code <span>"BNDL15OFF"</span> at checkout. Limited time offer!' ); ?></p>
					</div>
					<div class="homepage-setup-theme-bundle-two">
						<p><?php echo wp_kses_post( 'Extra <span>15%</span> OFF' ); ?></p>
					</div>
					<div class="homepage-setup-theme-bundle-three">
						<img src="<?php echo esc_url( get_template_directory_uri() . '/inc/homepage-setup/assets/homepage-setup-images/bundle-banner.png' ); ?>" alt="<?php echo esc_attr__( 'Theme Bundle Image', 'carpenter-shop' ); ?>">
					</div>
					<div class="homepage-setup-theme-bundle-four">
						<p><?php echo wp_kses_post( '<span>$2795</span>$69' ); ?></p>
						<a target="_blank" href="<?php echo esc_url( CARPENTER_SHOP_BUNDLE_BUTTON ); ?>"><?php echo esc_html__( 'SHOP NOW', 'carpenter-shop' ); ?> <span class="dashicons dashicons-arrow-right-alt2"></span></a>
					</div>
				</div>
			</div>
			
			<div class="card whizzie-wrap">
				<div class="demo_content_image">
					<div class="demo_content">
						<?php
							$carpenter_shop_steps = $this->get_steps();
							echo '<ul class="whizzie-menu">';
							foreach ( $carpenter_shop_steps as $carpenter_shop_step ) {
								$class = 'step step-' . esc_attr( $carpenter_shop_step['id'] );
								echo '<li data-step="' . esc_attr( $carpenter_shop_step['id'] ) . '" class="' . esc_attr( $class ) . '">';
								printf( '<h2>%s</h2>', esc_html( $carpenter_shop_step['title'] ) );

								$content = call_user_func( array( $this, $carpenter_shop_step['view'] ) );
								if ( isset( $content['summary'] ) ) {
									printf(
										'<div class="summary">%s</div>',
										wp_kses_post( $content['summary'] )
									);
								}
								if ( isset( $content['detail'] ) ) {
									printf(
										'<div class="detail">%s</div>',
										wp_kses_post( $content['detail'] )
									);
								}
								if ( isset( $carpenter_shop_step['button_text'] ) && $carpenter_shop_step['button_text'] ) {
									printf( 
										'<div class="button-wrap"><a href="#" class="button button-primary do-it" data-callback="%s" data-step="%s">%s</a></div>',
										esc_attr( $carpenter_shop_step['callback'] ),
										esc_attr( $carpenter_shop_step['id'] ),
										esc_html( $carpenter_shop_step['button_text'] )
									);
								}
								echo '</li>';
							}
							echo '</ul>';
						?>
						
						<ul class="whizzie-nav">
							<?php
							$step_number = 1;	
							foreach ( $carpenter_shop_steps as $carpenter_shop_step ) {
								echo '<li class="nav-step-' . esc_attr( $carpenter_shop_step['id'] ) . '">';
								echo '<span class="step-number">' . esc_html( $step_number ) . '</span>';
								echo '</li>';
								$step_number++;
							}
							?>
							<div class="blank-border"></div>
						</ul>

						<div class="homepage-setup-links">
							<div class="homepage-setup-links buttons">
								<a href="<?php echo esc_url( CARPENTER_SHOP_LITE_DOCS_PRO ); ?>" target="_blank" class="button button-primary"><?php echo esc_html__( 'Free Documentation', 'carpenter-shop' ); ?></a>
								<a href="<?php echo esc_url( CARPENTER_SHOP_BUY_NOW ); ?>" class="button button-primary" target="_blank"><?php echo esc_html__( 'Get Premium', 'carpenter-shop' ); ?></a>
								<a href="<?php echo esc_url( CARPENTER_SHOP_DEMO_PRO ); ?>" class="button button-primary" target="_blank"><?php echo esc_html__( 'Premium Demo', 'carpenter-shop' ); ?></a>
								<a href="<?php echo esc_url( CARPENTER_SHOP_SUPPORT_FREE ); ?>" target="_blank" class="button button-primary"><?php echo esc_html__( 'Support Forum', 'carpenter-shop' ); ?></a>
							</div>
						</div> <!-- .demo_image -->

						<div class="step-loading"><span class="spinner"></span></div>
					</div> <!-- .demo_content -->

					<div class="homepage-setup-image">
						<div class="homepage-setup-theme-buynow">
							<div class="homepage-setup-theme-buynow-one">
								<h1><?php echo wp_kses_post( 'Carpenter<br>WordPress Theme' ); ?></h1>
								<p><?php echo wp_kses_post( '<span>25%<br>Off</span> SHOP NOW' ); ?></p>
							</div>
							<div class="homepage-setup-theme-buynow-two">
								<img src="<?php echo esc_url( get_template_directory_uri() . '/inc/homepage-setup/assets/homepage-setup-images/carpenter-shop.png' ); ?>" alt="<?php echo esc_attr__( 'Theme Bundle Image', 'carpenter-shop' ); ?>">
							</div>
							<div class="homepage-setup-theme-buynow-three">
								<p><?php echo wp_kses_post( 'Get <span>25% OFF</span> on Premium Carpenter WordPress Theme Use code <span>"NYTHEMES25"</span> at checkout.' ); ?></p>
							</div>
							<div class="homepage-setup-theme-buynow-four">
								<a target="_blank" href="<?php echo esc_url( CARPENTER_SHOP_BUY_NOW ); ?>"><?php echo esc_html__( 'Upgrade To Pro With Just $40', 'carpenter-shop' ); ?></a>
							</div>
						</div>
					</div> <!-- .demo_image -->

				</div> <!-- .demo_content_image -->
			</div> <!-- .whizzie-wrap -->
		</div> <!-- .wrap -->
		<?php
	}


	/**
	 * Set options for the steps
	 * Incorporate any options set by the theme dev
	 * Return the array for the steps
	 * @return Array
	 */
	public function get_steps() {
		$carpenter_shop_dev_steps = $this->config_steps;
		$carpenter_shop_steps = array( 
			'plugins' => array(
				'id'			=> 'plugins',
				'title'			=> __( 'Install and Activate Essential Plugins', 'carpenter-shop' ),
				'icon'			=> 'admin-plugins',
				'view'			=> 'get_step_plugins',
				'callback'		=> 'install_plugins',
				'button_text'	=> __( 'Install Plugins', 'carpenter-shop' ),
				'can_skip'		=> true
			),
			'widgets' => array(
				'id'			=> 'widgets',
				'title'			=> __( 'Setup Home Page', 'carpenter-shop' ),
				'icon'			=> 'welcome-widgets-menus',
				'view'			=> 'get_step_widgets',
				'callback'		=> 'carpenter_shop_install_widgets',
				'button_text'	=> __( 'Start Home Page Setup', 'carpenter-shop' ),
				'can_skip'		=> false
			),
			'done' => array(
				'id'			=> 'done',
				'title'			=> __( 'Customize Your Site', 'carpenter-shop' ),
				'icon'			=> 'yes',
				'view'			=> 'get_step_done',
				'callback'		=> ''
			)
		);
		
		// Iterate through each step and replace with dev config values
		if( $carpenter_shop_dev_steps ) {
			// Configurable elements - these are the only ones the dev can update from homepage-setup-settings.php
			$can_config = array( 'title', 'icon', 'button_text', 'can_skip' );
			foreach( $carpenter_shop_dev_steps as $carpenter_shop_dev_step ) {
				// We can only proceed if an ID exists and matches one of our IDs
				if( isset( $carpenter_shop_dev_step['id'] ) ) {
					$id = $carpenter_shop_dev_step['id'];
					if( isset( $carpenter_shop_steps[$id] ) ) {
						foreach( $can_config as $element ) {
							if( isset( $carpenter_shop_dev_step[$element] ) ) {
								$carpenter_shop_steps[$id][$element] = $carpenter_shop_dev_step[$element];
							}
						}
					}
				}
			}
		}
		return $carpenter_shop_steps;
	}

	/**
	 * Get the content for the plugins step
	 * @return $content Array
	 */
	public function get_step_plugins() {
		$plugins = $this->get_plugins();
		$content = array(); 
		
		// Add plugin name and type at the top
		$content['detail'] = '<div class="plugin-info">';
		$content['detail'] .= '<p><strong>Plugin</strong></p>';
		$content['detail'] .= '<p><strong>Type</strong></p>';
		$content['detail'] .= '</div>';
		
		// The detail element is initially hidden from the user
		$content['detail'] .= '<ul class="whizzie-do-plugins">';
		
		// Add each plugin into a list
		foreach( $plugins['all'] as $slug=>$plugin ) {
			if ( $slug != 'easy-post-views-count') {
				$content['detail'] .= '<li data-slug="' . esc_attr( $slug ) . '">' . esc_html( $plugin['name'] ) . '<span>';
				$keys = array();
				if ( isset( $plugins['install'][ $slug ] ) ) {
					$keys[] = 'Installation';
				}
				if ( isset( $plugins['update'][ $slug ] ) ) {
					$keys[] = 'Update';
				}
				if ( isset( $plugins['activate'][ $slug ] ) ) {
					$keys[] = 'Activation';
				}
				$content['detail'] .= implode( ' and ', $keys ) . ' required';
				$content['detail'] .= '</span></li>';
			}
		}
		
		$content['detail'] .= '</ul>';
		
		return $content;
	}
	
	/**
	 * Print the content for the widgets step
	 * @since 1.1.0
	 */
	public function get_step_widgets() { ?> <?php }
	
	/**
	 * Print the content for the final step
	 */
	public function get_step_done() { ?>
		<div id="carpenter-shop-demo-setup-guid">
			<div class="customize_div">
				<div class="customize_div finish">
					<div class="customize_div finish btns">
						<h3><?php echo esc_html( 'Your Site Is Ready To View' ); ?></h3>
						<div class="btnsss">
							<a target="_blank" href="<?php echo esc_url( get_home_url() ); ?>" class="button button-primary">
								<?php esc_html_e( 'View Your Site', 'carpenter-shop' ); ?>
							</a>
							<a target="_blank" href="<?php echo esc_url( admin_url( 'customize.php' ) ); ?>" class="button button-primary">
								<?php esc_html_e( 'Customize Your Site', 'carpenter-shop' ); ?>
							</a>
							<a href="<?php echo esc_url(admin_url()); ?>" class="button button-primary">
								<?php esc_html_e( 'Finsh', 'carpenter-shop' ); ?>
							</a>
						</div>
					</div>
					<div class="carpenter-shop-setup-finish">
						<img src="<?php echo esc_url( get_template_directory_uri() . '/screenshot.png' ); ?>"/>
					</div>
				</div>
			</div>
		</div>
	<?php }

	/**
	 * Get the plugins registered with TGMPA
	 */
	public function get_plugins() {
		$instance = call_user_func( array( get_class( $GLOBALS['tgmpa'] ), 'get_instance' ) );
		$plugins = array(
			'all' 		=> array(),
			'install'	=> array(),
			'update'	=> array(),
			'activate'	=> array()
		);
		foreach( $instance->plugins as $slug=>$plugin ) {
			if( $instance->is_plugin_active( $slug ) && false === $instance->does_plugin_have_update( $slug ) ) {
				// Plugin is installed and up to date
				continue;
			} else {
				$plugins['all'][$slug] = $plugin;
				if( ! $instance->is_plugin_installed( $slug ) ) {
					$plugins['install'][$slug] = $plugin;
				} else {
					if( false !== $instance->does_plugin_have_update( $slug ) ) {
						$plugins['update'][$slug] = $plugin;
					}
					if( $instance->can_plugin_activate( $slug ) ) {
						$plugins['activate'][$slug] = $plugin;
					}
				}
			}
		}
		return $plugins;
	}

	/**
	 * Get the widgets.wie file from the /content folder
	 * @return Mixed	Either the file or false
	 * @since 1.1.0
	 */
	public function has_widget_file() {
		if( file_exists( $this->widget_file_url ) ) {
			return true;
		}
		return false;
	}
	
	public function setup_plugins() {
		if ( ! check_ajax_referer( 'whizzie_nonce', 'wpnonce' ) || empty( $_POST['slug'] ) ) {
			wp_send_json_error( array( 'error' => 1, 'message' => esc_html__( 'No Slug Found','carpenter-shop' ) ) );
		}
		$json = array();
		// send back some json we use to hit up TGM
		$plugins = $this->get_plugins();
		
		// what are we doing with this plugin?
		foreach ( $plugins['activate'] as $slug => $plugin ) {
			if ( $_POST['slug'] == $slug ) {
				$json = array(
					'url'           => admin_url( $this->tgmpa_url ),
					'plugin'        => array( $slug ),
					'tgmpa-page'    => $this->tgmpa_menu_slug,
					'plugin_status' => 'all',
					'_wpnonce'      => wp_create_nonce( 'bulk-plugins' ),
					'action'        => 'tgmpa-bulk-activate',
					'action2'       => - 1,
					'message'       => esc_html__( 'Activating Plugin','carpenter-shop' ),
				);
				break;
			}
		}
		foreach ( $plugins['update'] as $slug => $plugin ) {
			if ( $_POST['slug'] == $slug ) {
				$json = array(
					'url'           => admin_url( $this->tgmpa_url ),
					'plugin'        => array( $slug ),
					'tgmpa-page'    => $this->tgmpa_menu_slug,
					'plugin_status' => 'all',
					'_wpnonce'      => wp_create_nonce( 'bulk-plugins' ),
					'action'        => 'tgmpa-bulk-update',
					'action2'       => - 1,
					'message'       => esc_html__( 'Updating Plugin','carpenter-shop' ),
				);
				break;
			}
		}
		foreach ( $plugins['install'] as $slug => $plugin ) {
			if ( $_POST['slug'] == $slug ) {
				$json = array(
					'url'           => admin_url( $this->tgmpa_url ),
					'plugin'        => array( $slug ),
					'tgmpa-page'    => $this->tgmpa_menu_slug,
					'plugin_status' => 'all',
					'_wpnonce'      => wp_create_nonce( 'bulk-plugins' ),
					'action'        => 'tgmpa-bulk-install',
					'action2'       => - 1,
					'message'       => esc_html__( 'Installing Plugin','carpenter-shop' ),
				);
				break;
			}
		}
		if ( $json ) {
			$json['hash'] = md5( serialize( $json ) ); // used for checking if duplicates happen, move to next plugin
			wp_send_json( $json );
		} else {
			wp_send_json( array( 'done' => 1, 'message' => esc_html__( 'Success','carpenter-shop' ) ) );
		}
		exit;
	}


	public function carpenter_shop_customizer_nav_menu() {

		/* -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+- Carpenter Shop Primary Menu -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/

		$carpenter_shop_themename = 'Carpenter Shop';
		$carpenter_shop_menuname = $carpenter_shop_themename . ' Primary Menu';
		$carpenter_shop_menulocation = 'carpenter-shop-primary-menu';
		$carpenter_shop_menu_exists = wp_get_nav_menu_object($carpenter_shop_menuname);

		if (!$carpenter_shop_menu_exists) {
			$carpenter_shop_menu_id = wp_create_nav_menu($carpenter_shop_menuname);

			// Home
			wp_update_nav_menu_item($carpenter_shop_menu_id, 0, array(
				'menu-item-title' => __('Home', 'carpenter-shop'),
				'menu-item-classes' => 'home',
				'menu-item-url' => home_url('/'),
				'menu-item-status' => 'publish'
			));

			// About
			$carpenter_shop_page_about = get_page_by_path('about');
			if($carpenter_shop_page_about){
				wp_update_nav_menu_item($carpenter_shop_menu_id, 0, array(
					'menu-item-title' => __('About', 'carpenter-shop'),
					'menu-item-classes' => 'about',
					'menu-item-url' => get_permalink($carpenter_shop_page_about),
					'menu-item-status' => 'publish'
				));
			}

			// Services
			$carpenter_shop_page_services = get_page_by_path('services');
			if($carpenter_shop_page_services){
				wp_update_nav_menu_item($carpenter_shop_menu_id, 0, array(
					'menu-item-title' => __('Services', 'carpenter-shop'),
					'menu-item-classes' => 'services',
					'menu-item-url' => get_permalink($carpenter_shop_page_services),
					'menu-item-status' => 'publish'
				));
			}

			// Shop Page (WooCommerce)
			if (class_exists('WooCommerce')) {
				$carpenter_shop_shop_page_id = wc_get_page_id('shop');
				if ($carpenter_shop_shop_page_id) {
					wp_update_nav_menu_item($carpenter_shop_menu_id, 0, array(
						'menu-item-title' => __('Shop', 'carpenter-shop'),
						'menu-item-classes' => 'shop',
						'menu-item-url' => get_permalink($carpenter_shop_shop_page_id),
						'menu-item-status' => 'publish'
					));
				}
			}

			// Blog
			$carpenter_shop_page_blog = get_page_by_path('blog');
			if($carpenter_shop_page_blog){
				wp_update_nav_menu_item($carpenter_shop_menu_id, 0, array(
					'menu-item-title' => __('Blog', 'carpenter-shop'),
					'menu-item-classes' => 'blog',
					'menu-item-url' => get_permalink($carpenter_shop_page_blog),
					'menu-item-status' => 'publish'
				));
			}

			// 404 Page
			$carpenter_shop_notfound = get_page_by_path('404 Page');
			if($carpenter_shop_notfound){
				wp_update_nav_menu_item($carpenter_shop_menu_id, 0, array(
					'menu-item-title' => __('404 Page', 'carpenter-shop'),
					'menu-item-classes' => '404',
					'menu-item-url' => get_permalink($carpenter_shop_notfound),
					'menu-item-status' => 'publish'
				));
			}

			if (!has_nav_menu($carpenter_shop_menulocation)) {
				$carpenter_shop_locations = get_theme_mod('nav_menu_locations');
				$carpenter_shop_locations[$carpenter_shop_menulocation] = $carpenter_shop_menu_id;
				set_theme_mod('nav_menu_locations', $carpenter_shop_locations);
			}
		}
	}

	
	/**
	 * Imports the Demo Content
	 * @since 1.1.0
	 */
	public function carpenter_shop_setup_widgets(){

		/* -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+- MENUS PAGES -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
		
			// Creation of home page //
			$carpenter_shop_home_content = '';
			$carpenter_shop_home_title = 'Home';
			$carpenter_shop_home = array(
					'post_type' => 'page',
					'post_title' => $carpenter_shop_home_title,
					'post_content'  => $carpenter_shop_home_content,
					'post_status' => 'publish',
					'post_author' => 1,
					'post_slug' => 'home'
			);
			$carpenter_shop_home_id = wp_insert_post($carpenter_shop_home);

			add_post_meta( $carpenter_shop_home_id, '_wp_page_template', 'frontpage.php' );

			$carpenter_shop_home = get_page_by_path( 'Home' );
			update_option( 'page_on_front', $carpenter_shop_home->ID );
			update_option( 'show_on_front', 'page' );

			// Creation of blog page //
			$carpenter_shop_blog_title = 'Blog';
			$carpenter_shop_blog_check = get_page_by_path('blog');
			if (!$carpenter_shop_blog_check) {
				$carpenter_shop_blog = array(
					'post_type'    => 'page',
					'post_title'   => $carpenter_shop_blog_title,
					'post_status'  => 'publish',
					'post_author'  => 1,
					'post_name'    => 'blog'
				);
				$carpenter_shop_blog_id = wp_insert_post($carpenter_shop_blog);

				if (!is_wp_error($carpenter_shop_blog_id)) {
					update_option('page_for_posts', $carpenter_shop_blog_id);
				}
			}

			// Creation of about page //
			$carpenter_shop_about_title = 'About';
			$carpenter_shop_about_content = 'What is Lorem Ipsum?
														Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industrys standard dummy text ever since the 1500s, when an unknown printer took a galley of type and scrambled it to make a type specimen book. It has survived not only five centuries, but also the leap into electronic typesetting, remaining essentially unchanged. It was popularised in the 1960s with the release of Letraset sheets containing Lorem Ipsum passages, and more recently with desktop publishing software like Aldus PageMaker including versions of Lorem Ipsum.
														&nbsp;
														Why do we use it?
														It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout. The point of using Lorem Ipsum is that it has a more-or-less normal distribution of letters, as opposed to using Content here, content here, making it look like readable English. Many desktop publishing packages and web page editors now use Lorem Ipsum as their default model text, and a search for lorem ipsum will uncover many web sites still in their infancy. Various versions have evolved over the years, sometimes by accident, sometimes on purpose (injected humour and the like).
														&nbsp;
														Where does it come from?
														There are many variations of passages of Lorem Ipsum available, but the majority have suffered alteration in some form, by injected humour, or randomised words which dont look even slightly believable. If you are going to use a passage of Lorem Ipsum, you need to be sure there isnt anything embarrassing hidden in the middle of text. All the Lorem Ipsum generators on the Internet tend to repeat predefined chunks as necessary, making this the first true generator on the Internet. It uses a dictionary of over 200 Latin words, combined with a handful of model sentence structures, to generate Lorem Ipsum which looks reasonable. The generated Lorem Ipsum is therefore always free from repetition, injected humour, or non-characteristic words etc.
														&nbsp;
														Why do we use it?
														It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout. The point of using Lorem Ipsum is that it has a more-or-less normal distribution of letters, as opposed to using Content here, content here, making it look like readable English. Many desktop publishing packages and web page editors now use Lorem Ipsum as their default model text, and a search for lorem ipsum will uncover many web sites still in their infancy. Various versions have evolved over the years, sometimes by accident, sometimes on purpose (injected humour and the like).
														&nbsp;
														Where does it come from?
														There are many variations of passages of Lorem Ipsum available, but the majority have suffered alteration in some form, by injected humour, or randomised words which dont look even slightly believable. If you are going to use a passage of Lorem Ipsum, you need to be sure there isnt anything embarrassing hidden in the middle of text. All the Lorem Ipsum generators on the Internet tend to repeat predefined chunks as necessary, making this the first true generator on the Internet. It uses a dictionary of over 200 Latin words, combined with a handful of model sentence structures, to generate Lorem Ipsum which looks reasonable. The generated Lorem Ipsum is therefore always free from repetition, injected humour, or non-characteristic words etc.';
			$carpenter_shop_about_check = get_page_by_path('about');
			if (!$carpenter_shop_about_check) {
				$carpenter_shop_about = array(
					'post_type'    => 'page',
					'post_title'   => $carpenter_shop_about_title,
					'post_content'   => $carpenter_shop_about_content,
					'post_status'  => 'publish',
					'post_author'  => 1,
					'post_name'    => 'about'
				);
				wp_insert_post($carpenter_shop_about);
			}

			// Creation of services page //
			$carpenter_shop_services_title = 'Services';
			$carpenter_shop_services_content = 'What is Lorem Ipsum?
														Lorem Ipsum is simply dummy text of the printing and typesetting industry. Lorem Ipsum has been the industrys standard dummy text ever since the 1500s, when an unknown printer took a galley of type and scrambled it to make a type specimen book. It has survived not only five centuries, but also the leap into electronic typesetting, remaining essentially unchanged. It was popularised in the 1960s with the release of Letraset sheets containing Lorem Ipsum passages, and more recently with desktop publishing software like Aldus PageMaker including versions of Lorem Ipsum.
														&nbsp;
														Why do we use it?
														It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout. The point of using Lorem Ipsum is that it has a more-or-less normal distribution of letters, as opposed to using Content here, content here, making it look like readable English. Many desktop publishing packages and web page editors now use Lorem Ipsum as their default model text, and a search for lorem ipsum will uncover many web sites still in their infancy. Various versions have evolved over the years, sometimes by accident, sometimes on purpose (injected humour and the like).
														&nbsp;
														Where does it come from?
														There are many variations of passages of Lorem Ipsum available, but the majority have suffered alteration in some form, by injected humour, or randomised words which dont look even slightly believable. If you are going to use a passage of Lorem Ipsum, you need to be sure there isnt anything embarrassing hidden in the middle of text. All the Lorem Ipsum generators on the Internet tend to repeat predefined chunks as necessary, making this the first true generator on the Internet. It uses a dictionary of over 200 Latin words, combined with a handful of model sentence structures, to generate Lorem Ipsum which looks reasonable. The generated Lorem Ipsum is therefore always free from repetition, injected humour, or non-characteristic words etc.
														&nbsp;
														Why do we use it?
														It is a long established fact that a reader will be distracted by the readable content of a page when looking at its layout. The point of using Lorem Ipsum is that it has a more-or-less normal distribution of letters, as opposed to using Content here, content here, making it look like readable English. Many desktop publishing packages and web page editors now use Lorem Ipsum as their default model text, and a search for lorem ipsum will uncover many web sites still in their infancy. Various versions have evolved over the years, sometimes by accident, sometimes on purpose (injected humour and the like).
														&nbsp;
														Where does it come from?
														There are many variations of passages of Lorem Ipsum available, but the majority have suffered alteration in some form, by injected humour, or randomised words which dont look even slightly believable. If you are going to use a passage of Lorem Ipsum, you need to be sure there isnt anything embarrassing hidden in the middle of text. All the Lorem Ipsum generators on the Internet tend to repeat predefined chunks as necessary, making this the first true generator on the Internet. It uses a dictionary of over 200 Latin words, combined with a handful of model sentence structures, to generate Lorem Ipsum which looks reasonable. The generated Lorem Ipsum is therefore always free from repetition, injected humour, or non-characteristic words etc.';
			$carpenter_shop_services_check = get_page_by_path('services');
			if (!$carpenter_shop_services_check) {
				$carpenter_shop_services = array(
					'post_type'    => 'page',
					'post_title'   => $carpenter_shop_services_title,
					'post_content'   => $carpenter_shop_services_content,
					'post_status'  => 'publish',
					'post_author'  => 1,
					'post_name'    => 'services'
				);
				wp_insert_post($carpenter_shop_services);
			}

			// Creation of 404 page //
			$carpenter_shop_notfound_title = '404 Page';
			$carpenter_shop_notfound = array(
				'post_type'   => 'page',
				'post_title'  => $carpenter_shop_notfound_title,
				'post_status' => 'publish',
				'post_author' => 1,
				'post_slug'   => '404'
			);
			$carpenter_shop_notfound_id = wp_insert_post($carpenter_shop_notfound);
			add_post_meta($carpenter_shop_notfound_id, '_wp_page_template', '404.php');



			/* -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+- SLIDER POST -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/

				$carpenter_shop_slider_title = array('Women Fashion Style Collections');
				for($carpenter_shop_i=1;$carpenter_shop_i<=1;$carpenter_shop_i++){

					$carpenter_shop_title = $carpenter_shop_slider_title[$carpenter_shop_i-1];
					$carpenter_shop_content = 'Women Jackets From Popular Brands.';

					// Create post object
					$carpenter_shop_my_post = array(
							'post_title'    => wp_strip_all_tags( $carpenter_shop_title ),
							'post_content'  => $carpenter_shop_content,
							'post_status'   => 'publish',
							'post_type'     => 'post',
					);
					// Insert the post into the database
					$carpenter_shop_post_id = wp_insert_post( $carpenter_shop_my_post );

					wp_set_object_terms($carpenter_shop_post_id, 'Slider', 'category', true);

					wp_set_object_terms($carpenter_shop_post_id, 'Slider', 'post_tag', true);

					$carpenter_shop_image_url = get_template_directory_uri().'/inc/homepage-setup/assets/homepage-setup-images/slider-img'.$carpenter_shop_i.'.png';

					$carpenter_shop_image_name= 'slider-img'.$carpenter_shop_i.'.png';
					$upload_dir       = wp_upload_dir();
					// Set upload folder
					$carpenter_shop_image_data       = file_get_contents($carpenter_shop_image_url);
					// Get image data
					$carpenter_shop_unique_file_name = wp_unique_filename( $upload_dir['path'], $carpenter_shop_image_name );

					$carpenter_shop_filename = basename( $carpenter_shop_unique_file_name ); 
					
					// Check folder permission and define file location
					if( wp_mkdir_p( $upload_dir['path'] ) ) {
							$carpenter_shop_file = $upload_dir['path'] . '/' . $carpenter_shop_filename;
					} else {
							$carpenter_shop_file = $upload_dir['basedir'] . '/' . $carpenter_shop_filename;
					}
					// Create the image  file on the server
					// Generate unique name
					if ( ! function_exists( 'WP_Filesystem' ) ) {
						require_once( ABSPATH . 'wp-admin/includes/file.php' );
					}
					
					WP_Filesystem();
					global $wp_filesystem;
					
					if ( ! $wp_filesystem->put_contents( $carpenter_shop_file, $carpenter_shop_image_data, FS_CHMOD_FILE ) ) {
						wp_die( 'Error saving file!' );
					}
					// Check image file type
					$wp_filetype = wp_check_filetype( $carpenter_shop_filename, null );
					// Set attachment data
					$carpenter_shop_attachment = array(
							'post_mime_type' => $wp_filetype['type'],
							'post_title'     => sanitize_file_name( $carpenter_shop_filename ),
							'post_content'   => '',
							'post_type'     => 'post',
							'post_status'    => 'inherit'
					);
					// Create the attachment
					$carpenter_shop_attach_id = wp_insert_attachment( $carpenter_shop_attachment, $carpenter_shop_file, $carpenter_shop_post_id );
					// Include image.php
					require_once(ABSPATH . 'wp-admin/includes/image.php');
					// Define attachment metadata
					$carpenter_shop_attach_data = wp_generate_attachment_metadata( $carpenter_shop_attach_id, $carpenter_shop_file );
					// Assign metadata to attachment
						wp_update_attachment_metadata( $carpenter_shop_attach_id, $carpenter_shop_attach_data );
					// And finally assign featured image to post
					set_post_thumbnail( $carpenter_shop_post_id, $carpenter_shop_attach_id );

	 			}


			/* -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+- PRODUCTS -+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-*/
			
				$carpenter_shop_uncategorized_term = get_term_by('name', 'Uncategorized', 'product_cat');
				if ($carpenter_shop_uncategorized_term) {
					wp_delete_term($carpenter_shop_uncategorized_term->term_id, 'product_cat');
				}

				$carpenter_shop_product_category= array(
					'Furniture' => array(
								'Furniture King Product Title 1',
								'Furniture King Product Title 2',
								'Furniture King Product Title 3',
					),
					'Art & Entertainment' => array(
					),
					'Baby & Toddler' => array(
					),
					'Electronics' => array(
					),
					'Animals & Pet Supplies' => array(
					),
					'Hardware' => array(
					),
					'Healthy & Beauty' => array(
					)
				);
				$carpenter_shop_i = 1;
				foreach ( $carpenter_shop_product_category as $carpenter_shop_product_cats => $carpenter_shop_products_name ) {

					// Insert porduct cats Start
					$carpenter_shop_content = 'Lorem ipsum dolor sit amet';
					$carpenter_shop_parent_category	=	wp_insert_term(
					$carpenter_shop_product_cats, // the term
					'product_cat', // the taxonomy
					array(
						'description'=> $carpenter_shop_content,
						'slug' => 'product_cat'.$carpenter_shop_i
					));

					$carpenter_shop_image_url = get_template_directory_uri().'/inc/homepage-setup/assets/homepage-setup-images/slider-img'.$carpenter_shop_i.'.png';

					$carpenter_shop_image_name= 'slider-img'.$carpenter_shop_i.'.png';
					$upload_dir       = wp_upload_dir();
					// Set upload folder
					$carpenter_shop_image_data= file_get_contents($carpenter_shop_image_url);
					// Get image data
					$carpenter_shop_unique_file_name = wp_unique_filename( $upload_dir['path'], $carpenter_shop_image_name );
					// Generate unique name
					$carpenter_shop_filename= basename( $carpenter_shop_unique_file_name );
					// Create image file name

					// Check folder permission and define file location
					if( wp_mkdir_p( $upload_dir['path'] ) ) {
					$file = $upload_dir['path'] . '/' . $carpenter_shop_filename;
					} else {
					$file = $upload_dir['basedir'] . '/' . $carpenter_shop_filename;
					}

					// Create the image  file on the server
					if ( ! function_exists( 'WP_Filesystem' ) ) {
						require_once( ABSPATH . 'wp-admin/includes/file.php' );
					}
					
					WP_Filesystem();
					global $wp_filesystem;
					
					if ( ! $wp_filesystem->put_contents( $file, $carpenter_shop_image_data, FS_CHMOD_FILE ) ) {
						wp_die( 'Error saving file!' );
					}
					
					// Check image file type
					$wp_filetype = wp_check_filetype( $carpenter_shop_filename, null );

					// Set attachment data
					$carpenter_shop_attachment = array(
					'post_mime_type' => $wp_filetype['type'],
					'post_title'     => sanitize_file_name( $carpenter_shop_filename ),
					'post_content'   => '',
					'post_type'     => 'product',
					'post_status'    => 'inherit'
					);

					// Create the attachment
					$carpenter_shop_attach_id = wp_insert_attachment( $carpenter_shop_attachment, $file, $post_id );

					// Include image.php
					require_once(ABSPATH . 'wp-admin/includes/image.php');

					// Define attachment metadata
					$carpenter_shop_attach_data = wp_generate_attachment_metadata( $carpenter_shop_attach_id, $file );

					// Assign metadata to attachment
					wp_update_attachment_metadata( $carpenter_shop_attach_id, $carpenter_shop_attach_data );

					update_woocommerce_term_meta( $carpenter_shop_parent_category['term_id'], 'thumbnail_id', $carpenter_shop_attach_id );

					// create Product START
					foreach ( $carpenter_shop_products_name as $key => $carpenter_shop_product_title ) {

						$carpenter_shop_content = 'Te obtinuit ut adepto satis somno.';
						// Create post object
						$my_post = array(
							'post_title'    => wp_strip_all_tags( $carpenter_shop_product_title ),
							'post_content'  => $carpenter_shop_content,
							'post_status'   => 'publish',
							'post_type'     => 'product',
						);

						// Insert the post into the database
						$post_id    = wp_insert_post($my_post);

						wp_set_object_terms( $post_id, 'product_cat' . $carpenter_shop_i, 'product_cat', true );

						update_post_meta($post_id, '_regular_price', '130');
						update_post_meta($post_id, '_sale_price', '100');
						update_post_meta($post_id, '_price', '100');

						// Now replace meta w/ new updated value array
						$carpenter_shop_image_url = get_template_directory_uri().'/inc/homepage-setup/assets/homepage-setup-images/'.str_replace( " ", "-", $carpenter_shop_product_title).'.png';

						echo $carpenter_shop_image_url . "<br>";

						$carpenter_shop_image_name       = $carpenter_shop_product_title.'.png';
						$upload_dir = wp_upload_dir();
						// Set upload folder
						$carpenter_shop_image_data = file_get_contents(esc_url($carpenter_shop_image_url));

						// Get image data
						$carpenter_shop_unique_file_name = wp_unique_filename($upload_dir['path'], $carpenter_shop_image_name);
						// Generate unique name
						$carpenter_shop_filename = basename($carpenter_shop_unique_file_name);
						// Create image file name

						// Check folder permission and define file location
						if (wp_mkdir_p($upload_dir['path'])) {
							$file = $upload_dir['path'].'/'.$carpenter_shop_filename;
						} else {
							$file = $upload_dir['basedir'].'/'.$carpenter_shop_filename;
						}

						// Create the image  file on the server
						if ( ! function_exists( 'WP_Filesystem' ) ) {
							require_once( ABSPATH . 'wp-admin/includes/file.php' );
						}
						
						WP_Filesystem();
						global $wp_filesystem;
						
						if ( ! $wp_filesystem->put_contents( $file, $carpenter_shop_image_data, FS_CHMOD_FILE ) ) {
							wp_die( 'Error saving file!' );
						}

						// Check image file type
						$wp_filetype = wp_check_filetype($carpenter_shop_filename, null);

						// Set attachment data
						$carpenter_shop_attachment = array(
							'post_mime_type' => $wp_filetype['type'],
							'post_title'     => sanitize_file_name($carpenter_shop_filename),
							'post_type'      => 'product',
							'post_status'    => 'inherit',
						);

						// Create the attachment
						$carpenter_shop_attach_id = wp_insert_attachment($carpenter_shop_attachment, $file, $post_id);

						// Include image.php
						require_once (ABSPATH.'wp-admin/includes/image.php');

						// Define attachment metadata
						$carpenter_shop_attach_data = wp_generate_attachment_metadata($carpenter_shop_attach_id, $file);

						// Assign metadata to attachment
						wp_update_attachment_metadata($carpenter_shop_attach_id, $carpenter_shop_attach_data);

						// And finally assign featured image to post
						set_post_thumbnail($post_id, $carpenter_shop_attach_id);
					}
					// Create product END
					++$carpenter_shop_i;
				}

        
        $this->carpenter_shop_customizer_nav_menu();

	    exit;
	}
}