document.addEventListener('DOMContentLoaded', () => {
    // Select the main elements of the slider with safety checks
    const container = document.querySelector('.slider-container');

    // If container doesn't exist, stop execution
    if (!container) {
        // console.warn('Slider container not found. Slider functionality disabled.');
        return;
    }

    const wrapper = container.querySelector('.slider-wrapper');
    const slides = container.querySelectorAll('.slide');
    const prevButton = container.querySelector('.slider-prev');
    const nextButton = container.querySelector('.slider-next');

    // Verify if all required elements exist
    if (!wrapper) {
        // console.error('Slider wrapper not found!');
        return;
    }

    if (slides.length === 0) {
        //console.error('No slides found!');
        return;
    }

    if (!prevButton || !nextButton) {
        // console.warn('Navigation buttons not found. Slider will work without manual navigation.');
    }

    let currentIndex = 0;
    let slideInterval;
    let isAnimating = false;

    // Function to update slider position
    function updateSlider() {
        if (isAnimating) return;

        isAnimating = true;

        try {
            const slideWidth = slides[0].clientWidth;
            const newPosition = -currentIndex * slideWidth;
            wrapper.style.transition = 'transform 0.5s ease-in-out';
            wrapper.style.transform = `translateX(${Math.floor(newPosition)}px)`;

            // Reset animation flag after transition
            setTimeout(() => {
                isAnimating = false;
            }, 500);

        } catch (error) {
            //console.error('Error updating slider:', error);
            isAnimating = false;
        }
    }

    // Go to next slide
    function goToNextSlide() {
        if (isAnimating) return;

        currentIndex = (currentIndex < slides.length - 1) ? currentIndex + 1 : 0;
        updateSlider();
    }

    // Go to previous slide
    function goToPrevSlide() {
        if (isAnimating) return;

        currentIndex = (currentIndex > 0) ? currentIndex - 1 : slides.length - 1;
        updateSlider();
    }

    // Start automatic slideshow
    function startSlideShow() {
        stopSlideShow(); // Clear any existing interval
        slideInterval = setInterval(goToNextSlide, 5000);
    }

    // Stop automatic slideshow
    function stopSlideShow() {
        if (slideInterval) {
            clearInterval(slideInterval);
            slideInterval = null;
        }
    }

    // Add event listeners only if buttons exist
    if (prevButton) {
        prevButton.addEventListener('click', goToPrevSlide);
    }

    if (nextButton) {
        nextButton.addEventListener('click', goToNextSlide);
    }

    // Pause on hover
    container.addEventListener('mouseenter', stopSlideShow);
    container.addEventListener('mouseleave', startSlideShow);

    // Handle window resize
    function handleResize() {
        if (!isAnimating) {
            updateSlider();
        }
    }

    // Debounced resize handler
    let resizeTimeout;
    window.addEventListener('resize', () => {
        clearTimeout(resizeTimeout);
        resizeTimeout = setTimeout(handleResize, 250);
    });

    // Initialize slider
    function initSlider() {
        // Set initial position
        wrapper.style.transition = 'none';
        wrapper.style.transform = `translateX(0px)`;

        // Force reflow
        wrapper.offsetHeight;

        // Start slideshow
        startSlideShow();

        // console.log('Slider initialized successfully with', slides.length, 'slides');
    }

    // Start the slider
    initSlider();

    // Public methods (optional - for external control)
    window.sliderAPI = {
        next: goToNextSlide,
        prev: goToPrevSlide,
        start: startSlideShow,
        stop: stopSlideShow,
        goTo: (index) => {
            if (index >= 0 && index < slides.length && !isAnimating) {
                currentIndex = index;
                updateSlider();
            }
        }
    };
});