<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package Business_Process
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function business_process_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}
    // Add class for Slidebar layout.
    $sidebar_layout = business_process_get_option( 'layout_options' ); 
    $sidebar_layout = apply_filters( 'business_process_theme_global_layout', $sidebar_layout );
    $classes[] = 'global-layout-' . esc_attr( $sidebar_layout );  

	// Adds a class of no-sidebar when there is no sidebar present.
	if ( ! is_active_sidebar( 'sidebar-1' ) ) {
		$classes[] = 'global-layout-no-sidebar';
	}

	return $classes;
}
add_filter( 'body_class', 'business_process_body_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function business_process_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'business_process_pingback_header' );

if ( ! function_exists( 'business_process_the_excerpt' ) ) :

    /**
     * Generate excerpt.
     *
     * @since 1.0.0
     *
     * @param int     $length Excerpt length in words.
     * @param WP_Post $post_obj WP_Post instance (Optional).
     * @return string Excerpt.
     */
    function business_process_the_excerpt( $length = 0, $post_obj = null ) {

        global $post;

        if ( is_null( $post_obj ) ) {
            $post_obj = $post;
        }

        $length = absint( $length );

        if ( 0 === $length ) {
            return;
        }

        $source_content = $post_obj->post_content;

        if ( ! empty( $post_obj->post_excerpt ) ) {
            $source_content = $post_obj->post_excerpt;
        }

        $source_content = preg_replace( '`\[[^\]]*\]`', '', $source_content );
        $trimmed_content = wp_trim_words( $source_content, $length, '&hellip;' );
        return $trimmed_content;

    }

endif;

if ( ! function_exists( 'business_process_breadcrumb' ) ) :

    /**
     * Simple breadcrumb.
     *
     * @since 1.0.0
     *
     * @link: https://gist.github.com/melissacabral/4032941
     *
     * @param  array $args Arguments
     */
    function business_process_breadcrumb( $args = array() ) {
        // Bail if Home Page.
        if ( is_front_page() || is_home() ) {
            return;
        }

        if ( ! function_exists( 'breadcrumb_trail' ) ) {
            require_once trailingslashit(get_template_directory()) . '/inc/breadcrumbs.php';
        }

        $breadcrumb_args = array(
            'container'   => 'div',
            'show_browse' => false,
        );
        breadcrumb_trail( $breadcrumb_args );
       
    }

endif;

if ( ! function_exists( 'business_process_header_banner_title' ) ) :

    /**
     * Display custom header title in frontpage and blog
     */
    function business_process_header_banner_title() {
        if ( ! is_front_page() && is_home() || is_singular() ): ?>
            <h2 class="page-title"><?php single_post_title(); ?></h2>
        <?php elseif ( is_archive() ) : 
            the_archive_title( '<h2 class="page-title">', '</h2>' );
        elseif ( is_search() ) : ?>
            <h2 class="page-title"><?php printf( esc_html__( 'Search Results for: %s', 'business-process' ), get_search_query() ); ?></h2>
        <?php elseif ( is_404() ) :
            echo '<h2 class="page-title">' . esc_html__( 'Oops! That page can&#39;t be found.', 'business-process' ) . '</h2>';
        endif;
    }
endif;

if ( ! function_exists( 'business_process_content_length' ) ):
    /**
     * Displays an optional post content.
     *
     */
    function business_process_content_length() {
        $post_length = business_process_get_option( 'post_length' );
        $post_content = business_process_get_option( 'post_content' );

        if ( 'excerpt' == $post_content ){
            $excerpt = business_process_the_excerpt( absint( $post_length ) );
            echo wp_kses_post( wpautop( $excerpt ) );
        } else {
            the_content();
        }
    }


endif; 


/**
 * Register the required plugins for this theme.
 * 
 * This function is hooked into `tgmpa_register`, which is fired on the WP `init` action on priority 10.
 */
function business_process_register_required_plugins() {
    /*
     * Array of plugin arrays. Required keys are name and slug.
     * If the source is NOT from the .org repo, then source is also required.
     */
    $plugins = array(
        array(
            'name'      => esc_html__( 'Contact Form 7', 'business-process' ), //The plugin name
            'slug'      => 'contact-form-7',  // The plugin slug (typically the folder name)
            'required'  => false,  // If false, the plugin is only 'recommended' instead of required.
            'force_activation'   => false, // If true, plugin is activated upon theme activation and cannot be deactivated until theme switch.
            'force_deactivation' => false, // If true, plugin is deactivated upon theme switch, useful for theme-specific plugins.
        ),        

    );

    $config = array(
        'id'           => 'business-process',        // Unique ID for hashing notices for multiple instances of TGMPA.
        'default_path' => '',                      // Default absolute path to bundled plugins.     
        'has_notices'  => true,                    // Show admin notices or not.
        'dismissable'  => true,                    // If false, a user cannot dismiss the nag message.
        'dismiss_msg'  => '',                      // If 'dismissable' is false, this message will be output at top of nag.
        'is_automatic' => false,                   // Automatically activate plugins after installation or not.
        'message'      => '',                      // Message to output right before the plugins table.
        );

    tgmpa( $plugins, $config );
}
add_action( 'tgmpa_register', 'business_process_register_required_plugins' );