<?php
/**
 * Business Process functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Business_Process
 */

if ( ! function_exists( 'business_process_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function business_process_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Business Process, use a find and replace
		 * to change 'business-process' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'business-process', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'business-prcoess-blog', 285, 287, array( 'center', 'center') );
		
		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'menu-1' => esc_html__( 'Primary', 'business-process' ),
			'social-media' => esc_html__( 'Social Media', 'business-process' ),
			'top-menu' => esc_html__( 'Top Menu', 'business-process' ),
			'footer-menu' => esc_html__( 'Footer Menu', 'business-process' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'business_process_custom_background_args', array(
			'default-color' => 'ffffff',
			'default-image' => '',
		) ) );

		// Add theme support for selective refresh for widgets.
		add_theme_support( 'customize-selective-refresh-widgets' );

		// Add support for Block Styles.
		add_theme_support( 'wp-block-styles' );

		// Add support for full and wide align images.
		add_theme_support( 'align-wide' );

        // Responsive embeds
        add_theme_support( 'responsive-embeds' );

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support( 'custom-logo', array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		) );
	}
endif;
add_action( 'after_setup_theme', 'business_process_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function business_process_content_width() {
	// This variable is intended to be overruled from themes.
	// Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$GLOBALS['content_width'] = apply_filters( 'business_process_content_width', 640 );
}
add_action( 'after_setup_theme', 'business_process_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function business_process_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Sidebar', 'business-process' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here.', 'business-process' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

	for ($i=1; $i < 3 ; $i++) { 
		register_sidebar( array(
		'name'          => sprintf( esc_html__( 'Footer %d', 'business-process' ), $i ),
		'id'            => 'footer-'.$i,
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
		) );
	}
	
}
add_action( 'widgets_init', 'business_process_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function business_process_scripts() {
	$header_type 		= business_process_get_option( 'header_type' );

	wp_enqueue_style( 'business-process-fonts', business_process_fonts_url(), array(), null );

	//fontawesome css
	wp_enqueue_style( 'fontawesome', get_template_directory_uri() . '/assets/css/fontawesome-all.css' );

	//meanmenu css
	wp_enqueue_style( 'meanmenu', get_template_directory_uri() . '/assets/css/meanmenu.css' );

	//animate css
	wp_enqueue_style( 'animate', get_template_directory_uri() . '/assets/css/animate.css' );	

	//owl-carousel css
	wp_enqueue_style( 'owl-carousel', get_template_directory_uri() . '/assets/css/owl.carousel.css' );

	//owl theme default css
	wp_enqueue_style( 'owl-theme-default', get_template_directory_uri() . '/assets/css/owl.theme.default.css' );

	wp_enqueue_style( 'business-process-style', get_stylesheet_uri() );

	//Responsive css
	wp_enqueue_style( 'business-process-responsive', get_template_directory_uri() . '/assets/css/responsive.css' );


    //carousel
    wp_enqueue_script( 'owl-carousel', get_template_directory_uri() . '/assets/js/owl.carousel.js', array('jquery'), 'v2.2.1', true );

    //wow js
    wp_enqueue_script( 'wow', get_template_directory_uri() . '/assets/js/wow.js', array('jquery'), 'v2.2.1', true );

    //easyResponsiveTabs js
    wp_enqueue_script( 'easyResponsiveTabs', get_template_directory_uri() . '/assets/js/easyResponsiveTabs.js', array('jquery'), 'v2.2.1', true );

    //jquery-meanmenu
    wp_enqueue_script( 'meanmenu', get_template_directory_uri() . '/assets/js/jquery.meanmenu.js', array('jquery'), 'v2.0.8', true );

	if ( 'sticky' == $header_type ){
		wp_enqueue_script( 'business-process-sticky', get_template_directory_uri() . '/assets/js/sticky.js', array(), '1.0.0', true );
	}

	wp_enqueue_script( 'business-process-navigation', get_template_directory_uri() . '/assets/js/navigation.js', array(), '20151215', true );

    wp_enqueue_script( 'business-process-keyboard-navigation', get_template_directory_uri() . '/assets/js/keyboard-navigation.js', array(), '20151215', true );

	wp_enqueue_script( 'business-process-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix.js', array(), '20151215', true );

	//custom
	wp_enqueue_script( 'business-process-custom', get_template_directory_uri() . '/assets/js/custom.js', array(), '20180506', true );


	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'business_process_scripts' );

/**
 * Load init.
 */
require_once trailingslashit( get_template_directory() ) . '/inc/init.php';
