<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package Business_Aarambha
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function business_aarambha_body_classes( $classes ) {

    // Adds a class of hfeed to non-singular pages.
    if ( ! is_singular() ) {
        $classes[] = 'hfeed';
    }

    // Adds a class of no-sidebar when there is no sidebar present.
    if ( ! is_active_sidebar( 'sidebar-1' ) || ! Business_Aarambha_Helper::get_sidebar_layout() ) {
        $classes[] = 'no-sidebar';
    }

    if ( is_front_page() && is_home() || is_home() || is_search() || is_archive() || is_404() ) {
        $classes[] = 'business-aarambha-blog';
    }

    // Is Sticky Sidebar
    $is_sticky_sidebar = get_theme_mod(
        'business_aarambha_sidebar_sticky',
        ''
    );
    if ( $is_sticky_sidebar ) {
        $classes[]  = 'has-sticky-sidebar';
    }

    // Placeholder or Thumbnail
    if ( is_singular() ) {
        if ( has_post_thumbnail() ) {
            $classes[]  = 'has-thumbnail';
        }
    }
    else {
        $classes[]  = 'has-blog-thumbnail';
    }

    return array_unique($classes);
}
add_filter( 'body_class', 'business_aarambha_body_classes' );

/**
 * Adds custom classes to the array of post classes.
 *
 * @param array $classes Classes for the post element.
 * @return array
 */
function business_aarambha_post_classes( $classes ) {
    // Featured Image Condition
    $classes[] = 'post';
    if ( has_post_thumbnail() || ( is_archive() || is_front_page() && is_home() ) || is_search() ) {
        $classes[] = 'has-featured-image';
    }
    else {
        $classes[] = 'no-featured-image';
    }

    return array_unique($classes);
}
add_filter( 'post_class', 'business_aarambha_post_classes' );

/**
 * Add a pingback url auto-discovery header for single posts, pages, or attachments.
 */
function business_aarambha_pingback_header() {
    if ( is_singular() && pings_open() ) {
        printf( '<link rel="pingback" href="%s">', esc_url( get_bloginfo( 'pingback_url' ) ) );
    }
}
add_action( 'wp_head', 'business_aarambha_pingback_header' );

/**
 * Return an array of all icons.
 */
function business_aarambha_get_fontawesome() {
        // Bail if the nonce doesn't check out
        if ( ! isset( $_POST['business_aarambha_customize_nonce'] ) || ! wp_verify_nonce( sanitize_key( $_POST['business_aarambha_customize_nonce'] ), 'business_aarambha_customize_nonce' ) ) {
            wp_die();
        }

        // Do another nonce check
        check_ajax_referer( 'business_aarambha_customize_nonce', 'business_aarambha_customize_nonce' );

        // Bail if user can't edit theme options
        if ( ! current_user_can( 'edit_theme_options' ) ) {
            wp_die();
        }

        // Get all of our fonts
        $fonts = Business_Aarambha_Font_Awesome_Icons::$icons;

        ob_start();
        if( $fonts ){ ?>
            <ul class="font-group">
                <?php
                foreach( $fonts as $font => $val ){
                    echo '<li data-font="' . esc_attr( $font ) . '"><i class="fa ' . esc_attr( $font ) . '"></i></li>';
                }
                ?>
            </ul>
            <?php
        }
        $output = ob_get_clean();
        echo apply_filters( 'business_aarambha_get_fontawesome', $output ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped

        // Exit
        wp_die();
    }
add_action( 'wp_ajax_business_aarambha_get_fontawesome', 'business_aarambha_get_fontawesome' );

if ( ! function_exists( 'business_aarambha_get_template_part' ) ) {

	/**
	 * business_aarambha_get_template_part
	 *
	 * @param      $id
	 * @param      $slug
	 * @param null $name
	 */
	function business_aarambha_get_template_part( $id, $slug, $name = null ) {

		$templates = array();
		$name      = (string) $name;
		if ( '' !== $name ) {
			$templates[] = "{$slug}-{$name}.php";
		}

		$templates[] = "{$slug}.php";
		$template    = locate_template( $templates );

		// Allow 3rd party plugins to filter template file from their plugin.
		$template = apply_filters( 'business_aarambha_get_template_part', $template, $id, $slug, $name );
		if ( $template ) {
			load_template( $template, false );
		}
	}
}

/**
 * Add font-awesome 4 support for the elementor
 */
function business_aarambha_set_elementor_load_fa4_shim() {
    $elementor_load_fa4_shim = get_option( 'elementor_load_fa4_shim' );

    if ( ! $elementor_load_fa4_shim || '' === $elementor_load_fa4_shim ) {
        update_option( 'elementor_load_fa4_shim', 'yes' );
    }
}
add_action( 'after_setup_theme', 'business_aarambha_set_elementor_load_fa4_shim' );

/**
 * Disable Getting Start After activating Elementor.
 */
add_action( 'admin_init', function() {
	if ( did_action( 'elementor/loaded' ) ) {
		remove_action( 'admin_init', [ \Elementor\Plugin::$instance->admin, 'maybe_redirect_to_getting_started' ] );
	}
}, 1 );