<?php
/**
 * Brilliant functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Brilliant
 * @since Brilliant 1.0
 */


/**
* Make theme available for translation.
*/
add_action( 'init', function() {
	load_theme_textdomain( 'brilliant', get_template_directory() . '/languages' );
} );


if ( ! function_exists( 'brilliant_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function brilliant_setup() {

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	// Let WordPress manage the document title
	add_theme_support( 'title-tag' );

	// Enable support for custom logo.
	add_theme_support( 'custom-logo', array(
		'height'      => 400,
		'width'       => 400,
		'flex-width'  => true,
		'flex-height' => true,
	) );

	// Enable support for Post Thumbnail
	add_theme_support( 'post-thumbnails' );
	add_image_size( 'brilliant-large', 780, 520, true );
	add_image_size( 'brilliant-fullwidth', 1200, 520, true );

	// Set the default content width.
	$GLOBALS['content_width'] = 1160;

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'main_menu' => esc_html__( 'Main Menu', 'brilliant' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array( 'comment-form', 'comment-list', 'gallery', 'caption' ) );

	// Enable support for Post Formats
	add_theme_support('post-formats', array( 'image', 'video', 'audio', 'gallery', 'quote' ) );

	// Set up the WordPress Custom Background Feature.
	$defaults = array(
		'default-color'	=> 'f4f4f4',
		'default-image'	=> '',
	);
	add_theme_support( 'custom-background', $defaults );

	// Add AMP support
	add_theme_support( 'amp' );

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	// Add support for responsive embeds.
	add_theme_support( 'responsive-embeds' );

	// Add support for full and wide align images.
	add_theme_support( 'align-wide' );

	// Load regular editor styles into the new block-based editor.
	add_theme_support( 'editor-styles' );

	// This theme styles the visual editor to resemble the theme style,
	add_editor_style( array( 'inc/css/editor-style.css', brilliant_fonts_url() ) );

	// Add support for custom color scheme.
	add_theme_support( 'editor-color-palette', array(
		array(
			'name'  => __( 'Black', 'brilliant' ),
			'slug'  => 'black',
			'color' => '#000000',
		),
		array(
			'name'  => __( 'Medium Gray', 'brilliant' ),
			'slug'  => 'medium-gray',
			'color' => '#444444',
		),
		array(
			'name'  => __( 'Light Gray', 'brilliant' ),
			'slug'  => 'light-gray',
			'color' => '#999999',
		),
		array(
			'name'  => __( 'White', 'brilliant' ),
			'slug'  => 'white',
			'color' => '#ffffff',
		),
		array(
			'name'  => __( 'Accent Color', 'brilliant' ),
			'slug'  => 'accent',
			'color' => esc_attr( get_theme_mod( 'accent_color', '#e63855' ) ),
		),
	) );

}
endif;
add_action( 'after_setup_theme', 'brilliant_setup' );


if ( ! function_exists( 'brilliant_fonts_url' ) ) :
/**
 * Register Google fonts.
 *
 * @return string Google fonts URL for the theme.
 */
function brilliant_fonts_url() {
	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';

	/* translators: If there are characters in your language that are not supported by Raleway, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Raleway: on or off', 'brilliant' ) ) {
		$fonts[] = 'Raleway:400,700,400italic,700italic';
	}

	/* translators: If there are characters in your language that are not supported by Playfair Display, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Playfair Display: on or off', 'brilliant' ) ) {
		$fonts[] = 'Playfair Display:400,700';
	}

	/* translators: If there are characters in your language that are not supported by Crimson Text, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== _x( 'on', 'Crimson Text: on or off', 'brilliant' ) ) {
		$fonts[] = 'Crimson Text:400,400italic';
	}

	/* translators: To add an additional character subset specific to your language, translate this to 'greek', 'cyrillic', 'devanagari' or 'vietnamese'. Do not translate into your own language. */
	$subset = _x( 'no-subset', 'Add new subset (greek, cyrillic, devanagari, vietnamese)', 'brilliant' );

	if ( 'cyrillic' == $subset ) {
		$subsets .= ',cyrillic,cyrillic-ext';
	} elseif ( 'greek' == $subset ) {
		$subsets .= ',greek,greek-ext';
	} elseif ( 'devanagari' == $subset ) {
		$subsets .= ',devanagari';
	} elseif ( 'vietnamese' == $subset ) {
		$subsets .= ',vietnamese';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
			'display' => 'fallback',
		), 'https://fonts.googleapis.com/css' );
	}

	return $fonts_url;
}
endif;


/**
 * Add preconnect for Google Fonts.
 *
 * @since Brilliant 1.2.0
 *
 * @param array  $urls URLs to print for resource hints.
 * @param string $relation_type The relation type the URLs are printed.
 * @return array URLs to print for resource hints.
 */
function brilliant_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'brilliant-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}
	return $urls;
}
add_filter( 'wp_resource_hints', 'brilliant_resource_hints', 10, 2 );


/**
 * Enqueue scripts and styles.
 */
function brilliant_scripts() {

	// Add Google fonts, used in the main stylesheet.
	wp_enqueue_style( 'brilliant-fonts', brilliant_fonts_url(), array(), null );

	// Add Icons font, used in the main stylesheet.
	wp_enqueue_style( 'brilliant-icons', get_template_directory_uri() . '/assets/css/elegant-icons.min.css', array(), '1.0.1' );

	// Theme stylesheet.
	$theme_version = wp_get_theme()->get( 'Version' );
	wp_enqueue_style( 'brilliant-style', get_stylesheet_uri(), array(), $theme_version );

	if( ! brilliant_is_amp() ) {
		// Main js.
		wp_enqueue_script( 'brilliant-script', get_template_directory_uri() . '/assets/js/script.js', array(), '20210930', true );
		// Comment reply script.
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}
}
add_action( 'wp_enqueue_scripts', 'brilliant_scripts' );


/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function brilliant_widgets_init() {
	register_sidebar( array(
		'name'          => __( 'Sidebar', 'brilliant' ),
		'id'            => 'sidebar-1',
		'description'   => __( 'Add widgets here to appear in your Sidebar.', 'brilliant' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h3 class="widget-title">',
		'after_title'   => '</h3>',
	) );
	register_sidebar( array(
		'name'          => __( 'Footer Widget Area - Left', 'brilliant' ),
		'id'            => 'footer-1',
		'description'   => __( 'Add widgets here to appear in your footer.', 'brilliant' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title"><span>',
		'after_title'   => '</span></h3>',
	) );
	register_sidebar( array(
		'name'          => __( 'Footer Widget Area - Center', 'brilliant' ),
		'id'            => 'footer-2',
		'description'   => __( 'Add widgets here to appear in your footer.', 'brilliant' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title"><span>',
		'after_title'   => '</span></h3>',
	) );
	register_sidebar( array(
		'name'          => __( 'Footer Widget Area - Right', 'brilliant' ),
		'id'            => 'footer-3',
		'description'   => __( 'Add widgets here to appear in your footer.', 'brilliant' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h3 class="widget-title"><span>',
		'after_title'   => '</span></h3>',
	) );
}
add_action( 'widgets_init', 'brilliant_widgets_init' );


/**
 * Shim for wp_body_open, ensuring backward compatibility with versions of WordPress older than 5.2.
 */
if ( ! function_exists( 'wp_body_open' ) ) {
	function wp_body_open() {
		do_action( 'wp_body_open' );
	}
}


/**
 * Include a skip to content link at the top of the page so that users can bypass the menu.
 */
function brilliant_skip_link() {
	echo '<a class="skip-link screen-reader-text" href="#content">' . __( 'Skip to content', 'brilliant' ) . '</a>';
}
add_action( 'wp_body_open', 'brilliant_skip_link', 5 );


/**
 * Determine whether the current response being served as AMP.
 */
function brilliant_is_amp() {
	return function_exists( 'is_amp_endpoint' ) && is_amp_endpoint();
}


/**
 * Adds a Sub Nav Toggle to the Mobile Menu.
 */
function brilliant_add_sub_menu_toggles( $output, $item, $depth, $args ) {
	if( ! brilliant_is_amp() ) {
		if ( isset( $args->show_sub_menu_toggles ) && $args->show_sub_menu_toggles && in_array( 'menu-item-has-children', $item->classes, true ) ) {
			$output = $output . '<button class="dropdown-toggle" aria-expanded="false"><span class="screen-reader-text">' . __( 'Show sub menu', 'brilliant' ) . '</span>' . '</button>';
		}
	} else {
		if ( isset( $args->show_sub_menu_toggles ) && $args->show_sub_menu_toggles && in_array( 'menu-item-has-children', $item->classes, true ) ) {
			$output = $output . '<button data-amp-bind-class="visible' . $item->ID . ' ? \'dropdown-toggle is-open\' : \'dropdown-toggle is-closed\'" on="tap:AMP.setState({visible' . $item->ID . ': !visible' . $item->ID . '})" class="dropdown-toggle" aria-expanded="false"><span class="screen-reader-text">' . __( 'Show sub menu', 'brilliant' ) . '</span>' . '</button>';
		}
	}
	return $output;
}
add_filter( 'walker_nav_menu_start_el', 'brilliant_add_sub_menu_toggles', 10, 4 );


/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';
require get_template_directory() . '/inc/customizer-css.php';


/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';


/**
 * Main Menu Fallback
 */
function brilliant_fallback_menu() {
	echo '<ul class="main-menu">';
	wp_list_pages( 'title_li=' );
	echo '</ul>';
}


/**
 * Display Custom Logo
 */
function brilliant_custom_logo() {
	if ( function_exists( 'the_custom_logo' ) && has_custom_logo() ) : ?>
		<div class="site-logo"><?php the_custom_logo(); ?></div>
	<?php
	else :
		if ( is_front_page() && is_home() ) { ?>
			<h1 class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
		<?php } else { ?>
			<p class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" rel="home"><?php bloginfo( 'name' ); ?></a></p>
		<?php }
	endif;
}


/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function brilliant_body_classes( $classes ) {
	// Adds a class of group-blog to blogs with more than 1 published author.
	if ( is_multi_author() ) {
		$classes[] = 'group-blog';
	}

	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	$blog_layout = esc_attr( get_theme_mod('blog_layout', 'blog-layout2') );
	$archive_layout = esc_attr( get_theme_mod('archive_layout', 'archive-layout2') );

	// Adds a class to Homepage
	if ( is_home() ) {
		$classes[] = $blog_layout;
	}

	// Adds a class to Archive and Search Pages
	if ( is_archive() || is_search() ) {
		$classes[] = $archive_layout;
	}

	return $classes;
}
add_filter( 'body_class', 'brilliant_body_classes' );


/**
 * Filter the except length.
 */
function brilliant_excerpt_length( $excerpt_length ) {

	if ( is_admin() ) {
		return $excerpt_length;
	}

	if ( is_home() ) {
		$excerpt_length = get_theme_mod( 'blog_excerpt_length', 25 );
	} elseif ( is_archive() || is_search() ) {
		$excerpt_length = get_theme_mod( 'archive_excerpt_length', 25 );
	} else {
		$excerpt_length = 25;
	}
	return intval($excerpt_length);
}
add_filter( 'excerpt_length', 'brilliant_excerpt_length', 999 );


/**
 * Filter the "read more" excerpt string link to the post.
 *
 * @param string $more "Read more" excerpt string.
 */
function brilliant_excerpt_more( $more ) {
	if ( is_admin() ) {
		return $more;
	}

	if ( get_theme_mod( 'show_read_more', 1 ) ) {
		$more = sprintf( '<span class="read-more-link"><a class="read-more" href="%1$s">%2$s &rsaquo;</a></span>',
		esc_url( get_permalink( get_the_ID() ) ),
		__( 'Read More', 'brilliant' )
		);
		return ' &hellip; ' . $more;
	} else {
		return $more;
	}
}
add_filter( 'excerpt_more', 'brilliant_excerpt_more' );


/**
 * Blog: Post Templates
 */
function brilliant_blog_template() {
	$blog_layout = get_theme_mod('blog_layout', 'blog-layout4');

	if ( $blog_layout == 'blog-layout2' || $blog_layout == 'blog-layout4' ) {
		$blog_template = 'content-grid';
	} else {
		$blog_template = 'content';
	}
	return sanitize_file_name($blog_template);
}


/**
 * Blog: Post Columns
 */
function brilliant_blog_column() {
	$blog_layout = get_theme_mod('blog_layout', 'blog-layout4');

	if ( $blog_layout == 'blog-layout2' || $blog_layout == 'blog-layout4' ) {
		$blog_column = 'col-6 col-sm-6';
	} else {
		$blog_column = 'col-12';
	}
	return esc_attr($blog_column);
}


/**
 * Archive: Post Templates
 */
function brilliant_archive_template() {
	$archive_layout = get_theme_mod('archive_layout', 'archive-layout2');

	if ( $archive_layout == 'archive-layout2' ) {
		$archive_template = 'content-grid';
	} else {
		if ( is_search() ) {
		$archive_template = 'content-search';
		} else {
		$archive_template = 'content';
		}
	}
	return sanitize_file_name($archive_template);
}


/**
 * Archive: Post Columns
 */
function brilliant_archive_column() {
	$archive_layout = get_theme_mod('archive_layout', 'archive-layout2');

	if ( $archive_layout == 'archive-layout2' ) {
		$archive_column = 'col-6 col-sm-6';
	} else {
		$archive_column = 'col-12';
	}
	return esc_attr($archive_column);
}


/**
 * Prints Credits in the Footer
 */
function brilliant_credits() {
	$website_credits = '';
	$website_author = get_bloginfo('name');
	$website_date = date_i18n(__( 'Y', 'brilliant' ) );
	$website_credits = '&copy; ' . $website_date . ' ' . $website_author;
	echo esc_html($website_credits);
}


/**
 * WooCommerce
 */
if ( ! function_exists( 'brilliant_is_woocommerce_active' ) ) {
	// Query WooCommerce activation
	function brilliant_is_woocommerce_active() {
		return class_exists( 'woocommerce' ) ? true : false;
	}
}

if ( brilliant_is_woocommerce_active() ) {
	// Declare WooCommerce support.
	function woocommerce_support() {
		add_theme_support( 'woocommerce' );
		add_theme_support( 'wc-product-gallery-zoom' );
		add_theme_support( 'wc-product-gallery-lightbox' );
		add_theme_support( 'wc-product-gallery-slider' );
	}
	add_action( 'after_setup_theme', 'woocommerce_support' );

	// WooCommerce Hooks.
	remove_action( 'woocommerce_before_main_content', 'woocommerce_output_content_wrapper', 10);
	remove_action( 'woocommerce_after_main_content', 'woocommerce_output_content_wrapper_end', 10);

	add_action('woocommerce_before_main_content', 'brilliant_wrapper_start', 10);
	add_action('woocommerce_after_main_content', 'brilliant_wrapper_end', 10);

	function brilliant_wrapper_start() {
		echo '<div id="primary" class="content-area"><main id="main" class="site-main" role="main"><div class="woocommerce-content">';
	}

	function brilliant_wrapper_end() {
		echo '</div></main></div>';
	}
}


/**
 * Add Upsell "pro" link to the customizer
 */
require_once( trailingslashit( get_template_directory() ) . '/inc/customize-pro/class-customize.php' );


/**
 * Add About Page
 */
require_once get_template_directory() . '/inc/about-page/about-brilliant.php';


/**
 * Add Upsell notice
 */
function brilliant_notice() {
	$user_id = get_current_user_id();
	if ( ! get_user_meta( $user_id, 'brilliant_notice_dismissed' ) ) {
	?>
	<div class="updated notice notice-success is-dismissible brilliant-admin-notice">
		<h2 class="welcome-title">
			<?php esc_html_e( 'Welcome! Thank you for choosing Brilliant WordPress Theme', 'brilliant' ); ?>
		</h2>
		<p>
			<?php echo wp_kses_post( __( '<strong>To fully take advantage</strong> of the best our theme can offer, please visit our', 'brilliant' ) ); ?> <a href="<?php echo esc_url( admin_url( 'themes.php?page=about_brilliant' ) ); ?>"><strong><?php echo esc_html__( 'Welcome Page', 'brilliant' ); ?></strong></a>
		</p>
		<p>
			<a class="button button-primary" href="<?php echo esc_url( 'https://www.designlabthemes.com/brilliant-plus-wordpress-theme/?utm_source=WordPress&utm_medium=notice&utm_campaign=brilliant_upsell' ); ?>" target="_blank">
				<?php esc_html_e( 'View Brilliant Plus', 'brilliant' ); ?>
			</a>
			<a style="color: #646970;margin-left: 0.5em;" href="<?php echo esc_url( '?brilliant-dismissed' ); ?>">
				<?php esc_html_e( 'Dismiss', 'brilliant' ); ?>
			</a>
		</p>
	</div>
	<?php
	}
}
add_action( 'admin_notices', 'brilliant_notice' );

function brilliant_notice_dismissed() {
	$user_id = get_current_user_id();
	if ( isset( $_GET['brilliant-dismissed'] ) ) {
		add_user_meta( $user_id, 'brilliant_notice_dismissed', 'true', true );
	}
}
add_action( 'admin_init', 'brilliant_notice_dismissed' );
